#!/usr/bin/python
#
# bigbrother
# http://snarfed.org/space/bigbrother
# Copyright 2003, 2004 Ryan Barrett <bigbrother@ryanb.org>
#
# File: graph.py
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

"""
Provides methods for generating graphs and plots of away message data generated
by bigbrother.py.
"""

import os
import gdchart
import chart
import bigbrother


# constants
TMP_DIR = '/tmp'
TMP_PREFIX = 'bigbrother_'
BIG_SIZE = (640, 320)
SMALL_SIZE = (320, 160)

COLORS = [0x0000ff, 0x00ff00, 0xff0000, 0x0080ff, 0x00ff80, 0xff0080, 0x8000ff,
          0xff8000, 0x8080ff, 0x80ff00, 0xff8080, 0xffff80, 0xff80ff, 0x80ffff,
          0x00ffff, 0xff00ff, 0xffff00, 0x808000, 0x800080, 0x008080]

# this is set from bigbrother.py
config = None

# a simple Chart wrapper that actually draws the chart on draw()
class BBChart(chart.Chart):
    def __init__(self):
        chart.Chart.__init__(self)
        self.option(format=gdchart.GDC_PNG)
#        self.option(bg_transparent=1)
        self.option(bg_color=0xffffff, edge_color=0x0, line_color=0x0,
                    plot_color=0x0000cd)
        self.option(set_color=COLORS)
        self.option(requested_yinterval=1)

    def draw(self, style, size, file, labels, *data):
        chart.Chart.draw(self)
        args = (style, size, file, labels) + data
        apply(gdchart.chart, args)


def pygd_piechart(graphfile, points):
    # cut off at threshold
    threshold = config.getint('bigbrother', 'large_tier_size')
    points, other = (points[:threshold - 1], points[threshold:])
    other = reduce((lambda x, y: ('Other', x[1] + y[1])), other, ('', 0))
    points.append(other)

    # set up chart
    c = BBChart()
    c.option(pie_color=list(COLORS))

    # make chart
    labels = [x[0] for x in points]
    values = [x[1] for x in points]
    c.draw(gdchart.GDC_2DPIE, BIG_SIZE, graphfile, labels, values)


def pygd_barchart(graphfile, points, big_size=True, unixtimes=False):
    # set up chart
    if big_size:
        size = BIG_SIZE
    else:
        size = SMALL_SIZE

    if unixtimes:
        # convert to hours
        points = [(label, unixtime / 3600) for (label, unixtime) in points]

    c = BBChart()


    # make chart
    labels = [label for (label, val) in points]
    values = [val for (label, val) in points]
    c.draw(gdchart.GDC_BAR, size, graphfile, labels, values)


def pygd_timeplot(graphfile, points):
    # set up chart
    c = BBChart()

    # prepare data
    points.sort()
    timestrs = [bigbrother.secs_to_time(secs) for (secs, val) in points]
    smoothed = smooth([val for (secs, val) in points])

    # make chart
    c.draw(gdchart.GDC_LINE, SMALL_SIZE, graphfile, timestrs, smoothed)


def pygd_stack_timeplot(graphfile, *points_set):
    # set up chart
    c = BBChart()
    c.option(requested_yinterval=None)

    # prepare labels
    step = config.getint('bigbrother', 'sample_interval')
    intervals = range(0, bigbrother.SECS_IN_DAY, step)
    timestrs = [bigbrother.secs_to_time(secs) for secs in intervals]

    # prepare datasets
    datasets = []
    for points in points_set:
        all = dict(zip(intervals, [0] * len(intervals)))
        all.update(dict(points))
        allpoints = all.items()
        allpoints.sort()
        smoothed = smooth([val for (secs, val) in allpoints])
        datasets.append(smoothed)

    # make chart
    c.option(stack_type=gdchart.GDC_STACK_BESIDE)
    c.option(threed_angle=80)
    args = [gdchart.GDC_LINE, BIG_SIZE, graphfile, timestrs]
    apply(c.draw, args + datasets)


#
# utilities
#
def interpolate(points, factor):
  """ Takes a list of (x, y) tuples and performs linear interpolation on them.
  Returns a list of length factor * (len(points) - 1), containing the original
  points and the points derived by interpolation.
  """
  assert factor > 0
  if not points:
    return []

  interpolated = []

  lastx, lasty = points[0]
  for x, y in points[1:]:
    xstep = float(x - lastx) / factor
    ystep = float(y - lasty) / factor
    for i in range(0, factor):
      interpolated.append((lastx + xstep * i, lasty + ystep * i))
    lastx, lasty = x, y

  interpolated.append(points[-1])
  return interpolated

def smooth(values):
  """ Takes a list of values and smooths them using a linear range average. The
  smoothing factor is taking from the config option smoothing_factor, which
  defaults to 5. In other words, at point i, the smoothed value y_i_s is
  sum(y_i-5, y_i+5) / 11.
  """
  if not values:
    return []

  factor = config.getint('bigbrother', 'smoothing_factor')
  assert factor >= 0
  smoothed = []

  for i in range(0, len(values)):
    start = max(0, i - factor)
    end = min(len(values), i + 1 + factor)
    run = values[start:end]
    avg = float(bigbrother.sum(run)) / len(run)
    smoothed.append(avg)
  return smoothed

