#!/usr/bin/python
#
# bigbrother
# http://snarfed.org/space/bigbrother
# Copyright 2003, 2004 Ryan Barrett <bigbrother@ryanb.org>
#
# File: html.py
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

"""
Generates the HTML report with various away message statistics.
"""

import os
import sys
import string
import time
import bigbrother
import graph


# constants
HEADER = """
<!-- This file was automatically generated by bigbrother %(version)s. See
%(url)s for more information. -->
""" % {'version': bigbrother.VERSION, 'url': bigbrother.URL}

FOOTER = """
<hr />
<p>
Generated by
<a href="%(url)s"> bigbrother </a> %(version)s,
written by <a href="http://ryan.barrett.name/"> Ryan Barrett</a>.
</p>
</body>
</html>
""" % {'version': bigbrother.VERSION, 'url': bigbrother.URL}


TABLE_PAGE = """
<html>
<head><title> %(title)s </title></head>
<body>
<h2> %(title)s </h2>
<p> %(description)s </p>
<table>
%(table)s
</table>
"""

INDEX_TEMPLATE = os.path.join(bigbrother.BASE_DIR, 'index.html.template')

TABLE_ROW = """
<tr><td> %(msg1)s <br /><img src="%(filename1)s" alt="%(filename1)s" /></td>
<td> %(msg2)s <br /><img src="%(filename2)s" alt="%(filename2)s" /></td></tr>
"""

# this is set from bigbrother.py
config = None


def table_page(rows, title, description):
  """ Generates an HTML page with a table that contains the given rows.
  Rows should be a list of tuples (of any size).
  """
  table = ''
  
  # generate table
  for row in rows:
    strings = [`x` for x in row]
    htmlrow = string.join(strings, ' </td><td> ')
    table += '<tr><td> ' + htmlrow + ' </td></tr>\n'

  return TABLE_PAGE % vars()


def write_index_page(filename, day_msgs, week_msgs, weekday_msgs,
                     weekend_msgs):
  html = open(INDEX_TEMPLATE).read() % {
    'day_dist_table': two_column_table(day_msgs, '%s.png'),
    'week_dist_table': two_column_table(week_msgs, 'week_%s.png'),
    'timestamp': time.ctime(),
    'weekday_layer_key': make_layer_key(weekday_msgs),
    'weekend_layer_key': make_layer_key(weekend_msgs),
    }
  write_html_page(filename, html)


def make_layer_key(msgs):
  layer_keys = ['<span style="color: #%06X"> %s </span>' % (color, msg)
                for msg, color in zip(msgs, graph.COLORS)]
  return '\n'.join(layer_keys)
  

def two_column_table(msgs, filename_spec):
  """ Generates the table rows, but not the <table> open and close tags, for a
  table of away messages and their graphs. Takes a list of away messages and a
  filename spec, and generates a table with the graphs for those messages, in
  order. For each away message, filename = filename_spec % msg; a TypeError is
  thrown if filename_spec does not contain '%s'. (This is actually an
  oversimplification; other transformations also occur.)
  """
  html = ''
  for i in range(0, len(msgs), 2):
    msg1 = msgs[i]
    filename1 = (filename_spec % msg1).replace(' ', '_')

    if i < len(msgs) - 1:
      msg2 = msgs[i + 1]
      filename2 = (filename_spec % msg2).replace(' ', '_')
    else:
      msg2 = filename2 = ''

    html += TABLE_ROW % vars()

  return html


def write_html_page(filename, html):
  """ Performs final sanity checks, adds "generated by" tags, and writes the
  HTML content to the given file. Note that if the file exists but doesn't
  contain "<!-- This file was generated by bigbrother", it won't be
  overwritten. Instead, this method will complain to stderr.
  """
  if os.access(filename, os.F_OK):
    # file exists; make sure it was generated by bigbrother
    curfile = open(filename, 'r')
    tag = string.join(HEADER.split()[:8]) 
    if curfile.read().find(tag) == -1:
      print >> sys.stderr, (filename + ' exists and was not generated by '
                            'bigbrother; cowardly refusing to overwrite...')
      return
    
  newfile = open(filename, 'w')
  newfile.write(HEADER + html + FOOTER)
  newfile.close()
