#!/usr/bin/python
#
# BigBrother
# http://snarfed.org/space/bigbrother
# Copyright 2003, 2004 Ryan Barrett <bigbrother@ryanb.org>
#
# File: html_unittest.py
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA

"""
Unit tests for html.py.
"""

import os
import sys
import StringIO
import deluxetest
import html
import bigbrother

TMPFILE = 'html_unittest.out'
TUPLES = [('asdf', 7, 'A'), (5.0, 'yuiop', 'B'), (3, 'jkl;', 'C')]
TABLE ="""<tr><td> 'asdf' </td><td> 7 </td><td> 'A' </td></tr>
<tr><td> 5.0 </td><td> 'yuiop' </td><td> 'B' </td></tr>
<tr><td> 3 </td><td> 'jkl;' </td><td> 'C' </td></tr>
"""



class htmlTest(deluxetest.TestCase):

  def setUp(self):
    bigbrother.set_defaults()
    # redirect stderr so that write_html_page's complaints don't show up
    sys.stderr = StringIO.StringIO() #open('/dev/null')
    # make sure the temp file doesn't already exist
    assert not os.access(TMPFILE, os.F_OK)

  def tearDown(self):
    # reconnect stderr
    sys.stderr = sys.__stderr__
    # delete the temp file
    if os.access(TMPFILE, os.F_OK):
      os.remove(TMPFILE)


  def test_table_page(self):
    title = 'fulff_title_here!'
    description = 'fulff_description_here!'
    table = TABLE

    golden = html.TABLE_PAGE % vars()
    generated = html.table_page(TUPLES, title, description)
    self.assertEqual(golden, generated)


  def test_two_column_table(self):
    self.assertRaises(TypeError, html.two_column_table, ['a', 'b'], 'x')
    self.assertRaises(TypeError, html.two_column_table, ['a', 'b'], 'x %d')
    self.assertRaises(TypeError, html.two_column_table, ['a', 'b'], 'x %s %s')

    msgs = []
    self.assertEqual('', html.two_column_table(msgs, '%s'))

    msgs = ['foo bar']
    golden = html.TABLE_ROW % {'msg1': 'foo bar', 'filename1': 'a_foo_bar_b.c',
                                'msg2': '', 'filename2': '' }
    self.assertEqual(golden, html.two_column_table(msgs, 'a %s b.c'))

    msgs = ['foo', 'bar']
    golden = html.TABLE_ROW % {'msg1': 'foo', 'filename1': '_foo_',
                                'msg2': 'bar', 'filename2': '_bar_' }
    self.assertEqual(golden, html.two_column_table(msgs, ' %s '))

    msgs = ['foo', 'bar', 'baz']
    golden = html.TABLE_ROW % {'msg1': 'foo', 'filename1': 'foo.png',
                                'msg2': 'bar', 'filename2': 'bar.png' }
    golden += html.TABLE_ROW % {'msg1': 'baz', 'filename1': 'baz.png',
                                 'msg2': '', 'filename2': '' }
    self.assertEqual(golden, html.two_column_table(msgs, '%s.png'))


  def test_write_html_page(self):
    # the file doesn't exist; we should write to it
    golden = 'asdf'
    html.write_html_page(TMPFILE, golden)
    self.assertEqual(html.HEADER + golden + html.FOOTER, open(TMPFILE).read())

    # now the file exists, but was written by us; we should still write to it
    golden = 'qwert'
    html.write_html_page(TMPFILE, golden)
    self.assertEqual(html.HEADER + golden + html.FOOTER, open(TMPFILE).read())

    # scribble over the file so that we don't recognize it; write_html_page
    # should complain to stderr and not write to it if we ask it to
    file = open(TMPFILE, 'w')
    contents = 'written by LittleSister'
    file.write(contents)
    file.close()

    html.write_html_page(TMPFILE, golden)
    file = open(TMPFILE, 'r')
    self.assertEqual(contents, file.read())
    file.close()
    

if __name__ == '__main__':
  deluxetest.main()
