/**************************************************************************
 * Linux EarPlug Plugin - implements the Live Audio JavaScript methods, 
 *                      and a few extra
 * Copyright (c) 1999 J.S.Kinnersley.
 * e-mail: jsk@amath.washington.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **************************************************************************/
#include <errno.h>
#include <fcntl.h>
#include <linux/soundcard.h>
#include <math.h>
#include "npapi.h"
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#define BUFFSIZE 1024
#define HDRSIZE 150
#define NCHANS 30
#define AU 0
#define AIFF 1
#define WAV 2
#define SND 3
#define MIDI 4
#define FIFO "/tmp/play_fifo"

#define UTF(u) (((double)((long)(u - 2147483647L - 1))) + 2147483648.0)
#define SUN_MAGIC 	0x2e736e64		/* Really '.snd' */
#define SUN_INV_MAGIC	0x646e732e		/* '.snd' upside-down */
#define DEC_MAGIC	0x2e736400		/* Really '\0ds.' (for DEC) */
#define DEC_INV_MAGIC	0x0064732e		/* '\0ds.' upside-down */

typedef unsigned int UI;

typedef struct _Channel {
  char filename[200];
  int type,rate,bits,num_chans;
  int loop,paused;
  int fd,midiReady;
  FILE * file;
  int lvol,rvol;
} Channel;

/*
** Define IMPLEMENT_EarPlug before including EarPlug.h to state that we're
** implementing the native methods of this plug-in here, and consequently
** need to access it's protected and private members.
*/
#define IMPLEMENT_EarPlug
#include "EarPlug.h"
#include "netscape_plugin_Plugin.h"

#define DEBUG 0
#define STOPPED 0
#define PAUSED 1
#define PLAYING 2

/*******************************************************************************
 * Instance state information about the plugin.
 *
 * PLUGIN DEVELOPERS:
 *	Use this struct to hold per-instance information that you'll
 *	need in the various functions in this file.
 ******************************************************************************/

int nplugins=0;
int childpid=0;
XColor fg,bg,bgd,bgl;
int setupColors=FALSE;
XFontStruct * fi;

typedef struct _PluginInstance
{
  //  NPWindow* fWindow;
  //  uint16 fMode;
  
  Window window;
  Display *display;
  Colormap colormap;
  GC gc;
  uint32 x, y;
  uint32 width, height;
  int timid;
  int pmode,ready,loop,lvol,rvol,slide;
  int autostart;
  int disabled;
  XColor console;
  Pixmap dbuff;
  int index;
  char url[200];
  char midifile[100],filename[200],ConCol[30];
  struct EarPlug *self;
} PluginInstance;

void childplay(PluginInstance* This);
void tellChild(const char * filename,PluginInstance* This);
void tellChild0(const char * filename,int loop,int chan,int lvol,int rvol,
		PluginInstance* This);
void Redraw(Widget w,XtPointer closure, XEvent *e);
void redraw(PluginInstance* This);
void push(PluginInstance* This,int b);
void buttonPress(Widget w,XtPointer closure, XEvent *e);
void buttonRelease(Widget w,XtPointer closure, XEvent *e);
void mouseDrag(Widget w,XtPointer closure, XEvent *e);
XColor Lighten(XColor col);
XColor Darken(XColor col);

char*
NPP_GetMIMEDescription(void)
{
  return("
audio/basic:ul,au:ULAW audio;
audio/x-aiff:aiff,aif:AIFF audio;
audio/aiff:aiff,aif:AIFF audio;
audio/x-wav:wav:Microsoft WAV file;
audio/wav:wav:Microsoft WAV file;
audio/snd:snd:SND file;
audio/midi:mid,midi,zip:MIDI file;
audio/x-midi:mid,midi,zip:MIDI file;
");
}

#define PLUGIN_NAME "Linux EarPlug "VERSION
#define PLUGIN_DESCRIPTION \
"<a href=http://www.amath.washington.edu/~jsk/EarPlug.html>"\
"EarPlug version "VERSION"</a> audio plug-in for Linux. "\
"Controlable from Java and JavaScript.<br>"\
"Allows simultaneous playing of up to 30 sound files. "\
"Written by Jonathan S. Kinnersley."

NPError
NPP_GetValue(void *future, NPPVariable variable, void *value)
{
    NPError err = NPERR_NO_ERROR;
    if (variable == NPPVpluginNameString)
		*((char **)value) = PLUGIN_NAME;
    else if (variable == NPPVpluginDescriptionString)
		*((char **)value) = PLUGIN_DESCRIPTION;
    else
		err = NPERR_GENERIC_ERROR;

    return err;
}

/*******************************************************************************
 * General Plug-in Calls
 ******************************************************************************/

/*
** NPP_Initialize is called when your DLL is being loaded to do any
** DLL-specific initialization.
*/
NPError
NPP_Initialize(void)
{
    return NPERR_NO_ERROR;
}

/*
** We'll keep a global execution environment around to make our life
** simpler.
*/
JRIEnv* env;

/*
** NPP_GetJavaClass is called during initialization to ask your plugin
** what its associated Java class is. If you don't have one, just return
** NULL. Otherwise, use the javah-generated "use_" function to both
** initialize your class and return it. If you can't find your class, an
** error will be signalled by "use_" and will cause the Navigator to
** complain to the user.
*/
jref
NPP_GetJavaClass(void)
{
  struct java_lang_Class* myClass;
  env = NPN_GetJavaEnv();
  if (env == NULL)
    return NULL;		/* Java disabled */
  //  fprintf(stderr,"GetJavaClass\n");
  myClass = use_EarPlug(env);
  use_netscape_plugin_Plugin(env);
  if (myClass == NULL) {
    /*
    ** If our class doesn't exist (the user hasn't installed it) then
    ** don't allow any of the Java stuff to happen.
    */
    env = NULL;
  }
  return myClass;
}

/*
** NPP_Shutdown is called when your DLL is being unloaded to do any
** DLL-specific shut-down. You should be a good citizen and declare that
** you're not using your java class any more. This allows java to unload
** it, freeing up memory.
*/
void
NPP_Shutdown(void)
{
  if (env) {
    unuse_EarPlug(env);
    unuse_netscape_plugin_Plugin(env);
  }
}

/*
** NPP_New is called when your plugin is instantiated (i.e. when an EMBED
** tag appears on a page).
*/
NPError 
NPP_New(NPMIMEType pluginType,
	NPP instance,
	uint16 mode,
	int16 argc,
	char* argn[],
	char* argv[],
	NPSavedData* saved)
{
  NPError result = NPERR_NO_ERROR;
  PluginInstance* This;
  char factString[60];
  int i,vol;
  
  //  debug("New\n");
  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  mkfifo(FIFO,0777);
  instance->pdata = NPN_MemAlloc(sizeof(PluginInstance));
  
  This = (PluginInstance*) instance->pdata;
  
  This->index=nplugins++;
  if(nplugins==NCHANS) nplugins=0;

  This->timid=-1;
  This->pmode=0;
  This->ready=0;
  This->loop=1;
  This->slide=0;
  This->autostart=1;
  This->disabled=0;
  This->lvol=50;
  This->rvol=50;
  This->dbuff=(Pixmap)NULL;
  strcpy(This->filename,"");
  strcpy(This->midifile,"");
  strcpy(This->url,"");
  strcpy(This->ConCol,"lightblue");
  This->self=(EarPlug*)NPN_GetJavaPeer(instance);
  This->display=(Display*)NULL;
  
  for(i=0;i<argc;i++) {
    if(!strcasecmp(argn[i],"LOOP")) {
      if(!strcasecmp(argv[i],"false")) This->loop=1;
      else if(!strcasecmp(argv[i],"true")) This->loop=-1;
      else This->loop=atoi(argv[i]);
    }
    if(!strcasecmp(argn[i],"DISABLED")) {This->disabled=1;}
    if(!strcasecmp(argn[i],"COLOR")) {strcpy(This->ConCol,argv[i]);}
    if(!strcasecmp(argn[i],"AUTOSTART")) {
      if(!strcasecmp(argv[i],"false")) This->autostart=0;
    }
    if(!strcasecmp(argn[i],"VOLUME")) {
      vol=atoi(argv[i]);
      if(vol<0) vol=0; if(vol>100) vol=100;
      This->lvol=vol;
      This->rvol=vol;
    }
  }
  
  if (This == NULL)
    return NPERR_OUT_OF_MEMORY_ERROR;
  /* mode is NP_EMBED, NP_FULL, or NP_BACKGROUND (see npapi.h) */
  //	  This->fWindow = NULL;
  //	  This->fMode = mode;
  This->window = 0;	    
  return result;
}


NPError 
NPP_Destroy(NPP instance, NPSavedData** save)
{
  PluginInstance* This;
  
  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  This = (PluginInstance*) instance->pdata;
  if(childpid>0) kill(childpid,SIGKILL);
  if(This->timid>0) kill(This->timid,SIGKILL);

  if(This->display!=(Display*)NULL) {
    //  XUnloadFont(This->display,fi->fid);
    XFreeGC(This->display,This->gc);
    XFreePixmap(This->display,This->dbuff);
  }
  if(strlen(This->midifile)>0) {
    //    fprintf(stderr,"Unlinking %s\n",This->midifile);
    unlink(This->midifile);
  }
  
  if (This != NULL) {
    NPN_MemFree(instance->pdata);
    instance->pdata = NULL;
  }
  
  return NPERR_NO_ERROR;
}

NPError 
NPP_SetWindow(NPP instance, NPWindow* window)
{
  NPError result = NPERR_NO_ERROR;
  PluginInstance* This;
  XColor tmp;
  
  //  debug("SetWindow1\n");
  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;
  //  debug("SetWindow2\n");
  
  This = (PluginInstance*) instance->pdata;
  
  /*
   * PLUGIN DEVELOPERS:
   *	Before setting window to point to the
   *	new window, you may wish to compare the new window
   *	info to the previous window (if any) to note window
   *	size changes, etc.
   */
  {
    Widget netscape_widget;
    XGCValues gcv;
    Pixmap mask;
    
    //    if(This->window!=(Window)NULL) {
    This->window = (Window) window->window;
    This->x = window->x;
    This->y = window->y;
    This->width = window->width;
    This->height = window->height;
    if(This->width>200) This->width=200;
    if(This->height>100) This->height=100;
    This->display = ((NPSetWindowCallbackStruct *)
		     window->ws_info)->display;
    This->colormap = ((NPSetWindowCallbackStruct *)
		      window->ws_info)->colormap;
    This->gc = XCreateGC(This->display,This->window, 
			 GCForeground|GCBackground, &gcv);
    
    netscape_widget = XtWindowToWidget(This->display, This->window);
    XtVaGetValues(netscape_widget, XtNbackground, &gcv.background,
		  XtNforeground, &gcv.foreground, 0);
    XtAddEventHandler(netscape_widget, ExposureMask, FALSE, 
		      (XtEventHandler)Redraw, This);
    XtAddEventHandler(netscape_widget, ButtonPressMask, FALSE, 
		      (XtEventHandler)buttonPress, This);
    XtAddEventHandler(netscape_widget, ButtonReleaseMask, FALSE, 
		      (XtEventHandler)buttonRelease, This);
    XtAddEventHandler(netscape_widget, ButtonMotionMask, FALSE, 
		      (XtEventHandler)mouseDrag, This);
    
    XParseColor(This->display,This->colormap,
		This->ConCol,&(This->console));
    XAllocColor(This->display,This->colormap,&(This->console));
    tmp=Darken(This->console);
    XAllocColor(This->display,This->colormap,&tmp);
    
    if(!setupColors) {
      setupColors=TRUE;
      fi=XLoadQueryFont(This->display,"7x10");
      XParseColor(This->display,This->colormap,"darkgrey",&bg);      
      XParseColor(This->display,This->colormap,"darkblue",&fg);
      bgd=Darken(bg);  bgl=Lighten(bg);
      XAllocColor(This->display,This->colormap,&bg); 
      XAllocColor(This->display,This->colormap,&fg);
      XAllocColor(This->display,This->colormap,&bgd); 
      XAllocColor(This->display,This->colormap,&bgl);
    }
    if(fi!=NULL) XSetFont(This->display,This->gc,fi->fid);

    redraw(This);
    //    XFlush(This->display);
  }
  //  This->fWindow = window;
  return result;
}

NPError 
NPP_NewStream(NPP instance,
	      NPMIMEType type,
	      NPStream *stream,
	      NPBool seekable,
	      uint16 *stype)
{
  PluginInstance* This;
  char * ext;
  
  //  debug("NewStream\n");
  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;
  
  *stype=NP_ASFILEONLY;
  
  return NPERR_NO_ERROR;
}

int32 STREAMBUFSIZE = 0X0FFFFFFF;
int32 
NPP_WriteReady(NPP instance, NPStream *stream) {
	/* Number of bytes ready to accept in NPP_Write() */
	return STREAMBUFSIZE;
}


int32 
NPP_Write(NPP instance, NPStream *stream, int32 offset, int32 len, void *buffer) {
	return len;		/* The number of bytes accepted */
}


NPError 
NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason) {
  //  debug("destroy stream");
  if (instance == NULL)return NPERR_INVALID_INSTANCE_ERROR;
  return NPERR_NO_ERROR;
}

void 
NPP_StreamAsFile(NPP instance, NPStream *stream, const char* fname)
{
  PluginInstance* This;
  char * ext;
  char opfile[100];
  time_t t0;

  //  debug("StreamAsFile %s...\n",fname);
  if(fname==NULL) return;
  This = (PluginInstance*) instance->pdata;
  strcpy(This->filename,fname);
  strcpy(This->url,stream->url);
  This->ready=1;
  set_EarPlug_ready(env,This->self,1);

  if(This->timid>0) {
    kill(This->timid,SIGKILL); 
    This->timid=0;}
  ext=(char*)(fname+strlen(fname));
  if(!strcasecmp(ext-4,".mid") || !strcasecmp(ext-5,".midi") ||
     !strcasecmp(ext-4,".zip")	 ) {
    if(strlen(This->midifile)>0) {
      //      fprintf(stderr,"Unlinking %s\n",This->midifile);
      unlink(This->midifile);
    }
    This->timid=fork();
    if(This->timid==0) {
      sprintf(opfile,"-o%s%d%s","/tmp/earPlug",getpid(),".wavm");
      execlp("timidity","timidity","-Ow8M",opfile,
	     "-idqqqqqqq",fname,0);
      kill(getpid(),SIGKILL);
    }
    //    else waitpid(This->timid,(int*)0,WUNTRACED);
    sprintf(opfile,"%s%d%s","/tmp/earPlug",This->timid,".wavm");
    strcpy(This->midifile,opfile);
    strcpy(This->filename,opfile);
    //    if(This->timid>0) kill(This->timid,SIGKILL);
  }
  if(This->autostart==1) PPlay(This);
}

PStop(PluginInstance* This) {
  push(This,STOPPED);
  tellChild("stop",This);
  This->pmode=STOPPED;
  set_EarPlug_pmode(env,This->self,STOPPED);
}

PPause(PluginInstance* This) {
  push(This,PAUSED);
  tellChild("pause",This);
  This->pmode=PAUSED;
  set_EarPlug_pmode(env,This->self,PAUSED);
}

PPlay(PluginInstance* This) {
  push(This,PLAYING);
  if(This->ready==1) tellChild(This->filename,This);
  else This->autostart=1;
  //  drawButton(This,3,1);
  //  drawButton(This,4,1);
  //  redraw(This);
  //  XFlush(This->display);
  setDisplayString(This,This->url);
  This->pmode=PLAYING;
  set_EarPlug_pmode(env,This->self,PLAYING);
}

void push(PluginInstance* This,int button) {
  if(This->window==(Window)NULL) return;
  drawButton(This,button,1);
  if(button!=0) drawButton(This,0,0);
  if(button!=1) drawButton(This,1,0);
  if(button!=2) drawButton(This,2,0);
  redraw(This);
  //  XFlush(This->display);
}

void 
NPP_Print(NPP instance, NPPrint* printInfo)
{
  if(printInfo == NULL)
    return;
  
  if (instance != NULL) {
    PluginInstance* This = (PluginInstance*) instance->pdata;
    
    if (printInfo->mode == NP_FULL) {
       void* platformPrint =
	printInfo->print.fullPrint.platformPrint;
      NPBool printOne =
	printInfo->print.fullPrint.printOne;
      
      /* Do the default*/
      printInfo->print.fullPrint.pluginPrinted = FALSE;
    }
    else {	/* If not fullscreen, we must be embedded */
      NPWindow* printWindow =
	&(printInfo->print.embedPrint.window);
      void* platformPrint =
	printInfo->print.embedPrint.platformPrint;
    }
  }
}

/*******************************************************************************
 * Define the Java native methods
 ******************************************************************************/

setVol(PluginInstance* This,int vol) {
  int tvr=vol/256,tvl=vol%256;
  if(childpid==0 || waitpid(childpid,(int*)0,WNOHANG)!=0) return;
  This->lvol=tvl;
  This->rvol=tvr;
  drawButton(This,3,0);
  drawButton(This,4,0);
  redraw(This);
  //  XFlush(This->display);
  tellChild("setvol",This);
}

JRI_PUBLIC_API(void)
     native_EarPlug_disableConsole(JRIEnv* env, struct EarPlug* self) {
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  This->disabled=1;
}
JRI_PUBLIC_API(void)
     native_EarPlug_enableConsole(JRIEnv* env, struct EarPlug* self) {
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  This->disabled=0;
}

JRI_PUBLIC_API(void)
native_EarPlug_nSetVol(JRIEnv* env, struct EarPlug* self,
				   jint vol)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  setVol(This,vol);
}

JRI_PUBLIC_API(void)
native_EarPlug_nStop(JRIEnv* env, struct EarPlug* self)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  PStop(This);
}

JRI_PUBLIC_API(void)
native_EarPlug_stopAll(JRIEnv* env, struct EarPlug* self)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  tellChild0("stop",1,-1,100,100,This);
}

JRI_PUBLIC_API(void)
native_EarPlug_pauseAll(JRIEnv* env, struct EarPlug* self)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  tellChild0("pause",1,-1,100,100,This);
}

JRI_PUBLIC_API(void)
native_EarPlug_playAll(JRIEnv* env, struct EarPlug* self)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  tellChild0("NULL",1,-1,100,100,This);
}

JRI_PUBLIC_API(void)
native_EarPlug_pause(JRIEnv* env, struct EarPlug* self)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  PPause(This);
}

JRI_PUBLIC_API(void)
native_EarPlug_nPlay(JRIEnv* env, struct EarPlug* self, jint loop,
		   struct java_lang_String * s)
{
  NPP instance=(NPP)netscape_plugin_Plugin_getPeer(env, self);
  PluginInstance* This = (PluginInstance*) instance->pdata;
  const char* chars = JRI_GetStringUTFChars(env, s);
  char * str=This->url;
  //  if(strchr(str,'/')!=NULL) str=strchr(str,'/');
  This->loop=loop;
  if(!strcasecmp(chars,"NULL")) {
    if(strlen(This->url)!=0) PPlay(This);
    return;
  }
  if(strlen(str)==0 || strcmpEnd(str,chars)) {
    strcpy(This->url,chars);
    set_EarPlug_ready(env,This->self,0);
    This->ready=0;
    This->autostart=1;
    //    fprintf(stderr,"GetUrl %s\n",chars);
    NPN_GetURL(instance,chars, NULL);
  } else {
    //    fprintf(stderr,"Play\n");
    PPlay(This);
  }
}

int strcmpEnd(char *str1,char *str2) {
  int l1=strlen(str1), l2=strlen(str2);
  if(l1==l2) return strcmp(str1,str2);
  else if(l1<l2) return strcmp(str1,str2+l2-l1);
  else return strcmp(str2,str1+l1-l2);
}

/*******************************************************************************
// NPP_URLNotify:
// Notifies the instance of the completion of a URL request. 
// 
// NPP_URLNotify is called when Netscape completes a NPN_GetURLNotify or
// NPN_PostURLNotify request, to inform the plug-in that the request,
// identified by url, has completed for the reason specified by reason. The most
// common reason code is NPRES_DONE, indicating simply that the request
// completed normally. Other possible reason codes are NPRES_USER_BREAK,
// indicating that the request was halted due to a user action (for example,
// clicking the "Stop" button), and NPRES_NETWORK_ERR, indicating that the
// request could not be completed (for example, because the URL could not be
// found). The complete list of reason codes is found in npapi.h. 
// 
// The parameter notifyData is the same plug-in-private value passed as an
// argument to the corresponding NPN_GetURLNotify or NPN_PostURLNotify
// call, and can be used by your plug-in to uniquely identify the request. 
 ******************************************************************************/

void
NPP_URLNotify(NPP instance, const char* url, NPReason reason, void* notifyData)
{
}

void mouseDrag(Widget wid, XtPointer closure, XEvent *e) {
  PluginInstance* This = (PluginInstance*)closure;
  static int drawing=0;
  int x=e->xbutton.x;
  int xl=This->width/6,xr=This->width*5/6;
  int voll=This->lvol, volr=This->rvol;
  int nvoll=voll,nvolr=volr;
  int nvol=(x-xl)*100/(xr-xl);
  if(!This->slide || drawing) return;
  drawing=1;
  if(nvol<1) nvol=1;
  if(nvol>100) nvol=100;
  if(This->slide & 1) nvoll=nvol;
  if(This->slide & 2) nvolr=nvol;
  setVol(This,nvoll+nvolr*256);
  drawButton(This,3,0);
  drawButton(This,4,0);
  redraw(This);
  //  XFlush(This->display);
  drawing=0;
}

void buttonPress(Widget wid, XtPointer closure, XEvent *e) {
  PluginInstance* This = (PluginInstance*)closure;
  int width=This->width;
  int height=This->height;
  int button, xll,xlr,xrl,xrr;
  int y1=height/3, y2=height*7/12, y3=height*2/3, y4=height*11/12;
  int y5=(y4-y3)/3+y3;
  int y6=(y4-y3)*2/3+y3;
  int x=e->xbutton.x, y=e->xbutton.y;
  if(y > y1 && y < y2) {
    button=x*3/(width+1);
    if(This->disabled==0) {
      if(button==2) {This->loop=-1; PPlay(This);}
      if(!(childpid==0 || waitpid(childpid,(int*)0,WNOHANG)!=0)) {
	if(button==0) PStop(This);
	if(button==1) PPause(This);
      }
      push(This,button);
    }
  } else if(y > y3 && y < y4) {
    if(y>y5 && y<y6) This->slide=3;
    else if(y>y3 && y<y5) {
      xll=width/6+(4*width/6)*This->lvol/100-width/10;
      xlr=xll+width*2/3;
      if(x>xll && x<xlr) This->slide=1;
    } else {
      xrl=width/6+(4*width/6)*This->rvol/100-width/10;
      xrr=xrl+width*2/3;
      if(x>xrl && x<xrr) This->slide=2;
    }
    mouseDrag(wid,closure,e);
  }
}

void buttonRelease(Widget wid, XtPointer closure, XEvent *e) {
  PluginInstance* This = (PluginInstance*)closure;
  This->slide=0;
}


void Redraw(Widget wid, XtPointer closure, XEvent *e)
{
  redraw((PluginInstance*)closure);
}

void redraw(PluginInstance* This)
{
  GC gc; Display * display; Window window;
  int w,h;
  char str[]="EarPlug";
  Drawable d;

  gc = This->gc;
  display=This->display; window=This->window;
  w=This->width; h=This->height;

  if(This->window==(Window)NULL) return;
  if(This->dbuff==(Pixmap)NULL) {
    This->dbuff=XCreatePixmap(display,window,
			      w,h,DefaultDepth(display,0));
    d=This->dbuff;
    
    XSetForeground(display,gc,This->console.pixel);
    XFillRectangle(display,d,gc,0,0,w,h);

    XSetForeground(display,gc,0);
    XDrawRectangle(display, d, gc,0, 0, w-1, h-1);
    drawButton(This,0,0);
    drawButton(This,1,0);
    drawButton(This,2,0);
    drawButton(This,3,0);
    XDrawString(This->display, This->dbuff,This->gc,4,h*2/3+h/8-2,"L",1);
    drawButton(This,4,0);
    XDrawString(This->display, This->dbuff,This->gc,4,h*2/3+h/4+2,"R",1);
    setDisplayString(This,This->url);
  }
  XCopyArea(display,This->dbuff,window,gc,0,0,w,h,0,0);
  //  XFlush(This->display);
}

setDisplayString(PluginInstance* This,char * str) {
  char strw[]="EarPlug is waiting...";
  if(This->window==(Window)NULL) return;
  XSetForeground(This->display,This->gc,WhitePixel(This->display,0));
  XFillRectangle(This->display,This->dbuff,This->gc,2,2,This->width-4,12);
  XSetForeground(This->display,This->gc,BlackPixel(This->display,0));    
  if(strlen(str)!=0) {
    while(strchr(str,'/')!=NULL) str=strchr(str,'/')+1;
    XDrawString(This->display, This->dbuff,This->gc,4,12,str,strlen(str));
  } else {
    XDrawString(This->display, This->dbuff,This->gc,4,12,strw,strlen(strw));
  }
  redraw(This);
}

unsigned short min(int x1,int x2) { 
  if(x1<x2) return (unsigned short) x1; 
  return (unsigned short) x2; }

XColor Lighten(XColor col) {
  XColor ret;
  ret.red=min(65535,1.3*col.red); 
  ret.green=min(65535,1.3*col.green); 
  ret.blue=min(65535,1.3*col.blue); 
  return ret;
}

XColor Darken(XColor col) {
  XColor ret;
  ret.red=col.red*0.7; 
  ret.green=col.green*0.7; 
  ret.blue=col.blue*0.7; 
  return ret;
}

drawButton(PluginInstance* This,int button,int state) {
  
  Drawable window; Display *display; GC gc; Colormap cm;
  int x,y,w,h,width,height,hedge,vedge,xs;
  XPoint p[5];

  if(This->display==(Display*)NULL) return;
  gc      = This->gc;       cm     = This->colormap;
  display = This->display;  window = This->dbuff;
  width   = This->width;    height = This->height;
  hedge=width/48; vedge=height/48;
  if(button<3) {
    x=width*(1+button*8)/24; y=height/3; w=width/4; h=height/4;
  } else {
    w=width*4/6; h=height/8; x=width/6; y=height*2/3+(button-3)*h;
  }	
  
  if(button<3) {
    if(state==0) XSetForeground(display,gc,bgl.pixel);
    if(state==1) XSetForeground(display,gc,bgd.pixel);
    p[0].x=x; p[1].x=w; p[2].x=-hedge; p[3].x=-w+2*hedge; p[4].x=-hedge;
    p[0].y=y; p[1].y=0; p[2].y=vedge;  p[3].y=h-2*vedge;  p[4].y=vedge;
    XFillPolygon(display, window, gc, p, 5, Convex, CoordModePrevious);
    if(state==1) XSetForeground(display,gc,bgl.pixel);
    if(state==0) XSetForeground(display,gc,bgd.pixel);
    p[0].x=x;   p[1].x=w;  p[2].x=0; p[3].x=-hedge; p[4].x=-w+2*hedge;
    p[0].y=y+h; p[1].y=0; p[2].y=-h; p[3].y=vedge;  p[4].y=h-2*vedge;
    XFillPolygon(display, window, gc, p, 5, Convex, CoordModePrevious);
    if(state==0) XSetForeground(display,gc,bg.pixel);
    if(state==1) XSetForeground(display,gc,fg.pixel);
    XFillRectangle(display, window, gc, 
		   x+1+hedge, y+1+vedge, w-2*hedge-1, h-2*vedge-1);
  } else {  
    XSetForeground(display,gc,This->console.pixel);
    XFillRectangle(display, window, gc, x-w/10, y, w+w/5, h);	
    XSetForeground(display,gc,bgd.pixel);
    XFillRectangle(display, window, gc, x, y+h/3, w-1, h/3-1);	
    if(button==3) xs=This->lvol;
    if(button==4) xs=This->rvol;
    xs=x+w*xs/100-w/10;
    XSetForeground(display,gc,fg.pixel);
    XFillRectangle(display, window, gc, xs, y+1, w/5, h-1);	
  }

  if(state==0) XSetForeground(display,gc,fg.pixel);
  if(state==1) XSetForeground(display,gc,bg.pixel);
  switch(button) {
  case 0:
    XFillRectangle(display, window, gc, x+w/4, y+1+h/4, w/2, h/2);	
    break;
  case 1:
    XFillRectangle(display, window, gc, x+w/4, y+h/4+1, w/6, h/2);	
    XFillRectangle(display, window, gc, x+w*7/12, y+h/4+1, w/6, h/2);	
    break;
  case 2:
    p[0].x=x+w/3; p[1].x=w/3; p[2].x=-w/3;
    p[0].y=y+h/4+1; p[1].y=h/4; p[2].y=h/4;
    XFillPolygon(display, window, gc,p,3,Convex,CoordModePrevious);
    break;
  }
}

void ul2linSetup(int * ul2lin)
{
  static int exp_lut[8] = { 0, 132, 396, 924, 1980, 4092, 8316, 16764 };
  int sign, exponent, mantissa, res,ulb;

  for(ulb=0;ulb<256;ulb++) {
    sign = ( ulb & 0x80 );
    exponent = ( ulb >> 4 ) & 0x07;
    mantissa = ulb & 0x0F;
    res = exp_lut[exponent] + ( mantissa << ( exponent + 3 ) );
    if ( sign != 0 ) res = -res;
    ul2lin[ulb]=res;
  }
}

int readExtended(FILE *file)
{
  char buff[10];
  int exp;
  unsigned long himant;

  fread(buff,1,10,file);
  exp = (((buff[0] & 0x7F) << 8) | (buff[1] & 0xFF))-16414;
  himant    =    ((unsigned long)(buff[2] & 0xFF) << 24)
	|    ((unsigned long)(buff[3] & 0xFF) << 16)
	|    ((unsigned long)(buff[4] & 0xFF) << 8)
	|    ((unsigned long)(buff[5] & 0xFF));
  return (int)ldexp(UTF(himant),exp);
}

int readShort(FILE *file) {
  char buff[2];
  fread(buff,1,2,file);
  return (0xff & (UI)buff[1]) | (0xff00 & (UI)(buff[0])<<8) ;
}

int readShortR(FILE *file) {
  char buff[2];
  fread(buff,1,2,file);
  return (0xff & (UI)buff[0]) | (0xff00 & (UI)(buff[1])<<8) ;
}

int readLong(FILE *file) {
  char buff[4];
  fread(buff,1,4,file);
  return (0xff000000 & (UI)buff[0]<<24) 
	| (0xff0000 & (UI)(buff[1])<<16)
	| (0xff00 & (UI)buff[2]<<8) 
	| (0xff & (UI)(buff[3]));
}

int readLongR(FILE *file) {
  char buff[4];
  fread(buff,1,4,file);
  return (0xff000000 & (UI)buff[3]<<24) 
	| (0xff0000 & (UI)(buff[2])<<16)
	| (0xff00 & (UI)buff[1]<<8) 
	| (0xff & (UI)(buff[0]));
}

int readHeader(Channel *chan) {
  char header[HDRSIZE];
  int type,fd,hdrsize,magic,t0;
  type=chan->type;
  if(type==AU || type==SND) {
	chan->bits=8;
	magic=readLong(chan->file);
	if(magic!=SUN_MAGIC && magic!=SUN_INV_MAGIC &&
	   magic!=DEC_MAGIC && magic!=DEC_INV_MAGIC) {
	  chan->rate=8000;
	  chan->num_chans=1;
	  //	  debug("Can't find AU magic word\n");
	} else {
	  hdrsize=readLong(chan->file);
	  readLong(chan->file); //data_size;
	  readLong(chan->file); //encoding;
	  chan->rate=readLong(chan->file);
	  chan->num_chans=readLong(chan->file);
	  fread(header,1,hdrsize-24,chan->file); //info
	}
  }
  if(type==AIFF) {
    while(fread(header,1,4,chan->file)==4 && strncmp(header,"COMM",4));
	readLong(chan->file); //chunk length
	chan->num_chans=readShort(chan->file);
	readLong(chan->file); //frames
	chan->bits=readShort(chan->file);
	chan->rate=readExtended(chan->file);
	fread(header,1,40,chan->file);
	//    while(fread(header,1,4,chan->file)==4 && strncmp(header,"SSND",4));
  }
  if(type==WAV || type==MIDI) {
	fread(header,1,8,chan->file);
	fread(header,1,8,chan->file);
	header[8]=0;
	if(!strcmp(header,"WAVELIST")) fread(header,1,44,chan->file);
	fread(header,1,6,chan->file); //un-needed stuff
	chan->num_chans=readShortR(chan->file);
	chan->rate=readLongR(chan->file);
	fread(header,1,6,chan->file); //un-needed stuff
	chan->bits=readShortR(chan->file);
	fread(header,1,40,chan->file);
	//    while(fread(header,1,4,chan->file)==4 && strncmp(header,"data",4));
  }
  //  debug("%d chanels %d bits, rate=%d\n",chan->num_chans,chan->bits,chan->rate);
}

int readloop(Channel *chan,int*buf) {
  int cnt,i,paused,type;
  static int ul2lin[256];
  static int setup=0;
  char bufc[12*BUFFSIZE];
  char dummy[HDRSIZE];
  int bufrd;
  int *loop,*fd;
  char *fname;
  float fac;

  paused=chan->paused;
  type=chan->type;
  fd=&(chan->fd);
  loop=&(chan->loop);
  fname=chan->filename;

  if(setup==0) {setup=1; ul2linSetup(ul2lin);}
  if(*loop==0) return 0;
  if(type==AU || type==AIFF) memset(bufc,127,BUFFSIZE);
  if(type==WAV || type==MIDI) memset(bufc,0,BUFFSIZE);
  if(paused==1) { return BUFFSIZE; }

  if(type==MIDI && chan->midiReady++<12) return BUFFSIZE;

  if(*fd==0) { 
    *fd=open(fname,O_RDONLY,0); 
    chan->file=fdopen(*fd,"r");
    readHeader(chan);
  }

  bufrd=2*(int)(BUFFSIZE*(chan->rate)*(chan->bits)
				*(chan->num_chans)/(8000*16));

  cnt=fread(bufc,1,bufrd,chan->file);
  if(cnt<bufrd) {
	rewind(chan->file);
    if(*loop>0) *loop=*loop-1;
    if(*loop!=0) {
      readHeader(chan);
	  fread(bufc+cnt,1,bufrd-cnt,chan->file);
    }
  }
  if(type==AU) 
	for(i=0;i<BUFFSIZE;i++) buf[i]=ul2lin[(~bufc[i*bufrd/BUFFSIZE])&0xff];
  else if(type==AIFF)
	for(i=0;i<BUFFSIZE;i++) buf[i]=(unsigned int)
	      			 bufc[i*bufrd/BUFFSIZE]<<8;
  else if(type==WAV || type==SND || type==MIDI) {
	if(chan->bits==8) {
	  for(i=0;i<BUFFSIZE;i++) buf[i]= ((0xff00 &(unsigned int)
				 bufc[i*bufrd/BUFFSIZE]<<8) -0x8000);
  /* WAV data is unsigned 8-bit - need to convert it to signed 16 bit */
	} else {
	  for(i=0;i<BUFFSIZE;i++) buf[i]=
	       ((0xffffff00&(int)bufc[1+2*(int)(i*bufrd/(2*BUFFSIZE))]<<8) |
	       (0xff&(UI)bufc[2*(int)(i*bufrd/(2*BUFFSIZE))]));
	}
  }
  return BUFFSIZE;
}

void tellChild(const char * filename,PluginInstance* This) {
  tellChild0(filename,This->loop,This->index,This->lvol,This->rvol,This);
}

void tellChild0(const char * filename,int loop,int chan,int lvol,int rvol,
		PluginInstance* This) {
  int fres,res,fid,reopen=0;
  static int fifoid;
  char buf[400];
  sprintf(buf,"%d %d %d %d %s\n",loop,chan,lvol,rvol,filename);
  if(childpid==0 || waitpid(childpid,(int*)0,WNOHANG)!=0) reopen=1;

  if(reopen==1) {
    if(childpid>0) kill(childpid,SIGKILL);
    fid=fork();
    if(fid==0) { 
      childplay(This); 
      kill(getpid(),SIGKILL);
    } else if(fid>0) {
      childpid=fid;
      fifoid=open(FIFO,O_WRONLY);
      if(fifoid<0) {
	//	fprintf(stderr,"Can't connect with audio device\n"); 
	return;
      }
    }
  }
  if(fifoid>0) res=write(fifoid,buf,strlen(buf));
  //  debug("%d bytes written to child\n",res);
}

void childplay(PluginInstance* This) {
  static char *audiodev = "/dev/audio";
  static int frag=10,bits=16,stereo=1,speed=8000,vol=60*257;   
  /* fragment size is 2^frag */
  static char bufl[4*BUFFSIZE];
  static char buff[2000];
  static int buft[NCHANS][BUFFSIZE];
  char filename[150];
  int fifoid,fid,loop,i,ff,ft,res,nc,done,lvol,rvol;
  int maxv=0x4fff;
  char * str;
  int lsum,rsum;
  Channel chans[NCHANS];
  char * ext;
  audio_buf_info AI;
  int audiofd=-1,attempt=0;
  
  while(audiofd<0 && attempt++<20) {
    audiofd = open(audiodev, O_WRONLY);
    ioctl(audiofd,SNDCTL_DSP_STEREO,&stereo);
    ioctl(audiofd,SNDCTL_DSP_SPEED,&speed);
    ioctl(audiofd, SOUND_PCM_WRITE_BITS,&bits);
    ioctl(audiofd, SOUND_PCM_SETFRAGMENT,&frag);
    ioctl(audiofd, SOUND_MIXER_WRITE_VOLUME,&vol);
  }
  if(audiofd<0) {
    //    fprintf(stderr,"Problem opening sound device"); 
    kill(getpid(),SIGKILL);
  }
  
  for(i=0;i<NCHANS;i++) chans[i].loop=chans[i].fd=
	      chans[i].paused=chans[i].midiReady=0;
  done=0;
  
  fifoid=open(FIFO,O_RDONLY | O_NONBLOCK);
  while((res=read(fifoid,buff,2000))>0 || done==0) {
    if(res>0) {
      str=buff;
      while(sscanf(str,"%d %d %d %d %s\n",&loop,&nc,
		   &lvol,&rvol,filename)>0 && (str-buff<res)) {
	if(lvol<0) lvol=0;
	if(lvol>100) lvol=100;
	if(rvol<0) rvol=0;
	if(rvol>100) rvol=100;
	if(!strcmp(filename,"pause")) {
	  if(nc>-1) chans[nc].paused=1;
	  else {for(i=0;i<NCHANS;i++) chans[i].paused=1;}
	} else if(!strcmp(filename,"stop")) {
	  if(nc>-1) chans[nc].loop=0;
	  else {for(i=0;i<NCHANS;i++) chans[i].loop=0;}
	} else if(!strcmp(filename,"setvol")) {
	  if(nc>-1) {
	    chans[nc].lvol=lvol*lvol;
	    chans[nc].rvol=rvol*rvol;}
	  else {for(i=0;i<NCHANS;i++) {
	    chans[i].lvol=lvol*lvol;
	    chans[i].rvol=rvol*rvol;}}
	} else {
	  //	  if(nc==-1) { nc=nplugins; while(chans[nc++].loop!=0);}
	  if(nc==-1) { 
	    for(i=0;i<NCHANS;i++) chans[i].paused=0;
	  } else {
	    if(strlen(filename)>0) {
	      ext=strrchr(filename,'.');
	      
	      if(!strcasecmp(ext,".au") || !strcasecmp(ext,".ul"))
		chans[nc].type=AU; 
	      if(!strcasecmp(ext,".aif") || !strcasecmp(ext,".aiff"))
		chans[nc].type=AIFF; 
	      if(!strcasecmp(ext,".wav")) chans[nc].type=WAV; 
	      if(!strcasecmp(ext,".snd")) chans[nc].type=SND; 
	      if(!strcasecmp(ext,".wavm")) chans[nc].type=MIDI; 
	      
	      strcpy(chans[nc].filename,filename);
	      if(chans[nc].paused==0) chans[nc].fd=0;
	    }
	    chans[nc].paused=0;
	    if(loop>-2) chans[nc].loop=loop;
	    chans[nc].lvol=lvol*lvol;
	    chans[nc].rvol=rvol*rvol;
	  }
	}
	str = strchr(str,'\n')+1;
	//	debug("%d %d %d %s\n",loop,rvol,lvol,filename);
      }
    }
    
    ioctl(audiofd,SNDCTL_DSP_GETOSPACE,&AI);
    ft=AI.fragstotal;
    ff=AI.fragments;
    while(ft-ff>2) {
      usleep(10000);
      ioctl(audiofd,SNDCTL_DSP_GETOSPACE,&AI);
      ft=AI.fragstotal;
      ff=AI.fragments;
    }
    done=1;
    for(nc=0;nc<NCHANS;nc++) {
      res=readloop(&chans[nc],buft[nc]);
      if(res>0) done=0;
    }
    if(done==0) {
      for(i=0;i<BUFFSIZE;i++) {
	lsum=rsum=0;
	for(nc=0;nc<NCHANS;nc++)
	  if(chans[nc].loop!=0 && chans[nc].paused==0
	     && (chans[nc].type!=MIDI || chans[nc].midiReady>12) ) {
	    lsum+=buft[nc][i]*(chans[nc].lvol)/10000;
	    rsum+=buft[nc][i]*(chans[nc].rvol)/10000;
	  }
	if(lsum>maxv) lsum=maxv; if(lsum<-maxv) lsum=-maxv;
	bufl[4*i]=lsum;  bufl[4*i+1]=lsum>>8;
	if(rsum>maxv) rsum=maxv; if(rsum<-maxv) rsum=-maxv;
	bufl[4*i+2]=rsum;  bufl[4*i+3]=rsum>>8;
      }
      write(audiofd, bufl, 4*BUFFSIZE);  
    }
  }
  //  ioctl(audiofd,SNDCTL_DSP_SYNC);
  ioctl(audiofd,SNDCTL_DSP_RESET);
  close(audiofd);
  for(i=0;i<NCHANS;i++) {
    free(chans[i].filename); 
    free(buft[i]);
  }
  if(This->display!=(Display*)NULL) {
    XFreeGC(This->display,This->gc);
    XFreePixmap(This->display,This->dbuff);
  }
  kill(getpid(),SIGKILL);
  /**** IMPORTANT Use kill, and not exit which does not 
	seem to stop the child process *****/
}

//debug(char* str) {
//  if(DEBUG) fprintf(stderr,str);
//}
