/* inchannels.cpp - MuSE
   Copyright (c)2000-2001 Denis Roio aka jaromil
   see COPYING for licensing stuff

   different classes for different IN channels
   they are instantiated and used by the Stream_mixer class (jmixer.cpp)
*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>

#include "inchannels.h"

#ifdef HAVE_XAUDIO
/* xaudio lib */
#include "xaudio/include/stream_input.h"
#include "xaudio/include/file_input.h"
#include "xaudio/include/mpeg_codec.h"
#endif

extern "C" {
#include "mpeg.h"
}

static short buffo[32768];

/* ----- Parent Class Channel ----- */

bool Channel::set_mixer() {
  if(samplerate==44100) /* stereo */
    if(channels==2)
      mixxx = mixxx_stereo_44;
    else
      mixxx = mixxx_mono_44;
  else if(samplerate==22050)
    if(channels==2)
      mixxx = mixxx_stereo_22;
    else
      mixxx = mixxx_mono_22;
  else {
    fprintf(stderr, "HEY! i can mix only 44Khz or 22Khz streams\n");
    return(false);
  }
  return(true);
}

float Channel::upd_time() {
  long int secs;
  float res;

  update = false;

  res = (float)framepos/(float)frametot;

  if((res-time.f)>UPDATE_INTERVAL) {
    time.f = res;
    secs = framepos/fps;
    
    if(secs>3600) {
      time.h = (int) secs / 3600;
      secs -= time.h*3600;
    }
    if(secs>60) {
      time.m = (int) secs / 60;
      secs -= time.m*60;
    }
    time.s = (int) secs;
    update = true;
  }
  return(res);
}

#ifdef HAVE_XAUDIO
/* ----- XAudio MP3 CHANNEL ----- */

XaudioChannel::XaudioChannel() {
  
  type = MP3CHAN;

  decoder_new(&decoder);

   /* register mpeg audio codec */
    {
        XA_CodecModule module;

        mpeg_codec_module_register(&module);
        decoder_codec_module_register(decoder, &module);
    }

    /* register the file input module */
    {
        XA_InputModule module;

        file_input_module_register(&module);
        decoder_input_module_register(decoder, &module);
    }

    /* register the stream input module */
    {
        XA_InputModule module;
	stream_input_module_register(&module);
	decoder_input_module_register(decoder, &module);
    }

    input = decoder->input;
    status = decoder->status;
    buffer = decoder->output_buffer;
    
    speed = 100;
    volume = 1.0;
    playmode = PLAY;
    opened = false;
    on = false;
    update = false;
}

XaudioChannel::~XaudioChannel() {
  decoder_input_delete(decoder);
  decoder_delete(decoder);
}

int XaudioChannel::set(char *file) {
  int res;

  /* if a channel was already loaded then is automatically
     closed and deleted by decoder_input_new() */

  on = false;

  res = decoder_input_new(decoder,file,XA_DECODER_INPUT_AUTOSELECT);

  opened = false;

  if(res != XA_SUCCESS)
    return(false);

  res = decoder_input_open(decoder);
  if(res != XA_SUCCESS)
    return(false);

  opened = true;
  time.h = time.m = time.s = 0;
  time.f = position = 0.0;

  samplerate = status->info.frequency;
  channels = status->info.nb_channels;

  if(!(samplerate>0&&channels>0)) {
    fprintf(stderr, "XAudio: junk in the header! trying to guess samplerate and channels\n");
    res = decoder_decode(decoder,NULL);
    if(res != XA_SUCCESS) {
      fprintf(stderr, "XAudio: %s is an invalid mp3 file!\n",file);
      opened = false;
      return(opened);
    }
    samplerate = buffer->sample_rate;
    channels = buffer->nb_channels;
  }
  
  if(!set_mixer())
    opened = false;
  
  return(opened);
}

float XaudioChannel::mix(int *mixpcm) {
  float res;

  res = get_audio();
  if(res>1.0) return(res);

  in_smp = (*mixxx)(&mixpcm[0],&buffo[0],in_smp,volume);

  /* update time */
  time.h = status->timecode.h;
  time.m = status->timecode.m;
  time.s = status->timecode.s;
  time.f = status->timecode.f;

  return(res);
}

float XaudioChannel::get_audio() {
  int res;
  res = decoder_decode(decoder,NULL);

  if(res == XA_SUCCESS) {
    in_smp = resample(buffer->pcm_samples,buffer->nb_samples,buffo,speed);
    return(status->position);
  }

  if(res == XA_ERROR_INPUT_EOS) {/* the input stream ended */
    return(2.0); /* should do next thing: loop, nextsong or stop */
  }
  
  /*  fprintf(stderr,"XA: i have a bad feeling\n"); */
  return(3.0);
}

bool XaudioChannel::play() {
  if(!on) {
    if(time.f!=position) {
      if(decoder_input_seek_to_position(decoder,position)!=XA_SUCCESS)
	return(false);
      time.f = position;
    }
    on = true;
  }
  
  return(on);
}

bool XaudioChannel::stop() {
  on = false;

  position = 0.0;

  return(true);
}

bool XaudioChannel::pos(float pos) {
  if(on) {
    if(decoder_input_seek_to_position(decoder,pos)!=XA_SUCCESS)
      return(false);
    time.f = position = pos;
  } else {
    position = pos;
  }

  return(true);
}
#endif

#ifdef HAVE_VORBIS
/* ----- OggVorbis Channel ----- */

OggChannel::OggChannel() {
  type = OGGCHAN;
  vf = new OggVorbis_File;

  old_section = current_section = -1;

  volume = 1.0;
  speed = 100;
  playmode = PLAY;
  opened = false;
  on = false;
  update = false;
}

OggChannel::~OggChannel() {
  if(opened)
    ov_clear(vf);
  delete vf;
  //  delete vi;
}

float OggChannel::get_audio() {
  int res;
  float fres;

  old_section = current_section;  
  
  if(seekable)
    /* we check the position here 'cause ov_pcm_tell returns the NEXT frame */
    framepos = ov_pcm_tell(vf);
  
  res = 
    ov_read(vf, (char*)buffo, BUF_SIZE, 0, 2, 1, &current_section);
  
  if(res<0) {
    fprintf(stderr,"OggVorbis: stream error\n");
    in_smp = 0;
    return(3.0);
  }

  in_smp = res;

  /*  if((res==0)||(old_section != current_section && old_section != -1)) {
     with this we should check out entering into a new logical bitstream
     left here, will manage later, maybe, one day, oh yes */
  if(in_smp==0) {
    return(2.0);
  }

  if(seekable)
    fres = upd_time();
  else fres = 0.0;

  return(fres);

}

int OggChannel::set(char *file) {
  int res = 0;

  on = false;

  if(opened) {
    ov_clear(vf);
    opened = false;
  }

  oggfile = fopen(file,"r");
  if(oggfile==NULL)
    return(res);
  
  if(ov_open(oggfile,vf,NULL,0) < 0) {
    fprintf(stderr, "OggVorbis: input not an Ogg Vorbis audio stream.\n");
    return(res);
  }

  vi = ov_info(vf, -1);

  samplerate = vi->rate;
  channels = vi->channels;

  if(!set_mixer()) {
    ov_clear(vf);
    return(res);
  } 

  /* pcm position */
  framepos = 0;
  /* that's a bit shamanic but works fine for me */
  fps = samplerate;
  /* time position */
  time.h = time.m = time.s = 0;
  /* floating point position */
  position = time.f = 0.0;

  /* check if seekable */
  res = (ov_seekable(vf)) ? 1 : 2;
  seekable = (res>1) ? false : true;

  if(seekable)
    frametot = ov_pcm_total(vf,-1);

  opened = true;
  return(res);
}

float OggChannel::mix(int *mixpcm) {
  float res;
  
  res = get_audio();
  if(res>1.0) return(res);
  
  in_smp = (*mixxx)(&mixpcm[0],(short int *)&buffo[0],in_smp,volume);

  return(res);
}

bool OggChannel::play() {
  if(!on) {
    if(opened) {
      on = true;
      if(time.f!=position) pos(position);
    }
  }
  return(on);
}

bool OggChannel::stop() {
  on = false;

  position = 0.0;

  return(!on);
}

bool OggChannel::pos(float pos) {

  if(on) {
    if(pos==0.0) {
      if(ov_pcm_seek(vf,1)!=0)
	return(false);
    } else {
      if(ov_pcm_seek_page(vf,(ogg_int64_t)((double)frametot * pos))!=0)
	return(false);
    }

    time.f = position = pos;

  } else {
    position = pos;
  }

  return(true);
}

#endif

/* ----- Mpeg Channel ----- */

MpegChannel::MpegChannel() {

  type = MP3CHAN;

  loader = NULL;
  server = NULL;

  volume = 1.0;
  speed = 100;
  time.h = time.m = time.s = 0;
  position = time.f = 0.0;
  playmode = PLAY;
  opened = false;
  on = false;
  update = false;
}

MpegChannel::~MpegChannel() {
  if(opened) clean();
}

void MpegChannel::clean() {
  if(loader!=NULL) {
    delete loader;
    loader = NULL;
  }

  if(server!=NULL) {
    delete server;
    server = NULL;
  }
}

int MpegChannel::set(char *file) {
  int res = 0;
  
  on = false;
  
  if(opened) {
    clean();
    opened = false;
  }

  loader = Soundinputstream::hopen(file);
  if(loader == NULL) {
    clean();
    return(res);
  }

  server = new Mpegtoraw(loader);
  if(server == NULL) {
    clean();
    return(res);
  }

  server->initialize(file);

  samplerate = server->getfrequency();
  if(server->getmode()<3)
    channels = 2;
  else
    channels = 1;

  bitrate = server->getbitrate();

  if(!set_mixer()) {
    clean();
    return(res);
  }

  /* pcm position */
  framepos = 0;
  /* time position */
  time.h = time.m = time.s = 0;
  /* floating point position */
  position = time.f = 0.0;

  /* check if seekable */
  res = (loader->seekable) ? 1: 2;
  seekable = (res>1) ? false : true;

  if(seekable) { /* setup position tracking variables */
    /* total chunks into bitstream (if seekable) */
    frametot = server->gettotalframe();
    /* size of a chunk in bytes */
    framesize = server->getpcmperframe()<<2;
    /* how much frames make second */
    fps = samplerate/server->getpcmperframe();
  } else framesize = (1152*channels)<<1; /* hardcoded mp3 framesize in bytes */

  opened = true;
  return(res);
}

float MpegChannel::get_audio() {
  float res;
  /*  uhm, i guess that's not needed
      if(!server->run(-1)) // Initialize MPEG Layer 3
      return(3.0);
  */

  if(!server->run(1)) {
    return(2.0);
    /* should i set errorcodes here? */
  }

  in_smp = framesize;

  if(seekable) {
    framepos = server->getcurrentframe();
    res = upd_time();
  } else res = 0.0;
  /* resampling deactivated (speed)
     in_smp = resample((short int *)&server->rawdata[0],in_smp,buffo,speed);
     fprintf(stderr,"Mpeg: get_audio : in_smp %u\n",in_smp);
  */
  return(res);
}

bool MpegChannel::play() {
  if(!on) {
    if(opened) {
      on = true;
      if(time.f!=position) {
	pos(position);
      }
    }
  }
  return(on);
}

bool MpegChannel::stop() {
  on = false;

  position = 0.0;

  return(true);
}

bool MpegChannel::pos(float pos) {

  if(on) {
    if(pos==0.0) {
      server->setframe(0);
    } else {
      server->setframe((int) (frametot*pos));
    }
    time.f = position = pos;
  } else
    position = pos;
  
  return(true);
}

float MpegChannel::mix(int *mixpcm) {
  float res;
  
  res = get_audio();
  if(res>1.0) return(res);
  
  in_smp = (*mixxx)(&mixpcm[0],(short int *)&server->rawdata[0],in_smp,volume);
  
  return(res);
}

/* ----- LiveIN DSP Channel ----- */

LiveIn::LiveIn(int smpr, int chans, int *thedsp) {
  dsp = thedsp;
  sample_rate = smpr;
  channels = chans;

  num_samples=(int)((sample_rate*BUF_SIZE)/(sample_rate<<1));
  opt = (num_samples*channels);
  
  gotin = new short[opt];
  
  on = false;
}

LiveIn::~LiveIn() { 
  delete [ ] gotin;
};

int LiveIn::mix(int *mixpcm) {
  int res;

  res = get_audio();
  mixxx_stereo_44(mixpcm,gotin,res);

  return(res);
}


int LiveIn::get_audio() {
  int res;

  res = read(*dsp,gotin,opt);

  return(res);
}
