/* jmixer.cpp - part of MUSE
 * (c)2000 Denis Roio aka jaromil
 * c++ engine for multiple mp3 realtime stream mixing
 */

#include <iostream.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <termios.h>
#include <sys/time.h>
#include <config.h>

#include "jmixer.h"
#include "audioproc.h"

/* process_buffer BUF_SIZE is:
   BUF_SIZE of 32bit ints *2channels *8resampling_space

   audio_buffer BUF_SIZE is:
   BUF_SIZE of 16bit short *2channels *8resampling_space
*/

#define MUFHD0 ((BUF_SIZE<<1)<<3)
static short audio_buffer[MUFHD0];
static int process_buffer[MUFHD0];

Stream_mixer::Stream_mixer() {

  /* well... let's initialize channels with mp3 decoders */
  int i;
  for(i=0;i<MAX_CHANNELS;i++)
    xach[i] = NULL;

  dsp = 0;
  have_gui = false;

  dspout = false;
  icecastout = false;
  fileout = false;
  quit = false;

  /* Initialization as a synchronous object */
  synchronous.registerObjectName("Stream_mixer");

}

void Stream_mixer::run() {

  if(debug)
    cerr << selfName() << " running on thread " << (int)pthread_self() << "\n";

  while(!quit) {
    
    synchronous.select.accept(0, id_stream_mixer);
    
    synchronous.select.waitUntil(1, LTI::Kernel::now());
    synchronous.wait();

    cafudda();
  }

  cerr << selfName() << " exiting " << "\n";

}

Stream_mixer::~Stream_mixer() {
  int i;

  if(dsp!=0)
    close_soundcard();
  
  if(icecastout)
    stop_icecast();

  for(i=0;i<MAX_CHANNELS;i++)
    delete xach[i];

}

void Stream_mixer::register_gui(Amuse *reg_gui) { 
  gui = reg_gui; 
  have_gui = true;
  if(HAVE_LAME) {
    char temp[256];
    lame.version();
    sprintf(temp,"Encoding with Lame %s",lame.ver);
    gui->status_lame->value(temp);
  }
}

bool Stream_mixer::open_soundcard()
{
    int format,tstereo,speed,caps;
    static int val;
    
    if((dsp=open("/dev/dsp",O_RDWR))==-1) {
        perror("Error opening /dev/dsp: ");
	return(false);
    } else
      fprintf(stderr,"Found soundcard on /dev/dsp\n");

    /* BUFFER FRAGMENTATION */
    val = (FRAGCOUNT << 16) | FRAGSIZE;

    if(ioctl(dsp,SNDCTL_DSP_SETFRAGMENT,&val)==-1)
      fprintf(stderr,"failed to set dsp buffers\n");

    /* or LE */
    format = AFMT_S16_NE;

    tstereo = 1; /* only stereo _DSP_ in/out */
    speed = SAMPLE_RATE; /* 44100hz mixing */

    ioctl(dsp, SNDCTL_DSP_GETCAPS, &caps);
    if(caps & DSP_CAP_DUPLEX) {
      fullduplex = true;
      fprintf(stderr,"full duplex supported. good\n");
      ioctl(dsp, SNDCTL_DSP_SETDUPLEX, 0);
      ioctl(dsp, DSP_CAP_DUPLEX, 0);
    } else {
      fprintf(stderr,"full duplex is not supported\n");
      fullduplex = false;
    }

    /* FORMAT */
    if(ioctl(dsp,SNDCTL_DSP_SETFMT,&format)==-1)
      fprintf(stderr,"failed to set data format\n");

    /* CHANNELS */
    if(ioctl(dsp,SNDCTL_DSP_STEREO,&tstereo)==-1)
      fprintf(stderr,"something went wrong with the stereo setting\n");

    /* SAMPLERATE */
    if(ioctl(dsp,SNDCTL_DSP_SPEED,&speed)==-1)
      fprintf(stderr,"speed setting failed\n");

    /* GET FRAG SIZE BACK */
    if(ioctl(dsp,SNDCTL_DSP_GETBLKSIZE,&val)==-1)
      fprintf(stderr,"get block size failed to return\n");

    fprintf(stderr,"mixing 16bit %dHz stereo\n",speed);

    dspout = false;

    livein = new LiveIn(speed, tstereo+1, &dsp);

    return(true);
} /* open_soundcard */

bool Stream_mixer::set_icecast(char *ip, int port, char *pass, char *mountpoint, int bps, char *name, char *url, char *desc, int login) {

  if(!ice.set_server(ip,port,pass,mountpoint,bps,login)) {
    if(have_gui)
      gui->status_main->value("doom damnit! i can't resolve icecast hostname");
    else
      fprintf(stderr,"Can't resolve icecast server hostname\n");
    synchronous.reply();
    return(false);
  }

  ice.set_meta(name, url, desc);
      
  if((!lame.initialized)||(lame.bps!=bps)) {
    gui->status_main->value("ouch! lame is not initialized!");
    synchronous.reply();
    return(false);
  }

  /* ::::: we are connected */
  if(ice.start()) {
    icecastout = true;

    /* notice the user */
    if(have_gui) {
      char temp[256];
      gui->status_main->value("WE GOT MILK !!");
      sprintf(temp,"STREAMING ON [ http://%s:%u/%s ]",ip,port,mountpoint);
      gui->status_ice->value(temp);
    } else
      fprintf(stderr,"Estabilished %ubps stream on %s:%u/%s\n",bps,ip,port,mountpoint);

    
    
  }

  /* ::::: connection failed */
  else {
    
    icecastout = false;
    if(have_gui) {
      char temp[256];
      sprintf(temp,"Got problems streaming on server %s. No milk??",ip);
      gui->status_main->value(temp);
    } else
      fprintf(stderr,"Got problems streaming on server %s. No milk??\n",ip);
  }
  
  synchronous.reply();
  return(icecastout);
}

void Stream_mixer::set_lame(int bps, int freq, int mode, int quality, int lowpass, int highpass) {
  lame.init(bps, freq, mode, quality, lowpass, highpass);
  synchronous.reply();
}

void Stream_mixer::stop_icecast() {
  ice.stop();
  icecastout = false;

  if (have_gui) {
    gui->status_main->value("Stopped streaming to icecast server");
    gui->status_ice->value("");
  } else
    fprintf(stderr,"Stopped streaming to icecast server\n");

  synchronous.reply();
}

int Stream_mixer::set_mp3file(char *file, int bps) {
  int res;
  
  res = mp3file.openmp3(file);
  if(res>0) fileout = true;

  /*  
      if((!lame.initialized)||(lame.bps!=bps))
      lame.init(bps);
  */
  
  synchronous.reply();
  return(res);
}

void Stream_mixer::stop_mp3file() {
  mp3file.closemp3();
  fileout = false;
  synchronous.reply();
}

void Stream_mixer::cafudda()
{
  int i;
  float res;
  int max = 0;

  /* here memset takes byte num */
  memset(process_buffer,0,MUFHD0<<(SIZEOF_INT>>1));

  have_sound = false;

  for(i=0;i<MAX_CHANNELS;i++) {
    if(xach[i] != NULL) {
      if(xach[i]->on) {
	res = xach[i]->mix(process_buffer);
	updchan(i,res);
	max = (max<xach[i]->in_smp) ? xach[i]->in_smp : max;
	
	/* notice that we have pcm to process (at least 1 chan) */
	have_sound = true;
	
      } /* if(xach[i].on) */
    } /* if(xach[i] != NULL) */
  } /* for(i=0;i<MAX_CHANNELS;i++) */
  
  if(livein->on) {
    int ires;
    ires = livein->mix(process_buffer);
    max = (max<ires) ? ires : max;
    //    if(max < ires) max = ires;
    
    /* notice that we have pcm to process (at least 1 chan) */
    have_sound = true;
  }
  
  /* here: max = number of 32bit samples in process_buffer
     number of single 16bit stereo samples (max>>1)
  */

  if(have_sound) {
    
    /* CLIPPING
       this brings it back to a 16bit resolution
       i'm using a simple as effective alghoritm (audioproc.cpp) */
    clip_audio(&process_buffer[0],&audio_buffer[0],max);

    /* ENCODE AUDIO
       here lames need the number of BYTES
       so max*channels>>1
       (we mix in stereo domain so channels is always 2) */
    if(fileout||icecastout)
      lame.encode(&audio_buffer[0],max>>2);
    
    /* WRITE 2 MP3 FILE */
    if(fileout)
      mp3file.writemp3();

    /* SEND 2 ICECAST */
    if(icecastout) {
      if(ice.send()!=0) {
	if(have_gui)
	  gui->status_main->value("Send error: cannot send stream to icecast server");
	else
	  fprintf(stderr,"Send error: cannot send stream to icecast server\n");
      }
    }

    /* WRITE 2 DSP */
    if(dspout)
      /* write out interleaved stereo 16bit pcm 
	 dsp takes number of samples, the format
	 is being setted with ioctls in initialization
       */
      write(dsp,audio_buffer,max);

  }
  
  /* compute and draw levels */
  if(have_gui)
    if(gui->mix_monitor->showit)
      gui->mix_monitor->draw_levels(get_level(audio_buffer,max));
}

bool Stream_mixer::create_channel(int ch) {
  bool res = false;

  /* paranoia */
  if(xach[ch]!=NULL)
    delete xach[ch];

  /* by default we create a mpeg channel */
  xach[ch] = new MpegChannel();
  
  if(xach[ch]==NULL) return(res);

  res = true;
  synchronous.reply();
  return(res);
}

void Stream_mixer::delete_channel(int ch) { 
  if(xach[ch]!=NULL) {
    xach[ch]->stop();
    delete xach[ch];
    xach[ch] = NULL;
  }
  synchronous.reply();
}

void Stream_mixer::pause_channel(int ch) {

  if(xach[ch]->opened)
    xach[ch]->on = !xach[ch]->on;
  
  synchronous.reply();
}

/* set_channel returns:
   0 - error
   1 - bitstream opened (seekable)
   2 - bitstream opened (not seekable)
*/
int Stream_mixer::set_channel(int ch, int playlist_pos) {
  int res;
  char *sel = playlist[ch].song(playlist_pos);
  char temp[128];

  if(sel==NULL)
    return(false);
  
  if(strstr(sel,".ogg")!=NULL) {
#ifdef HAVE_VORBIS
    /* take care to delete the old channel if present*/
    if(xach[ch]!=NULL) {
      if(xach[ch]->type != OGGCHAN) {
	delete xach[ch];
    	xach[ch] = new OggChannel();
      }
    }
#else
    if(have_gui) {
      gui->status_main->value("can't open .ogg files, OggVorbis support not compiled.\n");
    } else {
      fprintf(stderr,"can't open .ogg files, OggVorbis support not compiled.\n");
    }
    synchronous.reply();
    return(false);
#endif
  } else if(strstr(sel,".mp3")!=NULL) {
    /* take care to delete the old channel if present*/
    if(xach[ch]!=NULL) {
      if(xach[ch]->type != MP3CHAN) {
	delete xach[ch];
#ifdef HAVE_XAUDIO
	xach[ch] = new XaudioChannel();
#else
	xach[ch] = new MpegChannel();
#endif
      }
    }
  }

  /* ok, we have the channel here, let's load the bitstream */
  res = xach[ch]->set(playlist[ch].song(playlist_pos));
  switch(xach[ch]->playmode) {
  case 0: /* LOOP :::::::::::::::::::::::: */
  case 1: /* CONT :::::::::::::::::::::::: */
    xach[ch]->play();
    break;
  default: /* PLAY :::::::::::::::::::::::: */
    break;
  }    

  if(res==0) {
    if(have_gui) {
      gui->status_main->value("Doom damnit! can't open the file selected");
    } else {
      fprintf(stderr,"error opening file %s\n", playlist[ch].song(playlist_pos));
    }
  } else {
    playlist[ch].select(playlist_pos);
    if(have_gui) {
      switch(xach[ch]->type) {
      case MP3CHAN:
	sprintf(temp,"Mpeg on chan[%u] %ukbit %ukhz %s"
		, ch, xach[ch]->bitrate, xach[ch]->samplerate,
		(xach[ch]->channels==1) ? "mono" : "stereo" );
	break;
      case OGGCHAN:
	sprintf(temp,"OggVorbis on chan[%u] %ukhz %s"
		, ch, xach[ch]->samplerate,
		(xach[ch]->channels==1) ? "mono" : "stereo" );
	break;
      }
      gui->status_main->value(temp);
    }
  }

  synchronous.reply();
  return(res);
}

void Stream_mixer::set_all_volumes(float *vol) {
  int ch;
  for(ch=0;ch<MAX_CHANNELS;ch++) {
    if(xach[ch]!=NULL)
      xach[ch]->volume = vol[ch];
  }
  synchronous.reply();
}

void Stream_mixer::set_volume(int ch, float vol) {

  xach[ch]->volume = vol;

  synchronous.reply();
}

void Stream_mixer::set_speed(int ch, int speed) {

  xach[ch]->speed = speed;

  synchronous.reply();
}

void Stream_mixer::play_channel(int ch) {
  
  if(xach[ch]->opened) {
    xach[ch]->play();
  }

  synchronous.reply();
}
  
bool Stream_mixer::stop_channel(int ch) {

  bool res;

  res = xach[ch]->stop();

  /* reset the speed when stopping
     xach[ch]->speed = 100;
     gui->chan[ch]->speed_slider->value(100);
  */

  synchronous.reply();
  return(res);

}

bool Stream_mixer::set_position(float pos, int ch) {
  bool res = true;

  res = xach[ch]->pos(pos);

  if(!res) {
    char temp[50];
    sprintf(temp,"Error in seeking file on channel [%u]",ch);      
    if(have_gui)
      gui->status_main->value(temp);
    else
      fprintf(stderr,"%s\n",temp);
  }

  synchronous.reply();
  return(res);
}

bool Stream_mixer::set_live(bool stat) {
  if(!((dspout)&&(!fullduplex)&&(stat)))
    livein->on = stat;

  synchronous.reply();
  return(livein->on&stat);
}

bool Stream_mixer::set_lineout(bool stat) {
  if(!((livein->on)&&(!fullduplex)&&(stat)))
    dspout = stat;

  synchronous.reply();
  return(dspout&stat);
}

void Stream_mixer::close_soundcard() {
  ioctl(dsp, SNDCTL_DSP_RESET, 0);
  close(dsp);
}

bool Stream_mixer::set_playlist(char *file, int ch) {
  bool res = false;
  res = playlist[ch].load(file);
  if(res) {
    if(have_gui) {
      gui->chan[ch]->playlist_browser->clear();
      int c;
      char *p;
      for(c=0;c<playlist[ch].length;c++) {
	p = playlist[ch].song(c);

	/* takes out the directory path */
	while(*p!='\0') p++;
	while(*p!='/') p--;

	gui->chan[ch]->playlist_browser->add(p+1,NULL);
      }
    }
  }
  synchronous.reply();
  return(res);
}

void Stream_mixer::upd_timeline(int ch, float res) {
  char temp[24];
  
  if(xach[ch]->update) {
    gui->chan[ch]->timeline_slider->value(xach[ch]->time.f);
    
    sprintf(temp,"%02u:%02u:%02u",xach[ch]->time.h,xach[ch]->time.m,xach[ch]->time.s);
    gui->chan[ch]->time->value(temp);
  }

}

/* updchan takes care that any action is taken (channel ends etc.)
   and updates the gui if registered */
void Stream_mixer::updchan(int ch, float res) {

  bool lastsong = false;

  if(res==3.0) {
    char temp[64];
    sprintf(temp,"arg... channel [%u] out of sync",ch);
    if(have_gui)
      gui->status_main->value(temp);
    else
      cerr << temp << endl;
  } else if(res==2.0) {
    switch(xach[ch]->playmode) {
    case 0: /* LOOP :::::::::::::::::::::::: */
      /*
	if(have_gui) {
	gui->chan[ch]->timeline_slider->value(0.0);
	gui->chan[ch]->time->value("00:00:00");
	}
      */
      xach[ch]->set(playlist[ch].selected->name);
      xach[ch]->play();
      break;
    case 1: /* CONT :::::::::::::::::::::::: */
      int newsong;
      xach[ch]->stop();
      newsong = gui->chan[ch]->playlist_browser->value();
      gui->chan[ch]->timeline_slider->value(0.0);
      gui->chan[ch]->time->value("00:00:00");

      do {
	newsong++;
	
	/* if last song in playlist */
	if(newsong>playlist[ch].length) {
	  lastsong = true;
	  break;
	}
	
	/* -1 'cause fltkbrowser starts from 1 */
	playlist[ch].select(newsong-1);
      } while(!xach[ch]->set(playlist[ch].selected->name));
	
      if(lastsong) {
	gui->chan[ch]->name->value("");
	break;
      }

      if(xach[ch]->opened) {
	gui->chan[ch]->name->value(gui->chan[ch]->playlist_browser->text(newsong));
	gui->chan[ch]->playlist_browser->value(newsong);
	xach[ch]->play();
      } else 
	gui->chan[ch]->name->value("");

      break;
    case 2: /* PLAY :::::::::::::::::::::::: */
      xach[ch]->set(playlist[ch].selected->name);
      break;
    }
  } else {
    if(have_gui)
      upd_timeline(ch, res);
  }
}
