#include "synchronous.h"
#include <assert.h>
#include <stdio.h>
#include <unistd.h>

/* Threads and usleep does not work, select is very portable */
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
void dosleep(unsigned int msec) {
	struct timeval time;
	time.tv_sec=0;
	time.tv_usec=msec;

	select(0, NULL, NULL, NULL, &time);
}

bool LTI::Thread::debug = false;

LTI::Thread::Thread()
{
	pthread_mutex_init(&mutex, NULL);
	pthread_cond_init(&cond, NULL);
}

void LTI::Thread::start()
{
	if (pthread_create(&thread, NULL, &startRoutine, this)) {
		perror("pthread_create:");
		//		exit(-2);
	}
	// pthread_detach(thread);
}

void LTI::Thread::lock()
{
	if (pthread_mutex_lock(&mutex)) {
		perror("pthread_mutex_lock:");
		//		exit(-2);
	}
	dprintf(stderr, "%d: %p->lock()\n", (int) pthread_self(), this);
}

void LTI::Thread::unlock()
{
	if (pthread_mutex_unlock(&mutex)) {
		perror("pthread_mutex_unlock:");
		//		exit(-2);
	}
	dprintf(stderr, "%d: %p->unlock()\n", (int) pthread_self(), this);
}

void LTI::Thread::wait()
{
	dprintf(stderr, "%d: %p->wait()...\n", (int) pthread_self(), this);
	if (pthread_cond_wait(&cond, &mutex)) {
		perror("pthread_cond_wait:");
		//		exit(-2);
	}
	dprintf(stderr, "%d: %p->wait() has been notified\n",
		(int) pthread_self(), this);
}

int LTI::Thread::notify()
{
	int r;

	dprintf(stderr, "%d: %p->notify()\n", (int) pthread_self(), this);
	if ((r = pthread_cond_signal(&cond))) {
		perror("pthread_cond_signal:");
	}
	return r;
}

int LTI::Thread::join()
{
	int r;
	if ((r = pthread_join(thread, NULL))) {
		perror("pthread_join:");
	}
	dprintf(stderr, "%d: %p->join()\n", (int) pthread_self(), this);
	return r;
}

int LTI::Thread::cancel()
{
	int r;

	if ((r = pthread_cancel(thread))) {
		perror("pthread_cancel:");
	}
	return r;
}

void *LTI::Thread::startRoutine(void *arg)
{
	((Thread *) arg)->run();
	return NULL;
}

void LTI::SyncList::call(int selectC, const Synchronous * pCallee, int nbMethod)
{
	assert(nbMethod > 0);
	assert(nbEntries < MAX_SELECT_STATEMENTS);
	
	entry[nbEntries] = nbMethod;
	selectCase[nbEntries] = selectC;
	callee[nbEntries] = pCallee;
	nbEntries++;
}

void LTI::SyncList::call(int selectC, const Object *pCallee, int nbMethod)
{
	call(selectC, pCallee->getSynchronous(), nbMethod);
}

void LTI::SyncList::accept(int selectC, int nbMethod)
{
	assert(nbMethod > 0);
	assert(nbEntries < MAX_SELECT_STATEMENTS);
	
	entry[nbEntries] = -nbMethod;
	selectCase[nbEntries] = selectC;
	nbEntries++;
}

void LTI::SyncList::waitUntil(int selectC, int timer)
{
	assert(nbEntries < MAX_SELECT_STATEMENTS);
	
	entry[nbEntries] = 0;
	timerValue[nbEntries] = timer;
	selectCase[nbEntries] = selectC;
	nbEntries++;
}

LTI::Synchronous::Synchronous(const char *myName)
{
	select.connect(this);
	guiUnlock.connect(this);
	_selfName = myName;
	pthread_mutex_init(&mutex, NULL);
	pthread_cond_init(&cond, NULL);
}

void LTI::Synchronous::call(Object *target, int methodID)
{
	select.call(0, target->getSynchronous(), methodID); wait();
}

void LTI::Synchronous::reply()
{
	pthread_mutex_lock(&mutex);
	pthread_cond_signal(&cond);
	pthread_mutex_unlock(&mutex);
}

int LTI::Synchronous::wait()
{
	pthread_mutex_lock(&mutex);
	
	Kernel::syncRequest(&select);
	pthread_cond_wait(&cond, &mutex);

	pthread_mutex_unlock(&mutex);
	return select.value;
}

void LTI::Synchronous::externalSync()
{
	pthread_mutex_unlock(&mutex);
	
	// if (trace) fprintf(stderr, "%d: in Kernel::Synch_Unlock\n", (int)pthread_self());
	if (guiUnlock.nbEntries != 0) {
		pthread_mutex_unlock(&mutex);
		return;		// last unlock still in the kernel
	}
	guiUnlock.selectCase[0] = 0;
	guiUnlock.entry[0] = 999;
	guiUnlock.callee[0] = this;
	guiUnlock.nbEntries = 1;
	Kernel::syncRequest(&guiUnlock);

	pthread_mutex_unlock(&mutex);
}

// FIXME: make a static instantiation instead ?
LTI::Kernel *LTI::Kernel::uniqueInstance = new Kernel();
bool LTI::Kernel::trace;
// FIXME: timer outside the kernel (syntax only)
LTI::Kernel::Timer *LTI::Kernel::timer;
LTI::SyncList *LTI::Kernel::plist;
int LTI::Kernel::currentTime;
int LTI::Kernel::activeObjectNb;
int LTI::Kernel::index;
unsigned long LTI::Kernel::syncStats;

LTI::Kernel::Kernel() : Thread()
{
	uniqueInstance = this;
	plist = NULL;
	currentTime = 0;
	activeObjectNb = 0;
	syncStats = 0;
	trace = false;
	timer = new Timer();
	start();
	timer->start();
}

void LTI::Kernel::Timer::run()
{
	pthread_mutex_lock(&mutex);
	
	if (trace)
		fprintf(stderr,
			"Kernel::timer (%p) is running on thread %d\n",
			this, (int) pthread_self());
	while (1) {
		dosleep(resolution);
		currentTime += resolution;
		
		// wait for the kernel to sleep, or we risk loosing signals...
		uniqueInstance->lock();
		uniqueInstance->notify();
		uniqueInstance->unlock();
	}
	pthread_mutex_unlock(&mutex);
}

void LTI::Kernel::printWaitingList()
{
	SyncList *select;
	int i;

	fprintf(stderr, "-- new selection inserted into the kernel\n");
	for (select = plist; select != NULL; select = select->next) {
		fprintf(stderr, ". %s (%p)\n",
			select->object->selfName(), select->object);
		for (i = 0; i < select->nbEntries; i++) {
			if (select->entry[i] > 0)
				fprintf(stderr,
					"  calling method %d of %s\n",
					select->entry[i],
					select->callee[i]->selfName());
			else if (select->entry[i] < 0)
				fprintf(stderr, "  accepting method %d\n",
					-select->entry[i]);
			else
				fprintf(stderr, "  waiting %ds\n",
					select->timerValue[i] -
					Kernel::now());
		}
	}
}

void LTI::Kernel::syncRequest(SyncList * list)
{
	pthread_mutex_lock(&uniqueInstance->mutex);

	assert(list != NULL);
	
	syncStats++;
	
	list->next = plist;
	plist = list;
	if (trace) {
		printWaitingList();
	}
	uniqueInstance->notify();

	pthread_mutex_unlock(&uniqueInstance->mutex);
}

// look for a timeout		
bool LTI::Kernel::findTimeout(SyncList **p1, SyncList **p2)
{
	SyncList *select;
	int i;
	
	for (select = plist; select != NULL; select = select->next) {
		for (i = 0; i < select->nbEntries; i++) {
			if (select->entry[i] == 0) {
				if ((select->timerValue[i] - currentTime) < 0) {
					*p1 = select;
					(*p1)->value = select->selectCase[i];
					*p2 = NULL;
					return true;
				}
			}
		}
	}
	return false;
}

// look for a matching pair in two different Synch_List objects
bool LTI::Kernel::findMatch(SyncList **p1, SyncList **p2)
{
	SyncList *select1, *select2;
	int i, j;
	
	// for each selection in the list
	for (select1 = plist; select1 != NULL; select1 = select1->next) {
		// try to find a call
		for (i = 0; i < select1->nbEntries; i++) {
			// ok, we have a call
			if (select1->entry[i] > 0) {
				// rescan the selection list from the beginning
				for (select2 = plist; select2 != NULL; select2 = select2->next) {
					// looking for a selection belonging to the matching object
					if (select1->callee[i] == select2->object) {
						// look if the matching object accepts our call
						for (j = 0; j < select2->nbEntries; j++) {
							// ok: we have a match
							if (select1->entry[i] == -select2->entry[j]) {
								*p1 = select1;
								(*p1)->value = select1->selectCase[i];
								*p2 = select2;
								(*p2)->value = select2->selectCase[j];
								index = i;
								return true;
							}
						}
					}
				}
			}
		}
	}
	return false;
}

// display new awaken objects
void LTI::Kernel::displaySelection(SyncList *p1, SyncList *p2)
{
	if ((p1 != NULL) && (p2 != NULL)) {
		fprintf(stderr,
			"kernel: %s calls %s method num. %d (case %d)\n",
			p1->object->selfName(),
			p2->object->selfName(),
			p1->entry[index],
			p1->value);
	} else if ((p1 != NULL)) {
		fprintf(stderr, "kernel:  %s wakes up (case %d)\n",
			p1->object->selfName(), p1->value);
	} else {
		fprintf(stderr, "kernel:  nothing to do yet...\n");
	}
}

void LTI::Kernel::releaseSelection(SyncList *p1)
{
	SyncList *select;
	
	assert(p1 != NULL);
	
	if (plist == p1)
		plist = p1->next;
	else {
		for (select = plist; select != NULL; select = select->next) {
			if (select->next == p1)
				select->next = p1->next;
		}
	}
	// set the number of entries back to 0
	p1->nbEntries = 0;
}

void LTI::Kernel::run()
{
	pthread_mutex_lock(&uniqueInstance->mutex);

	SyncList *p1, *p2;

	if (trace)
		fprintf(stderr, "Synchronous Kernel: running on thread %d\n",
			(int) pthread_self());
	while(true) {
		wait();
		
		p1 = NULL; p2 = NULL; index = 0;
		
		if (findTimeout(&p1, &p2)) {
			releaseSelection(p1);
			
		} else {
			if (findMatch(&p1, &p2)) {
				releaseSelection(p1);
				releaseSelection(p2);
			}
		}

		if (trace) displaySelection(p1, p2);
		if (p1 != NULL) {		
			p1->object->reply();
		}
	}

	pthread_mutex_unlock(&uniqueInstance->mutex);
}
