#ifndef __SYNCHRONOUS_H
#define __SYNCHRONOUS_H

#include <pthread.h>

class LTI
{
// or use namespace LTI

  /* this goes into a usleep used by the LTI::Kernel::Timer */
#define DEFAULT_TIMER_RESOLUTION 100

#define MAX_SELECT_STATEMENTS 20

#define dprintf if (Thread::debug) fprintf

/**
   @short encapsulates a Posix thread

   The Thread class encapsulates a Posix thread.

   The thread runs the code of method 'run'. By default it just
   returns, thus terminating the thread. You need to override the
   method 'run' to actually do something usefull.

   The thread wont start execution of method 'run' until you call the
   method 'start'. It allows one to prepare a task and later activate
   it, for example when other tasks are ready to collaborate.

   To ease interaction with other threads, the Thread class contains
   both a mutex and a condition. To guaranty exclusive access to a
   thread's members and/or methods, you should first 'lock' it. The
   condition serves as a checkpoint when the thread needs to
   sleep. Method 'wait' waits on the condition, while 'notify' will
   wake a up the sleeping thread. The thread needs to hold the lock
   before going to sleep.

   The 'wait' method atomically releases the lock before sleeping, and
   reacquires the lock just before it starts running again.
   
   This class should work quite like the synchronized objects of the
   Java language.

   This class is mainly used internally, but you can take advantage of
   it in your own project if you need a convenient thread object.
*/
class Thread {
public:
	Thread();
	virtual ~Thread() {}

	/** the main part of the thread. when the method returns the
	    thread is terminated. */
	virtual void run(void) {}

	/** start the current thread */
	void start();

	/** request exclusive access to the thread.
	    This is a blocking call, ie, it wont return until you get
	    the lock */
	void lock();

	/** release the lock */
	void unlock();

	/** put the thread to sleep, waiting for a 'notify' */
	void wait();

	/** wake up the thread.

	    Be carefull though: if the thread is not currently
	    sleeping (in the wait method), then it won't receive you
	    signal.	You should always get the lock before calling
	    notify, and release it afterwards */ 
	int notify();

	/** wait for the termination of the thread. This is a blocking
	    call */
	int join();

	/** terminate the thread.

	    This won't work immediatly: the thread needs to reach a
	    cancellation point before reacting to this method. See
	    pthread_cancel(1) for more details */
	int cancel();

	/** debug flag: when set to 'true' _all_ threads will start
	    printing verbose debug informations to stderr */
	static bool debug;

protected:
	pthread_t thread;
	pthread_mutex_t mutex;
	pthread_cond_t cond;

	/** helper function used to jumpstart the thread. You
	    shouldn't need to redefine this unless you know what
	    you're doing...
	    The argument is ignored by the current implementation */
	static void *startRoutine(void *arg);
};

 public:
class Synchronous;
class Object;
class Kernel;

/**
    @short groups the set of methods an active objects might call or
    accept concurrently.

    This class groups the synchronous statements issued and/or
    accepted in parallel by an active object.
*/
class SyncList {
friend class Kernel;
friend class Synchronous;
public:
	/** creates a new SyncList linked to a synchronous object */
	SyncList(Synchronous *o = NULL) {
		object = o;
		nbEntries = 0;
	}

	/** links a Synchronous object to this SyncList */
	void connect(Synchronous *o) { object = o; }

	/** adds a call statement to the current set.
	    
	    This is seldom used in normal programs. You'll most likely
	    prefer to call this function with an Object class
	    parameter (see next function).

	    @param selectC a unique selection identifier representing this statement
	    @param pCallee the synchronous object which method is to be called
	    @param nbMethod the (integer) identifier representing the method to call
	*/
	void call(int selectC, const Synchronous *pCallee, int nbMethod);

	/** adds a call statement to the current set (with an Object reference)

	    @param selectC a unique selection identifier representing this statement
	    @param pCallee the synchronous object which method is to be called
	    @param nbMethod the (integer) identifier representing the method to call
	*/
	void call(int selectC, const Object *pCallee, int nbMethod);

	/** adds an accept statement to the current set. This means
	    that the current object accepts that others call a
	    specific method

	    @param selectC a unique selection identifier representing this statement
	    @param nbMethod the (integer) identifier representing the method to accept
	*/
	void accept(int selectC, int nbMethod);

	/** adds a timeout statement to the current set. This means
	    that the current object accepts to wait until the timer
	    reaches at least 'timer' units of time before being woken
	    up. This statement will be only be selected when no other
	    call or accept statement are chosen.

	    @param selectC a unique selection identifier representing this statement
	    @param timer the time after which the current object wants to wake up
	*/
	void waitUntil(int selectC, int timeout);

protected:
	int nbEntries;
	int value;
	SyncList *next;
	Synchronous *object;
	int entry[MAX_SELECT_STATEMENTS];
	int selectCase[MAX_SELECT_STATEMENTS];
	int timerValue[MAX_SELECT_STATEMENTS];
	const Synchronous *callee[MAX_SELECT_STATEMENTS];
};

/**
   @short provides a framework for using synchronous active objects

   The Synchronous class provides a framework for using synchronous
   active objects. It represents a "rendez-vous" between two objects.
*/
class Synchronous {
friend class Kernel;
public:
	/** instantiates a new synchronous object with an optionnal
	    object name. The name argument is mainly used for
	    debugging purposes, so that you can see your object more
	    clearly when looking at the synchronous kernel's
	    output. */
	Synchronous(const char *myName = "a synchronous object");

	/** expresses the intention to call an external method. The
	    method _does_not_ actually call a method, but indicates
	    that, provided the receiving object is ready to accept it,
	    we will call it's method */
	void call(Object *target, int methodID);

	/** expresses the intention to accept an external call */
	void accept(int methodID) { select.accept(0, methodID); wait(); }

	/** adds a synchronous wait to the list of desired synchronizations */
	void waitUntil(int timeout) { select.waitUntil(0, timeout); wait(); }

	/** used at the end of method call to notify the local active
	    object that the call is finished. This wakes up the host
	    object, i.e. the one implementing the method. */
	void reply();

	/** used from inside an object's body to ask the synchronous
	    kernel to take a decision about the pending accept/calls
	    issued by the active object. Once this method is called,
	    the active object goes to sleep and will only wake up when
	    one of it's offers is choosen by the kernel. */
	int wait();

	/** simply notifies this thread that a new (mostly GUI) event
	    has occured : this wakes up the thread in practice. This
	    method builds a dummy synchronization to immediatly wake
	    up the object. It's main purpose is to allow an external
	    object (which does not really respect the Synchronous
	    accept/call protocol) to still be able to notify the
	    current object without breaking it's behaviour. You're
	    free to use whatever method you please to transmit datas
	    before calling externalSync. Be carefull about race
	    conditions and thread safetyness though... */
	void externalSync();

	/** helper function used to rename the object in the debug
	    output generated by the synchronous kernel. You can make
	    the difference between multiple instances of the same
	    class with this method. */
	void registerObjectName(const char *objName) {
		_selfName = objName;
	}
	const char *selfName(void) const { return _selfName; }
	SyncList select;
protected:
	/** used by the 'externalSync' method to notify the object */
	SyncList guiUnlock;
	/** name of the current object for debugging output */
	const char *_selfName;
	pthread_mutex_t mutex;
	pthread_cond_t cond;
};

/**
    @short provides a framework for creating synchronous active objects

    The LTI::Object class provides a framework for using synchronous
    active objects.

    It is based on the Thread class and thus one must implement the
    'run' method to build the body of an active object.

    It owns a Synchronous object, to make synchronized call towards
    other synchronous object.

    The LTI::Object provides automatic thread support. This class also
    shows how one can use composition instead of inheritance to use
    active objects. In case you already have thread support provided
    by another framework, you only need to declare a Synchronous
    attribute in your own class.
*/
class Object: public Thread {
public:
	/** creates a new synchronous active object with an optionnal
	    object name.
	    @param myName the object name

	    This is mainly used for debugging purposes, so that you
	    can see your object more clearly when looking at the
	    synchronous kernel's output. */
	Object(const char *myName = "a synchronous active object") { synchronous.registerObjectName(myName); }

	/** provides a constant pointer to the object's synchronizer */
	const Synchronous *getSynchronous() const { return &synchronous; }

protected:
	const char *selfName(void) { return synchronous.selfName(); }
	Synchronous synchronous;
};

/**
    @short arbitrates synchronous method calls between active objects

    This class provides a synchronization point between multiple
    concurrent active objects (threads). A unique instance (singleton)
    is created at the beginning of a program and then used by the
    active objects.

    The body of the Kernel (method 'run') implements a choice
    algorithm to determine which pair of accept(s)/call(s) will be
    activated next, i.e. to select which object will be allowed to
    execute a method call towards another synchronous active object.
*/
class Kernel: public Thread {
public:
	/** singleton used by all synchronous objects */
  static Kernel *uniqueInstance;

	/** debug flag: set to 'true' to activate a listing of pending
	    accept/calls each time a synchronization is requested. */
  static bool trace;
  
	/** used for benchmarks only */
  static unsigned long syncStats;
  
	Kernel();

	/** adds a SyncList to the set of pending
	    synchronization. This will also ask the synchronous kernel
	    to rescan its list in search of a matching pair of
	    call/accept. */
	static void syncRequest(SyncList * list);

	/** the body of the synchronous kernel: chooses a pair of
	    matching call/accept. */
	virtual void run();

	/** helper function providing the current time. Used mainly in
	    'waitUntil' statements like <pre>
	select->waitUntil(Kernel::now() + 10);</pre>
	*/
	static int now() { return currentTime; }
	
protected:
	static int activeObjectNb;
	static int index;
	static int currentTime;
	static SyncList *plist;
	
	static void printWaitingList();
	static bool findTimeout(SyncList **p1, SyncList **p2);
	static bool findMatch(SyncList **p1, SyncList **p2);
	static void displaySelection(SyncList *p1, SyncList *p2);
	static void releaseSelection(SyncList *p1);

	class Timer: public Thread {
	public:
		Timer(int r = DEFAULT_TIMER_RESOLUTION) {
			resolution = r;
		}
		virtual void run();
	protected:
		int resolution;
	};
	static Timer *timer;
};

}; // namespace (or class) LTI

#endif // __SYNCHRONOUS_H
