
/*
 * main.c, utils.c, nolce.h : Copyright (C) 1997-98 Giuseppe Trovato
 *                                             (g.trovato@usa.net)
 * Version 1.9.2
 * Read LICENCE file before using the program.
 */

/****************************  INCLUDES  *************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <db.h>
#include <ctype.h>

#include "nolce.h"

/************************  EXTERN VARIABLES  *********************************/

char *netscape_dir, *lock_file, *curr_dir;
char g_or_G = '\0';
size_t len_cache_dir;
NODE *root = NULL;
long total = 0;
int status = 0;
bool yet_to_process = 0;

T_OPT opt = {0, 0, NULL, NULL, NULL, NULL};

enum vto
  { process_main, process_related, summarize };
enum pr
  { with_frames, is_image, normal };

/*************************** SIGNAL HANDLER **********************************/

void
sig_handler (int sig_number)
{
  if (sig_number == SIGINT)
    {
      char c;

      printf ("\n---- Do you really want to quit (y/n)? ");
      c = getchar ();
      while (getchar () != '\n');
      if ((c != 'y') && (c != 'Y'))
	{
	  signal (SIGINT, sig_handler);
	  if ((status == 1) && !(opt.var & SILENT))
	    printf ("Processing main HTML files:   ");
	  if ((status == 2) && !(opt.var & SILENT))
	    printf ("Processing related HTML files: ");
	  fflush (stdout);
	  return;
	}
    }
  if (sig_number == SIGSEGV)
    printf ("\nI feel unwell... it's better to die!\n");
  else
    printf ("\nBye!\n");
  
  unlink (lock_file);

  fflush (NULL);
  signal (sig_number, SIG_DFL);
  raise (sig_number);

}

/******************************  MAIN  ***************************************/

int
main (int argc, char **argv)
{
  int k, j;
  char temp[64];
  time_t n_hours = 0, t, cut_time;
  char *home, *err, *ls;
  int len_home;
  size_t currdir_size = 0;
  /*
   * Obtains home dir and Netscape's dir. 
   */
  home = getenv ("HOME");
  
  if (home)
    len_home = strlen (home);
  else
    perror_exit ("\nnolce:\nCould not obtain your home dir from the enviroment, exiting...")     
  
  do
    {
      currdir_size += 30;
      if (currdir_size > 10000)
	perror_exit ("\nnolce:\nCould not read current dir")
      curr_dir = (char *) gt_realloc (curr_dir, currdir_size);
    }
  while (!getcwd (curr_dir, currdir_size));

  netscape_dir = (char *) gt_malloc (len_home + 1 + 9 + 1);
  sprintf (netscape_dir, "%s/.netscape", home);
  lock_file = (char *) gt_malloc (len_home + 1 + 9 + 1 + 4 + 1);
  sprintf (lock_file, "%s/lock", netscape_dir);
  /*
   * Command line scanning. 
   */
  if (argc == 1)
    {
      char ans;

      printf ("Ok to process all files under %s/.netscape/cache ([y]es/[n]o/[h]elp) ? ", home);
      ans = getchar ();
      while (getchar () != '\n');
      if (ans == 'n')
	exit (EXIT_SUCCESS);
      else if (ans == 'h')
	help ();
    }

  for (k = 1; k < argc; k++)
    {
      if (!strncmp (argv[k], "--h", 3))
	help ();

      t = strtoul (argv[k], &err, 10);
      if (err && *err)		/* Arg isn't n_hours */
	{
	  char sw;
	  char **ptr;
	  size_t len = strlen (argv[k]);

	  for (j = 0; j < len; j++)
	    {
	      sw = argv[k][j];
	      ptr = NULL;

	      switch (sw)
		{
		case '-':		  		break;
		case 'w': opt.view_window = 1;		break;
		case 'W': opt.view_window = -1;	  	break;
		case 'm': opt.var |= MISSING_IMAGES;	break;
		case 's': opt.var |= SILENT;   		break;
		case 't': opt.var |= REPORT_TIME;	break;
		case 'f': opt.var |= NO_LINKS;	  	break;
		case 'p': opt.var |= WIN_CACHE; 	break;
	        case 'l': opt.var |= IGNORE_LOCK;	break;
		case 'k': opt.var |= LINK_FOR_IMAGES;	break;
		case 'c': ptr = &opt.cache_dir;		break;
		case 'd': ptr = &opt.dest_dir; 		break;
		case 'i': ptr = &opt.summary_file;	break;
		case 'g':
		case 'G':
		  g_or_G = sw;
		  ptr = &opt.str_tbc;
		  break;
		default:
		  print_error (sw, argv[0], illegal);
		}

	      if (ptr)
		{
		  bool oper = 1;

		  if (strchr ("igG", sw))
		    oper = 0;
     
		  if (*ptr)
		    print_error (sw, argv[0], already_supplied);

		  if (j == len - 1)
		    {
		      if (k < argc - 1)
			process_arg (argv[++k], ptr, oper);
		      else
			print_error (sw, argv[0], need_arg);
		    }
		  else
		    {
		      process_arg ((argv[k] + j + 1), ptr, oper);
		      break;
		    }
		}
	    }
	}
      else
	/*
	 * Arg is n_hours 
	 */
	{
	  if (n_hours)
	    print_error ('\0', argv[0], already_supplied);
	  else
	    n_hours = t;
	}
    }
  /*
   * End of command line scanning. 
   */

  /*
   * If some parameters aren't given, the defaults are used. 
   */
  if (!opt.cache_dir)
    {
      opt.cache_dir = (char *) gt_malloc (len_home + 1 + 9 + 1 + 5 + 1);
      sprintf (opt.cache_dir, "%s/cache", netscape_dir);
    }

  len_cache_dir = strlen (opt.cache_dir);

  if (!opt.dest_dir)
    {
      opt.dest_dir = (char *) gt_malloc (len_home + 1 + strlen (DEST_DIR) + 1);
      sprintf  (opt.dest_dir, "%s/%s", home, DEST_DIR);
    }

  if (!opt.summary_file)
    {
      opt.summary_file = (char *) gt_malloc (strlen (SUMMARY_FILE) + 1);
      strcpy (opt.summary_file, SUMMARY_FILE);
    }
  else
    /*
     * Takes only the final file name of a path
     * which may contain also the direcory. 
     */
  if ((ls = strrchr (opt.summary_file, '/')))
    {
      gt_strshift (opt.summary_file, 1 + ls - opt.summary_file);
      printf ("summary file is: %s/%s\n", opt.dest_dir, opt.summary_file);
    }
  /*
   * Turns signal handler on. 
   */
  signal (SIGINT, sig_handler);
  signal (SIGQUIT, sig_handler);
  signal (SIGTERM, sig_handler);
  signal (SIGSEGV, sig_handler);
  
  if (! (opt.var & IGNORE_LOCK))
    {
     /*
      * Creates lock file. 
      */
      if (!chdir (netscape_dir))
	{
	  sprintf (temp, "1.0.0.127:%ld", (long) getpid ());
	  if (symlink (temp, lock_file))
	    {
	      fprintf (stderr, "\nnolce:\nThe program can't get exclusive access to the cache."
		   "\nIf Netscape or another copy of nolce is running, exit from it and"
		   "\nretry, otherwise delete the file $HOME/.netscape/lock\n");
	      exit (EXIT_FAILURE);
	    }
	 }
      else
	perror_exit ("\nnolce:\nCould not access $HOME/.netscape dir")
    }
  /*
   * n_hours stuff.
   */
  if (n_hours != 0)
    {
      if (n_hours < 0)
	n_hours = -n_hours;
      if (n_hours > 175200)
	n_hours = 175200;	/*
				 * Max 20 years, for avoiding problems
				 * with data overflow.
				 */
      cut_time = time (NULL) - n_hours * 3600L;
    }
  else
    cut_time = 0;

  /* 
   * Reads the cache index. 
   */
  root = get_data (root, cut_time, opt.str_tbc);

  if (root)
    {
      if (!(opt.var & SILENT))
	printf ("   (done)"); 
      fflush (stdout); 
      /* 
       * Creates dest_dir. 
       */
      mkdir (opt.dest_dir, S_IRWXU | S_IRWXG | S_IRWXO);
      chdir (opt.dest_dir);
      mkdir ("nolce_files", S_IRWXU | S_IRWXG | S_IRWXO);
      start_summary_files (CONST, NULL);

      if (!(opt.var & SILENT))
	printf ("\nProcessing main HTML files:   ");
      status = 1;
      /* 
       * Reads the tree to process html files. 
       */
      visit_tree (root, process_main);
      
      if (yet_to_process && !(opt.var & SILENT))
	printf ("\nProcessing related HTML files: ");
      status = 2;
      do
	{
	  yet_to_process = 0;
	  visit_tree (root, process_related);
	}
      while (yet_to_process);

      if (!(opt.var & SILENT))
	{
	  printf ("\nWriting summary file `%s/%s'", opt.dest_dir, opt.summary_file);
	  fflush (stdout);
	}
      status = 3;
      /*
       * Creates the summary file. 
       */
      visit_tree (root, summarize);
      put_in_summary_files (NULL, 1);

      if (!(opt.var & SILENT))
	printf ("   (all done)\n");
    }
  else if (total == 0)
    printf ("\nNo documents were found, or could be accessed, matching requested conditions.\n");
  
  /* 
   * Removes lock file. 
   */
  if (! (opt.var & IGNORE_LOCK))
    unlink (lock_file);
  fflush (NULL);

  return 0;
}

/********************** Function get_data *************************************
* Reads the cache index file (index.db) and creates the binary tree with all  *
* the informations about cached files. The root of the tree is "start",       *
* "cut_time" is the correspondent of n_hours, "sub_str" is the string	      *
* supplied with -g or -G.						      *
******************************************************************************/

NODE *
get_data (NODE * start, time_t cut_time, char *sub_str)
{				/* Creates b-tree. */
  DB *index;
  DBT key, data;
  int type;
  size_t len;
  time_t d_time;
  bool tc;
  struct stat attr;
  char *url, *file, *c_url, *path, *cont;

  if (!(opt.var & WIN_CACHE))
    {
      path = (char *) gt_malloc (len_cache_dir + 1 + strlen (CACHE_FILE) + 1);
      sprintf (path, "%s/" CACHE_FILE, opt.cache_dir);
    }
  else
    {
      path = (char *) gt_malloc (len_cache_dir + 1 + strlen (WIN_CACHE_FILE) + 1);
      sprintf (path, "%s/" WIN_CACHE_FILE, opt.cache_dir);
    }

  chdir (opt.cache_dir);

  index = dbopen (path, O_RDONLY, 0, DB_HASH, NULL);
  if (index == NULL)
    {
	      fprintf (stderr, "\n\nnolce:\nthe supplied cache dir (%s) isn't valid, " 
			"or I can't read the index file.\n", opt.cache_dir);
      total = -1;
      return NULL;
    }
  
  if (!(opt.var & SILENT))
    {
      printf ("Processing cache information from `%s'", path);
      fflush (stdout);
    }

  while (!index->seq (index, &key, &data, R_NEXT))
    {
      url = (char *) (key.data);
      if (url)
	{
	  url += 8;

	  if (!url_is_valid (url))
	    continue;
	  
 	  file = (char *) (data.data);
	  cont = (char *) (data.data);
 	  file += 33;
	  d_time = *((time_t*) (cont + 12));
 	  cont += (71 + strlen (file));
	  
	  if (strstr (cont, "html") || strstr (cont, "x-www"))
	    type = 1;
	  else
	    type = 0;
	  
	  if (!strncmp (url, "wysiwyg", 7))
	    continue;

	  if ((opt.var & WIN_CACHE))
	    {
	      int k, len;

	      len = strlen (file);
	      for (k = 0; k < len; k++)
		file[k] = tolower (file[k]);
	    }

	  if (type)
	    {
	      if (stat (file, &attr))
		continue;

	      len = strlen (url) + 1;
	      c_url = (char *) gt_malloc (len);
	      strcpy (c_url, url);
	    }
	
	  standardize (url);
	  
	  if (type)
	    {
	      tc = d_time > cut_time;

	      if (tc && g_or_G)
		{
		  if (g_or_G == 'g')
		    tc = (strstr (c_url, sub_str) != NULL);
		  else
		    tc = (strstr (c_url, sub_str) == NULL);
		}

	      if (check_html_url (url, 1))
		start = add_url (start, c_url, url, (tc) ? (IS_HTML | NEED_INDEX | PR_MAIN | SHOW) : IS_HTML | NEED_INDEX, file, d_time);
	      else
		start = add_url (start, c_url, url, (tc) ? (IS_HTML | PR_MAIN | SHOW) : IS_HTML, file, d_time);

	      free (c_url);
	    }
	  else
	    start = add_url (start, "", url, 0, file, d_time);
	}
    }
  index->close (index);

  free (path);

  if (total > 0)
    return start;
  else
    return NULL;
}

/********************** Function add_url **************************************
* Adds informations for a file to the binary tree. First searches recursively *
* starting from "node" the right place for the new node, then creates it.     *
* Informations stored are: "or_url", the original url of the file; "url", the *
* url modified by standardize(); "type", a bitmapped variable indicating the  *
* type (image, html...) of the file; "file_name", the name under the cache    *
* dir; "mtime", the time when the file was created/modified.		      *
******************************************************************************/

NODE *
add_url (NODE * node, char *or_url, char *url, int type,
	 char *file_name, time_t mtime)
{
  int cmp = 1;

  if (node != NULL)
    cmp = strcmp (url, node->url);

  if (node == NULL)
    {
      size_t len;

      node = (NODE *) gt_malloc (sizeof (NODE));

      len = strlen (url);
      node->url = (char *) gt_malloc (len + 1);
      strcpy (node->url, url);

      node->type = type;
      node->mod_time = mtime;
      node->dup = 1;
      node->file_name = (char *) gt_malloc (strlen (file_name) + 1);
      strcpy (node->file_name, file_name);

      if (type & IS_HTML)
	{
	  node->or_url = (char *) gt_malloc (strlen (or_url) + 1);
	  strcpy (node->or_url, or_url);

	  node->title = (char *) gt_malloc (9);
	  strcpy (node->title, "Untitled");

	  if (type & NEED_INDEX)
	    {
	      node->url_w_index = (char *) gt_malloc (len + 1 + strlen (DEFAULT_HTML) + 1);
	      strcpy (node->url_w_index, url);
	      check_html_url (node->url_w_index, 0);

	      node->r_url = node->url_w_index;
	    }
	  else
	    node->r_url = node->url;
	}

      node->sx = NULL;
      node->dx = NULL;

      if (type & PR_MAIN)
	total++;
    }
  else if (cmp == 0)
    {
      if (strcmp (node->file_name, file_name))
	{
	  if (type & IS_HTML)
	    {
	      char *temp;
	      temp = (char *) gt_malloc (strlen (url) + 8 + 1);
	      sprintf (temp, "%s-%u", url, ++(node->dup));

	      if (mtime > node->mod_time)
		{
		  node = add_url (node, node->or_url, temp, node->type, node->file_name, node->mod_time);

		  if ((type & PR_MAIN) && !(node->type & PR_MAIN))
		    total++;
		  
		  node->type = type;
		  node->mod_time = mtime;
		  node->file_name = (char *) gt_realloc (node->file_name, strlen (file_name) + 1);
		  strcpy (node->file_name, file_name);
		}
	      else
		node = add_url (node, or_url, temp, type, file_name, mtime);

	      free (temp);
	    }
	  else if (mtime > node->mod_time)
	    {
	      node->mod_time = mtime;
	      node->file_name = (char *) gt_realloc (node->file_name, strlen (file_name) + 1);
	      strcpy (node->file_name, file_name);
	    }
	}
    }
  else if (cmp < 0)
    node->sx = add_url (node->sx, or_url, url, type, file_name, mtime);
  else if (cmp > 0)
    node->dx = add_url (node->dx, or_url, url, type, file_name, mtime);

  return node;
}

/********************** Function visit_tree ***********************************
* Visits (reads) the binary tree in pre-order, and, depending of "operation", *
* calls process_html_file() or put_in_summary_files();			      *
******************************************************************************/

void
visit_tree (NODE * node, int operation)
{
  static long c = 0;

  if (node == NULL)
    return;
  else
    {
      visit_tree (node->sx, operation);

      if ((operation == process_main) && (node->type & PR_MAIN))
	{
	  if (!(opt.var & SILENT))
	    {
	      printf ("\r\t\t\t\t%ld/%ld", ++c, total);
	      fflush (stdout);
	    }
	  process_html_file (node);
	}
      if ((operation == process_related) && (node->type & PR_REL)
	  && !(node->type & PROCESSED))
	{
	  if (!(opt.var & SILENT))
	    {
	      printf ("\r\t\t\t\t%ld", ++c - total);
	      fflush (stdout);
	    }
	  node->type |= PROCESSED;
	  process_html_file (node);
	}
      if ((operation == summarize) && (node->type & SHOW))
	put_in_summary_files (node, 0);

      visit_tree (node->dx, operation);
    }
}

/********************** Function find_url *************************************
* Finds the node in the tree whose root is "node" containing informations     *
* about the url "url".					 		      *
******************************************************************************/

NODE *
find_url (NODE * node, char *url)
{
  int comp;

  if (node != NULL)
    {
      comp = strcmp (url, node->url);
      if (comp == 0)
	return node;
      else if (comp < 0)
	return (find_url (node->sx, url));
      else
	return (find_url (node->dx, url));
    }
  else
    return NULL;
}

/********************** Function process_html_file ****************************
* Scans, through yylex(), the html file referred by "node" and writes it,     *
* with the adjustments provided by process_reference() to make links working, *
* the correct directory under dest_dir.					      *
******************************************************************************/

extern int yylex (void);
extern FILE *yyin, *yyout;
#ifdef array
extern char yytext[];
#else
extern char *yytext;
#endif
extern unsigned char i_a_tag, i_frame, found;
extern char title[], img_other[];
extern char *img_src;

void
process_html_file (NODE * node)
{
  char *http_base, *mod_base, *fname, *dup_or_url, *path;
  FILE *orig, *dest;
  size_t len = strlen (node->or_url) + strlen (DEFAULT_HTML) + 2;

  http_base = (char *) gt_malloc (len);
  dup_or_url = (char *) gt_malloc (len);
  mod_base = (char *) gt_malloc (strlen (node->r_url) + 1);

  cut_path (node->r_url, mod_base);

  strcpy (dup_or_url, node->or_url);
  check_html_url (dup_or_url, 0);
  cut_path (dup_or_url, http_base);

  path = (char *) gt_malloc (len_cache_dir + 1 + strlen (node->file_name) + 1);
  sprintf (path, "%s/%s", opt.cache_dir, node->file_name);
  orig = fopen (path, "r");

  if (orig != NULL)
    {
      enum rets
	{
	  TITLE = 1, BASE, REF, IMG
	};
      int what;
      
      fname = dirs_and_name (node->r_url);
      
      dest = fopen (fname, "w");
      if (dest == NULL)
	perror_exit ("\nnolce:\nCouldn't open html file for writing")
 
      yyin = orig;
      yyout = dest;

      i_a_tag = 0;
      found = 0;
      while ((what = yylex ()))
	switch (what)
	  {
	  case BASE:
	    http_base = (char *) gt_realloc (http_base, strlen (yytext) + strlen (DEFAULT_HTML) + 2);
	    if (!url_is_valid (yytext))
	      break;
	    strcpy (http_base, yytext);
	    check_html_url (http_base, 0);
	    *(strrchr (http_base, '/') + 1) = '\0';
	    break;
	  case TITLE:
	    if (title[0] != '\0')
	      {
		node->title = (char *) gt_realloc (node->title, strlen (title) + 1);
		strcpy (node->title, title);
	      }
	    break;
	  case REF:
	    if (i_frame)
	      process_reference (yytext, http_base, mod_base, fname, yyout, with_frames);
	    else if (i_a_tag)
	      found = !process_reference (yytext, http_base, mod_base, fname, yyout, normal);
	    else
	      process_reference (yytext, http_base, mod_base, fname, yyout, normal);
	    break;
	  case IMG:
	    process_reference (img_src, http_base, mod_base, fname, yyout, is_image);
	    break;
	  }
      fclose (orig);
      fclose (dest);

      free (http_base);
      free (mod_base);
    }
  else
    {
      node->type &= (~SHOW);
      perror ("\n\nnolce:\nWARNING: Couldn't open cache file for reading");
    }
  free (path);
}

/********************** Function process_reference ****************************
* Adjusts links. First transforms the link to an absolute url, with 	      *
* check_path(), then checks if that url corresponds to a file in the cache,   *
* with find_url(). If yes, the url is retransformed to a a relative path,     *
* with relative_position, then, if it doesn't refer to an html document,      *
* make_link() or copy_file() are called.				      *
* "ref" is the link to process, "base" is the BASE url of the document to     *
* whom the link belongs, "mod_base" is like the previous, but in the format   *
* of standardize(), "file_name" is the filename of the document, "out" the    *
* file with the adjusted version of the document, "operation"  is   used in   *
* in case that link refers to an image.				              *
******************************************************************************/

bool
process_reference (char *ref, char *base, char *mod_base,
		   char *file_name, FILE *out, int operation)
{
  NODE *point = NULL;
  char *cleaned_url, *abs_url, *a, *anchor = NULL;
  bool ret_value = 0;
  char *topic, *eff;
  size_t len = strlen (ref), lbase = strlen (base), ldh = 1 + strlen (DEFAULT_HTML),
    lname = strlen (file_name);

  if (!strcmp (ref, " photo.html"))
    printf ("\n*****************");
  if (ref && (eff = strtok (ref, "'  \"")))
    {
      topic = (char *) gt_malloc (lbase * 2 + len + ldh + lname + 1);
      strcpy (topic, eff);
    }
  else
    return 1;

  if (!strstr (topic, "internal-gopher"))
    {
      cleaned_url = (char *) gt_malloc (lbase + len + ldh + lname + 1);
      abs_url = (char *) gt_malloc (lbase + len + ldh + lname + 1);

      if ((a = strrchr (topic, '#')))
	{
	  anchor = (char *) gt_malloc (strlen (a) + 1);
	  strcpy (anchor, a);		/* If we have an anchor, we must */
	  if (a == topic)		/* eliminate it when checking if */
	    strcpy (topic, file_name);	/* the file is in the cache.     */
	  else
	    *a = '\0';
	}

      check_path (cleaned_url, base, topic);
      if (!url_is_valid (cleaned_url))
	return 1;
      strcpy (abs_url, cleaned_url);
      standardize (cleaned_url);
      point = find_url (root, cleaned_url);

      if (!point)
	{
	  char *sl;

	  sl = strrchr (cleaned_url, '/');
	  if (!strcmp (sl + 1, DEFAULT_HTML))
	    {
	      *sl = '\0';
	      point = find_url (root, cleaned_url);
	      if (!point)
		*sl = '/';
	    }
	}

      if (point)
	{
	  if (point->type & NEED_INDEX)
	    strcpy (cleaned_url, point->url_w_index);

	  relative_position (cleaned_url, mod_base, topic);

	  if (!(point->type & IS_HTML))
	    {
	      if (opt.var & LINK_FOR_IMAGES)
		ret_value = make_link (point->url, point->file_name);
	      else
		ret_value = copy_file (point->url, point->file_name);
	    }
	  else 
	    if (operation == with_frames)
	      point->type &= (~SHOW);
	  /* 
	   * When a link points to a document present in the cache, but not 
	   * processed because it doesn't satisfy the n_hour condition, 
	   * this action is taken.
	   */
	  if ((point->type & IS_HTML) && !(point->type & PR_MAIN))
	    {
	      if (!(opt.var & NO_LINKS))
		{
		  point->type |= PR_REL;
		  yet_to_process = 1;
		}
	      else
		ret_value = 1;
	    }

	  if (anchor)
	    {
	      strcat (topic, anchor);
	      free (anchor);
	    }
	}
      else
	ret_value = 1;

      if (ret_value)
	strcpy (topic, abs_url);

      free (cleaned_url);
      free (abs_url);
    }
  if (operation == is_image)
    {
      if ((opt.var & MISSING_IMAGES) || (!(opt.var & MISSING_IMAGES) && !ret_value))
	fprintf (out, "<IMG SRC=\"%s\" %s>", topic, img_other);
    }
  else
    fprintf (out, "\"%s\"", topic);

  free (topic);
  return ret_value;
}

/********************** Function make_link ************************************
* Makes a symbolic link to a non-html file in the cache.                      *
******************************************************************************/

bool
make_link (char *url, char *file)
{
  int ret;
  char *link_name, *orig_file;
  struct stat attr;

  link_name = dirs_and_name (url);
    
  orig_file = (char *) gt_malloc (len_cache_dir + 1 + strlen (file) + 1);
  sprintf (orig_file, "%s/%s", opt.cache_dir, file);

  ret = stat (orig_file, &attr);
    
  if (!ret)
    {
      unlink (link_name);
      ret = symlink (orig_file, link_name);
    }
  else
    {
      free (orig_file); 
      return 1;
    }

  free (orig_file); 
  
  if (ret)
    perror_exit ("\nnolce:\nProblems with symbolic links:"
	     " probably file system doesn't support them,\nexiting...");

  return 0;
}

/********************** Function copy_file ************************************
* Copies a non html file from the cache to dest_dir.			      *
******************************************************************************/

bool
copy_file (char *url, char *file)
{
  FILE *to, *from;
  char *dest_file, *orig_file;
  char buf[512];
  size_t char_size, n_read, n_write;

  dest_file = dirs_and_name (url);
  unlink (dest_file);
  to = fopen(dest_file, "w");
    
  orig_file = (char *) gt_malloc (len_cache_dir + 1 + strlen (file) + 1);
  sprintf (orig_file, "%s/%s", opt.cache_dir, file);
  from = fopen (orig_file, "r");
  
  if (!to || !from)
    return 1;
    
  free (orig_file);
  
  char_size = sizeof (char);
  
  while (1)
    {
      n_read = fread (buf, char_size, 512, from);
      if (!n_read) 
	break;
      n_write = fwrite (buf, char_size, n_read, to);
      if (n_read != n_write)
	perror_exit ("\nnolce:\nCould not copy images.");
    }	  
  
  fclose(from);
  fclose(to);

  return 0;
}

/********************** Function dirs_and_name ********************************
* Creates the directory structure under dest_dir which reflects the given     *
* url, and returns the file name. The current directory remains the last      *
* created.          						 	      *
******************************************************************************/

char *
dirs_and_name (char *url)
{
  char *base;
  char *dest_file, *dir;
  
  base = (char *) gt_malloc (strlen (url) + 1);
  dest_file = cut_path (url, base);
  /* 
   * Creation of directories structure. 
   */
  chdir (opt.dest_dir);

  dir = strtok (base, "/");	/* base contains at least a '/' */
  do
    {
      mkdir (dir, S_IRWXU | S_IRWXG | S_IRWXO);
      chdir (dir);
    }
  while ((dir = strtok (NULL, "/")) != NULL);
  
  free (base);
  return dest_file;
}
