/* -*- c -*- */
/******************************************************************************
 * $Id: npampconf.bison,v 1.2 1997/11/16 17:00:24 okke Exp $
 *
 * npampconf.y - small parser for .npamp config files
 * Copyright (C) 1997 Okke Timm 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 *****************************************************************************/

%{

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "amp.h"
#include "audio.h"
#include "npampconf.h"

void
set_debugflag(char* string)
{
  int idx;

  for(idx=0; debugLookup[idx].name != 0; idx++)
    if (strcmp(debugLookup[idx].name, string) == 0) {
      *(debugLookup[idx].var) = TRUE;
      break;
    }
  if (debugLookup[idx].name == 0) {
    warn("Debug option '%s' does not exist\n", string);
  }
}

%}

%union {
  int  val;
  char str[255];
}

%token <val>  NUM_T        /* Numbers (integers) */
%token <str>  STRING_T     /* Strings */
%token AUDIO_BUFFER_SIZE_T QUIET_T SHOW_TIME_T VOLUME_T SET_DEBUGFLAG_T
%token ON_T OFF_T          /* Toggles */

%right '='

%type <val> toggle         /* Toggles */

/******************************************************************************
 * Grammar follows 
 *****************************************************************************/

%%
input:    /* empty */
        | input statement
;

statement: ';'   
        | action ';'
;
     
action:   QUIET_T '=' toggle            { A_QUIET = $3; }
        | SHOW_TIME_T '=' toggle        { A_SHOW_TIME = $3; }
        | VOLUME_T '=' NUM_T            { 
                                          if ($3 > 100)  {
                                            A_SET_VOLUME = 100;
                                          } else if ($3 < 0) {
					    A_SET_VOLUME = 0;
                                          } else {
                                            A_SET_VOLUME = $3; 
					  }
                                        }
        | AUDIO_BUFFER_SIZE_T '=' NUM_T {
                                          if (($3 != 0) && ($3 < 64))  {
                                            AUDIO_BUFFER_SIZE = 64 * 1024;
                                          } else {
                                          AUDIO_BUFFER_SIZE = $3 * 1024;
                                          }
                                        }
        | SET_DEBUGFLAG_T STRING_T      { set_debugflag($2); }
;

toggle:   ON_T   { $$ = TRUE }
        | OFF_T  { $$ = FALSE }


/* End of grammar */
%%

struct tab {
  char *name;
  short token;
};

struct tab stmttab[] = {
  { "audio_buffer_size", AUDIO_BUFFER_SIZE_T },
  { "quiet", QUIET_T },
  { "show_time", SHOW_TIME_T },
  { "volume", VOLUME_T },
  { "debug", SET_DEBUGFLAG_T },
  { "on", ON_T },
  { "off", OFF_T },
  { NULL, 0 }
};

short
lookup(struct tab *p, char *cmd)
{
  for (; p->name != NULL; p++)
    if (strcmp(cmd, p->name) == 0)
      return (p->token);
  return (0);
}

/******************************************************************************
 * Lexical analyzer
 *****************************************************************************/

int
yylex ()
{
  int c;
     
 next_char:
  /* skip white space  */
  while ((c = fgetc(conf_file)) == ' ' || c == '\t' || c == '\n')
    ;

  /* skip comments */
  if (c == '#') {
    while ((c = fgetc(conf_file)) && c != '\n')
      ;
    goto next_char;
  }

  /* process numbers   */
  if (isdigit (c)) {
      ungetc (c, conf_file);
      fscanf (conf_file, "%i", &yylval.val);
      DB(conf, msg("Number: %i\n", yylval.val));
      return NUM_T;
    }

  /* process strings */
  if (c == '"') {
      int i=0;
      while ((c = fgetc(conf_file)) && c != '"')
	yylval.str[i++]=c;
      yylval.str[i]=0;
      DB(conf, msg("String: %s\n", yylval.str));
      return STRING_T;
    }
    
  /* process statements */
  if (isalpha(c)) {
      int i=0;
      short token;
      char stmtbuf[255];
      do {
	stmtbuf[i++]=c;
      } while ((c = fgetc(conf_file)) && (isalpha(c) || c == '_'));
      stmtbuf[i]=0;
      ungetc(c, conf_file);
      token = lookup(stmttab, stmtbuf);
      DB(conf, msg("Statement: %s Token: %i\n", stmtbuf, token)); 
      return token;
    }
  
  /* return end-of-file  */
  if (c == EOF)
    return 0;

  /* return single chars */
  return c;
}

/******************************************************************************
 * Error routine
 *****************************************************************************/

int
yyerror (char *s)  /* Called by yyparse on error */
{
  warn("Error in config file: %s\n", s);
}
