/*************************************************************************/
/*                                                                       */
/*        scheduler.ch   : Eric Lavillonniere 2005                       */
/*        company        : Mitsubishi Electric ITE                       */
/*                                                                       */
/*************************************************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation; either version 2.1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.
   
   You should have received a copy of the GNU Lesser General Public License
   along with this program; see the file COPYING. If not, write to 
   the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   */
language syndex;

// system includes
#include <map>

// private includes
#include "tablist.h"
#include "operatormedia.h"
#include "scheduler.h"
#include "syndex.h"
#include "individual.h"
#include "computeunit.h"
#include <list>

typedef enum { PARAMETERS, VARIABLE, SET_EMPTY, SET_FULL, IS_EMPTY, IS_FULL, FULL_SEMAPHORE, EMPTY_SEMAPHORE, ALLOC_VARIABLE, CHECK_EMITTED }   ModeType ;
typedef std::multimap<int, TransTrunk> ::iterator    TransTrunkIterator ;
typedef std::multimap<int, TransTrunk> ::const_iterator    TransTrunkConstIterator ;

// TreatedInputVar
bool TreatedInputVar ( ComputeUnit &task, const EString &channel )
{
    return false ;
}

// ComputeNameVar : compute the name of an allocation variable
// parameters :
//               refVar : the referencing operation
//               channel : channel for this var
// return : name of variable
static  EString ComputeNameVar ( PTREE refVar, PTREE channel )
{
    EString                     nameVar ;
    std::list<PTREE>            names ;
    std::list<PTREE> ::iterator iterName ;
    
    while ( refVar ) {
        switch ( refVar ) {
            case <ALGORITHM> : 
            case <SENSOR> : 
            case <ACTUATOR> : 
            case <MEMORY> : 
            case <CONSTANT> : 
                {
                    names.push_front(Value(refVar));
                }
                break ;
            default : ;
        }
        refVar = refVar ^ ;
    }
    
    bool    first = true ;
    
    names.pop_front();
    names.push_back(channel);
    for ( iterName = names.begin() ; iterName != names.end() ; iterName++ ) {
        if ( !first ) 
            nameVar << "_";
        else 
            first = false ;
        nameVar << Value(*iterName);
    }
    return nameVar ;
}

static  int IsImplodeExplode ( ComputeUnit &task )
{
    {
        PTREE   type = task.Reference();
        VString explode = "Explode";
        VString implode = "Implode";
        if ( EString(Value(type)).substring(0, explode.length()) == explode ) 
            return 1 ;
        if ( EString(Value(type)).substring(0, implode.length()) == implode ) 
            return 2 ;
    }
    return false ;
}

static  void SetPrefixSuffix ( ModeType modeType, EString &prefix, EString &suffix )
{
    switch ( modeType ) {
        case IS_FULL : 
            {
                suffix = "_full";
                prefix = "Suc0_";
            }
            break ;
        case EMPTY_SEMAPHORE : 
        case SET_EMPTY : 
            {
                suffix = "_empty";
                prefix = "Pre0_";
            }
            break ;
        case IS_EMPTY : 
            {
                suffix = "_empty";
                prefix = "Suc0_";
            }
            break ;
        case FULL_SEMAPHORE : 
        case SET_FULL : 
            {
                suffix = "_full";
                prefix = "Pre0_";
            }
            break ;
        default : prefix = suffix = "Unknown-type";
    }
}

static  bool CheckDataLink ( ComputeUnit &task, ComputeUnit &currUnit, EString channel = "" )
{
    
    // look on which channel it is connected
    std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
    bool                                                    data = false ; // no data if condition on box not real data (no real channel : name == "")
    bool                                                    loop = true ;
    
    for ( iterIn = currUnit.InEndConnectionsRef().begin() ; loop && iterIn != currUnit.InEndConnectionsRef().end() ; iterIn++ ) {
        TransUnit   *pTransUnit = (TransUnit *)((*iterIn).second);
        
        // if not selected channel continue
        if ( channel != "" && channel != (*iterIn).first ) 
            continue ;
        
        // search for input
        unsigned int    indexComm ;
        ComputeUnit     *pUpUnitIn ;
        for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
            pUpUnitIn = pTransUnit->ConnectedOut()[indexComm];
            if ( pUpUnitIn == &task ) {
                PTREE                       treeName = <IDENT,PTREE((*iterIn).first)>;
                std::vector<ComputeUnit *>  connected ;
                currUnit.SearchConnected(treeName, connected, 0, false);
                if ( !connected.empty() ) 
                    data = true ;
                loop = false ;
                break ;
            }
        }
    }
    return data ;
}

static void ComputeParameters (OperatorMedia &operatorMedia, ComputeUnit &task, ComputeUnit &taskDest, ComputeUnit &last
    , ModeType modeType = PARAMETERS, std::set<EString> *pListDone = 0, EString *pRetVal = 0, TransUnit *pSelectedUnit = 0) ;
static void ComputeParameters (OperatorMedia &operatorMedia, ComputeUnit &task, bool out = false, ModeType modeType = PARAMETERS
    , EString channel = "", std::set<EString> *pListDone = 0, EString *pRetVal = 0) ;

static  void ComputeParametersDown ( OperatorMedia &operatorMedia, ComputeUnit &top, ComputeUnit &task, ComputeUnit &last
    , ModeType modeType = PARAMETERS, EString channel = "", std::set<EString> *pListDone = 0, EString *pRetVal = 0 )
{
    
    // create the variable
    {
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        for ( iterOut = task.OutEndConnectionsRef().begin() ; iterOut != task.OutEndConnectionsRef().end() ; iterOut++ ) {
            TransUnit   *pTransUnit = (TransUnit *)((*iterOut).second);
            PTREE       type = pTransUnit->Reference();
            
            // if channel is indicated treat only this one
            if ( channel != "" && channel != (*iterOut).first ) 
                continue ;
            
            // explore all attached application
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                ComputeUnit                                             *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                
                // search for name of channel
                EString                                                 channelName ;
                std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
                bool                                                    foundChannel = false ;
                for ( iterIn = pComputeUnit->InEndConnectionsRef().begin() ; !foundChannel && iterIn != pComputeUnit->InEndConnectionsRef().end() ; 
                        iterIn++ ) {
                    TransUnit   *pTransUnitDown = (TransUnit *)((*iterIn).second);
                    if ( pTransUnit == pTransUnitDown ) {
                        channelName = (*iterIn).first ;
                        foundChannel = true ;
                    }
                }
                if ( !foundChannel ) {
                    std::cerr << "Channel not found \n";
                }
                
                // treat
                if ( !IsImplodeExplode(*pComputeUnit) ) {
                    if ( modeType == IS_FULL
                            || modeType == IS_EMPTY
                            || modeType == SET_FULL
                            || modeType == SET_EMPTY
                            || modeType == FULL_SEMAPHORE
                            || modeType == EMPTY_SEMAPHORE ) {
                        if ( &pComputeUnit->Proc() == &top.Proc() ) 
                            continue ;
                    }
                    
                    // display
                    ComputeParameters(operatorMedia, top, *pComputeUnit, (&last == &top ? *pComputeUnit : last), modeType, pListDone, pRetVal
                        , pTransUnit);
                } else {
                    
                    // if explode and not real data link it is a condition and it will be done directly for box under
                    if ( !CheckDataLink(task, *pComputeUnit, channelName) ) 
                        continue ;
                    
                    // creates as many semaphores as duplication of var
                    std::map<EString, ComputeUnit *, LessString> ::iterator iterOutExplode ;
                    for ( iterOutExplode = pComputeUnit->OutEndConnectionsRef().begin() ; 
                            iterOutExplode != pComputeUnit->OutEndConnectionsRef().end() ; 
                            iterOutExplode++ ) {
                        TransUnit       *pTransUnit = (TransUnit *)((*iterOutExplode).second);
                        unsigned int    indexComm1 ;
                        for ( indexComm1 = 0 ; indexComm1 < pTransUnit->ConnectedIn().size() ; indexComm1++ ) {
                            ComputeUnit                                             *pDownUnit = pTransUnit->ConnectedIn()[indexComm1];
                            
                            // search for name of channel
                            EString                                                 channelName ;
                            bool                                                    foundChannel = false ;
                            std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
                            for ( iterIn = pDownUnit->InEndConnectionsRef().begin() ; 
                                    !foundChannel && iterIn != pDownUnit->InEndConnectionsRef().end() ; 
                                    iterIn++ ) {
                                TransUnit   *pTransUnitDown = (TransUnit *)((*iterIn).second);
                                if ( pTransUnit == pTransUnitDown ) {
                                    channelName = (*iterIn).first ;
                                    foundChannel = true ;
                                }
                            }
                            
                            // some checks
                            if ( !foundChannel ) {
                                std::cerr << "Channel not found \n";
                                return ;
                            }
#                           if 0
                                if ( IsImplodeExplode(*pDownUnit) ) {
                                    std::cerr << "Double explode \n";
                                    return ;
                                }
#                           endif
                            if ( IsImplodeExplode(*pDownUnit) ) {
                                
                                // ComputeParametersDown(*pDownUnit, modeType);
                                ComputeParametersDown(operatorMedia, top, *pDownUnit, (&last == &top ? *pDownUnit : last), modeType
                                    , "" /*channelName*/ , pListDone, pRetVal);
                            } else {
                                
                                // display
                                ComputeParameters(operatorMedia, top, *pComputeUnit, (&last == &top ? *pComputeUnit : last), modeType, pListDone
                                    , pRetVal);
                            }
                        }
                    }
                }
            }
        }
    }
}

// ComputeParameters : compute parameters for calling operation knowning the predecessor
static  void ComputeParameters ( OperatorMedia &operatorMedia, ComputeUnit &task, ComputeUnit &taskDest, ComputeUnit &last, ModeType modeType
    , std::set<EString> *pListDone, EString *pRetVal, TransUnit *pSelectedTransUnit )
{
    
    // if no real data transfer between the two of them do nothing
    // if ( !CheckDataLink(task, taskDest) ) 
    //    return ;
    // explode box
    if ( IsImplodeExplode(task) == 1 ) {
        
        // search for the box uplevel
        ComputeUnit                                             *pBeforeExplodeUnit ;
        TransUnit                                               *pTransUnit = 0 ;
        
        // search for the task before explode
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        for ( iterIn = task.InEndConnectionsRef().begin() ; iterIn != task.InEndConnectionsRef().end() ; iterIn++ ) {
            pTransUnit = (TransUnit *)((*iterIn).second);
            
            // search for input
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                pBeforeExplodeUnit = pTransUnit->ConnectedOut()[indexComm];
                
                // there should be only one input
                break ;
            }
        }
        
        // search for the index in explode 
        // search for the box uplevel
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        unsigned int                                            indexChannel = 0 ;
        unsigned int                                            currIndex ;
        PTREE                                                   type ;
        for ( iterOut = task.OutEndConnectionsRef().begin(), currIndex = 1 ; !indexChannel && iterOut != task.OutEndConnectionsRef().end() ; 
                iterOut++, currIndex++ ) {
            TransUnit   *pTransUnit = 0 ;
            pTransUnit = (TransUnit *)((*iterOut).second);
            type = (*iterOut).second->Reference();
            
            // search for input
            unsigned int    indexComm ;
            ComputeUnit     *pAfterExplodeUnit ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                pAfterExplodeUnit = pTransUnit->ConnectedIn()[indexComm];
                if ( pAfterExplodeUnit == &taskDest ) 
                    indexChannel = currIndex ;
            }
        }
        
        // generates var
        switch ( modeType ) {
            case PARAMETERS : 
            case VARIABLE : 
                {
                    if ( !pRetVal ) {
                        "Access_(";
                        ComputeParameters(operatorMedia, *pBeforeExplodeUnit, taskDest, task, modeType);
                        ", sizeof(" PrintString(Value(type)) ")," PrintString(CompactItoa(indexChannel)) ")";
                    } else {
                        EString retVal1 ;
                        ComputeParameters(operatorMedia, *pBeforeExplodeUnit, taskDest, task, VARIABLE, 0, &retVal1);
                        *pRetVal = retVal1 ;
                    }
                }
                break ;
            case CHECK_EMITTED : 
                {
                    EString name ;
                    ComputeParameters(operatorMedia, *pBeforeExplodeUnit, taskDest, task, VARIABLE, 0, &name);
                    name << "_" << CompactItoa(indexChannel);
                    if ( !pListDone || pListDone->empty() || pListDone->find(name) == pListDone->end() ) {
                        *pRetVal = name ;
                        if ( pListDone ) 
                            pListDone->insert(name);
                    }
                }
                break ;
            case ALLOC_VARIABLE : 
                {
                    
                    // if explode alloc all buffer
                    ComputeParameters(operatorMedia, *pBeforeExplodeUnit, taskDest, task, ALLOC_VARIABLE, pListDone);
                }
                break ;
            case IS_FULL : ;
            case IS_EMPTY : ;
            case SET_EMPTY : 
            case SET_FULL : 
            case FULL_SEMAPHORE : 
            case EMPTY_SEMAPHORE : 
                {
                    EString suffix ;
                    EString prefix ;
                    SetPrefixSuffix(modeType, prefix, suffix);
                    
                    // compute name
                    EString name ;
                    ComputeParameters(operatorMedia, *pBeforeExplodeUnit, taskDest, task, VARIABLE, 0, &name);
                    name << "_";
                    if ( &operatorMedia == &taskDest.Proc() ) 
                        name << taskDest.Name();
                    else {
                        name << taskDest.Proc().Name();
                    }
                    name << "_" << CompactItoa(indexChannel) << suffix ;
                    
                    // display
                    switch ( modeType ) {
                        case IS_FULL : 
                        case SET_EMPTY : 
                        case IS_EMPTY : 
                        case SET_FULL : 
                            {
                                if ( !pListDone ) {
                                    PrintString(prefix) "(" PrintString(name) ")" <NL>
                                } else {
                                    if ( pListDone->empty() || pListDone->find(name) == pListDone->end() ) {
                                        PrintString(prefix) "(" PrintString(name) ")" <NL>
                                        pListDone->insert(name);
                                    }
                                }
                            }
                            break ;
                        case FULL_SEMAPHORE : 
                        case EMPTY_SEMAPHORE : 
                            {
                                if ( !pListDone ) {
                                    "," <NL>
                                    PrintString(name);
                                } else if ( pListDone->empty() || pListDone->find(name) == pListDone->end() ) {
                                    if ( !pListDone->empty() ) {
                                        "," <NL>
                                    }
                                    PrintString(name);
                                    pListDone->insert(name);
                                }
                            }
                            break ;
                        default : ;
                    }
                }
                break ;
            default : ;
        }
        
        // finish
        return ;
    }
    
    // implode box
    if ( IsImplodeExplode(task) == 2 && (modeType == IS_FULL || modeType == IS_EMPTY || modeType == SET_FULL || modeType == SET_EMPTY) ) {
        
        // search for the box uplevel
        ComputeUnit                                             *pUpUnit ;
        
        // search for the task before implode
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        for ( iterIn = task.InEndConnectionsRef().begin() ; iterIn != task.InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            EString         channelName ;
            
            // search for input
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                pUpUnit = pTransUnit->ConnectedOut()[indexComm];
                
                // there should be only one input
                break ;
            }
            
            // search for name of channel
            std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
            for ( iterOut = pUpUnit->OutEndConnectionsRef().begin() ; channelName == "" && iterOut != pUpUnit->OutEndConnectionsRef().end() ; 
                    iterOut++ ) {
                TransUnit   *pUpTransUnit = (TransUnit *)((*iterOut).second);
                
                // search for input
                if ( pUpTransUnit == pTransUnit ) {
                    channelName = (*iterOut).first ;
                }
            }
            
            // if no real data link forget
            if ( !CheckDataLink(*pUpUnit, task, (*iterIn).first) ) 
                continue ;
            
            // generates var
            switch ( modeType ) {
                case IS_FULL : ;
                case IS_EMPTY : ;
                case SET_EMPTY : 
                case SET_FULL : 
                case FULL_SEMAPHORE : 
                case EMPTY_SEMAPHORE : 
                    {
                        EString suffix ;
                        EString prefix ;
                        SetPrefixSuffix(modeType, prefix, suffix);
                        
                        // compute name
                        EString name ;
                        name = EString("") << pUpUnit->Name() << "_" << channelName << "_";
                        if ( &operatorMedia == &taskDest.Proc() ) 
                            name << taskDest.Name();
                        else {
                            name << taskDest.Proc().Name();
                        }
                        name << suffix ;
                        
                        // display
                        switch ( modeType ) {
                            case IS_FULL : 
                            case SET_EMPTY : 
                            case IS_EMPTY : 
                            case SET_FULL : 
                                {
                                    if ( !pListDone ) {
                                        PrintString(prefix) "(" PrintString(name) ")" <NL>
                                    } else {
                                        if ( pListDone->empty() || pListDone->find(name) == pListDone->end() ) {
                                            PrintString(prefix) "(" PrintString(name) ")" <NL>
                                            pListDone->insert(name);
                                        }
                                    }
                                }
                                break ;
                            case FULL_SEMAPHORE : 
                            case EMPTY_SEMAPHORE : 
                                {
                                    if ( !pListDone ) {
                                        "," <NL>
                                        PrintString(name);
                                    } else if ( pListDone->empty() || pListDone->find(name) == pListDone->end() ) {
                                        if ( !pListDone->empty() ) {
                                            "," <NL>
                                        }
                                        PrintString(name);
                                        pListDone->insert(name);
                                    }
                                }
                                break ;
                            default : ;
                        }
                    }
                    break ;
                default : ;
            }
        }
        
        // finish
        return ;
    }
    
    // standard box or implode box
    {
        
        // search for name of channel
        EString                                                 channelName ;
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        TransUnit                                               *pTransUnit ;
        bool                                                    isImplode = false ;
        for ( iterOut = task.OutEndConnectionsRef().begin() ; iterOut != task.OutEndConnectionsRef().end() && channelName == "" ; iterOut++ ) {
            isImplode = false ;
            pTransUnit = (TransUnit *)((*iterOut).second);
            
            // search for channel
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() && channelName == "" ; indexComm++ ) {
                ComputeUnit *pDownUnit = pTransUnit->ConnectedIn()[indexComm];
                if ( pDownUnit == &last && (pSelectedTransUnit == 0 || pSelectedTransUnit == pTransUnit) ) {
                    channelName = (*iterOut).first ;
                }
                isImplode |= IsImplodeExplode(*pDownUnit) == 2 && CheckDataLink(task, *pDownUnit);
            }
            
            // if channel found exit
            if ( channelName != "" ) 
                break ;
        }
        
        // compute name of this variable
        EString nameVar = task.Name();
        nameVar << "_" << channelName ;
        
        // generate the var
        switch ( modeType ) {
            case PARAMETERS : 
            case VARIABLE : 
                {
                    if ( !pRetVal ) {
                        
                        // display it
                        if ( isImplode ) {
                            ComputeParameters(operatorMedia, task, true, modeType, channelName);
                        } else 
                            PrintString(nameVar) 
                    } else {
                        *pRetVal = nameVar ;
                    }
                }
                break ;
            case ALLOC_VARIABLE : 
                {
                    PTREE   type ;
                    if ( isImplode ) {
                        ComputeParameters(operatorMedia, task, true, modeType, channelName, pListDone);
                    } else if ( !pListDone || pListDone->empty() || pListDone->find(nameVar) == pListDone->end() ) {
                        PTREE   taskReference = task.Reference();
                        type = pTransUnit->Reference();
                        type == <IN,type>;
                        type == <OUT,type>;
                        if ( taskReference == <MEMORY> ) {
                            "alloc_(" PrintString(Value(type)) ", " PrintString(nameVar) <SEPO> "_buffer, " PrintString(Value(type [2])) ")" <NL>
                            "alias_(" PrintString(nameVar) ", " PrintString(nameVar) <SEPO> "_buffer,0, " PrintString(Value(type [2])) ")" <NL>
                        } else {
                            "alloc_(" PrintString(Value(type)) ", " PrintString(nameVar) ", " PrintString(Value(type [2])) ")" <NL>
                        }
                        if ( pListDone ) 
                            pListDone->insert(nameVar);
                    }
                }
                break ;
            case CHECK_EMITTED : 
                {
                    if ( isImplode ) {
                        ComputeParameters(operatorMedia, task, true, modeType, channelName, pListDone, pRetVal);
                    } else if ( !pListDone || pListDone->empty() || pListDone->find(nameVar) == pListDone->end() ) {
                        *pRetVal = nameVar ;
                        if ( pListDone ) 
                            pListDone->insert(nameVar);
                    }
                }
                break ;
            case IS_FULL : ;
            case SET_EMPTY : 
            case IS_EMPTY : ;
            case SET_FULL : 
            case FULL_SEMAPHORE : 
            case EMPTY_SEMAPHORE : 
                {
                    EString suffix ;
                    EString prefix ;
                    switch ( modeType ) {
                        case IS_FULL : 
                            {
                                suffix = "_full";
                                prefix = "Suc0_";
                            }
                            break ;
                        case EMPTY_SEMAPHORE : 
                        case SET_EMPTY : 
                            {
                                suffix = "_empty";
                                prefix = "Pre0_";
                            }
                            break ;
                        case IS_EMPTY : 
                            {
                                suffix = "_empty";
                                prefix = "Suc0_";
                            }
                            break ;
                        case FULL_SEMAPHORE : 
                        case SET_FULL : 
                            {
                                suffix = "_full";
                                prefix = "Pre0_";
                            }
                            break ;
                        default : ;
                    }
                    
                    // compute the list of names
                    std::set<EString>   names ;
                    if ( IsImplodeExplode(taskDest) == 2 ) {
                        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
                        TransUnit                                               *pTransUnit ;
                        for ( iterOut = taskDest.OutEndConnectionsRef().begin() ; iterOut != taskDest.OutEndConnectionsRef().end() ; iterOut++ ) {
                            pTransUnit = (TransUnit *)((*iterOut).second);
                            
                            // search for channel
                            unsigned int    indexComm ;
                            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                                ComputeUnit *pDownUnit = pTransUnit->ConnectedIn()[indexComm];
                                if ( IsImplodeExplode(*pDownUnit) == 2 && CheckDataLink(task, *pDownUnit) ) {
                                    std::cout << "Error cascaded implode \n";
                                }
                                
                                // compute name
                                EString name ;
                                name = nameVar << "_";
                                if ( &operatorMedia == &pDownUnit->Proc() ) 
                                    name << pDownUnit->Name();
                                else 
                                    name << pDownUnit->Proc().Name();
                                name << suffix ;
                                names.insert(name);
                            }
                        }
                    } else {
                        EString name ;
                        name = nameVar << "_";
                        if ( &operatorMedia == &taskDest.Proc() ) 
                            name << taskDest.Name();
                        else 
                            name << taskDest.Proc().Name();
                        name << suffix ;
                        names.insert(name);
                    }
                    
                    // display for the list of names
                    std::set<EString> ::iterator    iterNames ;
                    for ( iterNames = names.begin() ; iterNames != names.end() ; iterNames++ ) {
                        
                        // display
                        switch ( modeType ) {
                            case IS_FULL : 
                            case SET_EMPTY : 
                            case IS_EMPTY : 
                            case SET_FULL : 
                                {
                                    if ( !pListDone ) {
                                        PrintString(prefix) "(" PrintString(*iterNames) ")" <NL>
                                    } else {
                                        if ( pListDone->empty() || pListDone->find(*iterNames) == pListDone->end() ) {
                                            PrintString(prefix) "(" PrintString(*iterNames) ")" <NL>
                                            pListDone->insert(*iterNames);
                                        }
                                    }
                                }
                                break ;
                            case FULL_SEMAPHORE : 
                            case EMPTY_SEMAPHORE : 
                                {
                                    if ( !pListDone ) {
                                        "," <NL>
                                        PrintString(*iterNames);
                                    } else if ( pListDone->empty() || pListDone->find(*iterNames) == pListDone->end() ) {
                                        if ( !pListDone->empty() ) {
                                            "," <NL>
                                        }
                                        PrintString(*iterNames);
                                        pListDone->insert(*iterNames);
                                    }
                                }
                                break ;
                            default : ;
                        }
                    }
                }
                break ;
            default : ;
        }
    }
}

// ComputeParameters : compute parameters for calling operation
// parameters :
//               channel : display only this channel variable (for send and recv)
void ComputeParameters ( OperatorMedia &operatorMedia, ComputeUnit &task, bool out, ModeType modeType, EString channel, std::set<EString> *pListDone
    , EString *pRetVal )
{
    if ( !out ) {
        
        // in parameters
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        for ( iterIn = task.InEndConnectionsRef().begin() ; iterIn != task.InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // compute unit
            unsigned int    indexComm ;
            ComputeUnit     *pComputeUnit ;
            bool            outsideInput = false ;
            int             explodeImplode = 0 ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                
                // there should be only one input
                break ;
            }
            
            // if no real data transfer between the two of them do nothing
            // if ( !CheckDataLink(*pComputeUnit, task) ) 
            //    continue ;
            // compute parameters knowing the predecessor
            if ( channel == "" || channel == (*iterIn).first ) {
                if ( modeType == PARAMETERS ) {
                    
                    // if only condition do not generate : no real data transfer
                    if ( !CheckDataLink(*pComputeUnit, task, (*iterIn).first) ) 
                        continue ;
                    
                    // --
                    ", ";
                }
                if ( modeType == IS_FULL
                        || modeType == IS_EMPTY
                        || modeType == SET_FULL
                        || modeType == SET_EMPTY
                        || modeType == FULL_SEMAPHORE
                        || modeType == EMPTY_SEMAPHORE ) {
                    if ( &pComputeUnit->Proc() == &task.Proc() ) 
                        continue ;
                }
                ComputeParameters(operatorMedia, *pComputeUnit, task, task, modeType, pListDone, pRetVal, pTransUnit);
            }
        }
    } else {
        
        // out parameters
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        for ( iterOut = task.OutEndConnectionsRef().begin() ; iterOut != task.OutEndConnectionsRef().end() ; iterOut++ ) {
            TransUnit   *pTransUnit = (TransUnit *)((*iterOut).second);
            
            // if not good channel go on
            if ( channel != "" && channel != (*iterOut).first ) 
                continue ;
            
            // search if implode explode downstream
            unsigned int    indexComm ;
            ComputeUnit     *pImplodeUnit ;
            ComputeUnit     *pCurrUnit ;
            bool            outsideInput = false ;
            int             nbImplode = 0 ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                pCurrUnit = pTransUnit->ConnectedIn()[indexComm];
                if ( IsImplodeExplode(*pCurrUnit) == 2 ) {
                    
                    // bool rejected
                    bool    rejected = !CheckDataLink(task, *pCurrUnit);
                    
                    // if rejected do not take into account
                    if ( rejected ) 
                        continue ;
                    
                    // tag this implode unit
                    nbImplode++ ;
                    pImplodeUnit = pCurrUnit ;
                }
            }
            
            // if more than one implode not treated yet
            if ( nbImplode > 1 ) {
                EString message("Too many implode not treated yet !!!!!!!!!!!!!! \n");
                std::cerr << message.c_str();
            }
            
            // if simple parameter display 
            if ( !nbImplode || !(modeType == VARIABLE || modeType == PARAMETERS || modeType == ALLOC_VARIABLE || modeType == CHECK_EMITTED) ) {
                
                // compute name of this variable
                EString nameVar = task.Name();
                nameVar << "_" << (*iterOut).first ;
                
                // display it
                if ( channel == "" || channel == (*iterOut).first ) {
                    switch ( modeType ) {
                        case IS_FULL : 
                        case SET_EMPTY : 
                        case IS_EMPTY : 
                        case SET_FULL : 
                        case FULL_SEMAPHORE : 
                        case EMPTY_SEMAPHORE : 
                            {
                                
                                // propagates downstream
                                ComputeParametersDown(operatorMedia, task, task, task, modeType, (*iterOut).first, pListDone, pRetVal);
                            }
                            break ;
                        case ALLOC_VARIABLE : 
                            {
                                PTREE   type ;
                                PTREE   taskReference = task.Reference();
                                if ( !pListDone || pListDone->empty() || pListDone->find(nameVar) == pListDone->end() ) {
                                    type = pTransUnit->Reference();
                                    type == <IN,type>;
                                    type == <OUT,type>;
                                    if ( taskReference == <MEMORY> ) {
                                        "alloc_(" PrintString(Value(type)) ", " PrintString(nameVar) <SEPO> "_buffer, " PrintString(Value(type [2])) 
                                            ")" <NL>
                                        "alias_(" PrintString(nameVar) ", " PrintString(nameVar) <SEPO> "_buffer,0, " PrintString(Value(type [2])) ")" 
                                            <NL>
                                    } else {
                                        "alloc_(" PrintString(Value(type)) ", " PrintString(nameVar) ", " PrintString(Value(type [2])) ")" <NL>
                                    }
                                    if ( pListDone ) 
                                        pListDone->insert(nameVar);
                                }
                            }
                            break ;
                        case CHECK_EMITTED : 
                            {
                                std::cerr << " !! Error not implemented check emitted \n";
                            }
                            break ;
                        default : 
                            {
                                if ( modeType == PARAMETERS ) 
                                    ", " 
                                PrintString(nameVar);
                            }
                            break ;
                    }
                }
            } else {
                
                // search channel name and type for output of implode box
                PTREE                                                   type ;
                PTREE                                                   typeOut ;
                EString                                                 channelName ;
                std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
                for ( iterOut = pImplodeUnit->OutEndConnectionsRef().begin() ; iterOut != pImplodeUnit->OutEndConnectionsRef().end() ; iterOut++ ) {
                    channelName = (*iterOut).first ;
                    typeOut = (*iterOut).second->Reference();
                }
                
                // compute index for this input
                std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
                unsigned int                                            indexChannel = 0 ;
                unsigned int                                            currIndex ;
                for ( iterIn = pImplodeUnit->InEndConnectionsRef().begin(), currIndex = 1 ; 
                        !indexChannel && iterIn != pImplodeUnit->InEndConnectionsRef().end() ; 
                        iterIn++, currIndex++ ) {
                    TransUnit   *pTransUnitDown = (TransUnit *)((*iterIn).second);
                    type = (*iterIn).second->Reference();
                    
                    // if not data channel do not take it (condition)
                    {
                        PTREE                       treeName = <IDENT,PTREE((*iterIn).first)>;
                        std::vector<ComputeUnit *>  connected ;
                        pImplodeUnit->SearchConnected(treeName, connected, 0, false);
                        if ( connected.empty() ) {
                            currIndex-- ;
                            continue ;
                        }
                    }
                    
                    // search for input
                    if ( pTransUnitDown == pTransUnit ) 
                        indexChannel = currIndex ;
                }
                
                // compute name of this variable
                EString nameVar = pImplodeUnit->Name();
                nameVar << "_" << channelName ;
                
                // generates var
                switch ( modeType ) {
                    case VARIABLE : 
                    case PARAMETERS : 
                        {
                            if ( channel == "" ) {
                                ", ";
                            }
                            "Access_(" PrintString(nameVar) ", sizeof(" PrintString(Value(type)) ")," PrintString(CompactItoa(indexChannel)) ")";
                        }
                        break ;
                    case ALLOC_VARIABLE : 
                        {
                            PTREE   taskReference = task.Reference();
                            if ( !pListDone || pListDone->empty() || pListDone->find(nameVar) == pListDone->end() ) {
                                typeOut == <IN,typeOut>;
                                typeOut == <OUT,typeOut>;
                                if ( taskReference == <MEMORY> ) {
                                    "alloc_(" PrintString(Value(type)) ", " PrintString(nameVar) <SEPO> "_buffer, " PrintString(Value(type [2])) ")" 
                                        <NL>
                                    "alias_(" PrintString(nameVar) ", " PrintString(nameVar) <SEPO> "_buffer,0, " PrintString(Value(type [2])) ")" 
                                        <NL>
                                } else {
                                    "alloc_(" PrintString(Value(typeOut)) ", " PrintString(nameVar) ", " PrintString(Value(typeOut [2])) ")" <NL>
                                }
                                if ( pListDone ) 
                                    pListDone->insert(nameVar);
                            }
                        }
                        break ;
                    case CHECK_EMITTED : 
                        {
                            nameVar << "_" << CompactItoa(indexChannel);
                            if ( !pListDone || pListDone->empty() || pListDone->find(nameVar) == pListDone->end() ) {
                                if ( pListDone ) 
                                    pListDone->insert(nameVar);
                                *pRetVal = nameVar ;
                            }
                        }
                        break ;
                    default : ;
                }
            }
        }
    }
}

// GenerateFunctionModel : generate a model for each function
void Scheduler::GenerateArchitecture ()
{
    int     fileOutput ;
    EString fileName(Name());
    EString outputString ;
    int     oldOutput = output ;
    
    fileName << ".m4";
    if ( (fileOutput = _open(fileName.c_str(), O_CREAT | O_RDWR | O_BINARY | O_TRUNC, 0666 /*S_IWRITE | S_IREAD*/ )) > 0 ) {
        output = fileOutput ;
        "include(syndex.m4m)dnl" <NL>
        "architecture_(" PrintString(EString(Name())) "," <NL>
        "GenticSyndex" <NL>
        ")" <NL>
        <NL,2>
        
        // generate data for each operator
        std::vector<OperatorMedia *> ::iterator iterOperator ;
        for ( iterOperator = pvOperators.begin() ; iterOperator != pvOperators.end() ; iterOperator++ ) {
            if ( !(*iterOperator)->Planning().empty() ) {
                EString processorName = (*iterOperator)->Name();
                PTREE   reference = (*iterOperator)->Reference();
                "processor_(" PrintString(Value(reference)) "," PrintString(processorName);
                PTREE   gate, name, type ;
                foreach (<GATE>,reference,{
                    gate = for_elem ;
                    if ( gate == <,type,name> ) {
                        "," PrintString(Value(type)) "," PrintString(Value(name));
                    }
                })
                ")" <NL>
            }
        }
        <NL,2>
        
        // generate the links on the operator
        // if box already exists it is an error
        std::vector<OperatorMedia *> ::iterator iterMedia ;
        for ( iterMedia = Links().begin() ; iterMedia != Links().end() ; iterMedia++ ) {
            std::vector<OperatorMedia *> ::iterator iterOperator ;
            PTREE                                   reference = (*iterMedia)->Reference();
            "connect_(" PrintString(Value(reference)) "," PrintString((*iterMedia)->Name());
            for ( iterOperator = (*iterMedia)->Linked().begin() ; iterOperator != (*iterMedia)->Linked().end() ; iterOperator++ ) {
                "," PrintString((*iterOperator)->Name()) "," PrintString(((LinkUnit *)(*iterMedia))->ChannelByOperator()[(*iterOperator)->Name()]);
            }
            ")" <NL>
        }
        <NL,2>
        
        // end architecture
        "endarchitecture_" <NL,2>
        
        // close file
        _close(fileOutput);
    }
    output = oldOutput ;
}

// GenerateFunctionModel : generate a model for each function
void Scheduler::GenerateFunctionModel ()
{
    int                 fileOutput ;
    EString             fileName(Name());
    std::set<EString>   generated ;
    int                 oldOutput = output ;
    
    fileName << "_functions.m4x";
    if ( (fileOutput = _open(fileName.c_str(), O_CREAT | O_RDWR | O_BINARY | O_TRUNC, 0666 /*S_IWRITE | S_IREAD*/ )) > 0 ) {
        output = fileOutput ;
        
        // create a set with all the task
        std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo ;
        for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
            EString entry ;
            int     nbParam = 1 ;
            bool    first = true ;
            PTREE   refTree ;
            PTREE   listParameters ;
            EString refName ;
            refTree = (*iterAlgo).second->Reference();
            refName = Value(refTree);
            if ( generated.find(refName) == generated.end() ) {
                generated.insert(refName);
                "def(`" PrintString(Value(refTree)) "',`ifelse(" <NL>
                    <T> {{
                            {
                                {
                                    
                                    // for init
                                    "MGC,INIT, `_(" PrintString(Value(refTree)) <SEPO> "_FunctInit (";
                                    if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                                        PTREE   elem ;
                                        while ( elem = nextl(listParameters) ) {
                                            if ( !first ) 
                                                "," 
                                            else 
                                                first = false ;
                                            "$" <SEPO> PrintString(EString(nbParam++));
                                        }
                                    }
                                    if ( (*iterAlgo).second->Type() == ComputeUnit::MemoryUnit ) {
                                        foreach (<OUT>,refTree,{
                                            if ( !first ) 
                                                "," 
                                            else 
                                                first = false ;
                                            "$" <SEPO> PrintString(EString(nbParam++));
                                        })
                                    }
                                    ");)',";
                                }
                                <NL>
                                first = true ;
                                nbParam = 1 ;
                                {
                                    
                                    // for loop
                                    // for init
                                    "MGC,LOOP, `_(" PrintString(Value(refTree)) <SEPO> "_FunctLoop (";
                                    if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                                        PTREE   elem ;
                                        while ( elem = nextl(listParameters) ) {
                                            if ( !first ) 
                                                "," 
                                            else 
                                                first = false ;
                                            "$" <SEPO> PrintString(EString(nbParam++));
                                        }
                                    }
                                    foreach (<IN>,refTree,{
                                        if ( !first ) 
                                            "," 
                                        else 
                                            first = false ;
                                        "$" <SEPO> PrintString(EString(nbParam++));
                                    })
                                    foreach (<OUT>,refTree,{
                                        if ( !first ) 
                                            "," 
                                        else 
                                            first = false ;
                                        "$" <SEPO> PrintString(EString(nbParam++));
                                    })
                                    ");)')'";
                                }
                                ")";
                            }
                        }} <NL,2>
            }
        }
    }
    _close(fileOutput);
    output = oldOutput ;
}

// GenerateFunctionModel : generate a model for each function
void Scheduler::GenerateFunctionCode ()
{
    int                 fileOutput ;
    EString             fileName(Name());
    std::set<EString>   generated ;
    int                 oldOutput = output ;
    
    fileName << "_functions.h";
    if ( (fileOutput = _open(fileName.c_str(), O_CREAT | O_RDWR | O_BINARY | O_TRUNC, 0666 /*S_IWRITE | S_IREAD*/ )) > 0 ) {
        output = fileOutput ;
        
        // create a set with all the task
        std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo ;
        for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
            EString entry ;
            int     nbParam = 1 ;
            bool    first = true ;
            PTREE   refTree ;
            PTREE   listParameters ;
            EString refName ;
            PTREE   name ;
            refTree = (*iterAlgo).second->Reference();
            refName = Value(refTree);
            if ( generated.find(refName) == generated.end() ) {
                generated.insert(refName);
                "inline void " PrintString(Value(refTree)) <SEPO> "_FunctInit (";
                if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                    PTREE   elem ;
                    while ( elem = nextl(listParameters) ) {
                        if ( !first ) 
                            "," 
                        else 
                            first = false ;
                        "int " PrintString(Value(elem));
                    }
                }
                if ( (*iterAlgo).second->Type() == ComputeUnit::MemoryUnit ) {
                    foreach (<OUT>,refTree,{
                        PTREE   elem = for_elem ;
                        elem == <OUT,<TYPE,<>,<>,name>>;
                        if ( !first ) 
                            "," 
                        else 
                            first = false ;
                        PrintString(Value(elem)) "*" PrintString(Value(name));
                    })
                }
                ")" <NL>
                "{}" <NL,2>
                first = true ;
                "inline void " PrintString(Value(refTree)) <SEPO> "_FunctLoop (";
                if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                    PTREE   elem ;
                    while ( elem = nextl(listParameters) ) {
                        if ( !first ) 
                            "," 
                        else 
                            first = false ;
                        "int " PrintString(Value(elem));
                    }
                }
                foreach (<IN>,refTree,{
                    PTREE   elem = for_elem ;
                    elem == <IN,<TYPE,<>,<>,name>>;
                    if ( !first ) 
                        "," 
                    else 
                        first = false ;
                    PrintString(Value(elem)) "*" PrintString(Value(name));
                })
                foreach (<OUT>,refTree,{
                    PTREE   elem = for_elem ;
                    elem == <OUT,<TYPE,<>,<>,name>>;
                    if ( !first ) 
                        "," 
                    else 
                        first = false ;
                    PrintString(Value(elem)) "*" PrintString(Value(name));
                })
                ")" <NL>
                "{}" <NL,2>
            }
        }
        ")";
    }
    _close(fileOutput);
    output = oldOutput ;
}

// GenerateCode : generate the associated code
void Scheduler::GenerateCode ()
{
    
    int                                     fileOutput ; // fileOutput file
    
    // generate a file for each operator
    std::vector<OperatorMedia *> ::iterator iterOperator ;
    std::vector<ConditionType>              nullVector ;
    
    // display planning for processors    
    for ( iterOperator = pvOperators.begin() ; iterOperator != pvOperators.end() ; iterOperator++ ) {
        if ( !(*iterOperator)->Planning().empty() ) {
            
            // create file
            EString fileName = (*iterOperator)->Name();
            PTREE   type = (*iterOperator)->Reference();
            fileName << ".m4";
            if ( (fileOutput = _open(fileName.c_str(), O_CREAT | O_RDWR | O_BINARY | O_TRUNC, 0666 /*S_IWRITE | S_IREAD*/ )) > 0 ) {
                output = fileOutput ;
                "include(syndex.m4x) dnl" <NL,2>
                "processor_(" PrintString(Value(type)) ", " PrintString((*iterOperator)->Name()) ", " PrintString(Name()) ", GenticSyndex)" <NL,2>
                
                // extract function in chronological order
                std::multimap<time_t, unsigned> ::iterator  iterTask ;
                (*iterOperator)->Extract((*iterOperator)->Planning().begin(), nullVector);
                
                // the semaphores
                {
                    std::vector<OperatorMedia *> ::iterator iter ;
                    std::set<EString>                       listDone ;
                    "semaphores_(" <NL>
                        <T> 
                        {{
                            
                            // display the thread semaphores
                            {
                                
                                // put the semaphore for the transfer tasks in the links
                                std::vector<OperatorMedia *> ::iterator iter ;
                                for ( iter = (*iterOperator)->Linked().begin() ; iter != (*iterOperator)->Linked().end() ; iter++ ) {
                                    
                                    // extract planning
                                    (*iter)->Extract((*iter)->Planning().begin(), nullVector);
                                    
                                    // compute send and receive for all tasks
                                    for ( iterTask = (*iter)->PlanningExtract().begin() ; iterTask != (*iter)->PlanningExtract().end() ; iterTask++ ) {
                                        std::pair<TransTrunkIterator, TransTrunkIterator>   range
                                            = pvTransByNumber.equal_range((*iter)->TaskContent((*iterTask).second)->NumberRef());
                                        TransTrunkIterator      iterTransTrunk ;
                                        #if 0
                                        for ( iterTransTrunk = range.first ; iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                            TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                            EString channel = transTrunk.channel ;
                                            
                                            // do the transfer for this task
                                            if ( *iterOperator == transTrunk.start ) {
                                                
                                                // variable full : send
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , FULL_SEMAPHORE, channel
                                                    , &listDone);
                                            } else {
                                                
                                                // variable empty : receive
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , EMPTY_SEMAPHORE, channel
                                                    , &listDone);
                                            }
                                            
                                            // semaphores
                                            if ( *iterOperator == transTrunk.start ) {
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , EMPTY_SEMAPHORE, channel
                                                    , &listDone);
                                            } else {
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , FULL_SEMAPHORE, channel
                                                    , &listDone);
                                            }
                                        }
                                        #endif
                                    }
                                }
                                
                                // put the semaphores for the internal tasks
                                {
                                    for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; 
                                            iterTask != (*iterOperator)->PlanningExtract().end() ; 
                                            iterTask++ ) {
                                        ComputeUnit *pTask = LeafAlgorithm()[TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name())];
                                        if ( !IsImplodeExplode(*pTask) ) {
                                            
                                            // predecessor full
                                            ComputeParameters(**iterOperator, *pTask, false /* in */ , FULL_SEMAPHORE, "", &listDone);
                                            
                                            // successor empty
                                            ComputeParameters(**iterOperator, *pTask, true /* out */ , EMPTY_SEMAPHORE, "", &listDone);
                                            
                                            // write that input has been used
                                            ComputeParameters(**iterOperator, *pTask, false /* in */ , EMPTY_SEMAPHORE, "", &listDone);
                                            
                                            // write that output are full
                                            ComputeParameters(**iterOperator, *pTask, true /* out */ , FULL_SEMAPHORE, "", &listDone);
                                        }
                                    }
                                }
                            }
                        }} <NL>
                    ")" <NL,2>
                }
                
                // the variables
                {
                    std::vector<OperatorMedia *> ::iterator iter ;
                    std::set<EString>                       listDone ;
                    
                    // look at all the links
                    for ( iter = (*iterOperator)->Linked().begin() ; iter != (*iterOperator)->Linked().end() ; iter++ ) {
                        
                        // extract planning
                        (*iter)->Extract((*iter)->Planning().begin(), nullVector);
                        
                        // compute send and receive for all tasks
                        for ( iterTask = (*iter)->PlanningExtract().begin() ; iterTask != (*iter)->PlanningExtract().end() ; iterTask++ ) {
                            std::pair<TransTrunkIterator, TransTrunkIterator>   range
                                = pvTransByNumber.equal_range((*iter)->TaskContent((*iterTask).second)->NumberRef());
                            TransTrunkIterator                                  iterTransTrunk ;
                            for ( iterTransTrunk = range.first ; iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                EString channel = transTrunk.channel ;
                                
                                // name of var
                                ComputeParameters(**iterOperator, *transTrunk.endTask, false, ALLOC_VARIABLE, channel, &listDone);
                            }
                        }
                    }
                    
                    // display the main function
                    {
                        for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; iterTask != (*iterOperator)->PlanningExtract().end() ; 
                                iterTask++ ) {
                            ComputeUnit *pTask = LeafAlgorithm()[TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name())];
                            if ( !IsImplodeExplode(*pTask) ) {
                                
                                // display all parameters in
                                ComputeParameters(**iterOperator, *pTask, false, ALLOC_VARIABLE, "", &listDone);
                                
                                // display all parameters out
                                ComputeParameters(**iterOperator, *pTask, true, ALLOC_VARIABLE, "", &listDone);
                            }
                        }
                    }
                }
                
                // for each link display a thread
                {
                    std::vector<OperatorMedia *> ::iterator iterLink ;
                    for ( iterLink = (*iterOperator)->Linked().begin() ; iterLink != (*iterOperator)->Linked().end() ; iterLink++ ) {
                        OperatorMedia   *otherOperator = 0 ;
                        <NL,2>
                        "thread_(" PrintString(Value((*iterLink)->Reference())) ", " PrintString((*iterLink)->Name());
                        
                        // display operators linked 
                        {
                            std::vector<OperatorMedia *> ::iterator iterOper ;
                            for ( iterOper = (*iterLink)->Linked().begin() ; iterOper != (*iterLink)->Linked().end() ; iterOper++ ) {
                                if ( *iterOperator != *iterOper ) 
                                    otherOperator = *iterOper ;
                            }
                            ", " PrintString((*iterOperator)->Name()) ", " PrintString(otherOperator->Name());
                        }
                        ")" <NL>
                        
                        // display body of thread
                            <T> 
                            {{
                                {
                                    std::set<EString>       listSend ;
                                    std::set<EString>       listReceived ;
                                    bool                    okTransmit = true ;
                                    
                                    // extract planning
                                    (*iterLink)->Extract((*iterLink)->Planning().begin(), nullVector);
                                    
                                    // init                                
                                    for ( iterTask = (*iterLink)->PlanningExtract().begin() ; iterTask != (*iterLink)->PlanningExtract().end() ; 
                                            iterTask++ ) {
                                        std::set<EString>                                       listDone ;
                                        std::pair<TransTrunkIterator, TransTrunkIterator>       range
                                            = pvTransByNumber.equal_range((*iterLink)->TaskContent((*iterTask).second)->NumberRef());
                                        TransTrunkIterator                                      iterTransTrunk ;
                                        for ( iterTransTrunk = range.first ; iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                            TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                            EString channel = transTrunk.channel ;
                                            if ( transTrunk.media == *iterLink && *iterOperator != transTrunk.start ) {
                                                
                                                // output empty for all recv
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , SET_EMPTY, channel
                                                    , &listDone);
                                            }
                                        }
                                    }
                                    
                                    // compute send and receive for all tasks
                                    bool    first = true ;
                                    
                                    // compute send and receive for all tasks
                                    for ( iterTask = (*iterLink)->PlanningExtract().begin() ; iterTask != (*iterLink)->PlanningExtract().end() ; 
                                            iterTask++ ) {
                                        std::pair<TransTrunkIterator, TransTrunkIterator>   range
                                            = pvTransByNumber.equal_range((*iterLink)->TaskContent((*iterTask).second)->NumberRef());
                                        TransTrunkIterator      iterTransTrunk ;
                                        PTREE                   reference ;
                                        
                                        // check if sthing to transmit
                                        okTransmit = false ;
                                        for ( iterTransTrunk = range.first ; !okTransmit && iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                            TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                            EString channel = transTrunk.channel ;
                                            if ( transTrunk.media == *iterLink ) {
                                                
                                                // do the transfer for this task
                                                if ( *iterOperator == transTrunk.start ) {
                                                    
                                                    // check if variable must be emitted
                                                    EString retVal = "";
                                                    ComputeParameters(**iterOperator, *transTrunk.endTask, false, CHECK_EMITTED, channel
                                                        , 0 /*&listSend*/ , &retVal);
                                                    okTransmit |= retVal != "";
                                                } else {
                                                    
                                                    // check if variable must be received
                                                    EString retVal = "";
                                                    ComputeParameters(**iterOperator, *transTrunk.endTask, false, CHECK_EMITTED, channel
                                                        , 0 /* &listReceived*/ , &retVal);
                                                    okTransmit |= retVal != "";
                                                }
                                            }
                                        }
                                        if ( okTransmit ) {
                                            EString                 prefix ;
                                            ComputeUnit             *thisUnit ;
                                            std::set<EString>       listDone ;
                                            
                                            // put header if not done
                                            if ( first ) {
                                                "loop_" <NL>
                                                    <T>
                                                Mark();
                                                first = false ;
                                            }
                                            for ( iterTransTrunk = range.first ; iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                                TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                                EString channel = transTrunk.channel ;
                                                if ( transTrunk.media == *iterLink ) {
                                                    
                                                    // do the transfer for this task
                                                    if ( *iterOperator == transTrunk.start ) {
                                                        
                                                        // variable full
                                                        ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , IS_FULL, channel
                                                            , &listDone);
                                                        
                                                        // send
                                                        prefix = "ssend_(";
                                                        
                                                        // unit
                                                        thisUnit = transTrunk.startTask ;
                                                    } else {
                                                        reference = transTrunk.endTask->Reference();
                                                        
                                                        // variable empty
                                                        ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , IS_EMPTY, channel
                                                            , &listDone);
                                                        
                                                        // receive
                                                        prefix = "srecv_(";
                                                        
                                                        // unit
                                                        thisUnit = transTrunk.endTask ;
                                                    }
                                                }
                                            }
                                            
                                            // compute conditions
                                            // display conditions for this operation
                                            int nbCondition = 0 ;
                                            {
                                                std::vector<ConditionType> ::iterator   iterCond ;
                                                for ( iterCond = thisUnit->Conditions().begin() ; iterCond != thisUnit->Conditions().end() ; 
                                                        iterCond++ ) {
                                                    
                                                    // condition is up to date 
                                                    // predecessor full
                                                    VString condString = Value(pvConditionByNumber [NumberCondition(*iterCond)].condition);
                                                    if ( !nbCondition ) {
                                                        "if_((";
                                                    } else {
                                                        "||(";
                                                    }
                                                    
                                                    // emit
                                                    {
                                                        ComputeParameters(**iterOperator, *thisUnit, false, VARIABLE, condString);
                                                        "," PrintString(CompactItoa(ValCondition(*iterCond))) ")";
                                                        nbCondition++ ;
                                                    }
                                                    if ( nbCondition ) {
                                                        ")"     <T>
                                                        Mark();
                                                    }
                                                }
                                            }
                                            
                                            // function call
                                            {
                                                
                                                // name of var
                                                PrintString(prefix);
                                                TransTrunk & transTrunk = (*range.first).second ;
                                                EString channel = transTrunk.channel ;
                                                
                                                // since not same computer endTask should be correct
                                                ComputeParameters(**iterOperator, *transTrunk.endTask, false, VARIABLE, channel);
                                                
                                                // generate size
                                                {
                                                    PTREE   type ;
                                                    ", ";
                                                    type = transTrunk.trans->Reference();
                                                    PrintString(Value(type [2])) " * sizeof(" PrintString(Value(type)) ")";
                                                }
                                                
                                                // end of call
                                                ", " PrintString(transTrunk.start->Name()) ", " PrintString(transTrunk.end->Name()) ")" <NL>
                                            }
                                            
                                            // end conditions for this operation
                                            if ( nbCondition ) {
                                                UnMark();
                                                "endif_" <NL>
                                                <NL>
                                            }
                                            {
                                                for ( iterTransTrunk = range.first ; iterTransTrunk != range.second ; iterTransTrunk++ ) {
                                                    TransTrunk & transTrunk = (*iterTransTrunk).second ;
                                                    EString channel = transTrunk.channel ;
                                                    if ( transTrunk.media == *iterLink ) {
                                                        
                                                        // set full or empty
                                                        if ( *iterOperator == transTrunk.start ) {
                                                            ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , SET_EMPTY
                                                                , channel, &listDone);
                                                        } else {
                                                            ComputeParameters(**iterOperator, *transTrunk.endTask, false /* in */ , SET_FULL
                                                                , channel, &listDone);
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    if ( !first ) {
                                        UnMark();
                                        "endloop_" <NL>
                                    }
                                }
                            }}
                        "endthread_" <NL>
                        <NL,2>
                    }
                }
                
                // display the main function
                <NL,2>
                {
                    "main_" <NL>
                        <T> 
                        {{
                            {
                                
                                // do an init 
                                for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; // predecessor full
                                                                                              iterTask != (*iterOperator)->PlanningExtract().end() ; 
                                        iterTask++ ) {
                                    ComputeUnit *pTask = LeafAlgorithm()[TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name())];
                                    if ( !IsImplodeExplode(*pTask) ) {
                                        PTREE   refTree = pTask->Reference();
                                        "operation_init_(`" PrintString(Value(refTree)) "'";
                                        
                                        // display all parameters of operation
                                        PTREE   listParameters ;
                                        if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                                            PTREE   elem ;
                                            while ( elem = nextl(listParameters) ) {
                                                "," PrintString(Value(elem [2]));
                                            }
                                        }
                                        
                                        // display all parameters out
                                        ComputeParameters(**iterOperator, *pTask, true);
                                        
                                        //  
                                        ")" <NL>
                                        if ( pTask->Type() == ComputeUnit::MemoryUnit ) {
                                            
                                            // say that all output are empty
                                            std::set<EString>   listDone ;
                                            ComputeParameters(**iterOperator, *pTask, true /* out */ , SET_FULL, "", &listDone);
                                        } else {
                                            
                                            // say that all output are empty
                                            std::set<EString>   listDone ;
                                            ComputeParameters(**iterOperator, *pTask, true /* out */ , SET_EMPTY, "", &listDone);
                                        }
                                    }
                                }
                                
                                // start the threads
                                std::vector<OperatorMedia *> ::iterator iter ;
                                for ( iter = (*iterOperator)->Linked().begin() ; iter != (*iterOperator)->Linked().end() ; iter++ ) {
                                    "spawn_thread_(" PrintString((*iter)->Name()) ")" <NL>
                                }
                                
                                // the loop
                                "loop_" <NL>
                                    <T> 
                                    {{
                                        for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; 
                                                iterTask != (*iterOperator)->PlanningExtract().end() ; 
                                                iterTask++ ) {
                                            Task                *pCurrTask ;
                                            ComputeUnit         *pTask
                                                = LeafAlgorithm()[
                                                    TaskName((pCurrTask = (*iterOperator)->TaskContent((*iterTask).second))->Name())];
                                            PTREE               reference = pTask->Reference();
                                            if ( !IsImplodeExplode(*pTask) ) {
                                                
                                                // predecessor full
                                                std::set<EString>   listFull ;
                                                ComputeParameters(**iterOperator, *pTask, false /* in */ , IS_FULL, "", &listFull);
                                                
                                                // successor empty
                                                std::set<EString>   listEmpty ;
                                                ComputeParameters(**iterOperator, *pTask, true /* out */ , IS_EMPTY, "", &listEmpty);
                                                
                                                // display conditions for this operation
                                                int nbCondition = 0 ;
                                                {
                                                    std::vector<ConditionType> ::iterator   iterCond ;
                                                    for ( iterCond = pCurrTask->Conditions().begin() ; iterCond != pCurrTask->Conditions().end() ; 
                                                            iterCond++ ) {
                                                        
                                                        // condition is up to date 
                                                        // predecessor full
                                                        VString         condString = Value(pvConditionByNumber [NumberCondition(*iterCond)].condition);
                                                        
                                                        // condition
                                                        EString         retVal ;
                                                        bool            emit ;
                                                        
                                                        // check that this task is really concerned by this condition, it should appear in in-channel
                                                        {
                                                            std::set<EString>   listCheck ;
                                                            ComputeParameters(**iterOperator, *pTask, false, CHECK_EMITTED, condString, &listCheck
                                                                , &retVal);
                                                            emit = retVal != "";
                                                        }
                                                        
                                                        // if ok emit
                                                        if ( emit ) {
                                                            "if_(";
                                                            ComputeParameters(**iterOperator, *pTask, false, VARIABLE, condString);
                                                            "," PrintString(CompactItoa(ValCondition(*iterCond))) ")" <NL>
                                                                <T>
                                                            Mark();
                                                            nbCondition++ ;
                                                        }
                                                    }
                                                }
                                                
                                                // call function
                                                {
                                                    PTREE   refTree = pTask->Reference();
                                                    "operation_(";
                                                    if ( reference == <MEMORY> ) {
                                                        "`memory_shift_',1,1,1";
                                                        
                                                        // display all parameters in
                                                        ComputeParameters(**iterOperator, *pTask, false);
                                                        
                                                        // display all parameters out
                                                        ComputeParameters(**iterOperator, *pTask, true);
                                                        
                                                        // display all parameters out
                                                        ComputeParameters(**iterOperator, *pTask, true);
                                                        <SEPO> "_buffer";
                                                    } else {
                                                        "`" PrintString(Value(refTree)) "'";
                                                        
                                                        // display all parameters of operation
                                                        PTREE   listParameters ;
                                                        if ( refTree == <,<>,<>,<>,<CONS,listParameters>> ) {
                                                            PTREE   elem ;
                                                            while ( elem = nextl(listParameters) ) {
                                                                "," PrintString(Value(elem [2]));
                                                            }
                                                        }
                                                        
                                                        // display all parameters in
                                                        ComputeParameters(**iterOperator, *pTask, false);
                                                        
                                                        // display all parameters out
                                                        ComputeParameters(**iterOperator, *pTask, true);
                                                    }
                                                    
                                                    // --
                                                    ")" <NL>
                                                }
                                                
                                                // end conditions for this operations
                                                while ( nbCondition-- ) {
                                                    UnMark();
                                                    "endif_" <NL>
                                                    <NL>
                                                }
                                                
                                                // write that input has been used
                                                {
                                                    std::set<EString>   listEmpty ;
                                                    ComputeParameters(**iterOperator, *pTask, false /* in */ , SET_EMPTY, "", &listEmpty);
                                                }
                                                
                                                // write that output are full
                                                {
                                                    std::set<EString>   listFull ;
                                                    ComputeParameters(**iterOperator, *pTask, true /* out */ , SET_FULL, "", &listFull);
                                                }
                                            }
                                        }
                                    }}
                                "endloop_" <NL>
                            }
                        }}
                    "endmain_" <NL,2>
                    "endprocessor_" <NL>
                }
                _close(fileOutput);
            }
        }
    }
    
    // Generate architecture
    GenerateArchitecture();
    
    // generate function model
    GenerateFunctionModel();
    
    // generate function model in c
    GenerateFunctionCode();
}


