/*************************************************************************/
/*                                                                       */
/*        computeunit.h   : Eric Lavillonniere 2005                      */
/*        company         : Mitsubishi Electric ITE                      */
/*                                                                       */
/*************************************************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation; either version 2.1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.
   
   You should have received a copy of the GNU Lesser General Public License
   along with this program; see the file COPYING. If not, write to 
   the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   */
#ifndef COMPUTE_UNIT_FILE
#define COMPUTE_UNIT_FILE 

// system includes
#include <map>
#include <vector>
#include <set>

// private include
#include "erlstring.h"
#include "operatormedia.h"

// private includes

class ComputeUnit {
    
    public :
    
        enum UnitType {
            CompUnit, 
            MemoryUnit, 
            ConstantUnit, 
            TUnit, 
            None
        };
        
        // default constructor
        ComputeUnit ()
            : pvUnitType(None),  pvComputed(false),  pvCommTime(0),  pvIdent(0),  pvInDependant(true),  pvRefNumber(0)
        {
            pvProc = &OperatorMedia::DefaultMedia();
        }
        
        // constructor
        ComputeUnit ( UnitType type, EString name, PTREE &reference, std::vector<ConditionType> &conditions )
            : pvUnitType(type),  pvReference(reference),  pvComputed(false),  pvCommTime(0),  pvName(name),  pvIdent(0)
                ,  pvInDependant(true),  pvRefNumber(0),  pvConditions(conditions),  pvNameNumber(0)
        {
            pvProc = &OperatorMedia::DefaultMedia();
        }
        
        // constructor
        ComputeUnit ( UnitType type, EString name, PTREE &reference )
            : pvUnitType(type),  pvReference(reference),  pvComputed(false),  pvCommTime(0),  pvName(name),  pvIdent(0)
                ,  pvInDependant(true)
        {
            pvProc = &OperatorMedia::DefaultMedia();
        }
        
        // destructor 
        virtual ~ComputeUnit ()
        {
            std::map<EString, ComputeUnit *, LessString> ::iterator iter ;
            ComputeUnit *pUnit ;
            
            for ( iter = pvSubUnits.begin() ; iter != pvSubUnits.end() ; iter++ ) {
                if ( (*iter).second ) {
                    pUnit = (*iter).second ;
                    (*iter).second = 0 ;
                    delete pUnit ;
                }
            }
            pvSubUnits.clear();
            
            std::vector<ComputeUnit *> ::iterator   iterTrans ;
            int                                     index ;
            
            for ( iterTrans = TransVect().begin(), index = 0 ; iterTrans != TransVect().end() ; iterTrans++, index++ ) {
                if ( *iterTrans ) {
                    pUnit = *iterTrans ;
                    TransVect()[index] = (ComputeUnit *)0 ;
                    delete pUnit ;
                }
            }
        }
        
        // Unit : get a unit
        ComputeUnit &Unit ( EString name )
        {
            return *pvSubUnits [name];
        }
        
        // AddUnit : add unit in this processor unit
        void AddUnit ( EString name, ComputeUnit &computeUnit )
        {
            pvSubUnits [name] = &computeUnit ;
        }
        
        // Type : get unit type 
        UnitType Type ()
        {
            return pvUnitType ;
        }
        
        // Type : set unit type
        ComputeUnit &Type ( UnitType type )
        {
            pvUnitType = type ;
            return *this ;
        }
        
        // Name : get the name
        const EString &Name () const
        {
            return pvName ;
        }
        
        // Name : set the name
        ComputeUnit &Name ( EString name )
        {
            pvName = name ;
            return *this ;
        }
        
        // Name : get the name
        const unsigned int &NameNumber () const
        {
            return pvNameNumber ;
        }
        
        // Name : set the name
        ComputeUnit &NameNumber ( unsigned int nameNumber )
        {
            pvNameNumber = nameNumber ;
            return *this ;
        }
        
        // Reference : get reference for this unit
        PTREE &Reference ()
        {
            return pvReference ;
        }
        
        // Reference :  set  reference for this unit
        ComputeUnit &Reference ( PTREE &reference )
        {
            pvReference = reference ;
            return *this ;
        }
        
        // ComputeLinks : compute all the links
        void    ComputeLinks (PTREE &listDepedence, std::vector<std::set<PTREE> > &links) ;
        
        // set proc 
        ComputeUnit &Proc ( OperatorMedia &operatorMedia )
        {
            pvProc = &operatorMedia ;
            return *this ;
        }
        
        // get proc 
        OperatorMedia &Proc ()
        {
            return *pvProc ;
        }
        
        // Computed : tells if unit has been computed
        bool Computed ()
        {
            return pvComputed ;
        }
        
        // Computed : set computed flag
        ComputeUnit &Computed ( bool computed )
        {
            pvComputed = computed ;
            return *this ;
        }
        
        // StartTime : set start time
        ComputeUnit &StartTime ( time_t time )
        {
            pvStartTime = time ;
            return *this ;
        }
        
        // StartTime : get the start time
        time_t StartTime ()
        {
            return pvStartTime ;
        }
        
        // ComputeTime : set the compute time
        ComputeUnit &ComputeTime ( time_t time )
        {
            pvComputeTime = time ;
            return *this ;
        }
        
        // ComputeTime : get the compute time
        time_t ComputeTime ()
        {
            return pvComputeTime ;
        }
        
        // CommTime : set the comm time
        ComputeUnit &CommTime ( time_t time )
        {
            pvCommTime = time ;
            return *this ;
        }
        
        // CommTime : get the comm time
        time_t CommTime ()
        {
            return pvCommTime ;
        }
        
        // Connected : get the connected vector
        std::vector<ComputeUnit *> &ConnectedIn ()
        {
            return pvCommIn ;
        }
        
        // Connected : get the connected vector
        std::vector<ComputeUnit *> &ConnectedOut ()
        {
            return pvCommOut ;
        }
        
        // TransUnit : give the trans unit of this box
        std::vector<ComputeUnit *> &TransVect ()
        {
            return pvTransUnit ;
        }
        
        // SearchConnected : search the connected to a given port
        virtual void    SearchConnected (PTREE &name, std::vector<ComputeUnit *> &connected, ComputeUnit *transUnit
            , bool first) ;
        
        // InEndConnectionsRef 
        std::map<EString, ComputeUnit *, LessString> &InEndConnectionsRef ()
        {
            return pvInEndConnectionsRef ;
        }
        
        // OutEndConnectionsRef 
        std::map<EString, ComputeUnit *, LessString> &OutEndConnectionsRef ()
        {
            return pvOutEndConnectionsRef ;
        }
        
        // InEndConnectionsRef 
        std::map<EString, ComputeUnit *, LessString> &InOutEndConnectionsRef ()
        {
            return pvInOutEndConnectionsRef ;
        }
        
        // Identifier : get identifier for this algo
        unsigned int Identifier ()
        {
            return pvIdent ;
        }
        
        // Identifier : set identifier for this algo
        ComputeUnit &Identifier ( unsigned int identifier )
        {
            pvIdent = identifier ;
            return *this ;
        }
        
        // Display : display an unit 
        void    Display () ;
        
        // InDependant : must take into account for time computation, not true for memory(delay)
        bool InDependant ()
        {
            return pvInDependant ;
        }
        
        // InDependant : set in dependant flag
        ComputeUnit &InDependant ( bool inDependant )
        {
            pvInDependant = inDependant ;
            return *this ;
        }
        
        // Procs : get the procs for this algo
        std::vector<int> &Procs ()
        {
            return pvProcs ;
        }
        
        // Procs : set the procs
        ComputeUnit &Procs ( std::vector<int> &procs )
        {
            pvProcs = procs ;
            return *this ;
        }
        
        // Conditions : get the conditions
        std::vector<ConditionType> &Conditions ()
        {
            return pvConditions ;
        }
        
        // Conditions : set the conditions
        ComputeUnit &Conditions ( std::vector<ConditionType> &conditions )
        {
            pvConditions = conditions ;
            return *this ;
        }
        
        // RefNumber : set Ref number
        ComputeUnit &RefNumber ( unsigned int ref )
        {
            pvRefNumber = ref ;
            return *this ;
        }
        
        // RefNumber : get ref number
        unsigned int RefNumber () const
        {
            return pvRefNumber ;
        }
    
    protected :
    
        bool        pvInDependant ; // tells if must entry into account for time computation (not in case of delay)
        UnitType    pvUnitType ;    // the unit type        
    
    private :
    
        ComputeUnit ( ComputeUnit & ) {}
        
        ComputeUnit &operator= ( ComputeUnit & )
        {
            return *this ;
        }
        
        EString                                         pvName ;                   // the name of compute unit
        unsigned int                                    pvNameNumber ;             // number for name
        unsigned int                                    pvIdent ;                  // identifier number for this algo
        PTREE                                           pvReference ;              // reference for this unit
        std::map<EString, ComputeUnit *, LessString>    pvSubUnits ;               // the units inside this unit
        std::vector<ComputeUnit *>                      pvComm ;                   // link to in/out communications for a trans box
        std::vector<ComputeUnit *>                      pvCommIn ;                 // link to in/out communications for a trans box
        std::vector<ComputeUnit *>                      pvCommOut ;                // link to in/out communications for a trans box
        std::vector<ComputeUnit *>                      pvTransUnit ;              // the trans units in this box 
        std::map<EString, ComputeUnit *, LessString>    pvInEndConnectionsRef ;    // referencer of end connections for a leaf box
        std::map<EString, ComputeUnit *, LessString>    pvInOutEndConnectionsRef ; // referencer of end connections for a leaf box
        std::map<EString, ComputeUnit *, LessString>    pvOutEndConnectionsRef ;   // referencer of end connections for a leaf box
        OperatorMedia                                   *pvProc ;                  // the processor for execution it
        bool                                            pvComputed ;               // tells if time has been computed
        time_t                                          pvStartTime ;              // starting time
        time_t                                          pvComputeTime ;            // time used for computation
        time_t                                          pvCommTime ;               // time used for commmunications
        std::vector<int>                                pvProcs ;
        std::vector<ConditionType>                      pvConditions ;
        unsigned int                                    pvRefNumber ;
};

class ConditionalElem {
    
    public :
    
        // constructor
        ConditionalElem () {}
        
        // constructor with data
        ConditionalElem ( PTREE &condition, ComputeUnit &unit )
            : pvCondition(condition),  pvComputeUnit(&unit)
        {}
        
        // destructor 
        virtual ~ConditionalElem ()
        {
            if ( pvComputeUnit ) {
                delete pvComputeUnit ;
                pvComputeUnit = 0 ;
            }
        }
        
        // Condition : get the condition
        PTREE &Condition ()
        {
            return pvCondition ;
        }
        
        // Condition : set the conditionn
        ConditionalElem &Condition ( PTREE &condition )
        {
            pvCondition = condition ;
            return *this ;
        }
        
        // Unit : get the unit
        ComputeUnit &Unit ()
        {
            return *pvComputeUnit ;
        }
        
        // Unit : set the unit
        ConditionalElem &Unit ( ComputeUnit &unit )
        {
            pvComputeUnit = &unit ;
            return *this ;
        }
    
    private :
    
        PTREE       pvCondition ;    // the condition
        ComputeUnit *pvComputeUnit ; // the computation unit
};

class ConditionalUnit : public ComputeUnit {
    
    public :
    
        // constructor
        ConditionalUnit ()
            : ComputeUnit()
        {}
        
        // constructor with parameters
        ConditionalUnit ( EString name, PTREE &reference )
            : ComputeUnit(ComputeUnit::CompUnit, name, reference)
        {}
        
        // constructor with parameters
        ConditionalUnit ( EString name, PTREE &reference, std::vector<ConditionType> &conditions )
            : ComputeUnit(ComputeUnit::CompUnit, name, reference, conditions)
        {}
        
        // destructor
        virtual ~ConditionalUnit ()
        {
            std::vector<ConditionalElem *> ::iterator   iter ;
            
            for ( iter = pvConditionalElem.begin() ; iter != pvConditionalElem.end() ; iter++ ) {
                ConditionalElem *pCondElem = *iter ;
                delete pCondElem ;
            }
        }
        
        // AddUnit : add a unit in conditionalUnit
        void AddUnit ( PTREE &condition, ComputeUnit &unit )
        {
            ConditionalElem *pcondElem = new ConditionalElem(condition, unit);
            
            pvConditionalElem.push_back(pcondElem);
        }
        
        // SearchConnected : search the connected to a given port
        virtual void SearchConnected ( PTREE &name, std::vector<ComputeUnit *> &connected, ComputeUnit *transUnit
            , bool first )
        {
            std::vector<ConditionalElem *> ::iterator   iter ;
            
            for ( iter = pvConditionalElem.begin() ; iter != pvConditionalElem.end() ; iter++ ) {
                (*iter) -> Unit().SearchConnected(name, connected, transUnit, first);
            }
        }
    
    private :
    
        std::vector<ConditionalElem *>  pvConditionalElem ;
};

class ProcUnit : public ComputeUnit {
    
    public :
    
        // constructor
        ProcUnit ()
            : ComputeUnit()
        {}
        
        //constructor with init
        ProcUnit ( EString name, PTREE &reference )
            : ComputeUnit(ComputeUnit::CompUnit, name, reference)
        {}
        
        //constructor with init
        ProcUnit ( EString name, PTREE &reference, std::vector<ConditionType> &conditions )
            : ComputeUnit(ComputeUnit::CompUnit, name, reference, conditions)
        {}
        
        // destructor 
        virtual ~ProcUnit () {}
};

class MemoryUnit : public ProcUnit {
    
    public :
    
        // constructor
        MemoryUnit ()
            : ProcUnit()
        {
            pvUnitType = ComputeUnit::MemoryUnit ;
        }
        
        //constructor with init
        MemoryUnit ( EString name, PTREE &reference )
            : ProcUnit(name, reference)
        {
            pvUnitType = ComputeUnit::MemoryUnit ;
        }
        
        //constructor with init
        MemoryUnit ( EString name, PTREE &reference, std::vector<ConditionType> &conditions )
            : ProcUnit(name, reference, conditions)
        {
            pvUnitType = ComputeUnit::MemoryUnit ;
        }
        
        // destructor 
        virtual ~MemoryUnit () {}
};

class ConstantUnit : public ProcUnit {
    
    public :
    
        // constructor
        ConstantUnit ()
            : ProcUnit()
        {
            pvInDependant = false ;
            pvUnitType = ComputeUnit::ConstantUnit ;
        }
        
        //constructor with init
        ConstantUnit ( EString name, PTREE &reference )
            : ProcUnit(name, reference)
        {
            pvInDependant = false ;
            pvUnitType = ComputeUnit::ConstantUnit ;
        }
        
        //constructor with init
        ConstantUnit ( EString name, PTREE &reference, std::vector<ConditionType> &conditions )
            : ProcUnit(name, reference, conditions)
        {
            pvInDependant = false ;
            pvUnitType = ComputeUnit::ConstantUnit ;
        }
        
        // destructor 
        virtual ~ConstantUnit () {}
};

class TransUnit : public ComputeUnit {
    
    public :
    
        // constructor 
        TransUnit ()
            : ComputeUnit()
        {}
        
        // constructor with init
        TransUnit ( EString name, PTREE &reference )
            : ComputeUnit(ComputeUnit::TUnit, name, reference)
        {}
        
        // SearchConnected : search the connected to a given port
        virtual void SearchConnected ( PTREE &name, std::vector<ComputeUnit *> &connected, ComputeUnit *transUnit ) {}
        
        // destructor 
        virtual ~TransUnit () {}
};
#endif
