/*************************************************************************/
/*                                                                       */
/*        operatormedia.h   : Eric Lavillonniere 2005                    */
/*        company       : Mitsubishi Electric ITE                        */
/*                                                                       */
/*************************************************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation; either version 2.1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.
   
   You should have received a copy of the GNU Lesser General Public License
   along with this program; see the file COPYING. If not, write to 
   the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   */
#ifndef FILE_BOARD
#   define FILE_BOARD 
    
    // system file
#   include <map>
#   include <vector>
#   define CONDITION_BIT_REF 24
    typedef int ConditionType ;
    
    inline ConditionType CreateCondition ( unsigned int nbCondition, unsigned int val )
    {
        ConditionType   ret = 0 ;
        unsigned int    leftShift = sizeof(ConditionType) - CONDITION_BIT_REF ;
        unsigned int    valRef = ~((ConditionType) -1 << leftShift);
        unsigned int    nbRef = ~((ConditionType) -1 << (CONDITION_BIT_REF));
        
        if ( val > valRef || nbCondition > nbRef ) {
            cerr << "Condition invalid : too big \n";
        }
        ret = val << CONDITION_BIT_REF | nbCondition & ~(((ConditionType) -1) << CONDITION_BIT_REF);
        return ret ;
    }
    
    inline unsigned int NumberCondition ( ConditionType conditionType )
    {
        return conditionType & ~(((ConditionType) -1) << CONDITION_BIT_REF);
    }
    
    inline unsigned int ValCondition ( ConditionType conditionType )
    {
        return conditionType >> CONDITION_BIT_REF & ~(((ConditionType) -1) << 8 * sizeof(ConditionType) - CONDITION_BIT_REF);
    }
    
    inline bool MergeCondition ( std::vector<ConditionType> &outCondition, std::vector<ConditionType> &inCondition1, std::vector<ConditionType> &inCondition2 )
    {
        std::vector<ConditionType> ::iterator   iterCondition1 = inCondition1.begin();
        std::vector<ConditionType> ::iterator   iterCondition2 = inCondition2.begin();
        
        outCondition.clear();
        while ( true ) {
            
            // if one condition vector is empty task is inserted
            if ( iterCondition1 == inCondition1.end() ) {
                for (; iterCondition2 != inCondition2.end() ; iterCondition2++ ) 
                    outCondition.push_back(*iterCondition2);
                break ;
            }
            if ( iterCondition2 == inCondition2.end() ) {
                for (; iterCondition1 != inCondition1.end() ; iterCondition1++ ) 
                    outCondition.push_back(*iterCondition1);
                break ;
            }
            
            // if same number of task decide
            int nbCond1 = NumberCondition(*iterCondition1);
            int nbCond2 = NumberCondition(*iterCondition2);
            if ( nbCond1 == nbCond2 ) 
            {
                int valCond1 = ValCondition(*iterCondition1);
                int valCond2 = ValCondition(*iterCondition2);
                
                // not same branch so exit
                if ( valCond1 != valCond2 ) {
                    return false ;
                } else {
                    
                    // same branch, see otherConditions
                    outCondition.push_back(*iterCondition1);
                    iterCondition1++ ;
                    iterCondition2++ ;
                    continue ;
                }
            } else 
            // not same number advance the lower number
            if ( nbCond1 < nbCond2 ) {
                outCondition.push_back(*iterCondition1);
                iterCondition1++ ;
            } else {
                outCondition.push_back(*iterCondition2);
                iterCondition2++ ;
            }
        }
        return true ;
    }
    
    // #   define multimap map    
    inline void DisplayError ( EString error )
    {
        _write(2, error.c_str(), error.length());
    }
    
    class OperatorMedia ;
    
    struct OperatorConnect {
        OperatorMedia   *oper ;
        OperatorMedia   *link ;
    };
    
    class Task {
        
        public :
        
            Task ()
                : pvStart(0),  pvEnd(0)
            {}
            
            Task ( time_t start, time_t end )
                : pvStart(start),  pvEnd(end)
            {}
            
            virtual ~Task () {}
            
            // get start
            time_t Start () const
            {
                return pvStart ;
            }
            
            // set start
            Task &Start ( time_t start )
            {
                pvStart = start ;
                return *this ;
            }
            
            // get end
            time_t End () const
            {
                return pvEnd ;
            }
            
            // set end 
            Task &End ( time_t end )
            {
                pvEnd = end ;
                return *this ;
            }
            
            // Duration : get duration 
            unsigned int Duration ()
            {
                return pvEnd - pvStart ;
            }
            
            // Name : get name
            unsigned int Name ()
            {
                return pvName ;
            }
            
            // Name : set the name
            Task &Name ( unsigned int name )
            {
                pvName = name ;
                return *this ;
            }
            
            // Conditions : get the conditions
            std::vector<ConditionType> &Conditions ()
            {
                return pvConditions ;
            }
            
            // Conditions : set the conditions
            Task &Conditions ( std::vector<ConditionType> &conditions )
            {
                pvConditions = conditions ;
                return *this ;
            }
            
            // NumberRef : get number ref
            int NumberRef ()
            {
                return pvNumberRef ;
            }
            
            // Conditions : set the conditions
            Task &NumberRef ( int numberRef )
            {
                pvNumberRef = numberRef ;
                return *this ;
            }
        
        private :
        
            // some kind of pointer on real task
            time_t                      pvStart ;     // start of task
            time_t                      pvEnd ;       // end of task
            unsigned int                pvName ;
            std::vector<ConditionType>  pvConditions ;
            int                         pvNumberRef ; // reference of creator
    };
    
    class OperatorMedia {
        
        public :
        
            enum HardType { Proc, Link, None };
            
            OperatorMedia ()
                : pvIndex(0),  pvFpga(false)
            {
                Clear();
            }
            
            OperatorMedia ( EString name, HardType type, PTREE ref, unsigned numberProc = 0 )
                : pvName(name),  pvHardType(type),  pvIndex(numberProc),  pvFpga(false)
            {
                Reference(ref);
                Clear();
            }
            
            virtual ~OperatorMedia () {}
            
            // Type : get unit type 
            HardType Type ()
            {
                return pvHardType ;
            }
            
            // Type : set unit type
            OperatorMedia &Type ( HardType type )
            {
                pvHardType = type ;
                return *this ;
            }
            
            // get name 
            EString &Name ()
            {
                return pvName ;
            }
            
            // set name
            OperatorMedia &Name ( const EString &name )
            {
                pvName = name ;
                return *this ;
            }
            
            // Linked : set linked to this operator media
            OperatorMedia &Linked ( std::vector<OperatorMedia *> linked )
            {
                pvLinkedOperatorMedia = linked ;
                return *this ;
            }
            
            // Linked : get linked to this operator media
            std::vector<OperatorMedia *> &Linked ()
            {
                return pvLinkedOperatorMedia ;
            }
            
            // Reference : get reference 
            PTREE &Reference ()
            {
                return pvReference ;
            }
            
            // Reference : set reference 
            OperatorMedia   &Reference (PTREE &reference) ;
            
            // SearchConnection : search a connection between to proc
            virtual OperatorMedia *SearchConnection ( OperatorMedia *proc )
            {
                return 0 ;
            }
            
            // SearchConnection : search a connection between to proc
            virtual std::vector<OperatorConnect> AllConnections ()
            {
                std::vector<OperatorConnect>    vectRet ;
                
                return vectRet ;
            }
            
            // CreateTask : create a task on OperatorMedia
            virtual unsigned CreateTask ( unsigned int name, PTREE &ref, unsigned int nbRef, time_t startTime, const EString &label, unsigned int transferDelay = 0 )
            {
                std::vector<ConditionType>  conditions ;
                
                return CreateTask(name, ref, nbRef, startTime, conditions, label, transferDelay);
            }
            
            // CreateTask : create a task on OperatorMedia
            virtual unsigned ComputeTime ( EString &label, PTREE ref, unsigned int nbRef )
            {
                return 0 ;
            }
            
            // CreateTask : create a task on OperatorMedia
            virtual unsigned CreateTask ( unsigned int name, PTREE &ref, unsigned int nbRef, time_t startTime, std::vector<ConditionType> &conditions
                , const EString &label, unsigned int transferDelay = 0 )
            {
                return 0 ;
            }
            
            // InsertTask : insert a task in planning extract (task sorted by start time)
            unsigned    InsertTaskExtract (unsigned int name, time_t startTime, time_t transferTime, std::vector<ConditionType> &conditions) ;
            
            // InsertTask : insert a task in planning 
            unsigned    InsertTask (unsigned int name, time_t startTime, time_t transferTime, std::vector<ConditionType> &conditions, int nbRef = -1) ;
            
            // Planning : get the planning
            std::multimap<time_t, unsigned> &PlanningExtract ()
            {
                return pvPlanningExtract ;
            }
            
            // Planning : get the planning
            std::multimap<time_t, unsigned> &Planning ()
            {
                return pvPlanning ;
            }
            
            // default operator media
            static  OperatorMedia &DefaultMedia ()
            {
                return *pOperatorMedia ;
            }
            
            // Index : get index
            unsigned int Index ()
            {
                return pvIndex ;
            }
            
            // Index : set index
            OperatorMedia &Index ( unsigned int index )
            {
                pvIndex = index ;
                return *this ;
            }
            
            // TimeOper : set the time for an operation
            OperatorMedia &TimeOper ( EString &name, double time )
            {
                pvTimeOper [name] = time ;
                return *this ;
            }
            
            // TimeOper : get the time for an operation
            double TimeOper ( EString &name )
            {
                {
                    std::map<EString, double> ::iterator    iter = pvTimeOper.find(name);
                    if ( iter != pvTimeOper.end() ) {
                        return (*iter).second ;
                    } else {
                        EString message = "Time not found : ";
                        message << name << " on " << Name() << ";\n";
                        DisplayError(message);
                    }
                }
                return -1 ;
            }
            
            // TimeOper : get the time for an operation
            double GetTimeOper ( PTREE &ref, unsigned int nbRef )
            {
                if ( nbRef ) {
                    std::map<unsigned int, double> ::iterator   iter = pvTimeOperInt.find(nbRef);
                    if ( iter != pvTimeOperInt.end() ) {
                        return (*iter).second ;
                    } else {
                        VString name = Value(ref);
                        double  result = TimeOper(name);
                        return pvTimeOperInt [nbRef] = result ;
                    }
                } else {
                    VString name = Value(ref);
                    return TimeOper(name);
                }
            }
            
            // TaskContent : return the content of a task
            Task *TaskContent ( unsigned ref )
            {
                return &pvAllTasks [ref];
            }
            
            static OperatorMedia    *pOperatorMedia ;
            
            // Clear : clear everything
            void Clear ()
            {
                PlanningExtract().clear();
                Planning().clear();
                pvAllTasks.clear();
                pvTaskByRef.clear();
                
                Task    task ;
                
                pvAllTasks.push_back(task);
                pvStart = (unsigned int) -1 ;
            }
            
            // Extract : extract task and sort with starting time
            // parameters : 
            //              extract : where to start extraction
            void    Extract (std::multimap<time_t, unsigned> ::iterator extract, std::vector<ConditionType> &conditions) ;
            
            // Start : returns start
            time_t Start ()
            {
                return pvStart ;
            }
            
            // Start : set  start
            OperatorMedia &Start ( time_t start )
            {
                pvStart = start ;
                return *this ;
            }
            
            // Fpga : get fpga flag
            bool Fpga ()
            {
                return pvFpga ;
            }
            
            // Fpga : set fpga flag
            OperatorMedia &Fpga ( bool fpga )
            {
                pvFpga = fpga ;
                return *this ;
            }
        
        private :
        
            // InsertPlanning : insert a task in planning
            unsigned InsertPlanningInternal ( time_t startTime, time_t durationTime, unsigned int name, bool extract, std::vector<ConditionType> &conditions
                , int numberTask = -1, int numberRef = -1 )
            {
                if ( numberTask == -1 ) {
                    Task    task ;
                    task.Start(startTime).End(startTime + durationTime).Name(name).NumberRef(numberRef);
                    if ( !conditions.empty() ) {
                        task.Conditions(conditions);
                    }
                    pvAllTasks.push_back(task);
                    numberTask = pvAllTasks.size() - 1 ;
                }
                if ( extract ) 
                    PlanningExtract().insert(std::pair<time_t, unsigned> (startTime, numberTask));
                else 
                    Planning().insert(std::pair<time_t, unsigned> (startTime + durationTime, numberTask));
                return numberTask ;
            }
            
            // InsertPlanning : insert a task in planning
            unsigned InsertPlanning ( time_t startTime, time_t durationTime, unsigned int name, std::vector<ConditionType> &conditions, int numberTask = -1
                , int numberRef = -1 )
            {
                if ( ((unsigned long)startTime) < ((unsigned long)Start()) ) 
                    Start(startTime);
                numberTask = InsertPlanningInternal(startTime, durationTime, name, false, conditions, numberTask, numberRef);
                if ( numberRef != -1 ) 
                    pvTaskByRef [numberRef] = numberTask ;
                return numberTask ;
            }
            
            // InsertPlanning : insert a task in planning
            unsigned InsertPlanningExtract ( time_t startTime, time_t durationTime, unsigned int name, std::vector<ConditionType> &conditions, int numberTask = -1
                , int numberRef = -1 )
            {
                return InsertPlanningInternal(startTime, durationTime, name, true, conditions, numberTask, numberRef);
            }
            
            HardType                            pvHardType ;
            EString                             pvName ;                // name of operator media
            std::multimap<time_t, unsigned>     pvPlanning ;            // planning of tasks to be done on operator or media
            std::multimap<time_t, unsigned>     pvPlanningExtract ;     // planning of tasks to be done on operator or media
            std::vector<OperatorMedia *>        pvLinkedOperatorMedia ; // connection to operators or media
            PTREE                               pvReference ;
            unsigned int                        pvIndex ;
            std::map<EString, double>           pvTimeOper ;            // time for each operation
            std::map<unsigned, double>          pvTimeOperInt ;         // time for each operation
            std::vector<Task>                   pvAllTasks ;
            unsigned int                        pvStart ;               // start on this operator
            bool                                pvFpga ;                // tell if is an fpga
            std::map<int, unsigned>             pvTaskByRef ;
    };
    
    class HardProcUnit : public OperatorMedia {
        
        public :
        
            HardProcUnit ()
                : OperatorMedia()
            {}
            
            // constructor
            HardProcUnit ( EString name, PTREE &ref, unsigned int numberProc )
                : OperatorMedia(name, Proc, ref, numberProc)
            {}
            
            // destructor
            virtual ~HardProcUnit () {}
            
            // SearchConnection : search a connection between to proc
            virtual OperatorMedia                   *SearchConnection (OperatorMedia *proc) ;
            
            // SearchConnection : search a connection between to proc
            virtual std::vector<OperatorConnect>    AllConnections () ;
            
            // ComputeTime : compute time of a task
            virtual unsigned                        ComputeTime (EString &label, PTREE ref, unsigned int nbRef) ;
            
            // CreateTask : create a task on OperatorMedia
            virtual unsigned CreateTask ( unsigned int name, PTREE &ref, unsigned int nbRef, time_t startTime, std::vector<ConditionType> &conditions
                , const EString &labelParam, unsigned int transferDelay = 0 )
            {
                VString label = labelParam ;
                
                return InsertTask(name, startTime, ComputeTime(label, ref, nbRef) + transferDelay, conditions);
            }
    };
    
    class LinkUnit : public OperatorMedia {
        
        public :
        
            LinkUnit ()
                : OperatorMedia()
            {}
            
            // constructor
            LinkUnit ( EString name, PTREE &ref )
                : OperatorMedia(name, Link, ref)
            {}
            
            // destructor
            virtual ~LinkUnit () {}
            
            // ComputeTime : compute time of a task
            virtual unsigned    ComputeTime (EString &label, PTREE ref, unsigned int nbRef) ;
            
            // CreateTask : create a task on OperatorMedia
            virtual unsigned CreateTask ( unsigned int name, PTREE &ref, unsigned int nbRef, time_t startTime, std::vector<ConditionType> &conditions
                , const EString &labelParam, unsigned int transferDelay = 0 )
            {
                VString label = labelParam ;
                
                return InsertTask(name, startTime, ComputeTime(label, ref, nbRef) + transferDelay, conditions, nbRef);
            }
            
            // get all Operators
            std::vector<OperatorMedia *> &AllOperators ()
            {
                return pvAllOperators ;
            }
            
            // get ChannelByOperator
            std::map<EString, EString, LessString> &ChannelByOperator ()
            {
                return pvChannelByOperator ;
            }
        
        public :
        
            std::vector<OperatorMedia *>            pvAllOperators ;
            std::map<EString, EString, LessString>  pvChannelByOperator ;
    };
#endif
