/*************************************************************************/
/*                                                                       */
/*        scheduler.ch   : Eric Lavillonniere 2005                       */
/*        company        : Mitsubishi Electric ITE                       */
/*                                                                       */
/************************************* ************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation; either version 2.1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.
   
   You should have received a copy of the GNU Lesser General Public License
   along with this program; see the file COPYING. If not, write to 
   the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   */
language syndex;

// system includes
#include <map>

// private includes
#include "tablist.h"
#include "operatormedia.h"
#include "scheduler.h"
#include "syndex.h"
#include "individual.h"

// Schedule : main function for scheduling tasks
void Scheduler::Explore ()
{
    PTREE   description = Description();
    PTREE   mainArchi ;
    PTREE   mainAlgo ;
    
    if ( description == <SYNDEX_TOP,<>,<>,description> ) {
        
        // search for the main algo
        foreach (<MAIN_ALGORITHM>,description,{
            mainAlgo = for_elem ;
            
            // suppress mainAlgo entry to avoid confusing it with real entry in table
            PTREE   null ;
            mainAlgo += null ;
            goto for_break ;
        })
        
        // search for the main architecture
        foreach (<MAIN_ARCHITECTURE>,description,{
            mainArchi = for_elem ;
            
            // suppress mainArchi entry to avoid confusing it with real entry in table
            PTREE   null ;
            mainArchi += null ;
            goto for_break ;
        })
        
        // suppress pos1 and pos2 in lists
        PTREE   elem ;
        PTREE   nextVal ;
        PTREE   father ;
        foreach ((),description,{
            elem = for_elem ;
            if ( elem == <LIST,<POS1>,nextVal> || elem == <LIST,<POS2>,nextVal> ) {
                father = elem ^ ;
                elem += nextVal ;
                for_elem = father ; /* replaced tree could be () */ 
            }
        })
        
        // get direct access on the element of syndex description
        TabList syndexDescript (description) ;
        syndexDescript.Sort();
        
        // expand this architecture
        ExpandArchi(syndexDescript, mainArchi);
        
        // create a symbol table for algo
        SymbolTable symbTable ;
        PTREE       parameters ;
        mainAlgo == <MAIN_ALGORITHM,<NAME_AND_PARAM,<>,parameters>>;
        
        // expand references of main algo
        PTREE   expandedAlgo = ExpandParameters(symbTable, syndexDescript, mainAlgo, parameters);
        PTREE   topAlgo = syndexDescript [mainAlgo];
        topAlgo += <,<>,<>,expandedAlgo [3]>;
        
        // dispatch the software constraint on each algorithm
        DispatchConstraint(syndexDescript, description, topAlgo);
        
        // Check & expand
        CheckExpand(syndexDescript, syndexDescript [mainAlgo]);
        
        // sort again after expansion 
        syndexDescript.Sort();
        
        // put back expanded algo in ref description
        foreach ((),pvRefDescription,{
            PTREE   algo = for_elem ;
            switch ( algo ) {
                case <ALGORITHM> : 
                case <SENSOR> : 
                case <ACTUATOR> : 
                case <MEMORY> : 
                case <CONSTANT> : 
                    {
                        PTREE   replacement = syndexDescript [algo];
                        for_elem = algo += replacement ;
                        
                        // if algo was not treated it might be useless
                        PTREE   indicator = replacement [4];
                        if ( indicator != <CONS> ) {
                            EString message ;
                            message << "Algo not used : " << Value(algo) << "\n";
                            DisplayError(message);
                        }
                        goto for_continue ;
                    }
                default : break ;
            }
        })
        
        // expand this main algo : create the real modeling C++ classes
        PTREE                       refAlgo = syndexDescript [mainAlgo];
        std::vector<ConditionType>  vectCond ;
        pvpComputeUnit = ExpandAlgo(syndexDescript, refAlgo, VString(Value(mainAlgo)), vectCond);
        
        // lock the names of algo in taskname array
        TaskNameLock();
        
        // update architecture there were some algo added with a new time
        ExpandArchi(syndexDescript, mainArchi, true /* update */ );
        
        // compute connectivity matrix
        ComputeConnectivity();
        
        // compute connectivity matrix for algo
#       if 0
            ComputeAlgoConnectivity();
#       endif
        
        // search heads of algo
        SearchAllHeads();
        
        // look if all leaf algo has got a processor
        std::map<EString, ComputeUnit *, LessString> ::iterator iterLeaf ;
        for ( iterLeaf = LeafAlgorithm().begin() ; iterLeaf != LeafAlgorithm().end() ; iterLeaf++ ) {
            if ( (*iterLeaf).second->Procs().size() <= 0 ) {
                EString message ;
                message << "No processor for algo  : " << (*iterLeaf).second->Name() << "\n";
                DisplayError(message);
            }
        }
        
        // update pvDescription with expanded types
        {
            PTREE   top = pvRefDescription ;
            PTREE   list ;
            PTREE   newList ;
            PTREE   elem ;
            top == <SYNDEX_TOP,<>,<>,list>;
            newList = GetTypesFromAlgo(CopyTree(topAlgo), topAlgo [1], "");
            while ( elem = nextl(list) ) {
                if ( elem != <ALGORITHM> && elem != <SENSOR> && elem != <ACTUATOR> && elem != <CONSTANT> && elem != <MEMORY> ) {
                    newList *= elem ;
                }
            }
            top += <,<>,<>,newList>;
        }
    }
}

// GetTypesFromAlgo : create new types from an expanded algo
PTREE Scheduler::GetTypesFromAlgo ( PTREE algo, PTREE typeName, EString prefix )
{
    PTREE   list ;
    
    // create types for all references
    {
        std::vector<PTREE>              vectConditions = ConditionsForAlgo(algo);
        std::vector<PTREE> ::iterator   iterCondition ;
        
        // expand them
        for ( iterCondition = vectConditions.begin() ; iterCondition != vectConditions.end() ; iterCondition++ ) {
            PTREE   elem = *iterCondition ;
            PTREE   references ;
            PTREE   dependences ;
            PTREE   condition ;
            if ( elem == <CONDITIONS,condition,references,dependences> ) {
                PTREE   listReference ;
                PTREE   listDependence ;
                
                // expand all the sub-box of this algorithm
                if ( references == <REFERENCES,listReference> ) {
                    while ( elem = nextl(listReference) ) {
                        PTREE   typeName = <IDENT,PTREE(EString("TYPE_") + prefix + "_" + Value(elem [3]))>;
                        EString newPrefix = prefix ;
                        newPrefix << "_" << Value(elem [3]);
                        elem += <,CopyTree(typeName)>;
                        if ( !list ) 
                            list = GetTypesFromAlgo(elem [5], typeName, newPrefix);
                        else 
                            list = AddListList(list, GetTypesFromAlgo(elem [5], typeName, newPrefix));
                    }
                }
            }
        }
    }
    
    // return this type
    algo += <,CopyTree(typeName)>;
    list *= algo ;
    return list ;
}

// ExpandArchi : expand an architecture from description
void Scheduler::ExpandArchi ( TabList &syndexDescr, PTREE &mainArchi, bool update )
{
    PTREE   archiDescription ;
    PTREE   operators ;   // the operators
    PTREE   medias ;      // the medias
    PTREE   connections ; // the connections
    
    // search the archi
    PTREE   archi = syndexDescr [mainArchi];
    
    if ( !archi ) {
        EString message ;
        message << "Main archi not found : " << Value(mainArchi) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    
    // expand it 
    if ( archi == <ARCHITECTURE,<>,<>,archiDescription> ) {
        
        // search the different element of this architecture
        foreach (<ARCH_OPERATOR_LIST>,archiDescription,{
            operators = for_elem ;
            operators == <ARCH_OPERATOR_LIST,operators>;
            goto for_break ;
        })
        foreach (<ARCH_MEDIA_LIST>,archiDescription,{
            medias = for_elem ;
            medias == <ARCH_MEDIA_LIST,medias>;
            goto for_break ;
        })
        foreach (<ARCH_CONNECTION_LIST>,archiDescription,{
            connections = for_elem ;
            connections == <ARCH_CONNECTION_LIST,connections>;
            goto for_break ;
        })
        
        // connect everything
        {
            
            // entries for operators 
            PTREE   oper ;
            while ( oper = nextl(operators) ) {
                PTREE   type, name ;
                if ( oper == <OPERATOR_DEF,type,name> ) {
                    
                    // search the associated type
                    type = syndexDescr [type];
                    if ( update ) {
                        std::vector<OperatorMedia *> ::iterator iterMedia ;
                        for ( iterMedia = Operators().begin() ; iterMedia != Operators().end() ; iterMedia++ ) {
                            if ( (*iterMedia)->Name() == Value(name) ) {
                                (*iterMedia)->Reference(type);
                                break ;
                            }
                        }
                        continue ;
                    }
                    
                    // if found type create
                    if ( type ) {
                        HardProcUnit    *pUnit = new HardProcUnit(EString(Value(name)), type, pvNumberProc++);
                        {
                            PTREE   code ;
                            PTREE   name ;
                            PTREE   val ;
                            
                            // look if unit is an fpga
                            foreach (<CODE_TIME>,type,{
                                code = for_elem ;
                                code == <,name,val>;
                                if ( VString(Value(name)) == "fpga" && atoi(Value(val)) == 1 ) {
                                    pUnit->Fpga(true);
                                }
                            })
                        }
                        
                        // if box already exists it is an error
                        std::vector<OperatorMedia *> ::iterator iterMedia ;
                        for ( iterMedia = Operators().begin() ; iterMedia != Operators().end() ; iterMedia++ ) {
                            if ( (*iterMedia)->Name() == Value(name) ) {
                                EString message ;
                                message << "Two operators with the same name : " << Value(name) << "\n";
                                DisplayError(message);
                                continue ;
                            }
                        }
                        
                        // insert it 
                        Operators().push_back(pUnit);
                    } else {
                        EString message ;
                        message << "Unknown operator : " << Value(name) << "\n";
                        DisplayError(message);
                    }
                }
            }
            
            // if just updating operator (time) it is ok here
            if ( update ) 
                return ;
            
            // entries for media 
            PTREE   media ;
            PTREE   mediaOrg = media ;
            PTREE   connectionsOrg = connections ;
            while ( media = nextl(medias) ) {
                PTREE   type, name ;
                if ( media == <MEDIA_DEF,type,name> ) {
                    
                    // search the associated type
                    type = syndexDescr [type];
                    
                    // if found type create
                    if ( type ) {
                        LinkUnit                                *pUnit = new LinkUnit(EString(Value(name)), type);
                        
                        // if box already exists it is an error
                        std::vector<OperatorMedia *> ::iterator iterMedia ;
                        for ( iterMedia = Links().begin() ; iterMedia != Links().end() ; iterMedia++ ) {
                            if ( (*iterMedia)->Name() == Value(name) ) {
                                EString message ;
                                message << "Two operators with the same name : " << Value(name) << "\n";
                                DisplayError(message);
                                continue ;
                            }
                        }
                        
                        // insert it
                        Links().push_back(pUnit);
                        
                        // links all the connected processors
                        PTREE                           connection ;
                        std::vector<OperatorMedia *>    procUnit ;
                        connections = connectionsOrg ;
                        while ( connection = nextl(connections) ) {
                            PTREE           end, nameConnection ;
                            unsigned int    nbConnected = 0 ;
                            if ( connection == <CONNECTION_DEF,end,nameConnection> && VString(Value(name)) == Value(nameConnection) ) {
                                unsigned int    index ;
                                for ( index = 0 ; index < pvOperators.size() ; index++ ) {
                                    if ( pvOperators [index]->Name() == Value(end) ) {
                                        procUnit.push_back(pvOperators [index]);
                                        pvOperators [index]->Linked().push_back(pUnit);
                                        nbConnected++ ;
                                        pUnit->AllOperators().push_back(pvOperators [index]);
                                        
                                        // store channel dependency
                                        PTREE   channelName ;
                                        if ( end == <END_DEPEND,<,<>,<,channelName>>> ) 
                                            ;
                                        pUnit->ChannelByOperator()[Value(end)] = Value(channelName);
                                    }
                                }
                                if ( nbConnected < 1 ) {
                                    EString message ;
                                    message << "Connection not found for  : " << Value(nameConnection) << "\n";
                                    DisplayError(message);
                                }
                            }
                        }
                        if ( procUnit.size() < 2 ) {
                            EString message ;
                            message << "Not enough Connections found for  : " << Value(name) << "\n";
                            DisplayError(message);
                        }
                        pUnit->Linked(procUnit);
                    } else {
                        EString message ;
                        message << "Unknown media : " << Value(name) << "\n";
                        DisplayError(message);
                    }
                }
            }
        }
    }
}

// ExpandAlgo : expand main algo from description
ComputeUnit *Scheduler::ExpandAlgo ( TabList &syndexDescr, PTREE &mainAlgo, EString nameBox, std::vector<ConditionType> &vectCond )
{
    PTREE   algoDescription, listAlgo ;
    PTREE   elem, list ;
    
    // expand all the elements 
    PTREE   algo = mainAlgo ;
    EString name ;
    PTREE   listProc ;
    
    name = Value(algo);
    if ( !algo ) {
        EString message ;
        message << "Algo not found : " << Value(mainAlgo) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    
    // if really an algo expand it
    if ( algo == <ALGORITHM,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <SENSOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <ACTUATOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <MEMORY,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <CONSTANT,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>> ) {
        list = listAlgo ;
        listProc = algo [5];
        
        // if it is a soft unit (a leaf it has not conditional unit)
        bool    foundCondition = false ;
        while ( elem = nextl(list) ) {
            if ( elem == <CONDITIONS> ) 
                foundCondition = true ;
        }
        
        // create a end point for all the port since software unit
        if ( !foundCondition ) {
            
            // if box already exists it is an error
            std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo = Algorithm().find(nameBox);
            if ( iterAlgo != Algorithm().end() ) {
                EString message ;
                message << "Two algo with the same name : " << nameBox << "\n";
                DisplayError(message);
            }
            
            // create a simple box for this algo
            ProcUnit    *pUnit ;
            switch ( algo ) {
                case <MEMORY> : 
                    pUnit = new MemoryUnit(nameBox, algo, vectCond);
                    pUnit->NameNumber(TaskName(nameBox));
                    pUnit->RefNumber(TaskName(Value(algo)));
                    break ;
                case <CONSTANT> : 
                    pUnit = new ConstantUnit(nameBox, algo, vectCond);
                    pUnit->NameNumber(TaskName(nameBox));
                    pUnit->RefNumber(TaskName(Value(algo)));
                    break ;
                default : 
                    pUnit = new ProcUnit(nameBox, algo, vectCond);
                    pUnit->NameNumber(TaskName(nameBox));
                    pUnit->RefNumber(TaskName(Value(algo)));
                    break ;
            }
            GetProcessor(pUnit, algo);
            PTREE   listPort = listAlgo ;
            PTREE   port ;
            while ( port = nextl(listPort) ) {
                switch ( port ) {
                    case <IN> : 
                    case <OUT> : 
                        {
                            
                            // AddEndConnection(port);
                        }
                        break ;
                    default : break ;
                }
            }
            
            // insert this box in dictionnary
            pvAlgo [nameBox] = pUnit ;
            pvAlgoByNumber [pUnit->NameNumber()] = pUnit ;
            LeafAlgorithm()[nameBox] = pUnit ;
            pUnit->Identifier(pvNumberAlgo++);
            
            // return this small box
            return pUnit ;
        } // not a leaf
        else {
            
            // if box already exists it is an error
            std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo = Algorithm().find(nameBox);
            if ( iterAlgo != Algorithm().end() ) {
                EString message ;
                message << "Two algo with the same name : " << nameBox << "\n";
                DisplayError(message);
            }
            
            // create a box for this algo
            ConditionalUnit *pConditionalUnit = new ConditionalUnit(nameBox, algo);
            pConditionalUnit->NameNumber(TaskName(nameBox));
            pConditionalUnit->RefNumber(TaskName(Value(algo)));
            GetProcessor(pConditionalUnit, algo);
            
            // insert this box in dictionnary
            pvAlgo [nameBox] = pConditionalUnit ;
            pvAlgoByNumber [pConditionalUnit->NameNumber()] = pConditionalUnit ;
            
            // for each condition expand
            list = listAlgo ;
            unsigned int                index = 0 ;
            unsigned int                nbCondition = 0 ;
            unsigned int                refCondition = 0 ;
            std::vector<ConditionType>  newCondition ;
            while ( elem = nextl(list) ) {
                PTREE   condition ;
                PTREE   references ;
                PTREE   dependences ;
                if ( elem == <CONDITIONS,condition,references,dependences> ) {
                    PTREE   listReference ;
                    PTREE   listDependence ;
                    
                    // if condition is not true creates a new condition
                    newCondition = vectCond ;
                    if ( condition != <TRUE> ) {
                        
                        // if condition doesn't exist create it
                        if ( nbCondition == 0 ) 
                            Condition(refCondition = Condition() + 1);
                        
                        // store this condition
                        pvConditionByNumber [refCondition] = ConditionDescriptor(condition);
                        
                        // this is another case of this condition
                        nbCondition += 1 ;
                        newCondition.push_back(CreateCondition(refCondition, nbCondition));
                    }
                    
                    // construct a unit for this condition
                    ProcUnit    *pCondUnit = new ProcUnit(VString(""), elem, newCondition);
                    GetProcessor(pCondUnit, algo);
                    
                    // insert it in master
                    pConditionalUnit->AddUnit(condition, *pCondUnit);
                    
                    // expand all the sub-box of this algorithm
                    if ( references == <REFERENCES,listReference> ) {
                        PTREE   aReference ;
                        while ( aReference = nextl(listReference) ) {
                            
                            // expand
                            EString nameBoxSon ;
                            nameBoxSon << nameBox << "_" << Value(aReference [3]);
                            PTREE       referenceTree = aReference [5];
                            ComputeUnit *pUnit ;
                            pUnit = ExpandAlgo(syndexDescr, referenceTree, nameBoxSon, newCondition);
                            
                            // add this subunit to current conditional unit
                            pCondUnit->AddUnit(VString(Value(aReference [3])), *pUnit);
                        }
                    }
                    
                    // do the data creation and connection
                    if ( dependences == <DEPENDENCES,listDependence> ) {
                        std::vector<std::vector<PTREE> >    links ;
                        ComputeLinks(listDependence, links);
                        
                        // compute all the list not connected to the outside
                        unsigned int    indexLinks ;
                        for ( indexLinks = 0 ; indexLinks < links.size() ; indexLinks++ ) {
                            bool                            outSideConnected = false ;
                            bool                            precedence = false ;
                            std::vector<PTREE> ::iterator   iter ;
                            for ( iter = links [indexLinks].begin() ; iter != links [indexLinks].end() ; iter++ ) {
                                PTREE   elem = *iter ;
                                if ( ListLength(elem) == 1 ) 
                                    outSideConnected = true ;
                                if ( elem == <,<PRECEDENCE_LINK>> ) 
                                    precedence = true ;
                            }
                            
                            // if not outside connected create link
                            // !! should do a find pass to find connected to outside but with exterior not connected
                            // should suppress this outside connection
                            if ( !outSideConnected || precedence ) {
                                std::vector<ComputeUnit *>  connected ;
                                
                                // create a trans link
                                TransUnit                   *pTrans = new TransUnit ;
                                bool                        first = true ;
                                unsigned int                nbOut = 0 ;
                                for ( iter = links [indexLinks].begin() ; iter != links [indexLinks].end() ; iter++, first = false ) {
                                    PTREE   box ;
                                    PTREE   port ;
                                    box = (*iter)[1];
                                    port = (*iter)[2];
                                    ComputeUnit *pUnit = &pCondUnit->Unit(VString(Value(box)));
                                    if ( !pUnit || pUnit->Type() == ComputeUnit::None ) {
                                        EString message ;
                                        message << " Error cannot find unit : " << Value(box) << "\n";
                                        DisplayError(message);
                                    } else {
                                        unsigned int    length = connected.size();
                                        if ( precedence ) 
                                            pUnit->SearchConnected(box, connected, pTrans, first);
                                        else 
                                            pUnit->SearchConnected(port, connected, pTrans, first);
                                        if ( connected.size() == length ) {
                                            EString message ;
                                            message << " No end connection found for  : " << Value(box) << "." << Value(port) << "\n";
                                            DisplayError(message);
                                        }
                                        if ( first ) 
                                            nbOut = connected.size();
                                    }
                                }
                                
                                // setup the connection
                                pTrans->ConnectedIn().clear();
                                pTrans->ConnectedOut().clear();
                                unsigned int    indexComm ;
                                
                                // because of conditions they can be several out for the in
                                for ( indexComm = 0 ; indexComm < nbOut ; indexComm++ ) 
                                    pTrans->ConnectedOut().push_back(connected [indexComm]);
                                for (; indexComm < connected.size() ; indexComm++ ) {
                                    pTrans->ConnectedIn().push_back(connected [indexComm]);
                                }
                                
                                // insert current box
                                pCondUnit->TransVect().push_back(pTrans);
                                pTrans->RefNumber(TaskName(Value(pTrans->Reference())));
                            }
                        }
                    }
                }
            }
            
            // return unit
            return pConditionalUnit ;
        }
    } else {
        DumpTree(algo);
        NewLine();
        EString message("What to do ? \n");
        DisplayError(message);
        exit(3);
    }
    return 0 ;
}

// ComputeLinks : compute the sets of links : in a set are all the interconnected port 
void Scheduler::ComputeLinks ( PTREE &listDependence, std::vector<std::vector<PTREE> > &links )
{
    PTREE   aDependence ;
    
    while ( aDependence = nextl(listDependence) ) {
        PTREE   start ;
        PTREE   end ;
        bool    found = false ;
        if ( aDependence == <DEPENDENCE,<>,<END_DEPEND,start>,<END_DEPEND,end>> ) 
            ;
        if ( aDependence == <PRECEDENCE,<>,<END_DEPEND,start>,<END_DEPEND,end>> ) {
            PTREE   name ;
            start = CopyTree(start);
            end = CopyTree(end);
            start == <,name>;
            start += <,<PRECEDENCE_LINK,name>>;
            end == <,name>;
            end += <,<PRECEDENCE_LINK,name>>;
        }
        
        // search all the sets to see if not already inserted here        
        unsigned int    indexLinks ;
        for ( indexLinks = 0 ; indexLinks < links.size() ; indexLinks++ ) {
            std::vector<PTREE> ::iterator   iter ;
            iter = links [indexLinks].begin();
            PTREE   entry = *iter ;
            if ( VString(Value(entry)) == VString(Value(start)) && VString(Value(entry [2])) == VString(Value(start [2])) ) {
                found = true ;
                links [indexLinks].push_back(end);
                break ;
            }
        }
        
        // if not found creates a new set
        if ( !found ) {
            std::vector<PTREE>  newSet ;
            newSet.push_back(start);
            newSet.push_back(end);
            links.push_back(newSet);
        }
    }
}

// AffectProcessor : affect algo to a processor
void Scheduler::AffectProcessor ( EString &algo, EString &processor )
{
    ComputeUnit     *pUnit = pvAlgo [algo];
    unsigned int    index ;
    
    if ( !pUnit ) {
        EString message ;
        message << "Unknown algo : " << algo << "\n";
        DisplayError(message);
        return ;
    }
    
    bool            done = false ;
    unsigned int    size = Operators().size();
    
    for ( index = 0 ; index < size ; index++ ) {
        if ( Operators()[index]->Name() == processor ) {
            pUnit->Proc(*Operators()[index]);
            done = true ;
        }
    }
    if ( !done ) {
        EString message ;
        message << "Cannot find proc  : " << processor << "\n";
        DisplayError(message);
    }
}

// GetProcessors : get processors for an algo
void Scheduler::GetProcessors ( int algo, std::vector<int> &procs )
{
    std::map<EString, ComputeUnit *, LessString> ::iterator iterProc = LeafAlgorithm().begin();
    
    if ( algo ) 
        std::advance(iterProc, algo);
    procs = (*iterProc).second->Procs();
}

// GetProcessor : affect a processor to a unit
void Scheduler::GetProcessor ( ComputeUnit *pUnit, PTREE &algo )
{
    PTREE               listOperator = algo [5];
    std::vector<int>    setOperator ;
    unsigned int        size = Operators().size();
    
    if ( !listOperator ) {
        for ( unsigned int index = 0 ; index < size ; index++ ) 
            setOperator.push_back(index);
    } else {
        PTREE   operatorElem ;
        while ( operatorElem = nextl(listOperator) ) {
            for ( unsigned int index = 0 ; index < size ; index++ ) {
                if ( Operators()[index]->Name() == Value(operatorElem) ) {
                    setOperator.push_back(index);
                }
            }
        }
    }
    pUnit->Procs(setOperator);
}

static int  indent = 0 ;
static int  indexCompute = 0 ;

#define STEP_INDENT 4

inline void StartTime ( ComputeUnit *comp, unsigned int dir )
{
#   ifdef DEBUG
        
        EString message ;
        int     index = indent ;
        
        while ( index-- ) {
            message << ' ';
        }
        if ( dir == 2 ) 
            message << "SU ";
        else if ( dir == 3 ) 
            message << "SD ";
        else 
            message << "S  ";
        message << comp->Name() << " " << comp->Proc().Name();
        while ( message.length() < 60 ) 
            message << " ";
        message << "(" << (int)indexCompute++ << ")" << "\n";
        indent += STEP_INDENT ;
        _write(2, message, strlen(message));
#   endif
}

inline void EndTime ( ComputeUnit *comp, unsigned int dir )
{
#   ifdef DEBUG
        indent -= STEP_INDENT ;
        
        EString message ;
        int     index = indent ;
        
        while ( index-- ) {
            message << ' ';
        }
        if ( dir == 2 ) 
            message << "EU ";
        else if ( dir == 3 ) 
            message << "ED ";
        else 
            message << "E  ";
        message << comp->Name() << " " << comp->Proc().Name();
        if ( dir == 1 ) 
            message << " : " << comp->StartTime() << " - " << comp->ComputeTime();
        while ( message.length() < 60 ) 
            message << " ";
        message << "(" << (int)indexCompute++ << ")";
        message << "\n";
        _write(2, message, strlen(message));
#   endif
}

// UpdateHead : update head after taking one element
void Scheduler::UpdateHead ( ComputeUnit *algo, std::set<ComputeUnit *> &head, std::set<ComputeUnit *> &computed )
{
    
    // compute the new entries
    std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
    
    // for memory do not insert anything
    if ( algo->Type() == ComputeUnit::MemoryUnit ) 
        return ;
    
    // otherwise compute
    for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
        TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
        
        // look all the in comm and compute
        unsigned int    indexComm ;
        for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
            ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
            bool        readyToInsert = true ;
            
            // if this unit doesn't depend on its input don't put it in the trail it is already a head
            if ( !pComputeUnit->InDependant() ) 
                continue ;
            
            // if one predecesor has not been visited yet cannot insert
            std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
            for ( iterIn = pComputeUnit->InEndConnectionsRef().begin() ; readyToInsert && iterIn != pComputeUnit->InEndConnectionsRef().end() ; iterIn++ ) {
                TransUnit       *pTransUnit2 = (TransUnit *)((*iterIn).second);
                
                // look if predecessors computed
                unsigned int    indexComm2 ;
                for ( indexComm2 = 0 ; indexComm2 < pTransUnit2->ConnectedOut().size() ; indexComm2++ ) {
                    ComputeUnit *pComputeUnit2 = pTransUnit2->ConnectedOut()[indexComm2];
                    
                    // if predecessor is a memory no need to compute it before
                    if ( pComputeUnit2->Type() == ComputeUnit::MemoryUnit ) 
                        continue ;
                    
                    // otherwise look if computed
                    if ( computed.find(pComputeUnit2) == computed.end() ) {
                        readyToInsert = false ;
                        break ;
                    }
                }
            }
            
            // if ready insert
            if ( readyToInsert ) {
                if ( computed.find(pComputeUnit) != computed.end() ) {
                    EString message("Trying to reinsert : ");
                    message << pComputeUnit->Name() << "\n";
                    DisplayError(message);
                }
                head.insert(pComputeUnit);
            }
        }
    }
}

// RandomTrail : search a trail for computing time
std::vector<unsigned int> Scheduler::RandomTrail ()
{
    std::set<ComputeUnit *>     computed ;
    std::vector<unsigned int>   trail ;
    std::set<ComputeUnit *>     head = Heads();
    
    while ( !head.empty() ) {
        
        // select a unit and put it in trail
        int                                 index = RandomValue(0, head.size());
        std::set<ComputeUnit *> ::iterator  iterUnit = head.begin();
        if ( index ) 
            std::advance(iterUnit, index);
        ComputeUnit *algo = *iterUnit ;
        if ( algo == 0 ) {
            EString message("Trying to insert null unit \n");
            DisplayError(message);
            exit(3);
        }
        trail.push_back(algo->NameNumber());
        computed.insert(algo);
        head.erase(iterUnit);
        
        // update head
        UpdateHead(algo, head, computed);
    }
    if ( trail.size() != LeafAlgorithm().size() ) {
        EString message("Trail of wrong size in random initialization \n");
        DisplayError(message);
        std::set<ComputeUnit *>                                 selected ;
        std::vector<unsigned int> ::iterator                    iterTrail ;
        std::map<EString, ComputeUnit *, LessString> ::iterator iterLeaf ;
        for ( iterTrail = trail.begin() ; iterTrail != trail.end() ; iterTrail++ ) {
            selected.insert(pvAlgoByNumber [*iterTrail]);
        }
        message = "Missing : ";
        for ( iterLeaf = LeafAlgorithm().begin() ; iterLeaf != LeafAlgorithm().end() ; iterLeaf++ ) {
            if ( selected.find((*iterLeaf).second) == selected.end() ) {
                message << (*iterLeaf).second->Name() << " ";
            }
        }
        message << "\n";
        DisplayError(message);
        exit(3);
    }
    
    // return trail
    return trail ;
}

// MutateTrail : mutate a trail
void Scheduler::MutateTrail ( int *trail )
{
    std::set<ComputeUnit *>     computed ;
    std::set<ComputeUnit *>     head = Heads();
    int                         nbMutate = RandomValue(1, LeafAlgorithm().size() + 1);
    int                         *newTrail = new int [LeafAlgorithm().size()];
    
    if ( nbMutate > MAX_MODIFICATION ) 
        nbMutate = MAX_MODIFICATION ;
    
    std::set<unsigned int>  modifStep ;
    
    // select points of mutation for the invidual
    while ( nbMutate-- ) 
        modifStep.insert(RandomValue(0, LeafAlgorithm().size()));
    
    unsigned int    step ;
    unsigned int    trailStep = 0 ;
    
    // fill each slot of new trail
    for ( step = 0 ; !head.empty() ; step++ ) {
        
        // select a unit and put it in trail
        ComputeUnit *algo = 0 ;
        {
            int                                 index ;
            std::set<ComputeUnit *> ::iterator  iterUnit ;
            
            // we try to change it is impossible if head is alread of size 1
            if ( modifStep.find(step) == modifStep.end() || head.size() == 1 ) {
                
                // we try to take the next step in old trail
                if ( trailStep < LeafAlgorithm().size() ) {
                    do {
                        algo = pvAlgoByNumber [trail [trailStep++]];
                    } while ( computed.find(algo) != computed.end() && trailStep < LeafAlgorithm().size() );
                    
                    // this step is no longer here so will take one available
                    if ( head.find(algo) == head.end() ) {
                        algo = 0 ;
                        trailStep-- ;
                    }
                }
                
                // if we didn't find anything choose a random value in head
                if ( !algo ) {
                    index = RandomValue(0, head.size());
                    iterUnit = head.begin();
                    if ( index ) 
                        std::advance(iterUnit, index);
                    algo = *iterUnit ;
                }
            } else {
                
                // we want to take another trail, this step is a mutation point
                do {
                    index = RandomValue(0, head.size());
                    iterUnit = head.begin();
                    if ( index ) 
                        std::advance(iterUnit, index);
                    algo = *iterUnit ;
                } while ( algo->NameNumber() == (unsigned int)(trail [trailStep]) );
            }
        }
        
        // tread found algo
        newTrail [step] = algo->NameNumber();
        if ( algo == 0 ) {
            EString message("Trying to insert null unit \n");
            DisplayError(message);
            exit(3);
        }
        computed.insert(algo);
        head.erase(algo);
        
        // update head
        UpdateHead(algo, head, computed);
    }
    if ( step != LeafAlgorithm().size() ) {
        EString message("Trail of wrong size in random mutation \n");
        DisplayError(message);
        exit(3);
    }
    
    // return trail
    memcpy(trail, newTrail, LeafAlgorithm().size() * sizeof(int));
    delete [] newTrail ;
}

// SearchAllHeads
void Scheduler::SearchAllHeads ()
{
    
    // search the first proc unit
    ComputeUnit                                             *firstAlgo = (*LeafAlgorithm().begin()).second ;
    
    // the trail
    std::set<ComputeUnit *>                                 trail ;
    std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo = LeafAlgorithm().begin();
    
    for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
        if ( trail.find((*iterAlgo).second) == trail.end() ) {
            SearchHead((*iterAlgo).second, trail);
        }
    }
    if ( trail.size() != LeafAlgorithm().size() ) {
        EString message("SearchAllHeads didn't find all elements of trail \n");
        DisplayError(message);
        exit(3);
    }
}

// SearchHeadUp
void Scheduler::SearchHeadUp ( ComputeUnit *algo, std::set<ComputeUnit *> &trail )
{
    
    // start for this algo
    StartTime(algo, 2);
    
    // if this algo was already computed exclude
    if ( trail.find(algo) != trail.end() ) 
        return ;
    
    // for all the in connection compute time up
    std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
    std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
    
    if ( algo->InDependant() ) 
        for ( iterIn = algo->InEndConnectionsRef().begin() ; iterIn != algo->InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // look all the in comm and compute
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                if ( pComputeUnit->Type() != ComputeUnit::MemoryUnit ) 
                    SearchHead(pComputeUnit, trail, false);
            }
        }
    
    // end for this algo
    EndTime(algo, 2);
}

// SearchHeadDown
void Scheduler::SearchHeadDown ( ComputeUnit *algo, std::set<ComputeUnit *> &trail )
{
    
    // start for this algo
    StartTime(algo, 3);
    
    // compute all the entries down
    // for all the in connection compute time up
    std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
    
    // stop at the memory 
    if ( algo->Type() != ComputeUnit::MemoryUnit ) 
        for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
            
            // look all the in comm and compute
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                if ( pComputeUnit->InDependant() ) 
                    SearchHead(pComputeUnit, trail);
            }
        }
    
    // end for this algo
    EndTime(algo, 3);
}

// ComputeTime : compute time of an algo 
void Scheduler::SearchHead ( ComputeUnit *algo, std::set<ComputeUnit *> &trail, bool down )
{
    
    // if this algo was already computed exclude
    if ( trail.find(algo) != trail.end() ) 
        return ;
    
    // be sure that all up have been computed (up flow)
    SearchHeadUp(algo, trail);
    
    // end for this algo
    StartTime(algo, 1);
    
    // if no in put this algo in head
    if ( !algo->InDependant() || algo->InEndConnectionsRef().size() == 0 ) {
        Heads().insert(algo);
    } else {
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        
        // if all inputs are memory it is also ok
        bool                                                    sthingBefore = false ;
        for ( iterIn = algo->InEndConnectionsRef().begin() ; iterIn != algo->InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // look all the in comm and compute
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() && !sthingBefore ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                
                // if predecessor is a memory no need to compute it before
                if ( pComputeUnit->Type() != ComputeUnit::MemoryUnit ) 
                    sthingBefore = true ;
            }
        }
        
        // if nothing before insert
        if ( !sthingBefore ) 
            Heads().insert(algo);
    }
    
    // mark path
    trail.insert(algo);
    
    // end for this algo
    EndTime(algo, 1);
    
    // be sure that all down have been computed
    if ( down ) 
        SearchHeadDown(algo, trail);
}

void Scheduler::ComputeTransfert ( ComputeUnit *algo, ComputeUnit *pOutUnit, TransUnit *pTransUnit, time_t &delayTime )
{
    unsigned    sigmaTime = 0 ;
    
    if ( &pOutUnit->Proc() != &algo->Proc() ) {
        
        // search a connection between this algo and current one
        // OperatorMedia   *pOperatorMedia = pOutUnit->Proc().SearchConnection(&algo->Proc());
        std::vector<OperatorMedia *> ::iterator iterMedia ;
        std::vector<OperatorMedia *>            vectMedia = Connectivity()[MATRIX_ACCESS(pOutUnit->Proc().Index(), algo->Proc().Index())];
        
        // ask to affect a task about this transmission
        if ( vectMedia.size() >= 1 ) {
            PTREE                       null ;
            EString                     name ;
            
            // if conditions not compatible skip
            std::vector<ConditionType>  conditionVect ;
            if ( !MergeCondition(conditionVect, algo->Conditions(), pOutUnit->Conditions()) ) 
                return ;
            
            // add time for each link
            unsigned        indexTransaction = 1 ;
            OperatorMedia   *pStartProc = &pOutUnit->Proc();
            OperatorMedia   *pEndProc = 0 ;
            for ( iterMedia = vectMedia.begin() ; iterMedia != vectMedia.end() ; iterMedia++ ) {
                EString         taskName = name ;
                unsigned int    taskNumber ;
                if ( vectMedia.size() > 1 ) 
                    taskName << " (" << (int)indexTransaction++ << "/" << (int)vectMedia.size() << ")";
                sigmaTime += (*iterMedia)->ComputeTime(taskName, pTransUnit->Reference(), pTransUnit->RefNumber());
            }
        }
    }
    
    // update delay
    delayTime += sigmaTime ;
}

void Scheduler::UpdateStart ( ComputeUnit *algo, ComputeUnit *pOutUnit, TransUnit *pTransUnit, const EString &channelName, time_t &startTime )
{
    if ( &pOutUnit->Proc() != &algo->Proc() ) {
        
        // search a connection between this algo and current one
        // OperatorMedia   *pOperatorMedia = pOutUnit->Proc().SearchConnection(&algo->Proc());
        std::vector<OperatorMedia *>    vectMedia = Connectivity()[MATRIX_ACCESS(pOutUnit->Proc().Index(), algo->Proc().Index())];
        
        // ask to affect a task about this transmission
        if ( vectMedia.size() >= 1 ) {
            PTREE                       null ;
            EString                     name ;
            
            // if conditions not compatible skip
            std::vector<ConditionType>  conditionVect ;
            if ( !MergeCondition(conditionVect, algo->Conditions(), pOutUnit->Conditions()) ) 
                return ;
            
            // insert task
            name << pOutUnit->Name() << "->" << algo->Name();
            std::vector<OperatorMedia *> ::iterator iterMedia ;
            time_t                                  currentStart ;
            
            // if pOutUnit is a memory computation was done in previous cycle
            if ( pOutUnit->Type() == ComputeUnit::MemoryUnit || !algo->InDependant() ) {
                currentStart = 0 ;
            } else {
                currentStart = pOutUnit->StartTime() + pOutUnit->ComputeTime();
            }
            
            // add time for each link
            unsigned        indexTransaction = 1 ;
            OperatorMedia   *pStartProc = &pOutUnit->Proc();
            OperatorMedia   *pEndProc = 0 ;
            for ( iterMedia = vectMedia.begin() ; iterMedia != vectMedia.end() ; iterMedia++ ) {
                EString         taskName = name ;
                unsigned int    taskNumber ;
                if ( vectMedia.size() > 1 ) 
                    taskName << " (" << (int)indexTransaction++ << "/" << (int)vectMedia.size() << ")";
                unsigned    task
                    = (*iterMedia)->CreateTask((taskNumber = TaskName(taskName)), pTransUnit->Reference(), pTransUnit->RefNumber(), currentStart, conditionVect
                        , taskName);
                if ( task ) {
                    currentStart = (*iterMedia)->TaskContent(task)->End();
                    
                    // keep informations about this task
                    {
                        
                        // compute start and end proc
                        std::vector<OperatorMedia *> ::iterator iterProc ;
                        pEndProc = 0 ;
                        LinkUnit    *linkUnit = (LinkUnit *)(*iterMedia);
                        for ( iterProc = linkUnit->AllOperators().begin() ; iterProc != linkUnit->AllOperators().end() ; iterProc++ ) {
                            if ( *iterProc != pStartProc ) 
                                pEndProc = *iterProc ;
                        }
                        
                        // store data
                        pvTransByNumber.insert(
                            std::pair<int, TransTrunk> (pTransUnit->RefNumber()
                                , TransTrunk(pTransUnit, pStartProc, pEndProc, pOutUnit, algo, channelName, (*iterMedia), task)));
                        
                        // prepare next step
                        pStartProc = pEndProc ;
                        pEndProc = 0 ;
                    }
                }
            }
            if ( currentStart > startTime ) 
                startTime = currentStart ;
        } else {
            EString message ;
            message << "Missing connection for algo  : " << algo->Name() << " " << pOutUnit->Name() << "\n";
            DisplayError(message);
        }
        
        // update main task comm time
    } else {
        
        // if pOutUnit is a memory it is available at beginning
        if ( pOutUnit->Type() == ComputeUnit::MemoryUnit ) 
            return ;
        
        // Compute the time
        time_t  newTime = pOutUnit->StartTime() + pOutUnit->ComputeTime();
        if ( !algo->InDependant() ) 
            newTime = 0 ;
        if ( newTime > startTime ) 
            startTime = newTime ;
    }
}

// ComputeTime : compute time of an algo 
void Scheduler::ComputeTime ( std::vector<int> &trail )
{
    std::vector<int> ::iterator iterUnit ;
    
#   if 0
        
        // emit data coming from memory
        for ( iterUnit = trail.begin() ; iterUnit != trail.end() ; iterUnit++ ) {
            ComputeUnit *algo = pvAlgoByNumber [*iterUnit];
            if ( !algo || *iterUnit > (int)pvTaskNameLock ) {
                EString message("Invalid unit in trail \n");
                message << (int)*iterUnit << " " << (int)pvTaskNameLock << "\n";
                DisplayError(message);
                continue ;
            }
            if ( algo->Type() != ComputeUnit::MemoryUnit ) 
                continue ;
            
            // emit for each output
            StartTime(algo, 1);
            time_t                                                  startTime = 0 ;
            std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
            std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
            for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
                TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
                
                // look all the in comm and compute
                unsigned int    indexComm ;
                for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                    ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                    
                    // search the in
                    for ( iterIn = pComputeUnit->InEndConnectionsRef().begin() ; iterIn != pComputeUnit->InEndConnectionsRef().end() ; iterIn++ ) {
                        TransUnit   *pInTransUnit = (TransUnit *)((*iterOut).second);
                        if ( pInTransUnit == pTransUnit ) 
                            break ;
                    }
                    UpdateStart(pComputeUnit, algo, pTransUnit, (*iterIn).first, startTime, true /* take memory into account */ );
                }
            }
            
            // end for this algo
            EndTime(algo, 1);
        }
#   endif
    
    // dispatch the tasks on operators
    for ( iterUnit = trail.begin() ; iterUnit != trail.end() ; iterUnit++ ) {
        ComputeUnit *algo = pvAlgoByNumber [*iterUnit];
        if ( !algo || *iterUnit > (int)pvTaskNameLock ) {
            EString message("Invalid unit in trail \n");
            message << (int)*iterUnit << " " << (int)pvTaskNameLock << "\n";
            DisplayError(message);
            continue ;
        }
        
        // end for this algo
        StartTime(algo, 1);
        
        // than compute start time
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        time_t                                                  startTime = 0 ;
        for ( iterIn = algo->InEndConnectionsRef().begin() ; iterIn != algo->InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // search the out unit
            unsigned int    indexComm ;
            ComputeUnit     *pOutUnit = 0 ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                pOutUnit = pComputeUnit ;
                UpdateStart(algo, pOutUnit, pTransUnit, (*iterIn).first, startTime);
            }
            if ( !pOutUnit ) {
                EString message ;
                message << "Link without emitter \n";
                DisplayError(message);
                pTransUnit->Computed(true);
            }
        }
        
        // if no dma we must add transfer time to processor compute time
        time_t  transferTime = 0 ;
        if ( !UseDma() ) {
            std::set<OperatorMedia *>                               seenMedia ;
            std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
            for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
                TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
                
                // search the out unit
                unsigned int    indexComm ;
                for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                    ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                    
                    // search the in
                    for ( iterIn = pComputeUnit->InEndConnectionsRef().begin() ; iterIn != pComputeUnit->InEndConnectionsRef().end() ; iterIn++ ) {
                        TransUnit   *pInTransUnit = (TransUnit *)((*iterOut).second);
                        if ( seenMedia.find(&pComputeUnit->Proc()) == seenMedia.end() ) {
                            if ( pInTransUnit == pTransUnit ) {
                                ComputeTransfert(pComputeUnit, algo, pTransUnit, transferTime);
                                seenMedia.insert(&pComputeUnit->Proc());
                                break ;
                            }
                        }
                    }
                }
            }
        }
        
        // set the start time
        algo->StartTime(startTime);
        
        // compute the time in task
        unsigned    task
            = algo->Proc().CreateTask(algo->NameNumber(), algo->Reference(), algo->RefNumber(), algo->StartTime(), algo->Conditions(), algo->Name(), transferTime);
        
        // set compute time
        if ( task ) {
            Task    *pTask = algo->Proc().TaskContent(task);
            algo->StartTime(pTask->Start()).ComputeTime(pTask->End() - pTask->Start());
        } else {
            algo->ComputeTime(0);
            EString message ;
            message << "No proc for : " << algo->Name() << "\n";
            DisplayError(message);
        }
        
        // now this task is computed
        algo->Computed(true);
        
        // end for this algo
        EndTime(algo, 1);
    }
}

// DirectDescent : look if one is direct descent of the other
bool Scheduler::DirectDescent ( ComputeUnit *firstUnit, ComputeUnit *secondUnit )
{
    if ( firstUnit == secondUnit ) 
        return true ;
    if ( firstUnit->Identifier() < 1 || firstUnit->Identifier() >= pvNumberAlgo ) 
        return false ;
    if ( secondUnit->Identifier() < 1 || secondUnit->Identifier() >= pvNumberAlgo ) 
        return false ;
    if ( ALGO_MATRIX_GET(AlgoConnectivity(), firstUnit->Identifier(), secondUnit->Identifier())
            || ALGO_MATRIX_GET(AlgoConnectivity(), secondUnit->Identifier(), firstUnit->Identifier()) ) 
        return true ;
    return false ;
}

// ComputeAlgoConnectivity : compute the algorithm connectivity matrix
void Scheduler::ComputeAlgoConnectivity ()
{
    unsigned int    size = pvNumberAlgo ;
    unsigned int    index = ALGO_MATRIX_SIZE(pvNumberAlgo + 1);
    
    while ( index-- ) {
        AlgoConnectivity().push_back(0);
    }
    
    // fill with direct links
    std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo ;
    
    for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
        
        // if it is really a compute algorithm put it in matrix
        if ( (*iterAlgo).second->Identifier() ) {
            ComputeUnit                                             *algo = (*iterAlgo).second ;
            std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
            for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
                TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
                
                // look all the in comm and compute
                unsigned int    indexComm ;
                for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                    ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                    
                    // if other is also a compute algorithm put the connection
                    if ( pComputeUnit->Identifier() ) {
                        ALGO_MATRIX_PUT(AlgoConnectivity(), algo->Identifier(), pComputeUnit->Identifier(), true);
                    }
                }
            }
        }
    }
    
    // fill with known values
    std::vector<unsigned char>  newMatrix = AlgoConnectivity();
    
    // complete with more hops
    unsigned int                indexLine ;
    unsigned int                indexCol ;
    unsigned int                indexSearch ;
    bool                        sthingDone = true ;
    
    while ( sthingDone ) {
        sthingDone = false ;
        for ( indexLine = 1 ; indexLine <= size ; indexLine++ ) {
            for ( indexCol = 1 ; indexCol <= size ; indexCol++ ) {
                if ( indexLine != indexCol ) {
                    if ( !ALGO_MATRIX_GET(newMatrix, indexLine, indexCol) ) {
                        for ( indexSearch = 1 ; indexSearch <= size ; indexSearch++ ) {
                            if ( ALGO_MATRIX_GET(AlgoConnectivity(), indexLine, indexSearch) && ALGO_MATRIX_GET(AlgoConnectivity(), indexSearch, indexCol) ) {
                                ALGO_MATRIX_PUT(newMatrix, indexLine, indexCol, true);
                                sthingDone = true ;
                                break ;
                            }
                        }
                    }
                }
            }
        }
        AlgoConnectivity() = newMatrix ;
    }
}

// DisplayConnectivity : Display Connectivity matrix for algos
void Scheduler::DisplayConnectivity ()
{
    
    // fill with direct links
    std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo ;
    std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo1 ;
    EString                                                 message ;
    
    for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
        message << "\nSons for : " << (*iterAlgo).second->Name() << "\n";
        for ( iterAlgo1 = LeafAlgorithm().begin() ; iterAlgo1 != LeafAlgorithm().end() ; iterAlgo1++ ) {
            if ( ALGO_MATRIX_GET(AlgoConnectivity(), (*iterAlgo).second->Identifier(), (*iterAlgo1).second->Identifier()) ) {
                message << "\t" << (*iterAlgo1).second->Name() << "\n";
            }
        }
    }
    std::cerr << message ;
}

// ComputeConnectivity : compute the connectivity matrix
void Scheduler::ComputeConnectivity ()
{
    
    // create the matrix
    unsigned int                    size = pvOperators.size();
    std::vector<OperatorMedia *>    emptyVector ;
    int                             index = size * size ;
    
    while ( index-- ) 
        Connectivity().push_back(emptyVector);
    
    // fill with the direct links
    std::vector<OperatorMedia *> ::iterator iterOperator ;
    
    for ( iterOperator = pvOperators.begin() ; iterOperator != pvOperators.end() ; iterOperator++ ) {
        std::vector<OperatorConnect>            allConnections = (*iterOperator)->AllConnections();
        
        // put in the matrix
        std::vector<OperatorConnect> ::iterator iterMedia ;
        for ( iterMedia = allConnections.begin() ; iterMedia != allConnections.end() ; iterMedia++ ) {
            Connectivity()[MATRIX_ACCESS((*iterOperator)->Index(), (*iterMedia).oper->Index())].push_back((*iterMedia).link);
        }
    }
    
    // for  the empty slot fill them with level 2 connections
    {
        std::vector<std::vector<OperatorMedia *> >  newMatrix ;
        unsigned int                                indexLine ;
        unsigned int                                indexCol ;
        unsigned int                                indexSearch ;
        
        // fill with known values
        newMatrix = Connectivity();
        
        // complete with one more hop
        bool    sthingDone = true ;
        while ( sthingDone ) {
            sthingDone = false ;
            for ( indexLine = 1 ; indexLine <= size ; indexLine++ ) {
                for ( indexCol = 1 ; indexCol <= size ; indexCol++ ) {
                    if ( indexLine != indexCol ) {
                        if ( newMatrix [MATRIX_ACCESS(indexLine, indexCol)].size() == 0 ) {
                            for ( indexSearch = 1 ; indexSearch <= size ; indexSearch++ ) {
                                if ( Connectivity()[MATRIX_ACCESS(indexLine, indexSearch)].size() >= 1
                                        && Connectivity()[MATRIX_ACCESS(indexSearch, indexCol)].size() >= 1 ) {
                                    std::vector<OperatorMedia *> ::iterator iterLink ;
                                    newMatrix [MATRIX_ACCESS(indexLine, indexCol)] = Connectivity()[MATRIX_ACCESS(indexLine, indexSearch)];
                                    for ( iterLink = Connectivity()[MATRIX_ACCESS(indexSearch, indexCol)].begin() ; 
                                            iterLink != Connectivity()[MATRIX_ACCESS(indexSearch, indexCol)].end() ; 
                                            iterLink++ ) 
                                        newMatrix [MATRIX_ACCESS(indexLine, indexCol)].push_back(*iterLink);
                                    sthingDone = true ;
                                    break ;
                                }
                            }
                        }
                    }
                }
            }
            Connectivity() = newMatrix ;
        }
    }
}

// Costs : compute latency and mean time
void Scheduler::Cost ( unsigned int &latency, unsigned int &mean )
{
    
    // search time for ending
    std::vector<OperatorMedia *> ::iterator iterOperator ;
    time_t                                  finish = 0 ;
    
    for ( iterOperator = Operators().begin() ; iterOperator != Operators().end() ; iterOperator++ ) {
        time_t  end ;
        if ( !(*iterOperator)->Planning().empty() ) {
            std::multimap<time_t, unsigned> ::iterator  iterTask = (*iterOperator)->Planning().end();
            iterTask-- ;
            time_t  end = (*iterOperator)->TaskContent((*iterTask).second)->End();
            if ( end > finish ) 
                finish = end ;
        }
    }
    for ( iterOperator = Links().begin() ; iterOperator != Links().end() ; iterOperator++ ) {
        time_t  end ;
        if ( !(*iterOperator)->Planning().empty() ) {
            std::multimap<time_t, unsigned> ::iterator  iterTask = (*iterOperator)->Planning().end();
            iterTask-- ;
            time_t  end = (*iterOperator)->TaskContent((*iterTask).second)->End();
            if ( end > finish ) 
                finish = end ;
        }
    }
    
    // then compute the overlap
    time_t  overlap = finish ;
    
    for ( iterOperator = Operators().begin() ; iterOperator != Operators().end() ; iterOperator++ ) {
        time_t  end ;
        if ( !(*iterOperator)->Planning().empty() ) {
            std::multimap<time_t, unsigned> ::iterator  iterTask = (*iterOperator)->Planning().end();
            iterTask-- ;
            time_t  end = (*iterOperator)->TaskContent((*iterTask).second)->End();
            time_t  start = (*iterOperator)->Start();
            time_t  oper_over = finish - end + start ;
            if ( oper_over < overlap ) 
                overlap = oper_over ;
        }
    }
    for ( iterOperator = Links().begin() ; iterOperator != Links().end() ; iterOperator++ ) {
        time_t  end ;
        if ( !(*iterOperator)->Planning().empty() ) {
            std::multimap<time_t, unsigned> ::iterator  iterTask = (*iterOperator)->Planning().end();
            iterTask-- ;
            time_t  end = (*iterOperator)->TaskContent((*iterTask).second)->End();
            time_t  start = (*iterOperator)->Start();
            time_t  oper_over = finish - end + start ;
            if ( oper_over < overlap ) 
                overlap = oper_over ;
        }
    }
    
    // set return values
    latency = finish ;
    mean = finish - overlap ;
}

// DisplaySchedule : display the schedule
void Scheduler::DisplaySchedule ()
{
    EString                                 message("");
    std::vector<OperatorMedia *> ::iterator iterOperator ;
    std::vector<ConditionType>              nullVector ;
    
    // display planning for processors    
    for ( iterOperator = Operators().begin() ; iterOperator != Operators().end() ; iterOperator++ ) {
        if ( !(*iterOperator)->Planning().empty() ) {
            message << "\n=============================\n";
            message << (*iterOperator)->Name() << "\n";
            message << "=============================\n\n";
            std::multimap<time_t, unsigned> ::iterator  iterTask ;
            (*iterOperator)->Extract((*iterOperator)->Planning().begin(), nullVector);
            for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; iterTask != (*iterOperator)->PlanningExtract().end() ; iterTask++ ) {
                unsigned int    startLength = message.length();
                message << (int)(*iterTask).first << ":" << (int)((*iterOperator)->TaskContent((*iterTask).second)->End());
                while ( message.length() - startLength < 10 ) 
                    message << ' ';
                message << TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name()) << "\n";
            }
        }
    }
    _write(2, message.c_str(), message.length());
    message = "";
    
    // display planning for links
    for ( iterOperator = Links().begin() ; iterOperator != Links().end() ; iterOperator++ ) {
        if ( !(*iterOperator)->Planning().empty() ) {
            message << "\n=============================\n";
            message << (*iterOperator)->Name() << "\n";
            message << "=============================\n\n";
            std::multimap<time_t, unsigned> ::iterator  iterTask ;
            (*iterOperator)->Extract((*iterOperator)->Planning().begin(), nullVector);
            for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; iterTask != (*iterOperator)->PlanningExtract().end() ; iterTask++ ) {
                unsigned int    startLength = message.length();
                message << (int)(*iterTask).first << ":" << (int)((*iterOperator)->TaskContent((*iterTask).second)->End());
                while ( message.length() - startLength < 10 ) 
                    message << ' ';
                message << TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name()) << "\n";
            }
        }
    }
    _write(2, message.c_str(), message.length());
}

// ClearSchedule : clear current schedule
void Scheduler::ClearSchedule ()
{
    std::vector<OperatorMedia *> ::iterator                 iterOperator ;
    std::map<EString, ComputeUnit *, LessString> ::iterator iter ;
    
    for ( iter = Algorithm().begin() ; iter != Algorithm().end() ; iter++ ) {
        (*iter).second->Computed(false);
    }
    
    // clear planning for processors    
    for ( iterOperator = Operators().begin() ; iterOperator != Operators().end() ; iterOperator++ ) {
        (*iterOperator)->Clear();
    }
    
    // clear planning for links
    for ( iterOperator = Links().begin() ; iterOperator != Links().end() ; iterOperator++ ) {
        (*iterOperator)->Clear();
    }
    
    // clear the trans names
    pvTransByNumber.clear();
    
    // clear task names
    ClearNames();
}

// EvalParameter : eval a parameter
PTREE Scheduler::EvalParameter ( SymbolTable &symbTab, PTREE parameter )
{
    PTREE   val1, val2 ;
    
    switch ( parameter ) {
        case <SIZE,val1> : 
        case <PAR,val1> : 
        case <VAL,val1> : 
            {
                return EvalParameter(symbTab, val1);
            }
        case <NEG,val1> : 
            {
                return <NUMB,PTREE(CompactItoa( -1 * atoi(Value(EvalParameter(symbTab, val1)))))>;
            }
        case <ADD,val1,val2> : 
            {
                return <NUMB,PTREE(CompactItoa(atoi(Value(EvalParameter(symbTab, val1))) + atoi(Value(EvalParameter(symbTab, val2)))))>;
            }
        case <SUB,val1,val2> : 
            {
                return <NUMB,PTREE(CompactItoa(atoi(Value(EvalParameter(symbTab, val1))) - atoi(Value(EvalParameter(symbTab, val2)))))>;
            }
        case <MUL,val1,val2> : 
            {
                return <NUMB,PTREE(CompactItoa(atoi(Value(EvalParameter(symbTab, val1))) * atoi(Value(EvalParameter(symbTab, val2)))))>;
            }
        case <DIV,val1,val2> : 
            {
                return <NUMB,PTREE(CompactItoa(atoi(Value(EvalParameter(symbTab, val1))) / atoi(Value(EvalParameter(symbTab, val2)))))>;
            }
        case <NUMB> : 
            {
                return parameter ;
            }
        case <IDENT> : 
            {
                PTREE   val = symbTab [parameter];
                if ( !val ) {
                    EString message ;
                    message << "\nUnknown parameter : " << Value(parameter) << "\n";
                    DisplayError(message);
                } else {
                    val == <CONS,<>,val>;
                }
                return EvalParameter(symbTab, val);
            }
        default : 
            {
                DumpTree(parameter);
                NewLine();
                EString message ;
                message << "\nUnknown operation \n";
                DisplayError(message);
            }
            break ;
    }
    return PTREE(0);
}

// expand references of main algo
PTREE Scheduler::ExpandParameters ( SymbolTable &symbTab, TabList &syndexDescr, PTREE algoName, PTREE parameters )
{
    PTREE   algoDescription, listAlgo ;
    PTREE   elem, list ;
    PTREE   listEvaluatedParam ;
    
    // add this level to symbol table
    symbTab.AddLevel();
    
    // expand all the elements 
    PTREE   algo = syndexDescr [algoName];
    EString name ;
    
    // put a flag to tell that this algo was seen
    algo += <,<>,<>,<>,<CONS,(),()>>;
    algo = CopyTree(algo);
    name = Value(algo);
    if ( !algo ) {
        EString message ;
        message << "Algo not found : " << Value(algoName) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    
    // if really an algo expand it
    PTREE   nameParam ;
    
    if ( algo == <ALGORITHM,nameParam,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <SENSOR,nameParam,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <ACTUATOR,nameParam,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <CONSTANT,nameParam,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <MEMORY,nameParam,<>,algoDescription = <ALGO_ELEM,listAlgo>> ) {
        list = listAlgo ;
        
        // if the algo has parameters do the match between real parameters and symbolic parameters
        PTREE   symbolicParameters ;
        PTREE   nameOnly ;
        if ( nameParam == <NAME_AND_PARAM,nameOnly,symbolicParameters> || nameParam == <MEMORY_IDENT,nameOnly,<>,symbolicParameters> ) {
            PTREE   listParameters ;
            PTREE   listRealParameters ;
            PTREE   aParameter ;
            PTREE   realParameter ;
            symbolicParameters == <SYMBOLIC_PARAMETER,listParameters>;
            if ( parameters == <PARAMETER,listRealParameters> || parameters == <SYMBOLIC_PARAMETER,listRealParameters> ) 
                ;
            while ( aParameter = nextl(listParameters) ) {
                realParameter = nextl(listRealParameters);
                if ( realParameter == () ) {
                    EString message ;
                    message << "Not enough real parameters for : " << Value(algoName) << "\n";
                    DisplayError(message);
                    return PTREE(0);
                }
                
                // affect this parameter in symbol table
                PTREE   added = <CONS,CopyTree(aParameter),CopyTree(EvalParameter(symbTab, realParameter))>;
                symbTab.AddVar(added);
                
                // add to the list of evaluated params
                listEvaluatedParam *= CopyTree(added);
            }
            
            // suppress list of parameters other would replace inside this list and display would not be good (transformed sdx)
            if ( nameParam == <NAME_AND_PARAM> ) 
                nameParam += nameOnly ;
            else if ( nameParam == <MEMORY_IDENT> ) {
                PTREE   null ;
                nameParam.ReplaceTree(3, null);
            }
        }
        
        // replace all val by real value
        foreach (<VAL>,algo,{
            PTREE   elem = for_elem ;
            PTREE   replacement ;
            replacement = CopyTree(symbTab [elem]);
            
            // if no replacement was found it is an error
            if ( replacement == () ) {
                EString message ;
                message << "Real value not found for : " << Value(elem) << " in : " << Value(algoName) << "\n";
                DisplayError(message);
                return PTREE(0);
            }
            
            // do the replacement
            replacement = replacement [2];
            for_elem = elem += replacement ;
        })
        
        // for each condition treat
        list = listAlgo ;
        unsigned int    index = 0 ;
        while ( elem = nextl(list) ) {
            PTREE   condition ;
            PTREE   references ;
            PTREE   dependences ;
            if ( elem == <CONDITIONS,condition,references,dependences> ) {
                PTREE   listReference ;
                PTREE   listDependence ;
                
                // expand all the sub-box of this algorithm
                // checkexpand must be done before to replace variables
                if ( references == <REFERENCES,listReference> ) {
                    PTREE   aReference ;
                    PTREE   topReference ;
                    topReference = listReference ;
                    while ( aReference = nextl(listReference) ) {
                        PTREE       name, parameter ;
                        unsigned    coeff = 1 ;
                        PTREE       typeName ;
                        aReference == <REFERENCE,typeName,parameter,name>;
                        
                        // check this box
                        PTREE   expandedRef = ExpandParameters(symbTab, syndexDescr, typeName, parameter);
                        aReference += <,<>,<>,<>,<>,expandedRef>;
                    }
                }
            }
        }
    } else {
        DumpTree(algo);
        NewLine();
        EString message("What to do ? \n");
        DisplayError(message);
        exit(3);
    }
    
    // remove the added level to the symbol table
    symbTab.RemoveLevel();
    
    // put the list of evaluated parameters in algo
    algo += <,<>,<>,<>,<CONS,listEvaluatedParam,()>>;
    
    // return the algo
    return algo ;
}

// AttachSoftwareEntry : attach a software entry to a list of processors
void Scheduler::AttachSoftwareEntry ( PTREE mainAlgo, PTREE entry, PTREE listProc )
{
    std::vector<PTREE> ::iterator   iterCondition ;
    
    // search the start of attachment
    if ( entry ) {
        if ( VString(Value(mainAlgo)) = Value(entry) ) {
            nextl(entry);
            if ( entry ) {
                {
                    std::vector<PTREE>  vectConditions = ConditionsForAlgo(mainAlgo);
                    
                    // expand them
                    for ( iterCondition = vectConditions.begin() ; iterCondition != vectConditions.end() ; iterCondition++ ) {
                        PTREE   elem = *iterCondition ;
                        PTREE   references ;
                        PTREE   dependences ;
                        PTREE   condition ;
                        if ( elem == <CONDITIONS,condition,references,dependences> ) {
                            PTREE   listReference ;
                            PTREE   listDependence ;
                            
                            // expand all the sub-box of this algorithm
                            if ( references == <REFERENCES,listReference> ) {
                                PTREE   algo = AlgoForReference(listReference, entry);
                                if ( algo ) 
                                    AttachSoftwareEntry(algo, entry, listProc);
                            }
                        }
                    }
                }
                return ;
            }
        } else 
            return ;
    }
    
    // entry is null so we have reached the start of attachement 
    // put attachment on all sub algo
    {
        std::vector<PTREE>  vectConditions = ConditionsForAlgo(mainAlgo);
        
        // expand them
        for ( iterCondition = vectConditions.begin() ; iterCondition != vectConditions.end() ; iterCondition++ ) {
            PTREE   elem = *iterCondition ;
            PTREE   references ;
            PTREE   dependences ;
            PTREE   condition ;
            if ( elem == <CONDITIONS,condition,references,dependences> ) {
                PTREE   listReference ;
                PTREE   listDependence ;
                
                // expand all the sub-box of this algorithm
                if ( references == <REFERENCES,listReference> ) {
                    while ( elem = nextl(listReference) ) 
                        AttachSoftwareEntry(elem [5], entry, listProc);
                }
            }
        }
    }
    
    // put attachment on algo
    PTREE   proc ;
    PTREE   finalList ;
    
    while ( proc = nextl(listProc) ) {
        proc == <PROC,<>,proc>;
        finalList *= CopyTree(proc);
    }
    mainAlgo.ReplaceTree(5, finalList);
}

// DispatchConstraint : dispatch the software constraint on each algorithm
void Scheduler::DispatchConstraint ( TabList &syndexDescript, PTREE description, PTREE mainAlgo )
{
    PTREE   descript ;
    PTREE   listConstraint ;
    PTREE   elem ;
    PTREE   constraintName ;
    PTREE   topDescript = description ;
    
    for (; description ; nextl(description) ) {
        elem = description [1];
        if ( elem == <CONSTRAINTS,listConstraint> ) {
            PTREE   constraint ;
            while ( constraint = nextl(listConstraint) ) {
                PTREE   consraintName ;
                PTREE   listProc ;
                if ( constraint == <CONSTRAINT,constraintName,listProc> ) {
                    
                    // get the list of softare for this constraint
                    PTREE   softwareList, listEntries, softwareEntry ;
                    foreach (<SOFT_COMPONENT>,topDescript,{
                        if ( VString(Value(constraint)) == Value(for_elem) ) {
                            softwareList = for_elem ;
                            goto for_break ;
                        }
                    })
                    
                    // we have to find the software list
                    if ( softwareList == <SOFT_COMPONENT,<>,<SOFT_ENTRIES,listEntries>> ) {
                        while ( softwareEntry = nextl(listEntries) ) {
                            AttachSoftwareEntry(mainAlgo, softwareEntry [1], listProc);
                        }
                    } else {
                        EString message ;
                        message << "Software Description incorrect : " << Value(constraint) << "\n";
                        DisplayError(message);
                    }
                }
            }
        }
    }
}

// CheckExpand : check that data are correctly connected and 
//               if width of channels are not the same adapt
void Scheduler::CheckExpand ( TabList &syndexDescr, PTREE algo )
{
    {
        
        // get list of conditions
        PTREE               elem ;
        std::vector<PTREE>  listConditions ;
        listConditions = ConditionsForAlgo(algo);
        std::vector<PTREE> ::iterator   iterCondition ;
        
        // expand them
        for ( iterCondition = listConditions.begin() ; iterCondition != listConditions.end() ; iterCondition++ ) {
            elem = *iterCondition ;
            PTREE   references ;
            PTREE   dependences ;
            PTREE   condition ;
            if ( elem == <CONDITIONS,condition,references,dependences> ) {
                PTREE   listReference ;
                PTREE   listDependence ;
                
                // expand all the sub-box of this algorithm
                if ( references == <REFERENCES,listReference> ) {
                    
                    // expand all the sub-box of this algorithm
                    {
                        PTREE   aReference ;
                        while ( aReference = nextl(listReference) ) {
                            unsigned    coeff = 1 ;
                            PTREE       content ;
                            aReference == <REFERENCE,<>,<>,<>,<>,content>;
                            
                            // check this box
                            CheckExpand(syndexDescr, content);
                        }
                    }
                    
                    // do the expansion
                    {
                        listReference = references [1];
                        PTREE   aReference ;
                        while ( aReference = nextl(listReference) ) {
                            PTREE           name, parameter ;
                            unsigned        coeff = 1 ;
                            unsigned int    currMult = 1 ;
                            PTREE           typeName ;
                            PTREE           hasOneOutput ; // remember if there are output with 1 coeff
                            aReference == <REFERENCE,typeName,parameter,name>;
                            PTREE               topReference = references [1];  // compute it at each loop otherwise topReference could point on part of list
                            
                            // get the list of in out port
                            std::set<EString>   inOut ;
                            inOut = GetInOut(syndexDescr, topReference, name);
                            
                            // check every dependance for this reference
                            if ( dependences == <DEPENDENCES,listDependence> ) {
                                PTREE   aDependence ;
                                PTREE   currentEnd, otherEnd ;
                                while ( aDependence = nextl(listDependence) ) {
                                    
                                    // get associated port if it corresponds to name
                                    bool    out ;
                                    PTREE   portReferences = GetAssociatedPort(syndexDescr, algo, topReference, name, aDependence, currentEnd, otherEnd, out);
                                    
                                    // try to compute a coeff
                                    PTREE   portDescrCurrent ;
                                    PTREE   portDescrOther ;
                                    if ( portReferences == <CONS,portDescrCurrent,portDescrOther> ) {
                                        
                                        // check if types are correct
                                        if ( VString(Value(portDescrCurrent)) != Value(portDescrOther) ) {
                                            EString message ;
                                            message << "Type of data is not the same for : " << Value(currentEnd) << " " << Value(currentEnd [2]) << " : "
                                                << Value(portDescrCurrent) << " / " << Value(otherEnd) << " " << Value(otherEnd [2]) << " : " << Value(portDescrOther)
                                                << "\n";
                                            DisplayError(message);
                                            continue ;
                                        }
                                        
                                        // check multipliers
                                        SymbolTable symbTab ;
                                        PTREE       res ;
                                        currMult = atoi(Value(res = EvalParameter(symbTab, ((PTREE)(portDescrCurrent [1]))[2])));
                                        unsigned int    otherMult = atoi(Value(res = EvalParameter(symbTab, ((PTREE)portDescrOther [1])[2])));
                                        if ( currMult == 0 ) 
                                            currMult = 1 ;
                                        if ( otherMult == 0 ) 
                                            otherMult = 1 ;
                                        unsigned    remain = otherMult % currMult ;
                                        unsigned    div = otherMult / currMult ;
                                        if ( remain && otherMult > currMult ) {
                                            EString message ;
                                            message << "Number of data are not multiple  for : " << Value(currentEnd) << " " << Value(currentEnd [2]) << " : "
                                                << Value(portDescrCurrent) << " / " << Value(otherEnd) << " " << Value(otherEnd [2]) << " : " << Value(portDescrOther)
                                                << "\n";
                                            DisplayError(message);
                                            continue ;
                                        }
                                        
                                        // check that only one value for multiplier
                                        if ( div > 1 ) {
                                            if ( coeff != 1 && coeff != div ) {
                                                EString message ;
                                                message << "Not fixed multiple for : " << Value(currentEnd) << " " << Value(currentEnd [2]) << " : "
                                                    << Value(portDescrCurrent) << " / " << Value(otherEnd) << " " << Value(otherEnd [2]) << " : "
                                                    << Value(portDescrOther) << " : " << (int)coeff << " " << (int)div << "\n";
                                                DisplayError(message);
                                                continue ;
                                            } else {
                                                coeff = div ;
                                            }
                                        } else if ( div == 1 ) {
                                            if ( portDescrCurrent == <OUT> ) {
                                                
                                                // if reference is not an in/out we will not be able to repeat block
                                                std::set<EString> ::iterator    iterInOut ;
                                                iterInOut = inOut.find(VString(Value(((PTREE)(portDescrCurrent [1]))[2])));
                                                if ( iterInOut != inOut.end() ) 
                                                    hasOneOutput = portDescrCurrent ;
                                            }
                                        }
                                    }
                                }
                                
                                // if coeff is not 1 we have to extend this reference
                                if ( coeff != 1 ) {
                                    
                                    // if has output with cardinality of one we will not be able to expand this box
                                    if ( hasOneOutput ) {
                                        EString message ;
                                        DumpTree(hasOneOutput);
                                        NewLine();
                                        DumpTree(aReference);
                                        NewLine();
                                        message << "Output of one for box with iteration   \n";
                                        DisplayError(message);
                                        continue ;
                                    }
                                    
                                    // expand
                                    ExpandReference(syndexDescr, coeff, currMult, algo, topReference, aReference, dependences, inOut);
                                }
                            }
                        }
                    }
                }
            }
        }
    }
}

// ExpandReference : expand a reference
void Scheduler::ExpandReference ( TabList &syndexDescr, unsigned int coeff, unsigned int currMult, PTREE algo, PTREE topReference, PTREE reference, PTREE dependences
    , std::set<EString> &inOut )
{
    if ( coeff == 1 ) 
        return ;
    
    // mark the start
    PTREE                           father = reference ^ ;
    PTREE                           next ;
    PTREE                           referenceName = reference [3];
    unsigned int                    indexReplace ;
    
    // and reconnect like the original one
    // check every dependance for this reference
    PTREE                           newList ;
    PTREE                           listDependence ;
    std::set<EString> ::iterator    iterInOut ;
    std::set<EString>               treatedInOut ;
    bool                            out = false ;
    
    if ( dependences == <DEPENDENCES,listDependence> ) {
        PTREE   aDependence ;
        PTREE   currentEnd, otherEnd ;
        PTREE   name ;
        while ( aDependence = nextl(listDependence) ) {
            aDependence = CopyTree(aDependence);
            
            // get associated ports if reference name otherwise returned value is not a CONS
            PTREE   references = GetAssociatedPort(syndexDescr, algo, topReference, referenceName, aDependence, currentEnd, otherEnd, out);
            
            // try to compute a coeff
            PTREE   portDescrCurrent ;
            PTREE   portDescrOther ;
            PTREE   boxTypeName ;
            PTREE   boxElemName ;
            bool    isInOut = false ;
            
            // get name of port and look if in/out
            PTREE   namePort = currentEnd ;
            {
                
                // get name of  port           
                if ( portDescrCurrent == <IN,<TYPE,<>,<>,namePort>> || portDescrCurrent == <OUT,<TYPE,<>,<>,namePort>> ) 
                    ;
                if ( ListLength(namePort) > 1 ) 
                    nextl(namePort);
                
                // look if in/out
                iterInOut = inOut.find(VString(Value(namePort)));
                isInOut = iterInOut != inOut.end();
            }
            
            // if it is a dependence insert replacement in new list otherwise keep list as it 
            unsigned int    indexType = pvIndexType++ ;
            if ( references == <CONS,portDescrCurrent,portDescrOther> ) {
                
                // check multipliers
                SymbolTable     symbTab ;
                PTREE           res ;
                unsigned int    currMult = atoi(Value(res = EvalParameter(symbTab, ((PTREE)(portDescrCurrent [1]))[2])));
                unsigned int    otherMult = atoi(Value(res = EvalParameter(symbTab, ((PTREE)(portDescrOther [1]))[2])));
                if ( currMult == 0 ) 
                    currMult = 1 ;
                if ( otherMult == 0 ) 
                    otherMult = 1 ;
                unsigned    div = otherMult / currMult ;
                
                // if div greater than 1 and it is an in/out it is an error since cardinality of in/out do not change
                if ( div > 1 && iterInOut != inOut.end() ) {
                    DumpTree(aDependence);
                    <NL>
                    DumpTree(portDescrCurrent);
                    <NL>
                    DumpTree(portDescrOther);
                    <NL>
                    EString message ;
                    message << "Cannot associate ports since current port is in/out in : " << Value(referenceName) << "\n";
                    MetaExit(3, message.c_str());
                }
                if ( div > 1 ) {
                    
                    // compute names
                    switch ( portDescrCurrent ) {
                        case <IN> : 
                            {
                                boxTypeName = <IDENT,PTREE(EString("Explode_") + CompactItoa(otherMult) + "_" + CompactItoa(currMult) + "_" + Value(portDescrCurrent))>;
                                boxElemName
                                    = <IDENT
                                        ,PTREE(
                                            EString("explode_") + CompactItoa(otherMult) + "_" + CompactItoa(currMult) + "_" + Value(portDescrCurrent) + "_"
                                                + CompactItoa(indexType))>;
                            }
                            break ;
                        case <OUT> : 
                            {
                                boxTypeName = <IDENT,PTREE(EString("Implode_") + CompactItoa(currMult) + "_" + CompactItoa(otherMult) + "_" + Value(portDescrCurrent))>;
                                boxElemName
                                    = <IDENT
                                        ,PTREE(
                                            EString("implode_") + CompactItoa(currMult) + "_" + CompactItoa(otherMult) + "_" + Value(portDescrCurrent) + "_"
                                                + CompactItoa(indexType))>;
                            }
                            break ;
                    }
                } else if ( 0 ) {
                    
                    // initialize names
                    boxTypeName = <IDENT,PTREE(EString("Algo_") + Value(algo) + "_" + Value(namePort))>;
                    boxElemName = <IDENT,PTREE(EString("algo_") + Value(referenceName) + "_" + Value(namePort))>;
                }
                
                // create the necessary boxes
                if ( div > 1 ) {
                    
                    // create the implode/explode box
                    PTREE   box ;
                    {
                        
                        // insert this new reference only if it does not exist
                        {
                            PTREE   listBox ;
                            PTREE   box ;
                            bool    foundBox = false ;
                            PTREE   privDescr = pvRefDescription ;
                            privDescr == <SYNDEX_TOP,<>,<>,listBox>;
                            while ( box = nextl(listBox) ) {
                                if ( VString(Value(box)) == Value(boxTypeName) ) {
                                    foundBox = true ;
                                    break ;
                                }
                            }
                            if ( !foundBox ) {
                                PTREE   listPort ;
                                PTREE   connector ;
                                switch ( portDescrCurrent ) {
                                    case <OUT> : 
                                        {
                                            for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
                                                PTREE   newName ;
                                                PTREE   connector ;
                                                newName = <IDENT,PTREE(EString("in") + CompactItoa(indexReplace))>;
                                                connector
                                                    = <IN
                                                        ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(currMult))>>,newName
                                                            ,<NUMB,PTREE(CompactItoa(indexReplace))>>
                                                        ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                                listPort *= connector ;
                                            }
                                            connector
                                                = <OUT
                                                    ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(coeff * currMult))>>,<IDENT,"out">
                                                        ,<NUMB,"1">>
                                                    ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                            listPort *= connector ;
                                        }
                                        break ;
                                    case <IN> : 
                                        {
                                            for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
                                                PTREE   newName ;
                                                PTREE   connector ;
                                                newName = <IDENT,PTREE(EString("out") + CompactItoa(indexReplace))>;
                                                connector
                                                    = <OUT
                                                        ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(currMult))>>,newName
                                                            ,<NUMB,PTREE(CompactItoa(indexReplace))>>
                                                        ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                                listPort *= connector ;
                                            }
                                            connector
                                                = <IN
                                                    ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(coeff * currMult))>>,<IDENT,"in">
                                                        ,<NUMB,"1">>
                                                    ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                            listPort *= connector ;
                                        }
                                        break ;
                                }
                                
                                // finalize box 
                                listPort *= <CODE_PHASE,<LOOPSEQ>>;
                                box = <ALGORITHM,<IDENT,CopyTree(boxTypeName)>,<POS1,<NUMB,"0">,<NUMB,"0">>,<ALGO_ELEM,listPort>,<CONS,(),()>,()>;
                                
                                // insert in working list
                                syndexDescr.InsertAt(box, syndexDescr.Size());
                                syndexDescr.Sort();
                                
                                // insert in ref list
                                {
                                    PTREE   privDescr = pvRefDescription ;
                                    PTREE   list ;
                                    PTREE   newElem ;
                                    if ( privDescr == <SYNDEX_TOP,<>,<>,list> ) {
                                        list = <LIST,CopyTree(box),list>;
                                        privDescr += <,<>,<>,list>;
                                    }
                                }
                                
                                // insert a default 0 time for this box
                                {
                                    AddBoxTime(Description(), boxTypeName);
                                    AddBoxTime(pvRefDescription, boxTypeName);
                                }
                            }
                        }
                    }
                } else if ( isInOut && 0 ) {
                    
                    // create the transparent algo box
                    PTREE   box ;
                    {
                        
                        // insert this new reference only if it does not exist
                        {
                            PTREE   listBox ;
                            PTREE   box ;
                            bool    foundBox = false ;
                            PTREE   privDescr = pvRefDescription ;
                            privDescr == <SYNDEX_TOP,<>,<>,listBox>;
                            while ( box = nextl(listBox) ) {
                                if ( VString(Value(box)) == Value(boxTypeName) ) {
                                    foundBox = true ;
                                    break ;
                                }
                            }
                            if ( !foundBox ) {
                                
                                // get characteristics
                                PTREE   type ;
                                PTREE   mult ;
                                PTREE   port, listPort ;
                                if ( portDescrCurrent == <IN,<TYPE,type,mult,name>> || portDescrCurrent == <OUT,<TYPE,type,mult,name>> ) 
                                    ;
                                port = <IN,<TYPE,CopyTree(type),CopyTree(mult),<IDENT,"in">,<NUMB,"1">>,()>;
                                listPort *= port ;
                                port = <OUT,<TYPE,CopyTree(type),CopyTree(mult),<IDENT,"out">,<NUMB,"1">>,()>;
                                listPort *= port ;
                                box = <ALGORITHM,<IDENT,CopyTree(boxTypeName)>,<POS1,<NUMB,"0">,<NUMB,"0">>,<ALGO_ELEM,listPort>,<CONS,(),()>,()>;
                                
                                // insert in working list
                                syndexDescr.InsertAt(box, syndexDescr.Size());
                                syndexDescr.Sort();
                                
                                // insert in ref list
                                {
                                    PTREE   privDescr = pvRefDescription ;
                                    PTREE   list ;
                                    PTREE   newElem ;
                                    if ( privDescr == <SYNDEX_TOP,<>,<>,list> ) {
                                        list = <LIST,CopyTree(box),list>;
                                        privDescr += <,<>,<>,list>;
                                    }
                                }
                                
                                // insert a default 0 time for this box
                                {
                                    AddBoxTime(Description(), boxTypeName);
                                    AddBoxTime(pvRefDescription, boxTypeName);
                                }
                            }
                        }
                    }
                }
                
                // if div > 1 create implode / explode type && instance of it 
                if ( div > 1 ) {
                    PTREE   newReference ;
                    PTREE   algoReference ;
                    PTREE   typeReference ;
                    next = father ;
                    nextl(next);
                    algoReference = AlgoForReference(topReference, referenceName);
                    newReference = <LIST,<REFERENCE,boxTypeName,(),boxElemName,(),typeReference = CopyTree(syndexDescr [boxTypeName])>,next>;
                    typeReference.ReplaceTree(5, CopyTree(algoReference [5]));
                    father += <,<>,newReference>;
                } else {
                    next = father ;
                    nextl(next);
                    PTREE   algo = GetAlgoDescr(syndexDescr, algo, topReference, currentEnd);
                    if ( isInOut ) {
                        
                        // for in out do both sides at once 
                        if ( treatedInOut.find(VString(Value(namePort))) != treatedInOut.end() ) {
                            
                            // this entry is already modified we have only to correct dependence
                            if ( !out ) {
                                PTREE   newName ;
                                PTREE   port ;
                                currentEnd = CopyTree(currentEnd);
                                newName = <IDENT,PTREE(EString(Value(referenceName)) + CompactItoa(1))>;
                                currentEnd += <,newName>;
                                port = currentEnd ;
                                nextl(port);
                                port += <,<IDENT,PTREE(EString(Value(port)) /* + "_in"*/ )>>;
                                aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(otherEnd)>,<END_DEPEND,CopyTree(currentEnd)>>;
                                newList *= CopyTree(aDependence);
                            } else {
                                PTREE   newName ;
                                currentEnd = CopyTree(currentEnd);
                                newName = <IDENT,PTREE(EString(Value(referenceName)) + CompactItoa(coeff))>;
                                currentEnd += <,newName>;
                                {
                                    PTREE   port = currentEnd ;
                                    if ( ListLength(port) > 1 ) 
                                        nextl(port);
                                    port += <,<IDENT,PTREE(EString(Value(port)) /* + "_out"*/ )>>;
                                }
                                aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(currentEnd)>,<END_DEPEND,CopyTree(otherEnd)>>;
                                newList *= aDependence ;
                            }
                            continue ;
                        }
                        treatedInOut.insert(VString(Value(namePort)));
                        
                        // creates ref
                        if ( 0 ) {
                            for ( indexReplace = 1 ; indexReplace < coeff ; indexReplace++ ) {
                                PTREE   algoReference ;
                                PTREE   typeReference ;
                                PTREE   newReference
                                    = <LIST
                                        ,<REFERENCE,CopyTree(boxTypeName),(),<IDENT,PTREE(EString(Value(boxElemName)) + CompactItoa(indexReplace))>,()
                                            ,typeReference = CopyTree(syndexDescr [boxTypeName])>
                                        ,next>;
                                
                                // put good processor
                                algoReference = AlgoForReference(topReference, referenceName);
                                typeReference.ReplaceTree(5, CopyTree(algoReference [5]));
                                next = newReference ;
                            }
                            
                            // insert this reference
                            father += <,<>,next>;
                        }
                    }
                }
                
                // do the connections
                PTREE   otherEndOrg = otherEnd ;
                for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
                    PTREE   newName ;
                    newName = <IDENT,PTREE(EString(Value(referenceName)) + CompactItoa(indexReplace))>;
                    currentEnd += <,newName>;
                    if ( div > 1 ) {
                        
                        // connect each box to Implode or Explode depending on the fact that it is an out or an in
                        switch ( portDescrCurrent ) {
                            case <IN> : 
                                {
                                    aDependence
                                        = <DEPENDENCE,(),<END_DEPEND,<LIST,CopyTree(boxElemName),<LIST,<IDENT,PTREE(EString("out") + CompactItoa(indexReplace))>,()>>>
                                            ,<END_DEPEND,CopyTree(currentEnd)>>;
                                    newList *= aDependence ;
                                }
                                break ;
                            case <OUT> : 
                                {
                                    aDependence
                                        = <DEPENDENCE,<>,<END_DEPEND,CopyTree(currentEnd)>
                                            ,<END_DEPEND,<LIST,CopyTree(boxElemName),<LIST,<IDENT,PTREE(EString("in") + CompactItoa(indexReplace))>,()>>>>;
                                    newList *= aDependence ;
                                }
                                break ;
                        }
                    } else {
                        
                        // if it is an out an we have an inverted multiplier that is n in current box and 1 in other box it is impossible to solve
                        // design should factorize boxes
                        if ( currMult != otherMult && portDescrCurrent == <OUT> ) {
                            EString message = "Inverted multiplier \n";
                            DisplayError(message);
                        }
                        
                        // if it is an in/out connect
                        if ( isInOut ) {
                            
                            // connect each box 
                            {
#                               if 1
                                    
                                    // add in for in name in in_out
                                    PTREE   currentEndModified = CopyTree(currentEnd);
                                    {
                                        PTREE   port = currentEndModified ;
                                        if ( ListLength(port) > 1 ) 
                                            nextl(port);
                                        port += <,<IDENT,PTREE(EString(Value(port)) /*+ "_in"*/ )>>;
                                    }
                                    
                                    // do the new connection
                                    if ( indexReplace == 1 && !out ) {
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(otherEndOrg)>,<END_DEPEND,CopyTree(currentEndModified)>>;
                                        newList *= CopyTree(aDependence);
                                    }
                                    if ( indexReplace == coeff && out ) {
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(currentEndModified)>,<END_DEPEND,CopyTree(otherEndOrg)>>;
                                        newList *= aDependence ;
                                    }
                                    
                                    // connect to next
                                    if ( indexReplace != 1 ) {
                                        PTREE   otherName = <IDENT,PTREE(EString(Value(referenceName)) + CompactItoa(indexReplace - 1))>;
                                        otherEnd = CopyTree(currentEndModified);
                                        otherEnd += <,otherName>;
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(otherEnd)>,<END_DEPEND,CopyTree(currentEndModified)>>;
                                        newList *= aDependence ;
                                    }
#                               else 
                                    
                                    // add in for in name in in_out
                                    PTREE   currentEndModified = CopyTree(currentEnd);
                                    {
                                        PTREE   port = currentEndModified ;
                                        if ( ListLength(port) > 1 ) 
                                            nextl(port);
                                        port += <,<IDENT,PTREE(EString(Value(port)) /*+ "_in"*/ )>>;
                                    }
                                    
                                    // do the new connection
                                    if ( indexReplace != 1 ) {
                                        PTREE   otherName = <IDENT,PTREE(EString(Value(boxElemName)) + CompactItoa(indexReplace - 1))>;
                                        otherEnd = <LIST,otherName,<LIST,<IDENT,"out">,()>>;
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(otherEnd)>,<END_DEPEND,CopyTree(currentEndModified)>>;
                                        newList *= aDependence ;
                                    } else if ( !out ) {
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(otherEndOrg)>,<END_DEPEND,CopyTree(currentEndModified)>>;
                                        newList *= CopyTree(aDependence);
                                    }
                                    
                                    // add out for out name in in_out
                                    currentEndModified = CopyTree(currentEnd);
                                    {
                                        PTREE   port = currentEndModified ;
                                        if ( ListLength(port) > 1 ) 
                                            nextl(port);
                                        port += <,<IDENT,PTREE(EString(Value(port)) /*+ "_out"*/ )>>;
                                    }
                                    
                                    // do the new connection
                                    if ( indexReplace != coeff ) {
                                        PTREE   otherName = <IDENT,PTREE(EString(Value(boxElemName)) + CompactItoa(indexReplace))>;
                                        otherEnd = <LIST,otherName,<LIST,<IDENT,"in">,()>>;
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(currentEndModified)>,<END_DEPEND,CopyTree(otherEnd)>>;
                                        newList *= aDependence ;
                                    } else if ( out ) {
                                        aDependence = <DEPENDENCE,(),<END_DEPEND,CopyTree(currentEndModified)>,<END_DEPEND,CopyTree(otherEndOrg)>>;
                                        newList *= aDependence ;
                                    }
#                               endif
                            }
                        } else {
                            newList *= CopyTree(aDependence);
                        }
                    }
                }
                if ( div > 1 ) {
                    
                    // connect the connection box to the other port 
                    // connect each box to Implode or Explode depending on the fact that it is an out or an in
                    switch ( portDescrCurrent ) {
                        case <OUT> : 
                            {
                                aDependence
                                    = <DEPENDENCE,(),<END_DEPEND,<LIST,CopyTree(boxElemName),<LIST,<IDENT,PTREE(EString("out"))>,()>>>,<END_DEPEND,CopyTree(otherEnd)>>;
                                newList *= aDependence ;
                            }
                            break ;
                        case <IN> : 
                            {
                                aDependence
                                    = <DEPENDENCE,<>,<END_DEPEND,CopyTree(otherEnd)>,<END_DEPEND,<LIST,CopyTree(boxElemName),<LIST,<IDENT,PTREE(EString("in"))>,()>>>>;
                                newList *= aDependence ;
                            }
                            break ;
                    }
                    
                    // create the implode/explode box
                    PTREE   box ;
                    {
                        
                        // insert this new reference only if it does not exist
                        {
                            PTREE   listBox ;
                            PTREE   box ;
                            bool    foundBox = false ;
                            PTREE   privDescr = pvRefDescription ;
                            privDescr == <SYNDEX_TOP,<>,<>,listBox>;
                            while ( box = nextl(listBox) ) {
                                if ( VString(Value(box)) == Value(boxTypeName) ) {
                                    foundBox = true ;
                                    break ;
                                }
                            }
                            if ( !foundBox ) {
                                PTREE   listPort ;
                                PTREE   connector ;
                                switch ( portDescrCurrent ) {
                                    case <OUT> : 
                                        {
                                            for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
                                                PTREE   newName ;
                                                PTREE   connector ;
                                                newName = <IDENT,PTREE(EString("in") + CompactItoa(indexReplace))>;
                                                connector
                                                    = <IN
                                                        ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(currMult))>>,newName
                                                            ,<NUMB,PTREE(CompactItoa(indexReplace))>>
                                                        ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                                listPort *= connector ;
                                            }
                                            connector
                                                = <OUT
                                                    ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(coeff * currMult))>>,<IDENT,"out">
                                                        ,<NUMB,"1">>
                                                    ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                            listPort *= connector ;
                                        }
                                        break ;
                                    case <IN> : 
                                        {
                                            for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
                                                PTREE   newName ;
                                                PTREE   connector ;
                                                newName = <IDENT,PTREE(EString("out") + CompactItoa(indexReplace))>;
                                                connector
                                                    = <OUT
                                                        ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(currMult))>>,newName
                                                            ,<NUMB,PTREE(CompactItoa(indexReplace))>>
                                                        ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                                listPort *= connector ;
                                            }
                                            connector
                                                = <IN
                                                    ,<TYPE,<IDENT,PTREE(Value(portDescrCurrent))>,<SIZE,<NUMB,PTREE(CompactItoa(coeff * currMult))>>,<IDENT,"in">
                                                        ,<NUMB,"1">>
                                                    ,<POS2,<NUMB,PTREE("0")>,<NUMB,PTREE("0")>>>;
                                            listPort *= connector ;
                                        }
                                        break ;
                                }
                                
                                // finalize box 
                                listPort *= <CODE_PHASE,<LOOPSEQ>>;
                                box = <ALGORITHM,<IDENT,CopyTree(boxTypeName)>,<POS1,<NUMB,"0">,<NUMB,"0">>,<ALGO_ELEM,listPort>,<CONS,(),()>>;
                                
                                // insert in working list
                                syndexDescr.InsertAt(box, syndexDescr.Size());
                                
                                // insert in ref list
                                {
                                    PTREE   privDescr = pvRefDescription ;
                                    PTREE   list ;
                                    PTREE   newElem ;
                                    if ( privDescr == <SYNDEX_TOP,<>,<>,list> ) {
                                        list = <LIST,CopyTree(box),list>;
                                        privDescr += <,<>,<>,list>;
                                    }
                                }
                                
                                // insert a default 0 time for this box
                                {
                                    AddBoxTime(Description(), boxTypeName);
                                    AddBoxTime(pvRefDescription, boxTypeName);
                                }
                            }
                        }
                    }
                } else if ( isInOut && 0 ) {
                    
                    // create the transparent algo box
                    PTREE   box ;
                    {
                        
                        // insert this new reference only if it does not exist
                        {
                            PTREE   listBox ;
                            PTREE   box ;
                            bool    foundBox = false ;
                            PTREE   privDescr = pvRefDescription ;
                            privDescr == <SYNDEX_TOP,<>,<>,listBox>;
                            while ( box = nextl(listBox) ) {
                                if ( VString(Value(box)) == Value(boxTypeName) ) {
                                    foundBox = true ;
                                    break ;
                                }
                            }
                            if ( !foundBox ) {
                                
                                // get characteristics
                                PTREE   type ;
                                PTREE   mult ;
                                PTREE   port, listPort ;
                                if ( portDescrCurrent == <IN,<TYPE,type,mult,name>> || portDescrCurrent == <OUT,<TYPE,type,mult,name>> ) 
                                    ;
                                port = <IN,<TYPE,CopyTree(type),CopyTree(mult),<IDENT,"in">,<NUMB,"1">>,()>;
                                listPort *= port ;
                                port = <OUT,<TYPE,CopyTree(type),CopyTree(mult),<IDENT,"out">,<NUMB,"1">>,()>;
                                listPort *= port ;
                                box = <ALGORITHM,<IDENT,CopyTree(boxTypeName)>,<POS1,<NUMB,"0">,<NUMB,"0">>,<ALGO_ELEM,listPort>,<CONS,(),()>>;
                                
                                // insert in working list
                                syndexDescr.InsertAt(box, syndexDescr.Size());
                                
                                // insert in ref list
                                {
                                    PTREE   privDescr = pvRefDescription ;
                                    PTREE   list ;
                                    PTREE   newElem ;
                                    if ( privDescr == <SYNDEX_TOP,<>,<>,list> ) {
                                        list = <LIST,CopyTree(box),list>;
                                        privDescr += <,<>,<>,list>;
                                    }
                                }
                                
                                // insert a default 0 time for this box
                                {
                                    AddBoxTime(Description(), boxTypeName);
                                    AddBoxTime(pvRefDescription, boxTypeName);
                                }
                            }
                        }
                    }
                }
            } else 
                newList *= aDependence ;
        }
    }
    dependences += <,newList>;
    
    // put references
    father = reference ^ ;
    next = father ;
    nextl(next);
    
    // create the coeff new box which replace the old one
    PTREE   list ;
    PTREE   newElem ;
    
    for ( indexReplace = 1 ; indexReplace <= coeff ; indexReplace++ ) {
        newElem = CopyTree(reference);
        newElem += <,<>,<>,<IDENT,PTREE(EString(Value(referenceName)) + CompactItoa(indexReplace))>>;
        list = <LIST,newElem,list>;
    }
    
    // put back in list
    {
        PTREE   marker = <CONS,(),()>;
        list = <LIST,marker,list>;
        list = ListPermutate(list);
        PTREE   tail = marker ^ ;
        tail += next ;
        father = father += list ;
    }
}

// ReferenceForAlgo : get references of an algo
std::vector<PTREE> Scheduler::ConditionsForAlgo ( PTREE &algo )
{
    PTREE               listAlgo ;
    PTREE               elem ;
    std::vector<PTREE>  listList ;
    
    if ( algo == <ALGORITHM,<>,<>,<ALGO_ELEM,listAlgo>>
            || algo == <SENSOR,<>,<>,<ALGO_ELEM,listAlgo>>
            || algo == <ACTUATOR,<>,<>,<ALGO_ELEM,listAlgo>>
            || algo == <CONSTANT,<>,<>,<ALGO_ELEM,listAlgo>>
            || algo == <MEMORY,<>,<>,<ALGO_ELEM,listAlgo>> ) {
        
        // for each condition treat
        while ( elem = nextl(listAlgo) ) {
            if ( elem == <CONDITIONS> ) {
                listList.push_back(elem);
            }
        }
    }
    return listList ;
}

// AlgoForReference : return the algo master for a reference
PTREE Scheduler::AlgoForReference ( PTREE listReferenceType, PTREE reference )
{
    PTREE   typeReference ;
    PTREE   list = listReferenceType ;
    PTREE   aReference ;
    PTREE   algo ;
    
    // search the type of box 
    while ( aReference = nextl(list) ) {
        if ( VString(Value(aReference [3])) == Value(reference) ) {
            typeReference = aReference ;
            break ;
        }
    }
    if ( !typeReference ) {
        PTREE   algo = listReferenceType ;
        while ( algo && algo != <ALGORITHM> && algo != <SENSOR> && algo != <ACTUATOR> && algo != <CONSTANT> && algo != <MEMORY> ) {
            algo = algo ^ ;
        }
        EString message ;
        message << "Type of reference not found : " << Value(reference) << " In " << Value(algo) << "\n";
        DisplayError(message);
        return PTREE(0);
    }
    
    // return description of box
    return aReference [5];
}

// GetAlgoDescr : get an algo description from a dependance description algoReference.port
// parameters :
//              referencerAlgo : algo looking for port description
//              listReferenceType : the list of boxes with their type in referencerAlgo
//              reference : contents is start end of reference
PTREE Scheduler::GetAlgoDescr ( TabList &syndexDescr, PTREE referencerAlgo, PTREE listReferenceType, PTREE reference )
{
    PTREE   algo ;
    
    if ( reference != <LIST> ) 
        return algo ;
    
    // if length of reference is 1 we have to look for current ports
    if ( ListLength(reference) == 1 ) {
        algo = referencerAlgo ;
    } else {
        algo = AlgoForReference(listReferenceType, reference);
    }
    return algo ;
}

// GetInOut : get the names of the in/out ports
std::set<EString> Scheduler::GetInOut ( TabList &syndexDescr, PTREE topReference, PTREE referenceName )
{
    PTREE                                           algo = AlgoForReference(topReference, referenceName);
    std::map<EString, PTREE, LessString>            mapPorts ;
    std::map<EString, PTREE, LessString> ::iterator iterPort ;
    PTREE                                           listAlgo ;
    std::set<EString>                               inOutSet ;
    PTREE                                           algoDescription ;
    PTREE                                           port ;
    PTREE                                           type, mult, name ;
    PTREE                                           otherPort ;
    
    if ( algo == <ALGORITHM,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <SENSOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <ACTUATOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <MEMORY,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <CONSTANT,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>> ) {
        PTREE   listPort = listAlgo ;
        PTREE   ref ;
        while ( port = nextl(listPort) ) {
            switch ( port ) {
                case <IN,<TYPE,type,mult,name>> : 
                    {
                        iterPort = mapPorts.find(VString(Value(name)));
                        if ( iterPort == mapPorts.end() ) 
                            mapPorts [VString(Value(name))] = port ;
                        else {
                            otherPort = (*iterPort).second ;
                            if ( otherPort == <OUT> ) {
                                if ( VString(Value(type)) == Value(((PTREE)otherPort [1])[1]) && qcomparetree(((PTREE)port [1])[2], ((PTREE)otherPort [1])[2]) ) {
                                    inOutSet.insert(VString(value(name)));
                                } else {
                                    DumpTree(algo);
                                    NewLine();
                                    EString message ;
                                    message << "\nIn/Out port description do not match : " << Value(name) << " in : " << Value(algo) << "\n";
                                    DisplayError(message);
                                    MetaExit(3, "");
                                }
                            } else {
                                DumpTree(algo);
                                NewLine();
                                EString message ;
                                message << "\nTwo ports with the same name : " << Value(name) << " in : " << Value(algo) << "\n";
                                DisplayError(message);
                                MetaExit(3, "");
                            }
                        }
                    }
                    break ;
                case <OUT,<TYPE,type,mult,name>> : 
                    {
                        iterPort = mapPorts.find(VString(Value(name)));
                        if ( iterPort == mapPorts.end() ) 
                            mapPorts [VString(Value(name))] = port ;
                        else {
                            otherPort = (*iterPort).second ;
                            if ( otherPort == <IN> ) {
                                if ( VString(Value(type)) == Value(((PTREE)otherPort [1])[1]) && qcomparetree(((PTREE)port [1])[2], ((PTREE)otherPort [1])[2]) ) {
                                    inOutSet.insert(VString(value(name)));
                                } else {
                                    DumpTree(algo);
                                    NewLine();
                                    EString message ;
                                    message << "\nIn/Out port description do not match : " << Value(name) << " in : " << Value(algo) << "\n";
                                    DisplayError(message);
                                    MetaExit(3, "");
                                }
                            } else {
                                DumpTree(algo);
                                NewLine();
                                EString message ;
                                message << "\nTwo ports with the same name : " << Value(name) << " in : " << Value(algo) << "\n";
                                DisplayError(message);
                                MetaExit(3, "");
                            }
                        }
                    }
                    break ;
                default : break ;
            }
        }
    } else {
        DumpTree(algo);
        NewLine();
        EString message ;
        message << "\nUnexpected algo type for : " << Value(referenceName) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    return inOutSet ;
}

// GetPortDescr : get a port description
// parameters :
//              referencerAlgo : algo looking for port description
//              listReferenceType : the list of boxes with their type in referencerAlgo
//              reference : contents is start end of reference
PTREE Scheduler::GetPortDescr ( TabList &syndexDescr, PTREE referencerAlgo, PTREE listReferenceType, PTREE reference )
{
    PTREE   portReference ;
    PTREE   algo = GetAlgoDescr(syndexDescr, referencerAlgo, listReferenceType, reference);
    
    // if really an algo expand it
    PTREE   algoDescription ;
    PTREE   listAlgo ;
    PTREE   port ;
    
    if ( ListLength(reference) == 1 ) 
        portReference = reference [1];
    else 
        portReference = reference [2];
    if ( algo == <ALGORITHM,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <SENSOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <ACTUATOR,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <MEMORY,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>>
            || algo == <CONSTANT,<>,<>,algoDescription = <ALGO_ELEM,listAlgo>> ) {
        PTREE   listPort = listAlgo ;
        PTREE   ref ;
        while ( port = nextl(listPort) ) {
            switch ( port ) {
                case <IN,ref> : 
                case <OUT,ref> : 
                    {
                        if ( VString(Value(ref [3])) == Value(portReference) ) {
                            return port ;
                        }
                    }
                    break ;
                default : break ;
            }
        }
    } else {
        DumpTree(algo);
        NewLine();
        EString message ;
        message << "\nUnexpected algo type for : " << Value(reference) << " In " << Value(referencerAlgo) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    {
        EString message ;
        message << "Port not found : " << Value(reference) << " " << Value(portReference) << " In " << Value(referencerAlgo) << "\n";
        DisplayError(message);
        MetaExit(3, "");
    }
    
    // failed
    return port ;
}

// GetAssociatedPort for a reference
PTREE Scheduler::GetAssociatedPort ( TabList &syndexDescr, PTREE algo, PTREE topReference, PTREE referenceName, PTREE aDependence, PTREE &currentEnd, PTREE &otherEnd
    , bool &out )
{
    PTREE   start ;
    PTREE   end ;
    bool    found = false ;
    
    currentEnd = ();
    otherEnd = ();
    aDependence == <DEPENDENCE,<>,<END_DEPEND,start>,<END_DEPEND,end>>;
    
    // if one end is the current reference check it
    if ( VString(Value(referenceName)) == Value(start) ) {
        currentEnd = start ;
        otherEnd = end ;
        out = true ;
    }
    if ( VString(Value(referenceName)) == Value(end) ) {
        currentEnd = end ;
        otherEnd = start ;
        out = false ;
    }
    if ( currentEnd ) {
        PTREE   portDescrCurrent = GetPortDescr(syndexDescr, algo, topReference, currentEnd);
        PTREE   portDescrOther = GetPortDescr(syndexDescr, algo, topReference, otherEnd);
        
        // if current port not found it is an error
        if ( !(portDescrCurrent == <IN> || portDescrCurrent == <OUT>) ) {
            EString message ;
            message << "Port not found for : " << Value(currentEnd) << " " << Value(currentEnd [2]) << "\n";
            DisplayError(message);
        }
        
        // if other port not found it is an error
        if ( !(portDescrOther == <IN> || portDescrOther == <OUT>) ) {
            EString message ;
            message << "Port not found for : " << Value(currentEnd) << " " << Value(currentEnd [2]) << "\n";
            DisplayError(message);
        }
        
        // ok
        return <CONS,CopyTree(portDescrCurrent),CopyTree(portDescrOther)>;
    }
    
    // fail
    return ();
}

// AddBoxTime : add time for a box on all operators
void Scheduler::AddBoxTime ( PTREE &top, PTREE &boxName )
{
    PTREE   list ;
    PTREE   elem ;
    PTREE   mediaList ;
    PTREE   operatorMedia ;
    
    if ( top == <SYNDEX_TOP,<>,<>,list> ) {
        while ( elem = nextl(list) ) {
            if ( elem == <OPERATOR,<>,<>,operatorMedia = <OPERATOR_MEDIA_LIST,mediaList>> ) {
                
                // looks if box exists and exit this operator in this case
                {
                    PTREE   listItem = mediaList ;
                    PTREE   item ;
                    while ( item = nextl(listItem) ) {
                        if ( item == <CODE_TIME> && VString(Value(boxName)) == Value(item) ) 
                            continue ;
                    }
                }
                
                // if box doesn't exist insert it
                {
                    while ( mediaList != <LIST,<CODE_TIME>> ) 
                        nextl(mediaList);
                    PTREE   newTime = <CODE_TIME,<IDENT,CopyTree(boxName)>,<TIME,<NUMB,"0">,()>>;
                    if ( mediaList ) {
                        PTREE   next ;
                        next = mediaList ;
                        nextl(next);
                        PTREE   newElem = <LIST,newTime,next>;
                        mediaList += <,<>,newElem>;
                    } else {
                        PTREE   newElem = <LIST,newTime,mediaList>;
                        operatorMedia += <,newElem>;
                    }
                }
            }
        }
    }
}

// CheckSchedule : check a schedule
void Scheduler::CheckSchedule ()
{
    std::set<ComputeUnit *>                                 algos ;
    
    // create a set with all the task
    std::map<EString, ComputeUnit *, LessString> ::iterator iterAlgo ;
    
    for ( iterAlgo = LeafAlgorithm().begin() ; iterAlgo != LeafAlgorithm().end() ; iterAlgo++ ) {
        algos.insert((*iterAlgo).second);
    }
    
    std::vector<OperatorMedia *> ::iterator iterOperator ;
    std::vector<ConditionType>              nullVector ;
    
    // display planning for processors    
    for ( iterOperator = Operators().begin() ; iterOperator != Operators().end() ; iterOperator++ ) {
        if ( !(*iterOperator)->Planning().empty() ) {
            std::multimap<time_t, unsigned> ::iterator  iterTask ;
            (*iterOperator)->Extract((*iterOperator)->Planning().begin(), nullVector);
            for ( iterTask = (*iterOperator)->PlanningExtract().begin() ; iterTask != (*iterOperator)->PlanningExtract().end() ; iterTask++ ) {
                ComputeUnit *pTask = LeafAlgorithm()[TaskName((*iterOperator)->TaskContent((*iterTask).second)->Name())];
                algos.erase(pTask);
                
                // check that all task before end before
                // for all the in connection compute time up
                std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
                for ( iterIn = pTask->InEndConnectionsRef().begin() ; iterIn != pTask->InEndConnectionsRef().end() ; iterIn++ ) {
                    TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
                    
                    // look all the out comm and check
                    unsigned int    indexComm ;
                    for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                        ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                        
                        // if pOutUnit is a memory no dependancy
                        if ( pComputeUnit->Type() == ComputeUnit::MemoryUnit ) 
                            continue ;
                        
                        // else look
                        if ( pComputeUnit->StartTime() + pComputeUnit->ComputeTime() > pTask->StartTime() && pTask->InDependant() ) {
                            EString message = "Error for scheduling : ";
                            message << pComputeUnit->Name() << " ends after " << pTask->Name() << "\n";
                            std::cerr << message ;
                        }
                    }
                }
            }
        }
    }
    
    // if some tasks are remaining it is an error
    if ( !algos.empty() ) {
        EString                             message = "Error Task not scheduled : ";
        std::set<ComputeUnit *> ::iterator  iterRemain ;
        for ( iterRemain = algos.begin() ; iterRemain != algos.end() ; iterRemain++ ) {
            message << (*iterRemain)->Name() << " ";
        }
        message << "\n";
        std::cerr << message ;
    }
}

#if 0
    
    // ComputeTime : compute the time
    void Scheduler::ComputeAllTime ()
    {
        
        // search the first proc unit
        ComputeUnit *firstAlgo = (*LeafAlgorithm().begin()).second ;
        
        // compute the time downflow
        if ( firstAlgo ) 
            ComputeTime(firstAlgo);
    }
    
    // ComputeTimeUp : compute the time upflow
    void Scheduler::ComputeTimeUp ( ComputeUnit *algo )
    {
        
        // if already computed up is ok
        if ( algo->Computed() ) 
            return ;
        
        // start for this algo
        StartTime(algo, 2);
        
        // for all the in connection compute time up
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        
        for ( iterIn = algo->InEndConnectionsRef().begin() ; iterIn != algo->InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // look all the in comm and compute
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                ComputeTime(pComputeUnit, false);
            }
        }
        
        // end for this algo
        EndTime(algo, 2);
    }
    
    // ComputeTimeDown : compute the time downflow starting at an algo
    void Scheduler::ComputeTimeDown ( ComputeUnit *algo )
    {
        
        // start for this algo
        StartTime(algo, 3);
        
        // compute all the entries down
        // for all the in connection compute time up
        std::map<EString, ComputeUnit *, LessString> ::iterator iterOut ;
        
        for ( iterOut = algo->OutEndConnectionsRef().begin() ; iterOut != algo->OutEndConnectionsRef().end() ; iterOut++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterOut).second);
            
            // look all the in comm and compute
            unsigned int    indexComm ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedIn().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedIn()[indexComm];
                ComputeTime(pComputeUnit);
            }
        }
        
        // end for this algo
        EndTime(algo, 3);
    }
    
    // ComputeTime : compute time of an algo 
    void Scheduler::ComputeTime ( ComputeUnit *algo, bool down )
    {
        
        // if already done do nothing
        if ( algo->Computed() ) 
            return ;
        
        // be sure that all up have been computed
        ComputeTimeUp(algo);
        
        // end for this algo
        StartTime(algo, 1);
        
        // than compute start time
        std::map<EString, ComputeUnit *, LessString> ::iterator iterIn ;
        time_t                                                  startTime = 0 ;
        
        for ( iterIn = algo->InEndConnectionsRef().begin() ; iterIn != algo->InEndConnectionsRef().end() ; iterIn++ ) {
            TransUnit       *pTransUnit = (TransUnit *)((*iterIn).second);
            
            // search the out unit
            unsigned int    indexComm ;
            ComputeUnit     *pOutUnit = 0 ;
            for ( indexComm = 0 ; indexComm < pTransUnit->ConnectedOut().size() ; indexComm++ ) {
                ComputeUnit *pComputeUnit = pTransUnit->ConnectedOut()[indexComm];
                pOutUnit = pComputeUnit ;
                {
                    if ( &pOutUnit->Proc() != &algo->Proc() ) {
                        
                        // search a connection between this algo and current one
                        // OperatorMedia   *pOperatorMedia = pOutUnit->Proc().SearchConnection(&algo->Proc());
                        std::vector<OperatorMedia *>    vectMedia = Connectivity()[MATRIX_ACCESS(pOutUnit->Proc().Index(), algo->Proc().Index())];
                        
                        // ask to affect a task about this transmission
                        if ( vectMedia.size() >= 1 ) {
                            PTREE                       null ;
                            EString                     name ;
                            std::vector<ConditionType>  conditionVect ;
                            
                            // if conditions are not compatible forget it
                            if ( !MergeCondition(conditionVect, algo->Conditions(), pOutUnit->Conditions()) ) 
                                continue ;
                            
                            // create entry
                            name << pOutUnit->Name() << "->" << algo->Name();
                            time_t  currentStart = pOutUnit->StartTime() + pOutUnit->ComputeTime();
                            if ( !algo->InDependant() ) 
                                currentStart = 0 ;
                            
                            // send data
                            {
                                std::vector<OperatorMedia *> ::iterator iterMedia ;
                                unsigned                                indexTransaction = 1 ;
                                OperatorMedia                           *pStartProc = &pOutUnit->Proc();
                                OperatorMedia                           *pEndProc = 0 ;
                                for ( iterMedia = vectMedia.begin() ; iterMedia != vectMedia.end() ; iterMedia++ ) {
                                    EString         taskName = name ;
                                    unsigned int    taskNumber ;
                                    if ( vectMedia.size() > 1 ) 
                                        taskName << " (" << (int)indexTransaction++ << "/" << (int)vectMedia.size() << ")";
                                    unsigned    task
                                        = (*iterMedia)->CreateTask((taskNumber = TaskName(taskName)), pTransUnit->Reference(), pTransUnit->RefNumber(), currentStart
                                            , conditionVect, taskName);
                                    if ( task ) {
                                        currentStart = (*iterMedia)->TaskContent(task)->End();
                                        
                                        // keep informations about this task
                                        {
                                            
                                            // compute start and end proc
                                            std::vector<OperatorMedia *> ::iterator iterProc ;
                                            pEndProc = 0 ;
                                            LinkUnit    *linkUnit = (LinkUnit *)(*iterMedia);
                                            for ( iterProc = linkUnit->AllOperators().begin() ; iterProc != linkUnit->AllOperators().end() ; iterProc++ ) {
                                                if ( *iterProc != pStartProc ) 
                                                    pEndProc = *iterProc ;
                                            }
                                            
                                            // store data
                                            pvTransByNumber.insert(
                                                std::pair<int, TransTrunk> (pTransUnit->RefNumber()
                                                    , TransTrunk(pTransUnit, pStartProc, pEndProc, pOutUnit, algo, (*iterIn).first, (*iterMedia), task)));
                                            
                                            // prepare next step
                                            pStartProc = pEndProc ;
                                            pEndProc = 0 ;
                                        }
                                    }
                                }
                            }
                            if ( currentStart > startTime ) 
                                startTime = currentStart ;
                        } else {
                            EString message ;
                            message << "Missing connection for algo  : " << algo->Name() << " " << pOutUnit->Name() << "\n";
                            DisplayError(message);
                        }
                        
                        // update main task comm time
                    } else {
                        
                        // Compute the time
                        time_t  newTime = pOutUnit->StartTime() + pOutUnit->ComputeTime();
                        if ( !algo->InDependant() ) 
                            newTime = 0 ;
                        if ( newTime > startTime ) 
                            startTime = newTime ;
                    }
                }
                if ( !pOutUnit ) {
                    EString message ;
                    message << "Link without emitter \n";
                    DisplayError(message);
                    pTransUnit->Computed(true);
                }
            }
            
            // set the start time
            algo->StartTime(startTime);
            
            // compute the time in task
            unsigned    task = algo->Proc().CreateTask(algo->NameNumber(), algo->Reference(), algo->RefNumber(), algo->StartTime(), algo->Conditions(), algo->Name());
            
            // set compute time
            if ( task ) {
                Task    *pTask = algo->Proc().TaskContent(task);
                algo->StartTime(pTask->Start()).ComputeTime(pTask->End() - pTask->Start());
            } else {
                algo->ComputeTime(0);
                EString message ;
                message << "No proc for : " << algo->Name() << "\n";
                DisplayError(message);
            }
            
            // now this task is computed
            algo->Computed(true);
            
            // end for this algo
            EndTime(algo, 1);
            
            // compute time down
            if ( down ) 
                ComputeTimeDown(algo);
        }
    }
    
#endif


