/*************************************************************************/
/*                                                                       */
/*        scheduler.h   : Eric Lavillonniere 2005                        */
/*        company       : Mitsubishi Electric ITE                        */
/*                                                                       */
/*************************************************************************/
/*
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation; either version 2.1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.
   
   You should have received a copy of the GNU Lesser General Public License
   along with this program; see the file COPYING. If not, write to 
   the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   */
#ifndef FILE_SCHEDULER
#   define FILE_SCHEDULER 
    
    // system includes
#   include <vector>
#   include <map>
    
    // private includes
#   include "operatormedia.h"
#   include "computeunit.h"
#   include "ptree.h"
#   include "tablist.h"
#   include "symb.h"
#   include "syndex.h"
#   include "decsyndex.h"
#   include "fcntl.h"
#   define MATRIX_ACCESS(x, y) ((((x)-1)*(pvOperators.size()))+((y)-1))
    
    // #define ALGO_MATRIX_SIZE(p) (((p) * (2*(pvNumberProc) - (p) - 1 ) ) / 2)
    // #define ALGO_MATRIX_ACCESS(x, y) ALGO_MATRIX_SIZE((x-1)) + ((y) -1)
    static unsigned char    bitPattern [] = { 0x1, 0x2, 0x4, 0x8, 0x10, 0x20, 0x40, 0x80 };
    
    inline void MatrixAccessError ( int x, int y, int pos )
    {
        EString message ;
        
        message << "Matrix access error : " << x << " " << y << " " << pos << "\n";
        std::cerr << message ;
    }
    
#   define ALGO_MATRIX_SIZE(p) ((p) * (pvNumberAlgo/8+1))
#   define ALGO_MATRIX_ACCESS_SLOT(x, y) ALGO_MATRIX_SIZE((x-1)) + ((y) -1)/8
#   define ALGO_MATRIX_ACCESS_BIT(x, y) ((y) -1)%8
#   define ALGO_MATRIX_GET(mat, x, y) (ALGO_MATRIX_ACCESS_SLOT(x,y) < (mat) .size () ? (mat)[ALGO_MATRIX_ACCESS_SLOT(x,y)] & bitPattern[ALGO_MATRIX_ACCESS_BIT(x,y)]:(MatrixAccessError(x,y,ALGO_MATRIX_ACCESS_SLOT(x,y)),0))
#   define ALGO_MATRIX_PUT(mat, x, y, z) (ALGO_MATRIX_ACCESS_SLOT(x,y) < (mat) .size () ? (!z? \
                                                                                                                           ((mat)[ALGO_MATRIX_ACCESS_SLOT(x,y)] &= ~(bitPattern[ALGO_MATRIX_ACCESS_BIT(x,y)])) \
                                                                                                                           :((mat)[ALGO_MATRIX_ACCESS_SLOT(x,y)] |= (bitPattern[ALGO_MATRIX_ACCESS_BIT(x,y)]))):(MatrixAccessError(x,y,ALGO_MATRIX_ACCESS_SLOT(x,y)),0))
    
    class ConditionDescriptor {
        
        public :
        
            ConditionDescriptor ( PTREE &tree )
            {
                condition = tree ;
            }
            
            ConditionDescriptor ()
            {
                condition = PTREE(0);
            }
            
            PTREE   condition ;
    };
    
    class TransTrunk {
        
        public :
        
            TransTrunk ()
                : trans(0),  start(0),  end(0)
            {}
            
            TransTrunk ( TransUnit *pTransUnit, OperatorMedia *pStart, OperatorMedia *pEnd, ComputeUnit *pStartTask, ComputeUnit *pEndTask, const EString &pChannel
                , OperatorMedia *pMedia, unsigned paramTaskNumber )
                : trans(pTransUnit),  start(pStart),  end(pEnd),  startTask(pStartTask),  endTask(pEndTask),  channel(pChannel),  media(pMedia)
                    ,  taskNumber(paramTaskNumber)
            {}
            
            TransUnit       *trans ;
            OperatorMedia   *start ;
            OperatorMedia   *end ;
            ComputeUnit     *startTask ;
            ComputeUnit     *endTask ;
            EString         channel ;
            OperatorMedia   *media ;
            unsigned        taskNumber ;
    };
    
    class Scheduler {
        
        public :
        
            Scheduler ( PTREE description )
                : pvOMNumerator(0),  pvNumberProc(1),  pvIndexType(1),  pvNumberAlgo(1),  pvpComputeUnit(0),  pvCondition(0),  pvTaskNameLock(0),  pvUseDma(true)
            {
                pvDescription = CopyTree(description);
                pvRefDescription = CopyTree(description);
                
                EString null("");
                
                TaskName(null);
            }
            
            virtual ~Scheduler ()
            {
                std::vector<OperatorMedia *> ::iterator iter ;
                
                for ( iter = Operators().begin() ; iter != Operators().end() ; iter++ ) {
                    delete *iter ;
                }
                for ( iter = Links().begin() ; iter != Links().end() ; iter++ ) {
                    delete *iter ;
                }
                if ( pvpComputeUnit ) {
                    delete pvpComputeUnit ;
                    pvpComputeUnit = 0 ;
                }
                ClearNames(true);
            }
            
            // GetTypesFromAlgo : create new types from an expanded algo
            PTREE   GetTypesFromAlgo (PTREE algo, PTREE typeName, EString prefix) ;
            
            // GetPortDescr : get a port description
            PTREE   GetPortDescr (TabList &syndexDescr, PTREE algo, PTREE listReferenceType, PTREE reference) ;
            
            // EvalParameter : eval a parameter
            PTREE   EvalParameter (SymbolTable &symbTab, PTREE parameter) ;
            
            // AttachSoftwareEntry : attach a software entry to a list of processors
            void    AttachSoftwareEntry (PTREE mainAlgo, PTREE entry, PTREE listProc) ;
            
            // expand reference in algo
            PTREE   ExpandParameters (SymbolTable &symbTab, TabList &syndexDescript, PTREE algo, PTREE parameters) ;
            
            // dispatch the software constraint on each algorithm
            void    DispatchConstraint (TabList &syndexDescript, PTREE description, PTREE topAlgo) ;
            
            // CheckExpand : check that data are correctly connected and 
            //               if necessary adapat width of channels
            void    CheckExpand (TabList &syndexDescr, PTREE description) ;
            
            // Explore : construct architecture
            void    Explore () ;
            
#           if 0
                
                // Schedule : main function for scheduling tasks
                void Schedule ()
                {
                    ComputeAllTime();
                }
                
                // ComputeTime : compute time of this algo
                void    ComputeTime () ;
#           endif
            
            // Schedule : main function for scheduling tasks
            void Schedule ( std::vector<int> &trail )
            {
                ComputeTime(trail);
            }
            
            void        ComputeTransfert (ComputeUnit *algo, ComputeUnit *pOutUnit, TransUnit *pTransUnit, time_t &delayTime) ;
            void        UpdateStart (ComputeUnit *algo, ComputeUnit *pOutUnit, TransUnit *pTransUnit, const EString &channelName, time_t &startTime) ;
            
            // ComputeTime : compute time of an algo 
            void        ComputeTime (std::vector<int> &trail) ;
            
            // ExpandArchi : expand an architecture from description
            void        ExpandArchi (TabList &syndexDescr, PTREE &archi, bool update = false) ;
            
            // ExpandAlgo : expand main algo from description
            ComputeUnit *ExpandAlgo (TabList &syndexDescr, PTREE &algo, EString nameBox, std::vector<ConditionType> &vectCond) ;
            
            // AffectProcessor : affect algo to a processor
            void        AffectProcessor (EString &algo, EString &processor) ;
            
            // AffectProcessor : affect algo to a processor
            void AffectProcessor ( ComputeUnit *pUnit, unsigned int indexProcessor )
            {
                pUnit->Proc(*Operators()[indexProcessor]);
            }
            
            // GetProcessors : get processors for an algo
            void    GetProcessors (int algo, std::vector<int> &procs) ;
            
            // GetProcessor : affect a processor to a unit
            void    GetProcessor (ComputeUnit *pUnit, PTREE &algo) ;
            
            // Description : gives the description of the scene
            PTREE &Description ()
            {
                return pvDescription ;
            }
            
            // ComputeLinks : compute all the links
            void                        ComputeLinks (PTREE &listDepedence, std::vector<std::vector<PTREE> > &links) ;
            
            // UpdateHead : update head after taking one element
            void                        UpdateHead (ComputeUnit *algo, std::set<ComputeUnit *> &head, std::set<ComputeUnit *> &computed) ;
            
            // SearchComputeTrail : search a trail for computing time
            std::vector<unsigned int>   RandomTrail () ;
            
            // MutateTrail : mutate a trail
            void                        MutateTrail (int *trail) ;
            
            // SearchAllHeads
            void                        SearchAllHeads () ;
            
            // SearchHeadUp
            void                        SearchHeadUp (ComputeUnit *algo, std::set<ComputeUnit *> &trail) ;
            
            // SearchHeadDown
            void                        SearchHeadDown (ComputeUnit *algo, std::set<ComputeUnit *> &trail) ;
            
            // ComputeTime : compute time of an algo 
            void                        SearchHead (ComputeUnit *algo, std::set<ComputeUnit *> &trail, bool down = true) ;
            
            // ComputeTime : compute the time
            void                        ComputeAllTime () ;
            
            // ComputeTimeUp : compute the time upflow
            void                        ComputeTimeUp (ComputeUnit *algo) ;
            
            // ComputeTimeDown : compute the time downflow starting at an algo
            void                        ComputeTimeDown (ComputeUnit *algo) ;
            
            // ComputeTime : compute time of an algo 
            void                        ComputeTime (ComputeUnit *algo, bool down = true) ;
            
            // DisplaySchedule : display the schedule
            void                        DisplaySchedule () ;
            
            // GenerateArchitecture : generate architecture file
            void                        GenerateArchitecture () ;
            
            // GenerateFunctionModel : generate a model for each function
            void                        GenerateFunctionModel () ;
            
            // GenerateFunctionModel : generate a model for each function
            void                        GenerateFunctionCode () ;
            
            // GenerateCode : generate the associated code
            void                        GenerateCode () ;
            
            // ComputeConnectivity : compute connectivity matrix
            void                        ComputeConnectivity () ;
            
            // ComputeConnectivity : compute connectivity matrix for algo
            void                        ComputeAlgoConnectivity () ;
            
            // DirectDescent : look if one is direct descent of the other
            bool                        DirectDescent (ComputeUnit *firstUnit, ComputeUnit *secondUnit) ;
            
            // Connectivity : gives the connectivity matrix
            std::vector<std::vector<OperatorMedia *> > &Connectivity ()
            {
                return pvConnectivityMatrix ;
            }
            
            // Connectivity : gives the connectivity matrix
            std::vector<unsigned char> &AlgoConnectivity ()
            {
                return pvAlgoConnectivityMatrix ;
            }
            
            // Costs : compute latency and mean time
            void    Cost (unsigned int &latency, unsigned int &mean) ;
            
            // Algo : the algo by name 
            std::map<EString, ComputeUnit *, LessString> &Algorithm ()
            {
                return pvAlgo ;
            }
            
            // Algo : the algo by name 
            std::map<EString, ComputeUnit *, LessString> &LeafAlgorithm ()
            {
                return pvLeafAlgo ;
            }
            
            // Proc : the operators by name
            std::vector<OperatorMedia *> &Operators ()
            {
                return pvOperators ;
            }
            
            // Proc : the operators by name
            std::vector<OperatorMedia *> &Links ()
            {
                return pvLinks ;
            }
            
            // TaskName : insert task name
            unsigned int TaskName ( const EString &name )
            {
                taskName.push_back(name);
                //	    std::cout << " name : " << name.c_str() << "\n";
                return taskName.size() - 1 ;
            }
            
            // TaskName : get a task name
            EString &TaskName ( unsigned int taskNumber )
            {
                if ( taskNumber < taskName.size() ) 
                    return taskName [taskNumber];
                else 
                    return TaskName(0);
            }
            
            // TaskNameLock : lock the task name
            void TaskNameLock ()
            {
                pvTaskNameLock = taskName.size();
            }
            
            // ClearSchedule : clear current schedule
            void                ClearSchedule () ;
            
            // GetAlgoDescr : get an algo desciption knowing it's name in a box
            // parameters :
            //              referencerAlgo : algo looking for port description
            //              listReferenceType : the list of boxes with their type in referencerAlgo
            //              reference : contents is start end of reference
            PTREE               GetAlgoDescr (TabList &syndexDescr, PTREE referencerAlgo, PTREE listReferenceType, PTREE reference) ;
            
            // ReferenceForAlgo : get references of an algo
            std::vector<PTREE>  ConditionsForAlgo (PTREE &algo) ;
            
            // AlgoForReference : return the algo master for a reference
            PTREE               AlgoForReference (PTREE listReferenceType, PTREE reference) ;
            
            // GetInOut : get the names of the in/out ports
            std::set<EString>   GetInOut (TabList &syndexDescr, PTREE topReference, PTREE referenceName) ;
            
            // ClearNames : clear the names
            void ClearNames ( bool full = false )
            {
                if ( full ) 
                    taskName.clear();
                else {
                    std::vector<EString> ::iterator iter ;
                    iter = taskName.begin();
                    if ( pvTaskNameLock ) 
                        std::advance(iter, pvTaskNameLock);
                    taskName.erase(iter, taskName.end());
                }
            }
            
            // Dump : dump the modified tree
            void Dump ( EString file )
            {
                int oldOutput = output ;
                
                output = _open(file.c_str(), O_RDWR | O_TRUNC | O_CREAT, 0666);
                if ( output > 1 ) {
                    DecompSyndex    decompSyndex ;
                    decompSyndex.startComment = decompSyndex.middleComment = decompSyndex.endComment = decompSyndex.plusComment = "";
                    DecompSyndex::ptDecomp = &decompSyndex ;
                    decomp_syndex(pvRefDescription);
                    NewLine();
                }
                _close(output);
                output = oldOutput ;
            }
            
            // ExpandReference : expand a reference
            void                    ExpandReference (TabList &syndexDescr, unsigned int coeff, unsigned int currMult, PTREE algo, PTREE topReference, PTREE reference
                , PTREE dependences, std::set<EString> &inOut) ;
            
            // GetAssociatedPort
            PTREE                   GetAssociatedPort (TabList &syndexDescr, PTREE algo, PTREE topReference, PTREE referenceName, PTREE aDependence, PTREE &currentEnd
                , PTREE &otherEnd, bool &out) ;
            
            // AddBoxTime : add time for a box
            void                    AddBoxTime (PTREE &top, PTREE &boxName) ;
            
            // the names
            std::vector<EString>    taskName ;
            
            // Heads : get the list of heads
            std::set<ComputeUnit *> &Heads ()
            {
                return pvHeads ;
            }
            
            // DisplayConnectivity : Display Connectivity matrix for algos
            void    DisplayConnectivity () ;
            
            // CheckSchedule : check a schedule
            void    CheckSchedule () ;
            
            // Condition : return the current condition number
            unsigned int Condition ()
            {
                return pvCondition ;
            }
            
            // Condition : set condition
            Scheduler &Condition ( unsigned int condition )
            {
                pvCondition = condition ;
                return *this ;
            }
            
            // GetAlgoByNumber
            ComputeUnit *GetAlgoByNumber ( unsigned int ref )
            {
                return pvAlgoByNumber [ref];
            }
            
            // Name : set the name
            Scheduler &Name ( EString &name )
            {
                pvName = name ;
                return *this ;
            }
            
            // Name : get the name
            const EString &Name () const
            {
                return pvName ;
            }
            
            // UseDma : get use dma value
            bool UseDma () const
            {
                return pvUseDma ;
            }
            
            // UseDma : set use dma value
            Scheduler &UseDma ( bool useDma )
            {
                pvUseDma = useDma ;
                return *this ;
            }
        
        private :
        
            PTREE                                           pvDescription ;            // description of boards and tasks
            PTREE                                           pvRefDescription ;         // description which can be printed : no sort
            std::map<unsigned int, OperatorMedia>           pvOperatorMedia ;          // planning for operators and media
            unsigned int                                    pvOMNumerator ;            // for assigning a number to operator or media
            ComputeUnit                                     *pvpComputeUnit ;          // the computeunit 
            std::vector<OperatorMedia *>                    pvOperators ;              // the operator
            std::vector<OperatorMedia *>                    pvLinks ;                  // the links
            std::map<EString, ComputeUnit *, LessString>    pvAlgo ;                   // the links of algo by name 
            std::map<unsigned int, ComputeUnit *>           pvAlgoByNumber ;           // the links of algo by number
            std::multimap<int, TransTrunk>                  pvTransByNumber ;          // the links of transmission task by number
            std::map<unsigned int, ConditionDescriptor>     pvConditionByNumber ;      // the conditions
            std::map<EString, ComputeUnit *, LessString>    pvLeafAlgo ;               // the links of algo by name 
            std::vector<std::vector<OperatorMedia *> >      pvConnectivityMatrix ;     // interconnectivity matrix
            std::vector<unsigned char>                      pvAlgoConnectivityMatrix ; // connectivity matrix for algo
            unsigned int                                    pvNumberProc ;
            unsigned int                                    pvIndexType ;
            unsigned int                                    pvNumberAlgo ;
            std::set<ComputeUnit *>                         pvHeads ;                  // heads of algo
            unsigned int                                    pvCondition ;              // condition numbering
            unsigned int                                    pvTaskNameLock ;
            EString                                         pvName ;
            bool                                            pvUseDma ;
    };
#endif
