/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "opMainWindow.h"
#include "opMainWindowIf.h"
#include "opApplication.h"
#include "opMenuBarManager.h"
#include "opToolBarManager.h"
#include "opStatusBarManager.h"
#include "opToolBarIf.h"
#include "opStatusBarIf.h"
#include "opMenu.h"

/*****************************************************
 *
 * OPMainWindowIf
 *
 *****************************************************/

OPMainWindowIf::OPMainWindowIf( OPMainWindow* _main )
{
  ADD_INTERFACE( "IDL:OpenParts/MainWindow:1.0" );
  
  m_pMainWindow = _main;
  
  m_idActivePart = 0;
  m_id = 1;
}

OPMainWindowIf::~OPMainWindowIf()
{
  cleanUp();
}

void OPMainWindowIf::cleanUp()
{
  if ( m_bIsClean )
    return;

  KOMComponent::cleanUp();

  m_vActivePart = 0L;
  m_mapParts.clear();
  m_mapCaptions.clear();
}

OpenParts::Id OPMainWindowIf::registerPart( OpenParts::Part_ptr _part, const char *caption )
{
  cerr << "Registering part " << m_id << endl;

  OpenParts::Id id = m_id++;
  m_mapParts[ id ] = OpenParts::Part::_duplicate( _part );

  if ( caption && caption[0] != 0 )
    m_mapCaptions[ id ] =  caption;
  return id;
}

void OPMainWindowIf::unregisterPart( OpenParts::Id id )
{
  cerr << "Unregistering part " << id << endl;

  m_mapParts.erase( id );
  m_mapCaptions.erase( id );
}
  
void OPMainWindowIf::setActivePart( OpenParts::Id _id )
{
  int old_id = m_idActivePart;
  
  // No changes ?
  if ( _id == m_idActivePart )
    return;
  
  OpenParts::Part_var part;
  // Do we want to set a new active part ?
  if ( _id )
  {    
    // Find it
    part = findPart( _id );
    if( CORBA::is_nil( part ) )
    {
      cerr << "ERROR: void OPMainWindowIf::setActivePart( OpenParts::Id _id )" << endl;
      cerr << "       id " << _id << " is unknown" << endl;
      return;
    }
  }
  
  // Active part looses focus
  if ( !CORBA::is_nil( m_vActivePart ) )
    m_vActivePart->setFocus( false );

  // Change the active part
  if ( _id )
  {    
    m_vActivePart = OpenParts::Part::_duplicate( part );
    m_idActivePart = _id;
  }
  else
  {    
    m_vActivePart = 0L;
    m_idActivePart = 0;
  }

  // New part gets focus if there is any new part at all
  if ( !CORBA::is_nil( m_vActivePart ) )
  {
    map<OpenParts::Id,string>::iterator it = m_mapCaptions.find( m_idActivePart );
  
    if (m_pMainWindow->autoCaptionMode() && it != m_mapCaptions.end() )
      m_pMainWindow->setCaption( it->second.c_str() );
  
    m_vActivePart->setFocus( true );
  }    

  // Tell everyone ... this may for example cause changes in menu- and toolbars
  m_pMainWindow->emitActivePartChanged( _id, old_id );
}

OpenParts::Part_ptr OPMainWindowIf::activePart()
{
  return OpenParts::Part::_duplicate( m_vActivePart );
}

OpenParts::Id OPMainWindowIf::activePartId()
{
  return m_idActivePart;
}

CORBA::Boolean OPMainWindowIf::partClicked( OpenParts::Id part, CORBA::Long button )
{
  setActivePart( part );

  // Do NOT consume this event. That means the part will see
  // the mouse click.
  return false;
}

void OPMainWindowIf::setPosition( CORBA::Long _x, CORBA::Long _y )
{
  m_pMainWindow->move( _x, _y );
}

void OPMainWindowIf::setSize( CORBA::Long _width, CORBA::Long _height )
{
  m_pMainWindow->resize( _width, _height );
}

char *OPMainWindowIf::partCaption( OpenParts::Id part )
{
  map<OpenParts::Id,string>::iterator it = m_mapCaptions.find( part );
  if ( it == m_mapCaptions.end() )
    return CORBA::string_dup( "" );
    
  return CORBA::string_dup( it->second.c_str() );
}

void OPMainWindowIf::setPartCaption( OpenParts::Id part, const char *caption )
{
  cerr << "void OPMainWindowIf::setPartCaption( " << part << ", " << caption << " ) " << endl;

  map<OpenParts::Id,string>::iterator it = m_mapCaptions.find( part );
  if ( it == m_mapCaptions.end() )
    return;

  m_mapCaptions.erase( part );
  if ( !caption )
    return;
  m_mapCaptions[ part ] = caption;
  
  if (m_pMainWindow->autoCaptionMode() && m_idActivePart == part)
    m_pMainWindow->setCaption( caption );
}

OpenParts::Part_ptr OPMainWindowIf::findPart( OpenParts::Id _id )
{
  map<OpenParts::Id,OpenParts::Part_var>::iterator it = m_mapParts.find( _id );
  if ( it == m_mapParts.end() )
    return 0L;
  
  return OpenParts::Part::_duplicate( it->second );
}

OpenParts::MenuBarManager_ptr OPMainWindowIf::menuBarManager()
{
  // Do not call menuBarManager() directly since it would automatically create a manager.
  if ( !m_pMainWindow->hasMenuBarManager() )
    return 0L;
  
  return OpenParts::MenuBarManager::_duplicate( m_pMainWindow->menuBarManager() );
}

OpenParts::ToolBarManager_ptr OPMainWindowIf::toolBarManager()
{
  // Do not call toolBarManager() directly since it would automatically create a manager.
  if ( !m_pMainWindow->hasToolBarManager() )
    return 0L;
  
  return OpenParts::ToolBarManager::_duplicate( m_pMainWindow->toolBarManager() );
}

OpenPartsUI::ToolBarFactory_ptr OPMainWindowIf::toolBarFactory()
{
  // Do not call toolBarFactory() directly since it would automatically create a factory.
  if ( !m_pMainWindow->hasToolBarFactory() )
    return 0L;
  
  return OpenPartsUI::ToolBarFactory::_duplicate( m_pMainWindow->toolBarFactory() );
}

OpenParts::StatusBarManager_ptr OPMainWindowIf::statusBarManager()
{
  // Do not call statusBarManager() directly since it would automatically create a manager.
  if ( !m_pMainWindow->hasStatusBarManager() )
     return 0L;
     
  return OpenParts::StatusBarManager::_duplicate( m_pMainWindow->statusBarManager() );
}


/*****************************************************
 *
 * OPMainWindow
 *
 *****************************************************/

OPMainWindow::OPMainWindow( const char *_name = 0L ) : KTMainWindow( _name )
{
  m_pMenuBarManager = 0L;
  m_pToolBarManager = 0L;
  m_pToolBarFactory = 0L;
  m_pStatusBarManager = 0L;
  m_pInterface = 0L;
  m_pMenuBar = 0L;
  m_bAutoCaption = true;
}

OPMainWindow::~OPMainWindow()
{
  if ( m_pInterface )
    CORBA::release( m_pInterface );
  if ( m_pMenuBarManager )
    CORBA::release( m_pMenuBarManager );
  if ( m_pStatusBarManager )
    CORBA::release( m_pStatusBarManager );    
}

OPMainWindowIf* OPMainWindow::interface()
{
  if ( m_pInterface == 0L )
    m_pInterface = new OPMainWindowIf( this );
  return m_pInterface;
}

OPMenuBarManager* OPMainWindow::menuBarManager()
{
  if ( m_pMenuBarManager == 0L )
  {    
    interface()->addAggregate( ( m_pMenuBarManager = new OPMenuBarManager( this ) ),
			       "IDL:OpenParts/MenuBarManager:1.0" );
  }
  
  return m_pMenuBarManager;
}

OPToolBarManager* OPMainWindow::toolBarManager()
{
  if ( m_pToolBarManager == 0L )
  {    
    interface()->addAggregate( ( m_pToolBarManager = new OPToolBarManager( this ) ),
			       "IDL:OpenParts/ToolBarManager:1.0" );
  }
  
  return m_pToolBarManager;
}

OPToolBarFactory* OPMainWindow::toolBarFactory()
{
  if ( m_pToolBarFactory == 0L )
  {    
    interface()->addAggregate( ( m_pToolBarFactory = new OPToolBarFactory( this ) ),
			       "IDL:OpenPartsUI/ToolBarFactory:1.0" );
  }
  
  return m_pToolBarFactory;
}

OPStatusBarManager* OPMainWindow::statusBarManager()
{
  if ( m_pStatusBarManager == 0L )
  {
    interface()->addAggregate( ( m_pStatusBarManager = new OPStatusBarManager( this ) ),
                               "IDL:OpenParts/StatusBarManager:1.0" );
  }

  return m_pStatusBarManager;  
}

void OPMainWindow::emitActivePartChanged( unsigned long _new_part_id, unsigned long _old_part_id )
{
  emit activePartChanged( _new_part_id, _old_part_id );
}

OPToolBar* OPMainWindow::opToolBar( int _id )
{
  OPToolBar *bar = m_mapToolBars[ _id ];
  if ( bar )
    return bar;
  
  bar = new OPToolBar( this );
  addToolBar( bar );
  m_mapToolBars.insert( _id, bar );
  
  return bar;
}

OPMenuBar* OPMainWindow::opMenuBar()
{
  if ( m_pMenuBar )
    return m_pMenuBar;
  
  m_pMenuBar = new OPMenuBar( this );
  setMenu( m_pMenuBar );  

  return m_pMenuBar;
}

#include "opMainWindow.moc"
