/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>
 
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/     

#include "main.h"

#include <iostream>
#include <assert.h>

#include <komApplication.h>

CORBA::Object_ptr myActivator( KOMComponent* _core )
{
  return new Extension1( _core );
}

Core::Core() 
{
  cout << "+Core" << endl;

  ADD_INTERFACE( "IDL:MyComponent/Core:1.0" );
  
  addAggregate( myActivator, "IDL:MyComponent/Extension1:1.0" );
}  

Core::~Core()
{
  cout << "-Core" << endl;
 
  cleanUp();
}

void Core::sayHello()
{
  cout << "Hello, I am the core component" << endl;
}

void Core::cleanUp()
{
  if ( m_bIsClean )
    return;

  KOMComponent::cleanUp();
}

Extension1::Extension1( KOMComponent* _core )
{
  cout << "+Extension1" << endl;
}

Extension1::~Extension1()
{
  cout << "-Extension1" << endl;
}

void Extension1::sayHello()
{
  cout << "Hello, I am the Extension1. I am an builtin aggregate" << endl;
}

KOM::Aggregate_ptr Extension2Factory::create( KOM::Component_ptr _core )
{
  return KOM::Aggregate::_duplicate( new Extension2( _core ) );
}

Extension2::Extension2( KOM::Component_ptr _core ) : KOMAggregate( _core )
{
  cout << "+Extension2" << endl;
}

Extension2::~Extension2()
{
  cout << "-Extension2" << endl;
}

void Extension2::sayHello()
{
  cout << "Hello, I am the Extension2. I am a dynamic aggregate" << endl;
}

KOM::Aggregate_ptr WrapperFactory::create( KOM::Component_ptr _core )
{
  return KOM::Aggregate::_duplicate( new Wrapper( _core ) );
}

Wrapper::Wrapper( KOM::Component_ptr _core ) : KOMAggregate( _core )
{
  cout << "+Wrapper" << endl;
}

Wrapper::~Wrapper()
{
  cout << "-Wrapper" << endl;
}

void Wrapper::talk( const char* _message )
{
  cout << "Wrapper: " << _message << endl;
}

KOM::Aggregate_ptr Extension4Factory::create( KOM::Component_ptr _core )
{
  return KOM::Aggregate::_duplicate( new Extension4( _core ) );
}

Extension4::Extension4( KOM::Component_ptr _core ) : KOMAggregate( _core )
{
  cout << "+Extension4" << endl;

  CORBA::Object_var tmp = _core->getInterface( "IDL:MyInterfaces/Talk:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  m_vWrapper = MyInterfaces::Talk::_narrow( tmp );
  assert( !CORBA::is_nil( m_vWrapper ) );
}

Extension4::~Extension4()
{
  cout << "-Extension4" << endl;
  cleanUp();
}

void Extension4::cleanUp()
{
  if ( m_bIsClean )
    return;

  m_vWrapper = 0L;

  KOMAggregate::cleanUp();
}

void Extension4::sayNiceHello()
{
  m_vWrapper->talk( "Hello, I am the Extension4. I am a dynamic aggregate" );
  m_vWrapper->talk( "       I need the Wrapper aggregate to do my job here" );
}

KOM::Plugin_ptr Plugin1Factory::create( KOM::Component_ptr _core )
{
  return KOM::Plugin::_duplicate( new Plugin1( _core ) );
}

Plugin1::Plugin1( KOM::Component_ptr _core ) : KOMPlugin( _core )
{
  cout << "+Plugin1" << endl;
}

Plugin1::~Plugin1()
{
  cout << "-Plugin1" << endl;
}

void Plugin1::sayHello()
{
  cout << "Hello, I am the Plugin1. I am a dynamic plugin" << endl;
}

KOM::Plugin_ptr Plugin2Factory::create( KOM::Component_ptr _core )
{
  return KOM::Plugin::_duplicate( new Plugin2( _core ) );
}

Plugin2::Plugin2( KOM::Component_ptr _core ) : KOMPlugin( _core )
{
  cout << "+Plugin2" << endl;

  // Get a reference to the wrapper. The wrapper will always 
  // be there, because we added this as a dependecy for this plugin.
  CORBA::Object_var tmp = _core->getPluginInterface( "IDL:MyExtension/PluginWrapper:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  m_vWrapper = MyExtension::PluginWrapper::_narrow( tmp );
  assert( !CORBA::is_nil( m_vWrapper ) );
}

Plugin2::~Plugin2()
{
  cout << "-Plugin2" << endl;

  cleanUp();
}

void Plugin2::cleanUp()
{
  if ( m_bIsClean )
    return;

  m_vWrapper = 0L;

  KOMPlugin::cleanUp();
}

void Plugin2::sayHello()
{
  m_vWrapper->talk( "Hello, I am the Plugin2. I am a dynamic plugin" );
  m_vWrapper->talk( "       I need the PluginWrapper to work" );
}

KOM::Plugin_ptr PluginWrapperFactory::create( KOM::Component_ptr _core )
{
  return KOM::Plugin::_duplicate( new PluginWrapper( _core ) );
}

PluginWrapper::PluginWrapper( KOM::Component_ptr _core ) : KOMPlugin( _core )
{
  cout << "+PluginWrapper" << endl;
}

PluginWrapper::~PluginWrapper()
{
  cout << "-PluginWrapper" << endl;
}

void PluginWrapper::talk( const char* _message )
{
  cout << "PluginWrapper: " << _message << endl;
}

int main( int argc, char **argv )
{
  KOMApplication app( argc, argv );
  
  /***********
   * Create a component
   ***********/
  // We have to duplicate here. Later on we want to call "core->destroy()".
  // This will internally call "CORBA::release( this )". Upon creation the
  // reference count of "core" is 1. If we want to hold a reference on the "core"
  // in a *_var, then we have to duplicate once again => reference = 2.
  // When we finally call "core->destroy" => reference = 1 and the "core"
  // calls its "cleanUp" function to get rid of its plugins and aggregates.
  // And at the end "core = 0L" => reference = 0 => destructor.
  MyComponent::Core_var core = MyComponent::Core::_duplicate( new Core );
  core->sayHello();
  
  cout << "1 ---------------------------------------------------" << endl;
  
  /************
   * Ask for a certain interface. The "core" will have
   * to activate a builtin aggregate to do so.
   ************/
  CORBA::Object_var tmp = core->getInterface( "IDL:MyComponent/Extension1:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  MyComponent::Extension1_var ext1 = MyComponent::Extension1::_narrow( tmp );
  assert( !CORBA::is_nil( ext1 ) );
  ext1->sayHello();
  // Clean up
  tmp = 0L;
  ext1 = 0L;
  
  cout << "2 ---------------------------------------------------" << endl;

  /************
   * Lets add Extension2 and Extension4.
   * Extension4 needs the Wrapper which is not registered yet.
   ************/
  // Factories are no components => they dont have a destroy method
  // => No need to duplicate here like we did for "core" ( see above ).
  // The factory will die once the *_var variables go out of scope.
  KOM::AggregateFactory_var factory2 = new Extension2Factory;
  KOM::AggregateFactory_var factory4 = new Extension4Factory;
  KOM::InterfaceSeq prov;
  prov.length(1);
  prov[0] = CORBA::string_dup( "IDL:MyInterfaces/Hello:1.0" );
  KOM::InterfaceSeq seq;
  seq.length( 0 );
  // By default do NOT activate this aggregate
  core->addAggregate( factory2, seq, prov, false );
  // Activate this interface as soon as possible
  // Extension4 required Wrapper to work correctly
  seq.length(1);
  seq[0] = CORBA::string_dup( "IDL:MyInterfaces/Talk:1.0" );
  prov[0] = CORBA::string_dup( "IDL:MyInterfaces/NiceHello:1.0" );
  core->addAggregate( factory4, seq, prov, true );

  // Lets print all available interfaces
  KOM::InterfaceSeq_var s = core->interfaces();
  cout << "Got " << s->length() << " interfaces" << endl;
  for( CORBA::ULong l = 0; l < s->length(); ++l )
    cout << " Supports: " << (*s)[l].in() << endl;
  
  cout << "3 ---------------------------------------------------" << endl;

  /************
   * Ask for a certain interface. The "core" will have
   * to activate a dynamic aggregate to do so.
   ************/
  tmp = core->getInterface( "IDL:MyInterfaces/Hello:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  MyInterfaces::Hello_var hello = MyInterfaces::Hello::_narrow( tmp );
  assert( !CORBA::is_nil( hello ) );
  hello->sayHello();
  // Clean up
  tmp = 0L;
  hello = 0L;
  
  cout << "4 ---------------------------------------------------" << endl;

  /*************
   * Lets add the Wrapper interface. This should automatically
   * fire up the Extension4 and this in turn will start the Wrapper.
   *************/
  KOM::AggregateFactory_var factoryw = new WrapperFactory;
  seq.length( 0 );
  prov[0] = CORBA::string_dup( "IDL:MyInterfaces/Talk:1.0" );
  // By default do NOT activate this aggregate
  core->addAggregate( factoryw, seq, prov, false );

  // Lets print all available interfaces
  s = core->interfaces();
  cout << "Got " << s->length() << " interfaces" << endl;
  for( CORBA::ULong l = 0; l < s->length(); ++l )
    cout << " Supports: " << (*s)[l].in() << endl;

  // Now ask for extension 4.
  // It is already running ...
  tmp = core->getInterface( "IDL:MyInterfaces/NiceHello:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  MyInterfaces::NiceHello_var hello2 = MyInterfaces::NiceHello::_narrow( tmp );
  assert( !CORBA::is_nil( hello2 ) );
  hello2->sayNiceHello();
  // Clean up
  tmp = 0L;
  hello2 = 0L;
  
  cout << "5 ---------------------------------------------------" << endl;

  /************
   * Register two plugins. Activate the first, but not the
   * second one. The second one has a dependecy that
   * can not yet be resolved => this plugin is not
   * yet composed.
   ************/
  KOM::PluginFactory_var factoryp1 = new Plugin1Factory;
  KOM::PluginFactory_var factoryp2 = new Plugin2Factory;
  KOM::InterfaceSeq seq2;
  seq2.length( 0 );
  seq.length( 0 );
  prov[0] = CORBA::string_dup( "IDL:MyExtension/Plugin1:1.0" );
  core->addPlugin( factoryp1, seq2, seq, prov, true );
  // Plugin2 requires some interfaces, lets mention them
  seq.length( 1 );
  seq[0] = CORBA::string_dup( "IDL:MyExtension/PluginWrapper:1.0" );
  prov[0] = CORBA::string_dup( "IDL:MyExtension/Plugin2:1.0" );
  core->addPlugin( factoryp2, seq2, seq, prov, false );

  // Print all available plugins
  KOM::Component::PluginInfoSeq_var ps = core->describePlugins();
  for( CORBA::ULong l = 0; l < ps->length(); ++l )
  {
    cout << "Plugin " << (*ps)[l].id << endl;
    cout << " Interfaces:" << endl;
    for( CORBA::ULong k = 0; k < (*ps)[l].interfaces.length(); ++k )
      cout << "    " << (*ps)[l].interfaces[k].in() << endl;
    cout << " Composed: " << ( (*ps)[l].composed ? "Yes" : "No" ) << endl;
  }
  
  cout << "6 ---------------------------------------------------" << endl;

  /************
   * Lets register the wrapper plugin. Now Plugin2 is automatically
   * composed.
   ************/
  KOM::PluginFactory_var factorypw = new PluginWrapperFactory;
  seq.length( 0 );
  prov[0] = CORBA::string_dup( "IDL:MyExtension/PluginWrapper:1.0" );
  core->addPlugin( factorypw, seq2, seq, prov, false );

  // Print all available plugins
  ps = core->describePlugins();
  for( CORBA::ULong l = 0; l < ps->length(); ++l )
  {
    cout << "Plugin " << (*ps)[l].id << endl;
    cout << " Interfaces:" << endl;
    for( CORBA::ULong k = 0; k < (*ps)[l].interfaces.length(); ++k )
      cout << "    " << (*ps)[l].interfaces[k].in() << endl;
    cout << " Composed: " << ( (*ps)[l].composed ? "Yes" : "No" ) << endl;
  }

  cout << "7 ---------------------------------------------------" << endl;

  /***********
   * Print all interfaces supported by the plugins
   * and get us one.
   ***********/
  s = core->pluginInterfaces();
  cout << "Got " << s->length() << " interfaces" << endl;
  for( CORBA::ULong l = 0; l < s->length(); ++l )
    cout << " Supports: " << (*s)[l].in() << endl;

  // Lets get the reference to Plugin2
  // This will create the plugin and force the creation of Wrapper.
  tmp = core->getPluginInterface( "IDL:MyExtension/Plugin2:1.0" );
  assert( !CORBA::is_nil( tmp ) );
  MyExtension::Plugin2_var p2 = MyExtension::Plugin2::_narrow( tmp );
  assert( !CORBA::is_nil( p2 ) );
  p2->sayHello();
  // Clean up
  p2 = 0L;
  tmp = 0L;
  
  cout << "8 ---------------------------------------------------" << endl;

  /***********
   * Delete the component
   ***********/  
  core->destroy();
  cout << "9 ---------------------------------------------------" << endl;
  core = 0L;
  cout << "10 ---------------------------------------------------" << endl;

  return 0;
}
