/* 
 * $Id: bdm.c,v 1.5 1998/07/12 12:16:08 pisa Exp $
 *
 * Linux Device Driver BDM Interface
 * based on the PD driver package by Scott Howard, Feb 93
 * PD version ported to Linux by M.Schraut, Feb 95
 * enhancements from W. Eric Norum (eric@skatter.usask.ca), who did 
 *   a Next-Step-Port in Jun 95
 * tested for kernel version 1.3.57
 * (C) 1995, 1996 Technische Universitaet Muenchen, 
 * Lehrstuhl fuer Prozessrechner
 *
 * Modified by Pavel Pisa  pisa@cmp.felk.cvut.cz 1997 - 1998
 * tested for kernel version 2.1.78 with IDC and PD cables
 * new k_compat.h for kernels up to 2.1.107
 

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

 */

#undef REALLY_SLOW_IO

#include <linux/config.h>
#include <linux/module.h>
#include <linux/modversions.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/sys.h>
#include <linux/fs.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/io.h>
#include "bdm.h"
#include "k_compat.h"

#if 0
  #undef  outb
  #define outb outb_p
  #undef  inb
  #define inb inb_p
#endif

#define	BDM_DRIVER_VERSION	2
/*
 * Parallel port bit assignments for the PD interface (minor-numbers 0-2)
 *
 * Status register (bits 0-2 not used):
 * +---+---+---+---+---+---+---+---+
 * | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 * +---+---+---+---+---+---+---+---+
 *   |   |   |   |   |
 *   |   |   |   |   +--- Target FREEZE line
 *   |   |   |   |           1 - Target is in background mode
 *   |   |   |   |           0 - Target is not background mode
 *   |   |   |   |
 *   |   |   |   +------- Not used
 *   |   |   |
 *   |   |   +----------- Serial data from target
 *   |   |                    1 - `0' from target
 *   |   |                    0 - `1' from target
 *   |   |
 *   |   +--------------- Target power
 *   |                        1 - Target power is ON
 *   |                        0 - Target power is OFF
 *   |
 *   +------------------- Target connected
 *                            1 - Target is connected
 *                            0 - Target is not connected
 *
 * Control register (bits 4-7 not used):
 * +---+---+---+---+
 * | 3 | 2 | 1 | 0 |
 * +---+---+---+---+
 *   |   |   |   |
 *   |   |   |   +--- Target BKPT* /DSCLK line
 *   |   |   |           Write 1 - Drive BKPT* /DSCLK line LOW
 *   |   |   |           Write 0 - Drive BKPT* /DSCLK line HIGH
 *   |   |   |
 *   |   |   +------- Target RESET* line
 *   |   |               Write 1 - Force RESET* LOW
 *   |   |               Write 0 - Allow monitoring of RESET*
 *   |   |                         Read 1 - RESET* is LOW
 *   |   |                         Read 0 - RESET* is HIGH
 *   |   |
 *   |   +----------- Serial data to target
 *   |                    Write 0 - Send `0' to target
 *   |                    Write 1 - Send `1' to target
 *   |
 *   +--------------- Control single-step flip-flop
 *                        Write 1 - Clear flip-flop
 *                                  BKPT* /DSCLK is controlled by bit 0.
 *                        Write 0 - Allow flip-flop operation
 *                                  Falling edge of IFETCH* /DSI clocks a `1'
 *                                  into the flip-flop and drive BKPT* /DSCLK
 *                                  LOW, causing a breakpoint.
 */

/*
 * Parallel port bit assignments for the ICD interface (minor-numbers 4-6)
 *
 * Status register 
 * +---+---+---+---+---+---+---+---+
 * | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 * +---+---+---+---+---+---+---+---+
 *   |   |   |   |   |
 *   |   |   |   |   +--- Not used
 *   |   |   |   +------- Not used
 *   |   |   +----------- Not used
 *   |   |
 *   |   +--------------- Target FREEZE line
 *   |                        1 - Target is in background mode
 *   |                        0 - Target is not background mode
 *   |
 *   +------------------- Serial data from target
 *                            1 - `0' from target
 *                            0 - `1' from target
 *        
 * Data register 
 * +---+---+---+---+---+---+---+---+
 * | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 * +---+---+---+---+---+---+---+---+
 *   |   |   |   |   |   |   |   |
 *   |   |   |   |   |   |   |   +---  Serial data to target
 *   |   |   |   |   |   |   |            Write 1: Send 1 to target
 *   |   |   |   |   |   |   |            Write 0: Send 0 to target
 *   |   |   |   |   |   |   |            Signal gets to target, if OE is 1
 *   |   |   |   |   |   |   |            and target is in FREEZE mode
 *   |   |   |   |   |   |   |
 *   |   |   |   |   |   |   +-------  Clock 
 *   |   |   |   |   |   |                if target in freeze mode, then:
 *   |   |   |   |   |   |                Write 1: drive BKPT* /DSCLK 1
 *   |   |   |   |   |   |                Write 0: drive BKPT* /DSCLK 0
 *   |   |   |   |   |   |
 *   |   |   |   |   |   +-----------  BREAK
 *   |   |   |   |   |                    if target not in freeze mode, then:
 *   |   |   |   |   |                    Write 0: drive BKPT* /DSCLK 0
 *   |   |   |   |   |                    line determines single stepping
 *   |   |   |   |   |                    on leaving BGND mode:
 *   |   |   |   |   |                    Write 0: do single step
 *   |   |   |   |   |                    Write 1: continue normally
 *   |   |   |   |   |
 *   |   |   |   |   +---------------  RESET
 *   |   |   |   |                        Write 0: pull reset low
 *   |   |   |   |                        Write 1: release reset line
 *   |   |   |   |
 *   |   |   |   +--- OE
 *   |   |   |           Write 0 - DSI is tristated
 *   |   |   |           Write 1 - DSI pin is forced to level of serial data
 *   |   |   |
 *   |   |   +------- LED
 *   |   |               Write 1 - turn on LED
 *   |   |               Write 0 - turn off LED
 *   |   |
 *   |   +----------- ERROR
 *   |                   Write 0 - BERR output is tristated
 *   |                   Write 1 - BERR is pulled low
 *   |
 *   +--------------- spare
 */

/*
  #if !defined MODVERSIONS
  #undef MOD_INC_USE_COUNT
  #define MOD_INC_USE_COUNT
  #undef MOD_DEC_USE_COUNT
  #define MOD_DEC_USE_COUNT
  #endif
 */

#if !defined PD_INTERFACE && !defined IDC_INTERFACE
#error	You have to decide for the PD interface or the IDC interface
#error	Check the Makefile and uncomment the right line/s
#endif

#define BDM_WAITCNT    0xffff	/* no of loops to wait for response   */
#define BITS           15		/* shift 16bit into 32 bit by 15      */
								/* -.1.Bit always 0 (Statusbit)       */

/* declaration of local variables */
#if LP_PORT_RESTORE
static int bdm_old_ctl;			/* save old control port settings     */
#endif
static int bdm_speed = 30;		/* delay for fast CPU like 486        */
static unsigned int bdm_debug_level = BDM_DEBUG_NONE;
static int bdm_version = BDM_DRIVER_VERSION;
static int bdm_sense = 1;		/* do sense power & connected */

/* different debug levels             */

static void bdm_delay(int counter);

static void bdm_error(int bdm_type);

/*************************************************************************/

typedef unsigned int status_bitset;

struct bdm_driver_t {
	int		bd_port_adr;
	status_bitset	(*bd_init)(unsigned int minor);
	int 		(*bd_deinit)(unsigned int minor);
	status_bitset	(*bd_getstatus)(unsigned int minor);
	int             (*bd_restart)(unsigned int minor);
	int		(*bd_stop)(unsigned int minor);
	int		(*bd_step)(unsigned int minor);
	int		(*bd_release)(unsigned int minor);
	int		(*bd_reset)(unsigned int minor);
	long		(*bd_ser_clk)(unsigned int minor, 
				short bdm_value, int delay_bits);
	};

struct bdm_driver_t *bdm_drivers[];

#define	BDM_DATA(minor)	(bdm_drivers[minor]->bd_port_adr)
#define	BDM_STAT(minor)	(bdm_drivers[minor]->bd_port_adr+1)
#define	BDM_CTRL(minor)	(bdm_drivers[minor]->bd_port_adr+2)

#define bdm_init_hardware(minor) (bdm_drivers[minor]->bd_init(minor))
#define bdm_deinit(minor)	(bdm_drivers[minor]->bd_deinit(minor))
#define bdm_getstatus(minor)	(bdm_drivers[minor]->bd_getstatus(minor))
#define bdm_restart_chip(minor) (bdm_drivers[minor]->bd_restart(minor))
#define bdm_stop_chip(minor)	(bdm_drivers[minor]->bd_stop(minor))
#define bdm_step_chip(minor)	(bdm_drivers[minor]->bd_step(minor))
#define bdm_release_chip(minor)	(bdm_drivers[minor]->bd_release(minor))
#define bdm_reset_chip(minor)	(bdm_drivers[minor]->bd_reset(minor))
#define bdm_ser_clk(minor,bdm_value,delay_bits)	\
		(bdm_drivers[minor]->bd_ser_clk(minor,bdm_value,delay_bits))


/*************************************************************************/
/* PD interface */

#ifdef PD_INTERFACE

#define BDMPD_DSCLK      1		/* data shift clock / breakpoint pin  */
#define BDMPD_RST_OUT    2		/* set high to force reset on MCU     */
#define BDMPD_DSI        4		/* data shift input  Host->MCU        */
#define BDMPD_STEP_OUT   8		/* set low to gate IFETCH onto BKPT   */

#define BDMPD_FREEZE     8		/* FREEZE asserted when MCU stopped   */
#define BDMPD_DSO        0x20	/* data shift output MCU-.Host        */
#define BDMPD_PWR_DWN    0x40	/* power down - low when Vcc failed   */
#define BDMPD_NC         0x80	/* not connected - low when unplugged */

static status_bitset
bdm_pd_init(unsigned int minor)
{
	status_bitset status;

	outb(BDMPD_STEP_OUT | BDMPD_DSCLK, BDM_CTRL(minor));	

	status = bdm_getstatus(minor);		/* eg connected, power etc. */
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_init_hardware status %d, port: %#x\n", 
				status, BDM_DATA(minor));
	}
	return status;
}


static int
bdm_pd_deinit(unsigned int minor)
{
#if LP_PORT_RESTORE
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_deinit: restoring %#x\n", bdm_old_ctl);
	}
	outb(bdm_old_ctl, BDM_CTRL(minor));	/* restore old control port settings */
#endif /* LP_PORT_RESTORE */
	return BDM_NORETURN;
}


/* read status byte from statusport and translate into bitset */
static status_bitset
bdm_pd_getstatus(unsigned int minor)
{
	unsigned char temp = inb(BDM_STAT(minor));

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_getstatus minor=%d res=%#x\n", minor, temp);	 
	}
	if (bdm_sense) {
		if (!(temp & BDMPD_NC))
			return BDM_TARGETNC;
		if (!(temp & BDMPD_PWR_DWN))
			return BDM_TARGETPOWER;
	}
	return (temp & BDMPD_FREEZE ? BDM_TARGETSTOPPED : 0) |
	(inb(BDM_CTRL(minor)) & BDMPD_RST_OUT ? BDM_TARGETRESET : 0);
}

/* this routine is last resort one, it is able resyncronize with MCU,
   when some spike on DSCLK has disturbed MCU */
static status_bitset
bdm_pd_resyncro(unsigned int minor)
{
	int i;
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	i=17;	/* maximal number disturb in bit shift */
	while(i--)
	{
		if((inb(BDM_STAT(minor)) & BDMPD_DSO))
			return BDM_NORETURN; /* DSO becomes 0 => OK */
		outb( BDMPD_STEP_OUT | BDMPD_DSCLK, BDM_CTRL(minor)); /* DSCLK=0 */
		bdm_delay(bdm_speed);
		outb( BDMPD_STEP_OUT , BDM_CTRL(minor));	/* DSCLK=1 */
		bdm_delay(bdm_speed);
	};
	return BDM_FAULT_UNKNOWN;
};

/* restart chip and stop on first execution fetch */
static int
bdm_pd_restart(unsigned int minor)
{
	unsigned bdm_Loop_Count = 0;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_restart_chip minor=%d\n", minor);
	}

	outb(BDMPD_RST_OUT | BDMPD_DSCLK, BDM_CTRL(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMPD_DSCLK, BDM_CTRL(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMPD_STEP_OUT | BDMPD_DSCLK, BDM_CTRL(minor));
	for (bdm_Loop_Count = 0xffff; bdm_Loop_Count; bdm_Loop_Count--)
		if (inb(BDM_STAT(minor)) & BDMPD_FREEZE)
			break;
	if (!bdm_Loop_Count) {
		bdm_error(BDM_FAULT_RESPONSE);
		return BDM_FAULT_RESPONSE;
	} else {
		return BDM_NORETURN;
	}

}


/* stop running target */
static int
bdm_pd_stop(unsigned int minor)
{
	short bdm_ctr;
	int return_value;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_stop_chip minor=%d\n", minor);
	}

	if (inb(BDM_STAT(minor)) & BDMPD_FREEZE) {
		/* ensures no spike on BKPT/DSCLK in bdm_ser_clk */
		outb(BDMPD_DSCLK, BDM_CTRL(minor));
		bdm_delay(bdm_speed / 3 + 1);
		outb(BDMPD_DSCLK | BDMPD_STEP_OUT, BDM_CTRL(minor));
		return BDM_NORETURN;	/* target was already halted */
	};

	return_value = BDM_FAULT_UNKNOWN;

    /*	outb(BDMPD_DSCLK | BDMPD_STEP_OUT, BDM_CTRL(minor)); */
	outb(BDMPD_DSCLK*0, BDM_CTRL(minor));

	for (bdm_ctr = BDM_WAITCNT; bdm_ctr; bdm_ctr--) {
		if ((inb(BDM_STAT(minor))) & BDMPD_FREEZE) { /* FIXME */
			return_value = BDM_NORETURN;	/* target is now halted      */
			break;
		}
		bdm_delay(1);
	}

	outb(BDMPD_DSCLK, BDM_CTRL(minor)); 
	bdm_delay(bdm_speed + 1);
	outb(BDMPD_DSCLK | BDMPD_STEP_OUT, BDM_CTRL(minor));

	if (!bdm_ctr) {
		bdm_error(BDM_FAULT_RESPONSE);
		return BDM_FAULT_RESPONSE;
	}
    #ifdef BDM_TRY_RESYNCHRO
	bdm_pd_resyncro(minor); /* tries to find correct 17 bit boundary */
    #endif /* BDM_TRY_RESYNCHRO */
	return return_value;
}



/* single stepping target mcu */
static int
bdm_pd_step(unsigned int minor)
{
	unsigned long tester;
	unsigned char DataOut;
	status_bitset bdm_stat = bdm_getstatus(minor);

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_step_chip minor=%d\n", minor);
	}

	if (bdm_stat & BDM_TARGETRESET) {
		bdm_error(BDM_FAULT_RESET);
		return BDM_FAULT_RESET;
	}
	if (bdm_stat & BDM_TARGETNC) {
		bdm_error(BDM_FAULT_CABLE);
		return BDM_FAULT_CABLE;
	}
	if (bdm_stat & BDM_TARGETPOWER) {
		bdm_error(BDM_FAULT_POWER);
		return BDM_FAULT_POWER;
	}
	tester = bdm_ser_clk(minor, BDM_GO_CMD, 1);
	switch (tester) {
		case BDM_FAULT_RESET:
		case BDM_FAULT_CABLE:
		case BDM_FAULT_POWER:
			return tester;
		default:
	}
	if (bdm_debug_level >= BDM_DEBUG_ALL) {
		printk("stepchip sent: %x got : %lx\n", BDM_GO_CMD, tester);
		printk("last answerbit : %x\n", !(inb(BDM_STAT(minor)) & BDMPD_DSO));
		/* FIXME */
	}

	DataOut=(BDM_GO_CMD & 1 ? BDMPD_DSI : 0);
	outb(BDMPD_DSCLK | BDMPD_STEP_OUT | DataOut, BDM_CTRL(minor));
	bdm_delay(bdm_speed + 1);
	outb(BDMPD_DSCLK | DataOut, BDM_CTRL(minor));
	bdm_delay(bdm_speed + 1);
	outb(DataOut, BDM_CTRL(minor));
	/* FREEZE goes low after */
	/* 10 CPU32 clock cycles, it is about 700ns @ 16 MHz */
	bdm_delay(3*bdm_speed + 1);
    #ifdef BDM_TRY_RESYNCHRO
	bdm_stop_chip(minor);
	return bdm_pd_resyncro(minor);
    #else  /* BDM_TRY_RESYNCHRO */
	return bdm_stop_chip(minor);
    #endif /* BDM_TRY_RESYNCHRO */
}

/* reset target chip without asserting freeze */
static int
bdm_pd_release(unsigned int minor)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_release_chip minor=%d\n", minor);
	}

	outb(BDMPD_DSCLK | BDMPD_STEP_OUT, BDM_CTRL(minor));
	bdm_delay(10);
	outb(BDMPD_DSCLK | BDMPD_RST_OUT | BDMPD_STEP_OUT, BDM_CTRL(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMPD_STEP_OUT, BDM_CTRL(minor));
	bdm_delay(10);

	return BDM_NORETURN;
}

/*
 * Restart chip, enable background debugging mode, halt on first fetch
 *
 * The software from the Motorola BBS tries to have the target
 * chip begin execution, but that doesn't work very reliably.
 * The RESETH* line rises rather slowly, so sometimes the BKPT* / DSCLK
 * would be seen low, and sometimes it wouldn't.
 */
static int
bdm_pd_reset(unsigned int minor)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_reset_chip minor=%d\n", minor);
	}

	/* this is suggested by Eric Norum */
	outb(BDMPD_DSCLK | BDMPD_RST_OUT, BDM_CTRL(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMPD_DSCLK, BDM_CTRL(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMPD_DSCLK | BDMPD_STEP_OUT, BDM_CTRL(minor));

	return BDM_NORETURN;
}


/* serial software protokoll for bdm-pd-interface */
static long
bdm_pd_ser_clk(unsigned int minor, short bdm_value, int delay_bits)
{
	unsigned long ShiftRegister;
	unsigned char DataOut;
	unsigned counter;
	unsigned bdm_stat = bdm_getstatus(minor);

	ShiftRegister = bdm_value;
	ShiftRegister <<= BITS;
	if (bdm_stat & BDM_TARGETRESET) {	/*error checking */
		bdm_error(BDM_FAULT_RESET);
		return BDM_FAULT_RESET;
	}
	if (bdm_stat & BDM_TARGETNC) {
		bdm_error(BDM_FAULT_CABLE);
		return BDM_FAULT_CABLE;
	}
	if (bdm_stat & BDM_TARGETPOWER) {
		bdm_error(BDM_FAULT_POWER);
		return BDM_FAULT_POWER;
	}
	if(!(bdm_stat & BDM_TARGETSTOPPED))
	{ 	if (bdm_debug_level >= BDM_DEBUG_SOME)
			printk("bdm_ser_clk stop target first minor=%d\n", minor);
		if(bdm_stop_chip(minor)==BDM_FAULT_RESPONSE)
		{	printk("bdm_ser_clk can't stop it minor=%d\n", minor);
			return BDM_FAULT_RESPONSE;
		}
	}
	
	counter = 32 - BITS - delay_bits;
	while (counter--) {
		DataOut = ((ShiftRegister & 0x80000000) ? BDMPD_DSI : 0);
		/*if 1 then dsi=1*/

		if (bdm_debug_level >= BDM_DEBUG_ALL) {
			printk("%c", DataOut ? '1' : '0');
		}
		ShiftRegister <<= 1;	/*shift one bit left*/
		outb(DataOut | BDMPD_STEP_OUT | BDMPD_DSCLK, BDM_CTRL(minor));
		bdm_delay(bdm_speed / 2 + 1);
		if (!(inb(BDM_STAT(minor)) & BDMPD_DSO)) {
			ShiftRegister |= 1;	/*put 1 on 0.bit */
			if (bdm_debug_level >= BDM_DEBUG_ALL) {
				printk("1-");
			}
		} else {
			if (bdm_debug_level >= BDM_DEBUG_ALL) {
				printk("0-");
			}
		}
		outb(DataOut | BDMPD_STEP_OUT, BDM_CTRL(minor));
		bdm_delay(bdm_speed + 1);
	}
	return ShiftRegister;
}

#endif

/*************************************************************************/
/* IDC interface */

#ifdef IDC_INTERFACE

/* outputs */
#define BDMIDC_DSI        1		/* data shift input  Host->MCU        */
#define BDMIDC_DSCLK      2		/* data shift clock / breakpoint pin  */
#define BDMIDC_STEP_OUT   4		/* set low to force breakpoint        */
#define BDMIDC_RST_OUT    8		/* set low to force reset on MCU      */
#define BDMIDC_OE         0x10		/* set to a 1 to enable DSI           */
#define BDMIDC_LED        0x20		/* optional led indicator             */
#define BDMIDC_FORCE_BERR 0x40		/* set to a 1 to force BEER on target */
/* inputs */
#define BDMIDC_FREEZE     0x40		/* FREEZE asserted when MCU stopped   */
#define BDMIDC_DSO        0x80		/* data shift output - MCU->PC        */

static status_bitset
bdm_idc_init(unsigned int minor)
{
	status_bitset status;

	outb(BDMIDC_STEP_OUT | BDMIDC_DSCLK | BDMIDC_RST_OUT, BDM_DATA(minor));	

	status = bdm_getstatus(minor);		/* eg connected, power etc. */
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_init_hardware status %d, port: %#x\n", 
				status, BDM_DATA(minor));
	}
	return status;
}


static int
bdm_idc_deinit(unsigned int minor)
{
#if LP_PORT_RESTORE
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_deinit: restoring %#x\n", bdm_old_ctl);
	}
	outb(bdm_old_ctl, BDM_CTRL(minor));	/* restore old control port settings */
#endif /* LP_PORT_RESTORE */
	return BDM_NORETURN;
}


/* read status byte from statusport and translate into bitset */
static status_bitset
bdm_idc_getstatus(unsigned int minor)
{
	unsigned char temp = inb(BDM_STAT(minor));

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_getstatus minor=%d res=%#x\n", minor, temp);	 
	}
	return (temp & BDMIDC_FREEZE ? BDM_TARGETSTOPPED : 0);
}

/* this routine is last resort one, it is able resyncronize with MCU,
   when some spike on DSCLK has disturbed MCU */
static status_bitset
bdm_idc_resyncro(unsigned int minor)
{
	int i;
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	bdm_ser_clk(minor, BDM_NOP_CMD, 0); /* send NOP command */
	i=17;	/* maximal number disturb in bit shift */
	while(i--)
	{
		if((inb(BDM_STAT(minor)) & BDMIDC_DSO))
			return BDM_NORETURN; /* DSO becomes 0 => OK */
		outb(BDMIDC_RST_OUT | BDMIDC_OE | BDMIDC_STEP_OUT,
			BDM_DATA(minor));		/* DSCLK=0 */
		bdm_delay(bdm_speed);
		outb(BDMIDC_RST_OUT | BDMIDC_OE | BDMIDC_STEP_OUT |
			BDMIDC_DSCLK, BDM_DATA(minor));	/* DSCLK=1 */
		bdm_delay(bdm_speed);
	};
	return BDM_FAULT_UNKNOWN;
};

/* restart chip and stop on first execution fetch */
static int
bdm_idc_restart(unsigned int minor)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_restart_chip minor=%d\n", minor);
	}

   	outb(BDMIDC_DSCLK, BDM_DATA(minor));
	bdm_delay(BDM_WAITCNT);
	return bdm_stop_chip(minor);
}


/* stop running target */
static int
bdm_idc_stop(unsigned int minor)
{
	short bdm_ctr;
	int return_value;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_stop_chip minor=%d\n", minor);
	}

	return_value = BDM_FAULT_UNKNOWN;
	
#if 1
	if ((inb(BDM_STAT(minor))) & BDMIDC_FREEZE)
	{ /* already frozen, ensure no spikes on BKPT/DSCLK in bdm_ser_clk  */
		outb(BDMIDC_RST_OUT, BDM_DATA(minor));
		bdm_delay(bdm_speed / 3 + 1);
		outb(BDMIDC_STEP_OUT | BDMIDC_RST_OUT, BDM_DATA(minor));
		return BDM_NORETURN;
	};
#endif	

#if 1
	outb(BDMIDC_RST_OUT, BDM_DATA(minor));
#else
	outb(BDMIDC_DSCLK | BDMIDC_RST_OUT, BDM_DATA(minor));
#endif

	for (bdm_ctr = BDM_WAITCNT; bdm_ctr; bdm_ctr--) {
		if ((inb(BDM_STAT(minor))) & BDMIDC_FREEZE) {
			return_value = BDM_NORETURN;	/* target is now halted      */
			break;
		}
		bdm_delay(1);
	}

	if (!bdm_ctr) {
#if 1
		outb(BDMIDC_RST_OUT | BDMIDC_FORCE_BERR, BDM_DATA(minor));
#else
		outb(BDMIDC_DSCLK | BDMIDC_RST_OUT | BDMIDC_FORCE_BERR,
			BDM_DATA(minor));
#endif			
 		bdm_delay(1);
 		for (bdm_ctr = BDM_WAITCNT; bdm_ctr; bdm_ctr--) {
 			if ((inb(BDM_STAT(minor))) & BDMIDC_FREEZE) {
 				return_value = BDM_NORETURN;	/* target is now halted      */
 				break;
 			}
 			bdm_delay(1);
 		}
 	}

#if 1
	outb(BDMIDC_RST_OUT, BDM_DATA(minor));
	bdm_delay(bdm_speed + 1);
	outb(BDMIDC_RST_OUT | BDMIDC_STEP_OUT, BDM_DATA(minor));
#else
	outb(BDMIDC_DSCLK | BDMIDC_RST_OUT | BDMIDC_STEP_OUT, BDM_DATA(minor));
#endif

	if (!bdm_ctr) {
		bdm_error(BDM_FAULT_RESPONSE);
		return BDM_FAULT_RESPONSE;
	}
    #ifdef BDM_TRY_RESYNCHRO
	bdm_idc_resyncro(minor); /* tries to find correct 17 bit boundary */
    #endif /* BDM_TRY_RESYNCHRO */
	return return_value;
}


/* single stepping target mcu */
static int
bdm_idc_step(unsigned int minor)
{
	unsigned long tester;
	unsigned char DataOut;
	status_bitset bdm_stat = bdm_getstatus(minor);

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_step_chip minor=%d\n", minor);
	}

	if (bdm_stat & BDM_TARGETRESET) {
		bdm_error(BDM_FAULT_RESET);
		return BDM_FAULT_RESET;
	}
	if (bdm_stat & BDM_TARGETNC) {
		bdm_error(BDM_FAULT_CABLE);
		return BDM_FAULT_CABLE;
	}
	if (bdm_stat & BDM_TARGETPOWER) {
		bdm_error(BDM_FAULT_POWER);
		return BDM_FAULT_POWER;
	}
	tester = bdm_ser_clk(minor, BDM_GO_CMD, 1);
	switch (tester) {
		case BDM_FAULT_RESET:
		case BDM_FAULT_CABLE:
		case BDM_FAULT_POWER:
			return tester;
		default:
	}
	if (bdm_debug_level >= BDM_DEBUG_ALL) {
		printk("stepchip sent: %x got : %lx\n", BDM_GO_CMD, tester);
		printk("last answerbit : %x\n", (inb(BDM_STAT(minor))
						&BDMIDC_DSO)?0:1);
		/* FIXME */
	}
	DataOut=(BDM_GO_CMD & 1 ? BDMIDC_DSI : 0);
	outb(BDMIDC_OE | BDMIDC_STEP_OUT | DataOut | BDMIDC_RST_OUT,
		BDM_DATA(minor));
	bdm_delay(bdm_speed + 1);
	outb(BDMIDC_OE | DataOut | BDMIDC_RST_OUT, BDM_DATA(minor));
	outb(BDMIDC_DSCLK | BDMIDC_RST_OUT, BDM_DATA(minor));
    #ifdef BDM_TRY_RESYNCHRO
	bdm_stop_chip(minor);
	return bdm_idc_resyncro(minor);
    #else  /* BDM_TRY_RESYNCHRO */
	return bdm_stop_chip(minor);
    #endif /* BDM_TRY_RESYNCHRO */
}

/* reset target chip without asserting freeze */
static int
bdm_idc_release(unsigned int minor)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_release_chip minor=%d\n", minor);
	}

	outb(BDMIDC_DSCLK | BDMIDC_STEP_OUT, BDM_DATA(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMIDC_DSCLK | BDMIDC_RST_OUT | BDMIDC_STEP_OUT,
		BDM_DATA(minor));
	bdm_delay(10);

	return BDM_NORETURN;
}

/*
 * Restart chip, enable background debugging mode, halt on first fetch
 *
 * The software from the Motorola BBS tries to have the target
 * chip begin execution, but that doesn't work very reliably.
 * The RESETH* line rises rather slowly, so sometimes the BKPT* / DSCLK
 * would be seen low, and sometimes it wouldn't.
 */
static int
bdm_idc_reset(unsigned int minor)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_reset_chip minor=%d\n", minor);
	}

	/* this is according to Scott Howard */
        outb(BDMIDC_DSCLK | BDMIDC_STEP_OUT, BDM_DATA(minor));
	bdm_delay(BDM_WAITCNT);
	outb(BDMIDC_DSCLK | BDMIDC_STEP_OUT | BDMIDC_RST_OUT, BDM_DATA(minor));

	/*
	 *Tom Hoover's Error or Optimization? has to be checked
	 */
	/*
	outb(BDMIDC_DSCLK, BDM_DATA(minor));
	bdm_delay(BDM_WAITCNT);
	bdm_stop_chip(minor);
	 */

	return BDM_NORETURN;
}


/* serial software protokoll for bdm-pd-interface */
static long
bdm_idc_ser_clk(unsigned int minor, short bdm_value, int delay_bits)
{
	unsigned long ShiftRegister;
	unsigned char DataOut;
	unsigned counter;
	unsigned bdm_stat = bdm_getstatus(minor);

	ShiftRegister = bdm_value;
	ShiftRegister <<= BITS;
	if (bdm_stat & BDM_TARGETRESET) {	/*error checking */
		bdm_error(BDM_FAULT_RESET);
		return BDM_FAULT_RESET;
	}
	if (bdm_stat & BDM_TARGETNC) {
		bdm_error(BDM_FAULT_CABLE);
		return BDM_FAULT_CABLE;
	}
	if (bdm_stat & BDM_TARGETPOWER) {
		bdm_error(BDM_FAULT_POWER);
		return BDM_FAULT_POWER;
	}
	if(!(bdm_stat & BDM_TARGETSTOPPED))
	{ 	if (bdm_debug_level >= BDM_DEBUG_SOME)
			printk("bdm_ser_clk stop target first minor=%d\n", minor);
		if(bdm_stop_chip(minor)==BDM_FAULT_RESPONSE)
		{	printk("bdm_ser_clk can't stop it minor=%d\n", minor);
			return BDM_FAULT_RESPONSE;
		}
	}

	counter = 32 - BITS - delay_bits;
	while (counter--) {
		DataOut = ((ShiftRegister & 0x80000000) ? BDMIDC_DSI : 0);
		/*if 1 then DSI=1*/
		if (bdm_debug_level >= BDM_DEBUG_ALL) {
			printk("%c", DataOut ? '1' : '0');
		}

		outb(DataOut | BDMIDC_RST_OUT | BDMIDC_OE | BDMIDC_STEP_OUT | 
			BDMIDC_LED, BDM_DATA(minor));		/* DSCLK=0 */
		bdm_delay(bdm_speed + 1);

		ShiftRegister <<= 1;	/*shift one bit left*/
		if (!(inb(BDM_STAT(minor)) & BDMIDC_DSO)) {
			ShiftRegister |= 1;	/*put 1 on 0.bit */
			if (bdm_debug_level >= BDM_DEBUG_ALL) {
				printk("1-");
			}
		} else {
			if (bdm_debug_level >= BDM_DEBUG_ALL) {
				printk("0-");
			}
		}

		outb(DataOut | BDMIDC_RST_OUT | BDMIDC_OE | BDMIDC_STEP_OUT |
			BDMIDC_DSCLK | BDMIDC_LED, BDM_DATA(minor)); /* DSCLK=1 */
		bdm_delay((bdm_speed >> 1) + 1);
	}
	outb(BDMIDC_DSCLK | BDMIDC_STEP_OUT | BDMIDC_RST_OUT, BDM_DATA(minor));
	return ShiftRegister;
}

#endif

/*************************************************************************/

#ifdef PD_INTERFACE

static struct bdm_driver_t 
bdm_driver_0= { 0x378,bdm_pd_init,bdm_pd_deinit,bdm_pd_getstatus,
		bdm_pd_restart,bdm_pd_stop,bdm_pd_step,
		bdm_pd_release,bdm_pd_reset,bdm_pd_ser_clk};

static struct bdm_driver_t 
bdm_driver_1= { 0x278,bdm_pd_init,bdm_pd_deinit,bdm_pd_getstatus,
		bdm_pd_restart,bdm_pd_stop,bdm_pd_step,
		bdm_pd_release,bdm_pd_reset,bdm_pd_ser_clk};

static struct bdm_driver_t 
bdm_driver_2= { 0x3bc,bdm_pd_init,bdm_pd_deinit,bdm_pd_getstatus,
		bdm_pd_restart,bdm_pd_stop,bdm_pd_step,
		bdm_pd_release,bdm_pd_reset,bdm_pd_ser_clk};

#endif

#ifdef IDC_INTERFACE

static struct bdm_driver_t 
bdm_driver_4= { 0x378,bdm_idc_init,bdm_idc_deinit,bdm_idc_getstatus,
		bdm_idc_restart,bdm_idc_stop,bdm_idc_step,
		bdm_idc_release,bdm_idc_reset,bdm_idc_ser_clk};

static struct bdm_driver_t 
bdm_driver_5= { 0x278,bdm_idc_init,bdm_idc_deinit,bdm_idc_getstatus,
		bdm_idc_restart,bdm_idc_stop,bdm_idc_step,
		bdm_idc_release,bdm_idc_reset,bdm_idc_ser_clk};

static struct bdm_driver_t 
bdm_driver_6= { 0x3bc,bdm_idc_init,bdm_idc_deinit,bdm_idc_getstatus,
		bdm_idc_restart,bdm_idc_stop,bdm_idc_step,
		bdm_idc_release,bdm_idc_reset,bdm_idc_ser_clk};

#endif

struct bdm_driver_t *bdm_drivers[]={
    #ifdef PD_INTERFACE
	&bdm_driver_0,
	&bdm_driver_1,
	&bdm_driver_2,
	&bdm_driver_0	/* to shift ICD by 4 */
    #endif
    #ifdef IDC_INTERFACE
	#ifdef PD_INTERFACE
	,
	#endif
	&bdm_driver_4,
	&bdm_driver_5,
	&bdm_driver_6
    #endif
	};

/* number of defined hardware drivers */
const int bdm_drivers_num=sizeof(bdm_drivers)/sizeof(struct bdm_driver_t *);

/*************************************************************************/

/* slow down host for not overrunning target */
static void
bdm_delay(int counter)
{
	while (counter--) {
		asm volatile ("nop");
	}
}


static int
bdm_open(struct inode *inode, struct file *file)
{
	status_bitset bdm_okay;
	unsigned int minor = MINOR(inode->i_rdev);

	if (minor >= bdm_drivers_num) {
		return -ENODEV;
	}

	if (check_region(BDM_DATA(minor), 3)) {
		return -EBUSY;
	} else {
		request_region(BDM_DATA(minor), 3, "bdm");
	}

	MOD_INC_USE_COUNT;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_open\n");
	}

	bdm_okay = bdm_init_hardware(minor);

printk("bdm_open: init_hardware returns 0x%x\n", bdm_okay);
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		if (!(bdm_okay & BDM_TARGETNC)) {
			printk("BDM detected !!\n");
		} else {
			printk("Target not connected ! Error %d \n", bdm_okay);
		}
	}
#if xxNEVER
	/* we dont care on open... */
	if (bdm_okay & BDM_TARGETNC) {
		MOD_DEC_USE_COUNT;
		return BDM_FAULT_CABLE;
	} else if (bdm_okay & BDM_TARGETPOWER) {
		MOD_DEC_USE_COUNT;
		return BDM_FAULT_POWER;
	}
#endif /* NEVER */

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_open successful\n");
	}
printk("bdm_open returning %d\n", BDM_NORETURN);
	return BDM_NORETURN;
}


static CLOSERET 
bdm_close(struct inode *inode, struct file *file)
{
	unsigned int minor = MINOR(inode->i_rdev);
	bdm_deinit(minor);
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_release\n");
	}
	MOD_DEC_USE_COUNT;
	release_region(BDM_DATA(minor), 3);
        return (CLOSERET)0;
}

#define	do_case(x) case x: ret = #x; break;

static char * 
bdm_get_ioctl_name(int ioctl)
{
char *ret;
	switch (ioctl) {
		do_case(BDM_INIT);
		do_case(BDM_DEINIT);
		do_case(BDM_RESET_CHIP);
		do_case(BDM_RESTART_CHIP);
		do_case(BDM_STOP_CHIP);
		do_case(BDM_STEP_CHIP);
		do_case(BDM_GET_STATUS);
		do_case(BDM_SPEED);
		do_case(BDM_RELEASE_CHIP);
		do_case(BDM_DEBUG_LEVEL);
		do_case(BDM_GET_VERSION);
		do_case(BDM_SENSECABLE);
		default:
			ret = "Unknown ioctl";
	}
	return ret;
}

static int
bdm_ioctl(struct inode *inode, struct file *file, 
		unsigned int cmd, unsigned long arg)
{
	unsigned retval = BDM_NORETURN;
	status_bitset status; 
	unsigned int minor = MINOR(inode->i_rdev);

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_ioctl minor=%d cmd=%d (%s)  arg=0x%lx\n", 
			minor, cmd, bdm_get_ioctl_name(cmd), arg);
	}
	switch (cmd) {
	  case BDM_INIT:
		status = bdm_init_hardware(minor);
		if (status & BDM_TARGETNC) 
			retval = BDM_FAULT_CABLE;
		else if (status & BDM_TARGETPOWER) 
			retval = BDM_FAULT_POWER;
		break;
	  case BDM_DEINIT:
		retval = bdm_deinit(minor);
		break;
	  case BDM_RESET_CHIP:
		retval = bdm_reset_chip(minor);		
		/* hardware reset on MCU - running state */
		break;
	  case BDM_RESTART_CHIP:	
	  	/* reset target and stops execution on first instruction fetch */
		retval = bdm_restart_chip(minor);
		break;
	  case BDM_STOP_CHIP:		
	  	/* stop running target (FREEZE) */
		retval = bdm_stop_chip(minor);
		break;
	  case BDM_STEP_CHIP:
		/* one step on target */
		bdm_step_chip(minor);	
		retval = BDM_NORETURN;
		break;
	  case BDM_GET_STATUS:
		retval = bdm_getstatus(minor);
		break;
	  case BDM_SPEED:			
		/* Change speed value */
		bdm_speed = (int) arg;
		retval = BDM_NORETURN;
		break;
	  case BDM_RELEASE_CHIP:	
		/* reset without bdm, so quit bdm mode */
		retval = bdm_release_chip(minor);
		break;
	  case BDM_DEBUG_LEVEL:
		/* change debug level 0,1,2 */
		if (arg < 3) {
			bdm_debug_level = (unsigned int) arg;	
			retval = BDM_NORETURN;
		} else {
			retval = -EINVAL;
		}
		break;
	  case BDM_GET_VERSION:
		/* read counter and return it to *arg */
		if (verify_area(VERIFY_WRITE, (int *) arg, sizeof(int))) {
			retval = -EINVAL;
		} else {
			kc_put_user_long(bdm_version, arg);
		}
		break;
	  case BDM_SENSECABLE:
		bdm_sense = arg;
		printk("bdm_ioctl: setting bdm_sense to %d\n", bdm_sense);
	    break;
	  default:
		retval = -EINVAL;
	}
	return retval;
}



static RWRET
bdm_write(WRITE_PARAMETERS)
{
	unsigned int minor = MINOR(RWINODE->i_rdev);
	short bdm_word;
	long ret;
	int written = 0;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_write minor %d len %d\n", minor, (int) count);	 
	}
	if (verify_area(VERIFY_READ, buf, count))
		return -EINVAL;

	while (count > 0) {
		bdm_word = kc_get_user_word(buf);
		ret = bdm_ser_clk(minor, bdm_word, 0);
		if (bdm_debug_level >= BDM_DEBUG_ALL) {
			printk("\n write gesendet:  %x bekommen: %lx\n", bdm_word, ret);
		}
		if (written) {
			switch (ret) {
				case 0x1ffff:	/* no valid command return */
					return BDM_FAULT_NVC;
				case 0x10001:	/* Buserror */
					return BDM_FAULT_BERR;
				case BDM_FAULT_RESET:
				case BDM_FAULT_CABLE:
				case BDM_FAULT_POWER:
					return ret;
				default:
			}
		}
		count -= 2;
		buf += 2;
		written += 2;
		if (!(written % 64)) {
			schedule();
		}
	}
	return written;
}

static RWRET
bdm_read(READ_PARAMETERS)
{
	unsigned int minor = MINOR(RWINODE->i_rdev);
	short bdm_word = 0x0, wert;
	long ret;
	int read = 0, timeout = 0;

	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		printk("bdm_read minor=%d len=%d\n", minor, (int) count);	 
	}
	if (verify_area(VERIFY_WRITE, buf, count))
		return -EINVAL;

	while (count > 0) {
		ret = bdm_ser_clk(minor, bdm_word, 0);

		if (bdm_debug_level >= BDM_DEBUG_ALL) {
			printk("\n read gesendet:  %x bekommen: %lx\n", bdm_word, ret);
		}
		switch (ret) {
			case 0x10001:	/* test if BERR*/
				return BDM_FAULT_BERR;
			case 0x1ffff:	/* no valid command return */
				return BDM_FAULT_NVC;
			case 0x10000:
				timeout++;
				if (timeout - read / 2 == 4)
					return BDM_FAULT_RESPONSE;
			case BDM_FAULT_RESET:
			case BDM_FAULT_CABLE:
			case BDM_FAULT_POWER:
				return ret;
			default:
				wert = ret;
				kc_put_user_word(wert, buf);
				buf += 2;
				count -= 2;
				read += 2;
		}

	}
	return read;
}

static struct file_operations bdm_fops =
{
	NULL,						/* lseek               */
	bdm_read,					/* read                */
	bdm_write,					/* write               */
	NULL,						/* readdir             */
	NULL,						/* select              */
	bdm_ioctl,					/* ioctl               */
	NULL,						/* mmap                */
	bdm_open,					/* open                */
	bdm_close,					/* close               */
	NULL,						/* fsync               */
	NULL,						/* fasync              */
	NULL,						/* check_media_change  */
	NULL						/* revalidate          */
};

static void
bdm_error(int bdm_type)
{
	if (bdm_debug_level >= BDM_DEBUG_SOME) {
		switch (bdm_type) {
		  case BDM_FAULT_UNKNOWN:
			  printk("Unknown Error - check speed\n");
			  break;
		  case BDM_FAULT_POWER:
			  printk("Power failed on Target MCU\n");
			  break;
		  case BDM_FAULT_CABLE:
			  printk("Cable disconnected on Target MCU\n");
			  break;
		  case BDM_FAULT_RESPONSE:
			  printk("No response from Target MCU\n");
			  break;
		  case BDM_FAULT_RESET:
			  printk("Can t clock Target MCU while on Reset\n");
			  break;
		  case BDM_FAULT_PORT:
			  printk("Wrong Port\n");
			  break;
		  case BDM_FAULT_BERR:
			  printk("Buserror\n");
			  break;
		}
	}
}

int
init_module(void)
{

	printk("BDM init_module\n   %s\n   %s\n   %s\n",
		   "$RCSfile: bdm.c,v $", "$Revision: 1.5 $", "$Date: 1998/07/12 12:16:08 $");
		   /*"$Id: bdm.c,v 1.21 1996/07/21 12:16:08 magin Exp $", */
	printk("   Version %s\n   Compiled at %s %s\n",
#ifdef PD_INTERFACE
		   " PD-interface"
#endif
#ifdef IDC_INTERFACE
		   " ICD-interface with P&E-Cable"
#endif
		   ,__DATE__, __TIME__);
	if (register_chrdev(BDM_MAJOR_NUMBER, "bdm", &bdm_fops)) {
		printk("Unable to get major number for BDM driver\n");
		return -EIO;
	}
	printk("BDM driver succesfully registered major %d minors %d!!\n",
	         BDM_MAJOR_NUMBER,bdm_drivers_num);
	return 0;
}

void
cleanup_module(void)
{
	if (unregister_chrdev(BDM_MAJOR_NUMBER, "bdm") != 0)
		printk("BDM cleanup: Unregister failed\n");
	else
		printk("BDM cleanup: Unregister  O.K.\n");
}
