// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: MmFlat.cc
//   Flat data.
//   This is a concrete class.
//
// File Created:	19 Sep 1994		Michael Chastain
// Last Edited:		24 May 1995		Michael Chastain

#include <MmFlat.h>
#include <WhAbort.h>



// Constructor.
MmFlat::MmFlat( )
    : strFileName_	( "" )
    , icDataPos_	( 0  )
    , lcData_		(    )
{
    ;
}



// Read from a file.
MmRet MmFlat::fromFile( WhFile & fileFrom )
{
    strFileName_ = fileFrom.getName( );
    icDataPos_ = 0;
    MmRetCheck( fileFrom.readListChar( lcData_ ) );
    return mmRetOk;
}



// Write to a file.
MmRet MmFlat::toFile( WhFile & fileTo ) const
{
    return fileTo.writeListChar( lcData_ );
}



// Flat input.
MmRet MmFlat::fromFlat( MmFlat & flatFrom )
{
    MmRetCheck( flatFrom.getString   ( strFileName_ ) );
    icDataPos_ = 0;
    int icDataPos;
    MmRetCheck( flatFrom.getInt      ( icDataPos    ) );
    if ( icDataPos < 0 )
	return MmRetRaise( mmRetFlatIndexNegative );
    icDataPos_ = icDataPos;
    MmRetCheck( flatFrom.getListChar ( lcData_      ) );
    return mmRetOk;
}



// Flat output.
void MmFlat::toFlat( MmFlat & flatTo ) const
{
    flatTo.putString   ( strFileName_ );
    flatTo.putInt      ( icDataPos_   );
    flatTo.putListChar ( lcData_      );
}



// Get an address.
MmRet MmFlat::getAddr( MmAddr & addrGet )
{
    while ( icDataPos_ % sizeof(MmAddr) != 0 )
	++icDataPos_;
    if ( icDataPos_ + sizeof(MmAddr) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );
    addrGet     = * (const MmAddr *) (lcData_.address( ) + icDataPos_);
    icDataPos_ += sizeof(MmAddr);
    return mmRetOk;
}



// Get a boolean.
MmRet MmFlat::getBool( bool & fGet )
{
    while ( icDataPos_ % sizeof(char) != 0 )
	++icDataPos_;
    if ( icDataPos_ + sizeof(char) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );
    switch ( * (const char *) (lcData_.address( ) + icDataPos_) )
    {
    default: return MmRetRaise( mmRetFlatBoolBad );
    case 0:  fGet = false; break;
    case 1:  fGet = true;  break;
    }
    icDataPos_ += sizeof(char);

    return mmRetOk;
}



// Get an integer.
MmRet MmFlat::getInt( int & iGet )
{
    while ( icDataPos_ % sizeof(int) != 0 )
	++icDataPos_;
    if ( icDataPos_ + sizeof(int) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );
    iGet        = * (const int *) (lcData_.address( ) + icDataPos_);
    icDataPos_ += sizeof(int);
    return mmRetOk;
}



// Get a list of characters.
MmRet MmFlat::getListChar( WhList <char> & lcGet )
{
    // Get # of chars.
    int ncGet;
    MmRetCheck( getInt( ncGet ) );
    if ( ncGet < 0 )
	return MmRetRaise( mmRetFlatCountNegative );
    while ( icDataPos_ % sizeof(char) != 0 )
	++icDataPos_;
    if ( icDataPos_ + ncGet * sizeof(char) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );

    // Get chars.
    lcGet.clear( ncGet );
    lcGet.append( (const char *) (lcData_.address( ) + icDataPos_), ncGet );
    icDataPos_ += ncGet * sizeof(char);
    return mmRetOk;
}



// Get a list of words.
MmRet MmFlat::getListWord( WhList <MmWord> & lwGet )
{
    // Get # of words.
    int nwGet;
    MmRetCheck( getInt( nwGet ) );
    if ( nwGet < 0 )
	return MmRetRaise( mmRetFlatCountNegative );
    while ( icDataPos_ % sizeof(MmWord) != 0 )
	++icDataPos_;
    if ( icDataPos_ + nwGet * sizeof(MmWord) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );

    // Get words.
    lwGet.clear( nwGet );
    lwGet.append( (const MmWord *) (lcData_.address( ) + icDataPos_), nwGet );
    icDataPos_ += nwGet * sizeof(MmWord);
    return mmRetOk;
}



// Get a long.
MmRet MmFlat::getLong( long & lGet )
{
    while ( icDataPos_ % sizeof(long) != 0 )
	++icDataPos_;
    if ( icDataPos_ + sizeof(long) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );
    lGet        = * (const long *) (lcData_.address( ) + icDataPos_);
    icDataPos_ += sizeof(long);
    return mmRetOk;
}



// Get a string.
MmRet MmFlat::getString( WhString & strGet )
{
    // Get # of chars.
    int ncGet;
    MmRetCheck( getInt( ncGet ) );
    if ( ncGet < 0 )
	return MmRetRaise( mmRetFlatCountNegative );
    while ( icDataPos_ % sizeof(char) != 0 )
	++icDataPos_;
    if ( icDataPos_ + ncGet * sizeof(char) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );

    // Get chars.
    strGet.clear( );
    for ( int icGet = 0; icGet < ncGet; ++icGet )
    {
	strGet.appChrRaw(
	    ((const char *) (lcData_.address( ) + icDataPos_))[icGet] );
    }
    icDataPos_ += ncGet * sizeof(char);
    return mmRetOk;
}



// Get a word.
MmRet MmFlat::getWord( MmWord & wGet )
{
    while ( icDataPos_ % sizeof(MmWord) != 0 )
	++icDataPos_;
    if ( icDataPos_ + sizeof(MmWord) > lcData_.count( ) )
	return MmRetRaise( mmRetFlatEnd );
    wGet        = * (const MmWord *) (lcData_.address( ) + icDataPos_);
    icDataPos_ += sizeof(MmWord);
    return mmRetOk;
}



// Put an address.
void MmFlat::putAddr( MmAddr addrPut )
{
    while ( lcData_.count( ) % sizeof(MmAddr) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) &addrPut, sizeof(MmAddr) );
}



// Put a bool.
void MmFlat::putBool( bool fPut )
{
    while ( lcData_.count( ) % sizeof(char) != 0 )
	lcData_.append( '\0' );
    const char cPut = fPut ? 1 : 0;
    lcData_.append( (const char *) &cPut, sizeof(char) );
}



// Put an integer.
void MmFlat::putInt( int iPut )
{
    while ( lcData_.count( ) % sizeof(int) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) &iPut, sizeof(int) );
}



// Put a list of characters.
void MmFlat::putListChar( const WhList <char> & lcPut )
{
    putInt( lcPut.count( ) );
    while ( lcData_.count( ) % sizeof(char) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) lcPut.address( ),
	lcPut.count( ) * sizeof(char) );
}



// Put a list of words.
void MmFlat::putListWord( const WhList <MmWord> & lwPut )
{
    putInt( lwPut.count( ) );
    while ( lcData_.count( ) % sizeof(MmWord) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) lwPut.address( ),
	lwPut.count( ) * sizeof(MmWord) );
}



// Put a long.
void MmFlat::putLong( long lPut )
{
    while ( lcData_.count( ) % sizeof(long) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) &lPut, sizeof(long) );
}



// Put a string.
void MmFlat::putString( const WhString & strPut )
{
    putInt( strPut.count( ) );
    while ( lcData_.count( ) % sizeof(char) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) strPut.address( ),
	strPut.count( ) * sizeof(char) );
}



// Put a word.
void MmFlat::putWord( MmWord wPut )
{
    while ( lcData_.count( ) % sizeof(MmWord) != 0 )
	lcData_.append( '\0' );
    lcData_.append( (const char *) &wPut, sizeof(MmWord) );
}
