// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: MmMap.cc
//   Address map class.
//   This is a concrete class.
//
// File Created:	24 Apr 1995		Michael Chastain
// Last Edited:		11 Sep 1995		Michael Chastain

#include <MmMap.h>
#include <MmSeg.h>
#include <MmType.h>
#include <PrProc.h>
#include <TySegEnumLix.h>
#include <WhAbort.h>



// Constructor.
MmMap::MmMap( )
    : larea_	( )
{
    MmArea areaBlank;
    areaBlank.setCover( MmArea::tyAreaBlank );
    larea_.append( areaBlank );
}



// Get current 'brk' address.
MmAddr MmMap::getAddrBrk( ) const
{
    MmAddr addrBrk = 0;
    for ( int iArea = 0; iArea < count( ); ++iArea )
    {
	const MmArea & area = getArea( iArea );
	switch ( area.getTyArea( ) )
	{
	default:
	    break;
	case MmArea::tyAreaText:
	case MmArea::tyAreaData:
	case MmArea::tyAreaBss:
	case MmArea::tyAreaBrk:
	    addrBrk = area.getAddrMax( );
	    break;
	}
    }
#if 0
    ::printf( "MmMap::getAddrBrk: %p\n", addrBrk );
#endif
    return addrBrk;
}



// Get maximum stack pointer.
MmAddr MmMap::getAddrSpMax( ) const
{
    return mmAddrSpMax;
}



// Merge an area.
void MmMap::mergeArea( const MmArea & areaNew )
{
    // Verify bounds are inclusive.
    if ( areaNew.getAddrMin( ) < larea_[0].getAddrMin( )
    ||   areaNew.getAddrMax( ) > larea_[larea_.count( )-1].getAddrMax( ) )
	WhAbort( "MmMap::mergeArea: area out of range." );

    // Iterate over existing areas, make new ones.
    WhList <MmArea> lareaMerge;
    for ( int iareaOld = 0; iareaOld < larea_.count( ); ++iareaOld )
    {
	// Get old area.
	const MmArea & areaOld = larea_[iareaOld];
	
	// Get scalars for short names.
	const MmArea::TyArea tyAreaOld  = areaOld.getTyArea  ( );
	const MmAddr         addrOldMin = areaOld.getAddrMin ( );
	const MmAddr         addrOldMax = areaOld.getAddrMax ( );
	const MmArea::TyArea tyAreaNew  = areaNew.getTyArea  ( );
	const MmAddr         addrNewMin = areaNew.getAddrMin ( );
	const MmAddr         addrNewMax = areaNew.getAddrMax ( );

	// 1-3 fragments.  4!/(2*2) == 6 cases for endpoints to fall.
	WhList <MmArea> larea3;
	if ( addrOldMin <= addrNewMin )
	{
	    if      ( addrOldMax <= addrNewMin )
	    {
		larea3.append( areaOld );
	    }
	    else if ( addrOldMax <= addrNewMax )
	    {
		larea3.append( MmArea( tyAreaOld, addrOldMin, addrNewMin ) );
		larea3.append( MmArea( tyAreaNew, addrNewMin, addrOldMax ) );
	    }
	    else
	    {
		larea3.append( MmArea( tyAreaOld, addrOldMin, addrNewMin ) );
		larea3.append( MmArea( tyAreaNew, addrNewMin, addrNewMax ) );
		larea3.append( MmArea( tyAreaOld, addrNewMax, addrOldMax ) );
	    }
	}
	else
	{
	    if      ( addrNewMax <= addrOldMin )
	    {
		larea3.append( areaOld );
	    }
	    else if ( addrNewMax <= addrOldMax   )
	    {
		larea3.append( MmArea( tyAreaNew, addrOldMin, addrNewMax ) );
		larea3.append( MmArea( tyAreaOld, addrNewMax, addrOldMax ) );
	    }
	    else
	    {
		larea3.append( MmArea( tyAreaNew, addrOldMin, addrOldMax ) );
	    }
	}

	// Merge these fragments.
	for ( int iarea3 = 0; iarea3 < larea3.count( ); ++iarea3 )
	{
	    const MmArea & area3 = larea3[iarea3];
	    if ( area3.getAddrMin( ) < area3.getAddrMax( ) )
	    {
		if ( lareaMerge.count( ) > 0 )
		{
		    const MmArea & areaLast = lareaMerge[lareaMerge.count()-1];
		    if ( areaLast.getAddrMax( ) != area3.getAddrMin( ) )
			WhAbort( "MmMap::mergeArea: failed partition." );
		    if ( areaLast.getTyArea( ) == area3.getTyArea( ) )
		    {
			const MmArea areaJoin( area3.getTyArea( ),
			    areaLast.getAddrMin( ), area3.getAddrMax( ) );
			lareaMerge.removeLast( );
			lareaMerge.append( areaJoin );
		    }
		    else
		    {
			lareaMerge.append( area3 );
		    }
		}
		else
		{
		    lareaMerge.append( area3 );
		}
	    }
	}
    }

    // Substitute in new area list.
    if ( lareaMerge.count( ) == 0 )
	WhAbort( "MmMap::mergeArea: bad count." );
    larea_ = lareaMerge;

#if 0
    // Dump for inspection.
    ::printf( "MmMap::mergeArea: count: %d.\n", larea_.count( ) );
    for ( int iarea = 0; iarea < larea_.count( ); ++iarea )
    {
	const MmArea & area = larea_[iarea];
	::printf( "    0x%8.8X 0x%8.8X %d\n",
	    int( area.getAddrMin( ) ),
	    int( area.getAddrMax( ) ),
	    area.getTyArea( )
	    );
    }
#endif
}
