// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: MmSeg.cc
//   Memory segment.
//
// File Created:	11 Jun 1994		Michael Chastain
// Last Edited:		03 Jul 1995		Michael Chastain

#include <MmSeg.h>
#include <MmType.h>
#include <TySeg.h>
#include <WhAbort.h>



// Constructor.
MmSeg::MmSeg( )
    : fBound_		( false	)
    , fAreaNew_		( false	)
    , itySeg_		( 0	)
    , area_		(	)
    , lwData_		(	)
{
    ;
}



// Flat input.
MmRet MmSeg::fromFlat( MmFlat & flatFrom )
{
    MmRetCheck( flatFrom.getBool	( fBound_	) );
    MmRetCheck( flatFrom.getBool	( fAreaNew_	) );
    MmRetCheck( flatFrom.getInt		( itySeg_	) );
    MmRetCheck( area_.fromFlat		( flatFrom	) );
    MmRetCheck( flatFrom.getListWord	( lwData_	) );
    return mmRetOk;
}



// Flat output.
void MmSeg::toFlat( MmFlat & flatTo ) const
{
    flatTo.putBool	( fBound_		);
    flatTo.putBool	( fAreaNew_		);
    flatTo.putInt	( itySeg_		);
    area_.toFlat	( flatTo		);
    flatTo.putListWord	( lwData_		);
}



// Give away the address of the data in this segment.
const MmByte * MmSeg::address( ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::address: not bound." );
    return
	( (const MmByte *) lwData_.address( ) )
	+ MmAddrRem( area_.getAddrMin( ) );
}



// Return count.
int MmSeg::count( ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::count: not bound." );
    return area_.count( );
}



// Return first target address.
MmAddr MmSeg::getAddrFirst( ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::getAddrFirst: not bound." );
    return area_.getAddrMin( );
}



// Return last target address.
MmAddr MmSeg::getAddrLast( ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::getAddrLast: not bound." );
    return area_.getAddrMax( );
}



// Return whether two segments overlap.
bool MmSeg::hasOverlap( const MmSeg & seg2 ) const
{
    if ( !this->isBound( ) || !seg2.isBound( ) )
	return false;

    const MmAddr addr1First = this-> getAddrFirst ( );
    const MmAddr addr1Last  = this-> getAddrLast  ( );
    const MmAddr addr2First = seg2.  getAddrFirst ( );
    const MmAddr addr2Last  = seg2.  getAddrLast  ( );

    if ( addr1First >= addr2First && addr1First < addr2Last ) return true;
    if ( addr2First >= addr1First && addr2First < addr1Last ) return true;
    return false;
}



// Replace one byte with another.
void MmSeg::replaceByte( MmByte bLeft, MmByte bRight )
{
    WhList <MmWord> lwDataNew;
    lwDataNew.clear( lwData_.count( ) );

    for ( int iwData = 0; iwData < lwData_.count( ); ++iwData )
    {
	MmWord   wData  = lwData_[iwData];
	MmByte * pbData = (MmByte *) &wData;
	for ( int ibData=0; ibData < sizeof(MmWord)/sizeof(MmByte); ++ibData )
	{
	    if ( pbData[ibData] == bLeft )
		pbData[ibData] = bRight;
	}
	lwDataNew.append( wData );
    }

    lwData_ = lwDataNew;
}



// Set to unbound with a given area.
void MmSeg::setAreaFree( const MmArea & area )
{
    fBound_     = false;
    fAreaNew_   = true;
    itySeg_	= 0;
    area_       = area;
    lwData_.clear( );
}



// Check for const-char-star string.
void MmSeg::checkCcs( ) const
{
    if ( count( ) == 0 || count( ) % sizeof(MmByte) != 0 )
	WhAbort( "MmSeg::checkCcs: count mismatch." );
    const MmByte * pbStr = address( );
    for ( int ibStr = 0; ibStr < count( ) - 1; ++ibStr )
    {
	if ( pbStr[ibStr] == MmByte( '\0' ) )
	    WhAbort( "MmSeg::checkCcs: interior null." );
    }
    if ( pbStr[count( ) - 1] != MmByte( '\0' ) )
	WhAbort( "MmSeg::checkCcs: not nul at end." );
}



// Check count.
void MmSeg::checkCountList( int nByteCheck ) const
{
    if ( nByteCheck < 0 || count( ) % nByteCheck != 0 )
	WhAbort( "MmSeg::checkCountList: count mismatch." );
}



// Check count.
void MmSeg::checkCountMin( int nByteCheck ) const
{
    if ( count( ) < nByteCheck )
	WhAbort( "MmSeg::checkCountMin: count mismatch." );
}



// Check count.
void MmSeg::checkCountOne( int nByteCheck ) const
{
    if ( count( ) != nByteCheck )
	WhAbort( "MmSeg::checkCountOne: count mismatch." );
}



// Fetch from process with count.
MmRet MmSeg::fetchProc( const PrProc & procFetch, MmArea::TyArea tyArea,
    int itySeg, MmAddr addrFirst, int nItem )
{
    // Unbind.
    fBound_   = false;

    // Set segment type.
    itySeg_ = itySeg;

    // Clear existing data and hint at new size.
    const int nData = TySegGetSize( itySeg, nItem );
    if      ( nData >=  0 ) lwData_.clear( nData / sizeof(MmWord) + 2    );
    else if ( nData == -1 ) lwData_.clear( 60                            );
    else                    WhAbort( "MmSeg::fetchProc: negative count." );

    // Don't fetch at all for degenerate case.
    if ( nData == 0 )
    {
	area_   = MmArea( tyArea, addrFirst, addrFirst + nData );
	fBound_ = true;
	return mmRetOk;
    }

    // Fetch the child data.
    for ( MmAddr addrFetch  = MmAddrAlign(addrFirst);
                 ;
                 addrFetch += sizeof(MmWord) )
    {
	// Exit test for count-delimited fetch.
	if ( nData >= 0 )
	{
	    if ( addrFetch >= addrFirst + nData )
	    {
		area_     = MmArea( tyArea, addrFirst, addrFirst + nData );
		fBound_   = true;
		return mmRetOk;
	    }
	}

	// Fetch the word.
	MmWord wFetch;
	if ( !procFetch.fetchWord( tyArea, addrFetch, wFetch, 0 ) )
	    return MmRetRaise( mmRetSegWontFetch );
	lwData_.append( wFetch );

	// Exit test for zero-byte-delimited fetch.
	//   Little-endian.
	if ( nData == -1 )
	{
	    for ( int iByte = 0; iByte < sizeof(MmWord); ++iByte )
	    {
		if ( addrFetch + iByte >= addrFirst
		&&   ((const MmByte *) &wFetch)[iByte] == '\0' )
		{
		    area_   = MmArea( tyArea, addrFirst, addrFetch+iByte+1 );
		    fBound_ = true;
		    return mmRetOk;
		}
	    }
	}
    }
}



// Store to process.
//   I handle word alignment by filling partial words.
MmRet MmSeg::storeProc( PrProc & procStore ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::storeProc: not bound." );

    const MmAddr addrFirstAlign = MmAddrAlign( area_.getAddrMin( ) );
    for ( MmAddr addrStore  = addrFirstAlign;
		 addrStore  < area_.getAddrMax( );
		 addrStore += sizeof(MmWord) )
    {
	MmWord	wStore;

	// Construct word to store.
	if ( addrStore                  >= area_.getAddrMin( )
	&&   addrStore + sizeof(MmWord) <= area_.getAddrMax( ) )
	{
            wStore = lwData_[(addrStore - addrFirstAlign) / sizeof(MmWord)];
	}
	else
	{
	    // Partial word.
	    if ( !procStore.fetchWord( area_.getTyArea(), addrStore, wStore,
		0 ) )
	    {
		return MmRetRaise( mmRetSegWontFetch );
	    }

	    // Stuff bytes one at a time.
	    //   Little-endian.
	    for ( int iByte = 0; iByte < sizeof(MmWord); ++iByte )
	    {
		const MmAddr addrByte = addrStore + iByte;
		if ( addrByte >= area_.getAddrMin( )
		&&   addrByte <  area_.getAddrMax( ) )
		{
		    ((MmByte *) &wStore)[iByte] =
			((const MmByte *) lwData_.address( ))
			[addrByte - addrFirstAlign];
		}
	    }
	}

	// Store the word.
	if ( !procStore.storeWord( area_.getTyArea( ), addrStore, wStore, 0 ) )
	    return MmRetRaise( mmRetSegWontStore );
    }

    // That's all, folks.
    return mmRetOk;
}



// Return a printable representation.
void MmSeg::fmtSeg( WhString & strRet ) const
{
    if ( !isBound( ) )
	WhAbort( "MmSeg::fmtSeg: not bound." );
    TySegFmtSeg( itySeg_, strRet, *this );
}
