// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrProc.cc
//   Process handler.
//   This is an external resource handler class.
//
// File Created:	04 Oct 1994		Michael Chastain
// Last Reviewed:	23 Feb 1995		Michael Chastain
// Last Edited:		22 Sep 1995		Michael Chastain

#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <errno.h>
#include <unistd.h>

#if defined(MEC_TARGET_LIX)
#include <linux/unistd.h>
#include <signal.h>
#endif

#if defined(MEC_TARGET_SUN)
#include <FixHeadSun.h>
#include <sys/signal.h>
#include <sys/user.h>
#endif

#include <EvEvent.h>
#include <EvSci.h>
#include <EvSco.h>
#include <MmArea.h>
#include <MmSeg.h>
#include <MmType.h>
#include <PrProc.h>
#include <PrScr.h>
#include <TySegEnumLix.h>
#include <WhAbort.h>



// Constructor.
PrProc::PrProc( )
    : stExec_		( stExecBlank	)
    , stSys_		( stSysBlank	)
    , pid_		( 0		)
    , fWait_		( false		)
    , timeWait_		(		)
    , iStatusWait_	( 0		)
    , fSysEntry_	( false		)
    , wSysEntry_	( 0		)
    , wSubEntry_	( 0		)
    , fFollowRoot_	( false		)
    , fFollowCwd_	( false		)
{
    ;
}



// Destructor.
PrProc::~PrProc( )
{
    switch ( stExec_ )
    {
    default:
	break;

    case stExecRunning:
    case stExecStopSignal:
    case stExecStopSyscall:
	::errno = 0;
	::kill( pid_, SIGKILL );
	if ( ::errno != 0 )
	    WhAbort( "PrProc::~PrProc: failed kill." );
	break;
    }
}



// Get status for normal exit.
int PrProc::getExitNormal( ) const
{
    if ( stExec_ != stExecExitNormal )
	WhAbort( "PrProc::getExitNormal: bad state." );
    if ( !fWait_ )
	WhAbort( "PrProc::getExitNormal: not bound." );
    int iStatusWait = iStatusWait_;
    return WEXITSTATUS(iStatusWait);
}



// Get status for signal exit.
int PrProc::getExitSignal( ) const
{
    if ( stExec_ != stExecExitSignal )
	WhAbort( "PrProc::getExitSignal: bad state." );
    if ( !fWait_ )
	WhAbort( "PrProc::getExitSignal: not bound." );
    int iStatusWait = iStatusWait_;
    return WTERMSIG(iStatusWait);
}



// Get status for signal stop.
int PrProc::getStopSignal( ) const
{
    if ( stExec_ != stExecStopSignal )
	WhAbort( "PrProc::getStopSignal: bad state." );
    if ( !fWait_ )
	WhAbort( "PrProc::getStopSignal: not bound." );
    int iStatusWait = iStatusWait_;
    return WSTOPSIG(iStatusWait);
}



// Get time of last wait.
const PrTime & PrProc::getTimeWait( ) const
{
    if ( !fWait_ )
	WhAbort( "PrProc::getTimeWait: not bound." );
    return timeWait_;
}



// Continue the process.
void PrProc::contProc( )
{
    // Validate state.
    if ( stExec_ != stExecStopSignal && stExec_ != stExecStopSyscall )
	WhAbort( "PrProc::contProc: bad state." );

    // Continue the process.
    ::errno = 0;
    ::ptrace( PTRACE_SYSCALL, pid_, 0,
	stExec_ == stExecStopSignal ? getStopSignal( ) : 0 );
    if ( ::errno != 0 )
	WhAbort( "PrProc::contProc: failed ptrace." );

    // Move to new state.
    stExec_ = stExecRunning;
}



// Spawn a child process.
//   Doesn't honor $PATH.
void PrProc::spawnProc( const char * pstrName, const char * const * argv,
    const char * const * envp )
{
    // Validate parameters.
    if ( pstrName == 0 || argv == 0 || argv[0] == 0 || envp == 0 )
	WhAbort( "PrProc::spawnProc: zero pointer." );

    // Validate state.
    if ( stExec_ != stExecBlank
    &&   stExec_ != stExecExitNormal
    &&   stExec_ != stExecExitSignal )
	WhAbort( "PrProc::spawnProc: bad state." );

    // Make name point to unique memory.
    WhString strName( pstrName );
    strName.appChrRaw( '\0' );

    // Fork the process.
    ::errno = 0;
    pid_ = ::fork( );
    if ( ::errno != 0 )
	WhAbort( "PrProc::spawnProc: failed fork." );

    // Return for parent.
    //   After this is child process code.
    //   Child can never return: must abort on error.
    if ( pid_ != 0 )
    {
	stExec_		= stExecRunning;
	stSys_		= stSysBlank;
	fWait_		= false;
	fSysEntry_	= false;
	fFollowRoot_	= false;
	fFollowCwd_	= false;
	return;
    }

    // Enable tracing.
    ::errno = 0;
    ::ptrace( PTRACE_TRACEME, ::getpid( ), 0, 0 );
    if ( ::errno != 0 )
	WhAbort( "PrProc::spawnProc: failed ptrace." );

    // Trap for the parent.
    ::errno = 0;
    ::kill( ::getpid( ), SIGTRAP );
    if ( ::errno != 0 )
	WhAbort( "PrProc::spawnProc: failed kill." );

    // Exec the process.
    //   gcc 2.6.3: I believe the prototype for 'execve' is missing a const.
    ::errno = 0;
    ::execve( strName.address( ),
	(char * const []) argv, (char * const []) envp );
    WhAbort( "PrProc::spawnProc: failed execve." );
}



// Wait for status change in a process.
void PrProc::waitProc( WhList <PrProc*> & lpProcWait, PrProc * & pProcWaitRet )
{
    // Clear return value.
    pProcWaitRet = 0;

    // Wait for a process.
    ::errno = 0;
    int iStatusWait;
    const pid_t pidWait = ::wait4( -1, &iStatusWait, 0, 0 );
    if ( ::errno != 0 )
	WhAbort( "PrProc::waitProc: failed wait." );

    // Mark time.
    PrTime timeWait;
    timeWait.mark( );

    // Find process which changed.
    for ( int ipProcWait = 0; ipProcWait < lpProcWait.count( ); ++ipProcWait )
    {
	PrProc * pProcWait = lpProcWait[ipProcWait];
	if ( pProcWait == 0 )
	    WhAbort( "PrProc::waitProc: zero pointer." );

	if ( pProcWait->pid_ == pidWait )
	{
	    pProcWait->waitUpdate( iStatusWait, timeWait );
	    pProcWaitRet = pProcWait;
	    return;
	}
    }

    // Not found!
    WhAbort( "PrProc::waitProc: unlisted process." );
}



// Update state of a process after wait.
void PrProc::waitUpdate( int iStatusWait, const PrTime & timeWait )
{
    // Validate state.
    if ( stExec_ != stExecRunning )
	WhAbort( "PrProc::waitUpdate: bad state." );

    // Record status and time.
    iStatusWait_ = iStatusWait;
    timeWait_    = timeWait;
    fWait_       = true;

    // Advance exec state.
    if      ( WIFEXITED   (iStatusWait) )
    {
	stExec_ = stExecExitNormal;
	stSys_  = stSysBlank;
    }
    else if ( WIFSIGNALED (iStatusWait) )
    {
	stExec_ = stExecExitSignal;
	stSys_  = stSysBlank;
    }
    else if ( WIFSTOPPED  (iStatusWait) )
    {
	stExec_ = stExecStopSignal;
	if ( waitIsSyscall( iStatusWait ) )
	{
	    stExec_ = stExecStopSyscall;
	    waitDoSyscall( );
	}
    }
    else
    {
	// Can't happen according to 'man 2 wait'.
	WhAbort( "PrProc::waitUpdate: bad status." );
    }
}



// Return whether wait is a system call.
bool PrProc::waitIsSyscall( int iStatusWait ) const
{
    // Test trap number.
    if ( !WIFSTOPPED(iStatusWait) || WSTOPSIG(iStatusWait) != SIGTRAP )
	return false;

#if defined(MEC_TARGET_LIX)
    // Linux 1.2.9: hack for execve,
    //   which sends two traps with PC=0 during sys call.
    if ( stSys_ == stSysSci || stSys_ == stSysExec )
    {
	if ( !fSysEntry_ )
	    WhAbort( "PrProc::waitIsSyscall: no sys entry." );
	if ( wSysEntry_ == __NR_execve )
	    return true;
    }

    // Fetch program counter.
    MmWord wPc;
    if ( !fetchRegPc( wPc ) )
	WhAbort( "PrProc::waitIsSyscall: failed fetch." );
    const MmAddr addrPc = MmAddr( wPc );
    if ( addrPc < MmAddr( 2 ) )
	WhAbort( "PrProc::waitIsSyscall: bad pc." );

    // Fetch byte 0 of previous 2-byte instruction.
    MmWord wFetch0;
    if ( !fetchWord( MmArea::tyAreaText, MmAddrAlign(addrPc-2), wFetch0, 0 ) )
	WhAbort( "PrProc::waitIsSyscall: failed fetch." );
    const MmByte bFetch0 = ((MmByte *) &wFetch0)[MmAddrRem(addrPc-2)];

    // Fetch byte 1 of previous 2-byte instruction.
    MmWord wFetch1;
    if ( !fetchWord( MmArea::tyAreaText, MmAddrAlign(addrPc-1), wFetch1, 0 ) )
	WhAbort( "PrProc::waitIsSyscall: failed fetch." );
    const MmByte bFetch1 = ((MmByte *) &wFetch1)[MmAddrRem(addrPc-1)];

#if 0
    // Diagnostic output.
    ::printf( "PrProc::waitIsSyscall: %p %2.2X %2.2X.\n",
	addrPc-2, bFetch0, bFetch1 );
#endif

    // Test this instruction.
    //   Fails on instruction like 'mov ax, 0x80CD0000'!
    if ( bFetch0 != 0xCD || bFetch1 != 0x80 )
	return false;
#endif

#if defined(MEC_TARGET_SUN)
    WhAbort( "PrProc::waitIsSyscall: not implemented." );
#endif

    return true;
}



// Wait just returned a sys call.
void PrProc::waitDoSyscall( )
{
    // Advance sys state.
    switch ( stSys_ )
    {
    default:
	WhAbort( "PrProc::waitDoSyscall: bad state." );

    case stSysBlank:
	stSys_ = stSysFirst;
	break;

    case stSysFirst:
	stSys_ = stSysSci;
	break;

    case stSysSci:
	if ( !fSysEntry_ )
	    WhAbort( "PrProc::waitDoSyscall: no sys entry." );

	switch ( wSysEntry_ )
	{
	default:
	    stSys_ = stSysSco;
	    break;

#if defined(MEC_TARGET_LIX)
	case __NR_execve:
	    PrScr scrExec;
	    scrExec.fetchProc( *this, PrScr::tyScrInt );
	    stSys_ = scrExec.isError( ) ? stSysSco : stSysExec;
	    break;

	case __NR_vm86:
	    stSys_ = stSysSci;
	    break;
#endif
	}
	break;

    case stSysExec:
	stSys_ = stSysSco;
	break;

    case stSysSco:
	stSys_ = stSysSci;
	break;
    }

    // Fetch system call number.
    if ( stSys_ == stSysSci )
    {
	if ( !fetchSysEntry( wSysEntry_, wSubEntry_ ) )
	    WhAbort( "PrProc::waitDoSyscall: cannot fetch." );
	fSysEntry_ = true;
    }
}



// Fetch a register.
bool PrProc::fetchReg( int iReg, MmWord & wRegRet ) const
{
    // Validate state.
    if ( stExec_ != stExecStopSignal && stExec_ != stExecStopSyscall )
	WhAbort( "PrProc::fetchReg: bad state." );

    // Validate register.
    if ( iReg < 0 )
	WhAbort( "PrProc::fetchReg: negative index." );

#if defined(MEC_TARGET_LIX)
    // Fetch from process.
    const int wAddrReg = (EBX + iReg) * sizeof(MmWord);
    ::errno = 0;
    wRegRet = ::ptrace( PTRACE_PEEKUSR, pid_, wAddrReg, 0 );
    if ( ::errno != 0 )
	return false;
#endif

#if defined(MEC_TARGET_SUN)
    // Fetch from process.
    struct user user;
    const MmAddr addrReg =
	MmAddr( MmAddr( &user.u_arg[iReg] ) - MmAddr( &user ) );
    ::errno = 0;
    wRegRet = ::ptrace( PTRACE_PEEKUSER, pid_, addrReg, 0 );
    if ( ::errno != 0 )
	return false;
#endif

    // Success.
    return true;
}



// Fetch register: program counter.
bool PrProc::fetchRegPc( MmWord & wPcRet ) const
{
#if defined(MEC_TARGET_LIX)
    return fetchReg( EIP - EBX, wPcRet );
#endif

#if defined(MEC_TARGET_SUN)
    return fetchReg( PC, wPcRet );
#endif
}



// Fetch register: stack pointer.
bool PrProc::fetchRegSp( MmWord & wSpRet ) const
{
#if defined(MEC_TARGET_LIX)
    return fetchReg( UESP - EBX, wSpRet );
#endif

#if defined(MEC_TARGET_SUN)
    return fetchReg( SP, wSpRet );
#endif
}



// Fetch a list of registers.
bool PrProc::fetchListReg( int iRegFirst, int nReg, WhList <MmWord> & lwReg )
    const
{
    // Validate count.
    if ( nReg < 0 )
	WhAbort( "PrProc::fetchListReg: negative count." );

    // Fetch registers.
    lwReg.clear( nReg );
    for ( int iReg = 0; iReg < nReg; ++iReg )
    {
	MmWord wReg;
	if ( !fetchReg( iRegFirst + iReg, wReg ) )
	    return false;
	lwReg.append( wReg );
    }

    // Success.
    return true;
}



// Fetch a system call entry value.
bool PrProc::fetchSysEntry( MmWord & wSysEntryRet, MmWord & wSubEntryRet )
    const
{
#if defined(MEC_TARGET_LIX)
    if ( !fetchReg( ORIG_EAX - EBX, wSysEntryRet ) )
	return false;
    wSubEntryRet = 0;

    switch ( wSysEntryRet )
    {
    default:
	break;

    case __NR_ipc:
    case __NR_socketcall:
	if ( !fetchReg( 0, wSubEntryRet ) )
	    return false;
	break;
    }
#endif

#if defined(MEC_TARGET_SUN)
    if ( !fetchReg( 7, wSysEntryRet ) )
	return false;
    wSubEntryRet = 0;
#endif

    return true;
}



// Store a register.
bool PrProc::storeReg( int iReg, MmWord wReg )
{
    // Validate state.
    if ( stExec_ != stExecStopSignal && stExec_ != stExecStopSyscall )
	WhAbort( "PrProc::storeReg: bad state." );

    // Validate register.
    if ( iReg < 0 )
	WhAbort( "PrProc::storeReg: negative index." );

#if defined(MEC_TARGET_LIX)
    // Store to process.
    const int wAddrReg = (EBX + iReg) * sizeof(MmWord);
    ::errno = 0;
    ::ptrace( PTRACE_POKEUSR, pid_, wAddrReg, wReg );
    if ( ::errno != 0 )
	return false;
#endif

#if defined(MEC_TARGET_SUN)
    // Store to process.
    struct user user;
    const MmAddr addrReg =
	MmAddr( MmAddr( &user.u_arg[iReg] ) - MmAddr( &user ) );
    ::errno = 0;
    ::ptrace( PTRACE_POKEUSER, pid_, addrReg, wReg );
    if ( ::errno != 0 )
	return false;
#endif

    // Success.
    return true;
}



// Store a system call entry value.
bool PrProc::storeSysEntry( MmWord wSysEntry )
{
#if defined(MEC_TARGET_LIX)
    // Linux 1.2.0: kernel rejects this.
    if ( !storeReg( ORIG_EAX - EBX, wSysEntry ) )
	return false;
#endif

#if defined(MEC_TARGET_SUN)
    if ( !storeReg( 7, wSysEntry ) )
	return false;
#endif

    // Success.
    return true;
}



// Fetch a word from user struct, text space, or data space.
bool PrProc::fetchWord( MmArea::TyArea tyArea, MmAddr addrFetch,
    MmWord & wFetchRet, PrScr * pscrFetchRet ) const
{
    // Validate state.
    if ( stExec_ != stExecStopSignal && stExec_ != stExecStopSyscall )
	WhAbort( "PrProc::fetchWord: bad state." );

    // Declare return value.
    PrScr scrFetch;

#if defined(MEC_TARGET_LIX)
    // Fetch the word.
    const int reqSpace =
	(tyArea == MmArea::tyAreaUser) ? PTRACE_PEEKUSR  :
	(tyArea == MmArea::tyAreaText) ? PTRACE_PEEKTEXT
	                               : PTRACE_PEEKDATA;
    ::errno = 0;
    wFetchRet = ::ptrace( reqSpace, pid_, int( addrFetch ), 0 );
    const MmWord wErrno = ::errno;

    // Set return value.
    if ( wErrno != 0 )
	scrFetch.setError( PrScr::tyScrInt, wErrno );
    else
	scrFetch.setValue( PrScr::tyScrInt, 0, 0 );
#endif

#if defined(MEC_TARGET_SUN)
    // Fetch the word.
    const enum ptracereq reqSpace =
	(tyArea == MmArea::tyAreaUser) ? PTRACE_PEEKUSER :
	(tyArea == MmArea::tyAreaText) ? PTRACE_PEEKTEXT
	                               : PTRACE_PEEKDATA;
    ::errno = 0;
    wFetchRet = ::ptrace( reqSpace, pid_, addrFetch, 0 );
    const MmWord wErrno = ::errno;

    // Set return value.
    //   SunOS 4.1.3: this is a wild guess.
    if ( wErrno != 0 )
	scrFetch.setError( PrScr::tyScrInt, wErrno );
    else
	scrFetch.setValue( PrScr::tyScrInt, wFetchRet, 0 );
#endif

    // That's all, folks.
    if ( pscrFetchRet != 0 )
	*pscrFetchRet = scrFetch;
    return !scrFetch.isError( );
}



// Store a word to user struct, text space, or data space.
bool PrProc::storeWord( MmArea::TyArea tyArea, MmAddr addrStore,
    MmWord wStore, PrScr * pscrStoreRet )
{
    // Validate state.
    if ( stExec_ != stExecStopSignal && stExec_ != stExecStopSyscall )
	WhAbort( "PrProc::storeWord: bad state." );

    // Declare return value.
    PrScr scrStore;

#if defined(MEC_TARGET_LIX)
    // Store the word.
    const int reqSpace =
	(tyArea == MmArea::tyAreaUser) ? PTRACE_POKEUSR  :
	(tyArea == MmArea::tyAreaText) ? PTRACE_POKETEXT
	                               : PTRACE_POKEDATA;
    ::errno = 0;
    ::ptrace( reqSpace, pid_, int( addrStore ), wStore );
    const MmWord wErrno = ::errno;

    // Set return value.
    if ( wErrno != 0 )
	scrStore.setError( PrScr::tyScrInt, wErrno );
    else
	scrStore.setValue( PrScr::tyScrInt, 0, 0 );
#endif

#if defined(MEC_TARGET_SUN)
    // Store the word.
    const enum ptracereq reqSpace =
	(tyArea == MmArea::tyAreaUser) ? PTRACE_POKEUSER :
	(tyArea == MmArea::tyAreaText) ? PTRACE_POKETEXT
	                               : PTRACE_POKEDATA;
    ::errno = 0;
    ::ptrace( reqSpace, pid_, addrStore, wStore );
    const MmWord wErrno = ::errno;

    // Set return value.
    if ( wErrno != 0 )
	scrStore.setError( PrScr::tyScrInt, wErrno );
    else
	scrStore.setValue( PrScr::tyScrInt, 0, 0 );
#endif

    // That's all, folks.
    if ( pscrStoreRet != 0 )
	*pscrStoreRet = scrStore;
    return !scrStore.isError( );
}



// Follow root directory and current directory of child.
//   Needed so 'execve' and 'uselib' can read child's filenames.
void PrProc::followDir( const EvSci * pevSci, const EvSco * pevSco )
{
    // Check state.
    if ( stSys_ != stSysSco )
	return;

    // Validate pointers.
    if ( pevSci == 0 || pevSco == 0 )
	WhAbort( "PrProc::followDir: zero pointer." );

#if defined(MEC_TARGET_LIX)
    switch ( pevSci->getSysEntry( ) )
    {
    default:
	break;

    case __NR_chdir:
	if ( !pevSco->isScrError( ) )
	{
	    const bool fFollowCwdOld = fFollowCwd_;
	    fFollowCwd_ = false;
	    const MmSeg & segInName = pevSci->getSeg( )[0];
	    segInName.checkCcs( );
	    const char * pstrInName = (const char *) segInName.address( );
	    if ( pstrInName[0] == '/' ? fFollowRoot_ : fFollowCwdOld )
	    {
		::errno = 0;
		::chdir( pstrInName );
		if ( ::errno != 0 )
		    WhAbort( "PrProc::followDir: failed chdir." );
		fFollowCwd_ = true;
	    }
	}
	break;

    case __NR_chroot:
	if ( !pevSco->isScrError( ) )
	{
	    const bool fFollowRootOld = fFollowRoot_;
	    fFollowRoot_ = false;
	    const MmSeg & segInName = pevSci->getSeg( )[0];
	    segInName.checkCcs( );
	    const char * pstrInName = (const char *) segInName.address( );
	    if ( pstrInName[0] == '/' ? fFollowRootOld : fFollowCwd_ )
	    {
		::errno = 0;
		::chroot( pstrInName );
		if ( ::errno != 0 )
		    WhAbort( "PrProc::followDir: failed chroot." );
		fFollowRoot_ = true;
	    }
	}
	break;

    case __NR_fchdir:
	if ( !pevSco->isScrError( ) )
	{
	    // Loses.
	    //   Could be made to win sometimes by following open et cetera.
	    fFollowCwd_ = false;
	}
	break;
    }
#endif

#if defined(MEC_TARGET_SUN)
    // Unimplemented, lose.
    fFollowCwd_  = false;
    fFollowRoot_ = false;
#endif
}
