// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: PrScrLix.cc
//   System call return value.
//   This is a concrete class.
//
// File Created:	19 Jun 1995		Michael Chastain
// Last Edited:		30 Jun 1995		Michael Chastain

#include <sys/ptrace.h>
#define  __KERNEL__
#include <linux/errno.h>
#undef   __KERNEL__

#include <PrScr.h>
#include <WhAbort.h>



// Constructor.
PrScr::PrScr( )
    : tyScr_	( tyScrNot	)
    , fBound_	( false		)
    , wScr_	( 0		)
{
    ;
}



// Flat input.
MmRet PrScr::fromFlat( MmFlat & flatFrom )
{
    int ityScr;
    MmRetCheck( flatFrom.getInt  ( ityScr  ) );
    switch ( ityScr )
    {
    default:		return MmRetRaise( mmRetFlatEnumBad );
    case tyScrAddr:	tyScr_ = tyScrAddr; break;
    case tyScrHex:	tyScr_ = tyScrHex;  break;
    case tyScrInt:	tyScr_ = tyScrInt;  break;
    case tyScrNot:	tyScr_ = tyScrNot;  break;
    case tyScrOct:	tyScr_ = tyScrOct;  break;
    }
    MmRetCheck( flatFrom.getBool ( fBound_ ) );
    MmRetCheck( flatFrom.getWord ( wScr_   ) );
    return mmRetOk;
}



// Flat output.
void PrScr::toFlat( MmFlat & flatTo ) const
{
    flatTo.putInt  ( int( tyScr_ ) );
    flatTo.putBool ( fBound_       );
    flatTo.putWord ( wScr_         );
}



// Fetch from process.
void PrScr::fetchProc( const PrProc & procFetch, TyScr tyScr )
{
    if ( !procFetch.fetchReg( EAX - EBX, wScr_ ) )
	WhAbort( "PrScr::fetchProc: cannot fetch." );
    tyScr_  = tyScr;
    fBound_ = true;
}



// Store to process.
void PrScr::storeProc( PrProc & procStore ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::storeProc: not bound." );

    if ( !procStore.storeReg( EAX - EBX, wScr_ ) )
	WhAbort( "PrScr::storeProc: cannot store." );
}



// Format to string.
void PrScr::fmtScr( WhString & strRet ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::fmtScr: not bound." );

    if ( isError( ) )
    {
	fmtError( strRet );
    }
    else
    {
	switch ( tyScr_ )
	{
	default:
	    WhAbort( "PrScr::fmtStr: bad enum." );

	case tyScrAddr:
	    strRet.appPtrFmt( MmAddr( wScr_ ) );
	    break;

	case tyScrHex:	
	    strRet.appStrRaw( "0x" );
	    strRet.appIntFmt( wScr_, 16, 8 );
	    break;

	case tyScrInt:
	    strRet.appIntFmt( wScr_ );
	    break;

	case tyScrNot:
	    strRet.appStrRaw( "(UNEXPECTED!) " "0x" );
	    strRet.appIntFmt( wScr_, 16, 8 );
	    break;

	case tyScrOct:
	    strRet.appChrRaw( '0' );
	    strRet.appIntFmt( wScr_, 8, 0 );
	    break;
	}
    }
    strRet.appChrRaw( '\n' );
}



// Get a word.
MmWord PrScr::getWord( int iWord ) const
{
    if ( !fBound_ )
	WhAbort("PrScr::getWord: not bound." );
    if ( isError( ) )
	WhAbort( "PrScr::getWord: is error." );

    switch ( iWord )
    {
    default: WhAbort( "PrScr::getWord: bad index." );
    case 0:  return wScr_;
    }
}



// Return whether value represents an error.
bool PrScr::isError( ) const
{
    if ( !fBound_ )
	WhAbort( "PrScr::isError: not bound." );

    return ( tyScr_ == tyScrAddr || tyScr_ == tyScrInt )
        && ( int( wScr_ ) < 0 && int( wScr_ ) >= -1024 );
}



// Set value to represent a given error.
void PrScr::setError( TyScr tyScr, MmWord wError )
{
    tyScr_  = tyScr;
    wScr_   = 0 - wError;
    fBound_ = true;

    if ( !isError( ) )
	WhAbort( "PrScr::setError: not error." );
}



// Set value to represent a given non-error value.
void PrScr::setValue( TyScr tyScr, MmWord w1, MmWord )
{
    tyScr_  = tyScr;
    wScr_   = w1;
    fBound_ = true;

    if ( isError( ) )
	WhAbort( "PrScr::setValue: is error." );
}



// Format a string from an error value.
void PrScr::fmtError( WhString & strRet ) const
{
    strRet.appIntFmt( int( wScr_ ) );
    switch ( 0 - wScr_ )
    {
#define	when  break; case
    default:			strRet.appStrRaw( " (UNKNOWN)"		);
    when EPERM:			strRet.appStrRaw( " (EPERM)"		);
    when ENOENT:		strRet.appStrRaw( " (ENOENT)"		);
    when ESRCH:			strRet.appStrRaw( " (ESRCH)"		);
    when EINTR:			strRet.appStrRaw( " (EINTR)"		);
    when EIO:			strRet.appStrRaw( " (EIO)"		);
    when ENXIO:			strRet.appStrRaw( " (ENXIO)"		);
    when E2BIG:			strRet.appStrRaw( " (E2BIG)"		);
    when ENOEXEC:		strRet.appStrRaw( " (ENOEXEC)"		);
    when EBADF:			strRet.appStrRaw( " (EBADF)"		);
    when ECHILD:		strRet.appStrRaw( " (ECHILD)"		);
    when EAGAIN:		strRet.appStrRaw( " (EAGAIN)"		);
    when ENOMEM:		strRet.appStrRaw( " (ENOMEM)"		);
    when EACCES:		strRet.appStrRaw( " (EACCES)"		);
    when EFAULT:		strRet.appStrRaw( " (EFAULT)"		);
    when ENOTBLK:		strRet.appStrRaw( " (ENOTBLK)"		);
    when EBUSY:			strRet.appStrRaw( " (EBUSY)"		);
    when EEXIST:		strRet.appStrRaw( " (EEXIST)"		);
    when EXDEV:			strRet.appStrRaw( " (EXDEV)"		);
    when ENODEV:		strRet.appStrRaw( " (ENODEV)"		);
    when ENOTDIR:		strRet.appStrRaw( " (ENOTDIR)"		);
    when EISDIR:		strRet.appStrRaw( " (EISDIR)"		);
    when EINVAL:		strRet.appStrRaw( " (EINVAL)"		);
    when ENFILE:		strRet.appStrRaw( " (ENFILE)"		);
    when EMFILE:		strRet.appStrRaw( " (EMFILE)"		);
    when ENOTTY:		strRet.appStrRaw( " (ENOTTY)"		);
    when ETXTBSY:		strRet.appStrRaw( " (ETXTBSY)"		);
    when EFBIG:			strRet.appStrRaw( " (EFBIG)"		);
    when ENOSPC:		strRet.appStrRaw( " (ENOSPC)"		);
    when ESPIPE:		strRet.appStrRaw( " (ESPIPE)"		);
    when EROFS:			strRet.appStrRaw( " (EROFS)"		);
    when EMLINK:		strRet.appStrRaw( " (EMLINK)"		);
    when EPIPE:			strRet.appStrRaw( " (EPIPE)"		);
    when EDOM:			strRet.appStrRaw( " (EDOM)"		);
    when ERANGE:		strRet.appStrRaw( " (ERANGE)"		);
    when EDEADLK:		strRet.appStrRaw( " (EDEADLK)"		);
    when ENAMETOOLONG:		strRet.appStrRaw( " (ENAMETOOLONG)"	);
    when ENOLCK:		strRet.appStrRaw( " (ENOLCK)"		);
    when ENOSYS:		strRet.appStrRaw( " (ENOSYS)"		);
    when ENOTEMPTY:		strRet.appStrRaw( " (ENOTEMPTY)"	);
    when ELOOP:			strRet.appStrRaw( " (ELOOP)"		);
    when ENOMSG:		strRet.appStrRaw( " (ENOMSG)"		);
    when EIDRM:			strRet.appStrRaw( " (EIDRM)"		);
    when ECHRNG:		strRet.appStrRaw( " (ECHRNG)"		);
    when EL2NSYNC:		strRet.appStrRaw( " (EL2NSYNC)"		);
    when EL3HLT:		strRet.appStrRaw( " (EL3HLT)"		);
    when EL3RST:		strRet.appStrRaw( " (EL3RST)"		);
    when ELNRNG:		strRet.appStrRaw( " (ELNRNG)"		);
    when EUNATCH:		strRet.appStrRaw( " (EUNATCH)"		);
    when ENOCSI:		strRet.appStrRaw( " (ENOCSI)"		);
    when EL2HLT:		strRet.appStrRaw( " (EL2HLT)"		);
    when EBADE:			strRet.appStrRaw( " (EBADE)"		);
    when EBADR:			strRet.appStrRaw( " (EBADR)"		);
    when EXFULL:		strRet.appStrRaw( " (EXFULL)"		);
    when ENOANO:		strRet.appStrRaw( " (ENOANO)"		);
    when EBADRQC:		strRet.appStrRaw( " (EBADRQC)"		);
    when EBADSLT:		strRet.appStrRaw( " (EBADSLT)"		);
    when EDEADLOCK:		strRet.appStrRaw( " (EDEADLOCK)"	);
    when EBFONT:		strRet.appStrRaw( " (EBFONT)"		);
    when ENOSTR:		strRet.appStrRaw( " (ENOSTR)"		);
    when ENODATA:		strRet.appStrRaw( " (ENODATA)"		);
    when ETIME:			strRet.appStrRaw( " (ETIME)"		);
    when ENOSR:			strRet.appStrRaw( " (ENOSR)"		);
    when ENONET:		strRet.appStrRaw( " (ENONET)"		);
    when ENOPKG:		strRet.appStrRaw( " (ENOPKG)"		);
    when EREMOTE:		strRet.appStrRaw( " (EREMOTE)"		);
    when ENOLINK:		strRet.appStrRaw( " (ENOLINK)"		);
    when EADV:			strRet.appStrRaw( " (EADV)"		);
    when ESRMNT:		strRet.appStrRaw( " (ESRMNT)"		);
    when ECOMM:			strRet.appStrRaw( " (ECOMM)"		);
    when EPROTO:		strRet.appStrRaw( " (EPROTO)"		);
    when EMULTIHOP:		strRet.appStrRaw( " (EMULTIHOP)"	);
    when EDOTDOT:		strRet.appStrRaw( " (EDOTDOT)"		);
    when EBADMSG:		strRet.appStrRaw( " (EBADMSG)"		);
    when EOVERFLOW:		strRet.appStrRaw( " (EOVERFLOW)"	);
    when ENOTUNIQ:		strRet.appStrRaw( " (ENOTUNIQ)"		);
    when EBADFD:		strRet.appStrRaw( " (EBADFD)"		);
    when EREMCHG:		strRet.appStrRaw( " (EREMCHG)"		);
    when ELIBACC:		strRet.appStrRaw( " (ELIBACC)"		);
    when ELIBBAD:		strRet.appStrRaw( " (ELIBBAD)"		);
    when ELIBSCN:		strRet.appStrRaw( " (ELIBSCN)"		);
    when ELIBMAX:		strRet.appStrRaw( " (ELIBMAX)"		);
    when ELIBEXEC:		strRet.appStrRaw( " (ELIBEXEC)"		);
    when EILSEQ:		strRet.appStrRaw( " (EILSEQ)"		);
    when ERESTART:		strRet.appStrRaw( " (ERESTART)"		);
    when ESTRPIPE:		strRet.appStrRaw( " (ESTRPIPE)"		);
    when EUSERS:		strRet.appStrRaw( " (EUSERS)"		);
    when ENOTSOCK:		strRet.appStrRaw( " (ENOTSOCK)"		);
    when EDESTADDRREQ:		strRet.appStrRaw( " (EDESTADDRREQ)"	);
    when EMSGSIZE:		strRet.appStrRaw( " (EMSGSIZE)"		);
    when EPROTOTYPE:		strRet.appStrRaw( " (EPROTOTYPE)"	);
    when ENOPROTOOPT:		strRet.appStrRaw( " (ENOPROTOOPT)"	);
    when EPROTONOSUPPORT:	strRet.appStrRaw( " (EPROTONOSUPPORT)"	);
    when ESOCKTNOSUPPORT:	strRet.appStrRaw( " (ESOCKTNOSUPPORT)"	);
    when EOPNOTSUPP:		strRet.appStrRaw( " (EOPNOTSUPP)"	);
    when EPFNOSUPPORT:		strRet.appStrRaw( " (EPFNOSUPPORT)"	);
    when EAFNOSUPPORT:		strRet.appStrRaw( " (EAFNOSUPPORT)"	);
    when EADDRINUSE:		strRet.appStrRaw( " (EADDRINUSE)"	);
    when EADDRNOTAVAIL:		strRet.appStrRaw( " (EADDRNOTAVAIL)"	);
    when ENETDOWN:		strRet.appStrRaw( " (ENETDOWN)"		);
    when ENETUNREACH:		strRet.appStrRaw( " (ENETUNREACH)"	);
    when ENETRESET:		strRet.appStrRaw( " (ENETRESET)"	);
    when ECONNABORTED:		strRet.appStrRaw( " (ECONNABORTED)"	);
    when ECONNRESET:		strRet.appStrRaw( " (ECONNRESET)"	);
    when ENOBUFS:		strRet.appStrRaw( " (ENOBUFS)"		);
    when EISCONN:		strRet.appStrRaw( " (EISCONN)"		);
    when ENOTCONN:		strRet.appStrRaw( " (ENOTCONN)"		);
    when ESHUTDOWN:		strRet.appStrRaw( " (ESHUTDOWN)"	);
    when ETOOMANYREFS:		strRet.appStrRaw( " (ETOOMANYREFS)"	);
    when ETIMEDOUT:		strRet.appStrRaw( " (ETIMEDOUT)"	);
    when ECONNREFUSED:		strRet.appStrRaw( " (ECONNREFUSED)"	);
    when EHOSTDOWN:		strRet.appStrRaw( " (EHOSTDOWN)"	);
    when EHOSTUNREACH:		strRet.appStrRaw( " (EHOSTUNREACH)"	);
    when EALREADY:		strRet.appStrRaw( " (EALREADY)"		);
    when EINPROGRESS:		strRet.appStrRaw( " (EINPROGRESS)"	);
    when ESTALE:		strRet.appStrRaw( " (ESTALE)"		);
    when EUCLEAN:		strRet.appStrRaw( " (EUCLEAN)"		);
    when ENOTNAM:		strRet.appStrRaw( " (ENOTNAM)"		);
    when ENAVAIL:		strRet.appStrRaw( " (ENAVAIL)"		);
    when EISNAM:		strRet.appStrRaw( " (EISNAM)"		);
    when EREMOTEIO:		strRet.appStrRaw( " (EREMOTEIO)"	);
    when EDQUOT:		strRet.appStrRaw( " (EDQUOT)"		);
    when ERESTARTSYS:		strRet.appStrRaw( " (ERESTARTSYS)"	);
    when ERESTARTNOINTR:	strRet.appStrRaw( " (ERESTARTNOINTR)"	);
    when ERESTARTNOHAND:	strRet.appStrRaw( " (ERESTARTNOHAND)"	);
    when ENOIOCTLCMD:		strRet.appStrRaw( " (ENOIOCTLCMD)"	);
#undef	when
    }
}
