// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: TraceMain.cc
//   Main program for mec-trace.
//
// File Created:	21 Jun 1994		Michael Chastain
// Last Edited:		14 Sep 1995		Michael Chastain

// Program version.
static const char strVersion [] = "0.1";



// Unix library.
#include <stdio.h>
#include <stdlib.h>

// My library classes.
#include <EvHistory.h>
#include <MmFlat.h>
#include <MmMap.h>
#include <MmRet.h>
#include <PrProc.h>
#include <WhAbort.h>
#include <WhFile.h>

// My application classes.
#include <TraceArg.h>



// Override builtin 'new' and 'delete'.
//   g++ 2.6.0: profiler won't profile the builtins.
void *	operator new		( size_t s )        { return ::malloc( s ); }
void *	operator new    []	( size_t s )        { return ::malloc( s ); }
void	operator delete		( void *p, size_t ) { ::free( p ); }
void	operator delete []	( void *p, size_t ) { ::free( p ); }



// Local functions.
MmRet	do_trace		( const TraceArg & );



// Entry point.
int main( int argc, const char * const * argv, const char * const * envp )
{
    // Collect arguments.
    TraceArg argTrace( argc, argv, envp );

    // Handle argument errors.
    if ( argTrace.isError( ) )
    {
	argTrace.showUsage( strVersion );
	return 1;
    }

    // Handle '-h' (help).
    if ( argTrace.hasHelp( ) )
    {
	argTrace.showUsage( strVersion );
	return 0;
    }

    // Trace.
    const MmRet mmRet = do_trace( argTrace );
    if ( mmRet != mmRetOk )
    {
	::fflush  ( stdout );
	::fprintf ( stderr, "main: MmRet code %d.\n", int( mmRet ) );
	return 2;
    }

    // That's all, folks.
    return 0;
}



// Trace a process.
MmRet do_trace( const TraceArg & argTrace )
{
    // Spawn a child.
    PrProc procTrace;
    MmMap mapTrace;
    procTrace.spawnProc(
	argTrace.getTargetName( ),
	argTrace.getTargetArgv( ),
	argTrace.getTargetEnvp( ) );
    procTrace.setFollow( true, true );

    // Open output file.
    WhFile fileOut;
    MmRetCheck( fileOut.openName( argTrace.getFileOut( ),
	WhFile::tyOpenWriteData ) );

    // Declare history.
    EvHistory evhTrace;

    // Declare wait-list of children.
    WhList <PrProc *> lpProcWait;
    lpProcWait.append( &procTrace );

    // Monitor child's behaviour.
    for ( bool fExit = false; !fExit; )
    {
	// Wait for child.
	PrProc * pProcWait;
	PrProc::waitProc( lpProcWait, pProcWait );
	if ( pProcWait != &procTrace )
	    return MmRetRaise( mmRetTraceWaitMismatch );

	// Process trace input.
	switch ( procTrace.getExecState( ) )
	{
	default:
	    WhAbort( "do_trace: bad state." );

	case PrProc::stExecExitNormal:
	    ::printf( "Exit: status %d.\n", procTrace.getExitNormal( ) );
	    fExit = true;
	    break;

	case PrProc::stExecExitSignal:
	    ::printf( "Exit: signal %d.\n", procTrace.getExitSignal( ) );
	    fExit = true;
	    break;

	case PrProc::stExecStopSignal:
	    ::printf( "Signal: %d.\n", procTrace.getStopSignal( ) );
	    procTrace.contProc( );
	    break;

	case PrProc::stExecStopSyscall:
	    MmRetCheck( evhTrace.fetch( procTrace, mapTrace,
		argTrace.hasTime( ) ) );
	    procTrace.contProc( );
	    break;
	}
    }

    // Normal exit.
    MmFlat flatOut;
    evhTrace.toFlat( flatOut );
    MmRetCheck( flatOut.toFile( fileOut ) );
    MmRetCheck( fileOut.closeFile( ) );
    return mmRetOk;
}
