// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ControlMain.cc
//   Main program for mec-control.
//
// File Created:	05 Jun 1995		Michael Chastain
// Last Edited:		18 Nov 1995		Michael Chastain

// Program version.
static const char acVersion [] = "0.3";



// Unix library.
#include <sys/wait.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// My library classes.
#include <CxStore.hh>
#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <ErUser.hh>
#include <EvBase.hh>
#include <EvLog.hh>
#include <EvType.hh>
#include <MmDs.hh>
#include <PrProc.hh>
#include <PrSd.hh>
#include <PrSl.hh>
#include <WhFlatIn.hh>
#include <WhImage.hh>
#include <WhLap.hh>
#include <WhString.hh>

// My application classes.
#include <ControlArg.hh>
#include <ControlExecer.hh>
#include <ControlReplay.hh>
#include <ControlState.hh>
#include <ControlViewer.hh>
#include <ControlWait.hh>



// Override builtin 'new' and 'delete'.
//   g++ 2.6.0: profiler won't profile the builtins.
void *	operator new		( size_t s )        { return ::malloc( s ); }
void *	operator new    []	( size_t s )        { return ::malloc( s ); }
void	operator delete		( void *p, size_t ) { ::free( p ); }
void	operator delete []	( void *p, size_t ) { ::free( p ); }



// Local functions.
int	do_control		( const ControlArg & );



// Entry point.
int main( int, const char * argv [] )
{
    // Collect arguments.
    ControlArg argControl( argv );

    // Handle argument errors.
    if ( argControl.isError( ) )
    {
	argControl.showUsage( acVersion );
	return 2;
    }

    // Handle '-h' (help).
    if ( argControl.hasHelp( ) )
    {
	argControl.showUsage( acVersion );
	return 0;
    }

    // Prohibit root access.
    //   ... because root cannot annul fork(2).
    if ( ::getuid( ) == 0 )
    {
	ErUser( argControl.getNameSelf( ), "", "", "root not allowed" );
	return 2;
    }

    // Control.
    return do_control( argControl );
}



// Control processes.
int do_control( const ControlArg & argControl )
{
    // Get and check name of execer.
    const char * pcNameExecer = ::getenv( "EXECER" );
    if ( pcNameExecer == 0 )
    {
	ErUser( argControl.getNameSelf( ), "environment variable",
	    "EXECER", "not found" );
	return 1;
    }
    if ( ::access( pcNameExecer, X_OK ) != 0 )
    {
	ErUser( argControl.getNameSelf( ), "execer run file",
	    pcNameExecer, ::strerror( ::errno ) );
	return 1;
    }

    // Map trace file into image.
    WhImage * pimageTrace = new WhImage( argControl.getNameTrace( ) );
    if ( pimageTrace == 0 )
	ErMem( );
    if ( pimageTrace->isError( ) )
    {
	ErUser( argControl.getNameSelf( ), "trace file",
	    argControl.getNameTrace( ), pimageTrace->getError( ) );
	delete pimageTrace;
	return 1;
    }

    // Make data stash.
    WhString strDirStashData;
    strDirStashData.appStrRaw( "/tmp" );
    strDirStashData.appStrRaw( "/stash-" );
    strDirStashData.appIntFmt( ::getpid( ) );
    PrSd stashData( strDirStashData, WhString( pcNameExecer ) );

    // Get events.
    WhLap <EvBase> * plev = new WhLap <EvBase>;
    if ( plev == 0 )
	ErMem( );

    {
	// Transfer image to flat.
	WhFlatIn flatTrace( pimageTrace );

	// Check tag.
	{
	    int  iTag;
	    if ( flatTrace.countLeft( ) < int( sizeof(int) )
	    || ( flatTrace.getInt( iTag ), iTag != 0x0300FACE ) )
	    {
		ErUser( argControl.getNameSelf( ), "input file",
		    argControl.getNameTrace( ), "not a trace file" );
		delete plev;
		return 1;
	    }
	}

	// Input events.
	while ( !flatTrace.isAtEnd( ) )
	{
	    // Create event.
	    int ityEvent;
	    flatTrace.getInt( ityEvent );
	    EvBase * pevIn = TyEventCreateBase( TyEvent( ityEvent ) );
	    if ( pevIn == 0 )
		ErPtr( );

	    // Get event.
	    pevIn->fromFlat( flatTrace );

	    // Unreplayable event?
	    if ( pevIn->isModelFail( ) )
	    {
		ErUser( argControl.getNameSelf( ), "trace file",
		    argControl.getNameTrace( ), "has unreplayable events" );
		delete pevIn;
		delete plev;
		return 1;
	    }

	    // Stash datasets.
	    for ( int ids = 0; ids < pevIn->countDs( ); ++ids )
		stashData.stashFile( pevIn->getDs( ids ) );

	    // Save event.
	    plev->appendAp( pevIn );
	}

	// Check exec dataset.
	if ( stashData.countExec( ) == 0 )
	{
	    ErFatal( "do_control: missing exec dataset." );
	    delete plev;
	    return 2;
	}

	// Check exec dataset.
	if ( stashData.countExec( )  > 1 )
	{
	    ErFatal( "do_control: multiple exec datasets." );
	    delete plev;
	    return 2;
	}
    }

    // Create controlled process list.
    WhLap <CxStore> lcxControl;

    // Spawn a viewer.
    {
	WhString strNameExecDataNul( stashData.getNameExecData( 0 ) );
	strNameExecDataNul.appChrRaw( '\0' );
	strNameExecDataNul.checkCcs( );

	const int nargViewer = argControl.countArgViewer( );
	const char ** argvViewer = new const char * [nargViewer + 2];
	if ( argvViewer == 0 )
	    ErMem( );
	for ( int iargViewer = 0; iargViewer < nargViewer; ++iargViewer )
	    argvViewer[iargViewer] = argControl.getArgViewer( iargViewer );
	argvViewer[nargViewer+0] = strNameExecDataNul.address( );
	argvViewer[nargViewer+1] = 0;

	PrProc * pprocViewer = new PrProc( argvViewer );
	if ( pprocViewer == 0 )
	    ErMem( );
	delete [] argvViewer;

	WhString strDirStashLink;
	strDirStashLink.appStrRaw( "/tmp" );
	strDirStashLink.appStrRaw( "/stash-" );
	strDirStashLink.appIntFmt( pprocViewer->getPid( ) );
	PrSl * pstashLinkViewer = new PrSl( strDirStashLink );
	if ( pstashLinkViewer == 0 )
	    ErMem( );

	EvLog * plogViewer = new EvLog( plev );
	if ( plogViewer == 0 )
	    ErMem( );

	CxStore * pcxViewer = new CxStore( stTrackViewer, stRunRunning,
	    pprocViewer, stashData, pstashLinkViewer, plogViewer );
	lcxControl.appendAp( pcxViewer );
    }

    // Run and control.
    while ( lcxControl.count( ) > 0 )
    {
	{
	    // Wait.
	    CxStore & cxWait = PrProc::waitMulti( lcxControl );
	    if ( cxWait.getRunState( ) != stRunRunning )
		ErFatal( "do_control: bad state." );
	    cxWait.setRunState( stRunRunnable );

	    // Set run state set from event type.
	    switch ( cxWait.getProc( ).getTyEvent( ) )
	    {
	    default:
	    case tyEvBlank:
		ErAbort( "do_control: bad state." );

	    case tyEvFirst:
		break;

	    case tyEvExit:
		cxWait.setRunState( stRunExited );
		if ( cxWait.getProc( ).getPpid( ) == ::getpid( ) )
		{
		    if ( !cxWait.getProc( ).hasExeced( ) )
		    {
			// Failed.
			int istatusExit = cxWait.getProc( ).getStatusWait( );
			ErUser( argControl.getNameSelf( ), "viewer run file",
			    argControl.getArgViewer( 0 ),
			    ::strerror( WEXITSTATUS( istatusExit ) ) );
			return 2;
		    }
		    cxWait.setRunState( stRunDead );
		}
		break;

	    case tyEvExec:
	    case tyEvSignal:
	    case tyEvBpt:
	    case tyEvStep:
		if ( cxWait.getTrackState( ) != stTrackExecer && cxWait.isTraceMe( ) )
		    cxWait.setRunState( stRunStopped );
		break;

	    case tyEvSci:
	    case tyEvSco:
		if ( cxWait.getTrackState( ) == stTrackReplay )
		    cxWait.stepSc( );
		if ( cxWait.getTrackState( ) != stTrackExecer && cxWait.isTraceSc( ) )
		    cxWait.setRunState( stRunStopped );
		break;
	    }

	    // Filter.
	    switch ( cxWait.getTrackState( ) )
	    {
	    default:		ErAbort( "do_control: bad state." );
	    case stTrackViewer:	viewer_aw( cxWait, lcxControl ); break;
	    case stTrackExecer: execer_aw( cxWait, lcxControl ); break;
	    case stTrackReplay: replay_aw( cxWait, lcxControl ); break;
	    }
	}

	// Handle waiting processes.
	//   'wait' can cascade so must handle children before parents.
	for ( int icx1 = lcxControl.count( ) - 1; icx1 >= 0; --icx1 )
	{
	    CxStore & cxParent = lcxControl.refNonConst( icx1 );
	    if ( cxParent.getRunState( ) == stRunWaiting )
		control_wait( cxParent, lcxControl );
	}

	// Remove dead processes.
	for ( int icx2 = lcxControl.count( ) - 1; icx2 >= 0; --icx2 )
	{
	    if ( lcxControl.refNonConst( icx2 ).getRunState( ) == stRunDead )
		lcxControl.remove( icx2 );
	}

	// Continue runnable processes.
	for ( int icx3 = lcxControl.count( ) - 1; icx3 >= 0; --icx3 )
	{
	    CxStore & cxRunnable = lcxControl.refNonConst( icx3 );

	    if ( cxRunnable.getRunState( ) == stRunRunnable )
	    {
		// Filter.
		switch ( cxRunnable.getTrackState( ) )
		{
		default:	    ErAbort( "do_control: bad state." );
		case stTrackViewer: viewer_bc( cxRunnable, lcxControl ); break;
		case stTrackExecer: execer_bc( cxRunnable, lcxControl ); break;
		case stTrackReplay: replay_bc( cxRunnable, lcxControl ); break;
		}

		// Step or continue.
		if ( cxRunnable.nextIsStep( cxRunnable.getTrackState( ) == stTrackReplay ) )
		    cxRunnable.getProc( ).ptraceStep( );
		else
		    cxRunnable.getProc( ).ptraceCont( );

		// State transition.
		cxRunnable.setRunState( stRunRunning );
	    }
	}
    }

    // That's all, folks.
    return 0;
}
