// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ControlWait.cc
//   Wait handler.
//
// File Created:	02 Nov 1995		Michael Chastain
// Last Edited:		16 Nov 1995		Michael Chastain

#include <linux/unistd.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <errno.h>
#include <unistd.h>

#include <CxStore.hh>
#include <ErFatal.hh>
#include <MmScr.hh>
#include <MmSeg.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <PrProc.hh>
#include <TySegLix.hh>
#include <WhList.hh>

#include <ControlState.hh>
#include <ControlWait.hh>



// Wait handler.
void control_wait( CxStore & cxParent, WhLap <CxStore> & lcxControl )
{
    // Declare process.
    PrProc & procParent = cxParent.getProc( );

    // Get arguments.
    const MmWord wpidArg     = cxParent.getArgSci( 0 );
    const MmWord waddrStatus = cxParent.getArgSci( 1 );
    const MmWord wOption     = cxParent.getArgSci( 2 );
          MmWord waddrRusage = 0;

    // Check sys entry.
    switch ( procParent.getSysEntry( ) )
    {
    default:           ErFatal( "control_wait: bad sys entry." );
    case __NR_wait4:   waddrRusage = cxParent.getArgSci( 3 ); break;
    case __NR_waitpid: waddrRusage = 0;                       break;
    }

    // Check options I can't handle.
    if ( ( wOption & (__WCLONE|WUNTRACED) ) != 0 )
	ErFatal( "control_wait: cannot handle option." );

    // Synthetic return value.
    MmScr scrParent( MmScr::tyScrInt );
    bool fRelease        = false;
    PrProc * pprocTarget = 0;

    // Validate address of status.
    const MmAddr addrStatus = MmAddr( waddrStatus );
    if ( !fRelease )
    {
	if ( addrStatus != 0 )
	{
	    MmWord wStatus;
	    if ( !procParent.fetchWord( tyMmData, addrStatus, wStatus )
	    ||   !procParent.storeWord( tyMmData, addrStatus, wStatus ) )
	    {
		scrParent.setError( EFAULT );
		fRelease = true;
	    }
	}
    }

    // Validate address of rusage.
    const MmAddr addrRusage = MmAddr( waddrRusage );
    if ( !fRelease )
    {
	if ( addrRusage != 0 )
	{
	    WhList <MmWord> lwRusage;
	    if ( !procParent.fetchListWord( tyMmData, addrRusage,
		sizeof(struct rusage), lwRusage )
	    ||   !procParent.storeListWord( tyMmData, addrRusage,
		lwRusage.count( ) * sizeof(MmWord), lwRusage ) )
	    {
		scrParent.setError( EFAULT );
		fRelease = true;
	    }
	}
    }

    // Look for children.
    bool fChild = false;
    for ( int icx = 0; !fRelease && icx < lcxControl.count( ); ++icx )
    {
	CxStore & cxTry   = lcxControl.refNonConst( icx );
	PrProc  & procTry = cxTry.getProc( );

	if ( procTry.getPpid( ) == procParent.getPid( ) )
	{
	    // Match pid argument.
	    const int ipidArg = int( wpidArg );
	    if ( ipidArg  >  0 && procTry.getPid  ( ) == wpidArg
	    ||   ipidArg ==  0 && procTry.getPgid ( ) == procParent.getPgid ( )
	    ||   ipidArg == -1 && true
	    ||   ipidArg  < -1 && procTry.getPgid ( ) == 0 - wpidArg )
	    {
		fChild = true;

		switch ( cxTry.getRunState( ) )
		{
		case stRunExited:
		    cxTry.setRunState( stRunDead );
		    pprocTarget  = &procTry;
		    fRelease     = true;
		    break;

		case stRunStopped:
		    cxTry.setRunState( stRunReady );
		    pprocTarget = &procTry;
		    fRelease    = true;
		    break;
		}
	    }
	}
    }

    // Handle no child.
    if ( !fRelease )
    {
	if ( !fChild )
	{
	    scrParent.setError( ECHILD );
	    fRelease = true;
	}
    }

    // Handle no-hang option.
    if ( !fRelease )
    {
	if ( ( wOption & WNOHANG ) != 0 )
	{
	    scrParent.setValue( 0, 0 );
	    fRelease = true;
	}
    }

    // Release parent?
    if ( fRelease )
    {
	// Store values for success.
	if ( pprocTarget != 0 )
	{
	    // Store rusage.
	    if ( addrRusage != 0 )
	    {
		MmSeg segRusage;
		segRusage.fetchProc( procParent, tyMmData, tySegRusage,
		    addrRusage, 1 );
		if ( !segRusage.hasArea( ) || !segRusage.hasData( ) )
		    ErFatal( "control_wait: failed fetch." );
		segRusage.smashData(
		    (const MmByte *) & pprocTarget->getRusage( ),
		    sizeof(struct rusage ) );
		segRusage.storeProc( procParent );
	    }

	    // Store status.
	    if ( addrStatus != 0 )
	    {
		if ( !procParent.storeWord( tyMmData, addrStatus,
		    pprocTarget->getStatusWait( ) ) )
		{
		    ErFatal( "control_wait: failed store." );
		}
	    }

	    // Set return value.
	    scrParent.setValue( pprocTarget->getPid( ), 0 );
	}

	// Store into parent.
	scrParent.storeProc( procParent );
	procParent.storeReg( 0, wpidArg     );
	procParent.storeReg( 1, waddrStatus );

	// State transition.
	cxParent.setRunState( stRunRunnable );
	if ( cxParent.getTrackState( ) != stTrackExecer && cxParent.isTraceSc( ) )
	    cxParent.setRunState( stRunStopped );
    }
}
