// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: CxStore.hh
//   Context for store.
//   This is a reference class.
//
// File Created:	01 Nov 1995		Michael Chastain
// Last Edited:		18 Nov 1995		Michael Chastain

#if !defined(CX_STORE_HH)
#define CX_STORE_HH

#include <EvLog.hh>
#include <MmSeg.hh>
#include <PrBpt.hh>
#include <PrProc.hh>
#include <PrSd.hh>
#include <PrSl.hh>
#include <WhLap.hh>
#include <WhList.hh>



class	CxStore
{
    // Life cycle methods.
    public:
				CxStore		( int, int,
						  PrProc * &,
						  const PrSd &,
						  PrSl   * &,
						  EvLog  * &	   );
				~CxStore	(		   );

    // Forbidden copiers.
    private:
				CxStore		( const CxStore &  );
	void			operator =	( const CxStore &  );

    // Accessors.
    public:
	int			getTrackState	(		   ) const;
	int			getRunState	(		   ) const;
	bool			isTraceMe	(		   ) const;
	bool			isTraceSc	(		   ) const;
	bool			isStep		(		   ) const;
	MmAddr			getAddrCont	(		   ) const;
	MmWord			getArgSci	( int		   ) const;
	const MmSeg &		getSegSci	( int		   ) const;

    // Mutators.
    public:
	void			setTrackState	( int		   );
	void			setRunState	( int		   );
	void			setTraceMe	( bool		   );
	void			setTraceSc	( bool		   );
	void			setStep		( bool		   );
	void			stepSc		(		   );
	bool			nextIsStep	( bool		   );
	void			setAddrCont	( MmAddr	   );
	void			setSci		( const WhList <MmWord> &,
						  const WhLap  <MmSeg>  &
						  		   );

    // Component accessors and mutators.
    public:
	const PrProc &		getProc		(		   ) const;
	PrProc &		getProc		(		   );
	const PrBpt &		getBpt		(		   ) const;
	PrBpt &			getBpt		(		   );
	const PrSd &		getStashData	(		   ) const;
	const PrSl &		getStashLink	(		   ) const;
	PrSl &			getStashLink	(		   );
	const EvLog &		getLog		(		   ) const;
	EvLog &			getLog		(		   );

    // Instance data.
    private:
	int			istTrack_;	// Track state.
	int			istRun_;	// Run state.
	bool			fTraceMe_;	// Trace me.
	bool			fTraceSc_;	// Trace my system calls.
	bool			fStepWant_;	// Wants to step.
	int			nStepWait_;	// # events to wait to step.
	MmAddr			addrCont_;	// Continuation address.
	WhList <MmWord>		lwArgSci_;	// Args to last sci.
	WhLap  <MmSeg>		lsegSci_;	// Segs to last sci.
	PrProc *		pproc_;		// Process.
	PrBpt			bpt_;		// Breakpoints.
	const PrSd &		stashData_;	// Data stash.
	PrSl *			pstashLink_;	// Link stash.
	EvLog *			plog_;		// Log.
};



// Inline methods.

inline int CxStore::getTrackState( ) const
{
    return istTrack_;
}

inline int CxStore::getRunState( ) const
{
    return istRun_;
}

inline bool CxStore::isTraceMe( ) const
{
    return fTraceMe_;
}

inline bool CxStore::isTraceSc( ) const
{
    return fTraceSc_;
}

inline MmAddr CxStore::getAddrCont( ) const
{
    return addrCont_;
}

inline MmWord CxStore::getArgSci( int iArgSci ) const
{
    return lwArgSci_[iArgSci];
}

inline const MmSeg & CxStore::getSegSci( int isegSci ) const
{
    return lsegSci_[isegSci];
}

inline void CxStore::setTrackState( int istTrack )
{
    istTrack_ = istTrack;
}

inline void CxStore::setRunState( int istRun )
{
    istRun_ = istRun;
}

inline void CxStore::setTraceMe( bool fTraceMe )
{
    fTraceMe_ = fTraceMe;
}

inline void CxStore::setTraceSc( bool fTraceSc )
{
    fTraceSc_ = fTraceSc;
}

inline void CxStore::setStep( bool fStepWant )
{
    fStepWant_ = fStepWant;
    nStepWait_ = 0;
}

inline void CxStore::stepSc( )
{
    if ( nStepWait_ > 0 )
	--nStepWait_;
}

inline void CxStore::setAddrCont( MmAddr addrCont )
{
    addrCont_ = addrCont;
}

inline const PrProc & CxStore::getProc( ) const
{
    return *pproc_;
}

inline PrProc & CxStore::getProc( )
{
    return *pproc_;
}

inline const PrBpt & CxStore::getBpt( ) const
{
    return bpt_;
}

inline PrBpt & CxStore::getBpt( )
{
    return bpt_;
}

inline const PrSd & CxStore::getStashData( ) const
{
    return stashData_;
}

inline const PrSl & CxStore::getStashLink( ) const
{
    return *pstashLink_;
}

inline PrSl & CxStore::getStashLink( )
{
    return *pstashLink_;
}

inline const EvLog & CxStore::getLog( ) const
{
    return *plog_;
}

inline EvLog & CxStore::getLog( )
{
    return *plog_;
}



#endif
