// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ExeLix.cc
//   Interpret exe and lib files.
//
// File Created:	06 May 1995		Michael Chastain
// Last Reviewed:	08 Sep 1995		Michael Chastain
// Last Edited:		07 Nov 1995		Michael Chastain

#include <linux/a.out.h>

#include <CxFetch.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <EvBase.hh>
#include <Exe.hh>
#include <MmArea.hh>
#include <MmDs.hh>
#include <MmType.hh>



// Page alignment functions.
//   Linux 1.3.24: see 'PAGE_ALIGN' in 'include/asm-i386/page.h'.
inline int    PAGE_ALIGN ( int    a ) { return (a + (4096-1)) &~ (4096-1); }
inline MmAddr PAGE_ALIGN ( MmAddr a ) { return MmAddr(PAGE_ALIGN(int(a))); }



// Helper functions.
static bool ExeToAreaAout ( const MmDs &, CxFetch &, EvBase & );
static bool ExeToAreaElf  ( const MmDs &, CxFetch &, EvBase & );
static bool LibToAreaAout ( const MmDs &, CxFetch &, EvBase & );
static bool LibToAreaElf  ( const MmDs &, CxFetch &, EvBase & );



// Extract area list from an executable.
void ExeToArea( const MmDs & dsExe, CxFetch & cxFetch, EvBase & evTarget )
{
    if ( !ExeToAreaAout ( dsExe, cxFetch, evTarget )
    &&   !ExeToAreaElf  ( dsExe, cxFetch, evTarget ) )
	ErFatal( "ExeToArea: bad format." );
}



// Extract area list from an executable.
//   Linux 1.3.24: see 'load_aout_binary' in 'fs/exec.c'.
static bool ExeToAreaAout( const MmDs & dsExe, CxFetch & cxFetch, EvBase & evTarget )
{
    // Get reference to header.
    const char * pcExe = dsExe.address( );
    const int ncExe = dsExe.count( );
    if ( ncExe < int( sizeof(struct exec) ) )
	return false;
    const struct exec & exec = * (const struct exec *) pcExe;

    // Check header.
    if ( N_MAGIC(exec) != OMAGIC
    &&   N_MAGIC(exec) != ZMAGIC
    &&   N_MAGIC(exec) != QMAGIC )
	return false;

    // Check header.
    if ( ncExe < int( N_TXTOFF(exec) + exec.a_text + exec.a_data ) )
	ErFatal( "ExeToAreaAout: file short." );

    // Map over previous areas.
    {
	MmArea * pareaClear = new MmArea;
	if ( pareaClear == 0 )
	    ErMem( );
	pareaClear->setCover( tyMmBlank );
	cxFetch.getMap( ).mergeArea ( *pareaClear );
	evTarget.appArea            (  pareaClear );
    }

    // Map text and data areas.
    if ( N_MAGIC(exec) == OMAGIC )
    {
	// Map combined text+data area.
	const MmAddr addrTextData = MmAddr( N_TXTADDR(exec) );
	const int sTextData = PAGE_ALIGN( exec.a_text + exec.a_data );
	MmArea * pareaTextData = new MmArea( tyMmData,
	    addrTextData, addrTextData + sTextData );
	if ( pareaTextData == 0)
	    ErMem( );
	cxFetch.getMap( ).mergeArea ( *pareaTextData );
	evTarget.appArea            (  pareaTextData );
    }
    else
    {
	// Map text area.
	const MmAddr addrText = MmAddr( N_TXTADDR(exec) );
	const int sText = PAGE_ALIGN( exec.a_text );
	MmArea * pareaText = new MmArea( tyMmText,
	    addrText, addrText + sText );
	if ( pareaText == 0 )
	    ErMem( );
	cxFetch.getMap( ).mergeArea ( *pareaText );
	evTarget.appArea            (  pareaText );

	// Map data area.
	const MmAddr addrData = addrText + sText;
	const int sData = PAGE_ALIGN( exec.a_data );
	MmArea * pareaData = new MmArea( tyMmData,
	    addrData, addrData + sData );
	if ( pareaData == 0 )
	    ErMem( );
	cxFetch.getMap( ).mergeArea ( *pareaData );
	evTarget.appArea            (  pareaData );
    }

    // Map bss area.
    {
	const MmAddr addrDataEnd =
	    MmAddr( N_TXTADDR(exec) + exec.a_text + exec.a_data );
	const MmAddr addrBss = PAGE_ALIGN( addrDataEnd );
	const int sBss = PAGE_ALIGN( addrDataEnd + exec.a_bss ) - addrBss;
	MmArea * pareaBss = new MmArea( tyMmBss, addrBss, addrBss + sBss );
	if ( pareaBss == 0 )
	    ErMem( );
	cxFetch.getMap( ).mergeArea ( *pareaBss );
	evTarget.appArea            (  pareaBss );
    }

    // That's all, folks.
    return true;
}



// Extract area list from an executable.
//   Not implemented yet.
static bool ExeToAreaElf( const MmDs &, CxFetch &, EvBase & )
{
    return false;
}



// Extract area list from a library.
void LibToArea( const MmDs & dsLib, CxFetch & cxFetch, EvBase & evTarget )
{
    if ( !LibToAreaAout ( dsLib, cxFetch, evTarget )
    &&   !LibToAreaElf  ( dsLib, cxFetch, evTarget ) )
	ErFatal( "LibToArea: bad format." );
}



// Extract area list from a library.
//   Linux 1.3.24: see 'load_aout_library' in 'fs/exec.c'.
static bool LibToAreaAout( const MmDs & dsLib, CxFetch & cxFetch, EvBase & evTarget )
{
    // Get reference to header.
    const char * pcLib = dsLib.address( );
    const int ncLib = dsLib.count( );
    if ( ncLib < int( sizeof(struct exec) ) )
	return false;
    const struct exec & exec = * (const struct exec *) pcLib;

    // Check header.
    if ( N_MAGIC(exec) != ZMAGIC && N_MAGIC(exec) != QMAGIC )
	return false;

    // Check header.
    if ( ncLib < int( N_TXTOFF(exec) + exec.a_text + exec.a_data ) )
	ErFatal( "LibToAreaAout: file short." );

    // Map combined text+data area.
    const MmAddr addrTextData = MmAddr( exec.a_entry &~ (4096-1) );
    const int sTextData = PAGE_ALIGN( exec.a_text + exec.a_data );
    MmArea * pareaTextData = new MmArea( tyMmLib,
	addrTextData, addrTextData + sTextData );
    if ( pareaTextData == 0 )
	ErMem( );
    cxFetch.getMap( ).mergeArea ( *pareaTextData );
    evTarget.appArea            (  pareaTextData );

    // Map bss area.
    const MmAddr addrBss = addrTextData + sTextData;
    const int sBss =
	PAGE_ALIGN( exec.a_text + exec.a_data + exec.a_bss ) - sTextData;
    MmArea * pareaBss = new MmArea( tyMmLib, addrBss, addrBss + sBss );
    if ( pareaBss == 0 )
	ErMem( );
    cxFetch.getMap( ).mergeArea ( *pareaBss );
    evTarget.appArea            (  pareaBss );

    // That's all, folks.
    return true;
}



// Extract area list from a library.
//   Not implemented yet.
static bool LibToAreaElf( const MmDs &, CxFetch &, EvBase & )
{
    return false;
}
