// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ExecMain.cc
//   Main program for mec-exec.
//
// File Created:	28 Oct 1995		Michael Chastain
// Last Edited:		12 Nov 1995		Michael Chastain

// Program version.
static const char acVersion [] = "0.3";



// Unix library.
#include <sys/time.h>
#include <sys/resource.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>



// Library classes.
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErUser.hh>
#include <WhFlatIn.hh>



// Entry point.
int main( int argc, const char * const * argv )
{
    // Check args.
    if ( argc != 1 )
    {
	ErUser( argv[0], "", "", "arguments not allowed" );
	return 2;
    }

    // Prohibit root access.
    //   ... because root cannot annul fork(2).
    if ( ::getuid( ) == 0 )
    {
	ErUser( argv[0], "", "", "root not allowed" );
	return 2;
    }

    // Annul future forks.
    {
	struct rlimit rlimit;
	rlimit.rlim_cur = 0;
	rlimit.rlim_max = 0;
	if ( ::setrlimit( RLIMIT_NPROC, &rlimit ) != 0 )
	    ErFatal( "main: failed setrlimit." );
    }

    // Magic read for size.
    int ncMagic = 0;
    if ( ::read( -3, &ncMagic, sizeof(ncMagic) ) != sizeof(ncMagic) )
	ErFatal( "main: failed magic read." );
    if ( ncMagic <= 0 )
	ErFatal( "main: negative or zero count." );

    // Magic read for data.
    char * pcMagic = new char [ncMagic];
    if ( pcMagic == 0 )
	ErMem( );
    if ( ::read( -4, pcMagic, ncMagic ) != ncMagic )
	ErFatal( "main: failed magic read." );

    // Wrap flat around magic data.
    WhFlatIn flatMagic( pcMagic, ncMagic );

    // Get name of stash directory.
    const char * pcDirStash;
    flatMagic.getCcs( pcDirStash );

    // Change directory to stash.
    if ( ::chdir( pcDirStash ) != 0 )
    {
	ErUser( argv[0], "stash directory", pcDirStash, strerror( ::errno ) );
	return 1;
    }

    // Get name of executable.
    const char * pcNameExec;
    flatMagic.getCcs( pcNameExec );

    // Count args.
    int nArg;
    flatMagic.getInt( nArg );
    if ( nArg < 0 )
	ErFatal( "main: negative count." );

    // Get args.
    const char ** argvExec = new const char * [nArg + 1];
    if ( argvExec == 0 )
	ErMem( );
    for ( int iArg = 0; iArg < nArg; ++iArg )
	flatMagic.getCcs( argvExec[iArg] );
    argvExec[nArg] = 0;

    // Count envs.
    int nEnv;
    flatMagic.getInt( nEnv );
    if ( nEnv < 0 )
	ErFatal( "main: negative count." );

    // Get envs.
    const char ** envpExec = new const char * [nEnv + 1];
    if ( envpExec == 0 )
	ErMem( );
    for ( int iEnv = 0; iEnv < nEnv; ++iEnv )
	flatMagic.getCcs( envpExec[iEnv] );
    envpExec[nEnv] = 0;

    // Check end.
    if ( !flatMagic.isAtEnd( ) )
	ErFatal( "main: flat too long." );

    // Exec.
    ::execve( pcNameExec, (char * const *) argvExec, (char * const *) envpExec );
    ErUser( argv[0], "run file", pcNameExec, strerror( ::errno ) );
    return 2;
}
