// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: MmScrSun.cc
//   System call return value.
//   This is a concrete class.
//
// SunOS 4.1.3: I am guessing semantics from field names in 'struct user'!
//
// File Created:	19 Jun 1995		Michael Chastain
// Last Edited:		02 Nov 1995		Michael Chastain

#include <sys/types.h>
// #include <sys/ptrace.h>
#include <sys/user.h>
#include <errno.h>

#include <ErAbort.hh>
#include <ErFatal.hh>
#include <MmScr.hh>
#include <MmSpace.hh>
#include <PrProc.hh>
#include <WhFlatIn.hh>
#include <WhFlatOut.hh>
#include <WhString.hh>



// Constructor.
MmScr::MmScr( TyScr tyScr )
    : tyScr_	( tyScr	)
    , wError_	( 0    	)
    , w1_	( 0    	)
    , w2_	( 0    	)
{
    ;
}



// Flat input.
void MmScr::fromFlat( WhFlatIn & flatIn )
{
    int ityScr;
    flatIn.getInt( ityScr  );
    switch ( ityScr )
    {
    default:
	ErFatal( "MmScr::fromFlat: bad enum." );
    case tyScrUnbound:
    case tyScrAddr:
    case tyScrHex:
    case tyScrInt:
    case tyScrNot:
    case tyScrOct:
	tyScr_ = TyScr( ityScr );
	break;
    }

    flatIn.getWord ( wError_ );
    flatIn.getWord ( w1_     );
    flatIn.getWord ( w2_     );
}



// Flat output.
void MmScr::toFlat( WhFlatOut & flatOut ) const
{
    flatOut.putInt	( int( tyScr_ )	);
    flatOut.putWord	( wError_	);
    flatOut.putWord	( w1_		);
    flatOut.putWord	( w2_		);
}



// Fetch from process.
void MmScr::fetchProc( const PrProc & procFetch )
{
    struct user user;

    const MmAddr addrError = MmAddr( MmAddr(&user.u_error) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( tyMmUser, addrError, wError_ ) )
	ErFatal( "MmScr::fetchProc: failed fetch." );

    const MmAddr addr1 = MmAddr( MmAddr( &user.u_rval1 ) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( tyMmUser, addr1, w1_ ) )
	ErFatal( "MmScr::fetchProc: failed fetch." );

    const MmAddr addr2 = MmAddr( MmAddr( &user.u_rval2 ) - MmAddr( &user ) );
    if ( !procFetch.fetchWord( tyMmUser, addr2, w2_ ) )
	ErFatal( "MmScr::fetchProc: failed fetch." );
}



// Store to process.
void MmScr::storeProc( PrProc & procStore ) const
{
    if ( tyScr_ == tyScrUnbound )
	ErAbort( "MmScr::storeProc: not bound." );

    struct user user;

    const MmAddr addrError = MmAddr( MmAddr(&user.u_error) - MmAddr( &user ) );
    if ( !procStore.storeWord( tyMmUser, addrError, wError_ ) )
	ErFatal( "MmScr::storeProc: failed store." );

    const MmAddr addr1 = MmAddr( MmAddr( &user.u_rval1 ) - MmAddr( &user ) );
    if ( !procStore.storeWord( tyMmUser, addr1, w1_ ) )
	ErFatal( "MmScr::storeProc: failed store." );

    const MmAddr addr2 = MmAddr( MmAddr( &user.u_rval2 ) - MmAddr( &user ) );
    if ( !procStore.storeWord( tyMmUser, addr2, w2_ ) )
	ErFatal( "MmScr::storeProc: failed store." );
}



// Format to string.
void MmScr::fmtScr( WhString & strRet ) const
{
    if ( tyScr_ == tyScrUnbound )
	ErAbort( "MmScr::fmtScr: not bound." );

    if ( isError( ) )
    {
	fmtError( strRet );
    }
    else
    {
	switch ( tyScr_ )
	{
	default:
	    ErAbort( "MmScr::fmtStr: bad enum." );

	case tyScrAddr:
	    strRet.appPtrFmt( MmAddr( w1_ ) );
	    break;

	case tyScrHex:	
	    strRet.appStrRaw( "0x" );
	    strRet.appIntFmt( w1_, 16, 8 );
	    break;

	case tyScrInt:
	    strRet.appIntFmt( w1_ );
	    break;

	case tyScrNot:
	    strRet.appStrRaw( "(UNEXPECTED!) " "0x" );
	    strRet.appIntFmt( w1_, 16, 8 );
	    break;

	case tyScrOct:
	    strRet.appChrRaw( '0' );
	    strRet.appIntFmt( w1_, 8, 0 );
	    break;
	}
    }
    strRet.appChrRaw( '\n' );
}



// Get a word.
MmWord MmScr::getWord( int iWord ) const
{
    if ( tyScr_ == tyScrUnbound )
	ErAbort( "MmScr::getWord: not bound." );
    if ( isError( ) )
	ErAbort( "MmScr::getWord: is error." );

    switch ( iWord )
    {
    default: ErAbort( "MmScr::getWord: bad index." );
    case 0:  return wError_;
    case 1:  return w1_;
    case 2:  return w2_;
    }
}



// Return whether value represents an error.
bool MmScr::isError( ) const
{
    if ( tyScr_ == tyScrUnbound )
	ErAbort( "MmScr::isError: not bound." );

    return ( ( wError_ & 0xFF000000 ) >> 24 ) != 0;
}



// Return whether two values match.
bool MmScr::match( const MmScr & scrMatch ) const
{
    if ( tyScr_ != scrMatch.tyScr_ )
	return false;
    if ( tyScr_ == tyScrUnbound )
	return true;
    return wError_ == scrMatch.wError_ && w1_ == scrMatch.w1_ && w2_ == scrMatch.w2_;
}



// Set value to represent a given error.
void MmScr::setError( MmWord wError )
{
    wError_ = (wError & 0x000000FF) << 24;
    w1_     = 0;
    w2_     = 0;

    if ( !isError( ) )
	ErAbort( "MmScr::setError: not error." );
}



// Set value to represent a given non-error value.
void MmScr::setValue( MmWord w1, MmWord w2 )
{
    wError_ = 0;
    w1_     = w1;
    w2_     = w2;

    if ( isError( ) )
	ErAbort( "MmScr::setValue: not error." );
}



// Format a string from an error value.
void MmScr::fmtError( WhString & strRet ) const
{
    strRet.appIntFmt( ( wError_ & 0xFF000000 ) >> 24 );
    switch ( ( wError_ & 0xFF000000 ) >> 24 )
    {
#define	when  break; case
    default:			strRet.appStrRaw( " (UNKNOWN)"		);
    case EPERM:			strRet.appStrRaw( " (EPERM)"		);
    case ENOENT:		strRet.appStrRaw( " (ENOENT)"		);
    case ESRCH:			strRet.appStrRaw( " (ESRCH)"		);
    case EINTR:			strRet.appStrRaw( " (EINTR)"		);
    case EIO:			strRet.appStrRaw( " (EIO)"		);
    case ENXIO:			strRet.appStrRaw( " (ENXIO)"		);
    case E2BIG:			strRet.appStrRaw( " (E2BIG)"		);
    case ENOEXEC:		strRet.appStrRaw( " (ENOEXEC)"		);
    case EBADF:			strRet.appStrRaw( " (EBADF)"		);
    case ECHILD:		strRet.appStrRaw( " (ECHILD)"		);
    case EAGAIN:		strRet.appStrRaw( " (EAGAIN)"		);
    case ENOMEM:		strRet.appStrRaw( " (ENOMEM)"		);
    case EACCES:		strRet.appStrRaw( " (EACCES)"		);
    case EFAULT:		strRet.appStrRaw( " (EFAULT)"		);
    case ENOTBLK:		strRet.appStrRaw( " (ENOTBLK)"		);
    case EBUSY:			strRet.appStrRaw( " (EBUSY)"		);
    case EEXIST:		strRet.appStrRaw( " (EEXIST)"		);
    case EXDEV:			strRet.appStrRaw( " (EXDEV)"		);
    case ENODEV:		strRet.appStrRaw( " (ENODEV)"		);
    case ENOTDIR:		strRet.appStrRaw( " (ENOTDIR)"		);
    case EISDIR:		strRet.appStrRaw( " (EISDIR)"		);
    case EINVAL:		strRet.appStrRaw( " (EINVAL)"		);
    case ENFILE:		strRet.appStrRaw( " (ENFILE)"		);
    case EMFILE:		strRet.appStrRaw( " (EMFILE)"		);
    case ENOTTY:		strRet.appStrRaw( " (ENOTTY)"		);
    case ETXTBSY:		strRet.appStrRaw( " (ETXTBSY)"		);
    case EFBIG:			strRet.appStrRaw( " (EFBIG)"		);
    case ENOSPC:		strRet.appStrRaw( " (ENOSPC)"		);
    case ESPIPE:		strRet.appStrRaw( " (ESPIPE)"		);
    case EROFS:			strRet.appStrRaw( " (EROFS)"		);
    case EMLINK:		strRet.appStrRaw( " (EMLINK)"		);
    case EPIPE:			strRet.appStrRaw( " (EPIPE)"		);
    case EDOM:			strRet.appStrRaw( " (EDOM)"		);
    case ERANGE:		strRet.appStrRaw( " (ERANGE)"		);
    case EWOULDBLOCK_sys:	strRet.appStrRaw( " (EWOULDBLOCK_sys)"	);
    case EINPROGRESS:		strRet.appStrRaw( " (EINPROGRESS)"	);
    case EALREADY:		strRet.appStrRaw( " (EALREADY)"		);
    case ENOTSOCK:		strRet.appStrRaw( " (ENOTSOCK)"		);
    case EDESTADDRREQ:		strRet.appStrRaw( " (EDESTADDRREQ)"	);
    case EMSGSIZE:		strRet.appStrRaw( " (EMSGSIZE)"		);
    case EPROTOTYPE:		strRet.appStrRaw( " (EPROTOTYPE)"	);
    case ENOPROTOOPT:		strRet.appStrRaw( " (ENOPROTOOPT)"	);
    case EPROTONOSUPPORT:	strRet.appStrRaw( " (EPROTONOSUPPORT)"	);
    case ESOCKTNOSUPPORT:	strRet.appStrRaw( " (ESOCKTNOSUPPORT)"	);
    case EOPNOTSUPP:		strRet.appStrRaw( " (EOPNOTSUPP)"	);
    case EPFNOSUPPORT:		strRet.appStrRaw( " (EPFNOSUPPORT)"	);
    case EAFNOSUPPORT:		strRet.appStrRaw( " (EAFNOSUPPORT)"	);
    case EADDRINUSE:		strRet.appStrRaw( " (EADDRINUSE)"	);
    case EADDRNOTAVAIL:		strRet.appStrRaw( " (EADDRNOTAVAIL)"	);
    case ENETDOWN:		strRet.appStrRaw( " (ENETDOWN)"		);
    case ENETUNREACH:		strRet.appStrRaw( " (ENETUNREACH)"	);
    case ENETRESET:		strRet.appStrRaw( " (ENETRESET)"	);
    case ECONNABORTED:		strRet.appStrRaw( " (ECONNABORTED)"	);
    case ECONNRESET:		strRet.appStrRaw( " (ECONNRESET)"	);
    case ENOBUFS:		strRet.appStrRaw( " (ENOBUFS)"		);
    case EISCONN:		strRet.appStrRaw( " (EISCONN)"		);
    case ENOTCONN:		strRet.appStrRaw( " (ENOTCONN)"		);
    case ESHUTDOWN:		strRet.appStrRaw( " (ESHUTDOWN)"	);
    case ETOOMANYREFS:		strRet.appStrRaw( " (ETOOMANYREFS)"	);
    case ETIMEDOUT:		strRet.appStrRaw( " (ETIMEDOUT)"	);
    case ECONNREFUSED:		strRet.appStrRaw( " (ECONNREFUSED)"	);
    case ELOOP:			strRet.appStrRaw( " (ELOOP)"		);
    case ENAMETOOLONG:		strRet.appStrRaw( " (ENAMETOOLONG)"	);
    case EHOSTDOWN:		strRet.appStrRaw( " (EHOSTDOWN)"	);
    case EHOSTUNREACH:		strRet.appStrRaw( " (EHOSTUNREACH)"	);
    case ENOTEMPTY:		strRet.appStrRaw( " (ENOTEMPTY)"	);
    case EPROCLIM:		strRet.appStrRaw( " (EPROCLIM)"		);
    case EUSERS:		strRet.appStrRaw( " (EUSERS)"		);
    case EDQUOT:		strRet.appStrRaw( " (EDQUOT)"		);
    case ESTALE:		strRet.appStrRaw( " (ESTALE)"		);
    case EREMOTE:		strRet.appStrRaw( " (EREMOTE)"		);
    case ENOSTR:		strRet.appStrRaw( " (ENOSTR)"		);
    case ETIME:			strRet.appStrRaw( " (ETIME)"		);
    case ENOSR:			strRet.appStrRaw( " (ENOSR)"		);
    case ENOMSG:		strRet.appStrRaw( " (ENOMSG)"		);
    case EBADMSG:		strRet.appStrRaw( " (EBADMSG)"		);
    case EIDRM:			strRet.appStrRaw( " (EIDRM)"		);
    case EDEADLK:		strRet.appStrRaw( " (EDEADLK)"		);
    case ENOLCK:		strRet.appStrRaw( " (ENOLCK)"		);
    case ENONET:		strRet.appStrRaw( " (ENONET)"		);
    case ERREMOTE:		strRet.appStrRaw( " (ERREMOTE)"		);
    case ENOLINK:		strRet.appStrRaw( " (ENOLINK)"		);
    case EADV:			strRet.appStrRaw( " (EADV)"		);
    case ESRMNT:		strRet.appStrRaw( " (ESRMNT)"		);
    case ECOMM:			strRet.appStrRaw( " (ECOMM)"		);
    case EPROTO:		strRet.appStrRaw( " (EPROTO)"		);
    case EMULTIHOP:		strRet.appStrRaw( " (EMULTIHOP)"	);
    case EDOTDOT:		strRet.appStrRaw( " (EDOTDOT)"		);
    case EREMCHG:		strRet.appStrRaw( " (EREMCHG)"		);
    case ENOSYS:		strRet.appStrRaw( " (ENOSYS)"		);
#undef	when
    }
}
