// Copyright 1994, 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: MmSeg.cc
//   Memory segment.
//
// File Created:	11 Jun 1994		Michael Chastain
// Last Reviewed:	09 Oct 1995		Michael Chastain
// Last Edited:		08 Nov 1995		Michael Chastain

#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <MmSeg.hh>
#include <MmType.hh>
#include <PrProc.hh>
#include <TySeg.hh>
#include <WhFlatIn.hh>
#include <WhFlatOut.hh>
#include <WhString.hh>



// Constructor.
MmSeg::MmSeg( )
    : fArea_		( false )
    , fData_		( false )
    , itySeg_		( 0	)
    , nItem_		( 0	)
    , area_		(	)
    , lwData_		(	)
{
    ;
}



// Flat input.
void MmSeg::fromFlat( WhFlatIn & flatIn )
{
    flatIn.getBool	( fArea_	);
    flatIn.getBool	( fData_	);
    flatIn.getInt	( itySeg_	);
    flatIn.getInt	( nItem_	);
    area_.fromFlat	( flatIn	);
    flatIn.getListWord	( lwData_	);
}



// Flat output.
void MmSeg::toFlat( WhFlatOut & flatOut ) const
{
    flatOut.putBool	( fArea_	);
    flatOut.putBool	( fData_	);
    flatOut.putInt	( itySeg_	);
    flatOut.putInt	( nItem_	);
    area_.toFlat	( flatOut	);
    flatOut.putListWord	( lwData_	);
}



// Give away the address of the data in this segment.
const MmByte * MmSeg::address( ) const
{
    if ( !fArea_ || !fData_ )
	ErAbort( "MmSeg::address: not bound." );
    return ( (const MmByte *) lwData_.address( ) ) + MmAddrRem( area_.getAddrMin( ) );
}



// Return count.
int MmSeg::count( ) const
{
    if ( !fArea_ )
	ErAbort( "MmSeg::count: not bound." );
    return area_.count( );
}



// Return first target address.
MmAddr MmSeg::getAddrFirst( ) const
{
    if ( !fArea_ )
	ErAbort( "MmSeg::getAddrFirst: not bound." );
    return area_.getAddrMin( );
}



// Return last target address.
MmAddr MmSeg::getAddrLast( ) const
{
    if ( !fArea_ )
	ErAbort( "MmSeg::getAddrLast: not bound." );
    return area_.getAddrMax( );
}



// Return target area.
const MmArea & MmSeg::getArea( ) const
{
    if ( !fArea_ )
	ErAbort( "MmSeg::getArea: not bound." );
    return area_;
}



// Return whether two segments overlap.
bool MmSeg::hasOverlap( const MmSeg & seg2 ) const
{
    if ( !this->hasArea( ) || !seg2.hasArea( ) )
	return false;

    const MmAddr addr1First = this-> getAddrFirst ( );
    const MmAddr addr1Last  = this-> getAddrLast  ( );
    const MmAddr addr2First = seg2.  getAddrFirst ( );
    const MmAddr addr2Last  = seg2.  getAddrLast  ( );

    if ( addr1First >= addr2First && addr1First < addr2Last ) return true;
    if ( addr2First >= addr1First && addr2First < addr1Last ) return true;
    return false;
}



// Replace one byte with another.
void MmSeg::replaceByte( MmByte bLeft, MmByte bRight )
{
    // Check.
    if ( !fArea_ || !fData_ )
	ErAbort( "MmSeg::replaceByte: not bound." );

    // Smash the data.
    //   g++ 2.6.3: auto WhList won't inline.
    const MmByte * pbData = address( );
    const int nbData = count( );
    if ( nbData > 0 )
    {
	MmByte * pbNew = new MmByte [nbData];
	if ( pbNew == 0 )
	    ErMem( );
	for ( int ibData = 0; ibData < nbData; ++ibData )
	    pbNew[ibData] = pbData[ibData] == bLeft ? bRight : pbData[ibData];
	smashData( pbNew, nbData );
	delete [] pbNew;
    }
}



// Format to string.
void MmSeg::fmtStr( WhString & strRet ) const
{
    TySegFmtSeg( strRet, *this );
}



// Smash the data with an array of same size.
void MmSeg::smashData( const MmByte * pbSmash, int nbSmash )
{
    // Check.
    if ( !fArea_ || !fData_ )
	ErFatal( "MmSeg::smashData: not bound." );
    checkCountOne( nbSmash );

    // Smash the data.
    MmByte * pbData = ( (MmByte *) lwData_.addrNonConst( ) )
                    + MmAddrRem( area_.getAddrMin( ) );
    for ( int ibSmash = 0; ibSmash < nbSmash; ++ibSmash )
	pbData[ibSmash] = pbSmash[ibSmash];
}



// Check for const-char-star string.
void MmSeg::checkCcs( ) const
{
    checkCountList ( sizeof(MmByte) );
    checkCountMin  ( sizeof(MmByte) );
    const MmByte * pbStr = address( );
    const int nbStr = count( );
    for ( int ibStr = 0; ibStr < nbStr; ++ibStr )
    {
	if ( ( pbStr[ibStr] == MmByte( '\0' ) ) != ( ibStr == nbStr - 1 ) )
	    ErFatal( "MmSeg::checkCcs: not ccs." );
    }
}



// Check count.
void MmSeg::checkCountList( int nbCheck ) const
{
    if ( nbCheck < 0 || count( ) % nbCheck != 0 )
	ErAbort( "MmSeg::checkCountList: count mismatch." );
}



// Check count.
void MmSeg::checkCountMin( int nbCheck ) const
{
    if ( count( ) < nbCheck )
	ErAbort( "MmSeg::checkCountMin: count mismatch." );
}



// Check count.
void MmSeg::checkCountOne( int nbCheck ) const
{
    if ( count( ) != nbCheck )
	ErAbort( "MmSeg::checkCountOne: count mismatch." );
}



// Fetch from process, copying existing segment.
void MmSeg::fetchCopy( const PrProc & procFetch, const MmSeg & segFrom )
{
    // Unbind.
    fArea_    = false;
    fData_    = false;
    itySeg_   = 0;
    nItem_    = 0;

    // Fetch.
    if ( segFrom.hasArea( ) )
    {
	fetchProc( procFetch, segFrom.area_.getTyMmSpace( ),
	    segFrom.getItySeg( ), segFrom.getAddrFirst( ), segFrom.nItem_ );
    }
}



// Fetch from process with count.
void MmSeg::fetchProc( const PrProc & procFetch, TyMmSpace tyMmSpace,
    int itySeg, MmAddr addrFirst, int nItem )
{
    // Unbind.
    fArea_    = false;
    fData_    = false;

    // Set segment type.
    itySeg_ = itySeg;

    // Get size.
    nItem_ = nItem;
    const int nData = TySegGetSize( itySeg, nItem );

    if ( nData >= 0 )
    {
	// Ordinary non-negative length.
	area_  = MmArea( tyMmSpace, addrFirst, addrFirst + nData );
	fArea_ = true;
	if ( nData == 0 )
	{
	    // Easy out.
	    lwData_.clear( );
	    fData_ = true;
	}
	else
	{
	    // Fetch a list.
	    fData_ = procFetch.fetchListWord( tyMmSpace,
		MmAddrAlign( addrFirst ),
		MmAddrAlign( addrFirst + nData + sizeof(MmWord) - 1 )
		- MmAddrAlign( addrFirst ), lwData_ );
	}
    }
    else if ( nData == -1 || nData == -4 )
    {
	// Fetch segment terminated by MmByte( 0 ) or MmWord( 0 ).
	lwData_.clear( 60 );
	for ( MmAddr addrFetch  = MmAddrAlign( addrFirst );
		     ;
		     addrFetch += sizeof(MmWord) )
	{
	    // Fetch a word.
	    MmWord wFetch;
	    if ( !procFetch.fetchWord( tyMmSpace, addrFetch, wFetch ) )
	    {
		// Record area, such as it is.
		area_  = MmArea( tyMmSpace, addrFirst,
		    addrFetch > addrFirst ? addrFetch : addrFirst );
		fArea_ = true;
		fData_ = false;
		lwData_.clear( );
	    }
	    else
	    {
		// Store this data.
		lwData_.appendVal( wFetch );

		if ( nData == -1 )
		{
		    // Look for MmByte( 0 ) (little endian).
		    const int nbFetch = sizeof(MmWord) / sizeof(MmByte);
		    for ( int ibFetch  = 0; ibFetch < nbFetch; ++ibFetch )
		    {
			if ( addrFetch + ibFetch >= addrFirst
			&& ((const MmByte *) &wFetch)[ibFetch] == MmByte( 0 ) )
			{
			    area_  = MmArea( tyMmSpace, addrFirst,
				addrFetch + ibFetch + 1 );
			    fArea_ = true;
			    fData_ = true;
			    break;
			}
		    }
		}
		else
		{
		    // Look for MmWord( 0 ).
		    if ( addrFetch >= addrFirst )
		    {
			const MmWord * pwData = (const MmWord *)
			    ( ( (const MmByte *) lwData_.address( ) )
			    + MmAddrRem( addrFirst ) );
			const int iDataLast = lwData_.count( ) - 1
			    - ( MmAddrRem( addrFirst ) > 0 ? 1 : 0 );
			if ( iDataLast < 0 )
			    ErAbort( "MmSeg::fetchProc: negative index." );
			if ( pwData[iDataLast] == MmWord( 0 ) )
			{
			    area_  = MmArea( tyMmSpace, addrFirst,
				addrFirst + (iDataLast + 1)
				* sizeof(MmWord) / sizeof(MmByte) );
			    fArea_ = true;
			    fData_ = true;
			}
		    }
		}
	    }

	    // Did I find an area?
	    if ( fArea_ )
		break;
	}
    }
    else
    {
	ErAbort( "MmSeg::fetchProc: negative count." );
    }
}



// Store to process.
void MmSeg::storeProc( PrProc & procStore ) const
{
    if ( !fArea_ || !fData_ )
	ErAbort( "MmSeg::storeProc: not bound." );
    if ( area_.count( ) > 0 )
    {
	if ( !procStore.storeListWord( area_.getTyMmSpace( ),
	    MmAddrAlign( area_.getAddrMin( ) ),
	    MmAddrAlign( area_.getAddrMax( ) + sizeof(MmWord) - 1 )
	    - MmAddrAlign( area_.getAddrMin( ) ), lwData_ ) )
	{
	    ErFatal( "MmSeg::storeProc: failed store." );
	}
    }
}
