// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: ScLine.cc
//   System call line.
//   This is a concrete class.
//
// These are the methods.
//
// File Created:	10 Apr 1995		Michael Chastain
// Last Reviewed:	15 Oct 1995		Michael Chastain
// Last Edited:		11 Nov 1995		Michael Chastain

#include <CxFetch.hh>
#include <ErAbort.hh>
#include <ErFatal.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <EvBase.hh>
#include <EvSci.hh>
#include <EvSco.hh>
#include <EvType.hh>
#include <Exe.hh>
#include <MmDs.hh>
#include <MmSeg.hh>
#include <MmSpace.hh>
#include <MmType.hh>
#include <PrProc.hh>
#include <RiLine.hh>
#include <ScLine.hh>
#include <WhImage.hh>
#include <WhList.hh>
#include <WhString.hh>

#if defined(MEC_TARGET_LIX)
#include <sys/mman.h>
#include <TySegLix.hh>
#endif



// Format to string.
void ScLine::fmtStrEvSci( WhString & strRet, const EvSci & evSciFmt ) const
{
    // Format name.
    strRet.appStrRaw( pstrName_ );
    if ( !fKnown_ )
    {
	strRet.appStrRaw( "::"				);
	strRet.appIntFmt( evSciFmt.getSysEntry( )	);
    }
    strRet.appStrRaw( " (" );

    // Format arguments.
    if ( pstrTyArg_ == 0 )
	ErPtr( );
    int nArgFmt = 0;
    for ( int iArg = 0; pstrTyArg_[iArg] != '\0'; ++iArg )
    {
	// Get formatting char.
	const char ctyArg = pstrTyArg_[iArg];
	if ( ctyArg == '-' )
	    continue;

	// Comma for args past first arg.
	if ( nArgFmt > 0 )
	    strRet.appStrRaw( ", " );
	++nArgFmt;

	if ( !evSciFmt.hasArg( ) )
	{
	    // Arg not available.
	    strRet.appStrRaw( "..." );
	}
	else
	{
	    // Discriminate on type.
	    const MmWord wArg = evSciFmt.getArg( iArg );

	    switch ( ctyArg )
	    {
	    default:
		ErAbort( "ScLine::fmtStrEvSci: bad char." );

	    case 'D':
		strRet.appIntFmt( wArg );
		break;

	    case 'O':
		strRet.appChrFmt( '0' );
		strRet.appIntFmt( wArg, 8, 0 );
		break;

	    case 'P':
		strRet.appPtrFmt( MmAddr( wArg ) );
		break;

	    case 'R':
		{
		    WhList <const RiLine *> lpri;
		    RiLineGet( wArg, lpri );
		    for ( int ipri = 0; ipri < lpri.count( ); ++ipri )
		    {
			const RiLine * pri = lpri[ipri];
			if ( pri == 0 )
			    ErPtr( );
			strRet.appStrRaw( pri->pstrRequest	);
			strRet.appStrRaw( " = "			);
		    }
		    strRet.appStrRaw( "0x" );
		    strRet.appIntFmt( wArg, 16, 8 );
		}
		break;

	    case 'X':
		strRet.appStrRaw( "0x" );
		strRet.appIntFmt( wArg, 16, 8 );
		break;
	    }
	}
    }

    // Format end of arguments.
    if ( nArgFmt == 0 )
	strRet.appChrRaw( ' ' );
    strRet.appStrRaw( ")\n" );
}



// Fetch values for EvSci.
void ScLine::fetchSci( const PrProc & procFetch, bool & fArgTriedRet,
    bool & fArgFetchedRet, WhList <MmWord> & lwArgRet ) const
{
    // I tried.
    fArgTriedRet   = true;
    fArgFetchedRet = false;

    // Count arguments.
    if ( pstrTyArg_ == 0 )
	ErPtr( );
    int nArg = 0;
    while ( pstrTyArg_[nArg] != '\0' )
	++nArg;

    // Fetch arguments.
    switch ( tyArg_ )
    {
    default:
	ErAbort( "ScLine::fetchArg: bad enum." );

    case tyArgMem:
	fArgFetchedRet = procFetch.fetchListWord( tyMmData,
	    MmAddr( procFetch.fetchReg( iArgArg_ ) ),
	    nArg * sizeof(MmWord), lwArgRet );
	break;

    case tyArgNone:
	lwArgRet.clear( );
	fArgFetchedRet = true;
	break;

    case tyArgReg:
	procFetch.fetchListReg( iArgArg_, nArg, lwArgRet );
	fArgFetchedRet = true;
	break;
    }
}



// Fetch values for EvSco.
void ScLine::fetchSco( const PrProc & procFetch, bool & fCompareRet,
    WhList <MmWord> & lwCompareRet, WhList <MmWord> & lwSmashRet,
    MmScr & scrRet ) const
{
    // Switch on smash type.
    int nwSmash = 0;
    switch ( tySmash_ )
    {
    default:		ErAbort( "ScLine::fetchSco: bad enum." );
    case tySmashCant:	fCompareRet = false; nwSmash = 0; break;
    case tySmashDef:	fCompareRet = false; nwSmash = 1; break;
    case tySmashFile:	fCompareRet = true;  nwSmash = 1; break;
    case tySmashMmap:	fCompareRet = true;  nwSmash = 0; break;
    case tySmashMpi0:	fCompareRet = false; nwSmash = 1; break;
    case tySmashNop:	fCompareRet = false; nwSmash = 0; break;
    case tySmashPair:	fCompareRet = false; nwSmash = 2; break;
    case tySmashReplay:	fCompareRet = true;  nwSmash = 0; break;
    case tySmashWrite:	fCompareRet = false; nwSmash = 1; break;
    case tySmashZero:	fCompareRet = false; nwSmash = 1; break;
    }

    // Fetch registers for comparison.
    lwCompareRet.clear( );
    if ( fCompareRet )
	procFetch.fetchAllReg( lwCompareRet );

    // Fetch registers for restorataion after smash.
    procFetch.fetchListReg( 0, nwSmash, lwSmashRet );

    // Fetch return value.
    scrRet.setTyScr( tyScr_ );
    scrRet.fetchProc( procFetch );

    // Don't compare on failed file calls.
    if ( tySmash_ == tySmashFile && scrRet.isError( ) )
	fCompareRet = false;
}



// Fetch segments.
void ScLine::fetchSeg( CxFetch & cxFetch, EvBase & evTarget ) const
{
    // Mark unsmashable as model failures.
    if ( tySmash_ == tySmashCant )
    {
	WhString strModelFail;
	if ( fKnown_ )
	{
	    strModelFail.appStrRaw( pstrName_ );
	    strModelFail.appStrRaw( ": cannot replay" );
	}
	else
	{
	    strModelFail.appStrRaw( "unknown system call" );
	}
	evTarget.setModelFail( strModelFail );
    }

    // Need arguments to fetch segments.
    if ( cxFetch.getEvSci( ).hasArg( ) )
    {
	// Fetch ordinary segments.
	MsLine::fetchArraySeg( plmsLine_, nmsLine_, cxFetch, evTarget );

	// Fetch special segments.
	const FnFetch * pfnFetch =
	    (cxFetch.getProc( ).getTyEvent( ) == tyEvSci)
	    ? pfnFetchIn_ : pfnFetchOut_;
	if ( pfnFetch != 0 )
	    (*pfnFetch)( cxFetch, evTarget );
    }
}



// Fetch datasets.
void ScLine::fetchDs( CxFetch & cxFetch, EvBase & evTarget ) const
{
    // Check type.
    switch ( tyDs_ )
    {
    default:		ErAbort( "ScLine::fetchDs: bad enum." );
    case tyDsNone:	return;
    case tyDsExecve:	break;
    case tyDsUselib:	break;
    }

    // I want successful EvSco's.
    if ( cxFetch.getProc  ( ).getTyEvent ( ) != tyEvSco
    ||   cxFetch.getEvSco ( ).isScrError ( ) )
	return;

    // Get name, which must exist.
    const MmSeg & segInName = cxFetch.getEvSci( ).getSeg( 0 );
    segInName.checkCcs( );

    // Translate name.
    WhString strInName;
    if ( !cxFetch.getDir( ).translateName( cxFetch.getProc( ), segInName, strInName ) )
    {
	WhString strFail;
	strFail.appStrRaw( "cannot follow name: '"		);
	strFail.appStrRaw( (const char *) segInName.address( )	);
	strFail.appChrRaw( '\''					);
	evTarget.setModelFail( strFail );
	return;
    }

    // Map file into memory.
    WhImage * pimageIn = new WhImage( strInName );
    if ( pimageIn == 0 )
	ErMem( );
    if ( pimageIn->isError( ) )
    {
	WhString strFail;
	strFail.appStrRaw( "cannot fetch file: '"		);
	strFail.appStrRaw( (const char *) segInName.address( )	);
	strFail.appStrRaw( "': "				);
	strFail.appStrRaw( pimageIn->getError( )		);
	evTarget.setModelFail( strFail );
	delete pimageIn;
	return;
    }

    // Get a dataset.
    MmDs * pdsIn = new MmDs( segInName, pimageIn, cxFetch.nextSeqDs( ) );
    if ( pdsIn == 0 )
	ErMem( );

    // Get list of areas.
    switch ( tyDs_ )
    {
    default:		ErAbort( "ScLine::fetchDs: bad enum." );
    case tyDsExecve:	ExeToArea( *pdsIn, cxFetch, evTarget ); break;
    case tyDsUselib:	LibToArea( *pdsIn, cxFetch, evTarget ); break;
    }

    // Save dataset in target.
    evTarget.appDs( pdsIn );
}



// Smash registers to annul a system call.
void ScLine::smashProc( PrProc & procSmash, bool fSuccess ) const
{
    switch ( tySmash_ )
    {
    default:
	ErAbort( "ScLine::smashProc: bad enum." );

    case tySmashCant:
	break;

    case tySmashDef:
	procSmash.storeReg( 0, MmWord( ~0 ) );
	break;

    case tySmashFile:
	if ( fSuccess )
	{
	    // Smash name.
	    MmSeg segSmashName;
	    segSmashName.fetchProc( procSmash, tyMmData, tySegStrNul,
		MmAddr( procSmash.fetchReg( 0 ) ), 1 );
	    if ( !segSmashName.hasArea( ) || !segSmashName.hasData( ) )
		ErFatal( "ScLine::smashProc: failed fetch." );
	    // This should use the stash for name lookup.
	    segSmashName.replaceByte( MmByte( '/' ), MmByte( '%' ) );
	    segSmashName.storeProc( procSmash );
	}
	else
	{
	    // Failed in original.
	    procSmash.storeReg( 0, MmWord( ~0 ) );
	}
	break;

#if defined(MEC_TARGET_LIX)
    case tySmashMmap:
	// Smash an argument word.
	{
	    const MmAddr addrFlags =
		MmAddr( procSmash.fetchReg( 0 ) ) + 3 * sizeof(MmWord);
	    MmWord wFlags;
	    if ( procSmash.fetchWord( tyMmData, addrFlags, wFlags ) )
	    {
		wFlags = (wFlags & ~MAP_TYPE) | MAP_PRIVATE | MAP_ANONYMOUS;
		if ( !procSmash.storeWord( tyMmData, addrFlags, wFlags ) )
		    ErAbort( "ScLine::smash: failed store." );
	    }
	}
	break;
#endif

    case tySmashMpi0:
	procSmash.storeReg( 0, MmWord( ~0 ) >> 1 );
	break;

    case tySmashNop:
    	break;

    case tySmashPair:
    	procSmash.storeReg( 0, MmWord( ~0 ) );
    	procSmash.storeReg( 1, MmWord( ~0 ) );
	break;

    case tySmashReplay:
    	break;

    case tySmashWrite:
#if 1
	// Demo technology.
	if ( procSmash.fetchReg( 0 ) < 3 )
	    break;
#endif
	procSmash.storeReg( 0, MmWord( ~0 ) );
	break;

    case tySmashZero:
    	procSmash.storeReg( 0, 0 );
	break;
    }
}
