// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: TySegLix.cc
//   Enumeration of all types of segments.
//
// File Created:	10 Apr 1995		Michael Chastain
// Last Edited:		16 Nov 1995		Michael Chastain



// Ordinary include files.
#include <ErAbort.hh>
#include <MmSeg.hh>
#include <MmType.hh>
#include <TySeg.hh>
#include <TySegLix.hh>
#include <WhCount.hh>
#include <WhString.hh>



// Butt-load of system-specific type files.
#include <sys/types.h>
#include <asm/sigcontext.h>
#include <linux/ax25.h>
#include <linux/cdrom.h>
#include <linux/cyclades.h>
#include <linux/fcntl.h>
#include <linux/fd.h>
#include <linux/hdreg.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/if_eql.h>
#include <linux/if_ppp.h>
#include <linux/in.h>
#include <linux/ipx.h>
#include <linux/kd.h>
#include <linux/kernel.h>
#include <linux/ldt.h>
#include <linux/lp.h>
#include <linux/module.h>
#include <linux/mroute.h>
#include <linux/mtio.h>
#include <linux/netrom.h>
#include <linux/resource.h>
#include <linux/route.h>
#include <linux/scc.h>
#include <linux/signal.h>
#include <linux/socket.h>
#include <linux/soundcard.h>
#include <linux/termios.h>
#include <linux/time.h>
#include <linux/times.h>
#include <linux/timex.h>
#include <linux/types.h>
#include <linux/uio.h>
#include <linux/umsdos_fs.h>
#include <linux/utime.h>
#include <linux/utsname.h>
#include <linux/vm86.h>
#include <linux/vfs.h>
#include <linux/vt.h>

#define  __KERNEL__
#include <linux/linkage.h>
#include <linux/ipc.h>
#include <linux/msg.h>
#include <linux/sem.h>
#include <linux/serial.h>
#include <linux/shm.h>
#include <linux/stat.h>
#include <linux/tty.h>
#undef   __KERNEL__

// Linux 1.3.24: see 'set_selection' in 'drivers/char/selection.c'.
struct	selection
{
    unsigned short	xs1;
    unsigned short	ys1;
    unsigned short	xe1;
    unsigned short	ye1;
    int			sel_mode;
};



// Butt-load of data dumpers.
void	Fmt1MmByte			( WhString &, const MmSeg & );
void	Fmt1Selection			( WhString &, const MmSeg & );
void	Fmt4MmWord8			( WhString &, const MmSeg & );
void	FmtAddrStr			( WhString &, const MmSeg & );
void	FmtAny				( WhString &, const MmSeg & );
void	FmtArea				( WhString &, const MmSeg & );
void	FmtArpreq			( WhString &, const MmSeg & );
void	FmtArpreqOld			( WhString &, const MmSeg & );
void	FmtAsyncStruct			( WhString &, const MmSeg & );
void	FmtAudioBufInfo			( WhString &, const MmSeg & );
void	FmtAx25ParmsStruct		( WhString &, const MmSeg & );
void	FmtBrk				( WhString &, const MmSeg & );
void	FmtCdromBlk			( WhString &, const MmSeg & );
void	FmtCdromMsf			( WhString &, const MmSeg & );
void	FmtCdromMultisession		( WhString &, const MmSeg & );
void	FmtCdromReadAudio		( WhString &, const MmSeg & );
void	FmtCdromSubchnl			( WhString &, const MmSeg & );
void	FmtCdromTi			( WhString &, const MmSeg & );
void	FmtCdromTocentry		( WhString &, const MmSeg & );
void	FmtCdromTochdr			( WhString &, const MmSeg & );
void	FmtCdromVolctrl			( WhString &, const MmSeg & );
void	FmtConsolefontdesc		( WhString &, const MmSeg & );
void	FmtCoprBuffer			( WhString &, const MmSeg & );
void	FmtCoprDebugBuf			( WhString &, const MmSeg & );
void	FmtCoprMsg			( WhString &, const MmSeg & );
void	FmtCountInfo			( WhString &, const MmSeg & );
void	FmtCycladesMonitor		( WhString &, const MmSeg & );
void	FmtDirent			( WhString &, const MmSeg & );
void	FmtFlock			( WhString &, const MmSeg & );
void	FmtFloppyDriveParams		( WhString &, const MmSeg & );
void	FmtFloppyDriveStruct		( WhString &, const MmSeg & );
void	FmtFloppyFdcState		( WhString &, const MmSeg & );
void	FmtFloppyMaxErrors		( WhString &, const MmSeg & );
void	FmtFloppyRawCmd			( WhString &, const MmSeg & );
void	FmtFloppyStruct			( WhString &, const MmSeg & );
void	FmtFloppyWriteErrors		( WhString &, const MmSeg & );
void	FmtFormatDescr			( WhString &, const MmSeg & );
void	FmtHdDriveid			( WhString &, const MmSeg & );
void	FmtHdGeometry			( WhString &, const MmSeg & );
void	FmtIfconf			( WhString &, const MmSeg & );
void	FmtIfreq			( WhString &, const MmSeg & );
void	FmtIoctlCommand			( WhString &, const MmSeg & );
void	FmtIpcKludge			( WhString &, const MmSeg & );
void	FmtIpxConfigData		( WhString &, const MmSeg & );
void	FmtItimerval			( WhString &, const MmSeg & );
void	FmtKbdiacrs			( WhString &, const MmSeg & );
void	FmtKbentry			( WhString &, const MmSeg & );
void	FmtKbkeycode			( WhString &, const MmSeg & );
void	FmtKbsentry			( WhString &, const MmSeg & );
void	FmtListCdromFrame		( WhString &, const MmSeg & );
void	FmtListDirent			( WhString &, const MmSeg & );
void	FmtListExecAddr			( WhString &, const MmSeg & );
void	FmtListGidT			( WhString &, const MmSeg & );
void	FmtListIfreq			( WhString &, const MmSeg & );
void	FmtListInsn			( WhString &, const MmSeg & );
void	FmtListIovec			( WhString &, const MmSeg & );
void	FmtListKernelSym		( WhString &, const MmSeg & );
void	FmtListMmByte			( WhString &, const MmSeg & );
void	FmtListMmShort			( WhString &, const MmSeg & );
void	FmtListMmWord			( WhString &, const MmSeg & );
void	FmtListSembuf			( WhString &, const MmSeg & );
void	FmtListUnipair			( WhString &, const MmSeg & );
void	FmtLpStats			( WhString &, const MmSeg & );
void	FmtMasterConfig			( WhString &, const MmSeg & );
void	FmtMidiInfo			( WhString &, const MmSeg & );
void	FmtMmAddr			( WhString &, const MmSeg & );
void	FmtMmByte			( WhString &, const MmSeg & );
void	FmtMmByte8			( WhString &, const MmSeg & );
void	FmtMmByte16			( WhString &, const MmSeg & );
void	FmtMmByte48			( WhString &, const MmSeg & );
void	FmtMmByte8192			( WhString &, const MmSeg & );
void	FmtMmByteTabsz			( WhString &, const MmSeg & );
void	FmtMmLoff			( WhString &, const MmSeg & );
void	FmtMmShortTabsz			( WhString &, const MmSeg & );
void	FmtMmWord			( WhString &, const MmSeg & );
void	FmtMmWord2			( WhString &, const MmSeg & );
void	FmtMmWord8			( WhString &, const MmSeg & );
void	FmtModRoutines			( WhString &, const MmSeg & );
void	FmtModifyLdtLdtS		( WhString &, const MmSeg & );
void	FmtMpuCommandRec		( WhString &, const MmSeg & );
void	FmtMsgbuf			( WhString &, const MmSeg & );
void	FmtMsghdr			( WhString &, const MmSeg & );
void	FmtMsginfo			( WhString &, const MmSeg & );
void	FmtMsqidDs			( WhString &, const MmSeg & );
void	FmtMtconfiginfo			( WhString &, const MmSeg & );
void	FmtMtget			( WhString &, const MmSeg & );
void	FmtMtop				( WhString &, const MmSeg & );
void	FmtMtpos			( WhString &, const MmSeg & );
void	FmtNewStat			( WhString &, const MmSeg & );
void	FmtNewUtsname			( WhString &, const MmSeg & );
void	FmtNrParmsStruct		( WhString &, const MmSeg & );
void	FmtOldStat			( WhString &, const MmSeg & );
void	FmtOldUtsname			( WhString &, const MmSeg & );
void	FmtOldoldUtsname		( WhString &, const MmSeg & );
void	FmtPatmgrInfo			( WhString &, const MmSeg & );
void	FmtPidT				( WhString &, const MmSeg & );
void	FmtPppCompStats			( WhString &, const MmSeg & );
void	FmtPppIdle			( WhString &, const MmSeg & );
void	FmtPppOptionData		( WhString &, const MmSeg & );
void	FmtPppStats			( WhString &, const MmSeg & );
void	FmtRlimit			( WhString &, const MmSeg & );
void	FmtRtentry			( WhString &, const MmSeg & );
void	FmtRusage			( WhString &, const MmSeg & );
void	FmtSbiInstrument		( WhString &, const MmSeg & );
void	FmtSccModem			( WhString &, const MmSeg & );
void	FmtSccStat			( WhString &, const MmSeg & );
void	FmtSemidDs			( WhString &, const MmSeg & );
void	FmtSeminfo			( WhString &, const MmSeg & );
void	FmtSemun			( WhString &, const MmSeg & );
void	FmtSeqEventRec			( WhString &, const MmSeg & );
void	FmtSerialMultiportStruct	( WhString &, const MmSeg & );
void	FmtSerialStruct			( WhString &, const MmSeg & );
void	FmtShmInfo			( WhString &, const MmSeg & );
void	FmtShmidDs			( WhString &, const MmSeg & );
void	FmtShminfo			( WhString &, const MmSeg & );
void	FmtSigaction			( WhString &, const MmSeg & );
void	FmtSigcontextStruct		( WhString &, const MmSeg & );
void	FmtSigsetT			( WhString &, const MmSeg & );
void	FmtSiocSgReq			( WhString &, const MmSeg & );
void	FmtSiocVifReq			( WhString &, const MmSeg & );
void	FmtSlaveConfig			( WhString &, const MmSeg & );
void	FmtSlavingRequest		( WhString &, const MmSeg & );
void	FmtSockaddrAx25			( WhString &, const MmSeg & );
void	FmtStatfs			( WhString &, const MmSeg & );
void	FmtStrCount			( WhString &, const MmSeg & );
void	FmtStrNul			( WhString &, const MmSeg & );
void	FmtSymbolTable			( WhString &, const MmSeg & );
void	FmtSynthInfo			( WhString &, const MmSeg & );
void	FmtSysinfo			( WhString &, const MmSeg & );
void	FmtTermio			( WhString &, const MmSeg & );
void	FmtTermios			( WhString &, const MmSeg & );
void	FmtTimeT			( WhString &, const MmSeg & );
void	FmtTimeval			( WhString &, const MmSeg & );
void	FmtTimex			( WhString &, const MmSeg & );
void	FmtTimezone			( WhString &, const MmSeg & );
void	FmtTms				( WhString &, const MmSeg & );
void	FmtTtyStruct			( WhString &, const MmSeg & );
void	FmtUidT				( WhString &, const MmSeg & );
void	FmtUmsdosIoctl			( WhString &, const MmSeg & );
void	FmtUnimapdesc			( WhString &, const MmSeg & );
void	FmtUnimapinit			( WhString &, const MmSeg & );
void	FmtUstat			( WhString &, const MmSeg & );
void	FmtUtimbuf			( WhString &, const MmSeg & );
void	FmtVm86Struct			( WhString &, const MmSeg & );
void	FmtVtConsize			( WhString &, const MmSeg & );
void	FmtVtMode			( WhString &, const MmSeg & );
void	FmtVtSizes			( WhString &, const MmSeg & );
void	FmtVtStat			( WhString &, const MmSeg & );
void	FmtWinsize			( WhString &, const MmSeg & );



// Return desired size of a segment.
int TySegGetSize( int itySeg, int nItem )
{
    // Check count.
    if ( nItem < 0 )
	ErAbort( "TySegGetSize: negative count." );

    // Process items which take counts.
    switch ( itySeg )
    {
    default:			break;
    case tySegAny:		return nItem * sizeof(MmByte);
    case tySegArea:		return nItem * sizeof(MmByte);
    case tySegBrk:		return nItem * sizeof(MmByte);
    case tySegDirent:		return nItem * sizeof(MmByte);	// Ick!
    case tySegListCdromFrame:	return nItem * CD_FRAMESIZE_RAW;
    case tySegListDirent:	return nItem * sizeof(MmByte);	// Ick!
    case tySegListGidT:		return nItem * sizeof(gid_t);
    case tySegListIfreq:	return nItem * sizeof(MmByte);	// Ick!
    case tySegListInsn:		return nItem * sizeof(MmInsn);
    case tySegListIovec:	return nItem * sizeof(struct iovec);
    case tySegListKernelSym:	return nItem * sizeof(struct kernel_sym);
    case tySegListMmByte:	return nItem * sizeof(MmByte);
    case tySegListMmShort:	return nItem * sizeof(MmShort);
    case tySegListMmWord:	return nItem * sizeof(MmWord);
    case tySegListSembuf:	return nItem * sizeof(struct sembuf);
    case tySegListUnipair:	return nItem * sizeof(struct unipair);
    case tySegModifyLdtLdtS:	return nItem * sizeof(MmByte);	// Ick!
    case tySegMsgbuf:		return nItem * sizeof(MmByte);
    case tySegStrCount:		return nItem * sizeof(MmByte);
    case tySegSymbolTable:	return nItem * sizeof(MmByte);	// Ick!
    }

    // Count must be singular now.
    if ( nItem != 1 )
	ErAbort( "TySegGetSize: nonsingular count." );

    // Process items with singular counts.
    switch ( itySeg )
    {
    default:
    case tySegNil:
	ErAbort( "TySegGetSize: bad enum." );

    case tySeg1MmByte:		return sizeof(MmByte) + sizeof(MmByte);
    case tySeg1Selection:	return sizeof(MmByte)
				     + sizeof(struct selection);
    case tySeg4MmWord8:		return sizeof(MmWord) + sizeof(MmWord [8]);
    case tySegArpreq:		return sizeof(struct arpreq);
    case tySegArpreqOld:	return sizeof(struct arpreq_old);
    case tySegAsyncStruct:	return sizeof(struct async_struct);
    case tySegAudioBufInfo:	return sizeof(struct audio_buf_info);
    case tySegAx25ParmsStruct:	return sizeof(struct ax25_parms_struct);
    case tySegCdromBlk:		return sizeof(struct cdrom_blk);
    case tySegCdromMsf:		return sizeof(struct cdrom_msf);
    case tySegCdromMultisession:
    				return sizeof(struct cdrom_multisession);
    case tySegCdromReadAudio:	return sizeof(struct cdrom_read_audio);
    case tySegCdromSubchnl:	return sizeof(struct cdrom_subchnl);
    case tySegCdromTi:		return sizeof(struct cdrom_ti);
    case tySegCdromTocentry:	return sizeof(struct cdrom_tocentry);
    case tySegCdromTochdr:	return sizeof(struct cdrom_tochdr);
    case tySegCdromVolctrl:	return sizeof(struct cdrom_volctrl);
    case tySegConsolefontdesc:	return sizeof(struct consolefontdesc);
    case tySegCoprBuffer:	return sizeof(struct copr_buffer);
    case tySegCoprDebugBuf:	return sizeof(struct copr_debug_buf);
    case tySegCoprMsg:		return sizeof(struct copr_msg);
    case tySegCountInfo:	return sizeof(struct count_info);
    case tySegCycladesMonitor:	return sizeof(struct cyclades_monitor);
    case tySegFlock:		return sizeof(struct flock);
    case tySegFloppyDriveParams:
    				return sizeof(struct floppy_drive_params);
    case tySegFloppyDriveStruct:
				return sizeof(struct floppy_drive_struct);
    case tySegFloppyFdcState:	return sizeof(struct floppy_fdc_state);
    case tySegFloppyMaxErrors:	return sizeof(struct floppy_max_errors);
    case tySegFloppyRawCmd:	return sizeof(struct floppy_raw_cmd);
    case tySegFloppyStruct:	return sizeof(struct floppy_struct);
    case tySegFloppyWriteErrors:
    				return sizeof(struct floppy_write_errors);
    case tySegFormatDescr:	return sizeof(struct format_descr);
    case tySegHdDriveid:	return sizeof(struct hd_driveid);
    case tySegHdGeometry:	return sizeof(struct hd_geometry);
    case tySegIfconf:		return sizeof(struct ifconf);
    case tySegIfreq:		return sizeof(struct ifreq);
    case tySegIoctlCommand:	return sizeof(struct ioctl_command);
    case tySegIpcKludge:	return sizeof(struct ipc_kludge);
    case tySegIpxConfigData:	return sizeof(struct ipx_config_data);
    case tySegItimerval:	return sizeof(struct itimerval);
    case tySegKbdiacrs:		return sizeof(struct kbdiacrs);
    case tySegKbentry:		return sizeof(struct kbentry);
    case tySegKbkeycode:	return sizeof(struct kbkeycode);
    case tySegKbsentry:		return sizeof(struct kbsentry);
    case tySegListExecAddr:	return -4;
    case tySegLpStats:		return sizeof(struct lp_stats);
    case tySegMasterConfig:	return sizeof(struct master_config);
    case tySegMidiInfo:		return sizeof(struct midi_info);
    case tySegMmAddr:		return sizeof(MmAddr);
    case tySegMmByte:		return sizeof(MmByte);
    case tySegMmByte8:		return sizeof(MmByte [8]);
    case tySegMmByte16:		return sizeof(MmByte [16]);
    case tySegMmByte48:		return sizeof(MmByte [48]);
    case tySegMmByte8192:	return sizeof(MmByte [8192]);
    case tySegMmByteTabsz:	return sizeof(MmByte [E_TABSZ]);
    case tySegMmLoff:		return sizeof(MmLoff);
    case tySegMmShortTabsz:	return sizeof(MmShort [E_TABSZ]);
    case tySegMmWord:		return sizeof(MmWord);
    case tySegMmWord2:		return sizeof(MmWord [2]);
    case tySegMmWord8:		return sizeof(MmWord [8]);
    case tySegModRoutines:	return sizeof(struct mod_routines);
    case tySegMpuCommandRec:	return sizeof(struct mpu_command_rec);
    case tySegMsghdr:		return sizeof(struct msghdr);
    case tySegMsginfo:		return sizeof(struct msginfo);
    case tySegMsqidDs:		return sizeof(struct msqid_ds);
    case tySegMtconfiginfo:	return sizeof(struct mtconfiginfo);
    case tySegMtget:		return sizeof(struct mtget);
    case tySegMtop:		return sizeof(struct mtop);
    case tySegMtpos:		return sizeof(struct mtpos);
    case tySegNewStat:		return sizeof(struct new_stat);
    case tySegNewUtsname:	return sizeof(struct new_utsname);
    case tySegNrParmsStruct:	return sizeof(struct nr_parms_struct);
    case tySegOldStat:		return sizeof(struct old_stat);
    case tySegOldUtsname:	return sizeof(struct old_utsname);
    case tySegOldoldUtsname:	return sizeof(struct oldold_utsname);
    case tySegPatmgrInfo:	return sizeof(struct patmgr_info);
    case tySegPidT:		return sizeof(pid_t);
    case tySegPppCompStats:	return sizeof(struct ppp_comp_stats);
    case tySegPppIdle:		return sizeof(struct ppp_idle);
    case tySegPppOptionData:	return sizeof(struct ppp_option_data);
    case tySegPppStats:		return sizeof(struct ppp_stats);
    case tySegRlimit:		return sizeof(struct rlimit);
    case tySegRtentry:		return sizeof(struct rtentry);
    case tySegRusage:		return sizeof(struct rusage);
    case tySegSbiInstrument:	return sizeof(struct sbi_instrument);
    case tySegSccModem:		return sizeof(struct scc_modem);
    case tySegSccStat:		return sizeof(struct scc_stat);
    case tySegSemidDs:		return sizeof(struct semid_ds);
    case tySegSeminfo:		return sizeof(struct seminfo);
    case tySegSemun:		return sizeof(union  semun);
    case tySegSeqEventRec:	return sizeof(struct seq_event_rec);
    case tySegSerialMultiportStruct:
				return sizeof(struct serial_multiport_struct);
    case tySegSerialStruct:	return sizeof(struct serial_struct);
    case tySegShmInfo:		return sizeof(struct shm_info);
    case tySegShmidDs:		return sizeof(struct shmid_ds);
    case tySegShminfo:		return sizeof(struct shminfo);
    case tySegSigaction:	return sizeof(struct sigaction);
    case tySegSigcontextStruct:	return sizeof(struct sigcontext_struct);
    case tySegSigsetT:		return sizeof(sigset_t);
    case tySegSiocSgReq:	return sizeof(struct sioc_sg_req);
    case tySegSiocVifReq:	return sizeof(struct sioc_vif_req);
    case tySegSlaveConfig:	return sizeof(struct slave_config);
    case tySegSlavingRequest:	return sizeof(struct slaving_request);
    case tySegSockaddrAx25:	return sizeof(struct sockaddr_ax25);
    case tySegStatfs:		return sizeof(struct statfs);
    case tySegStrNul:		return -1;
    case tySegSynthInfo:	return sizeof(struct synth_info);
    case tySegSysinfo:		return sizeof(struct sysinfo);
    case tySegTermio:		return sizeof(struct termio);
    case tySegTermios:		return sizeof(struct termios);
    case tySegTimeT:		return sizeof(time_t);
    case tySegTimeval:		return sizeof(struct timeval);
    case tySegTimex:		return sizeof(struct timex);
    case tySegTimezone:		return sizeof(struct timezone);
    case tySegTms:		return sizeof(struct tms);
    case tySegTtyStruct:	return sizeof(struct tty_struct);
    case tySegUidT:		return sizeof(uid_t);
    case tySegUmsdosIoctl:	return sizeof(struct umsdos_ioctl);
    case tySegUnimapdesc:	return sizeof(struct unimapdesc);
    case tySegUnimapinit:	return sizeof(struct unimapinit);
    case tySegUstat:		return sizeof(struct ustat);
    case tySegUtimbuf:		return sizeof(struct utimbuf);
    case tySegVm86Struct:	return sizeof(struct vm86_struct);
    case tySegVtConsize:	return sizeof(struct vt_consize);
    case tySegVtMode:		return sizeof(struct vt_mode);
    case tySegVtSizes:		return sizeof(struct vt_sizes);
    case tySegVtStat:		return sizeof(struct vt_stat);
    case tySegWinsize:		return sizeof(struct winsize);
    }
}



// Format a segment.
void TySegFmtSeg( WhString & strRet, const MmSeg & segFmt )
{
    // No area or no data?
    if ( !segFmt.hasArea( ) || !segFmt.hasData( ) )
    {
	strRet.appStrRaw( "    <no segment>\n" );
	return;
    }

    switch ( TySegLix( segFmt.getItySeg( ) ) )
    {
    default:
    case tySegNil:
    	ErAbort( "TySegFmtSeg: bad enum." );

#define	when  break; case
    when tySeg1MmByte:		Fmt1MmByte		( strRet, segFmt );
    when tySeg1Selection:	Fmt1Selection		( strRet, segFmt );
    when tySeg4MmWord8:		Fmt4MmWord8		( strRet, segFmt );
    when tySegAny:		FmtAny			( strRet, segFmt );
    when tySegArea:		FmtArea			( strRet, segFmt );
    when tySegArpreq:		FmtArpreq		( strRet, segFmt );
    when tySegArpreqOld:	FmtArpreqOld		( strRet, segFmt );
    when tySegAsyncStruct:	FmtAsyncStruct		( strRet, segFmt );
    when tySegAudioBufInfo:	FmtAudioBufInfo		( strRet, segFmt );
    when tySegAx25ParmsStruct:	FmtAx25ParmsStruct	( strRet, segFmt );
    when tySegBrk:		FmtBrk			( strRet, segFmt );
    when tySegCdromBlk:		FmtCdromBlk		( strRet, segFmt );
    when tySegCdromMsf:		FmtCdromMsf		( strRet, segFmt );
    when tySegCdromMultisession:
    				FmtCdromMultisession	( strRet, segFmt );
    when tySegCdromReadAudio:	FmtCdromReadAudio	( strRet, segFmt );
    when tySegCdromSubchnl:	FmtCdromSubchnl		( strRet, segFmt );
    when tySegCdromTi:		FmtCdromTi		( strRet, segFmt );
    when tySegCdromTocentry:	FmtCdromTocentry	( strRet, segFmt );
    when tySegCdromTochdr:	FmtCdromTochdr		( strRet, segFmt );
    when tySegCdromVolctrl:	FmtCdromVolctrl		( strRet, segFmt );
    when tySegConsolefontdesc:	FmtConsolefontdesc	( strRet, segFmt );
    when tySegCoprBuffer:	FmtCoprBuffer		( strRet, segFmt );
    when tySegCoprDebugBuf:	FmtCoprDebugBuf		( strRet, segFmt );
    when tySegCoprMsg:		FmtCoprMsg		( strRet, segFmt );
    when tySegCountInfo:	FmtCountInfo		( strRet, segFmt );
    when tySegCycladesMonitor:	FmtCycladesMonitor	( strRet, segFmt );
    when tySegDirent:		FmtDirent		( strRet, segFmt );
    when tySegFlock:		FmtFlock		( strRet, segFmt );
    when tySegFloppyDriveParams:
    				FmtFloppyDriveParams	( strRet, segFmt );
    when tySegFloppyDriveStruct:
    				FmtFloppyDriveStruct	( strRet, segFmt );
    when tySegFloppyFdcState:	FmtFloppyFdcState	( strRet, segFmt );
    when tySegFloppyMaxErrors:	FmtFloppyMaxErrors	( strRet, segFmt );
    when tySegFloppyRawCmd:	FmtFloppyRawCmd		( strRet, segFmt );
    when tySegFloppyStruct:	FmtFloppyStruct		( strRet, segFmt );
    when tySegFloppyWriteErrors:
    				FmtFloppyWriteErrors	( strRet, segFmt );
    when tySegFormatDescr:	FmtFormatDescr		( strRet, segFmt );
    when tySegHdDriveid:	FmtHdDriveid		( strRet, segFmt );
    when tySegHdGeometry:	FmtHdGeometry		( strRet, segFmt );
    when tySegIfconf:		FmtIfconf		( strRet, segFmt );
    when tySegIfreq:		FmtIfreq		( strRet, segFmt );
    when tySegIoctlCommand:	FmtIoctlCommand		( strRet, segFmt );
    when tySegIpcKludge:	FmtIpcKludge		( strRet, segFmt );
    when tySegIpxConfigData:	FmtIpxConfigData	( strRet, segFmt );
    when tySegItimerval:	FmtItimerval		( strRet, segFmt );
    when tySegKbdiacrs:		FmtKbdiacrs		( strRet, segFmt );
    when tySegKbentry:		FmtKbentry		( strRet, segFmt );
    when tySegKbkeycode:	FmtKbkeycode		( strRet, segFmt );
    when tySegKbsentry:		FmtKbsentry		( strRet, segFmt );
    when tySegListCdromFrame:	FmtListCdromFrame	( strRet, segFmt );
    when tySegListDirent:	FmtListDirent		( strRet, segFmt );
    when tySegListExecAddr:	FmtListExecAddr		( strRet, segFmt );
    when tySegListGidT:		FmtListGidT		( strRet, segFmt );
    when tySegListIfreq:	FmtListIfreq		( strRet, segFmt );
    when tySegListInsn:		FmtListInsn		( strRet, segFmt );
    when tySegListIovec:	FmtListIovec		( strRet, segFmt );
    when tySegListKernelSym:	FmtListKernelSym	( strRet, segFmt );
    when tySegListMmByte:	FmtListMmByte		( strRet, segFmt );
    when tySegListMmShort:	FmtListMmShort		( strRet, segFmt );
    when tySegListMmWord:	FmtListMmWord		( strRet, segFmt );
    when tySegListSembuf:	FmtListSembuf		( strRet, segFmt );
    when tySegListUnipair:	FmtListUnipair		( strRet, segFmt );
    when tySegLpStats:		FmtLpStats		( strRet, segFmt );
    when tySegMasterConfig:	FmtMasterConfig		( strRet, segFmt );
    when tySegMidiInfo:		FmtMidiInfo		( strRet, segFmt );
    when tySegMmAddr:		FmtMmAddr		( strRet, segFmt );
    when tySegMmByte:		FmtMmByte		( strRet, segFmt );
    when tySegMmByte8:		FmtMmByte8		( strRet, segFmt );
    when tySegMmByte16:		FmtMmByte16		( strRet, segFmt );
    when tySegMmByte48:		FmtMmByte48		( strRet, segFmt );
    when tySegMmByte8192:	FmtMmByte8192		( strRet, segFmt );
    when tySegMmByteTabsz:	FmtMmByteTabsz		( strRet, segFmt );
    when tySegMmLoff:		FmtMmLoff		( strRet, segFmt );
    when tySegMmShortTabsz:	FmtMmShortTabsz		( strRet, segFmt );
    when tySegMmWord:		FmtMmWord		( strRet, segFmt );
    when tySegMmWord2:		FmtMmWord2		( strRet, segFmt );
    when tySegMmWord8:		FmtMmWord8		( strRet, segFmt );
    when tySegModRoutines:	FmtModRoutines		( strRet, segFmt );
    when tySegModifyLdtLdtS:	FmtModifyLdtLdtS	( strRet, segFmt );
    when tySegMpuCommandRec:	FmtMpuCommandRec	( strRet, segFmt );
    when tySegMsgbuf:		FmtMsgbuf		( strRet, segFmt );
    when tySegMsghdr:		FmtMsghdr		( strRet, segFmt );
    when tySegMsginfo:		FmtMsginfo		( strRet, segFmt );
    when tySegMsqidDs:		FmtMsqidDs		( strRet, segFmt );
    when tySegMtconfiginfo:	FmtMtconfiginfo		( strRet, segFmt );
    when tySegMtget:		FmtMtget		( strRet, segFmt );
    when tySegMtop:		FmtMtop			( strRet, segFmt );
    when tySegMtpos:		FmtMtpos		( strRet, segFmt );
    when tySegNewStat:		FmtNewStat		( strRet, segFmt );
    when tySegNewUtsname:	FmtNewUtsname		( strRet, segFmt );
    when tySegNrParmsStruct:	FmtNrParmsStruct	( strRet, segFmt );
    when tySegOldStat:		FmtOldStat		( strRet, segFmt );
    when tySegOldUtsname:	FmtOldUtsname		( strRet, segFmt );
    when tySegOldoldUtsname:	FmtOldoldUtsname	( strRet, segFmt );
    when tySegPatmgrInfo:	FmtPatmgrInfo		( strRet, segFmt );
    when tySegPidT:		FmtPidT			( strRet, segFmt );
    when tySegPppCompStats:	FmtPppCompStats		( strRet, segFmt );
    when tySegPppIdle:		FmtPppIdle		( strRet, segFmt );
    when tySegPppOptionData:	FmtPppOptionData	( strRet, segFmt );
    when tySegPppStats:		FmtPppStats		( strRet, segFmt );
    when tySegRlimit:		FmtRlimit		( strRet, segFmt );
    when tySegRtentry:		FmtRtentry		( strRet, segFmt );
    when tySegRusage:		FmtRusage		( strRet, segFmt );
    when tySegSbiInstrument:	FmtSbiInstrument	( strRet, segFmt );
    when tySegSccModem:		FmtSccModem		( strRet, segFmt );
    when tySegSccStat:		FmtSccStat		( strRet, segFmt );
    when tySegSemidDs:		FmtSemidDs		( strRet, segFmt );
    when tySegSeminfo:		FmtSeminfo		( strRet, segFmt );
    when tySegSemun:		FmtSemun		( strRet, segFmt );
    when tySegSeqEventRec:	FmtSeqEventRec		( strRet, segFmt );
    when tySegSerialMultiportStruct:
				FmtSerialMultiportStruct
							( strRet, segFmt );
    when tySegSerialStruct:	FmtSerialStruct		( strRet, segFmt );
    when tySegShmInfo:		FmtShmInfo		( strRet, segFmt );
    when tySegShmidDs:		FmtShmidDs		( strRet, segFmt );
    when tySegShminfo:		FmtShminfo		( strRet, segFmt );
    when tySegSigaction:	FmtSigaction		( strRet, segFmt );
    when tySegSigcontextStruct:	FmtSigcontextStruct	( strRet, segFmt );
    when tySegSigsetT:		FmtSigsetT		( strRet, segFmt );
    when tySegSiocSgReq:	FmtSiocSgReq		( strRet, segFmt );
    when tySegSiocVifReq:	FmtSiocVifReq		( strRet, segFmt );
    when tySegSlaveConfig:	FmtSlaveConfig		( strRet, segFmt );
    when tySegSlavingRequest:	FmtSlavingRequest	( strRet, segFmt );
    when tySegSockaddrAx25:	FmtSockaddrAx25		( strRet, segFmt );
    when tySegStatfs:		FmtStatfs		( strRet, segFmt );
    when tySegStrCount:		FmtStrCount		( strRet, segFmt );
    when tySegStrNul:		FmtStrNul		( strRet, segFmt );
    when tySegSymbolTable:	FmtSymbolTable		( strRet, segFmt );
    when tySegSynthInfo:	FmtSynthInfo		( strRet, segFmt );
    when tySegSysinfo:		FmtSysinfo		( strRet, segFmt );
    when tySegTermio:		FmtTermio		( strRet, segFmt );
    when tySegTermios:		FmtTermios		( strRet, segFmt );
    when tySegTimeT:		FmtTimeT		( strRet, segFmt );
    when tySegTimeval:		FmtTimeval		( strRet, segFmt );
    when tySegTimex:		FmtTimex		( strRet, segFmt );
    when tySegTimezone:		FmtTimezone		( strRet, segFmt );
    when tySegTms:		FmtTms			( strRet, segFmt );
    when tySegTtyStruct:	FmtTtyStruct		( strRet, segFmt );
    when tySegUidT:		FmtUidT			( strRet, segFmt );
    when tySegUmsdosIoctl:	FmtUmsdosIoctl		( strRet, segFmt );
    when tySegUnimapdesc:	FmtUnimapdesc		( strRet, segFmt );
    when tySegUnimapinit:	FmtUnimapinit		( strRet, segFmt );
    when tySegUstat:		FmtUstat		( strRet, segFmt );
    when tySegUtimbuf:		FmtUtimbuf		( strRet, segFmt );
    when tySegVm86Struct:	FmtVm86Struct		( strRet, segFmt );
    when tySegVtConsize:	FmtVtConsize		( strRet, segFmt );
    when tySegVtMode:		FmtVtMode		( strRet, segFmt );
    when tySegVtSizes:		FmtVtSizes		( strRet, segFmt );
    when tySegVtStat:		FmtVtStat		( strRet, segFmt );
    when tySegWinsize:		FmtWinsize		( strRet, segFmt );
#undef	when
    }
}



// Format a segment.
void Fmt1MmByte( WhString & strRet, const MmSeg & seg1MmByte )
{
    seg1MmByte.checkCountOne( sizeof(MmByte) + sizeof(MmByte) );
    const MmByte & bValue = * ( 1 + (const MmByte *) seg1MmByte.address( ) );

    strRet.appStrRaw( "    0x"					);
    strRet.appIntFmt( bValue, 16, 2				);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void Fmt1Selection( WhString & strRet, const MmSeg & seg1Selection )
{
    seg1Selection.checkCountOne( sizeof(MmByte) + sizeof(struct selection) );

    const struct selection & selection =
	 * (const struct selection *)
	 ( 1 + (const MmByte *) seg1Selection.address( ) );

    strRet.appStrRaw( "    xs1: "				);
    strRet.appIntFmt( selection.xs1				);
    strRet.appStrRaw( " ys1: "					);
    strRet.appIntFmt( selection.ys1				);
    strRet.appStrRaw( " xe1: "					);
    strRet.appIntFmt( selection.xe1				);
    strRet.appStrRaw( " ye1: "					);
    strRet.appIntFmt( selection.ye1				);
    strRet.appStrRaw( " sel_mode: "				);
    strRet.appIntFmt( selection.sel_mode			);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void Fmt4MmWord8( WhString & strRet, const MmSeg & seg4MmWord8 )
{
    seg4MmWord8.checkCountOne( sizeof(MmWord) + sizeof(MmWord [8]) );

    const MmWord * pw8 =
	( 1 + (const MmWord *) seg4MmWord8.address( ) );

    strRet.appStrRaw( "    0x"					);
    strRet.appIntFmt( pw8[0], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[1], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[2], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[3], 16, 8				);

    strRet.appStrRaw( "\n" "    0x"				);
    strRet.appIntFmt( pw8[4], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[5], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[6], 16, 8				);
    strRet.appStrRaw( " 0x"					);
    strRet.appIntFmt( pw8[7], 16, 8				);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtAny( WhString & strRet, const MmSeg & segAny )
{
    if ( segAny.count( ) % sizeof(MmWord) == 0 )
	FmtListMmWord( strRet, segAny );
    else
	FmtListMmByte( strRet, segAny );
}




// Format a segment.
void FmtArea( WhString & strRet, const MmSeg & segArea )
{
    strRet.appStrRaw( "    <area> "		);
    strRet.appPtrFmt( segArea.getAddrFirst( )	);
    strRet.appChrRaw( ' '			);
    strRet.appPtrFmt( segArea.getAddrLast( )	);
    strRet.appChrRaw( '\n'			);
}



// Format a segment.
void FmtArpreq( WhString & strRet, const MmSeg & segArpreq )
{
    segArpreq.checkCountOne( sizeof(struct arpreq) );
    FmtAny( strRet, segArpreq );
}



// Format a segment.
void FmtArpreqOld( WhString & strRet, const MmSeg & segArpreqOld )
{
    segArpreqOld.checkCountOne( sizeof(struct arpreq_old) );
    FmtAny( strRet, segArpreqOld );
}



// Format a segment.
void FmtAsyncStruct( WhString & strRet, const MmSeg & segAsyncStruct )
{
    segAsyncStruct.checkCountOne( sizeof(struct async_struct) );
    FmtAny( strRet, segAsyncStruct );
}



// Format a segment.
void FmtAx25ParmsStruct( WhString & strRet, const MmSeg & segAx25ParmsStruct )
{
    segAx25ParmsStruct.checkCountOne( sizeof(struct ax25_parms_struct) );
    FmtAny( strRet, segAx25ParmsStruct );
}



// Format a segment.
void FmtAudioBufInfo( WhString & strRet, const MmSeg & segAudioBufInfo )
{
    segAudioBufInfo.checkCountOne( sizeof(struct audio_buf_info) );

    const struct audio_buf_info & audio_buf_info =
	* (const struct audio_buf_info *) segAudioBufInfo.address( );

    strRet.appStrRaw( "    fragments: "			);
    strRet.appIntFmt( audio_buf_info.fragments		);
    strRet.appStrRaw( " fragstotal: "			);
    strRet.appIntFmt( audio_buf_info.fragstotal		);
    strRet.appStrRaw( " fragsize: "			);
    strRet.appIntFmt( audio_buf_info.fragsize		);
    strRet.appStrRaw( " bytes: "			);
    strRet.appIntFmt( audio_buf_info.bytes		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtBrk( WhString & strRet, const MmSeg & segBrk )
{
    segBrk.checkCountList( sizeof(MmByte) );

    bool fFmt = false;
    if ( segBrk.count( ) % sizeof(MmWord) != 0 )
    {
	fFmt = true;
    }
    else
    {
	const MmWord * pw = (const MmWord *) segBrk.address( );
	const int nw = segBrk.count( ) / sizeof(MmWord);
	for ( int iw = 0; iw < nw; ++iw )
	{
	    if ( pw[iw] != 0 )
	    {
		fFmt = true;
		break;
	    }
	}
    }

    if ( fFmt )
	FmtAny( strRet, segBrk );
}



// Format a segment.
void FmtCdromBlk( WhString & strRet, const MmSeg & segCdromBlk )
{
    segCdromBlk.checkCountOne( sizeof(struct cdrom_blk) );

    const struct cdrom_blk & cdrom_blk =
	* (const struct cdrom_blk *) segCdromBlk.address( );

    strRet.appStrRaw( "    from: "				);
    strRet.appIntFmt( cdrom_blk.from				);
    strRet.appStrRaw( " len: "					);
    strRet.appIntFmt( cdrom_blk.len				);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromMsf( WhString & strRet, const MmSeg & segCdromMsf )
{
    segCdromMsf.checkCountOne( sizeof(struct cdrom_msf) );

    const struct cdrom_msf & cdrom_msf =
	* (const struct cdrom_msf *) segCdromMsf.address( );

    strRet.appStrRaw( "    cdmsf_min0: "			);
    strRet.appIntFmt( cdrom_msf.cdmsf_min0			);
    strRet.appStrRaw( " cdmsf_sec0: "				);
    strRet.appIntFmt( cdrom_msf.cdmsf_sec0			);
    strRet.appStrRaw( " cdmsf_frame0: "				);
    strRet.appIntFmt( cdrom_msf.cdmsf_frame0			);

    strRet.appStrRaw( "\n" "    cdmsf_min1: "			);
    strRet.appIntFmt( cdrom_msf.cdmsf_min1			);
    strRet.appStrRaw( " cdmsf_sec1: "				);
    strRet.appIntFmt( cdrom_msf.cdmsf_sec1			);
    strRet.appStrRaw( " cdmsf_frame1: "				);
    strRet.appIntFmt( cdrom_msf.cdmsf_frame1			);
    strRet.appChrRaw( '\n'					);

}



// Format a segment.
void FmtCdromMultisession(
    WhString & strRet, const MmSeg & segCdromMultisession )
{
    segCdromMultisession.checkCountOne( sizeof(struct cdrom_multisession) );

    const struct cdrom_multisession & cdrom_multisession =
	* (const struct cdrom_multisession *) segCdromMultisession.address( );

    strRet.appStrRaw( "    addr.lba: 0x"			);
    strRet.appIntFmt( cdrom_multisession.addr.lba, 16, 8	);
    strRet.appStrRaw( " addr.msf: "				);
    strRet.appIntFmt( cdrom_multisession.addr.msf.minute	);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_multisession.addr.msf.second	);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_multisession.addr.msf.frame		);

    strRet.appStrRaw( "\n" "    xa_flag: "			);
    strRet.appIntFmt( cdrom_multisession.xa_flag		);
    strRet.appStrRaw( " addr_format: "				);
    strRet.appIntFmt( cdrom_multisession.addr_format		);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromReadAudio( WhString & strRet, const MmSeg & segCdromReadAudio )
{
    segCdromReadAudio.checkCountOne( sizeof(struct cdrom_read_audio) );

    const struct cdrom_read_audio & cdrom_read_audio =
	* (const struct cdrom_read_audio *) segCdromReadAudio.address( );

    strRet.appStrRaw( "    addr.lba: 0x"			);
    strRet.appIntFmt( cdrom_read_audio.addr.lba, 16, 8		);
    strRet.appStrRaw( " addr.msf: "				);
    strRet.appIntFmt( cdrom_read_audio.addr.msf.minute		);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_read_audio.addr.msf.second		);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_read_audio.addr.msf.frame		);
    strRet.appStrRaw( " addr_format: "				);
    strRet.appIntFmt( cdrom_read_audio.addr_format		);

    strRet.appStrRaw( "\n" "    nframes: "			);
    strRet.appIntFmt( cdrom_read_audio.nframes			);
    strRet.appStrRaw( " buf: "					);
    strRet.appPtrFmt( cdrom_read_audio.buf			);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromSubchnl( WhString & strRet, const MmSeg & segCdromSubchnl )
{
    segCdromSubchnl.checkCountOne( sizeof(struct cdrom_subchnl) );

    const struct cdrom_subchnl & cdrom_subchnl =
	* (const struct cdrom_subchnl *) segCdromSubchnl.address( );

    strRet.appStrRaw( "    cdsc_format: "				);
    strRet.appIntFmt( cdrom_subchnl.cdsc_format				);
    strRet.appStrRaw( " cdsc_audiostatus: "				);
    strRet.appIntFmt( cdrom_subchnl.cdsc_audiostatus			);
    strRet.appStrRaw( " cdsc_adr: "					);
    strRet.appIntFmt( cdrom_subchnl.cdsc_adr				);
    strRet.appStrRaw( " cdsc_ctrl: "					);
    strRet.appIntFmt( cdrom_subchnl.cdsc_ctrl				);

    strRet.appStrRaw( "\n" "    cdsc_trk: "				);
    strRet.appIntFmt( cdrom_subchnl.cdsc_trk				);
    strRet.appStrRaw( " cdsc_ind: "					);
    strRet.appIntFmt( cdrom_subchnl.cdsc_ind				);

    strRet.appStrRaw( "\n" "    cdsc_absaddr.lba: 0x"			);
    strRet.appIntFmt( cdrom_subchnl.cdsc_absaddr.lba, 16, 8		);
    strRet.appStrRaw( " cdsc_absaddr.msf: "				);
    strRet.appIntFmt( cdrom_subchnl.cdsc_absaddr.msf.minute		);
    strRet.appChrRaw( ' '						);
    strRet.appIntFmt( cdrom_subchnl.cdsc_absaddr.msf.second		);
    strRet.appChrRaw( ' '						);
    strRet.appIntFmt( cdrom_subchnl.cdsc_absaddr.msf.frame		);

    strRet.appStrRaw( "\n" "    cdsc_reladdr.lba: 0x"			);
    strRet.appIntFmt( cdrom_subchnl.cdsc_reladdr.lba, 16, 8		);
    strRet.appStrRaw( " cdsc_reladdr.msf: "				);
    strRet.appIntFmt( cdrom_subchnl.cdsc_reladdr.msf.minute		);
    strRet.appChrRaw( ' '						);
    strRet.appIntFmt( cdrom_subchnl.cdsc_reladdr.msf.second		);
    strRet.appChrRaw( ' '						);
    strRet.appIntFmt( cdrom_subchnl.cdsc_reladdr.msf.frame		);

    strRet.appChrRaw( '\n'						);
}



// Format a segment.
void FmtCdromTi( WhString & strRet, const MmSeg & segCdromTi )
{
    segCdromTi.checkCountOne( sizeof(struct cdrom_ti) );

    const struct cdrom_ti & cdrom_ti =
	* (const struct cdrom_ti *) segCdromTi.address( );

    strRet.appStrRaw( "    cdti_trk0: "				);
    strRet.appIntFmt( cdrom_ti.cdti_trk0			);
    strRet.appStrRaw( " cdti_ind0: "				);
    strRet.appIntFmt( cdrom_ti.cdti_ind0			);
    strRet.appStrRaw( " cdti_trk1: "				);
    strRet.appIntFmt( cdrom_ti.cdti_trk1			);
    strRet.appStrRaw( " cdti_ind1: "				);
    strRet.appIntFmt( cdrom_ti.cdti_ind1			);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromTocentry( WhString & strRet, const MmSeg & segCdromTocentry )
{
    segCdromTocentry.checkCountOne( sizeof(struct cdrom_tocentry) );

    const struct cdrom_tocentry & cdrom_tocentry =
	* (const struct cdrom_tocentry *) segCdromTocentry.address( );

    strRet.appStrRaw( "    cdte_track: "			);
    strRet.appIntFmt( cdrom_tocentry.cdte_track			);
    strRet.appStrRaw( " cdte_adr: "				);
    strRet.appIntFmt( cdrom_tocentry.cdte_adr			);
    strRet.appStrRaw( " cdte_ctrl: "				);
    strRet.appIntFmt( cdrom_tocentry.cdte_ctrl			);
    strRet.appStrRaw( " cdte_format: "				);
    strRet.appIntFmt( cdrom_tocentry.cdte_format		);

    strRet.appStrRaw( "\n" "    cdte_addr.lba: 0x"		);
    strRet.appIntFmt( cdrom_tocentry.cdte_addr.lba, 16, 8	);
    strRet.appStrRaw( " cdte_addr.msf: "			);
    strRet.appIntFmt( cdrom_tocentry.cdte_addr.msf.minute	);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_tocentry.cdte_addr.msf.second	);
    strRet.appChrRaw( ' '					);
    strRet.appIntFmt( cdrom_tocentry.cdte_addr.msf.frame	);
    strRet.appStrRaw( " cdte_datamode: "			);
    strRet.appIntFmt( cdrom_tocentry.cdte_datamode		);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromTochdr( WhString & strRet, const MmSeg & segCdromTochdr )
{
    segCdromTochdr.checkCountOne( sizeof(struct cdrom_tochdr) );

    const struct cdrom_tochdr & cdrom_tochdr =
	* (const struct cdrom_tochdr *) segCdromTochdr.address( );

    strRet.appStrRaw( "    cdth_trk0: "				);
    strRet.appIntFmt( cdrom_tochdr.cdth_trk0			);
    strRet.appStrRaw( " cdth_trk1: "				);
    strRet.appIntFmt( cdrom_tochdr.cdth_trk1			);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtCdromVolctrl( WhString & strRet, const MmSeg & segCdromVolctrl )
{
    segCdromVolctrl.checkCountOne( sizeof(struct cdrom_volctrl) );

    const struct cdrom_volctrl & cdrom_volctrl =
	* (const struct cdrom_volctrl *) segCdromVolctrl.address( );

    strRet.appStrRaw( "    channel0: "				);
    strRet.appIntFmt( cdrom_volctrl.channel0			);
    strRet.appStrRaw( " channel1: "				);
    strRet.appIntFmt( cdrom_volctrl.channel1			);
    strRet.appStrRaw( " channel2: "				);
    strRet.appIntFmt( cdrom_volctrl.channel2			);
    strRet.appStrRaw( " channel3: "				);
    strRet.appIntFmt( cdrom_volctrl.channel3			);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtConsolefontdesc( WhString & strRet, const MmSeg & segConsolefontdesc )
{
    segConsolefontdesc.checkCountOne( sizeof(struct consolefontdesc) );

    const struct consolefontdesc & consolefontdesc =
	* (const struct consolefontdesc *) segConsolefontdesc.address( );

    strRet.appStrRaw( "    charcount: "			);
    strRet.appIntFmt( consolefontdesc.charcount		);
    strRet.appStrRaw( " charheight: "			);
    strRet.appIntFmt( consolefontdesc.charheight	);
    strRet.appStrRaw( " chardata: "			);
    strRet.appPtrFmt( consolefontdesc.chardata		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtCoprBuffer( WhString & strRet, const MmSeg & segCoprBuffer )
{
    segCoprBuffer.checkCountOne( sizeof(struct copr_buffer) );

    const struct copr_buffer & copr_buffer =
	* (const struct copr_buffer *) segCoprBuffer.address( );

    strRet.appStrRaw( "    command: "			);
    strRet.appIntFmt( copr_buffer.command		);
    strRet.appStrRaw( " flags: 0x"			);
    strRet.appIntFmt( copr_buffer.flags, 16, 8		);
    strRet.appStrRaw( " offs: "				);
    strRet.appIntFmt( copr_buffer.offs			);

    FmtAny( strRet, segCoprBuffer );
}



// Format a segment.
void FmtCoprDebugBuf( WhString & strRet, const MmSeg & segCoprDebugBuf )
{
    segCoprDebugBuf.checkCountOne( sizeof(struct copr_debug_buf) );

    const struct copr_debug_buf & copr_debug_buf =
	* (const struct copr_debug_buf *) segCoprDebugBuf.address( );

    strRet.appStrRaw( "    command: "			);
    strRet.appIntFmt( copr_debug_buf.command		);
    strRet.appStrRaw( " parm1: "			);
    strRet.appIntFmt( copr_debug_buf.parm1		);
    strRet.appStrRaw( " parm2: "			);
    strRet.appIntFmt( copr_debug_buf.parm2		);
    strRet.appStrRaw( " flags: "			);
    strRet.appIntFmt( copr_debug_buf.flags		);
    strRet.appStrRaw( " len: "				);
    strRet.appIntFmt( copr_debug_buf.len		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtCoprMsg( WhString & strRet, const MmSeg & segCoprMsg )
{
    segCoprMsg.checkCountOne( sizeof(struct copr_msg) );

    const struct copr_msg & copr_msg =
	* (const struct copr_msg *) segCoprMsg.address( );

    strRet.appStrRaw( "    len: "			);
    strRet.appIntFmt( copr_msg.len			);
    FmtAny( strRet, segCoprMsg );
}



// Format a segment.
void FmtCountInfo( WhString & strRet, const MmSeg & segCountInfo )
{
    segCountInfo.checkCountOne( sizeof(struct count_info) );

    const struct count_info & count_info =
	* (const struct count_info *) segCountInfo.address( );

    strRet.appStrRaw( "    bytes: "			);
    strRet.appIntFmt( count_info.bytes			);
    strRet.appStrRaw( " blocks: "			);
    strRet.appIntFmt( count_info.blocks			);
    strRet.appStrRaw( " ptr: "				);
    strRet.appIntFmt( count_info.ptr			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtCycladesMonitor( WhString & strRet, const MmSeg & segCycladesMonitor )
{
    segCycladesMonitor.checkCountOne( sizeof(struct cyclades_monitor) );

    const struct cyclades_monitor & cyclades_monitor =
	* (const struct cyclades_monitor *) segCycladesMonitor.address( );

    strRet.appStrRaw( "    int_count: "			);
    strRet.appIntFmt( cyclades_monitor.int_count	);
    strRet.appStrRaw( " char_count: "			);
    strRet.appIntFmt( cyclades_monitor.char_count	);
    strRet.appStrRaw( " char_max: "			);
    strRet.appIntFmt( cyclades_monitor.char_max		);
    strRet.appStrRaw( " char_last: "			);
    strRet.appIntFmt( cyclades_monitor.char_last	);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtDirent( WhString & strRet, const MmSeg & segDirent )
{
    const struct dirent & dirent =
	* (const struct dirent *) segDirent.address( );
    segDirent.checkCountMin(
	(const char *) (&dirent.d_name[0]) - (const char *) (&dirent) );

    strRet.appStrRaw( "    d_ino: "				);
    strRet.appIntFmt( dirent.d_ino				);
    strRet.appStrRaw( " d_off: "				);
    strRet.appIntFmt( dirent.d_off				);
    strRet.appStrRaw( " d_reclen: "				);
    strRet.appIntFmt( dirent.d_reclen				);
    strRet.appStrRaw( " d_name: \""				);
    strRet.appStrFmt( &dirent.d_name[0], WhCount(dirent.d_name)	);
    strRet.appStrRaw( "\"\n"					);
}



// Format a segment.
void FmtFlock( WhString & strRet, const MmSeg & segFlock )
{
    segFlock.checkCountOne( sizeof(struct flock) );

    const struct flock & flock = * (const struct flock *) segFlock.address( );

    strRet.appStrRaw( "    l_type: "			);
    strRet.appIntFmt( flock.l_type			);
    strRet.appStrRaw( " l_whence: "			);
    strRet.appIntFmt( flock.l_whence			);
    strRet.appStrRaw( " l_start: "			);
    strRet.appIntFmt( flock.l_start			);
    strRet.appStrRaw( " l_len: "			);
    strRet.appIntFmt( flock.l_len			);
    strRet.appStrRaw( " l_pid: "			);
    strRet.appIntFmt( flock.l_pid			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtFloppyDriveParams
    ( WhString & strRet, const MmSeg & segFloppyDriveParams )
{
    segFloppyDriveParams.checkCountOne( sizeof(struct floppy_drive_params) );
    FmtAny( strRet, segFloppyDriveParams );
}



// Format a segment.
void FmtFloppyDriveStruct
    ( WhString & strRet, const MmSeg & segFloppyDriveStruct )
{
    segFloppyDriveStruct.checkCountOne( sizeof(struct floppy_drive_struct) );
    FmtAny( strRet, segFloppyDriveStruct );
}



// Format a segment.
void FmtFloppyFdcState( WhString & strRet, const MmSeg & segFloppyFdcState )
{
    segFloppyFdcState.checkCountOne( sizeof(struct floppy_fdc_state) );
    FmtAny( strRet, segFloppyFdcState );
}



// Format a segment.
void FmtFloppyMaxErrors( WhString & strRet, const MmSeg & segFloppyMaxErrors )
{
    segFloppyMaxErrors.checkCountOne( sizeof(struct floppy_max_errors) );
    FmtAny( strRet, segFloppyMaxErrors );
}



// Format a segment.
void FmtFloppyRawCmd( WhString & strRet, const MmSeg & segFloppyRawCmd )
{
    segFloppyRawCmd.checkCountOne( sizeof(struct floppy_raw_cmd) );
    FmtAny( strRet, segFloppyRawCmd );
}



// Format a segment.
void FmtFloppyStruct( WhString & strRet, const MmSeg & segFloppyStruct )
{
    segFloppyStruct.checkCountOne( sizeof(struct floppy_struct) );
    FmtAny( strRet, segFloppyStruct );
}



// Format a segment.
void FmtFloppyWriteErrors
    ( WhString & strRet, const MmSeg & segFloppyWriteErrors )
{
    segFloppyWriteErrors.checkCountOne( sizeof(struct floppy_write_errors) );
    FmtAny( strRet, segFloppyWriteErrors );
}



// Format a segment.
void FmtFormatDescr( WhString & strRet, const MmSeg & segFormatDescr )
{
    segFormatDescr.checkCountOne( sizeof(struct format_descr) );
    const struct format_descr & format_descr =
	* (const struct format_descr *) segFormatDescr.address( );

    strRet.appStrRaw( "    device: "			);
    strRet.appIntFmt( format_descr.device		);
    strRet.appStrRaw( " head: "				);
    strRet.appIntFmt( format_descr.head			);
    strRet.appStrRaw( " track: "			);
    strRet.appIntFmt( format_descr.track		);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtHdDriveid( WhString & strRet, const MmSeg & segHdDriveid )
{
    segHdDriveid.checkCountOne( sizeof(struct hd_driveid) );
    FmtAny( strRet, segHdDriveid );
}



// Format a segment.
void FmtHdGeometry( WhString & strRet, const MmSeg & segHdGeometry )
{
    segHdGeometry.checkCountOne( sizeof(struct hd_geometry) );

    const struct hd_geometry & hd_geometry =
	* (const struct hd_geometry *) segHdGeometry.address( );

    strRet.appStrRaw( "    heads: "			);
    strRet.appIntFmt( hd_geometry.heads			);
    strRet.appStrRaw( " sectors: "			);
    strRet.appIntFmt( hd_geometry.sectors		);
    strRet.appStrRaw( " cylinders: "			);
    strRet.appIntFmt( hd_geometry.cylinders		);
    strRet.appStrRaw( " start: "			);
    strRet.appIntFmt( hd_geometry.start			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtIfconf( WhString & strRet, const MmSeg & segIfconf )
{
    segIfconf.checkCountOne( sizeof(struct ifconf) );

    const struct ifconf & ifconf =
	* (const struct ifconf *) segIfconf.address( );

    strRet.appStrRaw( "    ifc_len: "			);
    strRet.appIntFmt( ifconf.ifc_len			);
    strRet.appStrRaw( " ifc_buf: "			);
    strRet.appPtrFmt( ifconf.ifc_buf			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtIfreq( WhString & strRet, const MmSeg & segIfreq )
{
    segIfreq.checkCountOne( sizeof(struct ifreq) );

    const struct ifreq & ifreq = * (const struct ifreq *) segIfreq.address( );

    strRet.appStrRaw( "    ifr_name: \""				);
    strRet.appStrFmt( &ifreq.ifr_name[0], WhCount(ifreq.ifr_name)	);
    strRet.appChrRaw( '"'						);

    const MmByte * pbIfru = (const MmByte *) &ifreq.ifr_ifru;
    const int nbIfru = sizeof(ifreq.ifr_ifru) / sizeof(MmByte);
    for ( int ibIfru  = 0; ibIfru < nbIfru; ++ibIfru )
    {
	if ( ibIfru % 8 == 0 )
	    strRet.appStrRaw( "\n" "    ifr_ifru:"		);
	strRet.appStrRaw( " 0x"					);
	strRet.appIntFmt( pbIfru[ibIfru], 16, 2			);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtIoctlCommand( WhString & strRet, const MmSeg & segIoctlCommand )
{
    segIoctlCommand.checkCountOne( sizeof(struct ioctl_command) );
    const struct ioctl_command & ioctl_command =
	* (const struct ioctl_command *) segIoctlCommand.address( );

    strRet.appStrRaw( "    command: "			);
    strRet.appIntFmt( ioctl_command.command		);
    strRet.appStrRaw( " param: "			);
    strRet.appIntFmt( ioctl_command.param		);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtIpcKludge( WhString & strRet, const MmSeg & segIpcKludge )
{
    segIpcKludge.checkCountOne( sizeof(struct ipc_kludge) );
    const struct ipc_kludge & ipc_kludge =
	* (const struct ipc_kludge *) segIpcKludge.address( );

    strRet.appStrRaw( "    msgp: "			);
    strRet.appPtrFmt( ipc_kludge.msgp			);
    strRet.appStrRaw( " msgtyp: "			);
    strRet.appIntFmt( ipc_kludge.msgtyp			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtIpxConfigData( WhString & strRet, const MmSeg & segIpxConfigData )
{
    segIpxConfigData.checkCountOne( sizeof(struct ipx_config_data) );

    const struct ipx_config_data & ipx_config_data =
	* (const struct ipx_config_data *) segIpxConfigData.address( );

    strRet.appStrRaw( "    ipxcfg_auto_select_primary: "		);
    strRet.appIntFmt( ipx_config_data.ipxcfg_auto_select_primary	);
    strRet.appStrRaw( " ipxcfg_auto_create_interfaces: "		);
    strRet.appIntFmt( ipx_config_data.ipxcfg_auto_create_interfaces	);
}



// Format a segment.
void FmtItimerval( WhString & strRet, const MmSeg & segItimerval )
{
    segItimerval.checkCountOne( sizeof(struct itimerval) );

    const struct itimerval & itimerval =
	* (const struct itimerval *) segItimerval.address( );

    strRet.appStrRaw( "    it_interval: "			);
    strRet.appIntFmt( itimerval.it_interval.tv_sec		);
    strRet.appChrRaw( '.'					);
    strRet.appIntFmt( itimerval.it_interval.tv_usec, 10, 6	);
    strRet.appStrRaw( " it_value: "				);
    strRet.appIntFmt( itimerval.it_value.tv_sec			);
    strRet.appChrRaw( '.'					);
    strRet.appIntFmt( itimerval.it_value.tv_usec, 10, 6		);
    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtKbdiacrs( WhString & strRet, const MmSeg & segKbdiacrs )
{
    segKbdiacrs.checkCountOne( sizeof(struct kbdiacrs) );

    const struct kbdiacrs & kbdiacrs =
	* (const struct kbdiacrs *) segKbdiacrs.address( );

    strRet.appStrRaw( "    kb_cnt: "			);
    strRet.appIntFmt( kbdiacrs.kb_cnt			);

    // Cap the count.
    const int	iDiacrMax = WhCount(kbdiacrs.kbdiacr);
    int		iDiacrMac = int(kbdiacrs.kb_cnt);
    if ( iDiacrMac > iDiacrMax )
    {
	strRet.appStrRaw( " (OVERFLOW)" );
	iDiacrMac = iDiacrMax;
    }

    // Format each entry.
    for ( int iDiacr = 0; iDiacr < iDiacrMac; ++iDiacr )
    {
	const struct kbdiacr & kbdiacr = kbdiacrs.kbdiacr[iDiacr];

	strRet.appStrRaw( "\n" "    diacr: '"		);
	strRet.appChrFmt( kbdiacr.diacr			);
	strRet.appStrRaw( "' base: '"			);
	strRet.appChrFmt( kbdiacr.base			);
	strRet.appStrRaw( "' result: '"			);
	strRet.appChrFmt( kbdiacr.result		);
	strRet.appChrRaw( '\''				);
    }
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtKbentry( WhString & strRet, const MmSeg & segKbentry )
{
    segKbentry.checkCountOne( sizeof(struct kbentry) );

    const struct kbentry & kbentry =
	* (const struct kbentry *) segKbentry.address( );

    strRet.appStrRaw( "    kb_table: "			);
    strRet.appIntFmt( kbentry.kb_table			);
    strRet.appStrRaw( " kb_index: "			);
    strRet.appIntFmt( kbentry.kb_index			);
    strRet.appStrRaw( " kb_value: "			);
    strRet.appIntFmt( kbentry.kb_value			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtKbkeycode( WhString & strRet, const MmSeg & segKbkeycode )
{
    segKbkeycode.checkCountOne( sizeof(struct kbkeycode) );

    const struct kbkeycode & kbkeycode =
	* (const struct kbkeycode *) segKbkeycode.address( );

    strRet.appStrRaw( "    scancode: "			);
    strRet.appIntFmt( kbkeycode.scancode		);
    strRet.appStrRaw( " keycode: "			);
    strRet.appIntFmt( kbkeycode.keycode			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtKbsentry( WhString & strRet, const MmSeg & segKbsentry )
{
    segKbsentry.checkCountOne( sizeof(struct kbsentry) );

    const struct kbsentry & kbsentry =
	* (const struct kbsentry *) segKbsentry.address( );

    strRet.appStrRaw( "    kb_func: "				);
    strRet.appIntFmt( kbsentry.kb_func				);
    strRet.appStrRaw( " kb_kbstring: \""			);
    strRet.appStrFmt( (const char *) &kbsentry.kb_string[0],
		      WhCount( kbsentry.kb_string )		);
    strRet.appStrRaw( "\"\n"					);

}



// Format a segment.
void FmtListCdromFrame( WhString & strRet, const MmSeg & segListCdromFrame )
{
    segListCdromFrame.checkCountList( CD_FRAMESIZE_RAW );

    const MmByte * plFrame = (const MmByte *) segListCdromFrame.address( );

    for ( int iFrame  = 0;
	      iFrame  < segListCdromFrame.count( ) / CD_FRAMESIZE_RAW;
	      iFrame += 1 )
    {
	const MmByte * pbFrame = plFrame + iFrame * CD_FRAMESIZE_RAW;

	strRet.appStrRaw( "    iFrame: "		);
	strRet.appIntFmt( iFrame			);
	strRet.appStrRaw( " pbFrame: "			);
	strRet.appPtrFmt( pbFrame			);

	for ( int ibFrame = 0; ibFrame < CD_FRAMESIZE_RAW; ++ibFrame )
	{
	    if ( ibFrame % 8 == 0 )
		strRet.appStrRaw( "\n       "		);
	    strRet.appStrRaw( " 0x"			);
	    strRet.appIntFmt( pbFrame[ibFrame], 16, 2	);
	}

	strRet.appChrRaw( '\n'				);
    }
}



// Format a segment.
void FmtListInsn( WhString & strRet, const MmSeg & segListInsn )
{
    FmtListMmByte( strRet, segListInsn );
}



// Format a segment.
void FmtListDirent( WhString & strRet, const MmSeg & segListDirent )
{
    // Grab pointer.
    const MmByte * pbDirent = (const MmByte *) segListDirent.address( );

    // Format directory entries.
    int ibDirent = 0;
    for ( ibDirent = 0; ibDirent < segListDirent.count( ); )
    {
	const struct dirent & dirent =
	    * (const struct dirent *) (pbDirent + ibDirent);

	const int sbDirent = dirent.d_reclen;
	if ( sbDirent <= 0 || ibDirent + sbDirent > segListDirent.count( ) )
	{
	    strRet.appStrRaw( "    <other> "				);
	    strRet.appIntFmt( segListDirent.count( ) - ibDirent		);
	    strRet.appChrRaw( '\n'					);
	    break;
	}

	strRet.appStrRaw( "    d_ino: "					);
	strRet.appIntFmt( dirent.d_ino					);
	strRet.appStrRaw( " d_off: "					);
	strRet.appIntFmt( dirent.d_off					);
	strRet.appStrRaw( " d_reclen: "					);
	strRet.appIntFmt( dirent.d_reclen				);
	strRet.appStrRaw( " d_name: \""					);
	strRet.appStrFmt( &dirent.d_name[0], WhCount(dirent.d_name)	);
	strRet.appStrRaw( "\"\n"					);

	ibDirent += sbDirent;
    }
}



// Format a segment.
void FmtListExecAddr( WhString & strRet, const MmSeg & segListExecAddr )
{
    segListExecAddr.checkCountList( sizeof(MmAddr) );

    const MmAddr * pa = (const MmAddr *) segListExecAddr.address( );
    const int na = segListExecAddr.count( ) / sizeof(MmAddr);

    for ( int ia = 0; ia < na; ++ia )
    {
	if ( ia % 4 == 0 )
	{
	    if ( ia > 0 )
		strRet.appChrRaw( '\n'			);
	    strRet.appStrRaw( "   "			);
	}
	strRet.appChrRaw( ' '				);
	strRet.appPtrFmt( pa[ia]			);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtListGidT( WhString & strRet, const MmSeg & segListGidT )
{
    segListGidT.checkCountList( sizeof(gid_t) );

    const gid_t * list_gid = (const gid_t *) segListGidT.address( );
    const int nGidT = segListGidT.count( ) / sizeof(gid_t);

    for ( int iGidT = 0; iGidT < nGidT; ++iGidT )
    {
	if ( iGidT % 8 == 0 )
	{
	    if ( iGidT > 0 )
		strRet.appChrRaw( '\n'			);
	    strRet.appStrRaw( "   "			);
	}
	strRet.appChrRaw( ' '				);
	strRet.appIntFmt( list_gid[iGidT]		);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtListIfreq( WhString & strRet, const MmSeg & segListIfreq )
{
    segListIfreq.checkCountList( sizeof(struct ifreq) );

    const struct ifreq * plIfreq = (const struct ifreq *) segListIfreq.address( );
    const int nIfreq = segListIfreq.count( ) / sizeof(struct ifreq);

    for ( int iIfreq = 0; iIfreq < nIfreq; ++iIfreq )
    {
	const struct ifreq &	ifreq	= plIfreq[iIfreq];
	strRet.appStrRaw( "    iIfreq: "			);
	strRet.appIntFmt( iIfreq				);

	strRet.appStrRaw( "\n" "        ifr_name: \""		);
	strRet.appStrFmt( &ifreq.ifr_name[0],
			  sizeof(ifreq.ifr_name) / sizeof(char)	);
	strRet.appChrRaw( '"'					);

	const MmByte * pbIfru = (const MmByte *) &ifreq.ifr_ifru;
	const int nbIfru = sizeof(ifreq.ifr_ifru) / sizeof(MmByte);
	for ( int ibIfru = 0; ibIfru < nbIfru; ++ibIfru )
	{
	    if ( ibIfru % 8 == 0 )
		strRet.appStrRaw( "\n" "        ifr_ifru:"	);
	    strRet.appStrRaw( " 0x"				);
	    strRet.appIntFmt( pbIfru[ibIfru], 16, 2		);
	}

	strRet.appChrRaw( '\n'					);
    }
}



// Format a segment.
void FmtListIovec( WhString & strRet, const MmSeg & segListIovec )
{
    segListIovec.checkCountList( sizeof(struct iovec) );

    const struct iovec * plIovec = (const struct iovec *) segListIovec.address( );
    const int nIovec = segListIovec.count( ) / sizeof(struct iovec);

    for ( int iIovec = 0; iIovec < nIovec; ++iIovec )
    {
	const struct iovec & iovec = plIovec[iIovec];
	strRet.appStrRaw( "    iIovec: "			);
	strRet.appIntFmt( iIovec				);
	strRet.appStrRaw( " iov_base: "				);
	strRet.appPtrFmt( iovec.iov_base			);
	strRet.appStrRaw( " iov_len: "				);
	strRet.appIntFmt( iovec.iov_len				);
	strRet.appChrRaw( '\n'					);
    }
}



// Format a segment.
void FmtListKernelSym( WhString & strRet, const MmSeg & segListKernelSym )
{
    segListKernelSym.checkCountList( sizeof(struct kernel_sym) );

    const struct kernel_sym * list_kernel_sym =
	(const struct kernel_sym *) segListKernelSym.address( );
    const int nSym = segListKernelSym.count( ) / sizeof(struct kernel_sym);

    for ( int iSym = 0; iSym < nSym; ++iSym )
    {
	const struct kernel_sym & kernel_sym = list_kernel_sym[iSym];
	strRet.appStrRaw( "    value: 0x"				);
	strRet.appIntFmt( kernel_sym.value, 16, 8			);
	strRet.appStrRaw( " name: \""					);
	strRet.appStrFmt( &kernel_sym.name[0], WhCount(kernel_sym.name)	);
	strRet.appStrRaw( "\"\n"					);
    }
}



// Format a segment.
void FmtListMmByte( WhString & strRet, const MmSeg & segListMmByte )
{
    segListMmByte.checkCountList( sizeof(MmByte) );

    const MmByte * pb = (const MmByte *) segListMmByte.address( );
    const int nb = segListMmByte.count( ) / sizeof(MmByte);

    for ( int ib = 0; ib < nb; ++ib )
    {
	if ( ib % 8 == 0 )
	{
	    if ( ib > 0 )
		strRet.appChrRaw( '\n'				);
	    strRet.appStrRaw( "   "				);
	}
	strRet.appStrRaw( " 0x"					);
	strRet.appIntFmt( pb[ib], 16, 2				);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtListMmShort( WhString & strRet, const MmSeg & segListMmShort )
{
    segListMmShort.checkCountList( sizeof(MmShort) );

    const MmShort * ph = (const MmShort *) segListMmShort.address( );
    const int nh = segListMmShort.count( ) / sizeof(MmShort);

    for ( int ih = 0; ih < nh; ++ih )
    {
	if ( ih % 8 == 0 )
	{
	    if ( ih > 0 )
		strRet.appChrRaw( '\n'				);
	    strRet.appStrRaw( "   "				);
	}
	strRet.appChrRaw( ' '					);
	strRet.appIntFmt( ph[ih]				);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtListMmWord( WhString & strRet, const MmSeg & segListMmWord )
{
    segListMmWord.checkCountList( sizeof(MmWord) );

    const MmWord * pw = (const MmWord *) segListMmWord.address( );
    const int nw = segListMmWord.count( ) / sizeof(MmWord);

    for ( int iw = 0; iw < nw; ++iw )
    {
	if ( iw % 4 == 0 )
	{
	    if ( iw > 0 )
		strRet.appChrRaw( '\n'			);
	    strRet.appStrRaw( "   "			);
	}
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( pw[iw], 16, 8			);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtListSembuf( WhString & strRet, const MmSeg & segListSembuf )
{
    segListSembuf.checkCountList( sizeof(struct sembuf) );

    const struct sembuf * plSembuf = (const struct sembuf *) segListSembuf.address( );
    const int nSembuf = segListSembuf.count( ) / sizeof(struct sembuf);

    for ( int iSembuf = 0; iSembuf < nSembuf; ++iSembuf )
    {
	const struct sembuf &	sembuf	= plSembuf[iSembuf];

	strRet.appStrRaw( "    iSembuf: "			);
	strRet.appIntFmt( iSembuf				);
	strRet.appStrRaw( " sem_num: "				);
	strRet.appIntFmt( sembuf.sem_num			);
	strRet.appStrRaw( " sem_op: "				);
	strRet.appIntFmt( sembuf.sem_op				);
	strRet.appStrRaw( " sem_flg: 0"				);
	strRet.appIntFmt( sembuf.sem_flg, 8, 0			);
	strRet.appChrRaw( '\n'					);
    }
}



// Format a segment.
void FmtListUnipair( WhString & strRet, const MmSeg & segListUnipair )
{
    segListUnipair.checkCountList( sizeof(struct unipair) );

    const struct unipair * plUnipair = (const struct unipair *) segListUnipair.address( );
    const int nUnipair = segListUnipair.count( ) / sizeof(struct unipair);

    for ( int iUnipair = 0; iUnipair < nUnipair; ++iUnipair )
    {
	const struct unipair & unipair = plUnipair[iUnipair];

	strRet.appStrRaw( "    iUnipair: "			);
	strRet.appIntFmt( iUnipair				);
	strRet.appStrRaw( " unicode: "				);
	strRet.appIntFmt( unipair.unicode			);
	strRet.appStrRaw( " fontpos: "				);
	strRet.appIntFmt( unipair.fontpos			);
	strRet.appChrRaw( '\n'					);
    }
}



// Format a segment.
void FmtLpStats( WhString & strRet, const MmSeg & segLpStats )
{
    segLpStats.checkCountOne( sizeof(struct lp_stats) );

    const struct lp_stats & lp_stats =
	* (const struct lp_stats *) segLpStats.address( );

    strRet.appStrRaw( "    chars: "				);
    strRet.appIntFmt( lp_stats.chars				);
    strRet.appStrRaw( " sleeps: "				);
    strRet.appIntFmt( lp_stats.sleeps				);
    strRet.appStrRaw( " maxrun: "				);
    strRet.appIntFmt( lp_stats.maxrun				);

    strRet.appStrRaw( "\n" "    maxwait: "			);
    strRet.appIntFmt( lp_stats.maxwait				);
    strRet.appStrRaw( " meanwait: "				);
    strRet.appIntFmt( lp_stats.meanwait				);
    strRet.appStrRaw( " mdev: "					);
    strRet.appIntFmt( lp_stats.mdev				);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtMasterConfig( WhString & strRet, const MmSeg & segMasterConfig )
{
    segMasterConfig.checkCountOne( sizeof(struct master_config) );
    FmtAny( strRet, segMasterConfig );
}



// Format a segment.
void FmtMidiInfo( WhString & strRet, const MmSeg & segMidiInfo )
{
    segMidiInfo.checkCountOne( sizeof(struct midi_info) );

    const struct midi_info & midi_info =
	* (const struct midi_info *) segMidiInfo.address( );

    strRet.appStrRaw( "    name: \""					);
    strRet.appStrFmt( &midi_info.name[0], WhCount(midi_info.name)	);

    strRet.appStrRaw( "\"\n" "    device: "				);
    strRet.appIntFmt( midi_info.device					);
    strRet.appStrRaw( " capabilities: 0x"				);
    strRet.appIntFmt( midi_info.capabilities, 16, 8			);
    strRet.appStrRaw( " dev_type: "					);
    strRet.appIntFmt( midi_info.dev_type				);

    for ( int iDummy = 0; iDummy < WhCount(midi_info.dummies); ++iDummy )
    {
	if ( iDummy % 8 == 0 )
	    strRet.appStrRaw( "\n    dummies:"				);
	strRet.appChrRaw( ' '						);
	strRet.appIntFmt( midi_info.dummies[iDummy]			);
    }

    strRet.appChrRaw( '\n'						);
}



// Format a segment.
void FmtMmAddr( WhString & strRet, const MmSeg & segMmAddr )
{
    segMmAddr.checkCountOne( sizeof(MmAddr) );

    const MmAddr & addrValue = * (const MmAddr *) segMmAddr.address( );

    strRet.appStrRaw( "    "				);
    strRet.appPtrFmt( addrValue				);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMmByte( WhString & strRet, const MmSeg & segMmByte )
{
    segMmByte.checkCountOne( sizeof(MmByte) );

    const MmByte & bValue = * (const MmByte *) segMmByte.address( );

    strRet.appStrRaw( "    0x"				);
    strRet.appIntFmt( bValue, 16, 2			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtMmByte8( WhString & strRet, const MmSeg & segMmByte8 )
{
    segMmByte8.checkCountOne( sizeof(MmByte [8]) );
    FmtAny( strRet, segMmByte8 );
}



// Format a segment.
void FmtMmByte16( WhString & strRet, const MmSeg & segMmByte16 )
{
    segMmByte16.checkCountOne( sizeof(MmByte [16]) );
    FmtAny( strRet, segMmByte16 );
}



// Format a segment.
void FmtMmByte48( WhString & strRet, const MmSeg & segMmByte48 )
{
    segMmByte48.checkCountOne( sizeof(MmByte [48]) );
    FmtAny( strRet, segMmByte48 );
}



// Format a segment.
void FmtMmByte8192( WhString & strRet, const MmSeg & segMmByte8192 )
{
    segMmByte8192.checkCountOne( sizeof(MmByte [8192]) );
    FmtAny( strRet, segMmByte8192 );
}



// Format a segment.
void FmtMmByteTabsz( WhString & strRet, const MmSeg & segMmByteTabsz )
{
    segMmByteTabsz.checkCountOne( sizeof(MmByte [E_TABSZ]) );
    FmtAny( strRet, segMmByteTabsz );
}



// Format a segment.
void FmtMmLoff( WhString & strRet, const MmSeg & segMmLoff )
{
    segMmLoff.checkCountOne( sizeof(MmLoff) );
    FmtAny( strRet, segMmLoff );
}



// Format a segment.
void FmtMmShortTabsz( WhString & strRet, const MmSeg & segMmShortTabsz )
{
    segMmShortTabsz.checkCountOne( sizeof(MmShort [E_TABSZ]) );
    FmtAny( strRet, segMmShortTabsz );
}



// Format a segment.
void FmtMmWord( WhString & strRet, const MmSeg & segMmWord )
{
    segMmWord.checkCountOne( sizeof(MmWord) );

    const MmWord & wValue = * (const MmWord *) segMmWord.address( );

    strRet.appStrRaw( "    0x"				);
    strRet.appIntFmt( wValue, 16, 8			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMmWord2( WhString & strRet, const MmSeg & segMmWord2 )
{
    segMmWord2.checkCountOne( sizeof(MmWord [2]) );
    const MmWord * pw2 = (const MmWord *) segMmWord2.address( );

    strRet.appStrRaw( "    0x"				);
    strRet.appIntFmt( pw2[0], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw2[1], 16, 8			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMmWord8( WhString & strRet, const MmSeg & segMmWord8 )
{
    segMmWord8.checkCountOne( sizeof(MmWord [8]) );
    const MmWord * pw8 = (const MmWord *) segMmWord8.address( );

    strRet.appStrRaw( "    0x"				);
    strRet.appIntFmt( pw8[0], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[1], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[2], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[3], 16, 8			);

    strRet.appStrRaw( "\n" "    0x"			);
    strRet.appIntFmt( pw8[4], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[5], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[6], 16, 8			);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( pw8[7], 16, 8			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtModRoutines( WhString & strRet, const MmSeg & segModRoutines )
{
    segModRoutines.checkCountOne( sizeof(struct mod_routines) );

    const struct mod_routines & mod_routines =
	* (const struct mod_routines *) segModRoutines.address( );

    strRet.appStrRaw( "    init: "			);
    strRet.appPtrFmt( (void *) (mod_routines.init)	);
    strRet.appStrRaw( " cleanup: "			);
    strRet.appPtrFmt( (void *) (mod_routines.cleanup)	);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtModifyLdtLdtS( WhString & strRet, const MmSeg & segModifyLdtLdtS )
{
    segModifyLdtLdtS.checkCountOne( sizeof(struct modify_ldt_ldt_s) );

    const struct modify_ldt_ldt_s & modify_ldt_ldt_s =
	* (const struct modify_ldt_ldt_s *) segModifyLdtLdtS.address( );

    strRet.appStrRaw( "    entry_number: "			);
    strRet.appIntFmt( modify_ldt_ldt_s.entry_number		);
    strRet.appStrRaw( " base_addr: 0x"				);
    strRet.appIntFmt( modify_ldt_ldt_s.base_addr, 16, 8		);
    strRet.appStrRaw( " limit: 0x"				);
    strRet.appIntFmt( modify_ldt_ldt_s.limit, 16, 8		);

    strRet.appStrRaw( "\n" "    seg_32bit: "			);
    strRet.appIntFmt( modify_ldt_ldt_s.seg_32bit		);
    strRet.appStrRaw( " contents: "				);
    strRet.appIntFmt( modify_ldt_ldt_s.contents			);
    strRet.appStrRaw( " read_exec_only: "			);
    strRet.appIntFmt( modify_ldt_ldt_s.read_exec_only		);
    strRet.appStrRaw( " limit_in_pages: "			);
    strRet.appIntFmt( modify_ldt_ldt_s.limit_in_pages		);

    strRet.appStrRaw( "\n" "    seg_not_present: "		);
    strRet.appIntFmt( modify_ldt_ldt_s.seg_not_present		);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtMpuCommandRec( WhString & strRet, const MmSeg & segMpuCommandRec )
{
    segMpuCommandRec.checkCountOne( sizeof(struct mpu_command_rec) );

    const struct mpu_command_rec & mpu_command_rec =
	* (const struct mpu_command_rec *) segMpuCommandRec.address( );

    strRet.appStrRaw( "    cmd: 0x"				);
    strRet.appIntFmt( mpu_command_rec.cmd, 16, 2		);
    strRet.appStrRaw( " nr_args: "				);
    strRet.appIntFmt( mpu_command_rec.nr_args			);
    strRet.appStrRaw( " nr_returns: "				);
    strRet.appIntFmt( mpu_command_rec.nr_returns		);

    for ( int iData  = 0; iData < WhCount(mpu_command_rec.data); ++iData )
    {
	if ( iData % 8 == 0 )
	    strRet.appStrRaw( "\n    data:"			);
	strRet.appStrRaw( " 0x"					);
	strRet.appIntFmt( mpu_command_rec.data[iData], 16, 2	);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtMsgbuf( WhString & strRet, const MmSeg & segMsgbuf )
{
    const struct msgbuf & msgbuf =
	* (const struct msgbuf *) segMsgbuf.address( );
    const int sHeader =
	(const char *) (&msgbuf.mtext[0]) - (const char *) (&msgbuf);
    segMsgbuf.checkCountMin( sHeader );
    const int sText = segMsgbuf.count( ) - sHeader;

    strRet.appStrRaw( "    mtype: "			);
    strRet.appIntFmt( msgbuf.mtype			);
    for ( int iText = 0; iText < sText; ++iText )
    {
	if ( iText % 8 == 0 )
	    strRet.appStrRaw( "\n" "    mtext:"		);
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( msgbuf.mtext[iText], 16, 2	);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMsghdr( WhString & strRet, const MmSeg & segMsghdr )
{
    segMsghdr.checkCountOne( sizeof(struct msghdr) );

    const struct msghdr & msghdr =
	* (const struct msghdr *) segMsghdr.address( );

    strRet.appStrRaw( "    msg_name: "			);
    strRet.appPtrFmt( msghdr.msg_name			);
    strRet.appStrRaw( " msg_namelen: "			);
    strRet.appIntFmt( msghdr.msg_namelen		);
    strRet.appStrRaw( " msg_iov: "			);
    strRet.appPtrFmt( msghdr.msg_iov			);
    strRet.appStrRaw( " msg_iovlen: "			);
    strRet.appIntFmt( msghdr.msg_iovlen			);

    strRet.appStrRaw( "\n" "    msg_accrights: "	);
    strRet.appPtrFmt( msghdr.msg_accrights		);
    strRet.appStrRaw( " msg_accrightslen: "		);
    strRet.appIntFmt( msghdr.msg_accrightslen		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMsginfo( WhString & strRet, const MmSeg & segMsginfo )
{
    segMsginfo.checkCountOne( sizeof(struct msginfo) );

    const struct msginfo & msginfo =
	* (const struct msginfo *) segMsginfo.address( );

    strRet.appStrRaw( "    msgpool: "			);
    strRet.appIntFmt( msginfo.msgpool			);
    strRet.appStrRaw( " msgmap: "			);
    strRet.appIntFmt( msginfo.msgmap			);
    strRet.appStrRaw( " msgmax: "			);
    strRet.appIntFmt( msginfo.msgmax			);
    strRet.appStrRaw( " msgmnb: "			);
    strRet.appIntFmt( msginfo.msgmnb			);
    strRet.appStrRaw( " msgmni: "			);
    strRet.appIntFmt( msginfo.msgmni			);

    strRet.appStrRaw( "\n" "    msgssz: "		);
    strRet.appIntFmt( msginfo.msgssz			);
    strRet.appStrRaw( " msgtql: "			);
    strRet.appIntFmt( msginfo.msgtql			);
    strRet.appStrRaw( " msgseg: "			);
    strRet.appIntFmt( msginfo.msgseg			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMsqidDs( WhString & strRet, const MmSeg & segMsqidDs )
{
    segMsqidDs.checkCountOne( sizeof(struct msqid_ds) );

    const struct msqid_ds & msqid_ds =
	* (const struct msqid_ds *) segMsqidDs.address( );

    strRet.appStrRaw( "    msg_perm.key: 0x"		);
    strRet.appIntFmt( msqid_ds.msg_perm.key, 16, 8	);
    strRet.appStrRaw( " msg_perm.uid: "			);
    strRet.appIntFmt( msqid_ds.msg_perm.uid		);
    strRet.appStrRaw( " msg_perm.gid: "			);
    strRet.appIntFmt( msqid_ds.msg_perm.gid		);

    strRet.appStrRaw( "\n" "    msg_perm.cuid: "	);
    strRet.appIntFmt( msqid_ds.msg_perm.cuid		);
    strRet.appStrRaw( " msg_perm.cgid: "		);
    strRet.appIntFmt( msqid_ds.msg_perm.cgid		);
    strRet.appStrRaw( " msg_perm.mode: 0"		);
    strRet.appIntFmt( msqid_ds.msg_perm.mode, 8, 0	);

    strRet.appStrRaw( "\n" "    msg_perm.seq: "		);
    strRet.appIntFmt( msqid_ds.msg_perm.seq		);

    strRet.appStrRaw( "\n" "    msg_first: "		);
    strRet.appPtrFmt( msqid_ds.msg_first		);
    strRet.appStrRaw( " msg_last: "			);
    strRet.appPtrFmt( msqid_ds.msg_last			);

    strRet.appStrRaw( "\n" "    msg_stime: "		);
    strRet.appIntFmt( msqid_ds.msg_stime		);
    strRet.appStrRaw( " msg_rtime: "			);
    strRet.appIntFmt( msqid_ds.msg_rtime		);
    strRet.appStrRaw( " msg_ctime: "			);
    strRet.appIntFmt( msqid_ds.msg_ctime		);

    strRet.appStrRaw( "\n" "    wwait: "		);
    strRet.appPtrFmt( msqid_ds.wwait			);
    strRet.appStrRaw( " rwait: "			);
    strRet.appPtrFmt( msqid_ds.rwait			);

    strRet.appStrRaw( "\n" "    msg_cbytes: "		);
    strRet.appIntFmt( msqid_ds.msg_cbytes		);
    strRet.appStrRaw( " msg_qnum: "			);
    strRet.appIntFmt( msqid_ds.msg_qnum			);
    strRet.appStrRaw( " msg_qbytes: "			);
    strRet.appIntFmt( msqid_ds.msg_qbytes		);

    strRet.appStrRaw( "\n" "    msg_lspid: "		);
    strRet.appIntFmt( msqid_ds.msg_lspid		);
    strRet.appStrRaw( " msg_lrpid: "			);
    strRet.appIntFmt( msqid_ds.msg_lrpid		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMtconfiginfo( WhString & strRet, const MmSeg & segMtconfiginfo )
{
    segMtconfiginfo.checkCountOne( sizeof(struct mtconfiginfo) );

    const struct mtconfiginfo & mtconfiginfo =
	* (const struct mtconfiginfo *) segMtconfiginfo.address( );

    strRet.appStrRaw( "    mt_type: 0x"				);
    strRet.appIntFmt( mtconfiginfo.mt_type, 16, 8		);
    strRet.appStrRaw( " ifc_type: 0x"				);
    strRet.appIntFmt( mtconfiginfo.ifc_type, 16, 8		);

    strRet.appStrRaw( "\n" "    irqnr: "			);
    strRet.appIntFmt( mtconfiginfo.irqnr			);
    strRet.appStrRaw( " dmanr: "				);
    strRet.appIntFmt( mtconfiginfo.dmanr			);
    strRet.appStrRaw( " port: 0x"				);
    strRet.appIntFmt( mtconfiginfo.port, 16, 4			);
    strRet.appStrRaw( " debug: 0x"				);
    strRet.appIntFmt( mtconfiginfo.debug			);

    strRet.appStrRaw( "\n    have_dens: "			);
    strRet.appStrRaw( mtconfiginfo.have_dens ? "1" : "0"	);
    strRet.appStrRaw( " have_bsf: "				);
    strRet.appStrRaw( mtconfiginfo.have_bsf  ? "1" : "0"	);
    strRet.appStrRaw( " have_fsr: "				);
    strRet.appStrRaw( mtconfiginfo.have_fsr  ? "1" : "0"	);
    strRet.appStrRaw( " have_eod: "				);
    strRet.appStrRaw( mtconfiginfo.have_eod  ? "1" : "0"	);
    strRet.appStrRaw( " have_seek: "				);
    strRet.appStrRaw( mtconfiginfo.have_seek ? "1" : "0"	);

    strRet.appStrRaw( "\n    have_tell: "			);
    strRet.appStrRaw( mtconfiginfo.have_tell ? "1" : "0"	);
    strRet.appStrRaw( " have_ras1: "				);
    strRet.appStrRaw( mtconfiginfo.have_ras1 ? "1" : "0"	);
    strRet.appStrRaw( " have_ras2: "				);
    strRet.appStrRaw( mtconfiginfo.have_ras2 ? "1" : "0"	);
    strRet.appStrRaw( " have_ras3: "				);
    strRet.appStrRaw( mtconfiginfo.have_ras3 ? "1" : "0"	);
    strRet.appStrRaw( " have_qfa: "				);
    strRet.appStrRaw( mtconfiginfo.have_qfa  ? "1" : "0"	);

    strRet.appStrRaw( "\n    pad1: 0x"				);
    strRet.appIntFmt( mtconfiginfo.pad1, 16, 2			);

    for ( int iRes = 0; iRes < WhCount(mtconfiginfo.reserved); ++iRes )
    {
	if ( iRes % 12 == 0 )
	    strRet.appStrRaw( "\n    reserved: 0x"		);
	else
	    strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( mtconfiginfo.reserved[iRes], 16, 2	);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtMtget( WhString & strRet, const MmSeg & segMtget )
{
    segMtget.checkCountOne( sizeof(struct mtget) );

    const struct mtget & mtget = * (const struct mtget *) segMtget.address( );

    strRet.appStrRaw( "    mt_type: 0x"			);
    strRet.appIntFmt( mtget.mt_type, 16, 8		);
    strRet.appStrRaw( " mt_resid: "			);
    strRet.appIntFmt( mtget.mt_resid			);

    strRet.appStrRaw( "\n" "    mt_dsreg: 0x"		);
    strRet.appIntFmt( mtget.mt_dsreg, 16, 8		);
    strRet.appStrRaw( " mt_gstat: 0x"			);
    strRet.appIntFmt( mtget.mt_gstat, 16, 8		);
    strRet.appStrRaw( " mt_erreg: 0x"			);
    strRet.appIntFmt( mtget.mt_erreg, 16, 8		);

    strRet.appStrRaw( "\n" "    mt_fileno: "		);
    strRet.appIntFmt( mtget.mt_fileno			);
    strRet.appStrRaw( " mt_blkno: "			);
    strRet.appIntFmt( mtget.mt_blkno			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtMtop( WhString & strRet, const MmSeg & segMtop )
{
    segMtop.checkCountOne( sizeof(struct mtop) );

    const struct mtop & mtop = * (const struct mtop *) segMtop.address( );

    strRet.appStrRaw( "    mt_op: "			);
    strRet.appIntFmt( mtop.mt_op			);
    strRet.appStrRaw( " mt_count: "			);
    strRet.appIntFmt( mtop.mt_count			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtMtpos( WhString & strRet, const MmSeg & segMtpos )
{
    segMtpos.checkCountOne( sizeof(struct mtpos) );

    const struct mtpos & mtpos = * (const struct mtpos *) segMtpos.address( );

    strRet.appStrRaw( "    mt_blkno: "			);
    strRet.appIntFmt( mtpos.mt_blkno			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtNewStat( WhString & strRet, const MmSeg & segNewStat )
{
    segNewStat.checkCountOne( sizeof(struct new_stat) );
    const struct new_stat & new_stat =
    	* (const struct new_stat *) segNewStat.address( );

    strRet.appStrRaw( "    st_dev: "			);
    strRet.appIntFmt( new_stat.st_dev			);
    strRet.appStrRaw( " st_ino: "			);
    strRet.appIntFmt( new_stat.st_ino			);
    strRet.appStrRaw( " st_mode: 0"			);
    strRet.appIntFmt( new_stat.st_mode,  8, 0		);
    strRet.appStrRaw( " st_nlink: "			);
    strRet.appIntFmt( new_stat.st_nlink			);
    strRet.appStrRaw( "\n" "    st_uid: "		);
    strRet.appIntFmt( new_stat.st_uid			);
    strRet.appStrRaw( " st_gid: "			);
    strRet.appIntFmt( new_stat.st_gid			);

    strRet.appStrRaw( "\n" "    st_rdev: "		);
    strRet.appIntFmt( new_stat.st_rdev			);
    strRet.appStrRaw( " st_size: "			);
    strRet.appIntFmt( new_stat.st_size			);
    strRet.appStrRaw( " st_blksize: "			);
    strRet.appIntFmt( new_stat.st_blksize		);
    strRet.appStrRaw( " st_blocks: "			);
    strRet.appIntFmt( new_stat.st_blocks		);

    strRet.appStrRaw( "\n" "    st_atime: "		);
    strRet.appIntFmt( new_stat.st_atime			);
    strRet.appStrRaw( " st_mtime: "			);
    strRet.appIntFmt( new_stat.st_mtime			);
    strRet.appStrRaw( " st_ctime: "			);
    strRet.appIntFmt( new_stat.st_ctime			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtNewUtsname( WhString & strRet, const MmSeg & segNewUtsname )
{
    segNewUtsname.checkCountOne( sizeof(struct new_utsname) );
    const struct new_utsname & new_utsname =
	* (const struct new_utsname *) segNewUtsname.address( );

    strRet.appStrRaw( "    sysname: \""				);
    strRet.appStrFmt( new_utsname.sysname,  WhCount( new_utsname.sysname  ) );
    strRet.appStrRaw( "\"\n" "    nodename: \""			);
    strRet.appStrFmt( new_utsname.nodename, WhCount( new_utsname.nodename ) );
    strRet.appStrRaw( "\"\n" "    release: \""			);
    strRet.appStrFmt( new_utsname.release,  WhCount( new_utsname.release  ) );
    strRet.appStrRaw( "\"\n" "    version: \""			);
    strRet.appStrFmt( new_utsname.version,  WhCount( new_utsname.version  ) );
    strRet.appStrRaw( "\"\n" "    machine: \""			);
    strRet.appStrFmt( new_utsname.machine,  WhCount( new_utsname.machine  ) );
    strRet.appStrRaw( "\"\n" "    domainname: \""		);
    strRet.appStrFmt( new_utsname.domainname, WhCount(new_utsname.domainname));
    strRet.appStrRaw( "\"\n"					);
}



// Format a segment.
void FmtNrParmsStruct( WhString & strRet, const MmSeg & segNrParmsStruct )
{
    segNrParmsStruct.checkCountOne( sizeof(struct nr_parms_struct) );

    const struct nr_parms_struct & nr_parms_struct =
	* (const struct nr_parms_struct *) segNrParmsStruct.address( );

    strRet.appStrRaw( "    quality: "			);
    strRet.appIntFmt( nr_parms_struct.quality		);
    strRet.appStrRaw( " obs_count: "			);
    strRet.appIntFmt( nr_parms_struct.obs_count		);
    strRet.appStrRaw( " ttl: "				);
    strRet.appIntFmt( nr_parms_struct.ttl		);
    strRet.appStrRaw( " timeout: "			);
    strRet.appIntFmt( nr_parms_struct.timeout		);

    strRet.appStrRaw( "\n" "    ack_delay: "		);
    strRet.appIntFmt( nr_parms_struct.ack_delay		);
    strRet.appStrRaw( " busy_delay: "			);
    strRet.appIntFmt( nr_parms_struct.busy_delay	);
    strRet.appStrRaw( " tries: "			);
    strRet.appIntFmt( nr_parms_struct.tries		);
    strRet.appStrRaw( " window: "			);
    strRet.appIntFmt( nr_parms_struct.window		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtOldStat( WhString & strRet, const MmSeg & segOldStat )
{
    segOldStat.checkCountOne( sizeof(struct old_stat) );

    const struct old_stat & old_stat =
	* (const struct old_stat *) segOldStat.address( );

    strRet.appStrRaw( "    st_dev: "			);
    strRet.appIntFmt( old_stat.st_dev			);
    strRet.appStrRaw( " st_ino: "			);
    strRet.appIntFmt( old_stat.st_ino			);
    strRet.appStrRaw( " st_mode: 0"			);
    strRet.appIntFmt( old_stat.st_mode, 8, 0		);
    strRet.appStrRaw( " st_nlink: "			);
    strRet.appIntFmt( old_stat.st_nlink			);

    strRet.appStrRaw( "\n" "    st_uid: "		);
    strRet.appIntFmt( old_stat.st_uid			);
    strRet.appStrRaw( " st_gid: "			);
    strRet.appIntFmt( old_stat.st_gid			);

    strRet.appStrRaw( "\n" "    st_rdev: "		);
    strRet.appIntFmt( old_stat.st_rdev			);
    strRet.appStrRaw( " st_size: "			);
    strRet.appIntFmt( old_stat.st_size			);

    strRet.appStrRaw( "\n" "    st_atime: "		);
    strRet.appIntFmt( old_stat.st_atime			);
    strRet.appStrRaw( " st_mtime: "			);
    strRet.appIntFmt( old_stat.st_mtime			);
    strRet.appStrRaw( " st_ctime: "			);
    strRet.appIntFmt( old_stat.st_ctime			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtOldUtsname( WhString & strRet, const MmSeg & segOldUtsname )
{
    segOldUtsname.checkCountOne( sizeof(struct old_utsname) );

    const struct old_utsname & old_utsname =
	* (const struct old_utsname *) segOldUtsname.address( );

    strRet.appStrRaw( "    sysname: \""					);
    strRet.appStrFmt( &old_utsname.sysname[0],
		      sizeof(old_utsname.sysname)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    nodename: \""				);
    strRet.appStrFmt( &old_utsname.nodename [0],
		      sizeof(old_utsname.nodename) / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    release: \""				);
    strRet.appStrFmt( &old_utsname.release  [0],
		      sizeof(old_utsname.release)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    version: \""				);
    strRet.appStrFmt( &old_utsname.version  [0],
		      sizeof(old_utsname.version)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    machine: \""				);
    strRet.appStrFmt( &old_utsname.machine  [0],
		      sizeof(old_utsname.machine)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n"						);
}



// Format a segment.
void FmtOldoldUtsname( WhString & strRet, const MmSeg & segOldoldUtsname )
{
    segOldoldUtsname.checkCountOne( sizeof(struct oldold_utsname) );

    const struct oldold_utsname & oldold_utsname =
	* (const struct oldold_utsname *) segOldoldUtsname.address( );

    strRet.appStrRaw( "    sysname: \""					);
    strRet.appStrFmt( &oldold_utsname.sysname  [0],
		      sizeof(oldold_utsname.sysname)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    nodename: \""				);
    strRet.appStrFmt( &oldold_utsname.nodename [0],
		      sizeof(oldold_utsname.nodename) / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    release: \""				);
    strRet.appStrFmt( &oldold_utsname.release  [0],
		      sizeof(oldold_utsname.release)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    version: \""				);
    strRet.appStrFmt( &oldold_utsname.version  [0],
    		      sizeof(oldold_utsname.version)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    machine: \""				);
    strRet.appStrFmt( &oldold_utsname.machine  [0],
    		      sizeof(oldold_utsname.machine)  / sizeof(char)	);

    strRet.appStrRaw( "\"\n"						);
}



// Format a segment.
void FmtPatmgrInfo( WhString & strRet, const MmSeg & segPatmgrInfo )
{
    segPatmgrInfo.checkCountOne( sizeof(struct patmgr_info) );

    const struct patmgr_info & patmgr_info =
	* (const struct patmgr_info *) segPatmgrInfo.address( );

    strRet.appStrRaw( "    key: "				);
    strRet.appIntFmt( patmgr_info.key				);
    strRet.appStrRaw( " device: "				);
    strRet.appIntFmt( patmgr_info.device			);
    strRet.appStrRaw( " command: 0x"				);
    strRet.appIntFmt( patmgr_info.command, 16, 4		);

    strRet.appStrRaw( "\n" "    parm1: 0x"			);
    strRet.appIntFmt( patmgr_info.parm1, 16, 8			);
    strRet.appStrRaw( " parm2: 0x"				);
    strRet.appIntFmt( patmgr_info.parm2, 16, 8			);
    strRet.appStrRaw( " parm3: 0x"				);
    strRet.appIntFmt( patmgr_info.parm3, 16, 8			);

    for ( int iData = 0; iData < WhCount(patmgr_info.data.data32); ++iData )
    {
	if ( iData % 4 == 0 )
	    strRet.appStrRaw( "\n    data:"			);
	strRet.appStrRaw( " 0x"					);
	strRet.appIntFmt( patmgr_info.data.data32[iData]	);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtPidT( WhString & strRet, const MmSeg & segPidT )
{
    segPidT.checkCountOne( sizeof(pid_t) );
    const pid_t & pid = * (const pid_t *) segPidT.address( );

    strRet.appStrRaw( "    pid: "		);
    strRet.appIntFmt( pid			);
    strRet.appChrRaw( '\n'			);

}



// Format a segment.
void FmtPppCompStats( WhString & strRet, const MmSeg & segPppCompStats )
{
    segPppCompStats.checkCountOne( sizeof(struct ppp_comp_stats) );
    FmtAny( strRet, segPppCompStats );
}



// Format a segment.
void FmtPppIdle( WhString & strRet, const MmSeg & segPppIdle )
{
    segPppIdle.checkCountOne( sizeof(struct ppp_idle) );

    const struct ppp_idle & ppp_idle =
	* (const struct ppp_idle *) segPppIdle.address( );

    strRet.appStrRaw( "    xmit_idle: "			);
    strRet.appIntFmt( ppp_idle.xmit_idle		);
    strRet.appStrRaw( " recv_idle: "			);
    strRet.appIntFmt( ppp_idle.recv_idle		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtPppOptionData( WhString & strRet, const MmSeg & segPppOptionData )
{
    segPppOptionData.checkCountOne( sizeof(struct ppp_option_data) );

    const struct ppp_option_data & ppp_option_data =
	* (const struct ppp_option_data *) segPppOptionData.address( );

    strRet.appStrRaw( "    ptr: "			);
    strRet.appPtrFmt( ppp_option_data.ptr		);
    strRet.appStrRaw( " length: "			);
    strRet.appIntFmt( ppp_option_data.length		);
    strRet.appStrRaw( " transmit: "			);
    strRet.appIntFmt( ppp_option_data.transmit		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtPppStats( WhString & strRet, const MmSeg & segPppStats )
{
    segPppStats.checkCountOne( sizeof(struct ppp_stats) );
    FmtAny( strRet, segPppStats );
}



// Format a segment.
void FmtRlimit( WhString & strRet, const MmSeg & segRlimit )
{
    segRlimit.checkCountOne( sizeof(struct rlimit) );
    const struct rlimit & rlimit =
	* (const struct rlimit *) segRlimit.address( );

    strRet.appStrRaw( "    rlim_cur: "			);
    strRet.appIntFmt( rlimit.rlim_cur			);
    strRet.appStrRaw( " rlim_max: "			);
    strRet.appIntFmt( rlimit.rlim_max			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtRtentry( WhString & strRet, const MmSeg & segRtentry )
{
    segRtentry.checkCountOne( sizeof(struct rtentry) );
    const struct rtentry & rtentry =
	* (const struct rtentry *) segRtentry.address( );

    strRet.appStrRaw( "    rt_hash: 0x"			);
    strRet.appIntFmt( rtentry.rt_hash, 16, 8		);

    strRet.appStrRaw( "\n" "    rt_dst:"		);
    strRet.appIntFmt( rtentry.rt_dst.sa_family		);
    for ( int iDstData  = 0;
	      iDstData  < WhCount(rtentry.rt_dst.sa_data);
	      iDstData += 1 )
    {
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( rtentry.rt_dst.sa_data[iDstData], 16, 2 );
    }

    strRet.appStrRaw( "\n" "    rt_gateway:"		);
    strRet.appIntFmt( rtentry.rt_gateway.sa_family	);
    for ( int iGatewayData  = 0;
	      iGatewayData  < WhCount(rtentry.rt_gateway.sa_data);
	      iGatewayData += 1 )
    {
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( rtentry.rt_gateway.sa_data[iGatewayData], 16, 2 );
    }

    strRet.appStrRaw( "\n" "    rt_genmask:"		);
    strRet.appIntFmt( rtentry.rt_genmask.sa_family	);
    for ( int iGenmaskData  = 0;
	      iGenmaskData  < WhCount(rtentry.rt_genmask.sa_data);
	      iGenmaskData += 1 )
    {
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( rtentry.rt_genmask.sa_data[iGenmaskData], 16, 2 );
    }

    strRet.appStrRaw( "\n" "    rt_flags: 0x"		);
    strRet.appIntFmt( rtentry.rt_flags, 16, 4		);
    strRet.appStrRaw( " rt_refcnt: "			);
    strRet.appIntFmt( rtentry.rt_refcnt			);
    strRet.appStrRaw( " rt_use: "			);
    strRet.appIntFmt( rtentry.rt_use			);
    strRet.appStrRaw( " rt_ifp: "			);
    strRet.appPtrFmt( rtentry.rt_ifp			);

    strRet.appStrRaw( "\n" "    rt_metric: "		);
    strRet.appIntFmt( rtentry.rt_metric			);
    strRet.appStrRaw( " rt_dev: "			);
    strRet.appPtrFmt( rtentry.rt_dev			);
    strRet.appStrRaw( " rt_mss: "			);
    strRet.appIntFmt( rtentry.rt_mss			);
    strRet.appStrRaw( " rt_window: "			);
    strRet.appIntFmt( rtentry.rt_window			);

    strRet.appStrRaw( "\n" "    rt_irtt: "		);
    strRet.appIntFmt( rtentry.rt_irtt			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtRusage( WhString & strRet, const MmSeg & segRusage )
{
    segRusage.checkCountOne( sizeof(struct rusage) );
    const struct rusage & rusage =
	* (const struct rusage *) segRusage.address( );

    strRet.appStrRaw( "    ru_utime: "			);
    strRet.appIntFmt( rusage.ru_utime.tv_sec		);
    strRet.appChrRaw( '.'				);
    strRet.appIntFmt( rusage.ru_utime.tv_usec, 10, 6	);
    strRet.appStrRaw( " ru_stime: "			);
    strRet.appIntFmt( rusage.ru_stime.tv_sec		);
    strRet.appChrRaw( '.'				);
    strRet.appIntFmt( rusage.ru_stime.tv_usec, 10, 6	);

    strRet.appStrRaw( "\n" "    ru_maxrss: "		);
    strRet.appIntFmt( rusage.ru_maxrss			);
    strRet.appStrRaw( " ru_ixrss: "			);
    strRet.appIntFmt( rusage.ru_ixrss			);
    strRet.appStrRaw( " ru_idrss: "			);
    strRet.appIntFmt( rusage.ru_idrss			);
    strRet.appStrRaw( " ru_isrss: "			);
    strRet.appIntFmt( rusage.ru_isrss			);

    strRet.appStrRaw( "\n" "    ru_minflt: "		);
    strRet.appIntFmt( rusage.ru_minflt			);
    strRet.appStrRaw( " ru_majflt: "			);
    strRet.appIntFmt( rusage.ru_majflt			);
    strRet.appStrRaw( " ru_nswap: "			);
    strRet.appIntFmt( rusage.ru_nswap			);
    strRet.appStrRaw( " ru_inblock: "			);
    strRet.appIntFmt( rusage.ru_inblock			);
    strRet.appStrRaw( " ru_oublock: "			);
    strRet.appIntFmt( rusage.ru_oublock			);

    strRet.appStrRaw( "\n" "    ru_msgsnd: "		);
    strRet.appIntFmt( rusage.ru_msgsnd			);
    strRet.appStrRaw( " ru_msgrcv: "			);
    strRet.appIntFmt( rusage.ru_msgrcv			);
    strRet.appStrRaw( " ru_nsignals: "			);
    strRet.appIntFmt( rusage.ru_nsignals		);
    strRet.appStrRaw( " ru_nvcsw: "			);
    strRet.appIntFmt( rusage.ru_nvcsw			);
    strRet.appStrRaw( " ru_nivcsw: "			);
    strRet.appIntFmt( rusage.ru_nivcsw			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSbiInstrument( WhString & strRet, const MmSeg & segSbiInstrument )
{
    segSbiInstrument.checkCountOne( sizeof(struct sbi_instrument) );

    const struct sbi_instrument & sbi_instrument =
	* (const struct sbi_instrument *) segSbiInstrument.address( );

    strRet.appStrRaw( "    key: 0x"			);
    strRet.appIntFmt( sbi_instrument.key, 16, 4		);
    strRet.appStrRaw( " device: "			);
    strRet.appIntFmt( sbi_instrument.device		);
    strRet.appStrRaw( " channel: "			);
    strRet.appIntFmt( sbi_instrument.channel		);

    for ( int iOperator  = 0;
	      iOperator  < WhCount(sbi_instrument.operators);
	      iOperator += 1 )
    {
	if ( iOperator % 8 == 0 )
	    strRet.appStrRaw( "\n    operators:"	);
	strRet.appStrRaw( " 0x"				);
	strRet.appIntFmt( sbi_instrument.operators[iOperator], 16, 2 );
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSccModem( WhString & strRet, const MmSeg & segSccModem )
{
    segSccModem.checkCountOne( sizeof(struct scc_modem) );
    const struct scc_modem & scc_modem =
	* (const struct scc_modem *) segSccModem.address( );

    strRet.appStrRaw( "    speed: "			);
    strRet.appIntFmt( scc_modem.speed			);
    strRet.appStrRaw( " clocksrc: "			);
    strRet.appIntFmt( scc_modem.clocksrc		);
    strRet.appStrRaw( " nrz: "				);
    strRet.appIntFmt( scc_modem.nrz			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSccStat( WhString & strRet, const MmSeg & segSccStat )
{
    segSccStat.checkCountOne( sizeof(struct scc_stat) );
    const struct scc_stat & scc_stat =
	* (const struct scc_stat *) segSccStat.address( );

    strRet.appStrRaw( "    rxints: "			);
    strRet.appIntFmt( scc_stat.rxints			);
    strRet.appStrRaw( " txints: "			);
    strRet.appIntFmt( scc_stat.txints			);
    strRet.appStrRaw( " exints: "			);
    strRet.appIntFmt( scc_stat.exints			);
    strRet.appStrRaw( " spints: "			);
    strRet.appIntFmt( scc_stat.spints			);

    strRet.appStrRaw( "\n" "    txframes: "		);
    strRet.appIntFmt( scc_stat.txframes			);
    strRet.appStrRaw( " rxframes: "			);
    strRet.appIntFmt( scc_stat.rxframes			);
    strRet.appStrRaw( " rxerrs: "			);
    strRet.appIntFmt( scc_stat.rxerrs			);
    strRet.appStrRaw( " txerrs: "			);
    strRet.appIntFmt( scc_stat.txerrs			);
        
    strRet.appStrRaw( "\n" "    nospace: "		);
    strRet.appIntFmt( scc_stat.nospace			);
    strRet.appStrRaw( " rx_over: "			);
    strRet.appIntFmt( scc_stat.rx_over			);
    strRet.appStrRaw( " tx_under: "			);
    strRet.appIntFmt( scc_stat.tx_under			);

    strRet.appStrRaw( "\n" "    tx_state: 0x"		);
    strRet.appIntFmt( scc_stat.tx_state, 16, 8		);
    strRet.appStrRaw( " tx_kiss_state: 0x"		);
    strRet.appIntFmt( scc_stat.tx_kiss_state, 16, 2	);
    strRet.appStrRaw( " rx_kiss_state: 0x"		);
    strRet.appIntFmt( scc_stat.rx_kiss_state, 16, 2	);
	
    strRet.appStrRaw( "\n" "    tx_queued: "		);
    strRet.appIntFmt( scc_stat.tx_queued		);
    strRet.appStrRaw( " rx_queued: "			);
    strRet.appIntFmt( scc_stat.rx_queued		);
	
    strRet.appStrRaw( "\n" "    rx_alloc: "		);
    strRet.appIntFmt( scc_stat.rx_alloc			);
    strRet.appStrRaw( " tx_alloc: "			);
    strRet.appIntFmt( scc_stat.tx_alloc			);
    strRet.appStrRaw( " used_buf: "			);
    strRet.appIntFmt( scc_stat.used_buf			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSemidDs( WhString & strRet, const MmSeg & segSemidDs )
{
    segSemidDs.checkCountOne( sizeof(struct semid_ds) );
    const struct semid_ds & semid_ds =
	* (const struct semid_ds *) segSemidDs.address( );

    strRet.appStrRaw( "    sem_perm.key: 0x"		);
    strRet.appIntFmt( semid_ds.sem_perm.key, 16, 8	);
    strRet.appStrRaw( " sem_perm.uid: "			);
    strRet.appIntFmt( semid_ds.sem_perm.uid		);
    strRet.appStrRaw( " sem_perm.gid: "			);
    strRet.appIntFmt( semid_ds.sem_perm.gid		);

    strRet.appStrRaw( "\n" "    sem_perm.cuid: "	);
    strRet.appIntFmt( semid_ds.sem_perm.cuid		);
    strRet.appStrRaw( " sem_perm.cgid: "		);
    strRet.appIntFmt( semid_ds.sem_perm.cgid		);
    strRet.appStrRaw( " sem_perm.mode: 0"		);
    strRet.appIntFmt( semid_ds.sem_perm.mode, 8, 0	);

    strRet.appStrRaw( "\n" "    sem_perm.seq: "		);
    strRet.appIntFmt( semid_ds.sem_perm.seq		);

    strRet.appStrRaw( "\n" "    sem_otime: "		);
    strRet.appIntFmt( semid_ds.sem_otime		);
    strRet.appStrRaw( " sem_ctime: "			);
    strRet.appIntFmt( semid_ds.sem_ctime		);
    strRet.appStrRaw( " sem_base: "			);
    strRet.appPtrFmt( semid_ds.sem_base			);

    strRet.appStrRaw( "\n" "    sem_pending: "		);
    strRet.appPtrFmt( semid_ds.sem_pending		);
    strRet.appStrRaw( " sem_pending_last: "		);
    strRet.appPtrFmt( semid_ds.sem_pending_last		);
    strRet.appStrRaw( " undo: "				);
    strRet.appPtrFmt( semid_ds.undo			);

    strRet.appStrRaw( "\n" "    sem_nsems: "		);
    strRet.appIntFmt( semid_ds.sem_nsems		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSeminfo( WhString & strRet, const MmSeg & segSeminfo )
{
    segSeminfo.checkCountOne( sizeof(struct seminfo) );
    const struct seminfo & seminfo =
	* (const struct seminfo *) segSeminfo.address( );

    strRet.appStrRaw( "    semmap: "			);
    strRet.appIntFmt( seminfo.semmap			);
    strRet.appStrRaw( " semmni: "			);
    strRet.appIntFmt( seminfo.semmni			);
    strRet.appStrRaw( " semmns: "			);
    strRet.appIntFmt( seminfo.semmns			);
    strRet.appStrRaw( " semmnu: "			);
    strRet.appIntFmt( seminfo.semmnu			);

    strRet.appStrRaw( "\n" "    semmsl: "		);
    strRet.appIntFmt( seminfo.semmsl			);
    strRet.appStrRaw( " semopm: "			);
    strRet.appIntFmt( seminfo.semopm			);
    strRet.appStrRaw( " semume: "			);
    strRet.appIntFmt( seminfo.semume			);
    strRet.appStrRaw( " semusz: "			);
    strRet.appIntFmt( seminfo.semusz			);

    strRet.appStrRaw( "\n" "    semvmx: "		);
    strRet.appIntFmt( seminfo.semvmx			);
    strRet.appStrRaw( " semaem: "			);
    strRet.appIntFmt( seminfo.semaem			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSemun( WhString & strRet, const MmSeg & segSemun )
{
    segSemun.checkCountOne( sizeof(union semun) );
    FmtAny( strRet, segSemun );
}



// Format a segment.
void FmtSeqEventRec( WhString & strRet, const MmSeg & segSeqEventRec )
{
    segSeqEventRec.checkCountOne( sizeof(struct seq_event_rec) );
    FmtListMmByte( strRet, segSeqEventRec );
}



// Format a segment.
void FmtSerialMultiportStruct(
    WhString & strRet, const MmSeg & segSerialMultiportStruct )
{
    segSerialMultiportStruct.checkCountOne(
	sizeof(struct serial_multiport_struct) );

    const struct serial_multiport_struct & serial_multiport_struct =
	* (const struct serial_multiport_struct *)
	segSerialMultiportStruct.address( );

    strRet.appStrRaw( " irq: "						);
    strRet.appIntFmt( serial_multiport_struct.irq			);

    strRet.appStrRaw( "\n" "    port1: "				);
    strRet.appIntFmt( serial_multiport_struct.port1			);
    strRet.appStrRaw( " mask1: "					);
    strRet.appIntFmt( serial_multiport_struct.mask1			);
    strRet.appStrRaw( " match1: "					);
    strRet.appIntFmt( serial_multiport_struct.match1			);

    strRet.appStrRaw( "\n" "    port2: "				);
    strRet.appIntFmt( serial_multiport_struct.port2			);
    strRet.appStrRaw( " mask2: "					);
    strRet.appIntFmt( serial_multiport_struct.mask2			);
    strRet.appStrRaw( " match2: "					);
    strRet.appIntFmt( serial_multiport_struct.match2			);

    strRet.appStrRaw( "\n" "    port3: "				);
    strRet.appIntFmt( serial_multiport_struct.port3			);
    strRet.appStrRaw( " mask3: "					);
    strRet.appIntFmt( serial_multiport_struct.mask3			);
    strRet.appStrRaw( " match3: "					);
    strRet.appIntFmt( serial_multiport_struct.match3			);

    strRet.appStrRaw( "\n" "    port4: "				);
    strRet.appIntFmt( serial_multiport_struct.port4			);
    strRet.appStrRaw( " mask4: "					);
    strRet.appIntFmt( serial_multiport_struct.mask4			);
    strRet.appStrRaw( " match4: "					);
    strRet.appIntFmt( serial_multiport_struct.match4			);

    strRet.appStrRaw( "\n" "    port_monitor: "				);
    strRet.appIntFmt( serial_multiport_struct.port_monitor		);

    strRet.appStrRaw( "\n" "    reserved: "				);
    for ( int ir = 0; ir < WhCount(serial_multiport_struct.reserved); ++ir )
    {
	if ( ir % 4 == 0 )
	    strRet.appStrRaw( "\n       "				);
	strRet.appStrRaw( " 0x"						);
	strRet.appIntFmt( serial_multiport_struct.reserved[ir], 16, 8	);
    }

    strRet.appChrRaw( '\n'						);
}



// Format a segment.
void FmtSerialStruct( WhString & strRet, const MmSeg & segSerialStruct )
{
    segSerialStruct.checkCountOne( sizeof(struct serial_struct) );

    const struct serial_struct & serial_struct =
	* (const struct serial_struct *) segSerialStruct.address( );

    strRet.appStrRaw( "    type: "			);
    strRet.appIntFmt( serial_struct.type		);
    strRet.appStrRaw( " line: "				);
    strRet.appIntFmt( serial_struct.line		);
    strRet.appStrRaw( " port: 0x"			);
    strRet.appIntFmt( serial_struct.port, 16, 4		);
    strRet.appStrRaw( " irq: "				);
    strRet.appIntFmt( serial_struct.irq			);
    strRet.appStrRaw( " flags: 0x"			);
    strRet.appIntFmt( serial_struct.flags, 16, 8	);

    strRet.appStrRaw( "\n" "    xmit_fifo_size: "	);
    strRet.appIntFmt( serial_struct.xmit_fifo_size	);
    strRet.appStrRaw( " custom_divisor: "		);
    strRet.appIntFmt( serial_struct.custom_divisor	);
    strRet.appStrRaw( " baud_base: "			);
    strRet.appIntFmt( serial_struct.baud_base		);

    strRet.appStrRaw( "\n" "    close_delay: "		);
    strRet.appIntFmt( serial_struct.close_delay		);
    strRet.appStrRaw( " reserved_char: "		);
    for ( int iRc = 0; iRc < WhCount(serial_struct.reserved_char); ++iRc )
    {
	if ( iRc > 0 )
	    strRet.appChrRaw( ' '			);
	strRet.appStrRaw( "0x"				);
	strRet.appIntFmt( serial_struct.reserved_char[iRc], 16, 2  );
    }
    strRet.appStrRaw( " hub6: "				);
    strRet.appIntFmt( serial_struct.hub6		);

    strRet.appStrRaw( "\n" "    closing_wait: "		);
    strRet.appIntFmt( serial_struct.closing_wait	);
    strRet.appStrRaw( " closing_wait2: "		);
    strRet.appIntFmt( serial_struct.closing_wait2	);

    strRet.appStrRaw( "\n" "    reserved: "		);
    for ( int iRes = 0; iRes < WhCount(serial_struct.reserved); ++iRes )
    {
	if ( iRes > 0 )
	    strRet.appChrRaw( ' '			);
	strRet.appIntFmt( serial_struct.reserved[iRes]	);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtShmInfo( WhString & strRet, const MmSeg & segShmInfo )
{
    segShmInfo.checkCountOne( sizeof(struct shm_info) );

    const struct shm_info & shm_info =
	* (const struct shm_info *) segShmInfo.address( );

    strRet.appStrRaw( "    used_ids: "			);
    strRet.appIntFmt( shm_info.used_ids			);
    strRet.appStrRaw( " shm_tot: "			);
    strRet.appIntFmt( shm_info.shm_tot			);
    strRet.appStrRaw( " shm_rss: "			);
    strRet.appIntFmt( shm_info.shm_rss			);
    strRet.appStrRaw( " shm_swp: "			);
    strRet.appIntFmt( shm_info.shm_swp			);

    strRet.appStrRaw( "\n" "    swap_attempts: "	);
    strRet.appIntFmt( shm_info.swap_attempts		);
    strRet.appStrRaw( " swap_successes: "		);
    strRet.appIntFmt( shm_info.swap_successes		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtShmidDs( WhString & strRet, const MmSeg & segShmidDs )
{
    segShmidDs.checkCountOne( sizeof(struct shmid_ds) );
    const struct shmid_ds & shmid_ds =
	* (const struct shmid_ds *) segShmidDs.address( );

    strRet.appStrRaw( "    shm_perm.key: 0x"		);
    strRet.appIntFmt( shmid_ds.shm_perm.key, 16, 8	);
    strRet.appStrRaw( " shm_perm.uid: "			);
    strRet.appIntFmt( shmid_ds.shm_perm.uid		);
    strRet.appStrRaw( " shm_perm.gid: "			);
    strRet.appIntFmt( shmid_ds.shm_perm.gid		);

    strRet.appStrRaw( "\n" "    shm_perm.cuid: "	);
    strRet.appIntFmt( shmid_ds.shm_perm.cuid		);
    strRet.appStrRaw( " shm_perm.cgid: "		);
    strRet.appIntFmt( shmid_ds.shm_perm.cgid		);
    strRet.appStrRaw( " shm_perm.mode: 0"		);
    strRet.appIntFmt( shmid_ds.shm_perm.mode, 8, 0	);

    strRet.appStrRaw( "\n" "    shm_perm.seq: "		);
    strRet.appIntFmt( shmid_ds.shm_perm.seq		);

    strRet.appStrRaw( "\n" "    shm_segsz: "		);
    strRet.appIntFmt( shmid_ds.shm_segsz		);

    strRet.appStrRaw( "\n" "    shm_atime: "		);
    strRet.appIntFmt( shmid_ds.shm_atime		);
    strRet.appStrRaw( " shm_dtime: "			);
    strRet.appIntFmt( shmid_ds.shm_dtime		);
    strRet.appStrRaw( " shm_ctime: "			);
    strRet.appIntFmt( shmid_ds.shm_ctime		);

    strRet.appStrRaw( "\n" "    shm_cpid: "		);
    strRet.appIntFmt( shmid_ds.shm_cpid			);
    strRet.appStrRaw( " shm_lpid: "			);
    strRet.appIntFmt( shmid_ds.shm_lpid			);
    strRet.appStrRaw( " shm_nattch: "			);
    strRet.appIntFmt( shmid_ds.shm_nattch		);

    strRet.appStrRaw( "\n" "    shm_npages: "		);
    strRet.appIntFmt( shmid_ds.shm_npages		);
    strRet.appStrRaw( " shm_pages: "			);
    strRet.appPtrFmt( shmid_ds.shm_pages		);
    strRet.appStrRaw( " attaches: "			);
    strRet.appPtrFmt( shmid_ds.attaches			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtShminfo( WhString & strRet, const MmSeg & segShminfo )
{
    segShminfo.checkCountOne( sizeof(struct shminfo) );
    const struct shminfo & shminfo =
	* (const struct shminfo *) segShminfo.address( );

    strRet.appStrRaw( "    shmmax: "			);
    strRet.appIntFmt( shminfo.shmmax			);
    strRet.appStrRaw( " shmmin: "			);
    strRet.appIntFmt( shminfo.shmmin			);
    strRet.appStrRaw( " shmmni: "			);
    strRet.appIntFmt( shminfo.shmmni			);
    strRet.appStrRaw( " shmseg: "			);
    strRet.appIntFmt( shminfo.shmseg			);
    strRet.appStrRaw( " shmall: "			);
    strRet.appIntFmt( shminfo.shmall			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSigaction( WhString & strRet, const MmSeg & segSigaction )
{
    segSigaction.checkCountOne( sizeof(struct sigaction) );
    const struct sigaction & sigaction =
	* (const struct sigaction *) segSigaction.address( );

    strRet.appStrRaw( "    sa_handler: "			);
    strRet.appPtrFmt( (const void *) sigaction.sa_handler	);
    strRet.appStrRaw( " sa_mask: 0x"				);
    strRet.appIntFmt( sigaction.sa_mask, 16, 8			);
    strRet.appStrRaw( " sa_flags: 0x"				);
    strRet.appIntFmt( sigaction.sa_flags, 16, 8			);

    strRet.appStrRaw( "\n" "    sa_restorer: "			);
    strRet.appPtrFmt( (const void *) sigaction.sa_restorer	);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtSigcontextStruct(
    WhString & strRet, const MmSeg & segSigcontextStruct )
{
    segSigcontextStruct.checkCountOne( sizeof(struct sigcontext_struct) );

    const struct sigcontext_struct & sigcontext_struct =
	* (const struct sigcontext_struct *) segSigcontextStruct.address( );

    strRet.appStrRaw( "     gs: 0x"				);
    strRet.appIntFmt( sigcontext_struct.__gsh << 16 | sigcontext_struct.gs,
			16, 8					);
    strRet.appStrRaw( "  fs: 0x"				);
    strRet.appIntFmt( sigcontext_struct.__fsh << 16 | sigcontext_struct.fs,
			16, 8					);
    strRet.appStrRaw( "  es: 0x"				);
    strRet.appIntFmt( sigcontext_struct.__esh << 16 | sigcontext_struct.es,
			16, 8					);
    strRet.appStrRaw( "  ds: 0x"				);
    strRet.appIntFmt( sigcontext_struct.__dsh << 16 | sigcontext_struct.ds,
			16, 8					);

    strRet.appStrRaw( "\n" "    edi: 0x"			);
    strRet.appIntFmt( sigcontext_struct.edi, 16, 8		);
    strRet.appStrRaw( " esi: 0x"				);
    strRet.appIntFmt( sigcontext_struct.esi, 16, 8		);
    strRet.appStrRaw( " ebp: 0x"				);
    strRet.appIntFmt( sigcontext_struct.ebp, 16, 8		);
    strRet.appStrRaw( " esp: 0x"				);
    strRet.appIntFmt( sigcontext_struct.esp, 16, 8		);

    strRet.appStrRaw( "\n" "    ebx: 0x"			);
    strRet.appIntFmt( sigcontext_struct.ebx, 16, 8		);
    strRet.appStrRaw( " edx: 0x"				);
    strRet.appIntFmt( sigcontext_struct.edx, 16, 8		);
    strRet.appStrRaw( " ecx: 0x"				);
    strRet.appIntFmt( sigcontext_struct.ecx, 16, 8		);
    strRet.appStrRaw( " eax: 0x"				);
    strRet.appIntFmt( sigcontext_struct.eax, 16, 8		);

    strRet.appStrRaw( "\n" "    trapno: "			);
    strRet.appIntFmt( sigcontext_struct.trapno			);
    strRet.appStrRaw( " err: "					);
    strRet.appIntFmt( sigcontext_struct.err			);
    strRet.appStrRaw( " cs:eip: 0x"				);
    strRet.appIntFmt( sigcontext_struct.__csh << 16 | sigcontext_struct.cs,
			16, 8					);
    strRet.appStrRaw( ":"					);
    strRet.appIntFmt( sigcontext_struct.eip, 16, 8		);
    strRet.appStrRaw( " eflags: 0x"				);
    strRet.appIntFmt( sigcontext_struct.eflags, 16, 8		);

    strRet.appStrRaw( "\n" "    ss:esp_at_signal: 0x"		);
    strRet.appIntFmt( sigcontext_struct.__ssh << 16 | sigcontext_struct.ss,
			16, 8					);
    strRet.appStrRaw( ":"					);
    strRet.appIntFmt( sigcontext_struct.esp_at_signal, 16, 8	);

    strRet.appStrRaw( "\n" "    i387: 0x"			);
    strRet.appIntFmt( sigcontext_struct.i387, 16, 8		);
    strRet.appStrRaw( " oldmask: "				);
    strRet.appIntFmt( sigcontext_struct.oldmask, 16, 8		);
    strRet.appStrRaw( " cr2: "					);
    strRet.appIntFmt( sigcontext_struct.cr2, 16, 8		);

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtSigsetT( WhString & strRet, const MmSeg & segSigsetT )
{
    segSigsetT.checkCountOne( sizeof(sigset_t) );
    const sigset_t & ss_value = * (const sigset_t *) segSigsetT.address( );

    strRet.appStrRaw( "    0x"				);
    strRet.appIntFmt( ss_value, 16, 8			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSiocSgReq( WhString & strRet, const MmSeg & segSiocSgReq )
{
    segSiocSgReq.checkCountOne( sizeof(struct sioc_sg_req) );
    FmtAny( strRet, segSiocSgReq );
}



// Format a segment.
void FmtSiocVifReq( WhString & strRet, const MmSeg & segSiocVifReq )
{
    segSiocVifReq.checkCountOne( sizeof(struct sioc_vif_req) );

    const struct sioc_vif_req & sioc_vif_req =
	* (const struct sioc_vif_req *) segSiocVifReq.address( );

    strRet.appStrRaw( "    vifi: 0x"			);
    strRet.appIntFmt( sioc_vif_req.vifi, 16, 8		);
    strRet.appStrRaw( " icount: "			);
    strRet.appIntFmt( sioc_vif_req.icount		);
    strRet.appStrRaw( " ocount: "			);
    strRet.appIntFmt( sioc_vif_req.ocount		);
    strRet.appStrRaw( " ibytes: "			);
    strRet.appIntFmt( sioc_vif_req.ibytes		);
    strRet.appStrRaw( " obytes: "			);
    strRet.appIntFmt( sioc_vif_req.obytes		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtSlaveConfig( WhString & strRet, const MmSeg & segSlaveConfig )
{
    segSlaveConfig.checkCountOne( sizeof(struct slave_config) );
    FmtAny( strRet, segSlaveConfig );
}



// Format a segment.
void FmtSlavingRequest( WhString & strRet, const MmSeg & segSlavingRequest )
{
    segSlavingRequest.checkCountOne( sizeof(struct slaving_request) );
    FmtAny( strRet, segSlavingRequest );
}



// Format a segment.
void FmtSockaddrAx25( WhString & strRet, const MmSeg & segSockaddrAx25 )
{
    segSockaddrAx25.checkCountOne( sizeof(struct sockaddr_ax25) );
    FmtListMmByte( strRet, segSockaddrAx25 );
}



// Format a segment.
void FmtStatfs( WhString & strRet, const MmSeg & segStatfs )
{
    segStatfs.checkCountOne( sizeof(struct statfs) );
    const struct statfs & statfs =
	* (const struct statfs *) segStatfs.address( );

    strRet.appStrRaw( "    f_type: 0x"			);
    strRet.appIntFmt( statfs.f_type, 16, 8		);

    strRet.appStrRaw( "\n" "    f_bsize: "		);
    strRet.appIntFmt( statfs.f_bsize			);
    strRet.appStrRaw( " f_blocks: "			);
    strRet.appIntFmt( statfs.f_blocks			);
    strRet.appStrRaw( " f_bfree: "			);
    strRet.appIntFmt( statfs.f_bfree			);
    strRet.appStrRaw( " f_bavail: "			);
    strRet.appIntFmt( statfs.f_bavail			);

    strRet.appStrRaw( "\n" "    f_files: "		);
    strRet.appIntFmt( statfs.f_files			);
    strRet.appStrRaw( " f_ffree: "			);
    strRet.appIntFmt( statfs.f_ffree			);
    strRet.appStrRaw( " f_fsid: 0x"			);
    strRet.appIntFmt( statfs.f_fsid.val[0], 16		);
    strRet.appStrRaw( " 0x"				);
    strRet.appIntFmt( statfs.f_fsid.val[1], 16		);
    strRet.appStrRaw( " f_namelen: "			);
    strRet.appIntFmt( statfs.f_namelen			);

    strRet.appStrRaw( "\n" "    f_spare: "		);
    strRet.appIntFmt( statfs.f_spare[0]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( statfs.f_spare[1]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( statfs.f_spare[2]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( statfs.f_spare[3]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( statfs.f_spare[4]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( statfs.f_spare[5]			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtStrCount( WhString & strRet, const MmSeg & segStr )
{
    segStr.checkCountList( sizeof(MmByte) );
    strRet.appStrRaw( "    \""					);
    const char * pstrRaw = (const char *) segStr.address( );
    const int ncRaw = segStr.count( ) / sizeof(char);
    if ( ncRaw > 0 )
    {
	bool fCcs = true;
	for ( int icRaw = 0; icRaw < ncRaw - 1; ++icRaw )
	{
	    if ( pstrRaw[icRaw] == '\0' )
		fCcs = false;
	    strRet.appChrFmt( pstrRaw[icRaw] );
	}
	if ( pstrRaw[ncRaw - 1] != '\0' )
	    fCcs = false;
	if ( !fCcs )
	    strRet.appChrFmt( pstrRaw[ncRaw - 1] );
    }
    strRet.appStrRaw( "\"\n"					);
}



// Format a segment.
void FmtStrNul( WhString & strRet, const MmSeg & segStr )
{
    segStr.checkCcs( );
    strRet.appStrRaw( "    \""					);
    const char * pstrRaw = (const char *) segStr.address( );
    const int ncRaw = segStr.count( ) / sizeof(char);
    for ( int icRaw = 0; icRaw < ncRaw - 1; ++icRaw )
	strRet.appChrFmt( pstrRaw[icRaw] );
    strRet.appStrRaw( "\"\n"					);
}



// Format a segment.
void FmtSymbolTable( WhString & strRet, const MmSeg & segSymbolTable )
{
    FmtAny( strRet, segSymbolTable );
}



// Format a segment.
void FmtSynthInfo( WhString & strRet, const MmSeg & segSynthInfo )
{
    segSynthInfo.checkCountOne( sizeof(struct synth_info) );
    const struct synth_info & synth_info =
	* (const struct synth_info *) segSynthInfo.address( );

    strRet.appStrRaw( "    name: \""				);
    strRet.appStrFmt( &synth_info.name[0],
    		      sizeof(synth_info.name) / sizeof(char)	);

    strRet.appStrRaw( "\"\n" "    device: "			);
    strRet.appIntFmt( synth_info.device				);
    strRet.appStrRaw( " synth_type: "				);
    strRet.appIntFmt( synth_info.synth_type			);
    strRet.appStrRaw( " synth_subtype: "			);
    strRet.appIntFmt( synth_info.synth_subtype			);
    strRet.appStrRaw( " perc_mode: "				);
    strRet.appIntFmt( synth_info.perc_mode			);

    strRet.appStrRaw( "\n" "    nr_voices: "			);
    strRet.appIntFmt( synth_info.nr_voices			);
    strRet.appStrRaw( " nr_drums: "				);
    strRet.appIntFmt( synth_info.nr_drums			);
    strRet.appStrRaw( " instr_bank_size: "			);
    strRet.appIntFmt( synth_info.instr_bank_size		);
    strRet.appStrRaw( " capabilities: 0x"			);
    strRet.appIntFmt( synth_info.capabilities, 16, 8		);

    for ( int iDummy = 0; iDummy < WhCount(synth_info.dummies); ++iDummy )
    {
	if ( iDummy % 8 == 0 )
	    strRet.appStrRaw( "\n    dummies:"			);
	strRet.appChrRaw( ' '					);
	strRet.appIntFmt( synth_info.dummies[iDummy]		);
    }

    strRet.appChrRaw( '\n'					);
}



// Format a segment.
void FmtSysinfo( WhString & strRet, const MmSeg & segSysinfo )
{
    segSysinfo.checkCountOne( sizeof(struct sysinfo) );
    const struct sysinfo & sysinfo =
	* (const struct sysinfo *) segSysinfo.address( );

    strRet.appStrRaw( "    uptime: "			);
    strRet.appIntFmt( sysinfo.uptime			);
    strRet.appStrRaw( " loads: "			);
    strRet.appIntFmt( sysinfo.loads[0]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( sysinfo.loads[1]			);
    strRet.appChrRaw( ' '				);
    strRet.appIntFmt( sysinfo.loads[2]			);

    strRet.appStrRaw( "\n" "    totalram: "		);
    strRet.appIntFmt( sysinfo.totalram			);
    strRet.appStrRaw( " freeram: "			);
    strRet.appIntFmt( sysinfo.freeram			);
    strRet.appStrRaw( " sharedram: "			);
    strRet.appIntFmt( sysinfo.sharedram			);

    strRet.appStrRaw( "\n" "    bufferram: "		);
    strRet.appIntFmt( sysinfo.bufferram			);
    strRet.appStrRaw( " totalswap: "			);
    strRet.appIntFmt( sysinfo.totalswap			);
    strRet.appStrRaw( " freeswap: "			);
    strRet.appIntFmt( sysinfo.freeswap			);

    strRet.appStrRaw( "\n" "    procs: "		);
    strRet.appIntFmt( sysinfo.procs			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTermio( WhString & strRet, const MmSeg & segTermio )
{
    segTermio.checkCountOne( sizeof(struct termio) );
    const struct termio & termio =
	* (const struct termio *) segTermio.address( );

    strRet.appStrRaw( "    c_iflag: 0x"			);
    strRet.appIntFmt( termio.c_iflag, 16, 4		);
    strRet.appStrRaw( " c_oflag: 0x"			);
    strRet.appIntFmt( termio.c_oflag, 16, 4		);
    strRet.appStrRaw( "\n" "    c_cflag: 0x"		);
    strRet.appIntFmt( termio.c_cflag, 16, 4		);
    strRet.appStrRaw( " c_lflag: 0x"			);
    strRet.appIntFmt( termio.c_lflag, 16, 4		);
    strRet.appStrRaw( "\n" "    c_line: "		);
    strRet.appIntFmt( termio.c_line			);

    for ( int iCc = 0; iCc < WhCount(termio.c_cc); ++iCc )
    {
	if ( iCc % 10 == 0 )
	    strRet.appStrRaw( "\n" "    c_cc: "		);
	else if ( iCc > 0 )
	    strRet.appChrRaw( ' '			);
	strRet.appStrRaw( "0x"				);
	strRet.appIntFmt( termio.c_cc[iCc], 16, 2	);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTermios( WhString & strRet, const MmSeg & segTermios )
{
    segTermios.checkCountOne( sizeof(struct termios) );
    const struct termios & termios =
	* (const struct termios *) segTermios.address( );

    strRet.appStrRaw( "    c_iflag: 0x"			);
    strRet.appIntFmt( termios.c_iflag, 16, 8		);
    strRet.appStrRaw( " c_oflag: 0x"			);
    strRet.appIntFmt( termios.c_oflag, 16, 8		);
    strRet.appStrRaw( "\n" "    c_cflag: 0x"		);
    strRet.appIntFmt( termios.c_cflag, 16, 8		);
    strRet.appStrRaw( " c_lflag: 0x"			);
    strRet.appIntFmt( termios.c_lflag, 16, 8		);
    strRet.appStrRaw( "\n" "    c_line: "		);
    strRet.appIntFmt( termios.c_line			);

    for ( int iCc = 0; iCc < WhCount(termios.c_cc); ++iCc )
    {
	if ( iCc % 10 == 0 )
	    strRet.appStrRaw( "\n" "    c_cc: "		);
	else if ( iCc > 0 )
	    strRet.appChrRaw( ' '			);
	strRet.appStrRaw( "0x"				);
	strRet.appIntFmt( termios.c_cc[iCc], 16, 2	);
    }

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTimeT( WhString & strRet, const MmSeg & segTimeT )
{
    segTimeT.checkCountOne( sizeof(time_t) );
    const time_t & tValue = * (const time_t *) segTimeT.address( );

    strRet.appStrRaw( "    "			);
    strRet.appIntFmt( tValue			);
    strRet.appChrRaw( '\n'			);
}



// Format a segment.
void FmtTimeval( WhString & strRet, const MmSeg & segTimeval )
{
    segTimeval.checkCountOne( sizeof(struct timeval) );
    const struct timeval & timeval =
	* (const struct timeval *) segTimeval.address( );

    strRet.appStrRaw( "    timeval: "			);
    strRet.appIntFmt( timeval.tv_sec			);
    strRet.appChrRaw( '.'				);
    strRet.appIntFmt( timeval.tv_usec, 10, 6		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTimex( WhString & strRet, const MmSeg & segTimex )
{
    segTimex.checkCountOne( sizeof(struct timex) );
    FmtAny( strRet, segTimex );
}



// Format a segment.
void FmtTimezone( WhString & strRet, const MmSeg & segTimezone )
{
    segTimezone.checkCountOne( sizeof(struct timezone) );
    const struct timezone & timezone =
	* (const struct timezone *) segTimezone.address( );

    strRet.appStrRaw( "    tz_minuteswest: "		);
    strRet.appIntFmt( timezone.tz_minuteswest		);
    strRet.appStrRaw( " tz_dsttime: "			);
    strRet.appIntFmt( timezone.tz_dsttime		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTms( WhString & strRet, const MmSeg & segTms )
{
    segTms.checkCountOne( sizeof(struct tms) );
    const struct tms & tms = * (const struct tms *) segTms.address( );

    strRet.appStrRaw( "    tms_utime: "			);
    strRet.appIntFmt( tms.tms_utime			);
    strRet.appStrRaw( " tms_stime: "			);
    strRet.appIntFmt( tms.tms_stime			);
    strRet.appStrRaw( " tms_cutime: "			);
    strRet.appIntFmt( tms.tms_cutime			);
    strRet.appStrRaw( " tms_cstime: "			);
    strRet.appIntFmt( tms.tms_cstime			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtTtyStruct( WhString & strRet, const MmSeg & segTtyStruct )
{
    segTtyStruct.checkCountOne( sizeof(struct tty_struct) );
    FmtAny( strRet, segTtyStruct );
}



// Format a segment.
void FmtUidT( WhString & strRet, const MmSeg & segUidT )
{
    segUidT.checkCountOne( sizeof(uid_t) );

    const uid_t & uidValue = * (const uid_t *) segUidT.address( );

    strRet.appStrRaw( "    "				);
    strRet.appIntFmt( uidValue				);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtUmsdosIoctl( WhString & strRet, const MmSeg & segUmsdosIoctl )
{
    segUmsdosIoctl.checkCountOne( sizeof(struct umsdos_ioctl) );
    const struct umsdos_ioctl & umsdos_ioctl =
	* (const struct umsdos_ioctl *) segUmsdosIoctl.address( );

    strRet.appStrRaw( "    version: "				);
    strRet.appIntFmt( umsdos_ioctl.version			);
    strRet.appStrRaw( "\n" " release: "				);
    strRet.appIntFmt( umsdos_ioctl.release			);
    strRet.appChrRaw( '\n'					);

    FmtAny( strRet, segUmsdosIoctl );
}



// Format a segment.
void FmtUnimapdesc( WhString & strRet, const MmSeg & segUnimapdesc )
{
    segUnimapdesc.checkCountOne( sizeof(struct unimapdesc) );
    const struct unimapdesc & unimapdesc =
	* (const struct unimapdesc *) segUnimapdesc.address( );

    strRet.appStrRaw( "    entry_ct: "			);
    strRet.appIntFmt( unimapdesc.entry_ct		);
    strRet.appStrRaw( " entries: "			);
    strRet.appPtrFmt( unimapdesc.entries		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtUnimapinit( WhString & strRet, const MmSeg & segUnimapinit )
{
    segUnimapinit.checkCountOne( sizeof(struct unimapinit) );
    const struct unimapinit & unimapinit =
	* (const struct unimapinit *) segUnimapinit.address( );

    strRet.appStrRaw( "    advised_hashsize: "		);
    strRet.appIntFmt( unimapinit.advised_hashsize	);
    strRet.appStrRaw( " advised_hashstep: "		);
    strRet.appIntFmt( unimapinit.advised_hashstep	);
    strRet.appStrRaw( " advised_hashlevel: "		);
    strRet.appIntFmt( unimapinit.advised_hashlevel	);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtUstat( WhString & strRet, const MmSeg & segUstat )
{
    segUstat.checkCountOne( sizeof(struct ustat) );
    const struct ustat & ustat = * (const struct ustat *) segUstat.address( );

    strRet.appStrRaw( "    f_tfree: "				);
    strRet.appIntFmt( ustat.f_tfree				);
    strRet.appStrRaw( " f_tinode: "				);
    strRet.appIntFmt( ustat.f_tinode				);
    strRet.appStrRaw( " f_fname: \""				);
    strRet.appStrFmt( &ustat.f_fname[0], WhCount(ustat.f_fname)	);
    strRet.appStrRaw( "\" f_fpack: \""				);
    strRet.appStrFmt( &ustat.f_fpack[0], WhCount(ustat.f_fpack)	);
    strRet.appStrRaw( "\"\n"					);
}



// Format a segment.
void FmtUtimbuf( WhString & strRet, const MmSeg & segUtimbuf )
{
    segUtimbuf.checkCountOne( sizeof(struct utimbuf) );
    const struct utimbuf & utimbuf =
	* (const struct utimbuf *) segUtimbuf.address( );

    strRet.appStrRaw( "    actime: "			);
    strRet.appIntFmt( utimbuf.actime			);
    strRet.appStrRaw( " modtime: "			);
    strRet.appIntFmt( utimbuf.modtime			);
    strRet.appChrRaw( '\n'				);

}



// Format a segment.
void FmtVm86Struct( WhString & strRet, const MmSeg & segVm86Struct )
{
    segVm86Struct.checkCountOne( sizeof(struct vm86_struct) );
    FmtAny( strRet, segVm86Struct );
}



// Format a segment.
void FmtVtConsize( WhString & strRet, const MmSeg & segVtConsize )
{
    segVtConsize.checkCountOne( sizeof(struct vt_consize) );
    const struct vt_consize & vt_consize =
	* (const struct vt_consize *) segVtConsize.address( );

    strRet.appStrRaw( "    v_rows: "			);
    strRet.appIntFmt( vt_consize.v_rows			);
    strRet.appStrRaw( " v_cols: "			);
    strRet.appIntFmt( vt_consize.v_cols			);
    strRet.appStrRaw( " v_vlin: "			);
    strRet.appIntFmt( vt_consize.v_vlin			);
    strRet.appStrRaw( " v_clin: "			);
    strRet.appIntFmt( vt_consize.v_clin			);
    strRet.appStrRaw( " v_vcol: "			);
    strRet.appIntFmt( vt_consize.v_vcol			);
    strRet.appStrRaw( " v_ccol: "			);
    strRet.appIntFmt( vt_consize.v_ccol			);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtVtMode( WhString & strRet, const MmSeg & segVtMode )
{
    segVtMode.checkCountOne( sizeof(struct vt_mode) );
    const struct vt_mode & vt_mode =
	* (const struct vt_mode *) segVtMode.address( );

    strRet.appStrRaw( "    mode: "			);
    strRet.appIntFmt( vt_mode.mode			);
    strRet.appStrRaw( " waitv: "			);
    strRet.appIntFmt( vt_mode.waitv			);
    strRet.appStrRaw( " relsig: "			);
    strRet.appIntFmt( vt_mode.relsig			);
    strRet.appStrRaw( " acqsig: "			);
    strRet.appIntFmt( vt_mode.acqsig			);
    strRet.appStrRaw( " frsig: "			);
    strRet.appIntFmt( vt_mode.frsig			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtVtSizes( WhString & strRet, const MmSeg & segVtSizes )
{
    segVtSizes.checkCountOne( sizeof(struct vt_sizes) );
    const struct vt_sizes & vt_sizes =
	* (const struct vt_sizes *) segVtSizes.address( );

    strRet.appStrRaw( "    v_rows: "			);
    strRet.appIntFmt( vt_sizes.v_rows			);
    strRet.appStrRaw( " v_cols: "			);
    strRet.appIntFmt( vt_sizes.v_cols			);
    strRet.appStrRaw( " v_scrollsize: "			);
    strRet.appIntFmt( vt_sizes.v_scrollsize		);

    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtVtStat( WhString & strRet, const MmSeg & segVtStat )
{
    segVtStat.checkCountOne( sizeof(struct vt_stat) );
    const struct vt_stat & vt_stat =
	* (const struct vt_stat *) segVtStat.address( );

    strRet.appStrRaw( "    v_active: "			);
    strRet.appIntFmt( vt_stat.v_active			);
    strRet.appStrRaw( " v_signal: "			);
    strRet.appIntFmt( vt_stat.v_signal			);
    strRet.appStrRaw( " v_state: "			);
    strRet.appIntFmt( vt_stat.v_state			);
    strRet.appChrRaw( '\n'				);
}



// Format a segment.
void FmtWinsize( WhString & strRet, const MmSeg & segWinsize )
{
    segWinsize.checkCountOne( sizeof(struct winsize) );
    const struct winsize & winsize =
	* (const struct winsize *) segWinsize.address( );

    strRet.appStrRaw( "    ws_row: "			);
    strRet.appIntFmt( winsize.ws_row			);
    strRet.appStrRaw( " ws_col: "			);
    strRet.appIntFmt( winsize.ws_col			);
    strRet.appStrRaw( " ws_xpixel: "			);
    strRet.appIntFmt( winsize.ws_xpixel			);
    strRet.appStrRaw( " ws_ypixel: "			);
    strRet.appIntFmt( winsize.ws_ypixel			);
    strRet.appChrRaw( '\n'				);
}
