// Copyright 1995 Michael Chastain
// Licensed under the Gnu Public License, Version 2
//
// File: WhFlatOut.cc
//   Flat data.
//   This is a concrete class.
//
// File Created:	25 Oct 1995		Michael Chastain
// Last Edited:		02 Nov 1995		Michael Chastain

#include <stdlib.h>
#include <string.h>

#include <ErAbort.hh>
#include <ErMem.hh>
#include <ErPtr.hh>
#include <WhFlatOut.hh>
#include <WhString.hh>



// Constructor.
WhFlatOut::WhFlatOut( int ncDataMax )
    : ncData_		( 0		)
    , ncDataMax_	( ncDataMax	)
    , pcData_		( 0		)
{
    if ( ncDataMax <= 0 )
	ErAbort( "WhFlatOut::WhFlatOut: negative or zero count." );
    pcData_ = (char *) ::malloc( ncDataMax );
    if ( pcData_ == 0 )
	ErMem( );
}



// Destructor.
WhFlatOut::~WhFlatOut( )
{
    ::free( pcData_ );
    pcData_ = 0;
}



// Expand.
void WhFlatOut::expand( int ncDataNew )
{
    // Check count.
    if ( ncDataNew < 0 )
	ErAbort( "WhFlatOut::expand: negative count." );

    // Compute new maximum and check it.
    int ncDataMaxNew = ncData_ + ncDataNew;
    if ( ncDataMaxNew < ncData_ )
	ErAbort( "WhFlatOut::expand: count overflow." );

    // Expand if needed.
    if ( ncDataMaxNew > ncDataMax_ )
    {
	// Heuristic pre-allocation.
	const int ncDataMaxFirst = 240;
	if ( ncDataMaxNew < ncDataMaxFirst ) ncDataMaxNew = ncDataMaxFirst;
	if ( ncDataMaxNew < 2 * ncDataMax_ ) ncDataMaxNew = 2 * ncDataMax_;

	// Expand the data.
	char * pcDataNew = (char *) ::realloc( pcData_, ncDataMaxNew );
	if ( pcDataNew == 0 )
	    ErMem( );

	// Swap in new values.
	pcData_    = pcDataNew;
	ncDataMax_ = ncDataMaxNew;
    }
}



// Put an address.
void WhFlatOut::putAddr( MmAddr addrPut )
{
    if ( ncDataMax_ - ncData_ < 2 * int( sizeof(MmAddr) ) )
	expand( 2 * sizeof(MmAddr) );
    while ( ncData_ % sizeof(MmAddr) != 0 )
	pcData_[ncData_++] = 0;
    * (MmAddr *) (pcData_ + ncData_) = addrPut;
    ncData_ += sizeof(MmAddr);
}



// Put a boolean.
void WhFlatOut::putBool( bool fPut )
{
    if ( ncDataMax_ - ncData_ < 2 * int( sizeof(bool) ) )
	expand( 2 * sizeof(bool) );
    while ( ncData_ % sizeof(bool) != 0 )
	pcData_[ncData_++] = 0;
    * (bool *) (pcData_ + ncData_) = fPut;
    ncData_ += sizeof(bool);
}



// Put a character.
void WhFlatOut::putChar( char cPut )
{
    if ( ncDataMax_ - ncData_ < 2 * int( sizeof(char) ) )
	expand( 2 * sizeof(char) );
    while ( ncData_ % sizeof(char) != 0 )
	pcData_[ncData_++] = 0;
    * (char *) (pcData_ + ncData_) = cPut;
    ncData_ += sizeof(char);
}



// Put an integer.
void WhFlatOut::putInt( int iPut )
{
    if ( ncDataMax_ - ncData_ < 2 * int( sizeof(int) ) )
	expand( 2 * sizeof(int) );
    while ( ncData_ % sizeof(int) != 0 )
	pcData_[ncData_++] = 0;
    * (int *) (pcData_ + ncData_) = iPut;
    ncData_ += sizeof(int);
}



// Put an array of character.
void WhFlatOut::putArrayChar( const char * pcPut, int ncPut )
{
    if ( ncPut < 0 )
	ErAbort( "WhFlatOut::putArrayChar: negative count." );
    putInt( ncPut );
    if ( ncPut > 0 )
    {
	if ( pcPut == 0 )
	    ErPtr( );
	if ( ncDataMax_ - ncData_ < (ncPut + 1) * int( sizeof(char) ) )
	    expand( (ncPut + 1) * sizeof(char) );
	while ( ncData_ % sizeof(char) != 0 )
	    pcData_[ncData_++] = 0;
	::memcpy( pcData_ + ncData_, pcPut, ncPut * sizeof(char) );
	ncData_ += ncPut * sizeof(char);
    }
}



// Put a list of characters.
void WhFlatOut::putListWord( const WhList <MmWord> & lwPut )
{
    putInt( lwPut.count( ) );
    if ( lwPut.count( ) > 0 )
    {
	if ( ncDataMax_ - ncData_ < (lwPut.count( ) + 1) * int( sizeof(MmWord) ) )
	    expand( (lwPut.count( ) + 1) * sizeof(MmWord) );
	while ( ncData_ % sizeof(MmWord) != 0 )
	    pcData_[ncData_++] = 0;
	::memcpy( pcData_ + ncData_, lwPut.address( ),
	    lwPut.count( ) * sizeof(MmWord) );
	ncData_ += lwPut.count( ) * sizeof(MmWord);
    }
}



// Put a string.
void WhFlatOut::putString( const WhString & strPut )
{
    putInt( strPut.count( ) );
    if ( strPut.count( ) > 0 )
    {
	if ( ncDataMax_ - ncData_ < (strPut.count( ) + 1) * int( sizeof(char) ) )
	    expand( (strPut.count( ) + 1) * sizeof(char) );
	while ( ncData_ % sizeof(char) != 0 )
	    pcData_[ncData_++] = 0;
	::memcpy( pcData_ + ncData_, strPut.address( ),
	    strPut.count( ) * sizeof(char) );
	ncData_ += strPut.count( ) * sizeof(char);
    }
}



// Put a word.
void WhFlatOut::putWord( MmWord wPut )
{
    if ( ncDataMax_ - ncData_ < 2 * int( sizeof(MmWord) ) )
	expand( 2 * sizeof(MmWord) );
    while ( ncData_ % sizeof(MmWord) != 0 )
	pcData_[ncData_++] = 0;
    * (MmWord *) (pcData_ + ncData_) = wPut;
    ncData_ += sizeof(MmWord);
}
