/*
 * ss5136dn.c -- Linux device driver for the S.S Technologies 5136-DN ISA bus
 * CAN-Network interface card.  Version 1.3.
 *
 * Copyright (c) 1997,1998,1999, 2000 The Laitram Corporation.
 * All rights reserved.
 *
 * Author: Mark Sutton (mes)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details. It is contained in
 * the file "COPYING" in the root directory of this distribution.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * Revision history:
 * -Initial writing (Version 0.01) May 5, 1997 (mes)
 * -Allow easy selection of 16k or 32k card memory aperature (Version 0.02)
 *  May 15, 1997 (mes)
 * -Give user level hook to do a hard reset of the card (Version 0.03)
 *  May 15, 1997 (mes)
 *
 * Version 0.04 Placed driver and user level code under the LGPL.
 * Added functions for deallocation and "tidy" of card memory to allow
 * dynamic re-assignment of card memory while running.
 * June 29, 1998. (mes)
 *
 * Version 0.05 Fixed bug that caused 5136-DN-PRO board to not work reliably.
 * Added capability to module loader to load encrypted modules.  Driver
 * now identifies whether board is a standard or a "pro" and adjusts some
 * paramaters slightly for the different boards.  User code can poll for what
 * type of board is installed.
 * July 21, 1998 (mes)
 *
 * Version 1.0 Cleaned up module loader based on new information from
 * engineers at SST.  Added the folowing programs to package:  devset (for
 * software setting macid and baud rate on devices), devqry (for quering a
 * device about it's input and output size and other paramaters), and 
 * devwho (for finding all devices on a network).
 * October 23, 1998 (mes)
 *
 * Version 1.1 Added a PCMCIA driver to the package.
 * January 7, 1999 (mes)
 *
 * Version 1.2 Modified to run correctly on 2.2.x kernels as well as 2.0.x
 * Should also compile and run correctly on 2.1.x series kernels.
 * March 5, 1999 (mes)
 * 
 * Version 1.3 Changed Major Number of device since SUSE inadvertantly usurped
 * the major number 144 for Encapsulated PPP.
 * Made a fix for determining kernel version when kernel version number has
 * a "-" in it.  Eg. RedHat 6.1 comes with 2.2.12-20.
 * March 14, 2000
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <asm/irq.h>
#include <asm/pgtable.h>
#include <linux/time.h>
#include <linux/malloc.h>
#include <asm/io.h>
#include <linux/errno.h>

#include "k_compat.h"

#include "ss5136dn.h"

static unsigned int ss5136dn_status;
char * cardmemptr;
int gCardType,gMaxReg;

static int ss5136dn_open(struct inode * inode, struct file * file)
{
  if (ss5136dn_status & SS5136DN_BUSY)
    {
      return(-EBUSY);
    }
  ss5136dn_status = ss5136dn_status | SS5136DN_BUSY; 
  if (MINOR(inode->i_rdev) != 0) return(-ENODEV);
  return(0);
}

static FS_RELEASE_T ss5136dn_close(struct inode * inode, struct file * file)
{
  ss5136dn_status = ss5136dn_status & (~SS5136DN_BUSY);
  return (FS_RELEASE_T)0;
}

/* Card is disabled ("turned off") when all three i/o registers set to 0 */
static void ss5136dn_shutdown(void)
{
  outb(0,SS5136DN_PORTBASE + BCR0);
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);
}

void cleanup_module(void)
{  
  unregister_chrdev(SS5136DN_MAJOR,"ss5136dn");
  ss5136dn_shutdown();
}

static int ss5136dn_init(void)
{

  unsigned char tstin;
  unsigned char portout;

  outb(0,SS5136DN_PORTBASE + BCR0);  /* Clear all i/o registers */
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);

/* Next, do a "card alive" test, line at 0x4 will almost immediately
reset itself if card is working correctly */
  outb(0x06,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0x02)
    {
      printk("SS5136DN card not present or failing to answer are you there poll! \n");
      printk("Aborting attempt to set up SS5136DN card!\n");
      ss5136dn_shutdown();
      return(-1);
    }
  outb(0,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0)
    {
      printk("SS5136DN board not responding properly. Aborting attempt to initialize it.\n");
      ss5136dn_shutdown();
      return(-1);
    }
/* Compute the memory base address code to load and load it */
  portout=((SS5136DN_MEMBASE >> 14) & 0x1f);
  if (PAGESIZEINBYTES == 0x4000) {
    portout=(portout | WIN16K);
    printk("ss5136dn: Setting a 16k aperature.\n");
  }
  else {
    printk("ss5136dn: Setting a 32k aperature.\n");
  }
  outb(portout,SS5136DN_PORTBASE + BCR1);
  tstin=inb(SS5136DN_PORTBASE + BCR1);
  if (tstin != portout)
    {
      printk("Unable to set SS5136DN board's base address! Aborting attempt to innitialize it!\n");
      ss5136dn_shutdown();
      return(-1);
    }
/* Setup the BCR0 register to initial condition:
   Bits are as followes:
   bit0 = Board interupt high enables board processor (on now).
   bit1 = Reserved (off now).
   bit2 = Interupt status latch (set/reset latch mirrors ISA interupt 
          generation, off now)
   bit3 = ISA interrupt enable (off now).
   bit4,5,6 = Page select (loaded with 001 or 011 depending on aperature size,
              contains the "Application Module Header".  This is how this
              will always be set unless loading application kernel mods.)
   bit7 = Memory enable (on now).
*/
  if (PAGESIZEINBYTES == 0x8000) {
    outb(0x91,SS5136DN_PORTBASE + BCR0);
  }
  else {
    outb(0xb1,SS5136DN_PORTBASE + BCR0);
  }
  tstin=inb(SS5136DN_PORTBASE + BCR7);
  if (tstin == 0x20)
    {
      gCardType = PRO;
      gMaxReg = 7;
    }
  else
    {
      gCardType = REGULAR;
      gMaxReg = 2;
    }
/* Turn the "Health indicator" LED green because card is now ready to use */
  outb(0x20,SS5136DN_PORTBASE + BCR2);
  cardmemptr = (char *)SS5136DN_MEMBASE;
  printk("ss5136dn: Driver for the S.S Technologies CAN Bus interface board, successfully loaded.\n");
  if (gCardType == PRO)
    printk("ss5136dn: This appears to be a PRO board.\n");
  else
    printk("ss5136dn: This appears to be a 5136-dn, basic board.\n");
  printk("ss5136dn: Copyright (c) 1997, 1998, 1999 The Laitram Corporation\n");
  printk("ss5136dn: Written By: Mark Sutton\n");
  return(0);
}

static int ss5136dn_reset(void)
{

  unsigned char tstin;
  unsigned char portout;

  outb(0,SS5136DN_PORTBASE + BCR0);  /* Clear all i/o registers */
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);

/* Next, do a "card alive" test, line at 0x4 will almost immediately
reset itself if card is working correctly */
  outb(0x06,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0x02)
    {
      ss5136dn_shutdown();
      return(-1);
    }
  outb(0,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0)
    {
      ss5136dn_shutdown();
      return(-1);
    }
/* Compute the memory base address code to load and load it */
  portout=((SS5136DN_MEMBASE >> 14) & 0x1f);
  if (PAGESIZEINBYTES == 0x4000) {
    portout=(portout | WIN16K);
  }
  outb(portout,SS5136DN_PORTBASE + BCR1);
  tstin=inb(SS5136DN_PORTBASE + BCR1);
  if (tstin != portout)
    {
      ss5136dn_shutdown();
      return(-1);
    }
/* Setup the BCR0 register to initial condition:
   Bits are as followes:
   bit0 = Board interupt high enables board processor (on now).
   bit1 = Reserved (off now).
   bit2 = Interupt status latch (set/reset latch mirrors ISA interupt 
          generation, off now)
   bit3 = ISA interrupt enable (off now).
   bit4,5,6 = Page select (loaded with 001 or 011 depending on aperature size,
              contains the "Application Module Header".  This is how this
              will always be set unless loading application kernel mods.)
   bit7 = Memory enable (on now).
*/
  if (PAGESIZEINBYTES == 0x8000) {
    outb(0x91,SS5136DN_PORTBASE + BCR0);
  }
  else {
    outb(0xb1,SS5136DN_PORTBASE + BCR0);
  }
    
/* Turn the "Health indicator" LED green because card is now ready to use */
  outb(0x20,SS5136DN_PORTBASE + BCR2);
  cardmemptr = (char *)SS5136DN_MEMBASE;
  return(0);
}

static FS_SIZE_T ss5136dn_read FOPS(struct inode * inode, struct file * file, char * buffer, U_FS_SIZE_T count, loff_t * dummy)
{

#if (LINUX_VERSION_CODE >= VERSION(2,1,0))
  int ctr;
#endif

  if (MINOR(F_INODE(file)->i_rdev) != 0) return(-ENODEV);
  if ((cardmemptr + count) > (char*)(SS5136DN_MEMBASE + PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
#if (LINUX_VERSION_CODE < VERSION(2,1,0))
  copy_to_user((char *)buffer,(char *)cardmemptr,count);
  cardmemptr = cardmemptr + count;
#else
  for(ctr=0;ctr<count;ctr++)
    {
      *(buffer + ctr) = readb(cardmemptr);
      cardmemptr++;
    }
#endif;
  return(count);
}

static FS_SIZE_T ss5136dn_write FOPS(struct inode * inode, struct file * file, const char * buffer, U_FS_SIZE_T count, loff_t * dummy)
{

#if (LINUX_VERSION_CODE >= VERSION(2,1,0))
  int ctr;
#endif

  if (MINOR(F_INODE(file)->i_rdev) != 0) return(-ENODEV);
  if ((cardmemptr + count) > (char*)(SS5136DN_MEMBASE + PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
#if (LINUX_VERSION_CODE < VERSION(2,1,0))
  copy_from_user((char *)cardmemptr,(char *)buffer,(int)count);
  cardmemptr = cardmemptr + count;
#else
  for (ctr=0;ctr<count;ctr++)
    {
      writeb(*(buffer+ctr),cardmemptr);
      cardmemptr++;
    }
#endif
  return(count);
}


static int ss5136dn_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
  char outchr,inchr;
  unsigned short portnum,portloc;
  int pageval;
  portiopair * argp;
  int argi;

  if (MINOR(inode->i_rdev) != 0) return(-ENODEV);
  switch(cmd){
  case SS5136DN_SETCARDMEMPTR:
    argi = (int)arg;
    if((argi < 0) || (argi > PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
    cardmemptr=(char *)(SS5136DN_MEMBASE + argi);
    break;
  case SS5136DN_BIGRESET:
    argi=ss5136dn_reset();
    if (argi)
      {
	ss5136dn_shutdown();
	return(-1);
      }
    break;
  case SS5136DN_SHUTDOWN:
    ss5136dn_shutdown();
    break;
  case SS5136DN_REPORT_APERATURE:
    return(PAGESIZEINBYTES);
    break;
  case SS5136DN_REPORT_CARDTYPE:
    return(gCardType);
    break;
  case SS5136DN_WRITEREG: /* Generic write to ss5136dn i/o register */
    argp = (portiopair *)arg;
    copy_from_user(&outchr,&argp->value,1);
    copy_from_user(&portnum,&argp->port,2);
    if(portnum > gMaxReg) return(-EINVAL);
    portloc = portnum + SS5136DN_PORTBASE;
    outb(outchr,portloc);
    break;
  case SS5136DN_READREG: /* Generic read to ss5136dn i/o register */
    argp = (portiopair *)arg;
    copy_from_user(&portnum,&argp->port,2);
    if(portnum > gMaxReg) return(-EINVAL);
    portloc = portnum + SS5136DN_PORTBASE;
    inchr = (char)inb(portloc);
    copy_to_user(&argp->value,&inchr,1);
    break;
  case SS5136DN_PAGECHANGE: /* Change the memory page on ss5136dn */
    pageval = (int)arg;
    if ((pageval < 0) || (pageval > MAXPAGES)) return(-EINVAL);
    pageval = pageval << 4;
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & SS5136DN_PAGEMASK; /* strip page bits from byte */
    outchr = outchr | (char)pageval;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    cardmemptr=(char *)SS5136DN_MEMBASE;
    break;
  case SS5136DN_MEMENABLE: /* Enable ss5136dn memory access */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_MEMENABLEMASK;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  case SS5136DN_MEMDISABLE: /* Disable ss5136dn memory access (usefull for detecting memory conflicts) */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_MEMENABLEMASK);
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  case SS5136DN_PROCENABLE: /* Enable the processor on the ss5136dn */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_PROCENABLEMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_PROCDISABLE: /* Disable the processor on the ss5136dn */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_PROCENABLEMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_WDENABLE: /* Enable the ss5136dn watchdog timer (true low) */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_WDENABLEMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_WDDISABLE: /* Disable the ss5136dn watchdog timer (true high) */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_WDENABLEMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;  
  case SS5136DN_HLTHGREEN: /* Turn the health LED green */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_HLTHMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_HLTHRED: /* Turn the health LED red */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_HLTHMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_CLINT: /* Clear the card's (output) interupt latch */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr | PCINT;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  case SS5136DN_STROBECINT: /* Strobe the ss5136dn interrupt line to make it process a loaded command */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_CINTMASK);
    outb_p(outchr,BCR0+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_CINTMASK;
    outb_p(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  default:
    return(-EINVAL);
    break;
  }
  return(0);
}
static struct file_operations ss5136dn_fops = {
        NULL,   	/* seek */
	ss5136dn_read, 	/* read */
	ss5136dn_write,	/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	ss5136dn_ioctl,	/* ioctl */
	NULL,     	/* mmap */
        ss5136dn_open,    /* open */
	NULL_FLUSH      /* flush  kernel > 2.1.117 only */
        ss5136dn_close,	/* release */
	NULL,		/* fsync */
	NULL,		/* fasync */
	NULL,		/* check_media_change */
	NULL		/* revalidate */
};


/* Designed to be used as a module */
#ifdef MODULE

int init_module(void)
{
  int ret = 0;

  if (register_chrdev(SS5136DN_MAJOR,"ss5136dn",&ss5136dn_fops)) {
    printk("ss5136dn: unable to get major %d for ss5136dn device\n",
	    SS5136DN_MAJOR);
    return -EIO;
  }
  ret=ss5136dn_init();
  if (ret)
    ss5136dn_shutdown();
  return ret;
}
#endif /*MODULE*/

