/*
 * dnuser.c -- User level function library for the ss5136dn card.
 *
 * Copyright (c) 1997, 1998 The Laitram Corporation.
 * All rights reserved.
 *
 * Author: Mark Sutton (mes)
 *
 * NOTE: The structures "DNS_SCANNER_CFG" and "DNS_DEVICE_CFG" and a number
 * of constants, are taken from example code Written by Geoff Jones and 
 * Copyright (c) 1996 S-S Technologies Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details. It is contained in
 * the file "COPYING" in the root directory of this distribution.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 *  Version 0.03 (Version number parallel's that of driver.)
 *  May 15, 1997 (mes)
 *
 * Version 0.04 Placed driver and user level code under the LGPL.
 * Added functions for deallocation and "tidy" of card memory to allow
 * dynamic re-assignment of card memory while running.
 * June 29, 1998. (mes)
 *
 * Version 0.05 Fixed bug that caused 5136-DN-PRO board to not work reliably.
 * Added capability to module loader to load encrypted modules.  Driver
 * now identifies whether board is a standard or a "pro" and adjusts some
 * paramaters slightly for the different boards.  User code can poll for what
 * type of board is installed.
 * July 21, 1998 (mes)
 *
 * Version 1.0 Cleaned up module loader based on new information from
 * engineers at SST.  Added the folowing programs to package:  devset (for
 * software setting macid and baud rate on devices), devqry (for quering a
 * device about it's input and output size and other paramaters), and 
 * devwho (for finding all devices on a network.
 * October 23, 1998 (mes)
 */

#include <stdio.h>
#include <fcntl.h>
#include <sys/termios.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

#include "../ss5136dn.h"
#include "dnuser.h"

#ifdef CANDECRYPT
unsigned char bDecrTable[256] =
{
  #include "decrypt.tbl"
};
#endif

FILE * ss1p;

int dni;
DNS_SCANNER_CFG ScannerCfg;
DNS_DEVICE_CFG DeviceCfg;
cardmemel * CardMemHead, * curmemp;

/* List of device configs, must malloc each one, use macid for index */
DNS_DEVICE_CFG * DevicesList[64];

/*--------------------------------------------------------------------------
  A waiting loop.  This loop delays aproximately "i" microseconds.  E.g. to
  delay for 1/10 of a second, pass it a value of 100000.  Caviets:
  probably highly O/S dependant.  Some UNIX flavors don't have "gettimeofday"
  others might measure in units other than microseconds.  Also, please do
  not use this for delays longer than 1 sec, use something like "sleep"
  instead.  Also note, due to context switching, this is really sloppy,
  it is not a precise timer!
 *-------------------------------------------------------------------------*/
void fnWait(long i)
{
  struct timeval timeback;
  struct timezone zoneback;
  int dummy;
  long curtime,initime,elapsedtime,cutouttime,sectime;

  dummy = gettimeofday(&timeback,&zoneback);
  initime = timeback.tv_usec;
  sectime = timeback.tv_sec;
  if ((initime+i) < 1000000)
    {
      do
	{
	  dummy = gettimeofday(&timeback,&zoneback);
	  curtime = timeback.tv_usec;
	  elapsedtime = curtime-initime;
	} while((elapsedtime < i) && (timeback.tv_sec == sectime));
    }
  else
    {
      cutouttime = initime+i-1000000;
      do
	{
	  dummy = gettimeofday(&timeback,&zoneback);
	  curtime = timeback.tv_usec;
	} while(((timeback.tv_sec == sectime) || (curtime < cutouttime)) && timeback.tv_sec < (sectime + 2));
    }
}

/*----------------------------------------------------------------------
 function ldappmod32p: called to load an application module if 
 we are in 32k aperature mode.
--------------------------------------------------------------------*/
int ldappmod32p(char * modulename)
{
  portiopair regpair;
  int dummy,i,j,decrypton;
  char inch,appmodcksum,instring[200], outstring[200];
  char giantin[PAGESIZEINBYTES+1], giantout[PAGESIZEINBYTES+1];
  short * pwordfollow, *pconvshort;
  time_t initialtime,passedtime,accumedtime;
  int wherefile,cardtype;
  unsigned char tmpchar;
  unsigned short shortback;

  cardtype = ioctl(dni,SS5136DN_REPORT_CARDTYPE);
  decrypton = 0;
#ifdef CANDECRYPT
  if ((strstr(modulename,".ss1")) != NULL)  /* Encrypted file for
                                               a regular board  */
    {
      if (cardtype == PRO)
	{
	  printf("You are trying to load a file with the extension '.ss1' into a PRO board.\n");
	  printf("PRO boards use module files with the extension '.ss2'.\n");
	  printf("If you are POSITIVE this is the right module for your board,\n");
	  printf("Rename it with an extension of '.ss2' and try again, otherwise, please try to\n");
	  printf("find the right module.\n");
	  printf("If you get this message and you are positive that you have a non-PRO board,\n");
	  printf("Send a gripe to mark.sutton@laitram.com.\n");
	  return(-1);
	}
      else
	{
	  decrypton=1;
	}
    }
  if ((strstr(modulename,".ss2")) != NULL)  /* Encrypted file for
                                               a PRO board  */
    {
      if (cardtype != PRO)
	{
	  printf("You are trying to load a file with the extension '.ss2' into a non-PRO board.\n");
	  printf("Non-PRO boards use module files with the extension '.ss1'.\n");
	  printf("If you are POSITIVE this is the right module for your board,\n");
	  printf("Rename it with an extension of '.ss1' and try again, otherwise, please try to\n");
	  printf("find the right module.\n");
	  printf("If you get this message and you are positive that you have a PRO board,\n");
	  printf("Send a gripe to mark.sutton@laitram.com.\n");
	  return(-1);
	}
      else
	{
	  decrypton=1;
	}
    }
#endif
  dummy=ioctl(dni,SS5136DN_BIGRESET); /* Hard reset the card. */
  dummy=ioctl(dni,SS5136DN_HLTHRED); /* Set the Health LED red */
  dummy=(dni,SS5136DN_PROCDISABLE); /* Disable on board processor while fooling with board's memory */
  pwordfollow = (short *)giantout;
  for (i = 0;i < PAGESIZEINSHORTS; i++)
    {
      *(pwordfollow + (2*i)) = 0xaa55; /* Load output string with test pattern */
    }
  dummy=ioctl(dni,SS5136DN_MEMDISABLE); /* Disable memory for conflict test */
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  dummy=write(dni,giantout,PAGESIZEINBYTES);
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }

/* Readback SHOULD NOT succeed, if it does, it means that there is another 
   device with memory at card's chosen address */

  dummy=read(dni,giantin,PAGESIZEINBYTES);
  for (i=0;i< PAGESIZEINBYTES;i++)
    {
      if (giantin[i] == giantout[i])
	{
	  printf("Possible memory confilct found with 5136dn card, aborting load.\n");
	  dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	  return(-1);
	}
    }
  dummy=ioctl(dni,SS5136DN_MEMENABLE); /* Enable the memory */
  for (j=0;j<4;j++)
    {
      if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,j) < 0) {
	printf("5136dn card page change failed! \n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      } 
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (write(dni,giantout,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error writing test string, aborting load.\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }

/* Readback SHOULD succeed this time.  Memory is enabled now */

      if (read(dni,giantin,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error with memory test readback, aborting load.\n");
      }
      for (i=0;i< PAGESIZEINBYTES;i++)
	{
	  if (giantin[i] != giantout[i])
	    {
	      printf("Failure in memory test, aborting load.\n");
	      dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	      return(-1);
	    }
	}
    }
  printf("5136dn card memory test successfull.\n");

/* Now zero out all pages of memory */
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i]=0;
  for (j=0;j<4;j++)
    {
      if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,j) < 0) {
	printf("5136dn card page change failed! \n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      } 
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (write(dni,giantout,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error writing zero string, aborting load.\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
    }
  if ((ss1p = fopen(modulename, "rb")) == NULL) {
    printf("Application module file not found!\n");
    return(-1);
  }
  appmodcksum = 0;
/* Set card to page "2" (32k mode)  */
  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,2) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
/*  Read in first 32k of file  */
  for (i=0;i<PAGESIZEINBYTES;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, first page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES);
/* Change card to page "3" (32k mode) */

  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,3) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i] = 0;
/* read in second 32k of file  */
  for (i=0;i<PAGESIZEINBYTES-1;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, second page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
  
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES-1);

/* Two 32k memory blocks should have exactly equalled file size, check this */
  if (fgetc(ss1p) != EOF) {
    printf("Application module file appears to be larger than 64k.\n");
    printf("This probably means it will not work!\n");
  } 
  if (appmodcksum != 0) {
    printf("Checksum is not equal to 0, this is almost certianly bad. %i\n",(int)appmodcksum);
  }
  fclose(ss1p);
  dummy=ioctl(dni,SS5136DN_PAGECHANGE,APPINTPAGE); /* Leave on page 1 always when application module is running */
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+PAGE_SIZE_LOCATION));
  outstring[0]=0x1;
  dummy=write(dni,outstring,1);
  dummy=ioctl(dni,SS5136DN_STROBECINT);
  dummy=ioctl(dni,SS5136DN_CLINT);  
  dummy=ioctl(dni,SS5136DN_WDDISABLE); 
  dummy=ioctl(dni,SS5136DN_PROCENABLE);  /* Bang on the process enable bit to */
  dummy=ioctl(dni,SS5136DN_PROCDISABLE);  /* Start the application running   */
  dummy=ioctl(dni,SS5136DN_PROCENABLE);
/* Note: The "double bang" above can be replaced with just a single call to
"SS5136DN_PROCENABLE" if you are using a 5136-DNP (pro) board or
the newest revisions of the 5136-DN boards.  The older revisions need this
"double bang".  If you are unsure, leave this the way it is.     */
  initialtime = time(&passedtime);
  do
    {
      regpair.port=BCR0;
      ioctl(dni,SS5136DN_READREG,&regpair);
      shortback = regpair.value & PCINT;
      accumedtime = time(&passedtime) - initialtime;
    } while((shortback == 0) && (accumedtime <6));
  /* Potential context switch race, check one more time after timeout.  */
  if (shortback == 0) {
      regpair.port=BCR0;
      ioctl(dni,SS5136DN_READREG,&regpair);
      shortback = regpair.value & PCINT;
  }
  if (shortback == 0) {
    printf("Interupt did not set. \n");
    return(-1);
  }
  initialtime = time(&passedtime);
  do
    {
      dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODULE_TYPE));
      read(dni,instring,2);
      pconvshort=(short *)instring;
      shortback = *pconvshort;
      accumedtime = time(&passedtime) - initialtime;
    } while ((shortback != ER) && (shortback != DN) && (accumedtime < 3));

/* Note: We check the value one more time after timeout because there
is a potential race here.  It's possible to only read once, then have the
task context switched until we allready timed out, meanwhile, the card may
have responded properly.  */

  if ((shortback != ER) && (shortback != DN))
    {
      dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODULE_TYPE));
      dummy=read(dni,instring,2);
      pconvshort=(short *)instring;
      shortback = *pconvshort;
    }
  if (shortback == ER) {
    dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
    read(dni,instring,64);
    printf("Card reported an error, error string was: \n %s \n",instring);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (shortback != DN) {
    dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
    read(dni,instring,64);
    printf("Unknown card error after module load, error string was \n %s \n",instring);
    regpair.port=BCR0;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 0: %i\n",(int)regpair.value);
    regpair.port=BCR1;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 1: %i\n",(int)regpair.value);
    regpair.port=BCR2;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 2: %i\n",(int)regpair.value);
    exit(0);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  dummy=ioctl(dni,SS5136DN_CLINT);
  dummy=ioctl(dni,SS5136DN_WDENABLE);
  dummy=ioctl(dni,SS5136DN_HLTHGREEN);

  printf("Application module successfully loaded.\n");
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+PAGE_SIZE_LOCATION));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  if (shortback) {
    printf("Card reports 32k memory aperature in use.\n");
  } else {
    printf("Card reports 16k memory aperature in use.\n");
  }
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+KERNELIDLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Kernel ID reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+KERNELREVLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Kernel revision level reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODIDLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Module ID reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODREVLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Module revision level reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+CARDIDLOC));
  read(dni,instring,16);
  printf("Card identifies itself as: %s\n",instring);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+CARDSERIALLOC));
  read(dni,instring,8);
  printf("Card serial number reported as: %s\n",instring);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
  read(dni,instring,64);
  printf("Running module identification: \n%s\n",instring);
  return(0);
}

/*----------------------------------------------------------------------
function ldappmod16p: called to load an application module if
we are in 16k aperature mode.
--------------------------------------------------------------------*/
int ldappmod16p(char * modulename)
{
  portiopair regpair;
  int dummy,i,j,decrypton;
  char inch,appmodcksum,instring[200], outstring[200];
  char giantin[PAGESIZEINBYTES+1], giantout[PAGESIZEINBYTES+1];
  short * pwordfollow, *pconvshort;
  time_t initialtime,passedtime,accumedtime;
  int wherefile,cardtype;
  unsigned char tmpchar;
  unsigned short shortback;

  cardtype = ioctl(dni,SS5136DN_REPORT_CARDTYPE);
  decrypton = 0;
#ifdef CANDECRYPT
  if ((strstr(modulename,".ss1")) != NULL)  /* Encrypted file for
                                               a regular board  */
    {
      if (cardtype == PRO)
	{
	  printf("You are trying to load a file with the extension '.ss1' into a PRO board.\n");
	  printf("PRO boards use module files with the extension '.ss2'.\n");
	  printf("If you are POSITIVE this is the right module for your board,\n");
	  printf("Rename it with an extension of '.ss2' and try again, otherwise, please try to\n");
	  printf("find the right module.\n");
	  printf("If you get this message and you are positive that you have a non-PRO board,\n");
	  printf("Send a gripe to mark.sutton@laitram.com.\n");
	  return(-1);
	}
      else
	{
	  decrypton=1;
	}
    }
  if ((strstr(modulename,".ss2")) != NULL)  /* Encrypted file for
                                               a PRO board  */
    {
      if (cardtype != PRO)
	{
	  printf("You are trying to load a file with the extension '.ss2' into a non-PRO board.\n");
	  printf("Non-PRO boards use module files with the extension '.ss1'.\n");
	  printf("If you are POSITIVE this is the right module for your board,\n");
	  printf("Rename it with an extension of '.ss1' and try again, otherwise, please try to\n");
	  printf("find the right module.\n");
	  printf("If you get this message and you are positive that you have a PRO board,\n");
	  printf("Send a gripe to mark.sutton@laitram.com.\n");
	  return(-1);
	}
      else
	{
	  decrypton=1;
	}
    }
#endif
  dummy=ioctl(dni,SS5136DN_BIGRESET); /* Hard reset the card. */
  dummy=ioctl(dni,SS5136DN_HLTHRED); /* Set the Health LED red */
  dummy=(dni,SS5136DN_PROCDISABLE); /* Disable on board processor while fooling with board's memory */
  pwordfollow = (short *)giantout;
  for (i = 0;i < PAGESIZEINSHORTS; i++)
    {
      *(pwordfollow + (2*i)) = 0xaa55; /* Load output string with test pattern */
    }
  dummy=ioctl(dni,SS5136DN_MEMDISABLE); /* Disable memory for conflict test */
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  dummy=write(dni,giantout,PAGESIZEINBYTES);
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }

/* Readback SHOULD NOT succeed, if it does, it means that there is another 
   device with memory at card's chosen address */

  dummy=read(dni,giantin,PAGESIZEINBYTES);
  for (i=0;i< PAGESIZEINBYTES;i++)
    {
      if (giantin[i] == giantout[i])
	{
	  printf("Possible memory confilct found with 5136dn card, aborting load.\n");
	  dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	  return(-1);
	}
    }
  dummy=ioctl(dni,SS5136DN_MEMENABLE); /* Enable the memory */
  for (j=0;j<8;j++)
    {
      if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,j) < 0) {
	printf("5136dn card page change failed! \n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      } 
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (write(dni,giantout,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error writing test string, aborting load.\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }

/* Readback SHOULD succeed this time.  Memory is enabled now   */

      if (read(dni,giantin,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error with memory test readback, aborting load.\n");
      }
      for (i=0;i< PAGESIZEINBYTES;i++)
	{
	  if (giantin[i] != giantout[i])
	    {
	      printf("Failure in memory test, aborting load.\n");
	      dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	      return(-1);
	    }
	}
    }
  printf("5136dn card memory test successfull.\n");

/* Now zero out all pages of memory */
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i]=0;
  for (j=0;j<8;j++)
    {
      if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,j) < 0) {
	printf("5136dn card page change failed! \n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      } 
      if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
	printf("SETCARDMEMPTR Failed!\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
      if (write(dni,giantout,PAGESIZEINBYTES) != PAGESIZEINBYTES){
	printf("Error writing zero string, aborting load.\n");
	dummy=ioctl(dni,SS5136DN_SHUTDOWN);
	return(-1);
      }
    }
  if ((ss1p = fopen(modulename, "rb")) == NULL) {
    printf("Application module file not found!\n");
    return(-1);
  }
  appmodcksum = 0;
/* Set card to page "4" (16k mode)  */
  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,4) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
/*  Read in first 16k of file  */
  for (i=0;i<PAGESIZEINBYTES;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, first page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES);
/* Set card to page "5" (16k mode)  */
  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,5) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i] = 0;
/*  Read in second 16k of file  */
  for (i=0;i<PAGESIZEINBYTES;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, second page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES);
/* Set card to page "6" (16k mode)  */
  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,6) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i] = 0;
/*  Read in third 16k of file  */
  for (i=0;i<PAGESIZEINBYTES;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, third page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES);
/* Change card to page "7" (16k mode) */

  if (dummy = ioctl(dni,SS5136DN_PAGECHANGE,7) < 0) {
    printf("5136dn card page change failed! \n",dummy);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (ioctl(dni,SS5136DN_SETCARDMEMPTR,0) < 0) {
    printf("SETCARDMEMPTR Failed!\n");
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  for (i=0;i<PAGESIZEINBYTES;i++) giantout[i] = 0;
/* read in fourth 16k of file  */
  for (i=0;i<PAGESIZEINBYTES-1;i++)
    {
      dummy = fgetc(ss1p);
      if (dummy == EOF)
	{
	  printf("Premature EOF reached on Application module, fourth page, aborting. %i\n",i);
	  return(-1);
	}
#ifdef CANDECRYPT
      if (decrypton)
	{
	  inch = bDecrTable[dummy];
	}
      else
	{
	  inch = (char)dummy;
	}
#else
      inch = (char)dummy;
#endif
      appmodcksum=appmodcksum+inch;
      giantout[i] = inch;
    }
  
/* Sock it into the card.  */
  dummy=write(dni,giantout,PAGESIZEINBYTES-1);

/* Four 16 memory blocks should have exactly equalled file size, check this */
  if (fgetc(ss1p) != EOF) {
    printf("Application module file appears to be larger than 64k.\n");
    printf("This probably means it will not work!\n");
  } 
  if (appmodcksum != 0) {
    printf("Checksum is not equal to 0, this is almost certianly bad. %i\n",(int)appmodcksum);
  }
  fclose(ss1p);
  dummy=ioctl(dni,SS5136DN_PAGECHANGE,APPINTPAGE); /* Leave on page 3 always when application module is running */
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+PAGE_SIZE_LOCATION));
  outstring[0]=0;
  dummy=write(dni,outstring,1);
  dummy=ioctl(dni,SS5136DN_STROBECINT);
  dummy=ioctl(dni,SS5136DN_CLINT);
  dummy=ioctl(dni,SS5136DN_WDDISABLE); 
  dummy=ioctl(dni,SS5136DN_PROCENABLE);  /* Bang on the process enable bit to */
  dummy=ioctl(dni,SS5136DN_PROCDISABLE);  /* Start the application running  */
  dummy=ioctl(dni,SS5136DN_PROCENABLE);
/* Note: The "double bang" above can be replaced with just a single call to
"SS5136DN_PROCENABLE" if you are using a 5136-DNP (pro) board or
the newest revisions of the 5136-DN boards.  The older revisions need this
"double bang".  If you are unsure, leave this the way it is.     */
  initialtime = time(&passedtime);
  do
    {
      regpair.port=BCR0;
      ioctl(dni,SS5136DN_READREG,&regpair);
      shortback = regpair.value & PCINT;
      accumedtime = time(&passedtime) - initialtime;
    } while((shortback == 0) && (accumedtime <3));
  /* Potential context switch race, check one more time after timeout.  */
  if (shortback == 0) {
      regpair.port=BCR0;
      ioctl(dni,SS5136DN_READREG,&regpair);
      shortback = regpair.value & PCINT;
  }
  if (shortback == 0) {
    printf("Interupt did not set. \n");
    return(-1);
  }
  initialtime = time(&passedtime);
  do
    {
      dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODULE_TYPE));
      read(dni,instring,2);
      pconvshort=(short *)instring;
      shortback = *pconvshort;
      accumedtime = time(&passedtime) - initialtime;
    } while ((shortback != ER) && (shortback != DN) && (accumedtime < 3));

/* Note: We check the value one more time after timeout because there
is a potential race here.  It's possible to only read a few times, then have 
the task context switched until we allready timed out, meanwhile, the card may
have responded properly.  */

  if ((shortback != ER) && (shortback != DN))
    {
      dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODULE_TYPE));
      dummy=read(dni,instring,2);
      pconvshort=(short *)instring;
      shortback = *pconvshort;
    }
  if (shortback == ER) {
    dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
    read(dni,instring,64);
    printf("Card reported an error, error string was: \n %s \n",instring);
    regpair.port=BCR0;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 0: %i\n",(int)regpair.value);
    regpair.port=BCR1;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 1: %i\n",(int)regpair.value);
    regpair.port=BCR2;
    ioctl(dni,SS5136DN_READREG,&regpair);
    printf("Value of port 2: %i\n",(int)regpair.value);
    exit(0);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  if (shortback != DN) {
    dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
    read(dni,instring,64);
    printf("Unknown card error after module load, error string was \n %s \n",instring);
    dummy=ioctl(dni,SS5136DN_SHUTDOWN);
    return(-1);
  }
  dummy=ioctl(dni,SS5136DN_CLINT);
  dummy=ioctl(dni,SS5136DN_WDENABLE);
  dummy=ioctl(dni,SS5136DN_HLTHGREEN);

  printf("Application module successfully loaded.\n");
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+PAGE_SIZE_LOCATION));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  if (shortback) {
    printf("Card reports 32k memory aperature in use.\n");
  } else {
    printf("Card reports 16k memory aperature in use.\n");
  }
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+KERNELIDLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Kernel ID reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+KERNELREVLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Kernel revision level reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODIDLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Module ID reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MODREVLOC));
  read(dni,instring,2);
  pconvshort=(short *)instring;
  shortback = *pconvshort;
  printf("Module revision level reported as %x\n",(int)shortback);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+CARDIDLOC));
  read(dni,instring,16);
  printf("Card identifies itself as: %s\n",instring);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+CARDSERIALLOC));
  read(dni,instring,8);
  printf("Card serial number reported as: %s\n",instring);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,(DATA_BASE+MESSAGE));
  read(dni,instring,64);
  printf("Running module identification: \n%s\n",instring);
  return(0);
}
/*--------------------------------------------------------------------*/
void dumpcardmem(char * outputfilename)
{
  unsigned char cardmemdump[0x3fff];
  FILE * outfile;
  int foo,i;

  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE);
  foo=read(dni,cardmemdump,0x3fff);
  outfile=fopen(outputfilename,"w");
  for (i=0;i<0x3fff;i++)
    {
      if ((i % 32) == 0)
	{
	  fprintf(outfile,"\n%x ",i);
	}
      fprintf(outfile,"%x ",(int)cardmemdump[i]);
    }
  fclose(outfile);
}

/*------------------------------------------------------------------------
   sendDeviceExplicit().  Sends an explicit message to a device and 
   returns the response.
   ------------------------------------------------------------------------*/
int sendDeviceExplicit(int dbg, int macid, explicit_request *req, explicit_response *retresp)
{
  int foo;
  unsigned char inflags,outflags;
  time_t initialtime,passedtime,accumedtime;
  unsigned int timeout = 3;

  if (dbg ==1) dumpcardmem("before.txt");
  initialtime = time(&passedtime);
  /* Check explicit message buffer interlock flag.  */
  do {
    foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)));
    foo=read(dni,&inflags,1);
    accumedtime = time(&passedtime) - initialtime;
  } while (((inflags & DCTL_EXPITRLCK) == 0) && (accumedtime < timeout));
  /* Once more to avoid racing  */
  if (inflags & DCTL_EXPITRLCK == 0)
    {
      foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)));
      foo=read(dni,&inflags,1);
    }
  if (inflags & DCTL_EXPITRLCK == 0) return(-2);
  /* Put the outgoing explicit message in the buffer  */
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+((*DevicesList[macid]).ExplicitOffset));
  foo=write(dni,req,sizeof(explicit_request));
  if (dbg ==1) dumpcardmem("written.txt");
  /* Clear the explicit message interlock flag */
  outflags = inflags & (~DCTL_EXPITRLCK);
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)));
  foo=write(dni,&outflags,1);
  /* Set the explicit message event flag */
  outflags = 1;
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)+EVFLAGOFF));
  foo=write(dni,&outflags,1);
  initialtime = time(&passedtime);
  /* Check that the explicit message event flag is not allready set */
  do {
    foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+EVENTTOP+macid);
    foo=read(dni,&inflags,1);
    accumedtime = time(&passedtime) - initialtime;
  } while ((inflags != 0) && (accumedtime < timeout));
  /* Once more to avoid racing  */
  if (inflags != 0)
    {
      foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+EVENTTOP+macid);
      foo=read(dni,&inflags,1);
    }
  if (inflags != 0) return(-3);
  /* Set the explicit message event flag */
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+EVENTTOP+macid);  
  outflags = 1;
  foo=write(dni,&outflags,1);
  /* Wait for scanner to set the explicit message event flag  */
  initialtime = time(&passedtime);
  do {
    foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+STATUSTOP+(macid*16)+4);  
    foo=read(dni,&inflags,1);
    accumedtime = time(&passedtime) - initialtime;
  } while ((inflags == 0) && (accumedtime < timeout));
  /* Once more to avoid racing  */
  if (inflags == 0)
    {
      foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+STATUSTOP+(macid*16)+4);
      foo=read(dni,&inflags,1);
    }
  if (inflags == 0) return(-4);
  if (inflags == 3)
    {
      printf("Warning, explicit message response received from device %i\n",macid);
      printf("but response was too long and overflowed the buffer.\n");
    }
  /* We clear the flag  */
  outflags=0;
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+STATUSTOP+(macid*16)+4);
  foo=write(dni,&outflags,1);
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)));
  /* Set the interlock flag */
  foo=read(dni,&inflags,1);
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)));
  outflags=inflags | DCTL_EXPITRLCK;
  foo=write(dni,&outflags,1);
  /* Read the return message */
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+((*DevicesList[macid]).ExplicitOffset));
  foo=read(dni,retresp,sizeof(explicit_response));
  if (dbg ==1) dumpcardmem("returned.txt");
  return(0);
}
  
/*------------------------------------------------------------------------
   initCardMemList().  This function initilises a linked list used to keep 
   track of the available free memory on the card.  Free memory is located
   between 0x1000 and 0x3FFF in the application module header.  This memory
   is used to map devicenet I/O, hold explicit message buffers, etc.  It's
   organisation is defined by the user application program.  
-------------------------------------------------------------------------*/
void initCardMemList(void)
{
  (void *)CardMemHead = malloc(sizeof(cardmemel));
  (*CardMemHead).blockadd = 0x1000;
  (*CardMemHead).blocksize = 0x3000;
  (*CardMemHead).freeyn = 1;
  (*CardMemHead).prev = NULL;
  (*CardMemHead).next = NULL;
}

/*-------------------------------------------------------------------------
CleanupCardMemList().  Deallocates the linked list used to keep track of
card free memory allocation.
-------------------------------------------------------------------------*/
void CleanupCardMemList(void)
{
  cardmemel * tmpmemp;
  
  curmemp = CardMemHead;
  while ((*curmemp).next != NULL) curmemp = (*curmemp).next;
  do
    {
      tmpmemp = (*curmemp).prev;
      free((cardmemel *)curmemp);
      curmemp = tmpmemp;
    } while (curmemp != NULL);
  CardMemHead = NULL;
}

/*------------------------------------------------------------------------
allocCardMem(). Allocates a chunk of card free memory of "size".  Returns a 
"pointer" (actually an unsigned short reference) to the memory address on the
card allocated.  Maintains a linked list "map" of the allocated memory on
the card.
---------------------------------------------------------------------------*/
unsigned short allocCardMem(unsigned short size)
{
  cardmemel * tmpmemp;

  curmemp = CardMemHead;
  while (((!(*curmemp).freeyn) || ((*curmemp).blocksize < size)) && (*curmemp).next != NULL) curmemp = (*curmemp).next;
  if (((*curmemp).next == NULL) && ((*curmemp).blocksize < size) || (!(*curmemp).freeyn))
    {
      /* No memory blocks big enough are left on card ! */
      printf("Card is out of memory!\n");
      return(-1);
    }

  else
    {
      /* We have found a memory block of sufficient size */
      if ((*curmemp).blocksize > size)
	{
	  (void *)tmpmemp = malloc(sizeof(cardmemel));
	  (*tmpmemp).blocksize = (*curmemp).blocksize - size;
	  (*tmpmemp).next = (*curmemp).next;
	  (*curmemp).next = tmpmemp;
	  (*tmpmemp).prev = curmemp;
	  (*tmpmemp).blockadd = ((*curmemp).blockadd)+size;
	  (*curmemp).blocksize = size;
	  (*tmpmemp).freeyn = 1;
	}
      (*curmemp).freeyn = 0;
      /*      printf("allocCardMem returning: %x\n",(int)(*curmemp).blockadd);  */
      return((*curmemp).blockadd);
    }
}

/*----------------------------------------------------------------------
deallocCardMem() marks a card memory block as free.
--------------------------------------------------------------------------*/
int deallocCardMem(unsigned short memloc)
{
  int notdone = 1;

  curmemp = CardMemHead;
  while ((curmemp != NULL) && (notdone))
    {
      if ((*curmemp).blockadd == memloc)
	{
	  (*curmemp).freeyn = 1;
	  notdone = 0;
	}
      curmemp = (*curmemp).next;
    }
  return(0);
}

/*----------------------------------------------------------------------
TidyCardMemList: Combines consecutive free blocks of card memory in the memory
map into larger blocks. 
--------------------------------------------------------------------------*/
int TidyCardMemList(void)
{
  cardmemel * firstone, * lastone, * tempone;
  int howmany = 0;
  int totsize = 0;

  firstone = CardMemHead;
  while ((!(*firstone).freeyn) && ((*firstone).next != NULL))
    firstone = (*firstone).next;
  lastone = firstone;
  while ((*lastone).freeyn && lastone != NULL)
    {
      totsize = totsize + (*lastone).blocksize;
      lastone = (*lastone).next;
      howmany++;
    }
  if (howmany > 1)
    {
      (*firstone).blocksize = totsize;
      (*firstone).next = lastone;
      if (lastone != NULL)
	{
	  tempone = lastone;
	  lastone = (*lastone).prev;
	  (*tempone).prev = firstone;
	  while (lastone != firstone)
	    {
	      tempone = lastone;
	      lastone = (*lastone).prev;
	      free((cardmemel *)tempone);
	    }
	}
      else
	{
	  lastone = firstone;
	  while ((*lastone).next != NULL) 
	    lastone = (*lastone).next;
	  while (lastone != firstone)
	    {
	      tempone = lastone;
	      lastone = (*lastone).prev;
	      free((cardmemel *)tempone);
	    }
	}
    }
  return(howmany);
}

/*-----------------------------------------------------------------------
function QueryInterrupt:  Polls for a specific card interupt by looking
at the interrupt status words in the Application Module Header
-----------------------------------------------------------------------*/

int QueryInterrupt(int flags)
{
  unsigned char IrqStatusA = 0, IrqStatusB = 0;
  char instring[2],outstring[2];
  int dummy;

  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+IRQSTATUSALOC);
  read(dni,instring,1);
  IrqStatusA |= instring[0];
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+IRQSTATUSALOC);
  outstring[0]=instring[0] & (unsigned char)~flags;
  write(dni,outstring,1);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+IRQSTATUSBLOC);
  read(dni,instring,1);
  IrqStatusB |= instring[0];
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+IRQSTATUSBLOC);
  outstring[0]=instring[0] & (unsigned char)~flags;

  if ((IrqStatusA|IrqStatusB) & (unsigned char)flags)
    {
      /* Specified interupt was present */
      return(1);
    }
  return(0);  /* Specified interupt was not present.  */
}

/*------------------------------------------------------------------------
CardCommand().  Issues a command to the card and strobes the card interupt
to execute the command, confirms proper card response.
-------------------------------------------------------------------------*/
int CardCommand(unsigned short incommand, unsigned int timeout)
{
  unsigned short command;
  unsigned char * cmdptr;
  int dummy;
  time_t initialtime,passedtime,accumedtime;

  command = incommand;
  dummy=QueryInterrupt(CM);
  cmdptr = (char *)&command;
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDREG);
  write(dni,cmdptr,2);
  dummy=ioctl(dni,SS5136DN_STROBECINT);
  initialtime = time(&passedtime);
  do
    {
      accumedtime = time(&passedtime) - initialtime;
    } while ((!QueryInterrupt(CM)) && (accumedtime < timeout));
  if (accumedtime >= timeout)
    {
      printf("Card did timeout at this point.\n");
      /* One more time to avoid context-switch racing */
      if (!QueryInterrupt(CM)) return(0xfffe);
    }
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDREG);
  read(dni,cmdptr,2);
  if ((*cmdptr) & COMMANDERR )
    {
      dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDREG+2);
      read(dni,cmdptr,2);
      if ((*cmdptr) == 0)
	{
	  /* Undefined error */
	  return(0xffff);
	}
      return(*cmdptr);
    }
  return(0);
}
/* GoOnline().  Puts the card "online" scanning the devicenet.  */
int GoOnline(unsigned short scannerid, unsigned short scannerbaud)
{
  unsigned char * config;
  int dummy;

  ScannerCfg.MacId = scannerid;
  ScannerCfg.BaudRate = scannerbaud;
  ScannerCfg.StrobeInterval = 50;
  ScannerCfg.Reserved = 0;
  /* enable poll and strobe connections  */
  ScannerCfg.Flags = DNS_EXP | DNS_P | DNS_ST;
  ScannerCfg.ExplicitRequestSize = 16;
  ScannerCfg.ExplicitRequestOffset = allocCardMem(16);
  ScannerCfg.ExplicitResponseSize = 16;
  ScannerCfg.ExplicitResponseOffset = allocCardMem(16);
  ScannerCfg.Io1Interval = 0;
  /* poll request size  */
  ScannerCfg.Output1Size = 2;
  /* poll request Offset  */
  ScannerCfg.Output1Offset = allocCardMem(2);
  ScannerCfg.Output1PathOffset = 0;
  /* poll response size */
  ScannerCfg.Input1Size = 2;
  /* poll response offset  */
  ScannerCfg.Input1Offset = allocCardMem(2);
  ScannerCfg.Input1PathOffset = 0;
  ScannerCfg.Io2Interval = 0;
  /* strobe request size  */
  ScannerCfg.Output2Size = 1;
  /* strobe request offset  */
  ScannerCfg.Output2Offset = allocCardMem(1);
  ScannerCfg.Output2PathOffset = 0;
  /* strobe response size  */
  ScannerCfg.Input2Size = 2;
  /* strobe response Offset  */
  ScannerCfg.Input2Offset = allocCardMem(2);
  ScannerCfg.Input2PathOffset = 0;
  
  config = (unsigned char*)&ScannerCfg;
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  /* write data to command data buffer  */
  dummy=write(dni,config,sizeof(DNS_SCANNER_CFG));
  fnWait(50000);
  return(CardCommand( ONLINE, INIT_TIMEOUT ));
}

int GoOffline(void)
  {
    return( CardCommand(OFFLINE, CMD_TIMEOUT) );
  }

int StartScan(void)
  {
    return( CardCommand(START_SCAN, CMD_TIMEOUT) );
  }

int StopScan(void)
  {
    return( CardCommand(STOP_SCAN, CMD_TIMEOUT) );
  }

int IOActive(void)
{
  unsigned short timeoutparam;
  int dummy;

  timeoutparam = 0;
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=write(dni,&timeoutparam,2);
  fnWait(50000);
  return(CardCommand( IO_ACTIVE, CMD_TIMEOUT));
}

int IOIdle(void)
{
  return(CardCommand( IO_IDLE, CMD_TIMEOUT));
}

int GetDevice( DNS_DEVICE_CFG * DeviceDescrip )
{
  int dummy;

  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=write(dni,&(*DeviceDescrip).MacId,2);
  if (CardCommand( GET_DEVICE, CMD_TIMEOUT ) < 0) return(-1);
  fnWait(50000);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=read(dni,DeviceDescrip,sizeof(DNS_DEVICE_CFG));
  return(0);
}

int DeleteDevice(short * ddevice)
{

  int dummy;
  
  if ((*DevicesList[*ddevice]).ExplicitOffset) deallocCardMem((*DevicesList[*ddevice]).ExplicitOffset);
  if ((*DevicesList[*ddevice]).Output1Offset) deallocCardMem((*DevicesList[*ddevice]).Output1Offset);
  if ((*DevicesList[*ddevice]).Output1LocalPathOffset) deallocCardMem((*DevicesList[*ddevice]).Output1LocalPathOffset);
  if ((*DevicesList[*ddevice]).Output1RemotePathOffset) deallocCardMem((*DevicesList[*ddevice]).Output1RemotePathOffset);
  if ((*DevicesList[*ddevice]).Input1Offset) deallocCardMem((*DevicesList[*ddevice]).Input1Offset);
  if ((*DevicesList[*ddevice]).Input1LocalPathOffset) deallocCardMem((*DevicesList[*ddevice]).Input1LocalPathOffset);
  if ((*DevicesList[*ddevice]).Input1RemotePathOffset) deallocCardMem((*DevicesList[*ddevice]).Input1RemotePathOffset);
  if ((*DevicesList[*ddevice]).Output2Offset) deallocCardMem((*DevicesList[*ddevice]).Output2Offset);
  if ((*DevicesList[*ddevice]).Output2LocalPathOffset) deallocCardMem((*DevicesList[*ddevice]).Output2LocalPathOffset);
  if ((*DevicesList[*ddevice]).Output2RemotePathOffset) deallocCardMem((*DevicesList[*ddevice]).Output2RemotePathOffset);
  if ((*DevicesList[*ddevice]).Input2Offset) deallocCardMem((*DevicesList[*ddevice]).Input2Offset);
  if ((*DevicesList[*ddevice]).Input2LocalPathOffset) deallocCardMem((*DevicesList[*ddevice]).Input2LocalPathOffset);
  if ((*DevicesList[*ddevice]).Input2RemotePathOffset) deallocCardMem((*DevicesList[*ddevice]).Input2RemotePathOffset);
  free((void *)DevicesList[*ddevice]);
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=write(dni,ddevice,2);
  return(CardCommand( DELETE_DEVICE, CMD_TIMEOUT ));
}

int RemoveDevice(short * ddevice)
{

  int dummy;
  
  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=write(dni,ddevice,2);
  return(CardCommand( DELETE_DEVICE, CMD_TIMEOUT ));
}

int SimpleAddDevice( DNS_DEVICE_CFG * DeviceDescrip)
{
  int dummy;

  dummy=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+COMMANDPARAMSLOC);
  dummy=write(dni,DeviceDescrip,sizeof(DNS_DEVICE_CFG));
  fnWait(50000);
  return(CardCommand( ADD_DEVICE, CMD_TIMEOUT ));
}

int AddDevice( DNS_DEVICE_CFG * DeviceDescrip, int maxfailures)
{
  int foo;
  int failurecounter;
  short tmpsht;
  unsigned char dstatus;
  time_t initialtime,passedtime,accumedtime;

  failurecounter = 0;
  do
    {
      foo=SimpleAddDevice(DeviceDescrip);
      if(foo)
	{
	  printf("Unable to add device with MacID %i, AddDevice returned %i\n",(*DeviceDescrip).MacId,foo);
	  break;
	}
      initialtime = time(&passedtime);
      do
	{
	  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(STATUSTOP+(((*DeviceDescrip).MacId)*16)));  
	  foo=read(dni,&dstatus,1);
	  accumedtime = time(&passedtime) - initialtime;
	  printf("Status of device %i is %x  \r",(*DeviceDescrip).MacId,dstatus);
	} while((dstatus != 2) && (dstatus != 3) && (accumedtime < DEVONLINEWAIT));
      printf("\n");
      foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(STATUSTOP+(((*DeviceDescrip).MacId)*16)));  
      foo=read(dni,&dstatus,1);
      if ((dstatus != 2) && (maxfailures > 0))
	{
	  failurecounter++;
	  printf("Having to retry MacID %i, attempt number %i\n",(*DeviceDescrip).MacId,failurecounter);
	  printf("Device status for device %i was %x\n",(*DeviceDescrip).MacId,dstatus);
	  tmpsht=(*DeviceDescrip).MacId;
	  RemoveDevice(&tmpsht);
	}
    } while((dstatus != 2) && (failurecounter < maxfailures));
  if (((failurecounter >= maxfailures) || (maxfailures == 0)) && (dstatus != 2))
    {
      printf("WARNING!! Unable to bring device MacID %i online!\n",(*DeviceDescrip).MacId);
      return(-10);
    }
  return(0);
}

/*--------------------------------------------------------------------------
EnableEventQueueing().  Enables queueing and interrupt generation when a
particular event associated with a device occurs.  "flags" sets the type
of event we wish to be notified about, namely, device status, I/O 1, I/O 2,
or Explicit message.
----------------------------------------------------------------------------*/
void EnableEventQueueing(unsigned short macid, unsigned short flags)
{
  int foo;
  unsigned short tmpflags;
  unsigned char tmpchr;

  printf("Entered EnableEventQueueing\n");
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)+EQFLAGOFF));
  printf("Computed address value = %x\n",(CONTROLTOP+(macid*DCONSIZE)+EQFLAGOFF));
  foo=read(dni,&tmpflags,2);
  tmpflags=tmpflags | flags;
  printf("Writing: %i\n",(int)tmpflags);
  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)+EQFLAGOFF));
  foo=write(dni,&tmpflags,2);
}
  
/*--------------------------------------------------------------------------
DisableEventQueueing().  Disables queueing and interrupt generation when a
particular event associated with a device occurs.
----------------------------------------------------------------------------*/
void DisableEventQueueing(unsigned short macid, unsigned short flags)
{
  int foo;
  unsigned short tmpflags;

  foo=ioctl(dni,SS5136DN_SETCARDMEMPTR,DATA_BASE+(CONTROLTOP+(macid*DCONSIZE)+EQFLAGOFF));
  foo=read(dni,&tmpflags,2);
  tmpflags=tmpflags & ~flags;
  foo=write(dni,&tmpflags,2);
}

