/*
 * ss5136_cs.c -- Linux device driver for the S.S Technologies 5136-DN-PCM
 * PCMCIA CAN-Network interface card.  Version 0.3.
 *
 * Copyright (c) 1997,1998,1999 The Laitram Corporation.
 * All rights reserved.
 *
 * Author: Mark Sutton (mes)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details. It is contained in
 * the file "COPYING" in the root directory of this distribution.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * Revision history:
 * -Initial release (Version 0.1) Jan 4, 1999 (mes)
 * This release, V 0.1 is based on version 1.0 of ss5136dn.c,
 * the driver for the ISA version of the SST card.
 *
 * -Version 0.2 March 9, 1999 (mes)
 * Version 0.2 is based on version 1.2 of ss5136dn.c,
 * the driver for the ISA version of the SST card.
 * This version makes the driver more independant of kernel version.
 *
 * -Version 0.3 March 14, 2000 (mes)
 * Changed major number from 144 to 183 since SUSE usurped 144 for their
 * encapsulated PPP driver.
 */

#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/module.h>
#include <linux/mm.h>
#include <asm/pgtable.h>
#include <linux/time.h>
#include <linux/timer.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/errno.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>

#include "ss5136dn_cs.h"

static unsigned int ss5136dn_status;
char * cardmemptr;
int gCardType,gMaxReg,memdis;

/*
   Include debugging code, normally disabled via config files.
*/
#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DEBUG(n, args...) if (pc_debug>(n)) printk(KERN_DEBUG args);
static char *version =
"ss5136_cs.c 0.2 1999/03/09 (Mark Sutton)";
#else
#define DEBUG(n, args...)
#endif

/*
   PCMCIA specific function prototypes.
*/

static void ss5136dn_config(dev_link_t *link);
static void ss5136dn_release(u_long arg);
static int ss5136dn_event(event_t event, int priority,
		       event_callback_args_t *args);
static dev_link_t *ss5136dn_attach(void);
static void ss5136dn_detach(dev_link_t *);
static int ss5136dn_init(void);
static void ss5136dn_shutdown(void);


static dev_info_t dev_info = "ss5136dn_cs";

static dev_link_t *dev_list = NULL;

typedef struct local_info_t {
    dev_node_t	node;
    int		stop;
} local_info_t;

/*====================================================================*/

static void cs_error(client_handle_t handle, int func, int ret)
{
    error_info_t err = { func, ret };
    CardServices(ReportError, handle, &err);
}

/*======================================================================
ss5136dn_attach.  This is a close copy of the dummy_attach function in
the example dummy_cs.c that comes with the linux PCMCIA package.
It works, so here it is for now.  I believe there is a bit of unnecessary
stuff in here vis-a-vis the 5136-DN-PCM card.  TODO: strip the unnecessary
stuff from here.
===================================================================*/


static dev_link_t *ss5136dn_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    local_info_t *local;
    int ret;
    
    DEBUG(0, "ss5136dn_attach()\n");

    /* Initialize the dev_link_t structure */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &ss5136dn_release;
    link->release.data = (u_long)link;

    link->conf.Attributes = 0;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;

    /* Allocate space for private device-specific data */
    local = kmalloc(sizeof(local_info_t), GFP_KERNEL);
    memset(local, 0, sizeof(local_info_t));
    link->priv = local;
    
    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask =
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &ss5136dn_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(link->handle, RegisterClient, ret);
	ss5136dn_detach(link);
	return NULL;
    }

    return link;
} /* ss5136dn_attach */

/*======================================================================

ss5136dn_detach: Again, mostly a copy of dummy_detach from the example.

======================================================================*/

static void ss5136dn_detach(dev_link_t *link)
{
    dev_link_t **linkp;

    DEBUG(0, "ss5136dn_detach(0x%p)\n", link);
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    /*
       If the device is currently configured and active, we won't
       actually delete it yet.  Instead, it is marked so that when
       the release() function is called, that will trigger a proper
       detach().
    */
    if (link->state & DEV_CONFIG) {
#ifdef PCMCIA_DEBUG
	printk(KERN_DEBUG "ss5136dn_cs: detach postponed, '%s' "
	       "still locked\n", link->dev->dev_name);
#endif
	link->state |= DEV_STALE_LINK;
	return;
    }

    /* Break the link with Card Services */
    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free pieces */
    *linkp = link->next;
    if (link->priv) {
	kfree_s(link->priv, sizeof(local_info_t));
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* ss5136dn_detach */

/*======================================================================

    ss5136dn_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    device available to the system.
    
======================================================================*/

#define CS_CHECK(fn, args...) \
while ((last_ret=CardServices(last_fn=(fn),args))!=0) goto cs_failed

#define CFG_CHECK(fn, args...) \
if (CardServices(fn, args) != 0) goto next_entry

static void ss5136dn_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    local_info_t *dev;
    int last_fn, last_ret;
    u_char buf[64];
    win_req_t req;
    /*    memreq_t map;  */
    int ret;
    
    handle = link->handle;
    dev = link->priv;

    DEBUG(0, "ss5136dn_config(0x%p)\n", link);

    /*
      Read the card's config tuple.
    */
    tuple.DesiredTuple = CISTPL_CONFIG;
    tuple.Attributes = 0;
    tuple.TupleData = buf;
    tuple.TupleDataMax = sizeof(buf);
    tuple.TupleOffset = 0;
    CS_CHECK(GetFirstTuple, handle, &tuple);
    CS_CHECK(GetTupleData, handle, &tuple);
    CS_CHECK(ParseTuple, handle, &tuple, &parse);
    link->conf.ConfigBase = parse.config.base;
    link->conf.Present = parse.config.rmask[0];
    
    /* Configure card */
    link->state |= DEV_CONFIG;

    /*
      Get all the configuration entries we can.
    */
    tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
    CS_CHECK(GetFirstTuple, handle, &tuple);
    while (1) {
	cistpl_cftable_entry_t dflt = { 0 };
	cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
	CFG_CHECK(GetTupleData, handle, &tuple);
	CFG_CHECK(ParseTuple, handle, &tuple, &parse);

	if (cfg->index == 0) goto next_entry;
	link->conf.ConfigIndex = cfg->index;
	
	/* Does this card need audio output? */
	if (cfg->flags & CISTPL_CFTABLE_AUDIO) {
	    link->conf.Attributes |= CONF_ENABLE_SPKR;
	    link->conf.Status = CCSR_AUDIO_ENA;
	}
	
	/* Use power settings for Vcc and Vpp */
	if (cfg->vcc.present & (1<<CISTPL_POWER_VNOM))
	    link->conf.Vcc = cfg->vcc.param[CISTPL_POWER_VNOM]/10000;
	else if (dflt.vcc.present & (1<<CISTPL_POWER_VNOM))
	    link->conf.Vcc = dflt.vcc.param[CISTPL_POWER_VNOM]/10000;
	    
	if (cfg->vpp1.present & (1<<CISTPL_POWER_VNOM))
	    link->conf.Vpp1 = link->conf.Vpp2 =
		cfg->vpp1.param[CISTPL_POWER_VNOM]/10000;
	else if (dflt.vpp1.present & (1<<CISTPL_POWER_VNOM))
	    link->conf.Vpp1 = link->conf.Vpp2 =
		dflt.vpp1.param[CISTPL_POWER_VNOM]/10000;
	
	/* IO window settings */
	link->io.NumPorts1 = link->io.NumPorts2 = 0;
	if ((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
	    cistpl_io_t *io = (cfg->io.nwin) ? &cfg->io : &dflt.io;
	    link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
	    if (!(io->flags & CISTPL_IO_8BIT))
		link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
	    if (!(io->flags & CISTPL_IO_16BIT))
		link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
	    link->io.BasePort1 = SS5136DN_PORTBASE;
	    link->io.NumPorts1 = SS5136DN_IOSIZE;
	}
	/* This reserves IO space but doesn't actually enable it */
	CFG_CHECK(RequestIO, link->handle, &link->io);

	/*
	  Set up memory block.
	*/
	if ((cfg->mem.nwin > 0) || (dflt.mem.nwin > 0)) {
	  /*	    cistpl_mem_t *mem =
		    (cfg->mem.nwin) ? &cfg->mem : &dflt.mem;  */
	    req.Attributes = WIN_DATA_WIDTH_16|WIN_MEMORY_TYPE_CM|WIN_ENABLE;
	    req.Base = SS5136DN_MEMBASE;
	    req.Size = SS5136DN_MEMSIZE;
	    req.AccessSpeed = 0;
	    link->win = (window_handle_t)link->handle;
	    CFG_CHECK(RequestWindow, &link->win, &req);
	    /*	    map.Page = 0; map.CardOffset = mem->win[0].card_addr;
		    CFG_CHECK(MapMemPage, link->win, &map);   */
	}
	break;
	
    next_entry:
	if (cfg->flags & CISTPL_CFTABLE_DEFAULT) dflt = *cfg;
	CS_CHECK(GetNextTuple, handle, &tuple);
    }
    
    /*
       This actually configures the PCMCIA socket -- setting up
       the I/O windows and the interrupt mapping, and putting the
       card and host interface into "Memory and IO" mode.
    */
    CS_CHECK(RequestConfiguration, link->handle, &link->conf);

    /*
      At this point, the dev_node_t structure(s) need to be
      initialized and arranged in a linked list at link->dev.
    */
    sprintf(dev->node.dev_name, "ss5136dn");
    dev->node.major = SS5136DN_MAJOR; dev->node.minor = 0;
    link->dev = &dev->node;

    /* Finally, report what we've done */
    printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d",
	   dev->node.dev_name, link->conf.ConfigIndex,
	   link->conf.Vcc/10, link->conf.Vcc%10);
    if (link->conf.Vpp1)
	printk(", Vpp %d.%d", link->conf.Vpp1/10, link->conf.Vpp1%10);
    /*    if (link->conf.Attributes & CONF_ENABLE_IRQ)
	  printk(", irq %d", link->irq.AssignedIRQ);  */
    if (link->io.NumPorts1)
	printk(", io 0x%04x-0x%04x", link->io.BasePort1,
	       link->io.BasePort1+link->io.NumPorts1-1);
    if (link->io.NumPorts2)
	printk(" & 0x%04x-0x%04x", link->io.BasePort2,
	       link->io.BasePort2+link->io.NumPorts2-1);
    if (link->win)
	printk(", mem 0x%06lx-0x%06lx", req.Base,
	       req.Base+req.Size-1);
    printk("\n");
    
    link->state &= ~DEV_CONFIG_PENDING;
   ret=ss5136dn_init();
   if (ret)
      ss5136dn_shutdown();
    return;

cs_failed:
    cs_error(link->handle, last_fn, last_ret);
    ss5136dn_release((u_long)link);

} /* ss5136dn_config */

/*======================================================================

    After a card is removed, ss5136dn_release() will unregister the
    device, and release the PCMCIA configuration.  If the device is
    still open, this will be postponed until it is closed.
    
======================================================================*/

static void ss5136dn_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;

    DEBUG(0, "ss5136dn_release(0x%p)\n", link);

    /*
       If the device is currently in use, we won't release until it
       is actually closed, because until then, we can't be sure that
       no one will try to access the device or its data structures.
    */
    if (link->open) {
	DEBUG(1, "ss5136dn_cs: release postponed, '%s' still open\n",
	      link->dev->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    /* Unlink the device chain */
    link->dev = NULL;

    /* Don't bother checking to see if these succeed or not */
    if (link->win)
	CardServices(ReleaseWindow, link->win);
    CardServices(ReleaseConfiguration, link->handle);
    if (link->io.NumPorts1)
	CardServices(ReleaseIO, link->handle, &link->io);
    /*    if (link->irq.AssignedIRQ)
	  CardServices(ReleaseIRQ, link->handle, &link->irq);  */
    link->state &= ~DEV_CONFIG;
    
    if (link->state & DEV_STALE_LINK)
	ss5136dn_detach(link);
    
} /* ss5136dn_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.

    When a CARD_REMOVAL event is received, we immediately set a
    private flag to block future accesses to this device.  All the
    functions that actually access the device should check this flag
    to make sure the card is still present.
    
======================================================================*/

static int ss5136dn_event(event_t event, int priority,
		       event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;

    DEBUG(1, "ss5136dn_event(0x%06x)\n", event);
    
    switch (event) {
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    ((local_info_t *)link->priv)->stop = 1;
	    link->release.expires = RUN_AT(HZ/20);
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	ss5136dn_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	/* Mark the device as stopped, to block IO until later */
	((local_info_t *)link->priv)->stop = 1;
	if (link->state & DEV_CONFIG)
	    CardServices(ReleaseConfiguration, link->handle);
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG)
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	((local_info_t *)link->priv)->stop = 0;
	/*
	  In a normal driver, additional code may go here to restore
	  the device state and restart IO. 
	*/
	break;
    }
    return 0;
} /* ss5136dn_event */

/*====================================================================*/

static int ss5136dn_open(struct inode * inode, struct file * file)
{
  if (ss5136dn_status & SS5136DN_BUSY)
    {
      return(-EBUSY);
    }
  ss5136dn_status = ss5136dn_status | SS5136DN_BUSY; 
  if (MINOR(inode->i_rdev) != 0) return(-ENODEV);
  return(0);
}

static FS_RELEASE_T ss5136dn_close(struct inode * inode, struct file * file)
{
  ss5136dn_status = ss5136dn_status & (~SS5136DN_BUSY);
  return (FS_RELEASE_T)0;
}

/* Card is disabled ("turned off") when all three i/o registers set to 0 */
static void ss5136dn_shutdown(void)
{
  outb(0,SS5136DN_PORTBASE + BCR0);
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);
}

void cleanup_module(void)
{
  unregister_chrdev(SS5136DN_MAJOR,"ss5136dn");
  ss5136dn_shutdown();
    DEBUG(0, "ss5136dn_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL) {
	if (dev_list->state & DEV_CONFIG)
	    ss5136dn_release((u_long)dev_list);
	ss5136dn_detach(dev_list);
    }
}

static int ss5136dn_init(void)
{

  unsigned char tstin;
  unsigned char portout;

  outb(0,SS5136DN_PORTBASE + BCR0);  /* Clear all i/o registers */
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);

/* Next, do a "card alive" test, line at 0x4 will almost immediately
reset itself if card is working correctly */
  outb(0x06,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0x02)
    {
      printk("SS5136DN card not present or failing to answer are you there poll! \n");
      printk("Aborting attempt to set up SS5136DN card!\n");
      ss5136dn_shutdown();
      return(-1);
    }
  outb(0,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0)
    {
      printk("SS5136DN board not responding properly. Aborting attempt to initialize it.\n");
      ss5136dn_shutdown();
      return(-1);
    }
/* Compute the memory base address code to load and load it */
  portout=((SS5136DN_MEMBASE >> 14) & 0x1f);
  if (PAGESIZEINBYTES == 0x4000) {
    portout=(portout | WIN16K);
    printk("ss5136dn: Setting a 16k aperature.\n");
  }
  else {
    printk("ss5136dn: Setting a 32k aperature.\n");
  }
  outb(portout,SS5136DN_PORTBASE + BCR1);
  tstin=inb(SS5136DN_PORTBASE + BCR1);
  if (tstin != portout)
    {
      printk("Unable to set SS5136DN board's base address! Aborting attempt to innitialize it!\n");
      ss5136dn_shutdown();
      return(-1);
    }
/* Setup the BCR0 register to initial condition:
   Bits are as followes:
   bit0 = Board interupt high enables board processor (on now).
   bit1 = Reserved (off now).
   bit2 = Interupt status latch (set/reset latch mirrors ISA interupt 
          generation, off now)
   bit3 = ISA interrupt enable (off now).
   bit4,5,6 = Page select (loaded with 001 or 011 depending on aperature size,
              contains the "Application Module Header".  This is how this
              will always be set unless loading application kernel mods.)
   bit7 = Memory enable (on now).
*/
  if (PAGESIZEINBYTES == 0x8000) {
    outb(0x91,SS5136DN_PORTBASE + BCR0);
  }
  else {
    outb(0xb1,SS5136DN_PORTBASE + BCR0);
  }
  tstin=inb(SS5136DN_PORTBASE + BCR7);
  if (tstin == 0x20)
    {
      gCardType = PRO;
      gMaxReg = 7;
    }
  else
    {
      gCardType = REGULAR;
      gMaxReg = 2;
    }
/* Turn the "Health indicator" LED green because card is now ready to use */
  outb(0x20,SS5136DN_PORTBASE + BCR2);
  cardmemptr = (char *)SS5136DN_MEMBASE;
  printk("ss5136dn: Driver for the S.S Technologies CAN Bus interface board, successfully loaded.\n");
  if (gCardType == PRO)
    printk("ss5136dn: This appears to be a PRO board.\n");
  else
    printk("ss5136dn: This appears to be a 5136-dn, basic board.\n");
  printk("ss5136dn: Copyright (c) 1997, 1998, 1999 The Laitram Corporation\n");
  printk("ss5136dn: Written By: Mark Sutton\n");
  return(0);
}

static int ss5136dn_reset(void)
{

  unsigned char tstin;
  unsigned char portout;

  outb(0,SS5136DN_PORTBASE + BCR0);  /* Clear all i/o registers */
  outb(0,SS5136DN_PORTBASE + BCR1);
  outb(0,SS5136DN_PORTBASE + BCR2);

/* Next, do a "card alive" test, line at 0x4 will almost immediately
reset itself if card is working correctly */
  outb(0x06,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0x02)
    {
      ss5136dn_shutdown();
      return(-1);
    }
  outb(0,SS5136DN_PORTBASE + BCR0);
  tstin = inb(SS5136DN_PORTBASE + BCR0);
  if (tstin != 0)
    {
      ss5136dn_shutdown();
      return(-1);
    }
/* Compute the memory base address code to load and load it */
  portout=((SS5136DN_MEMBASE >> 14) & 0x1f);
  if (PAGESIZEINBYTES == 0x4000) {
    portout=(portout | WIN16K);
  }
  outb(portout,SS5136DN_PORTBASE + BCR1);
  tstin=inb(SS5136DN_PORTBASE + BCR1);
  if (tstin != portout)
    {
      ss5136dn_shutdown();
      return(-1);
    }
/* Setup the BCR0 register to initial condition:
   Bits are as followes:
   bit0 = Board interupt high enables board processor (on now).
   bit1 = Reserved (off now).
   bit2 = Interupt status latch (set/reset latch mirrors ISA interupt 
          generation, off now)
   bit3 = ISA interrupt enable (off now).
   bit4,5,6 = Page select (loaded with 001 or 011 depending on aperature size,
              contains the "Application Module Header".  This is how this
              will always be set unless loading application kernel mods.)
   bit7 = Memory enable (on now).
*/
  if (PAGESIZEINBYTES == 0x8000) {
    outb(0x91,SS5136DN_PORTBASE + BCR0);
  }
  else {
    outb(0xb1,SS5136DN_PORTBASE + BCR0);
  }
    
/* Turn the "Health indicator" LED green because card is now ready to use */
  outb(0x20,SS5136DN_PORTBASE + BCR2);
  cardmemptr = (char *)SS5136DN_MEMBASE;
  return(0);
}

static FS_SIZE_T ss5136dn_read FOPS(struct inode * inode, struct file * file, char * buffer, U_FS_SIZE_T count, loff_t * dummy)
{

#if (LINUX_VERSION_CODE >= VERSION(2,1,0))
  int ctr;
#endif

  int i;
  char byteallones = 0xff;

  if (MINOR(F_INODE(file)->i_rdev) != 0) return(-ENODEV);
  if ((cardmemptr + count) > (char*)(SS5136DN_MEMBASE + PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
  /* YACK!!! Here's the butt ugly cludge!  Fake as if the memory is disabled.
     Find out what is really wrong and FIX THIS!  */
    if (memdis)
    {
      for (i=0;i<count;i++)
	{
#if (LINUX_VERSION_CODE < VERSION(2,1,0))
	  copy_to_user((char *)(buffer+i),&byteallones,1);
#else
	  *(buffer + i) = readb(cardmemptr);
	  cardmemptr++;
#endif
	}
    }
  else
    {
#if (LINUX_VERSION_CODE < VERSION(2,1,0))
      copy_to_user((char *)buffer,(char *)cardmemptr,count);
      cardmemptr = cardmemptr + count;
#else
  for(ctr=0;ctr<count;ctr++)
    {
      *(buffer + ctr) = readb(cardmemptr);
      cardmemptr++;
    }
#endif
    } 
  return(count);
}

static FS_SIZE_T ss5136dn_write FOPS(struct inode * inode, struct file * file, const char * buffer, U_FS_SIZE_T count, loff_t * dummy)
{

#if (LINUX_VERSION_CODE >= VERSION(2,1,0))
  int ctr;
#endif

  if (MINOR(F_INODE(file)->i_rdev) != 0) return(-ENODEV);
  if ((cardmemptr + count) > (char*)(SS5136DN_MEMBASE + PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
#if (LINUX_VERSION_CODE < VERSION(2,1,0))
  copy_from_user((char *)cardmemptr,(char *)buffer,(int)count);
  cardmemptr = cardmemptr + count;
#else
  for (ctr=0;ctr<count;ctr++)
    {
      writeb(*(buffer+ctr),cardmemptr);
      cardmemptr++;
    }
#endif
  return(count);
}


static int ss5136dn_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
  char outchr,inchr;
  unsigned short portnum,portloc;
  int pageval;
  portiopair * argp;
  int argi;

  if (MINOR(inode->i_rdev) != 0) return(-ENODEV);
  switch(cmd){
  case SS5136DN_SETCARDMEMPTR:
    argi = (int)arg;
    if((argi < 0) || (argi > PAGESIZEINBYTES)) return(-ECARDMEMOUTOFRANGE);
    cardmemptr=(char *)(SS5136DN_MEMBASE + argi);
    break;
  case SS5136DN_BIGRESET:
    argi=ss5136dn_reset();
    if (argi)
      {
	ss5136dn_shutdown();
	return(-1);
      }
    break;
  case SS5136DN_SHUTDOWN:
    ss5136dn_shutdown();
    break;
  case SS5136DN_REPORT_APERATURE:
    return(PAGESIZEINBYTES);
    break;
  case SS5136DN_REPORT_CARDTYPE:
    return(gCardType);
    break;
  case SS5136DN_WRITEREG: /* Generic write to ss5136dn i/o register */
    argp = (portiopair *)arg;
    copy_from_user(&outchr,&argp->value,1);
    copy_from_user(&portnum,&argp->port,2);
    if(portnum > gMaxReg) return(-EINVAL);
    portloc = portnum + SS5136DN_PORTBASE;
    outb(outchr,portloc);
    break;
  case SS5136DN_READREG: /* Generic read to ss5136dn i/o register */
    argp = (portiopair *)arg;
    copy_from_user(&portnum,&argp->port,2);
    if(portnum > gMaxReg) return(-EINVAL);
    portloc = portnum + SS5136DN_PORTBASE;
    inchr = (char)inb(portloc);
    copy_to_user(&argp->value,&inchr,1);
    break;
  case SS5136DN_PAGECHANGE: /* Change the memory page on ss5136dn */
    pageval = (int)arg;
    if ((pageval < 0) || (pageval > MAXPAGES)) return(-EINVAL);
    pageval = pageval << 4;
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & SS5136DN_PAGEMASK; /* strip page bits from byte */
    outchr = outchr | (char)pageval;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    cardmemptr=(char *)SS5136DN_MEMBASE;
    break;
  case SS5136DN_MEMENABLE: /* Enable ss5136dn memory access */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_MEMENABLEMASK;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    memdis = 0;
    break;
  case SS5136DN_MEMDISABLE: /* Disable ss5136dn memory access (usefull for detecting memory conflicts) */
    /* Disabling the memory is not working on the PCMCIA card, therfore, there
       is a kludge here, disable reading and writing by the driver.  */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_MEMENABLEMASK);
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    memdis = -1;
    break;
  case SS5136DN_PROCENABLE: /* Enable the processor on the ss5136dn */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_PROCENABLEMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_PROCDISABLE: /* Disable the processor on the ss5136dn */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_PROCENABLEMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_WDENABLE: /* Enable the ss5136dn watchdog timer (true low) */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_WDENABLEMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_WDDISABLE: /* Disable the ss5136dn watchdog timer (true high) */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_WDENABLEMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;  
  case SS5136DN_HLTHGREEN: /* Turn the health LED green */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_HLTHMASK;
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_HLTHRED: /* Turn the health LED red */
    inchr = (char)inb(BCR2+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_HLTHMASK);
    outb(outchr,BCR2+SS5136DN_PORTBASE);
    break;
  case SS5136DN_CLINT: /* Clear the card's (output) interupt latch */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr | PCINT;
    outb(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  case SS5136DN_STROBECINT: /* Strobe the ss5136dn interrupt line to make it process a loaded command */
    inchr = (char)inb(BCR0+SS5136DN_PORTBASE);
    outchr = inchr & ~(SS5136DN_CINTMASK);
    outb_p(outchr,BCR0+SS5136DN_PORTBASE);
    outchr = inchr | SS5136DN_CINTMASK;
    outb_p(outchr,BCR0+SS5136DN_PORTBASE);
    break;
  default:
    return(-EINVAL);
    break;
  }
  return(0);
}
static struct file_operations ss5136dn_fops = {
        NULL,   	/* seek */
	ss5136dn_read, 	/* read */
	ss5136dn_write,	/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	ss5136dn_ioctl,	/* ioctl */
	NULL,     	/* mmap */
        ss5136dn_open,    /* open */
	NULL_FLUSH
        ss5136dn_close,	/* release */
	NULL,		/* fsync */
	NULL,		/* fasync */
	NULL,		/* check_media_change */
	NULL		/* revalidate */
};

int init_module(void)
{

    servinfo_t serv;
    DEBUG(0, "%s\n", version);
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk(KERN_NOTICE "ss5136dn_cs: Card Services release "
	       "does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &ss5136dn_attach, &ss5136dn_detach);
  if (register_chrdev(SS5136DN_MAJOR,"ss5136dn",&ss5136dn_fops)) {
    printk("ss5136dn: unable to get major %d for ss5136dn device\n",
	    SS5136DN_MAJOR);
    return -EIO;
  }
  return 0;
}

