/*
 * output.c: handles a variety of tasks dealing with the output from the irc
 * program 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */

#include "irc.h"
#include <sys/stat.h>

#include <stdarg.h>

#include "output.h"
#include "vars.h"
#include "input.h"
#include "ircterm.h"
#include "lastlog.h"
#include "window.h"
#include "screen.h"
#include "hook.h"
#include "ctcp.h"
#include "log.h"
#include "misc.h"

	int	in_help = 0;

/* make this buffer *much* bigger than needed */

#define LARGE_BIG_BUFFER_SIZE BIG_BUFFER_SIZE * 4

static	char	putbuf[LARGE_BIG_BUFFER_SIZE + 1];
extern	LastMsg	last_servermsg[];

char three_stars[4] = "***";

/* unflash: sends a ^[c to the screen */
/* Must be defined to be useful, cause some vt100s really *do* reset when
   sent this command. >;-) */

/* Now that you can send ansi sequences, this is much less inportant.. */
void unflash _((void))
{
#ifdef HARD_UNFLASH
	fwrite("\033c\033(U", 5, 1, stdout);		/* hard reset */
#else
	fwrite("\033)0\033(U", 6, 1, stdout);		/* soft reset */
#endif
}

/*
 * refresh_screen: Whenever the REFRESH_SCREEN function is activated, this
 * swoops into effect 
 */
void refresh_screen (unsigned char dumb, char *dumber)
{
#ifndef WINNT
	term_clear_screen();
	unflash();
#else
	SetConsoleTitle("BitchX The Ultimate IRC client");
	NT_ClearScreen();
#endif
	if (term_resize())
		recalculate_windows();
	else
		redraw_all_windows();
	update_all_windows();
	update_input(UPDATE_ALL);
}

/* init_windows:  */
int init_screen _((void))
{
	if (term_init())
		return -1;
#ifdef WINNT
	NT_ClearScreen();
#else
	term_clear_screen();
#endif
	term_resize();
	new_window();
	recalculate_windows();
	update_all_windows();
	init_input();
	term_move_cursor(0, 0);
	return 0;
}

/*
 * put_it: the irc display routine.  Use this routine to display anything to
 * the main irc window.  It handles sending text to the display or stdout as
 * needed, add stuff to the lastlog and log file, etc.  Things NOT to do:
 * Dont send any text that contains \n, very unpredictable.  Tabs will also
 * screw things up.  The calling routing is responsible for not overwriting
 * the 1K buffer allocated.  
 *
 * For Ultrix machines, you can't call put_it() with floating point arguements.
 * It just doesn't work.  - phone, jan 1993.
 */
void put_it(const char *format, ...)
{
	if (window_display && format)
	{
		va_list args;
		memset(putbuf, 0, 200);
		va_start (args, format);
		vsnprintf(putbuf, LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		if (*putbuf)
		{
			add_to_log(irclog_fp, putbuf);
			add_to_screen(putbuf);
		}
	}
}

/* This is an alternative form of put_it which writes three asterisks
 * before actually putting things out.
 */
void say (const char *format, ...)
{
int len = 0;
	if (window_display && format)
	{
		va_list args;
		va_start (args, format);
		if (thing_ansi)
			len = strlen(thing_ansi);
		else
			len = 3;
		vsnprintf(&(putbuf[len+1]), LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		strcpy(putbuf, thing_ansi?thing_ansi:three_stars);
		putbuf[len] = ' ';
		if (strip_ansi_in_echo) {
		register char *ptr;
			for (ptr = putbuf + len; *ptr; ptr++)
				if (*ptr < 31 && *ptr > 13)
					if (*ptr != 15 && *ptr != 22)
						*ptr = (*ptr & 127) | 64;
		}
		add_to_log(irclog_fp, putbuf);
		add_to_screen(putbuf);
	}
}

void bitchsay (const char *format, ...)
{
int len;
	if (window_display && format)
	{
		va_list args;
		va_start (args, format);
		sprintf(putbuf, "%s \002%s\002: ", thing_ansi?thing_ansi:three_stars, version);
		len = strlen(putbuf);
		vsnprintf(&(putbuf[len]), LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		if (strip_ansi_in_echo) {
		register char *ptr;
			for (ptr = putbuf+len; *ptr; ptr++)
				if (*ptr < 31 && *ptr > 13)
					if (*ptr != 15 && *ptr != 22)
						*ptr = (*ptr & 127) | 64;
		}
		add_to_log(irclog_fp, putbuf);
		add_to_screen(putbuf);
	}
}

void	yell(const char *format, ...)
{
	if (format)
	{
		va_list args;
		va_start (args, format);
		vsnprintf(putbuf, LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		do_hook(YELL_LIST, "%s", putbuf);
		add_to_log(irclog_fp, putbuf);
		add_to_screen(putbuf);
	}
}


/* help_put_it: works just like put_it, but is specially used by help */
void	help_put_it (const char *topic, const char *format, ...)
{
	if (format)
	{
		va_list args;
		va_start (args, format);
		vsnprintf(putbuf, LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		in_help = 1;
		if (do_hook(HELP_LIST, "%s %s", topic, putbuf))
		{
			message_from(NULL, LOG_CURRENT);
			if (window_display)
			{
				add_to_log(irclog_fp, putbuf);
				add_to_screen(convert_output_format(get_string_var(FORMAT_HELP_VAR), "%s", putbuf));
			}
			message_from(NULL, LOG_CRAP);
		}
		in_help = 0;
	}
}

void serversay (int save, const char *format, ...)
{
	Window	*old_to_window = to_window;
        int     lastlog_level;
        lastlog_level = set_lastlog_msg_level(LOG_CRAP);
	if (get_int_var(OV_VAR))
		to_window = get_window_by_name("OPER_VIEW");
        if (window_display && format)
        {
		va_list args;
		va_start (args, format);
		vsnprintf(&(putbuf[strlen(get_string_var(SERVER_PROMPT_VAR))+1]), LARGE_BIG_BUFFER_SIZE, format, args);
		va_end(args);
		strcpy(putbuf, get_string_var(SERVER_PROMPT_VAR));
		putbuf[strlen(get_string_var(SERVER_PROMPT_VAR))] = ' ';

		if (*putbuf)
		{
			add_to_log(irclog_fp, putbuf);
			add_to_screen(putbuf);
		}
	}
	if (get_int_var(OV_VAR))
		to_window = old_to_window;
	set_lastlog_msg_level(lastlog_level);
	if (save)
		add_last_type(&last_servermsg[0], MAX_LAST_MSG, NULL, NULL, NULL, putbuf);
}
