	
#include <config.h>
#include <janbot.h>

/* Function: Parse and process an incoming DCC. */
void incoming_dcc(char *orig,char *args)
{
	char *cmd;
	for (cmd=args;(*args!=' ')&&(*args!='\0');args++);
	*args++='\0';

	ENTER2(1,"incoming_dcc(\"%s\",\"%s\")\n",orig,args);

	if (strcasecmp("SEND",cmd)==0)
	{
		open_incoming_file(orig,args);
		LEAVE(1);
		return;
	}
	else if (strcasecmp("RESUME",cmd)==0)
	{
		parse_resume_request(orig,args);
		LEAVE(1);
		return;
	}
	else if (strcasecmp("CHAT",cmd)==0)
	{
		send_to_server("NOTICE %s :\001ERRMSG I do not accept DCC CHAT without authorization. Use '/CTCP %s AUTH <passwd>' to connect.\001\n",getnick(orig),cfg.nick);
	}
	else
	{
		send_to_server("NOTICE %s :\001ERRMSG Unknown dcc type: %s.\001\n",getnick(orig),cmd);
	}	
	LEAVE(1);
}

/* Function: Preprocess an incoming file. */
void open_incoming_file(char *orig,char *args)
{
	struct user_t *tmp;
	struct chat_t *ctmp;
	char *addr,*port,*size,*file;
	unsigned long laddr,lsize;
	unsigned int iport;

	ENTER2(1,"open_incoming_dcc(\"%s\",\"%s\")\n",orig,args);
	
	if ((tmp=get_user(getnick(orig),&users))==NULL)
	{
		send_to_server("NOTICE %s :\001ERRMSG I do not accept files from strangers.\001\n",getnick(orig));
		LEAVE(1);
		return;
	}
	
	if ((ctmp=find_chat(tmp,&chatlist))==NULL)
	{
		send_to_server("NOTICE %s :\001ERRMSG I am sorry, I cannot accept a file unless you're connected.\001\n",getnick(orig));
		LEAVE(1);
		return;
	}

	if ((file=strtok(args," "))==NULL) { LEAVE(1); return; } /* This is a fake DCC */
	if ((addr=strtok(NULL," "))==NULL) { LEAVE(1); return; } /* Fake */
	if ((port=strtok(NULL," "))==NULL) { LEAVE(1); return; } /* Fake */
	if ((size=strtok(NULL," "))==NULL) { LEAVE(1); return; } /* Fake */
	sscanf(addr,"%lu",&laddr);
	sscanf(port,"%u",&iport);
	sscanf(size,"%lu",&lsize);

	if (laddr==0||iport<=1024||lsize==0) return; /* Hmmm, suspicous */
	
	send_to_child(ctmp,GET,"%s%s/%s %lu %u %lu",cfg.filedir,ctmp->chatter->dir,file,laddr,iport,lsize);
	log(DCC,"Receiving %s%s%s from %s.",cfg.filedir,ctmp->chatter->dir,file,ctmp->chatter->nick);
	
	LEAVE(1);
}

/* Function: Sets up a DCC SEND connection. */
void process_outgoing_file(char *dccinfo,int dcctype)
{
	char *dest,*fname,*tmpch,tempstr[NAME_MAX+1],command[NAME_MAX+1];
	struct dcc_t *tmp;
	struct stat statbuf;

	ENTER2(1,"process_outgoing_file(\"%s\",%d)\n",dccinfo,dcctype);

	for (dest=dccinfo;*dccinfo!=' ';dccinfo++);
	*dccinfo++='\0';
	fname=dccinfo;
	for (;(*dccinfo!='\n')&&(*dccinfo!='\0');dccinfo++);
	*dccinfo='\0';

	if (dcc_exists(fname))
	{
		send_to_parent(LIST,"%s \002%s%s\002 is already being DCCed.",dest,base_name(fname),dcctype&D_TAR?".tar":"");
		LEAVE(1);
		return;
	}

	if ((tmp=(struct dcc_t *)malloc(sizeof(struct dcc_t)))==NULL)
	{
		/* No mem */
		LEAVE(1);
		return;
	}	

	bzero(tmp,sizeof(struct dcc_t));
	strcpy(tmp->fname,fname);
	tmp->type=dcctype;
	tmp->active=0;
	tmp->transmit=0;
	tmp->bsize=cfg.dccblocksize;

	if ((tmp->buf=(char *)malloc(tmp->bsize))==NULL)
	{
		/* No mem for DCC buffer. */
		LEAVE(1);
		return;
	}

	if (dcctype&D_SEND)
	{
		if ((tmp->filefd=open(tmp->fname,O_RDONLY))==-1)
		{
			/* Cannot open file. Bummer. */
			free(tmp);
			LEAVE(1);
			return;
		}
	
		if (fstat(tmp->filefd,&statbuf)==-1)
		{
			/* I can't really imagine how this could happen... */
			close(tmp->filefd);
			free(tmp);
			LEAVE(1);
			return;
		}

		if ((tmp->total=(unsigned long)statbuf.st_size)==0)
		{
			close(tmp->filefd);
			free(tmp);
			LEAVE(1);
			return;
		}
#ifdef USE_RATIOS
		/* Do we have enough credits left for this? */
		if (ratio>0&&cred<tmp->total)
		{
			send_to_parent(LIST,"? You need \002%llu\002 more credits to receive \002%s\002.",tmp->total-cred,base_name(tmp->fname));
			close(tmp->filefd);
			free(tmp);
			LEAVE(1);
			return;
		}
#endif
	}
	else if (dcctype&D_TAR)
	{
		strcpy(tempstr,fname);
		if ((tmpch=strrchr(tempstr,(int)'/'))==NULL)
		{
			free(tmp);
			LEAVE(1);
			return;
		}	
		*tmpch='\0';
		if (chdir(tempstr))
		{
			free(tmp);
			LEAVE(1);
			return;
		}	

		/* 
		 * It's important to predict the correct size of the tar
		 * archive, because a lot of irc clients chops off the DCC
		 * when the given number of bytes has been received. I
		 * think this works, but it's possible that some tar bins
		 * use a different default block alignment (normal is 20*512).
		 */
		if ((tmp->total=calculate_tarsize(fname))==0)
		{
			free(tmp);
			LEAVE(1);
			return;
		}
		tmp->total+=1024; /* The size of the 2 end marker blocks. */
		tmp->total+=(unsigned long)((tmp->total%10240)?(10240-(tmp->total%10240)):0);

#ifdef USE_RATIOS
		if (ratio>0&&cred<tmp->total)
		{
			send_to_parent(LIST,"? You need \002%llu\002 more credits to receive \002%s.tar\002.",tmp->total-cred,base_name(tmp->fname));
			free(tmp);
			LEAVE(1);
			return;
		}
#endif
		/* Create the tar commad and redirect errors to /dev/null. */
		sprintf(command,"%s cf - %s 2>/dev/null",cfg.tarpath,base_name(fname));
		if ((tmp->tarfile=popen(command,"r"))==NULL)
		{
			/* Cannot open file. Bummer. */
			free(tmp);
			LEAVE(1);
			return;
		}
	}	

	if ((tmp->tmpfd=socket(AF_INET,SOCK_STREAM,0))<0)
	{
		if (dcctype&D_SEND) close(tmp->filefd);
		else if (dcctype&D_TAR) pclose(tmp->tarfile);
		free(tmp);
		LEAVE(1);
		return;
	}

	bzero((char *) &tmp->srv_addr,sizeof(tmp->srv_addr));
	tmp->srv_addr.sin_family=AF_INET;
	tmp->srv_addr.sin_addr.s_addr=htonl(INADDR_ANY);
	tmp->port=next_port();
	tmp->srv_addr.sin_port=htons(tmp->port);

	while (bind(tmp->tmpfd,(struct sockaddr *)&tmp->srv_addr,sizeof(tmp->srv_addr))<0)
	{
		if (errno!=EADDRINUSE)
		{
			/* Damnit, can't bind local address... */
			if (dcctype&D_SEND) close(tmp->filefd);
			else if (dcctype&D_TAR) pclose(tmp->tarfile);
			close(tmp->tmpfd);
			free(tmp);
			LEAVE(1);
			return;
		}
		else
		{
			tmp->port=next_port();
			tmp->srv_addr.sin_port=htons(tmp->port);
		}
	}

	if (listen(tmp->tmpfd,4)!=0)
	{
		if (dcctype&D_SEND) close(tmp->filefd);
		else if (dcctype&D_TAR) pclose(tmp->tarfile);
		close(tmp->tmpfd);
		free(tmp);
		LEAVE(1);
		return;
	}

	if (fcntl(tmp->tmpfd,F_SETFL,O_NONBLOCK)<0)
	{
		if (dcctype&D_SEND) close(tmp->filefd);
		else if (dcctype&D_TAR) pclose(tmp->tarfile);
		close(tmp->tmpfd);
		free(tmp);
		LEAVE(1);
		return;
	}

	time(&tmp->stime);
	tmp->next=dcclist;
	dcclist=tmp;

#ifdef USE_RATIOS
	/* This is done to avoid "double booked" credits. */
	if (ratio>0)
	{
		cred-=tmp->total;
		send_to_parent(RATIO,"CRED %llu",cred);
	}
#endif

	/* Report the number of open DCCs. */
	send_to_parent(NUM,"%d %d",dcc_count(),queue_count());

	/* Report back the DCC SEND offer. */
	send_to_parent(SEND,"%s %s%s %lu %u %lu",dest,base_name(tmp->fname),tmp->type&D_TAR?".tar":"",local_ip,tmp->port,tmp->total);
	LEAVE(1);
}


/* Function: Sets up a DCC GET connection. */
void process_incoming_file(char *dccinfo)
{
	char *name;
	unsigned long addr,size;
	int port,openflag;
	struct dcc_t *tmp;

	ENTER1(1,"process_incoming_file(\"%s\")\n",dccinfo);

	for (name=dccinfo;(*dccinfo!='\n')&&(*dccinfo!='\0');dccinfo++);
	if (*dccinfo=='\n') *dccinfo='\0';
	
	for (dccinfo=name;*dccinfo!=' ';dccinfo++);
	*dccinfo++='\0';
	
	sscanf(dccinfo,"%lu %u %lu",&addr,&port,&size);

	if (dcc_exists(name))
	{
		send_to_parent(LIST,"? \002%s\002 is already being DCCed. File skipped.",name);
		LEAVE(1);
		return;
	}

	if ((tmp=(struct dcc_t *)malloc(sizeof(struct dcc_t)))==NULL)
	{
		LEAVE(1);
		return;
	}

	/* Initialize the dcc structure. */
	bzero(tmp,sizeof(struct dcc_t));
	strcpy(tmp->fname,name);
	tmp->type=D_GET;
	tmp->active=1; /* Well, these are always active, I suppose. */
	tmp->total=size;
	tmp->transmit=0;
	tmp->bsize=0;
	tmp->buf=NULL;
	
	bzero((char *)&tmp->srv_addr,sizeof(tmp->srv_addr));
	
	tmp->srv_addr.sin_family=AF_INET;
	tmp->srv_addr.sin_addr.s_addr=htonl(addr);
	tmp->srv_addr.sin_port=htons(port);

	if ((tmp->sockfd=socket(AF_INET,SOCK_STREAM,0))<0)
	{
		free(tmp);
		LEAVE(1);
		return;
	}


	/* Open the file... */
	if (cfg.overwrite) openflag=O_WRONLY|O_CREAT|O_TRUNC;
	else openflag=O_WRONLY|O_CREAT|O_EXCL;
	
	if ((tmp->filefd=open(tmp->fname,openflag,0666))==-1)
	{
		close(tmp->sockfd);
		free(tmp);
		LEAVE(1);
		return;
	}

	if (fcntl(tmp->sockfd,F_SETFL,O_NONBLOCK)==-1)
	{
		close(tmp->sockfd);
		close(tmp->filefd);
		free(tmp);
		LEAVE(1);
		return;
	}

	/* Set up the connection */	
	if (connect(tmp->sockfd,(struct sockaddr *)&tmp->srv_addr,sizeof(tmp->srv_addr))<0)
	{
		if (errno!=EINPROGRESS)
		{
			close(tmp->sockfd);
			close(tmp->filefd);
			free(tmp);
			LEAVE(1);
			return;
		}
		else
		{
			tmp->active=0;
		}
	}

	/* Insert this DCC into the list */
	tmp->next=dcclist;
	dcclist=tmp;

	/* Report the number of open DCCs. */
	send_to_parent(NUM,"%d %d",dcc_count(),queue_count());
	LEAVE(1);
}

/* Function: Returns the number of open DCCs. */
int dcc_count()
{
	struct dcc_t *tmp;
	int count;

	ENTER0(3,"dcc_count()\n");
	for (tmp=dcclist,count=0;tmp!=NULL;tmp=tmp->next,count++);
	LEAVE(3);
	return(count);
}


/* Function: Returns the number of open DCCs. */
int queue_count()
{
	struct dccqueue_t *tmp;
	int count;

	ENTER0(3,"queue_count()\n");
	for (tmp=dccqueue,count=0;tmp!=NULL;tmp=tmp->next,count++);
	LEAVE(3);
	return(count);
}

/* Function: Puts a DCC on deathrow for removal. */
void put_on_deathrow(struct dcc_t *convict)
{
	struct deathrow_t *tmp;
	
	ENTER0(2,"put_on_deathrow(...)\n");
	
	if ((tmp=(struct deathrow_t *)malloc(sizeof(struct deathrow_t)))!=NULL)
	{
		bzero(tmp,sizeof(struct deathrow_t));
		tmp->convict=convict;
		tmp->next=deathrow;
		deathrow=tmp;
	}
	
	/* There ain't much we can do if we can't allocate mem */
	LEAVE(2);
}

/* Function: Removes any DCCs on deathrow. */
void clear_deathrow()
{
	struct dcc_t *tmp,*prev;
	struct deathrow_t *conv;
	
	ENTER0(3,"clear_deathrow()\n");
	
	if (deathrow!=NULL)
	{
		while (deathrow!=NULL)
		{
			tmp=deathrow->convict;
			if (tmp->buf!=NULL) free(tmp->buf);
			if (tmp==dcclist)
			{
				dcclist=tmp->next;
				free(tmp);
			}
			else
			{
				prev=dcclist;
				tmp=prev->next;
				while ((tmp!=deathrow->convict)&&(tmp!=NULL))
				{
					prev=tmp;
					tmp=tmp->next;
				}
				if (tmp!=NULL)
				{
					prev->next=tmp->next;
					free(tmp);
				}
			}
			conv=deathrow;
			deathrow=conv->next;
			free(conv);
		}		
		/* Report the number of open DCCs. */
		send_to_parent(NUM,"%d %d",dcc_count(),queue_count());
	}
	LEAVE(3);
}

/* Function: Checks all dccs and do whatever necessary. */
void check_dcc()
{
	fd_set readset;
	struct timeval tval={0,50};
	struct dcc_t *tmp;
	int biggest=0;

	ENTER0(2,"check_dcc()\n");

	FD_ZERO(&readset);

	for (tmp=dcclist;tmp!=NULL;tmp=tmp->next)
	{
		if (tmp->active)
		{
			if (tmp->sockfd>biggest) biggest=tmp->sockfd;
			FD_SET(tmp->sockfd,&readset);
		}
	}


	select(biggest+1,&readset,(fd_set *)0,(fd_set *)0,&tval);

	for (tmp=dcclist;tmp!=NULL;tmp=tmp->next)
	{
		if ((!tmp->active)||FD_ISSET(tmp->sockfd,&readset))
		{
			switch(tmp->type&(~D_RESUME))
			{
			case D_GET:	process_dccget(tmp);
					break;
			case D_SEND:
			case D_TAR:
					process_dccsend(tmp);
					break;
			default:
			}
		}
	}
	/* Let's get rid of some dead weight. */
	check_dcc_timeout();
	clear_deathrow();
	/* Maybe there are some DCCs in the queue. */
	check_dccqueue();
	LEAVE(2);
}

/* Function: Reads next chunk of data from a DCC GET, and writes to file. */
void process_dccget(struct dcc_t *tmp)
{
	unsigned long bytesread;
	unsigned long bytestmp;
	char tmpbuff[BIG_BUFFER_SIZE+1];
		
	ENTER0(2,"process_dccget(...)\n");

	if (tmp->active)
	{
		if ((bytesread=read(tmp->sockfd,tmpbuff,BIG_BUFFER_SIZE))<=0)
		{
#ifdef USE_RATIOS
			if (tmp->total==tmp->transmit)
			{
				ul+=tmp->total;
				cred+=tmp->total*ratio;
				send_to_parent(RATIO,"UL %llu",ul);
				send_to_parent(RATIO,"CRED %llu",cred);
			}
#endif
			close(tmp->filefd);
			close(tmp->sockfd);
			put_on_deathrow(tmp);
		}
		write(tmp->filefd,tmpbuff,bytesread);
		tmp->transmit+=bytesread;
		bytestmp=htonl(tmp->transmit);
		write(tmp->sockfd,&bytestmp,sizeof(bytestmp));
	}
	else if (connect(tmp->sockfd,(struct sockaddr *)&tmp->srv_addr,sizeof(tmp->srv_addr))<0&&errno!=EALREADY)
	{
		if (errno!=EINPROGRESS)
		{
			close(tmp->sockfd);
			close(tmp->filefd);
			put_on_deathrow(tmp);
			LEAVE(2);
			return;
		}
	}
	else
	{
		tmp->active=1;
		time(&tmp->stime);
	}
	LEAVE(2);
}


/* Function: Attends to an open DCC SEND and does what's needed. */
void process_dccsend(struct dcc_t *tmp)
{
	unsigned long ack=0;
	int bytesread,written;
	
	ENTER0(2,"process_dccsend(...)\n");

	if (!tmp->active)
	{
		if ((tmp->sockfd=accept(tmp->tmpfd,(struct sockaddr *) &tmp->cli_addr,&tmp->clilen))==-1)
		{
			if (errno!=EWOULDBLOCK)
			{
#ifdef USE_RATIOS
				if (ratio>0)
				{
					cred+=tmp->type&D_RESUME?tmp->resume:tmp->total;
					send_to_parent(RATIO,"CRED %llu",cred);
				}
#endif
				if (tmp->type&D_SEND) close(tmp->filefd);
				else if (tmp->type&D_TAR) pclose(tmp->tarfile);
				close(tmp->tmpfd);
				put_on_deathrow(tmp);
			}
			LEAVE(2);
			return;						
		}
		fcntl(tmp->sockfd,F_SETFL,O_NONBLOCK);
		tmp->active=1;
		tmp->remains=0;
		if (tmp->type&D_RESUME) ack=tmp->transmit;
		close(tmp->tmpfd);
		/* Reset start time for measurement purposes. */
		time(&tmp->stime);
	}
	else 
	if ((read(tmp->sockfd,&ack,sizeof(ack))<sizeof(ack))||(tmp->type&D_RESUME)?(ntohl(ack)==tmp->resume):(ntohl(ack)==tmp->total))
	{
#ifdef USE_RATIOS
		if (tmp->type&D_RESUME?ntohl(ack)==tmp->resume:ntohl(ack)==tmp->total)
		{
			dl+=ntohl(ack);
			send_to_parent(RATIO,"DL %llu",dl);
		}
		else if (ratio>0)
		{
			cred+=tmp->type&D_RESUME?tmp->resume:tmp->total;
			send_to_parent(RATIO,"CRED %llu",cred);
		}
#endif
		close(tmp->sockfd);
		if (tmp->type&D_SEND) close(tmp->filefd);
		else if (tmp->type&D_TAR) fclose(tmp->tarfile);
		put_on_deathrow(tmp);
		LEAVE(2);
		return;
	}
	if (ntohl(ack)<tmp->transmit) return;

	if (tmp->type&D_SEND)
	{
		bytesread=read(tmp->filefd,&tmp->buf[tmp->remains],tmp->bsize-tmp->remains);
	}

	if (tmp->type&D_TAR)
	{
		bytesread=fread(&tmp->buf[tmp->remains],1,tmp->bsize-tmp->remains,tmp->tarfile);
	}

	tmp->transmit+=bytesread;
	written=write(tmp->sockfd,tmp->buf,tmp->remains==tmp->bsize?tmp->bsize:bytesread);
	tmp->remains=bytesread-written;
	memmove(tmp->buf,&tmp->buf[written],tmp->remains);

	LEAVE(2);	
}

/* Function: Sends a DCC offer to a client. */
void send_dcc_offer(char *dccinfo)
{
	char *dest;

	/*ENTER1(1,"send_dcc_offer(\"%s\")\n",dccinfo);*/

	for (dest=dccinfo;*dccinfo!=' ';dccinfo++);
	*dccinfo++='\0';
	send_to_server("PRIVMSG %s :\001DCC SEND %s\001\n",dest,dccinfo);
	/*LEAVE(1);*/
}

/* Function: Checks for timed out DCCs and if necessary, put's them to death. */
void check_dcc_timeout()
{
	struct dcc_t *tmp;
	
	ENTER0(3,"check_dcc_timeout()\n");
	for (tmp=dcclist;tmp!=NULL;tmp=tmp->next)
	{
		if ((!tmp->active)&&(time(NULL)>(tmp->stime+cfg.dcctimeout)))
		{
			close(tmp->tmpfd);
			if (tmp->type&D_SEND) close(tmp->filefd);
			else if (tmp->type&D_TAR) pclose(tmp->tarfile);
			put_on_deathrow(tmp);
		}
	}
	LEAVE(3);
}

/* Function: Puts a file on the DCC queue. */
void put_on_dccqueue(int type,char *dccinfo)
{
	struct dccqueue_t *tmp,*p;
	char sometext[256],*word,*dest;

	ENTER2(1,"put_on_dccqueue(%d,\"%s\")\n",type,dccinfo);

	strcpy(sometext,dccinfo);

	dest=type&D_GET?"?":strtok(sometext," ");
	word=type&D_GET?strtok(sometext," "):strtok(NULL," ");

	if (is_in_queue(word)||dcc_exists(word))
	{
		send_to_parent(LIST,"%s \002%s%s\002 is already being DCCed or is in queue.",dest,base_name(word),(type==D_TAR)?".tar":"");
		LEAVE(1);
		return;
	}

	if ((tmp=(struct dccqueue_t *)malloc(sizeof(struct dccqueue_t)))==NULL)
	{
		LEAVE(1);
		return;
	}
	bzero(tmp,sizeof(struct dccqueue_t));
	tmp->type=type;
	strcpy(tmp->info,dccinfo);
	tmp->next=NULL;
	
	if (dccqueue==NULL)
	{
		dccqueue=tmp;
	}
	else
	{
		for (p=dccqueue;p->next!=NULL;p=p->next);
		p->next=tmp;
	}
	/* Report the number of open DCCs. */
	send_to_parent(NUM,"%d %d",dcc_count(),queue_count());
	send_to_parent(LIST,"%s \002%s%s\002 has been put on DCC queue.",dest,base_name(word),(type&D_TAR)?".tar":"");
	LEAVE(1);
}

/* Function: Checks the DCC queue for DCCs to be processed. */
void check_dccqueue()
{
	struct dccqueue_t *tmp;
	static int counter;

	/* It's a waste of time checking this too often */
	if ((!autodcc)||(counter++%50)||(dccqueue==NULL)) return;
	
	ENTER0(1,"check_dccqueue()\n");

	for (tmp=dccqueue;(tmp!=NULL)&&(dcc_count()<dcclimit);tmp=dccqueue)
	{
		switch(tmp->type)
		{
		case D_GET:	process_incoming_file(tmp->info);
				break;
		case D_SEND:
		case D_TAR:	process_outgoing_file(tmp->info,tmp->type);
				break;
		default:
		}
		dccqueue=tmp->next;
		free(tmp);
		send_to_parent(NUM,"%d %d",dcc_count(),queue_count());
	}
	LEAVE(1);
}

/* Functions: Displays a list of the files being DCCed. */
void show_dcc_list(char *msgtext)
{
	int index;
	struct dcc_t *tmp;
	char typetxt[10];
	char speed[10];
	char stat[10];
	
	ENTER1(1,"show_dcc_list(\"%s\")\n",msgtext);

	if (dcclist==NULL)
	{
		send_to_parent(LIST,"%s No open DCCs.",msgtext);
		LEAVE(1);
		return;
	}
	else
	{
		send_to_parent(LIST,"%s No. Type      Size  Prc.    K/s  Stat  Name",msgtext);
	}

	for (tmp=dcclist,index=1;tmp!=NULL;tmp=tmp->next,index++)
	{
		switch(tmp->type&(~D_RESUME))
		{
		case D_GET:	strcpy(typetxt,"GET");
				break;
		case D_SEND:	strcpy(typetxt,"SEND");
				break;
		case D_TAR:	strcpy(typetxt,"TAR");
				break;
		default:	strcpy(typetxt,"???");
		}
		sprintf(stat,tmp->active?"Actv":"Wait");
		if (tmp->transmit==0)
		{
			sprintf(speed,"  N/A ");
		}
		else	
		{
			sprintf(speed,"%3.2f",((float)tmp->transmit)/(float)(time(NULL)-(tmp->stime))/1024);
		}
		send_to_parent(LIST,"%s #%-2d %-4s %9lu  %3d%c  %-6s %s  \002%s%s\002",
			msgtext,index,typetxt,tmp->total,
			(int)(((unsigned long)100*(unsigned long)tmp->transmit)/(unsigned long)(tmp->type&D_RESUME?tmp->resume:tmp->total)),
			'%',speed,stat,base_name(tmp->fname),tmp->type&D_TAR?".tar":"");
	}
	send_to_parent(LIST,"%s End of DCC listing.",msgtext);
	LEAVE(1);
}

/* Function: Checks to see if a file is being DCCed. */
int dcc_exists(char *filename)
{
	struct dcc_t *p;

	ENTER1(2,"dcc_exists(\"%s\")\n",filename);

	for (p=dcclist;(p!=NULL)&&(strcmp(p->fname,filename)!=0);p=p->next);
	if (p!=NULL)
	{
		LEAVE(2);
		return(1);
	}
	else
	{
		LEAVE(2);
		return(0);
	}
}

/* Function: Checks to see if a file is in the DCC queue. */
int is_in_queue(char *filename)
{
	struct dccqueue_t *p;
	char sometext[256],*word;

	ENTER1(2,"is_in_queue(\"%s\")\n",filename);

	for (p=dccqueue;p!=NULL;p=p->next)
	{
		strcpy(sometext,p->info);
		word=strtok(sometext," ");
		if (p->type&D_GET&&strcmp(word,filename)==0)
		{
			LEAVE(2);
			return(1);
		}
		else if (strcmp(strtok(NULL," "),filename)==0)
		{
			LEAVE(2);
			return(1);
		}
	}
	LEAVE(2);
	return(0);
}

/* Function: Displays a list of the files in the DCC queue. */
void show_dcc_queue(char *msgtext)
{
	int idx;
	struct dccqueue_t *dq;
	char typetxt[10],sometext[256],*first;

	ENTER1(1,"show_dcc_queue(\"%s\")\n",msgtext);
	
	if (dccqueue==NULL)
	{
		send_to_parent(QLIST,"%s No files enqueued.",msgtext);
	}
	else
	{
		send_to_parent(QLIST,"%s No. Type   Name",msgtext);
		for (idx=1,dq=dccqueue;dq!=NULL;dq=dq->next,idx++)
		{
			switch (dq->type)
			{
			case D_GET:	strcpy(typetxt,"GET");
					break;
			case D_SEND:	strcpy(typetxt,"SEND");
					break;
			case D_TAR:	strcpy(typetxt,"TAR");
					break;
			default:	strcpy(typetxt,"???");
			}
			strcpy(sometext,dq->info);
			first=strtok(sometext," ");
			send_to_parent(QLIST,"%s #%-2d %-5s \002%s%s\002",msgtext,idx,typetxt,base_name(dq->type&D_GET?first:strtok(NULL," ")),(dq->type==D_TAR)?".tar":"");
		}
		send_to_parent(QLIST,"%s End of list.",msgtext);
	}
	LEAVE(1);
}


/* Function: Removes DCCs from the DCC list. */
void remove_dcc(char *args)
{
	char *dest,*num;
	int inum,idx;
	struct dcc_t *p;

	ENTER1(1,"remove_dcc(\"%s\")\n",args);
	
	dest=strtok(args," ");
	num=strtok(NULL," ");
	
	if (strcasecmp(num++,"#all")!=0)
	{
		sscanf(num,"%d",&inum);
	
		for (idx=1,p=dcclist;(p!=NULL)&&(idx!=inum);p=p->next,idx++);
		if ((p!=NULL)&&(idx==inum))
		{
			switch(p->type&(~D_RESUME))
			{
			case D_SEND:
			case D_TAR:	send_to_parent(LIST,"%s Closed DCC SEND \002%s%s\002 at \002%lu\002 bytes.",dest,base_name(p->fname),p->type&D_TAR?".tar":"",p->transmit);
					if (!p->active) close(p->tmpfd);
					else close(p->sockfd);
					if (p->type&D_SEND) close(p->filefd);
					else if (p->type&D_TAR) pclose(p->tarfile);
					break;
			case D_GET:	send_to_parent(LIST,"%s Closed DCC GET \002%s\002 at \002%lu\002 bytes.",dest,base_name(p->fname),p->transmit);
					close(p->sockfd);
					close(p->filefd);
					break;
			default:	send_to_parent(LIST,"%s Closed DCC ??? \002%s\002 at \002%lu\002 bytes.",dest,p->fname,p->transmit);
			}
#ifdef USE_RATIOS
			if (p->type&D_RESUME?p->transmit<p->resume:p->transmit<p->total&&ratio>0)
			{
				cred+=p->type&D_RESUME?p->resume:p->total;
				send_to_parent(RATIO,"CRED %llu",cred);
			}
#endif
			put_on_deathrow(p);
		}
	}
	else
	{
		for (p=dcclist;p!=NULL;p=p->next)
		{
			switch(p->type&(~D_RESUME))
			{
			case D_SEND:
			case D_TAR:	if (!p->active) close(p->tmpfd);
					else close(p->sockfd);
					if (p->type&D_SEND) close(p->filefd);
					else if (p->type&D_TAR) pclose(p->tarfile);
					break;
			case D_GET:	close(p->sockfd);
					close(p->filefd);
					break;
			default:
			}
#ifdef USE_RATIOS
			if (p->type&D_RESUME?p->transmit<p->resume:p->transmit<p->total&&ratio>0)
			{
				cred+=p->type&D_RESUME?p->resume:p->total;
				send_to_parent(RATIO,"CRED %llu",cred);
			}
#endif
			put_on_deathrow(p);
			clear_deathrow();
		}
		send_to_parent(LIST,"%s All DCCs closed.",dest);
	}
	clear_deathrow();
	LEAVE(1);
}

/* Function: Removes files from the DCC queue. */
void remove_queued(char *msgtext)
{
	int num,idx;
	char *dest,*arg;
	struct dccqueue_t *p,*q;
	
	ENTER1(1,"remove_queued(\"%s\")\n",msgtext);

	dest=strtok(msgtext," ");
	arg=strtok(NULL," ");
	
	if (strcasecmp(arg++,"#all"))
	{

		/* One file */
		if ((num=atoi(arg))==0)
		{
			send_to_parent(LIST,"%s Unable to remove item.",dest);
			return;
		}
		if (!queue_count())
		{
			send_to_parent(LIST,"%s Queue is already empty.",dest);
			return;
		}		
		if (num==1)
		{
			p=dccqueue;
			dccqueue=p->next;
			free(p);
			send_to_parent(LIST,"%s Item \002%d\002 has been dequeued.",dest,num);
		}
		else
		{
			for (idx=2,q=dccqueue,p=q->next;(p!=NULL)&&(idx<num);q=p,p=p->next,idx++);
			if (p!=NULL)
			{
				q->next=p->next;
				free(p);
				send_to_parent(LIST,"%s Item \002%d\002 has been dequeued.",dest,num);
			}
			else
			{
				send_to_parent(LIST,"%s Unable to remove item.",dest);
			}
		}
	}
	else
	{
		/* All files */
		for (p=dccqueue;p!=NULL;p=dccqueue)
		{
			dccqueue=p->next;
			free(p);
		}
		send_to_parent(LIST,"%s All files removed from queue.",dest);
	}
	send_to_parent(NUM,"%d %d",dcc_count(),queue_count());
	LEAVE(1);
}


/* Function: Pulls files from the dcc queue. */
void process_next(char *msgtext)
{
	int cnt,diff,idx;
	struct dccqueue_t *tmp;
	
	sscanf(msgtext,"%d",&cnt);
	diff=dcclimit-dcc_count();
	if (diff<1)
	{
		send_to_parent(NEXT,"? No files pulled from DCC queue.");
		return;
	}
	if (cnt>diff) cnt=diff;
	for (tmp=dccqueue,idx=cnt;(tmp!=NULL)&&(idx>0);tmp=dccqueue,idx--)
	{
		switch(tmp->type)
		{
		case D_GET:	process_incoming_file(tmp->info);
				break;
		case D_SEND:
		case D_TAR:	process_outgoing_file(tmp->info,tmp->type);
				break;
		default:
		}
		dccqueue=tmp->next;
		free(tmp);
	}
	send_to_parent(NEXT,"? \002%d\002 file%s pulled from DCC queue.",cnt-idx,(cnt-idx)>1?"s":"");
}

/* Function: Parses a mIRC Resume request, and notifies child. */
void parse_resume_request(char *orig,char *args)
{
	struct user_t *tmp;
	struct chat_t *ctmp;
	char filename[100];
	int port;
	unsigned long pos;

	ENTER2(1,"parse_resume_request(\"%s\",\"%s\")\n",orig,args);

	if ((tmp=get_user(getnick(orig),&users))==NULL||(ctmp=find_chat(tmp,&chatlist))==NULL)
	{
		send_to_server("NOTICE %s :\001ERRMSG Bogus resume request detected.\001\n",getnick(orig));
		LEAVE(1);
		return;
	}
	sscanf(args,"%s %d %lu",filename,&port,&pos);
	send_to_child(ctmp,RESUME,"%s %d %lu",filename,port,pos);
}

/* Function: Searches for resumable DCC and sends out acknowledgement. */
void process_resume(char *msg)
{
	char filename[100],tarbuf[10240];
	struct dcc_t *p;
	int port;
	unsigned long pos,bytes2read;
	
	sscanf(msg,"%s %d %lu",filename,&port,&pos);
	
	for (p=dcclist;p!=NULL&&p->port!=port;p=p->next);
	if (p!=NULL&&!p->active&&pos<p->total)
	{
		p->resume=p->total;
		p->total-=pos;
		p->transmit=pos;
		if (p->type&D_SEND)
		{
			lseek(p->filefd,pos,SEEK_SET);
		}
		else if (p->type&D_TAR)
		{
			/* This is a clumsy & slow way to seek through a */
			/* tar file. If anyone got a better suggestion, */
			/* please tell me all about it. */
			do
			{	
				bytes2read=(pos<10240)?pos:10240;
				pos-=bytes2read;
				fread((void *)tarbuf,(size_t)1,(size_t)bytes2read,p->tarfile);
			}
			while(pos);
		}
		send_to_parent(RESUME,"%s",msg);
		p->type|=D_RESUME;
		p->stime=time(NULL); /* Reset time to avoid premature removal. */
	}
}
