
#include <config.h>
#include <janbot.h>


/* Function: Adds a user to userlist, returns pointer to new user. */
struct user_t *add_user(char *nickname,struct user_t **ulist)
{
	struct user_t *tmp,*p,*prev;

	if (!isvalidnick(nickname)) return(NULL);
	if ((tmp=(struct user_t *)malloc(sizeof(struct user_t)))!=NULL)
	{
		bzero((char *)tmp,sizeof(struct user_t));
		strcpy(tmp->nick,nickname);
		prev=*ulist;
		if ((prev==NULL)||(strcasecmp(nickname,prev->nick)<=0))
		/* User list is empty, insert user at start. */
		{
			tmp->next=*ulist;
			*ulist=tmp;
		}
		else
		/* Userlist contains users, let's find the correct place. */
		{
			p=prev->next;
			while ((p!=NULL)&&(strcasecmp(nickname,p->nick)>0))
			{
				prev=p;
				p=prev->next;
			}
			tmp->next=p;
			prev->next=tmp;
		}
	}				

	/* Let's put in some default values. */
	tmp->dcclimit=cfg.dcclimit;
	tmp->flags=cfg.defaultflags;
	tmp->userhost=NULL;
	tmp->ratio=cfg.defaultratio;
	strcpy(tmp->dir,"/");

	return(tmp);
}
	

/* Function: Finds a user in userlist, returns pointer to user. */
struct user_t *get_user(char *nickname,struct user_t **ulist)
{
	struct user_t *tmp;
	tmp=*ulist;
	/* I could make this recursive. I love recursion. But I won't bother. */
	while ((tmp!=NULL)&&(strcasecmp(nickname,tmp->nick)!=0))
 	{
		tmp=tmp->next;
	}
	/* If we found a user, that's great, otherwise NULL is returned. */
	return(tmp);
}


/* Function: Empties the userlist. */
void free_users(struct user_t **ulist)
{
	struct user_t *p,*tmp;
	struct userhost_t *q,*r;
	p=*ulist;
	while (p!=NULL)
	{
	/* Why do I bother making these comments? Nobody reads them... */
		tmp=p->next;
		q=p->userhost;
		while(q!=NULL)
		{
			r=q->next;
			free(q);
			q=r;
		}
		free(p);
		p=tmp;
	}
	*ulist=NULL;
}


/* Function: Deletes a user from userlist. */
void del_user(char *nickname,struct user_t **ulist)
{
	struct user_t *p,*tmp;
	struct userhost_t *q,*r;
	p=*ulist;
	if ((tmp=get_user(nickname,ulist))!=NULL)
	{
		if (*ulist==tmp)
		{
			(*ulist)=tmp->next;
		}
		else
		{
			while (p->next!=tmp)
			{
				p=p->next;
			}
			p->next=tmp->next;
		}
		/* Next time I'll use C++, I swear... */
		for (q=tmp->userhost;q!=NULL;r=q->next,free(q),q=r);
		free(tmp);
	}	
}
	
/* Function: Opens userlist file. */
int initusers()
{
	if ((userfd=open(USERFILE,O_RDWR|O_CREAT,0600))==-1)
	{
		/* Unable to open file... */
		DEBUG1("WARNING: Unable to open userfile: %s\n",cfg.userfile);
		return(0);
	}
	DEBUG1("Opening userfile: %s\n",cfg.userfile);
	return(1);
}


/* Function: Reads all users from userfile. */
int read_users(struct user_t **ulist)
{
	struct user_t *tmp;
	int ucount=0;
	char buf[256],*tag,*val,*end;
	
	lseek(userfd,0,SEEK_SET);
	DEBUG0("Loading users... ");

	while(read_line(userfd,buf))
	{
		for (tag=buf;*tag==' ';tag++);
		for (val=tag;*val!=' ';val++);
		*val++='\0';
		for (;*val==' ';val++);
		for (end=val;*end!='\0';end++);
		for (end--;*end==' ';end--);
		*(++end)='\0';

		if (!strcasecmp("NICK",tag))
		{
			tmp=add_user(val,&users);
			ucount++;
		}
		else if (!strcasecmp("PASSWD",tag))
		{
			strcpy(tmp->passwd,val);
		}
		else if (!strcasecmp("LEVEL",tag))
		{
			sscanf(val,"%d",(int *)&tmp->level);
		}
		else if (!strcasecmp("LIMIT",tag))
		{
			sscanf(val,"%d",(int *)&tmp->dcclimit);
		}
		else if (!strcasecmp("DIR",tag))
		{
			strcpy(tmp->dir,val);
		}
		else if (!strcasecmp("FLAGS",tag))
		{
			sscanf(val,"%d",(int *)&tmp->flags);
		}
		else if (!strcasecmp("RATIO",tag))
		{
			sscanf(val,"%d",(int *)&tmp->ratio);
		}
		else if (!strcasecmp("UPLOAD",tag))
		{
			sscanf(val,"%llu",(unsigned long long*)&tmp->ul);
		}
		else if (!strcasecmp("DOWNLOAD",tag))
		{
			sscanf(val,"%llu",(unsigned long long*)&tmp->dl);
		}
		else if (!strcasecmp("CREDITS",tag))
		{
			sscanf(val,"%llu",(unsigned long long*)&tmp->cr);
		}
		else if (!strcasecmp("HOST",tag))
		{
			add_userhost(tmp->nick,val,ulist);
		}
	}
	DEBUG2("%d user%s loaded.\n",ucount,ucount==1?"":"s");
	return(ucount);
}


/* Function: Calls write_users() as scheduled. */
void write_scheduled(struct user_t **ulist)
{
	static time_t t;
	static int writepid=0;
	int idx;

	if (writepid)
	{
		if (waitpid(writepid,NULL,WNOHANG)>0)
		{
			writepid=0;
		}
		return;
	}
	if (t==0) t=time(NULL);
	if ((t+cfg.scheduledwrite)<=time(NULL))
	{
		if (!(writepid=fork()))
		{
			for (idx=0;idx<p_argc;idx++)
			{
				bzero(p_argv[idx],strlen(p_argv[idx]));
			}
			sprintf(p_argv[0],"%s%s",cfg.nick,USERLIST_CHILD);
			write_users(ulist);
			exit(0);
		}
		t=time(NULL);
	}
}

/* Function: Writes the userlist to the ascii user file. */
void write_users(struct user_t **ulist)
{
	int ucount=0;
#ifdef DEBUG
	time_t start=time(NULL);
#endif
	char buf[256];
	char big[2048];
	struct user_t *tmp;
	struct userhost_t *p;

	lseek(userfd,0,SEEK_SET);
	ftruncate(userfd,0);

	for (tmp=*ulist;tmp!=NULL;tmp=tmp->next)
	{
		sprintf(buf,"NICK %s\n",tmp->nick);
		strcpy(big,buf);
		sprintf(buf,"PASSWD %s\n",tmp->passwd);
		strcat(big,buf);
		sprintf(buf,"LEVEL %d\n",tmp->level);
		strcat(big,buf);
		sprintf(buf,"LIMIT %d\n",tmp->dcclimit);
		strcat(big,buf);
		sprintf(buf,"DIR %s\n",tmp->dir);
		strcat(big,buf);
		sprintf(buf,"FLAGS %d\n",tmp->flags);
		strcat(big,buf);
		sprintf(buf,"RATIO %d\n",tmp->ratio);
		strcat(big,buf);
		sprintf(buf,"UPLOAD %llu\n",tmp->ul);
		strcat(big,buf);
		sprintf(buf,"DOWNLOAD %llu\n",tmp->dl);
		strcat(big,buf);
		sprintf(buf,"CREDITS %llu\n",tmp->cr);
		strcat(big,buf);
		for (p=tmp->userhost;p!=NULL;p=p->next)
		{
			sprintf(buf,"HOST %s\n",p->uh);
			strcat(big,buf);
		}			
		write(userfd,big,strlen(big));
		ucount++;
	}
	DEBUG2("Wrote %d users to userfile in %d seconds.\n",ucount,(int)(time(NULL)-start));
}
	
/* Function: Adds a user@host mask to a user. */
void add_userhost(char *nick,char *hostmask,struct user_t **ulist)
{
	struct user_t *tmp;
	struct userhost_t *u,*p;
	
	if ((tmp=get_user(nick,ulist))!=NULL)
	{
		if ((u=(struct userhost_t *)malloc(sizeof(struct userhost_t)))==NULL) return;
		strncpy(u->uh,hostmask,USERHOST_LEN-1);
		u->next=NULL;
		if (tmp->userhost==NULL)
		{
			tmp->userhost=u;
		}
		else
		{
			for (p=tmp->userhost;p->next!=NULL;p=p->next);
			p->next=u;
		}
	}
}

/* Function: Removes a userhost mask from a user. */
void del_userhost(char *nick,int num,struct user_t **ulist)
{
	struct user_t *tmp;
	struct userhost_t *p,*q;
	int index;	

	/* This might get nasty if we have a lot of hosts, but for now... */

	if ((tmp=get_user(nick,ulist))==NULL) return;
	if ((p=tmp->userhost)==NULL) return;
	if (num==0)
	{
		tmp->userhost=p->next;
		free(p);
	}
	for (index=1;index<num;index++,p=p->next);
	q=p->next;
	p->next=q->next;
	free(q);
}

/* Function: Returns the number of registered userhost masks for a user. */
int userhost_count(struct userhost_t *p)
{
	if (p==NULL) return(0);
	else return (1+userhost_count(p->next));
}
	
/* Function: Authenticates a user by nick+userhost+passwd */
int authenticate(char *a_nick,char *a_userhost,char *a_passwd,struct user_t **ulist)
{
	struct user_t *usr;
	struct userhost_t *p;
	int matched=0;
	
	if ((usr=get_user(a_nick,ulist))==NULL)
	{
		/* No user matching the nick */
		DEBUG1("Auth error: No such nick: %s\n",a_nick);
		return(1);
	}


	/* From the irc2.9p1 server code:
	**
	** ident is fun.. ahem
	** prefixes used:
	**	none	I line with ident
	**	^	I line with OTHER type ident
	**	~	I line, no ident
	**	+	i line with ident
	**	=	i line with OTHER type ident
	**	-	i line, no ident
	*/

	if (*a_userhost=='^') a_userhost++;
	else if (*a_userhost=='~') a_userhost++;
	else if (*a_userhost=='+') a_userhost++;
	else if (*a_userhost=='=') a_userhost++;
	else if (*a_userhost=='-') a_userhost++;
	
	DEBUG1("Checking host: %s\n",a_userhost);
	p=usr->userhost;
	while(p!=NULL)
	{
		if (regex_match(p->uh,a_userhost))
		{
			matched=1;
		}
		p=p->next;
	}
	if (!matched)
	{
		/* No userhost matching of this user. */
		DEBUG0("Auth error: No matching userhost.\n");
		return(1);
	}

	if (chkpasswd(a_nick,a_passwd,ulist)!=0)
	{
		/* Password is incorrect. */
		DEBUG0("Auth error: Password mismatch.\n");
		return(1);
	}

	DEBUG1("Auth: %s authenticated.\n",usr->nick);
	return(0);	
}


/* Function: Sets a password for a user. */
int setpasswd(char *nick,char *passwd,struct user_t **ulist)
{
	struct user_t *usr;
	static char *saltlist="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789./";
	char newsalt[3]={0,0,0};
	
	if ((usr=get_user(nick,ulist))==NULL)
	{
		/* Oops, no such user. */
		return(1);
	}

	newsalt[0]=saltlist[rand()%strlen(saltlist)];
	newsalt[1]=saltlist[rand()%strlen(saltlist)];
	
	strncpy(usr->passwd,crypt(passwd,newsalt),13);
	
	return(0);	
}

/* Function: Validates a user's passwd */
int chkpasswd(char *nick,char *passwd,struct user_t **ulist)
{
	struct user_t *usr;
	char newsalt[3]={0,0,0};
	
	
	if ((usr=get_user(nick,ulist))==NULL)
	{
		/* No such luck. User does not exist. */
		return(1);
	}
	
	/* Get the salt encryption code. */
	strncpy(newsalt,usr->passwd,2);
	
	if (strcmp(crypt(passwd,newsalt),usr->passwd))
	{
		/* Encrypted strings doesn't match. This is not the passwd. */
		return(1);
	}
	return(0);
}

int setlevel(char *nick,int lvl,struct user_t **ulist)
{
	struct user_t *tmp;
	if ((tmp=get_user(nick,ulist))!=NULL)
	{
		tmp->level=lvl;
		return(0);
	}
	return(1);
}

int setlimit(char *nick,int lim,struct user_t **ulist)
{
	struct user_t *tmp;
	if ((tmp=get_user(nick,ulist))!=NULL)
	{
		tmp->dcclimit=lim;
		return(0);
	}
	return(1);
}

void parse_userhost_reply(char *args)
{

	struct chat_t *usr;
	struct user_t *tmp;
	char *n,*u,*h,pw[10];
	char nh[50]; /* I've seen some really long hostnames in my life. */

	if ((usr=find_chat(uhquery.caller,&chatlist))==NULL)
	{
		/* We won't do anything unless the caller is connected. */
		return;
	}
	
	if (strlen(args)==0)
	{
		tell_user(usr,"Cannot find \002%s\002 on iRC!\n",uhquery.nick);
		return;
	}

	n=strtok(args,"=");
	if (n[strlen(n)-1]=='*') n[strlen(n)-1]='\0'; 
	u=strtok(NULL,"@")+1;
	h=strtok(NULL,"@");

	/* Strip the ident characters. Fixed to 2.9p1 compliance. */
	if (*u=='^') u++;
	else if (*u=='~') u++;
	else if (*u=='+') u++;
	else if (*u=='=') u++;
	else if (*u=='-') u++;

	strcpy(nh,u);
	strcat(nh,"@");
	strcat(nh,create_hostmask(h));

	/* This will be the temporary password. */
	strcpy(pw,n);
	for (u=pw;*u!='\0';u++) *u=tolower(*u);

	if ((tmp=get_user(n,&users))!=NULL)
	{
		add_userhost(n,nh,&users);
		tell_user(usr,"Added hostmask \002%s\002 to user \002%s\002.\n",nh,tmp->nick);
	}
	else
	{
		if ((tmp=add_user(n,&users))!=NULL)
		{
			add_userhost(n,nh,&users);
			setpasswd(n,pw,&users);
			tell_user(usr,"Added new user \002%s\002 with hostmask \002%s\002.\n",n,nh);
			send_to_server("NOTICE %s :Hi there! You have just been added to my userlist\n",n);
			send_to_server("NOTICE %s :with the hostmask \002%s\002. To connect you must\n",n,nh);
			send_to_server("NOTICE %s :always use your current nick, issue the command\n",n);
			send_to_server("NOTICE %s :'/CTCP %s AUTH \002%s\002', and use the chat to play.\n",n,cfg.nick,pw);
			send_to_server("NOTICE %s :\002%s\002 is your password, and it can be changed later.\n",n,pw);
		}
		else
		{
			tell_user(usr,"Unable to add new user!\n");
		}
	}	
}

/* Function: Reads userdata from keyboard, primarily to set up first user. */
void enter_new_user()
{
	struct user_t *tmp;
	int lvl;
	char *pw,buf[MSG_LEN+1];

	do
	{
		printf("Enter nickname: ");
		(void)scanf("%s",buf);
		if (strlen(buf)>9)
		{
			printf("Invalid nickname!\n");
		}
	}
	while (strlen(buf)>9);

	if ((tmp=get_user(buf,&users))!=NULL)
	{
		fprintf(stderr,"\002%s\002 is already registered.\n",tmp->nick);
		return;
	}
	
	if ((tmp=add_user(buf,&users))==NULL)
	{
		fprintf(stderr,"Unable to add user.\n");
		return;
	}
	
	
	printf("Enter user@host mask: ");
	(void)scanf("%s",buf);
	add_userhost(tmp->nick,buf,&users);

	printf("Enter user level: ");
	(void)scanf("%d",&lvl);
	setlevel(tmp->nick,lvl,&users);
	
	pw=getpass("Enter password: ");
	setpasswd(tmp->nick,pw,&users);
	bzero(pw,strlen(pw));
	write_users(&users);
}
