//   $Id: kvidns.cpp,v 1.2 1998/10/06 14:41:55 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#ifdef HAVE_CONFIG_H
	#include <config.h>
#endif

#include <stdarg.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
//required by BSD
#include <netinet/in.h>

#include "kvidns.h"

#ifdef __KVI_NEED_MY_INET_ATON__

// 
// Check whether "cp" is a valid ascii representation
// of an Internet address and convert to a binary address.
// Returns 1 if the address is valid, 0 if not.
// This replaces inet_addr, the return value from which
// cannot distinguish between failure and a local broadcast address.
// Taken from ircd source.
//

int my_inet_aton(register const char *cp, struct in_addr *addr)
{
	register unsigned long val;
	register int base, n;
	register char c;
	unsigned int parts[4];
	register unsigned int *pp = parts;

	c = *cp;
	for (;;) {
		//
		// Collect number up to ``.''.
		// Values are specified as for C:
		// 0x=hex, 0=octal, isdigit=decimal.
		//
		if (!isdigit(c))
			return (0);
		val = 0; base = 10;
		if (c == '0') {
			c = *++cp;
			if (c == 'x' || c == 'X')
				base = 16, c = *++cp;
			else
				base = 8;
		}
		for (;;) {
			if (isascii(c) && isdigit(c)) {
				val = (val * base) + (c - '0');
				c = *++cp;
			} else if (base == 16 && isascii(c) && isxdigit(c)) {
				val = (val << 4) |
					(c + 10 - (islower(c) ? 'a' : 'A'));
				c = *++cp;
			} else
				break;
		}
		if (c == '.') {
			//
			// Internet format:
			//	a.b.c.d
			//	a.b.c	(with c treated as 16 bits)
			//	a.b	(with b treated as 24 bits)
			//
			if (pp >= parts + 3)
				return (0);
			*pp++ = val;
			c = *++cp;
		} else
			break;
	}
	// Check for trailing characters.
	if (c != '\0' && (!isascii(c) || !isspace(c)))
		return (0);
	//
	// Concoct the address according to
	// the number of parts specified.
	//
	n = pp - parts + 1;
	switch (n) {

	case 0:
		return (0);		/* initial nondigit */

	case 1:				/* a -- 32 bits */
		break;

	case 2:				/* a.b -- 8.24 bits */
		if (val > 0xffffff)
			return (0);
		val |= parts[0] << 24;
		break;

	case 3:				/* a.b.c -- 8.8.16 bits */
		if (val > 0xffff)
			return (0);
		val |= (parts[0] << 24) | (parts[1] << 16);
		break;

	case 4:				/* a.b.c.d -- 8.8.8.8 bits */
		if (val > 0xff)
			return (0);
		val |= (parts[0] << 24) | (parts[1] << 16) | (parts[2] << 8);
		break;
	}
	if (addr)
		addr->s_addr = htonl(val);
	return (1);
}

#endif //__KVI_NEED_INET_ATON__

void writeError(const char *query,const char *errStr)
{
	fprintf(stdout,"ERR:%s:%s\n",query,errStr);
}

int main( int argc , char **argv)
{
	if(argc<2){
		writeError("none","No host to resolve");
		return 0;
	}

	char *hostToResolve = argv[1];
	struct in_addr inAddr;
	struct hostent *hostEntry;

	if(inet_aton(hostToResolve,&inAddr)){
		hostEntry=gethostbyaddr((const char *)&inAddr,sizeof(inAddr),AF_INET);
	} else {
		hostEntry=gethostbyname(hostToResolve);
	}
	if(!hostEntry){
		switch(h_errno){
			case HOST_NOT_FOUND:
				writeError(hostToResolve,"Host not found");
				break;
			case NO_ADDRESS:
				writeError(hostToResolve,"Valid name but host has no IP address");
				break;
			case NO_RECOVERY:
				writeError(hostToResolve,"Unrecoverable name server error");
				break;
			case TRY_AGAIN:
				writeError(hostToResolve,"DNS temporaneous fault,try again");
				break;
			default:
				writeError(hostToResolve,"DNS Query failed , unable to resolve host");
				break;
		}
		return 0;
	}
	//lookup:query:hostname:ip:alias1:alias2
	
	if(hostEntry->h_aliases[0]){
		if(hostEntry->h_aliases[1]){
			fprintf(stdout,"DNS:%s:%s:%s:%s:%s\n",
							hostToResolve,
							hostEntry->h_name,
							inet_ntoa( * (struct in_addr*) hostEntry->h_addr),
							hostEntry->h_aliases[0],
							hostEntry->h_aliases[1]);	

		} else {
			fprintf(stdout,"DNS:%s:%s:%s:%s:none\n",
							hostToResolve,
							hostEntry->h_name,
							inet_ntoa( * (struct in_addr*) hostEntry->h_addr),
							hostEntry->h_aliases[0]);	
		}
	} else {
		fprintf(stdout,"DNS:%s:%s:%s:none:none\n",
							hostToResolve,
							hostEntry->h_name,
							inet_ntoa( * (struct in_addr*) hostEntry->h_addr));	
	}
	return 0;
}
