//   $Id: kviphone.cpp,v 1.0 1998/10/19 20:21:57 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

#ifdef HAVE_CONFIG_H
	#include "../config.h"
#endif

#include <signal.h>     /* for signal() and siginterrupt() */
#include <stdio.h>      /* for fprintf() and sscanf() */
#include <sys/types.h>  //
#include <sys/socket.h> //
#include <netinet/in.h> // for socket() inet_aton()
#include <arpa/inet.h>  //
#include <sys/time.h>   /* for timeval */
#include <sys/stat.h>   // for open()
#include <fcntl.h>      //
#include <string.h>     /* for bzero() and strcasecmp() */
#include <sys/ioctl.h>  /* for ioctl() */
#include <unistd.h>     /* for close() */

#ifdef HAVE_SYS_SELECT_H
	#include <sys/select.h>
#endif

#undef __KILL_KVIPHONE__

//Check for the *SS Api....we don't want to fail here...

#ifdef HAVE_LINUX_SOUNDCARD_H
	#include <linux/soundcard.h>
#else
	#ifdef HAVE_SYS_SOUNDCARD_H
		#include <sys/soundcard.h>
	#else
		#ifdef HAVE_SOUNDCARD_H
			#include <soundcard.h>			
		#else
			//CAN NOT COMPILE :(
			#define __KILL_KVIPHONE__
			#warning "Can not find the soundcard.h include : You will NOT be able to use DCC voice."
		#endif
	#endif
#endif

#include <errno.h>

#include "kviphone.h"

// ----------------------------->
//                 COMPATIBILITY
// ----------------------------->

#ifndef HAVE_INET_ATON
	#define inet_aton(__arg1,__arg2) my_inet_aton(__arg1,__arg2)
	#include <ctype.h>
	// 
	// Check whether "cp" is a valid ascii representation
	// of an Internet address and convert to a binary address.
	// Returns 1 if the address is valid, 0 if not.
	// This replaces inet_addr, the return value from which
	// cannot distinguish between failure and a local broadcast address.
	// Taken from ircd source.
	//
	int my_inet_aton(register const char *cp, struct in_addr *addr)
	{
		register unsigned long val;
		register int base, n;
		unsigned int parts[4];
		register unsigned int *pp = parts;	
		register char c = *cp;
		for (;;) {
			//
			// Collect number up to ``.''.
			// Values are specified as for C:
			// 0x=hex, 0=octal, isdigit=decimal.
			//
			if(!isdigit(c))return (0);
			val = 0; base = 10;
			if(c == '0') {
				c = *++cp;
				if (c == 'x' || c == 'X')base = 16, c = *++cp;
				else base = 8;
			}
			for (;;) {
				if (isascii(c) && isdigit(c)) {
					val = (val * base) + (c - '0');
					c = *++cp;
				} else if (base == 16 && isascii(c) && isxdigit(c)) {
					val = (val << 4) | (c + 10 - (islower(c) ? 'a' : 'A'));
					c = *++cp;
				} else break;
			}
			if (c == '.') {
				//
				// Internet format:
				//	a.b.c.d
				//	a.b.c	(with c treated as 16 bits)
				//	a.b	(with b treated as 24 bits)
				//
				if (pp >= parts + 3)return (0);
				*pp++ = val;
				c = *++cp;
			} else
				break;
		}
		// Check for trailing characters.
		if (c != '\0' && (!isascii(c) || !isspace(c)))return (0);
		//
		// Concoct the address according to
		// the number of parts specified.
		//
		n = pp - parts + 1;
		switch (n) {	
		case 0:
			return (0);		/* initial nondigit */		
		case 1:				/* a -- 32 bits */
			break;	
		case 2:				/* a.b -- 8.24 bits */
			if (val > 0xffffff)return (0);
			val |= parts[0] << 24;
			break;
		case 3:				/* a.b.c -- 8.8.16 bits */
			if (val > 0xffff)return (0);
			val |= (parts[0] << 24) | (parts[1] << 16);
			break;
		case 4:				/* a.b.c.d -- 8.8.8.8 bits */
			if (val > 0xff)return (0);
			val |= (parts[0] << 24) | (parts[1] << 16) | (parts[2] << 8);
			break;
		}
		if (addr)addr->s_addr = htonl(val);
		return (1);
	}
#endif //HAVE_INET_ATON

#ifndef HAVE_BZERO
	#ifdef HAVE_MEMSET
		#define bzero(__arg1,__arg2) memset((__arg1),0,(__arg2))
	#else
		#define bzero(__arg1,__arg2) my_bzero(__arg1,__arg2)
		void my_bzero(void * s,int n)
		{
			register char * p=(char *)s;
			while(n){
				*p=(char)0;
				p++;
				n--;
			}
		}
	#endif
#endif //HAVE_BZERO


#ifndef SIGUSR1
#define SIGUSR1 10
#endif

#ifndef SIGUSR2
#define SIGUSR2 12
#endif


// ------------------------------->
//                        KVIPHONE
// ------------------------------->

//DO NOT CHANGE THIS
#define KVI_AUDIO_DEVICE "/dev/dsp"
#define KVI_FRAG_SIZE 0x00080008
#define KVI_FORMAT AFMT_S16_LE
#define KVI_NUM_CHANNELS 1
#define KVI_SPEED 8000

//Read from sound card buffer in bytes!
#define KVI_READ_BUF_SIZE_IN_CHARS 2048 
#define KVI_READ_BUF_SIZE_IN_SHORTS 1024
//By now should be same as above
#define KVI_WRITE_BUF_SIZE_IN_CHARS 2048
#define KVI_WRITE_BUF_SIZE_IN_SHORTS 1024
//4:1 compression
#define KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS 512

//states
#define KVI_STATE_SILENCE 0;
#define KVI_STATE_TALK 1
#define KVI_STATE_LISTEN 2
#define KVI_STATE_LOCK 3
#define KVI_STATE_FINISHJOBANDEXIT 4

//a compressed saple readed or ready to send
typedef struct SampleStruct{
	char buffer[KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS];
	SampleStruct *next;
};

// -------------------------------->
//                          GLOBALS
// -------------------------------->

// send queue
static SampleStruct * m_sendHead;
static SampleStruct * m_sendTail;
// receive queue
static SampleStruct * m_recvHead;
static SampleStruct * m_recvTail;
static SampleStruct * m_tempBuff;
// internal state
static int m_state;
static int m_lastState;
// file descriptors
static int m_sound;
static int m_sock;
// socket description
static unsigned short int m_iPort;
static struct in_addr m_inAddr;
// buffers and counters
static struct ADPCM_state m_adpcm;
static char m_soundReadBuffer[KVI_READ_BUF_SIZE_IN_CHARS];
static char m_soundWriteBuffer[KVI_WRITE_BUF_SIZE_IN_CHARS];
static int m_completeBuffers;
// select variables
static fd_set m_fd;
static fd_set m_fdAux;
static struct timeval m_tm;
// in and out queue helpers
static int m_lastSentSize;
static int m_lastReadSize;
static int m_recvQueueCount;
//static int m_sendQueueCount;

#ifndef __KILL_KVIPHONE__

//------------------------------>
//               SIGNAL HANDLERS
//------------------------------>
// Handler for SIGUSR1
void signalStartTalk(int signalNumber)
{
	signal(SIGUSR1,signalStartTalk);
	siginterrupt(SIGUSR1,0);
	if(m_state != KVI_STATE_LOCK)m_state=KVI_STATE_TALK;
}
// Handler for SIGUSR2
void signalStopTalk(int signalNumber)
{
	signal(SIGUSR2,signalStopTalk);
	siginterrupt(SIGUSR2,0);
	if(m_state != KVI_STATE_LOCK)m_state=KVI_STATE_LISTEN;
}

// ----------------------------->
//                       CLEANUP
// ----------------------------->
void cleanup()
{
	//Clear everything possible before exiting
	if(m_sound > 0)close(m_sound);
	while(m_sendHead){
		SampleStruct *s=m_sendHead;
		m_sendHead=s->next;
		delete s;
	}
	while(m_recvHead){
		SampleStruct *s=m_recvHead;
		m_recvHead=s->next;
		delete s;
	}
	if(m_sock > 0)close(m_sock);
}

// ----------------------------->
//                        ERRORS
// ----------------------------->
void fatal(const char *message)
{
	fprintf(stderr,"E%s",message);
	fprintf(stderr,"\n");
	if(m_state!=KVI_STATE_LOCK){
		cleanup();
		exit(-1);
	} else m_state=KVI_STATE_FINISHJOBANDEXIT;
}

// ---------------------------->
//                      MESSAGE
// ---------------------------->
void message(const char *msg)
{
	fprintf(stderr,"M%s",msg);
	fprintf(stderr,"\n");
}
void msglistening(unsigned short int thePort)
{
	fprintf(stderr,"L%u",thePort);
	fprintf(stderr,"\n");
}

// ----------------------------->
//                     SOUNDCARD
// ----------------------------->
void openSoundCardForWriting()
{
	if(m_sound)fatal("openSoundCardForWriting : Sound card already open");
	m_sound=open(KVI_AUDIO_DEVICE,O_WRONLY);
	if(m_sound<0)fatal("openSoundCardForWriting : Unable to open /dev/dsp");
	static int frag=KVI_FRAG_SIZE;
	if(ioctl(m_sound,SNDCTL_DSP_SETFRAGMENT,&frag)<0)fatal("openSoundCardForWriting : Unable to set fragment size");
	static int fmt=KVI_FORMAT;
	if(ioctl(m_sound,SNDCTL_DSP_SETFMT,&fmt)<0)fatal("openSoundCardForWriting : Unsupported format");
	static int chans=KVI_NUM_CHANNELS;
	if(ioctl(m_sound,SNDCTL_DSP_CHANNELS,&chans)<0)fatal("openSoundCardForWriting : Unsupported channels number");
	static int speed=KVI_SPEED;
	if(ioctl(m_sound,SNDCTL_DSP_SPEED,&speed)<0)fatal("openSoundCardForWriting : Unsupported sample rate");
}
void openSoundCardForReading()
{
	if(m_sound)fatal("openSoundCardForReading : Sound card already open");
	m_sound=open(KVI_AUDIO_DEVICE,O_RDONLY);
	if(m_sound<0)fatal("openSoundCardForReading : Unable to open /dev/dsp");
	static int frag=KVI_FRAG_SIZE;
	if(ioctl(m_sound,SNDCTL_DSP_SETFRAGMENT,&frag)<0)fatal("openSoundCardForReading : Unable to set fragment size");
	static int fmt=KVI_FORMAT;
	if(ioctl(m_sound,SNDCTL_DSP_SETFMT,&fmt)<0)fatal("openSoundCardForReading : Unsupported format");
	static int chans=KVI_NUM_CHANNELS;
	if(ioctl(m_sound,SNDCTL_DSP_CHANNELS,&chans)<0)fatal("openSoundCardForReading : Unsupported channels number");
	static int speed=KVI_SPEED;
	if(ioctl(m_sound,SNDCTL_DSP_SPEED,&speed)<0)fatal("openSoundCardForReading : Unsupported sample rate");
}
void initSoundCard()
{
	openSoundCardForReading();
	close(m_sound);
	m_sound=0;
}

// ----------------------------->
//           SEND & RECEIVE DATA
// ----------------------------->

bool receiveData()
{
	m_tm.tv_sec = 0;
	m_tm.tv_usec = 1;
	FD_ZERO(&m_fd);
	FD_SET(m_sock,&m_fd);
	if(select(m_sock+1,&m_fd,0,0,&m_tm)>0){
		if(!m_lastReadSize)m_tempBuff=new SampleStruct; //new buffer begin
		//if m_lastReadSize is 0
		int toRead=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-m_lastReadSize;
		//toread=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-0;
		int readed=read(m_sock,(void *)((m_tempBuff->buffer)+m_lastReadSize),toRead);
		//read(m_sock,(void *)m_tempBuff->buffer+0),KVI_COMPRESSED_SAMPLE....);
		if(readed <= 0){
			if(readed==0)fatal("Remote end closed connection");
			else fatal("Connection terminated (socket error)");
		} else {
			if(readed==toRead){
				m_lastReadSize=0; //next time start a new buffer
				//append the buf
				m_tempBuff->next=0;
				if(m_recvHead){ //other data in queue
					m_recvTail->next=m_tempBuff;
					m_recvTail=m_tempBuff;
				} else { //no data in queue
					m_recvHead=m_tempBuff;
					m_recvTail=m_tempBuff;
				}
				//check if it was an EOF
				if(m_tempBuff->buffer[0]==0){
					if(m_tempBuff->buffer[1]==0){
						if(m_tempBuff->buffer[2]==0){
							if(m_tempBuff->buffer[3]==0){
								if(m_tempBuff->buffer[4]==0){
									if(m_tempBuff->buffer[5]==0)m_completeBuffers++;
								}
							}
						}
					}
				}
				m_recvQueueCount++;
				m_tempBuff=0;
			} else m_lastReadSize+=readed; //continue here
		}
		return true;
	} else return false;
}

bool sendToSocket()
{
	m_tm.tv_sec = 0;
	m_tm.tv_usec = 1;
	FD_ZERO(&m_fd);
	FD_SET(m_sock,&m_fd);
	if(select(m_sock+1,0,&m_fd,0,&m_tm)>0){
		//get the first available buffer and try to send it
		SampleStruct *s=m_sendHead;
		//m_lastSentSize can be either 0 here...
		int toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-m_lastSentSize;
		//so toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS;
		int wrote=write(m_sock,(void *)((s->buffer)+m_lastSentSize),toSend);
		// int wrote=write(m_sock,(void *)(s->buffer)+0,KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS);
		if(wrote<0)fatal("Can't write to socket , connection lost");
		else {
			if(wrote==toSend){
				m_lastSentSize=0;
				m_sendHead=s->next;
				if(!m_sendHead)m_sendTail=0;
				delete s;
//				m_sendQueueCount--;
//				fprintf(stderr,"B%d",m_sendQueueCount);
//				fprintf(stderr,"\n");
			} else m_lastSentSize+=wrote; //and finish next time

			return true;
		}
	} else return false;
}

void sendAndReceive()
{
	//only one select call
	m_tm.tv_sec = 0;
	m_tm.tv_usec = 1;
	FD_ZERO(&m_fd);
	FD_SET(m_sock,&m_fd);
	FD_ZERO(&m_fdAux);
	FD_SET(m_sock,&m_fdAux);
	if(select(m_sock+1,&m_fd,&m_fdAux,0,&m_tm)>0){
		if(FD_ISSET(m_sock,&m_fd)){ //can read
			if(!m_lastReadSize)m_tempBuff=new SampleStruct; //new buffer begin
			//if m_lastReadSize is 0
			int toRead=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-m_lastReadSize;
			//toread=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-0;
			int readed=read(m_sock,(void *)((m_tempBuff->buffer)+m_lastReadSize),toRead);
			//read(m_sock,(void *)m_tempBuff->buffer+0),KVI_COMPRESSED_SAMPLE....);
			if(readed <= 0){
				if(readed==0)fatal("Remote end closed connection");
				else fatal("Connection terminated (socket error)");
			} else {
				if(readed==toRead){
					m_lastReadSize=0; //next time start a new buffer
					//append the buf
					m_tempBuff->next=0;
					if(m_recvHead){ //other data in queue
						m_recvTail->next=m_tempBuff;
						m_recvTail=m_tempBuff;
					} else { //no data in queue
						m_recvHead=m_tempBuff;
						m_recvTail=m_tempBuff;
					}
					if(m_tempBuff->buffer[0]==0){
						if(m_tempBuff->buffer[1]==0){
							if(m_tempBuff->buffer[2]==0){
								if(m_tempBuff->buffer[3]==0){
									if(m_tempBuff->buffer[4]==0){
										if(m_tempBuff->buffer[5]==0)m_completeBuffers++;
									}
								}
							}
						}
					}
					m_recvQueueCount++;
					fprintf(stderr,"V%d",m_recvQueueCount);
					fprintf(stderr,"\n");
					m_tempBuff=0;
				} else m_lastReadSize+=readed; //continue here
			}
		}
		if(FD_ISSET(m_sock,&m_fdAux)){ //can write
			//get the first available buffer and try to send it
			SampleStruct *s=m_sendHead;
			//m_lastSentSize can be either 0 here...
			int toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS-m_lastSentSize;
			//so toSend=KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS;
			int wrote=write(m_sock,(void *)((s->buffer)+m_lastSentSize),toSend);
			// int wrote=write(m_sock,(void *)(s->buffer)+0,KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS);
			if(wrote<0)fatal("Can't write to socket , connection lost");
			else {
				if(wrote==toSend){
					m_lastSentSize=0;
					m_sendHead=s->next;
					if(!m_sendHead)m_sendTail=0;
					delete s;
//					m_sendQueueCount--;
//					fprintf(stderr,"B%d",m_sendQueueCount);
//					fprintf(stderr,"\n");
				} else m_lastSentSize+=wrote; //and finish next time
			}
		}
	}
}

// ----------------------------->
//                      PLAYBACK
// ----------------------------->
void playback()
{
	fprintf(stderr,"I");
	fprintf(stderr,"\n");
	m_state=KVI_STATE_LOCK;
	//playback Loop
	bool notEof=true;
	while(m_recvHead && notEof){
		SampleStruct *s=m_recvHead;
		m_recvHead=s->next;
		if(m_recvHead==0)m_recvTail=0;
		//check if it is EOF for us
		//gnary hack but works (6 bytes 0 is a complete rumour);
		if(s->buffer[0]==0){
			if(s->buffer[1]==0){
				if(s->buffer[2]==0){
					if(s->buffer[3]==0){
						if(s->buffer[4]==0){
							if(s->buffer[5]==0){
								notEof=false;
								m_completeBuffers--;
							}
						}
					}
				}
			}
		}
		m_recvQueueCount--;
		if(notEof){ //play the sample
			m_adpcm.index=0;
			m_adpcm.valprev=0;
			ADPCM_uncompress(s->buffer,(short *)m_soundWriteBuffer,KVI_WRITE_BUF_SIZE_IN_SHORTS,&m_adpcm);
			//block TODO : check for space in the buffer
			write(m_sound,(void *)m_soundWriteBuffer,KVI_WRITE_BUF_SIZE_IN_CHARS);
			//do something while playing 
			if(m_state==KVI_STATE_LOCK){
				if(m_sendHead)sendToSocket();	
				else {
					if(!receiveData()){
						fprintf(stderr,"V%d",m_recvQueueCount);
						fprintf(stderr,"\n");
						usleep(10); //<---try it later
					}
					///receiveData();
				}
			} else {
				//we are in FINISHJOBANDEXIT
				//finish playback and exit
				fprintf(stderr,"V%d",m_recvQueueCount);
				fprintf(stderr,"\n");
			}
		}
		delete s;
	}
	usleep(100);
	if(m_state == KVI_STATE_FINISHJOBANDEXIT){ //the connection was closed while playing
		fatal("Playback complete : Exiting.");
	}
	m_state=KVI_STATE_LISTEN;
	fprintf(stderr,"U");
	fprintf(stderr,"\n");
}

// ----------------------------->
//                     MAIN LOOP
// ----------------------------->
int mainLoop(){
	//setup
	openSoundCardForWriting();
	m_state=KVI_STATE_LISTEN;
	m_lastState=KVI_STATE_LISTEN;
	//loop
	for(;;){
		if(m_state==KVI_STATE_TALK){
			//we're talking
			if(m_lastState != KVI_STATE_TALK){
				//just started
				close(m_sound);
				m_sound=0;
				openSoundCardForReading();
				m_lastState=KVI_STATE_TALK;
			}
			//read a sample (2048 bytes)
			read(m_sound,(void *)m_soundReadBuffer,KVI_READ_BUF_SIZE_IN_CHARS);
			//compress
			m_adpcm.index=0;
			m_adpcm.valprev=0;
			SampleStruct *s=new SampleStruct;
			//compress (1024 shorts to 512 byte samples)
			ADPCM_compress((short *)m_soundReadBuffer,s->buffer,KVI_READ_BUF_SIZE_IN_SHORTS,&m_adpcm);
			//append to the list
			s->next=0;
			if(m_sendHead){
				//other data in queue
				m_sendTail->next=s;
				m_sendTail=s;
			} else {
				//no data in queue
				m_sendHead=s;
				m_sendTail=s;
			}
//			m_sendQueueCount++;
			//check and maybe remove this is there are pauses in the sampling
			if(!sendToSocket())receiveData();
			//spit out our current buf state
//			fprintf(stderr,"B%d",m_sendQueueCount);
//			fprintf(stderr,"\n");
		} else { //KVI_STATE_LISTEN
			//we're listening for data
			if(m_lastState != KVI_STATE_LISTEN){
				//just finished talking
				//append an empty buffer as a EOF sign
				SampleStruct *s=new SampleStruct;
				bzero((void *)s->buffer,KVI_COMPRESSED_SAMPLE_SIZE_IN_CHARS);
				s->next=0;
				if(m_sendHead){ //other data in queue
					m_sendTail->next=s;
					m_sendTail=s;
				} else { //no data in queue
					m_sendHead=s;
					m_sendTail=s;
				}
//				m_sendQueueCount++;
				//and send it if possible
				sendToSocket();
//				fprintf(stderr,"B%d",m_sendQueueCount);
//				fprintf(stderr,"\n");
				close(m_sound);
				m_sound=0;
				openSoundCardForWriting();
				m_lastState=KVI_STATE_LISTEN;
			}
			//send the remaining part (if any) and read data from socket
			if(m_sendHead)sendAndReceive(); //one select call
			else {
				if(receiveData()){
					fprintf(stderr,"V%d",m_recvQueueCount);
					fprintf(stderr,"\n");
				} else usleep(200); //release the CPU for a while
			}
			//start playback when have a complete audio buffer
			if(m_completeBuffers){
				playback();
				fprintf(stderr,"V%d",m_recvQueueCount);
				fprintf(stderr,"\n");
			}
		}
	}
}

// --------------------------->
//                  CONNECTION
// --------------------------->

void listenLoop()
{
	message("Preparing to listen...");
	struct sockaddr_in sockAddress;
	bzero((char *)&sockAddress, sizeof(sockAddress));
	sockAddress.sin_family = AF_INET;
	sockAddress.sin_port   = htons(0);
	sockAddress.sin_addr.s_addr = INADDR_ANY;
	int aSock;
	if((aSock = socket(AF_INET,SOCK_STREAM,0)) < 0)fatal("Unable to create a stream socket");
	if((bind(aSock,(struct sockaddr *) &sockAddress,sizeof(sockAddress))<0)||(listen(aSock,100)<0) ) {
		close(aSock);
		fatal("Unable to setup a listening socket");
	}
	if(fcntl(aSock, F_SETFL, O_NONBLOCK) < 0){
		close(aSock);
		fatal("Unable to create non-blocking socket");
	}
	ksize_t size=sizeof(sockAddress);
	getsockname(aSock, (struct sockaddr*)&sockAddress, &size );
	unsigned short int thePort = ntohs(sockAddress.sin_port);
	msglistening(thePort);
	//now enter in loop
	for(;;){
		//connected ,accept it*/
		size=sizeof(sockAddress);
		m_sock=accept(aSock, (struct sockaddr*)&sockAddress, &size);
		if(m_sock != -1){
			close(aSock);
			fprintf(stderr,"CConnected to %s on port %u",inet_ntoa(sockAddress.sin_addr),sockAddress.sin_port);
			fprintf(stderr,"\n");
			return;
		} else {
			if(errno != EWOULDBLOCK)fatal("Unable to accept a connection");
			usleep(200);
		}
	}
}

void connectLoop()
{
	message("Preparing to connect...");
	m_sock=socket(AF_INET,SOCK_STREAM,0);
	if(m_sock<0)fatal("Unable to create stream socket");
	if(fcntl(m_sock, F_SETFL, O_NONBLOCK) < 0)fatal("Unable to create non-blocking socket");
	struct sockaddr_in sain;
	sain.sin_family = AF_INET;
	sain.sin_port   = htons(m_iPort);
	sain.sin_addr   = m_inAddr;
	if(connect(m_sock,(struct sockaddr*)(&sain),sizeof(sain))<0){
		if(errno != EINPROGRESS){
			switch(errno){
				case ECONNREFUSED:
					fatal("Connection refused");
					break;
				case ENETUNREACH:
					fatal("Network is unreachable");
					break;
				case ETIMEDOUT:
					fatal("Connection timed out");
					break;
				default:
					fatal("Unable to connect remote host");
					break;
			}
			//newer arrive here
			fprintf(stderr,"NEWER ARRIVE HERE!\n");
			exit(-1);
		} else message("Connection in progress...");
	} else {
		fprintf(stderr,"CConnected (was blocking)");
		fprintf(stderr,"\n");
		return;
	}
	for(;;){
		m_tm.tv_sec = 0;
		m_tm.tv_usec = 1;
		FD_ZERO(&m_fd);
		FD_SET(m_sock,&m_fd);
		if(select(m_sock+1,0,&m_fd,0,&m_tm)){
			int nSockErr;
			ksize_t iSize=sizeof(int);
			if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&nSockErr,&iSize)==-1)nSockErr=EBADF;
			if(nSockErr != 0){
				switch(nSockErr){
					case ECONNREFUSED:
						fatal("Connection refused");
						break;
					case ENETUNREACH:
						fatal("Network is unreachable");
						break;
					case ETIMEDOUT:
						fatal("Connection timed out");
						break;
					default:
						fatal("Unable to connect remote host");
						break;
				}
				//newer arrive here
				fprintf(stderr,"NEWER ARRIVE HERE!");
				exit(-1);
			} else {
				fprintf(stderr,"CConnected (was select)");
				fprintf(stderr,"\n");
			}
			//connected
			return;
		} else usleep(200);
	}
}

#endif //__KILL_KVIPHONE__

// ----------------------------->
//                          MAIN
// ----------------------------->
int main(int argc,char ** argv)
{
	//set the signal handlers

#ifndef __KILL_KVIPHONE__

	signal(SIGUSR1,signalStartTalk);
	siginterrupt(SIGUSR1,0);
	signal(SIGUSR2,signalStopTalk);
	siginterrupt(SIGUSR2,0);
	//reset
	m_state=KVI_STATE_SILENCE;
	m_sound=0;
	m_sendHead=0;
	m_sendTail=0;
	m_recvHead=0;
	m_recvTail=0;
	m_tempBuff=0;
	m_completeBuffers=0;
	m_sock=0;
	m_lastSentSize=0;
	m_lastReadSize=0;
	m_recvQueueCount=0;
//	m_sendQueueCount=0;
	//check for /dev/dsp
	initSoundCard();
	//check for the params
	bool bNeedListen=false;
	switch(argc){
		case 2:
			if(strcasecmp(argv[1],"listen")){ fatal("Bad arguments"); }
			else bNeedListen=true;
			break;
		case 3:
			if(!inet_aton(argv[1],&m_inAddr)){ fatal("Bad arguments"); }
			else {
				if(sscanf(argv[2],"%u",&m_iPort) !=1){ fatal("Bad arguments"); }
				else bNeedListen=false;
			}
			break;
		default:
			fatal("Bad arguments");
			break;
	}
	if(bNeedListen)listenLoop();
	else connectLoop();
	mainLoop();
	cleanup();
#endif //__KILL_KVIPHONE__

	return 0;
}
