//   $Id: kvi_ctcp.cpp,v 1.11 1998/10/06 14:42:10 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviCtcp"

#include "kvi_macros.h"
#include "kvi_ctcp.h"
#include "kvi_frame.h"
#include "kvi_debug.h"
#include "kvi_socket.h"
#include "kvi_sparser.h"
#include "kvi_token.h"
#include "kvi_mdi.h"
#include "kvi_opt.h"
#include "kvi_chat.h"
#include "kvi_status.h"
#include "kvi_app.h"
#include "kvi_send.h"
#include "kvi_event.h"
#include "kvi_global.h"
#include "kvi_uparser.h"
#include "kvi_audio.h"
#include "kvi_ident.h"
#include "kvi_voice.h"
//#include <kfiledialog.h>
#include <kmsgbox.h>


//============ KviCTCP ============//

KviCTCP::KviCTCP(KviFrame *lpFrame,KviServerParser *lpPar) : QObject(lpFrame)
{
	_debug_entertrace("KviCTCP");
	m_lpFrm       = lpFrame;
	m_lpGlb       = m_lpFrm->m_lpGlb;
	m_lpUsr       = m_lpFrm->m_lpUsr;
	m_lpOpt       = m_lpFrm->m_lpOpt;
	m_lpSock      = m_lpFrm->m_lpSock;
	m_iCTCPCount  = 0;
	m_lpPar       = lpPar;
	m_lpCTCPTimer = new QTimer(this);
	connect(m_lpCTCPTimer,SIGNAL(timeout()),this,SLOT(resetCTCPCount()));
	_debug_leavetrace("KviCTCP");
}

//============ ~KviCTCP ============//

KviCTCP::~KviCTCP()
{
	_debug_entertrace("~KviCTCP");
	if(m_lpCTCPTimer->isActive())m_lpCTCPTimer->stop();
	delete m_lpCTCPTimer;
	_debug_leavetrace("~KviCTCP");
}

//============ replyCTCPErrormsg ============//

void KviCTCP::replyCTCPErrormsg(QString &szNick,const char *szReply)
{
	_debug_entertrace("replyCTCPErrormsg");
	m_lpFrm->m_lpConsole->doFmtOutput(KVI_OUT_CTCP,"Replying CTCP ERROR to %s : %s",szNick.data(),szReply);
	m_lpSock->sendFmtData("NOTICE %s :%c%s%c",szNick.data(),0x01,szReply,0x01);
	_debug_leavetrace("replyCTCPErrormsg");
}

//============ handleCTCPActionRequest ============//

void KviCTCP::handleCTCPActionRequest(QString &szNick,QString &target,KviIrcToken *lpData)
{
	_debug_entertrace("handleCTCPActionRequest");
	KviMdiChild *lpC=0;
	if(target.find('#') != -1)lpC=m_lpPar->findWin(target.data());
	else lpC=m_lpPar->findWin(szNick.data());
	lpC->doFmtOutput(KVI_OUT_ACTION,"%s %s",szNick.data(),lpData->szString.data());
	_debug_leavetrace("handleCTCPActionRequest");
}

//============ handleCTCPPageRequest ============//

void KviCTCP::handleCTCPPageRequest(QString &szNick,QString &target,KviIrcToken *lpData)
{
	_debug_entertrace("handleCTCPPageRequest");
	KviMdiChild *lpC=0;
	if(target.find('#') != -1)lpC=m_lpPar->findWin(target.data());
	else lpC=m_lpPar->findWin(szNick.data());
	lpC->doFmtOutput(KVI_OUT_CTCP,"[PAGE from %s] %s",szNick.data(),lpData->szString.data());
	_debug_leavetrace("handleCTCPPageRequest");
}

//============ handleDCCChat ============//

void KviCTCP::handleDCCChat(QString &szNick,QString &szMask,QString &,QString &szAddr,QString &szPort)
{
	_debug_entertrace("handleDCCChat");
	if(!m_lpOpt->bIgnoreDCCChat){
		QString szReq=szNick+" ["+szMask+i18n("]\nrequests a DCC CHAT [")+szAddr+" : "+szPort+i18n("]\nDo you accept?");
		if(!m_lpOpt->bAutoAcceptDCCChat){
			if(KMsgBox::yesNo(m_lpFrm, i18n("DCC CHAT Request"), szReq.data())==1){ // Fritz: check retcode
				QString szChatName="[DCC-CHAT]:-"+szNick;		
				KviChatWnd *lpWnd=m_lpFrm->createChatWnd(szChatName.data());
				lpWnd->acceptChat(szPort,szAddr,szNick.data());
			} else m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Refusing DCC CHAT from %s [%s]"),szNick.data(),szMask.data());
		} else {
			QString szChatName="[DCC-CHAT]:-"+szNick;
			KviChatWnd *lpWnd=m_lpFrm->createChatWnd(szChatName.data());
			lpWnd->acceptChat(szPort,szAddr,szNick.data());			
		}
	} else replyCTCPErrormsg(szNick,"DCC CHAT protocol is currently disabled. (user request)");
	_debug_leavetrace("handleDCCChat");
}

//============ handleDCCVoice ============//

void KviCTCP::handleDCCVoice(QString &szNick,QString &szMask,QString &,QString &szAddr,QString &szPort)
{
	_debug_entertrace("handleDCCChat");
	if(!m_lpOpt->bIgnoreDCCVoice){
		if(!m_lpFrm->m_lpVoiceWnd){
			QString szReq=szNick+" ["+szMask+i18n("]\nrequests a DCC VOICE [")+szAddr+" : "+szPort+i18n("]\nDo you accept?");
			if(!m_lpOpt->bAutoAcceptDCCVoice){
				if(KMsgBox::yesNo(m_lpFrm, i18n("DCC VOICE Request"), szReq.data())==1){ // Fritz: check retcode
					QString szChatName="[DCC-VOICE]:-"+szNick;		
					KviVoiceWnd *lpWnd=m_lpFrm->createVoiceWnd(szChatName.data());
					lpWnd->acceptConnection(szPort,szAddr,szNick.data());
				} else m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Refusing DCC VOICE from %s [%s]"),szNick.data(),szMask.data());
			} else { //auto accept it
				QString szChatName="[DCC-VOICE]:-"+szNick;		
				KviVoiceWnd *lpWnd=m_lpFrm->createVoiceWnd(szChatName.data());
				lpWnd->acceptConnection(szPort,szAddr,szNick.data());
			}
		} else replyCTCPErrormsg(szNick,"DCC VOICE : Sorry , another VOICE connection in progress.");
	} else replyCTCPErrormsg(szNick,"DCC VOICE protocol is currently disabled. (user request)");
	_debug_leavetrace("handleDCCVoice");
}

//============ getDCCSaveDirectory ============//

QString KviCTCP::getDCCSaveDirectory(QString &szStr)
{
	_debug_entertrace("getDCCSaveDirectory");
	//Get the correct default directory
	// Sounds goes in Audio
	// Pictures in Images
	// The rest in Incoming
	QString szHome(_macro_getKVircHomeDirectory("Incoming"));
	bool bRecogn=(szStr.find(".wav") != -1);
	if(!bRecogn)bRecogn=(szStr.find(".mid") != -1);
	if(!bRecogn)bRecogn=(szStr.find(".au") != -1);
	if(!bRecogn)bRecogn=(szStr.find(".mp3") != -1);
	if(bRecogn)szHome=QString(_macro_getKVircHomeDirectory("Audio"));
	else {
		//Check for images....maybe it's too long....
		bool bRecogn=(szStr.find(".jpg") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".gif") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".xpm") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".ppm") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".bmp") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".jpeg") != -1);
		if(!bRecogn)bRecogn=(szStr.find(".tif") != -1);
		if(bRecogn)szHome=QString(_macro_getKVircHomeDirectory("Images"));	
	}
	_debug_leavetrace("getDCCSaveDirectory");
	return szHome;
}

//============ getDCCSaveFileName ============//

bool KviCTCP::getDCCSaveFileName(QString &szNewFileName,QString &szOldFileName,QString &szFileSize)
{
	_debug_entertrace("getDCCSaveFileName");
	QString szHome(getDCCSaveDirectory(szOldFileName));
	QDir aDir(szHome.data());
	if(!aDir.exists())aDir.mkdir(szHome.data(),true);
	szNewFileName=_macro_getSaveFileName(0,szHome.data(),szOldFileName.data());

	bool bOK=false;
	uint uFileSize=szFileSize.toUInt(&bOK);
	if(!bOK)uFileSize=0;

	if(!szNewFileName.isEmpty()){
		QFile testFile(szNewFileName.data());
		bool bExists=testFile.exists();
		if(bExists){
			//check the file size;
			bool bShouldResume=false;
			QFileInfo testFInfo(testFile);
			QString szShortFileName=testFInfo.fileName();
			if(!strcmp(szShortFileName.data(),szOldFileName.data())){
				//selected the same file name!..if the size that whe have is smaller than the offered one...
				if(testFInfo.size()<uFileSize)bShouldResume=true;
			}
			if(bShouldResume){
				//QString szTxt;
				//
				//Fritz : QString's sprintf seems to have problems with big buffers.
				//        I got segfaults in chunk_relloc inside QGArray::resize()
				//        with a long (but common for mp3s) szNewFileName
				//        like : /root/.kde/share/apps/kvirc/Audio/Dave_Matthews_Band_-_Dont_Drink_The_Water_-_Radio_Version.mp3
				// #0  0x404cd88e in ?? () from /lib/libc.so.6 <--this is chunk_realloc() at malloc.c:3015(where?)
				// #1  0x405253fc in ?? () from /lib/libc.so.6
				// #2  0x404cd818 in ?? () from /lib/libc.so.6
				// #3  0x402b99b4 in QGArray::resize ()
				// #4  0x402c12af in QString::resize ()
				// #5  0x402c1378 in QString::sprintf ()
				// #6  0x806ee0c in KviCTCP::getDCCSaveFileName (this=0x81e2cd8,
				//     szNewFileName=@0xbffff0ac, szOldFileName=@0xbffff128,
				//     szFileSize=@0xbffff0bc) at kvi_ctcp.cpp:194              
				//
				//szTxt.sprintf(i18n("The selected file\n%s\nalready exists.\n"
				//         "It seems to be smaller in size than the offered one,\n"
				//         "so it could be an incomplete file from a previous\n"
				//         "unsuccesfull DCC Send.\n"
				//         "Do you want to overwrite it with the offered one,\n"
				//         "rename the new file or attempt to resume the previous DCC Send?\n"
				//was      "If you choose to rename the file, KVirc will auto-rename it to\n"
				//line 194:"%s.rnm"), szNewFileName.data(), szNewFileName.data());
				//
				//  It is not the first time that I get a (think that this is a Qt bug in QGarray)
				//  segfault in QGarray functions. This is why I prefer to 'sum' the strings
				//  rather than sprintf it , and to always =X.copy() or =X.data() the string
				//  instead of sharing it.
				//  It's like QGarray sometimes gets angry and deletes a random chunk somewhere :)))
				//
				QString szTxt=i18n("The selected file\n");
				szTxt+=szNewFileName.data();
				szTxt+=i18n("\nalready exists.\nIt seems to be smaller in size than the offered one,\n"
							"so it could be an incomplete file from a previous unsuccesfull DCC Send.\n"
							"Do you want to overwrite it with the offered one,\nrename the new file "
							"or attempt to resume the previous DCC Send?\nIf you choose to rename the file,"
							"KVirc will auto-rename it to\n");
				szTxt+=szNewFileName.data();
				szTxt+=".rnm";
				m_lpFrm->m_bFocusManagingEnabled=false;
				int nResult=KMsgBox::yesNoCancel(m_lpFrm,i18n("File exists"), szTxt.data(), KMsgBox::QUESTION,
								 i18n("Overwrite"), i18n("Rename"), i18n("Resume")) - 1;
				m_lpFrm->m_bFocusManagingEnabled=true;
				if(nResult==0)return false; //overwrite , not resume
				else if(nResult==2)return true; //resume the prev dcc send
				else { //rename until get a good file name
					while(bExists){
						QFile aFile(szNewFileName);
						bExists=aFile.exists();
						if(bExists)szNewFileName+=".rnm";
					}
					return false;
				}
			} else {
				//QString szTxt;
				//szTxt.sprintf(i18n("The selected file\n%s\nalready exists.\n"
				//		   "Do you want to overwrite it with the offered one,\n"
				//		   "or rename the new file?\n"
				//		   "If you choose to rename the file,KVirc will auto-rename it to\n"
				//		   "%s.rnm"), szNewFileName.data(), szNewFileName.data());
				//Same as above
				QString szTxt=i18n("The selected file\n");
				szTxt+=szNewFileName.data();
				szTxt+=i18n("\nalready exists.\nDo you want to overwrite it with the offered one,\n"
							"or rename the new file?\nIf you choose to rename the file,KVirc will "
							"auto-rename it to\n");
				szTxt+=szNewFileName.data();
				szTxt+=".rnm";
				m_lpFrm->m_bFocusManagingEnabled=false;
				int nResult=KMsgBox::yesNo(m_lpFrm,i18n("File exists"), szTxt.data(), KMsgBox::QUESTION,
							   i18n("Overwrite"), i18n("Rename")) - 1;
				m_lpFrm->m_bFocusManagingEnabled=true;
				if(nResult==0)return false; //overwrite , not resume
				else { //rename until get a good file name
					while(bExists){
						QFile aFile(szNewFileName);
						bExists=aFile.exists();
						if(bExists)szNewFileName+=".rnm";
					}
					return true;
				}
			}
		}
	}
	_debug_leavetrace("getDCCSaveFileName");
	return false; //unique file name
}

//============ handleDCCSend ============//

void KviCTCP::handleDCCSend(QString &szNick,QString &szMask,QString &szPara,QString &szAddr,QString &szPort,KviIrcToken *lpData)
{
	_debug_entertrace("handleDCCSend");
	if(!m_lpOpt->bIgnoreDCCSend){
		QString szFileSize=lpData->nextWord();
		if(szFileSize.isEmpty())szFileSize=i18n("unknown size");
		QString szReq;
		szReq.sprintf(i18n("%s [%s]\nrequests a DCC SEND [%s:%s]\nfor file %s %s bytes long\nDo you accept?"),
			      szNick.data(), szMask.data(), szAddr.data(), szPort.data(), szPara.data(), szFileSize.data());
		if(!m_lpOpt->bAutoAcceptDCCSend){
			if(KMsgBox::yesNo(m_lpFrm, i18n("DCC SEND Request"), szReq.data())==1){
				//Make the $home/Incoming directory (if is not there)
				QString szFileName;
				bool bResume=getDCCSaveFileName(szFileName, szPara, szFileSize);
				if(!szFileName.isEmpty()){
					QString szChatName="[DCC-GET]:-"+szNick+" "+szFileName;		
					KviSendWnd *lpWnd=m_lpFrm->createSendWnd(szChatName.data(),m_lpOpt->bMinimizeDCCSend);
					lpWnd->acceptSend(szNick.data(),szPort,szAddr,szFileName.data(),szFileSize,bResume);
				} else m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Refusing DCC SEND from %s [%s]"),szNick.data(),szMask.data());	
			} else m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Refusing DCC SEND from %s [%s]"),szNick.data(),szMask.data());
		} else {
			QString szFileName(getDCCSaveDirectory(szPara));
			QDir aDir(szFileName.data());
			if(!aDir.exists())aDir.mkdir(szFileName.data(),true);
			if(szPara[0] != '/')szFileName+='/';
			szFileName+=szPara;
			bool bExists=true;
			while(bExists){
				QFile aFile(szFileName.data());
				bExists=aFile.exists();
				if(bExists)szFileName+=".rnm";
			}
			m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_INTERNAL,i18n("Auto-accepting DCC SEND from %s [%s] for file [%s]"),szNick.data(),szMask.data(),szFileName.data());
			QString szChatName="[DCC-GET]:-"+szNick+" "+szFileName;		
			KviSendWnd *lpWnd=m_lpFrm->createSendWnd(szChatName.data(),m_lpOpt->bMinimizeDCCSend);
			lpWnd->acceptSend(szNick.data(),szPort,szAddr,szFileName.data(),szFileSize);			
		}
	} else replyCTCPErrormsg(szNick,"DCC SEND protocol is currently disabled. (user request)");
	_debug_leavetrace("handleDCCSend");
}

//============ handleCTCPDCCRequest ============//

void KviCTCP::handleCTCPDCCRequest(QString &szNick,KviIrcToken *lpData,QString &szMask)
{
	_debug_entertrace("handleCTCPDCCRequest");
	
	QString szType=lpData->nextWord().upper();
	QString szPara=lpData->nextWord();
	QString szAddr=lpData->nextWord();
	QString szPort=lpData->nextWord();
	
	if(szType.isEmpty() || szPara.isEmpty() || szAddr.isEmpty() || szPort.isEmpty()){
			m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,i18n("Malformed CTCP DCC from %s [%s]:[%s:%s:%s:%s]"),
						szNick.data(),szMask.data(),szType.data(),szPara.data(),szAddr.data(),szPort.data());
	} else {
		//got a correct ctcp DCC
		if(m_lpOpt->bBeepOnDCCRequest){
			_macro_beep(20);
			_macro_beep(40);
			_macro_beep(60);
			_macro_beep(80);
			_macro_beep(100);
		}
		if(szType=="CHAT")handleDCCChat(szNick,szMask,szPara,szAddr,szPort);
		else if(szType=="VOICE")handleDCCVoice(szNick,szMask,szPara,szAddr,szPort);
		else if(szType=="SEND")handleDCCSend(szNick,szMask,szPara,szAddr,szPort,lpData);
		else if(szType=="RESUME"){
			KviMdiChild *lpC=0;
			QString szFile=szPara.data();
			QString szNewPort=szAddr.data();
			QString szPosition=szPort.data();
			bool bOK=false;
			unsigned short int iPort=(unsigned short int)szNewPort.toUInt(&bOK);
			m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,
					i18n("Received CTCP DCC RESUME from %s [ file :%s port :%s position :%s ]"),
					szNick.data(),szFile.data(),szNewPort.data(),szPosition.data());
			if(bOK){
				for(lpC=m_lpFrm->m_lpMdi->m_lpChildList->first();lpC;lpC=m_lpFrm->m_lpMdi->m_lpChildList->next()){
					if(lpC->type()==KVI_WND_TYPE_SEND){
						KviSendWnd *lpW=(KviSendWnd*)lpC;
						if(!strcasecmp(lpW->m_szRemoteNick.data(),szNick.data())){
							if(lpW->m_iPort==iPort){ //got it
								if(lpW->m_bSending){ //look if we not sending something ourself
									uint iPosition=szPosition.toUInt(&bOK);
									if(bOK && (iPosition < lpW->m_iFileLength)){
										if(lpW->dccResumeForCurrentSend(iPosition)){
											m_lpSock->sendFmtData("PRIVMSG %s :%cDCC ACCEPT %s %s %s%c",
												szNick.data(),0x01,szFile.data(),szNewPort.data(),szPosition.data(),0x01);
										} else replyCTCPErrormsg(szNick,"DCC RESUME : File position requested is out of range.");
									} else replyCTCPErrormsg(szNick,"DCC RESUME : File position is out of range.");
									return;
								}
							}
						}
					}
				}
			}
			m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,
					i18n("Refusing DCC RESUME from %s [ file :%s port :%s position :%s ] : Transfer not initiated."),
					szNick.data(),szFile.data(),szNewPort.data(),szPosition.data());
			replyCTCPErrormsg(szNick,"DCC RESUME : Transfer not initiated.");
		} else if(szType=="ACCEPT"){
			KviMdiChild *lpC=0;
			QString szFile=szPara.data();
			QString szNewPort=szAddr.data();
			QString szPosition=szPort.data();
			bool bOK=false;
			unsigned short int iPort=(unsigned short int)szNewPort.toUInt(&bOK);
			if(bOK){
				for(lpC=m_lpFrm->m_lpMdi->m_lpChildList->first();lpC;lpC=m_lpFrm->m_lpMdi->m_lpChildList->next()){
					if(lpC->type()==KVI_WND_TYPE_SEND){
						KviSendWnd *lpW=(KviSendWnd*)lpC;
						if(!strcasecmp(lpW->m_szRemoteNick.data(),szNick.data())){
							if(lpW->m_iPort==iPort){ //got it
								if(!lpW->m_bSending){
									lpW->initiateResumeGet();
									return;
								}
							}
						}
					}
				}
			}
			m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,
					i18n("Refusing DCC ACCEPT from %s [ file :%s port :%s position :%s ] : No DCC RESUME Requested."),
					szNick.data(),szFile.data(),szNewPort.data(),szPosition.data());
			replyCTCPErrormsg(szNick,"DCC ACCEPT : No DCC RESUME Requested.");
		} else replyCTCPErrormsg(szNick,"DCC WHAT ??? [ Known DCC types: SEND,GET,RESUME,ACCEPT,VOICE ]");
	}
	_debug_leavetrace("handleCTCPDCCRequest");
}

void KviCTCP::handleCTCPRequest(KviNewNick &nick,QString &target,KviIrcToken *lpData){
	//remove the 0x01
	lpData->szString.remove(0,1);
	lpData->szString.remove(lpData->szString.length()-1,1);
	//Get the CTCP type
	QString szCTCP=lpData->nextWord();
	//Get the params (if any)
	if(lpData->szString.isEmpty())lpData->szString="***";
	//Get the source nick mask
	QString szMask;
	nick.mask(szMask,1);
	if(m_lpOpt->bBeepOnCTCPRequest){
		_macro_beep(50);
		_macro_beep(100);
		_macro_beep(50);
		_macro_beep(100);
		_macro_beep(50);
	}
	target=target.stripWhiteSpace();
	szCTCP=szCTCP.upper();
	
	if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpRequest]->bEnabled){
			QString szPara=nick.szNick.data();
			szPara+=" ";
			szPara+=szMask+" "+szCTCP+" "+lpData->szString;
			if(m_lpFrm->m_lpUserParser->executeEvent(m_lpFrm->m_lpConsole,
								m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpRequest],
								szPara)==KVI_EventErr_Halt)return; //allow disabling the reply
 	}

	if(szCTCP=="ACTION"){ handleCTCPActionRequest(nick.szNick,target,lpData); return; }
	else if(szCTCP=="PAGE"){ handleCTCPPageRequest(nick.szNick,target,lpData); return; }

	//Start the timer anc check for flood
	if(m_lpOpt->iTimerCTCP>=1){
		if(!m_lpCTCPTimer->isActive())m_lpCTCPTimer->start(m_lpOpt->iTimerCTCP * 1000);
		m_iCTCPCount++;
	}
	if((m_iCTCPCount==m_lpOpt->iMaxCTCP)&&(m_lpOpt->iMaxCTCP>3)){
		m_lpPar->topWin()->doFmtOutput(KVI_OUT_FLOOD,i18n("WARNING : CTCP flood detected. Last: CTCP %s from %s [%s]"),
			szCTCP.data(),nick.szNick.data(),szMask.data());

		if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpFlood]->bEnabled){
			QString szPara=nick.szNick.data();
			szPara+=" ";
			szPara+=szMask+" "+szCTCP+" "+lpData->szString;
			m_lpFrm->m_lpUserParser->executeEvent(m_lpFrm->m_lpConsole,
					m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpFlood],szPara);
		}
	
	}

	if((m_iCTCPCount>m_lpOpt->iMaxCTCP)||(m_lpOpt->iTimerCTCP<1)){
		m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_FLOOD,i18n("Ignoring CTCP %s from %s [%s] [%s]"),
			szCTCP.data(),nick.szNick.data(),szMask.data(),lpData->szString.data());
		return;
	}

	if((target[0] != '#')&&(target[0] != '&')){
		KviMdiChild *lpTrg=(m_lpOpt->bCtcpRequestsToActive ? m_lpPar->topWin() : m_lpPar->m_lpCons);
		lpTrg->doFmtOutput(KVI_OUT_CTCP,i18n("CTCP %s request from %s [%s] [%s]"),
					szCTCP.data(),nick.szNick.data(),szMask.data(),lpData->szString.data());
	} else {
		if(szCTCP != "SOUND")m_lpPar->findWin(target.data())->doFmtOutput(KVI_OUT_CTCP,i18n("Channel CTCP %s request from %s [%s] [%s]"),
					szCTCP.data(),nick.szNick.data(),szMask.data(),lpData->szString.data());
		else m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_CTCP,i18n("Channel CTCP SOUND request from %s [%s] [%s] for channel %s"),
					nick.szNick.data(),szMask.data(),lpData->szString.data(),target.data());
	}

	QString szReply;
	if(szCTCP=="PING"){
		replyCTCP(nick,"PING",lpData->szString.data());
	} else if(szCTCP=="VERSION"){
		szReply="Running ";
		szReply+=KVI_VERSION;
		QString szScript=m_lpOpt->szScriptVer.copy();
		m_lpFrm->m_lpUserParser->m_lpIdentifiers->substitute(m_lpFrm->m_lpConsole,szScript,0);
		if(!szScript.isEmpty()){
			szReply+=' ';
			szReply+=szScript;
		}
		replyCTCP(nick,"VERSION",szReply.data());
	} else if(szCTCP=="SOURCE"){
		szReply=KVI_CTCP_SOURCE_REPLY;
		replyCTCP(nick,"SOURCE",szReply.data());
	} else if(szCTCP=="TIME"){
		replyCTCP(nick,"TIME",QDateTime::currentDateTime().toString().data());
	} else if(szCTCP=="CLIENTINFO"){
		szReply=KVI_CLIENTINFO;
		szReply+=" - [Accepts:PING TIME VERSION CLIENTINFO SOURCE USERINFO FINGER DCC [CHAT SEND RESUME] ACTION PAGE SOUND]";
		replyCTCP(nick,"CLIENTINFO",szReply);
	} else if(szCTCP=="USERINFO"){
		if(m_lpOpt->szUserInfo.isEmpty())szReply=m_lpGlb->szUserName+" "+m_lpGlb->szRealName;
		else {
			szReply=m_lpOpt->szUserInfo.copy();
			m_lpFrm->m_lpUserParser->m_lpIdentifiers->substitute(m_lpFrm->m_lpConsole,szReply,0);
		}
		replyCTCP(nick,"USERINFO",szReply);
	} else if(szCTCP=="FINGER"){
		szReply=m_lpOpt->szFinger.copy();
		m_lpFrm->m_lpUserParser->m_lpIdentifiers->substitute(m_lpFrm->m_lpConsole,szReply,0);
		replyCTCP(nick,"FINGER",szReply);
	} else if(szCTCP=="DCC"){
		handleCTCPDCCRequest(nick.szNick,lpData,szMask);
	} else if(szCTCP=="SOUND") {
		QString szWav=lpData->nextWord();
		m_lpPar->findWin(target.data())->doFmtOutput(KVI_OUT_SOUND,i18n("%s plays %s"),nick.szNick.data(),szWav.data());
		if(m_lpOpt->bListenToCtcpSound){
			int nResult=m_lpFrm->m_lpAudio->playSound(szWav);
			if(nResult==KVI_AUDIO_FILE_NOT_FOUND){
				if(m_lpOpt->bSendSoundRequest){
					QString szTo=nick.szNick.data();
					if((target[0]=='#')||(target[0]=='&'))szTo=target.data();
					m_lpSock->sendFmtData("PRIVMSG %s :!%s %s",szTo.data(),nick.szNick.data(),szWav.data());
				}
				if(m_lpOpt->bPlayDefSound){
					szWav="default.wav";
					nResult=m_lpFrm->m_lpAudio->playSound(szWav);
				}
			}
			switch(nResult){
				case KVI_AUDIO_FILE_NOT_FOUND:
					m_lpPar->m_lpCons->doFmtOutput(KVI_OUT_ERROR,i18n("[CTCP SOUND] : Sound file %s not found"),szWav.data());
					break;
				case KVI_AUDIO_NO_SOUND_SERVER:
					m_lpPar->m_lpCons->doOutput(KVI_OUT_ERROR,i18n("[CTCP SOUND] : Sound server not active or not ready."));
					break;
			}
		}
	} else if(szCTCP=="COMMAND"){
		if(m_lpOpt->bEnableActionsOnCtcpCommand){
			KviUserStruct *lpU=m_lpFrm->findUserStruct(nick.szNick,szMask);
			if(lpU){
				if(lpU->bEnableActionOnCtcpCommand){
					KviEventStruct *lpE=new KviEventStruct;
					lpE->szName="Event_OnCtcpCommand";
					lpE->bEnabled=true;
					lpE->szBuffer=lpU->szActionOnCtcpCommand.copy();
					QString szParam=lpU->szNick.data();
					szParam+=' ';
					szParam+=lpU->szNick;
					szParam+='!';
					szParam+=szMask;
					szParam+=' ';
					szParam+=lpData->szString;
					m_lpFrm->m_lpUserParser->executeEvent(m_lpPar->m_lpCons,lpE,szParam);
					delete lpE;
				}
			}
		}
	} else { //Can't handle it...
		replyCTCPErrormsg(nick.szNick,"Huh ??? [ Try CLIENTINFO for a list of CTCP types that this client can handle ]");
		m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,
			i18n("%s: %s :Unknown CTCP type ... replying warning msg.(Huh???)"),nick.szNick.data(),szCTCP.data());
 	}	
}

//============ replyCTCP ============//

void KviCTCP::replyCTCP(KviNewNick &nick,const char *szType,const char * szReply)
{
	_debug_entertrace("replyCTCP");
	m_lpSock->sendFmtData("NOTICE %s :%c%s %s%c",nick.szNick.data(),0x01,szType,szReply,0x01);
	_debug_leavetrace("replyCTCP");
}


void KviCTCP::resetCTCPCount(){
	m_lpCTCPTimer->stop();
	m_iCTCPCount=0;
}

void KviCTCP::handleCTCPReply(KviNewNick &nick,QString &target,KviIrcToken *lpData){
	lpData->szString.remove(0,1);
	lpData->szString.remove(lpData->szString.length()-1,1);
	QString szCTCP=lpData->nextWord();

	if(lpData->szString.isEmpty())lpData->szString="***";
	QString szMask;
	nick.mask(szMask,1);
	target=target.stripWhiteSpace();
	szCTCP=szCTCP.upper();

	if(m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpReply]->bEnabled){
			QString szPara=nick.szNick.data();
			szPara+=" ";
			szPara+=szMask;
			szPara+=" ";
			szPara+=szCTCP;
			szPara+=" ";
			szPara+=lpData->szString.data();
			m_lpFrm->m_lpUserParser->executeEvent(m_lpFrm->m_lpConsole,
					m_lpFrm->m_lpEventManager->lpEvent[KVI_Event_OnCtcpReply],szPara);
	}

	if(szCTCP=="ACTION"){ handleCTCPActionRequest(nick.szNick,target,lpData); return; }
	else if(szCTCP=="PAGE"){ handleCTCPPageRequest(nick.szNick,target,lpData); return; }

	if(szCTCP != "PING"){
		m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,i18n("CTCP %s reply from %s [%s]:%s"),
					szCTCP.data(),nick.szNick.data(),szMask.data(),lpData->szString.data());
	} else {
		uint oldTime = (uint)(atoi(lpData->szString.data()));
		time_t curTime = time(0);
		uint elapsed = (uint)curTime - oldTime;
		m_lpPar->topWin()->doFmtOutput(KVI_OUT_CTCP,i18n("CTCP PING reply from %s [%s]:%u seconds"),
					nick.szNick.data(),szMask.data(),elapsed);
	}
}
#include "m_kvi_ctcp.moc"

//
// $Log: kvi_ctcp.cpp,v $
// Revision 1.11  1998/10/06 14:42:10  pragma
// Tons of changes
//
// Revision 1.10  1998/10/01 00:53:39  pragma
// Ctcp requests optionally to the active window
//
// Revision 1.9  1998/09/28 13:55:46  pragma
// Minor changes and bugfixes all around.
//
// Revision 1.8  1998/09/23 12:41:35  pragma
// Another big commit.
// Added user list (Still have to fix ignore things)
// User list dialogs
// Some code rearrangements
// Minor bugfixes
//
// Revision 1.7  1998/09/20 20:22:21  fritz
// reorganized includes.
// More work on srvdlg - still not finished.
//
// Revision 1.6  1998/09/17 19:29:55  fritz
// First translations.
// Fixed some typos and inconsistent messages.
//
// Revision 1.5  1998/09/17 16:14:31  pragma
// Moving to handle mp3 , au and mid files by external players.
//
// Revision 1.4  1998/09/17 12:53:58  pragma
// In a way to handle CTCP sound mp3,au and mid.
// (Huge user requests)
//
// Revision 1.3  1998/09/16 17:16:07  fritz
// Starting i18n.
//
// Revision 1.2  1998/09/16 16:13:10  pragma
// Moving to use a dynamic kvirc home directory.
// Big commit :)
//
//
