//   $Id: kvi_link.cpp,v 1.3 1998/10/29 14:25:00 pragma Exp $
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1998 Szymon Stefanek (stefanek@tin.it)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public
//   License as published by the Free Software Foundation; either
//   version 2 of the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Library General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; see the file COPYING.  If not, write to
//   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//   Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CLASS_NAME_ "KviLinkWnd"

#include "kvi_defs.h"
#include "kvi_macros.h"
#include "kvi_link.h"
#include "kvi_debug.h"
#include "kvi_view.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_socket.h"
#include "kvi_mdi.h"
#include "kvi_taskbar.h"
#include "kvi_opt.h"
#include "kvi_int.h"
#include "kvi_support.h"

#include <stdarg.h>

//============ KviListViewItem ============//

KviListViewItem::KviListViewItem(QListView *parent,const char *str1,const char *str2,const char *str3,const char *str4):
	QListViewItem(parent,str1,str2,str3,str4)
{
	_debug_entertrace("KviListViewItem");
	_debug_leavetrace("KviListViewItem");
}
KviListViewItem::KviListViewItem(KviListViewItem *parent,const char *str1,const char *str2,const char *str3,const char *str4):
	QListViewItem(parent,str1,str2,str3,str4)
{
	_debug_entertrace("KviListViewItem");
	_debug_leavetrace("KviListViewItem");
}
//============ ~KviListViewItem ============//
KviListViewItem::~KviListViewItem()
{
	_debug_entertrace("~KviListViewItem");
	_debug_leavetrace("~KviListViewItem");
}
//============ paintBranches ============//
void KviListViewItem::paintBranches(QPainter *p,const QColorGroup &cg,int w,int y,int h,GUIStyle)
{
	_debug_entertrace("paintBranches");
	p->fillRect( 0, 0, w, h, cg.base() );
	QListViewItem * child = firstChild();
	int linebot = 0;
	// skip the stuff above the exposed rectangle
	while ( child && y + child->height() <= 0 ) {
		y += child->totalHeight();
		child = child->nextSibling();
	}
	int bx = w / 2;
	// paint stuff in the magical area
	while ( child && (y < h) ) {
		linebot = y + child->height()/2;
		if ( child->isExpandable() || child->childCount() ) {
			int x = bx - 4;
			int y = linebot - 4;
			int d = 9;
			QPointArray *a=new QPointArray();
			if ( child->isOpen() ) {
				// DownArrow
				a->setPoints( 3, x, y, x+d, y, x+d/2, y+d );
			} else {
			    //RightArrow
				a->setPoints( 3, x, y, x, y+d, x+d, y+d/2 );
			}
			p->setPen( cg.foreground() );
			p->drawPolygon( *a );
			delete a;
		}
		y += child->totalHeight();
		child = child->nextSibling();
    }
	_debug_leavetrace("paintBranches");
}



KviLinkWnd::KviLinkWnd(KviMdiManager *parent,KviFrame *frame,int aid)
				:KviMdiChild(parent,frame,KVI_LINKS_WND_NAME,aid,KVI_WND_TYPE_LINKS)
{
	m_lpPanner=new KNewPanner(this,"KviPanner",KNewPanner::Horizontal,KNewPanner::Percent,70);
	m_lpView = new QListView(m_lpPanner);
	m_lpOutput=new KviView(m_lpPanner,frame,"KviViewClass");
	m_lpPanner->activate(m_lpView,m_lpOutput);
	m_lpView->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	m_lpView->addColumn(i18n("Host"));
	m_lpView->addColumn(i18n("Link"));
	m_lpView->addColumn(i18n("Hops"));
	m_lpView->addColumn(i18n("Description"));
	setFocusOwner(m_lpView);
	connect(this,SIGNAL(closeButtonPressed()),this,SLOT(closeSlot()));
	applyOptions();
	if(!m_lpFrm->m_lpSock->m_bConnected)return;
	m_bDone=false;
	m_lpLinkList=new QList<KviLinkStruct>;
	m_lpLinkList->setAutoDelete(true);
	if(m_lpFrm->m_lpSock->sendData("LINKS")){
		m_bDone=true;
		doOutput(KVI_OUT_ERROR,i18n("Unable to get links...no connection active."));
	}
	doOutput(KVI_OUT_INTERNAL,i18n("Requesting links from server..."));
}

KviLinkWnd::~KviLinkWnd()
{
	while(!m_lpLinkList->isEmpty())m_lpLinkList->removeLast();
	delete m_lpLinkList;
	m_lpView->clear();
	delete m_lpView;
	delete m_lpOutput;
	delete m_lpPanner;
}

//============ closeSlot ============//

void KviLinkWnd::closeSlot()
{
	_debug_entertrace("closeSlot");
	hide();
	_macro_kviApplication->processEvents();
	blockSignals(true);
	m_lpMdi->m_lpTaskBar->removeButton(id());
	m_lpMdi->m_lpChildList->setAutoDelete(false);
	m_lpMdi->m_lpChildList->removeRef(this);
	m_lpMdi->focusTopChild();
	m_lpMdi->fillWinListPopup();
	m_lpFrm->queryExternalDestroy(this);
	_debug_leavetrace("closeSlot");
}

void KviLinkWnd::applyOptions(){
	m_lpOutput->m_bShowPixmaps=m_lpFrm->m_lpOpt->bShowPixmaps;
	m_lpOutput->m_bTimestamp=m_lpFrm->m_lpOpt->bTimestamp;
	m_lpOutput->setFont(m_lpInt->fnt_output);
	const QSize sz=size();
	QResizeEvent e(sz,sz);
	resizeEvent(&e);
}

void KviLinkWnd::resizeEvent(QResizeEvent *){
	_debug_entertrace("resizeEvent");
	updateRects();
	QRect rct=viewSizeHint();
	m_lpPanner->setGeometry(rct);
	_debug_leavetrace("resizeEvent");
}
void KviLinkWnd::doFmtOutput(int nType,const char *szFmt,...){
	char szText[600]; //It should be big enough... I hope...
	va_list list;
	va_start(list,szFmt);
	if(vsnprintf(szText,600,szFmt,list)==-1)debug("WARNING : Output string truncated"); // Fritz: vsnprintf
	va_end(list);
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,(const char *) &szText);
}
void KviLinkWnd::doOutput(int nType,const char *szText){
	m_lpMdi->highlightWindow(m_iId);
	m_lpOutput->appendText(nType,szText);
}
//============ getChildItem ============//
KviListViewItem *KviLinkWnd::getChildItem(KviListViewItem *parent,const char *parenthost)
{
	_debug_entertrace("getChildItem");
	if(!strcasecmp(parent->text(0),parenthost))return parent;
	else {
		KviListViewItem *lpI=(KviListViewItem *)parent->firstChild();
		if(lpI){
			KviListViewItem *lpX=getChildItem(lpI,parenthost);
			if(lpX)return lpX;
		}
		lpI=(KviListViewItem *)parent->nextSibling();
		if(lpI){
			KviListViewItem *lpX=getChildItem(lpI,parenthost);
			if(lpX)return lpX;
		}
	}
	return 0;
	_debug_leavetrace("getChildItem");
}
//============ getItemByHost ============//
KviListViewItem *KviLinkWnd::getItemByHost(const char *parenthost)
{
	_debug_entertrace("getItemByHost");
	KviListViewItem *lpI=(KviListViewItem *)m_lpView->firstChild();
	if(!lpI)return 0;
	return getChildItem(lpI,parenthost);
	_debug_leavetrace("getItemByHost");
}
//============ endOfLinks ============//
void KviLinkWnd::endOfLinks()
{
	_debug_entertrace("endOfLinks");
	m_bDone=true;
	doOutput(KVI_OUT_INTERNAL,i18n("Received END OF LINKS : processing internal list."));
	uint nHosts=m_lpLinkList->count();
	int nMaxHops=0;
	int totHops=0;
	int nDirect=0;
	int nNear=0;
	int nFar=0;
	while(!m_lpLinkList->isEmpty()){
		KviLinkStruct *lpL=m_lpLinkList->first();
		if(lpL->nHops>nMaxHops)nMaxHops=lpL->nHops;
		if(lpL->nHops==1)nDirect++;
		if(lpL->nHops<4)nNear++;
		else nFar++;
		totHops+=lpL->nHops;
		insertLinkInternal(lpL->host.data(),lpL->parenthost.data(),lpL->hops.data(),lpL->description.data());
		m_lpLinkList->removeFirst();
	}
	doOutput(KVI_OUT_INTERNAL,i18n("Completed."));
	doOutput(KVI_OUT_INTERNAL,i18n("Network statistics:"));
	doOutput(KVI_OUT_INTERNAL,"-----------------------------------------------");
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Hosts in the network         : %u"),nHosts);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Total hops from root server  : %d"),totHops);
	if(nHosts)doFmtOutput(KVI_OUT_INTERNAL,i18n("Average hops number          : %d"),totHops/nHosts);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Max hops number              : %d"),nMaxHops);
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Direct links                 : %d"),nDirect);
	if(nNear)doFmtOutput(KVI_OUT_INTERNAL,i18n("Near links (<4 hops)         : %d (~%d %%)"),nNear,(nNear*100)/nHosts);
	else doFmtOutput(KVI_OUT_INTERNAL,i18n("Near links (<4 hops)         : %d (0 %%)"),nNear);
	if(nFar)doFmtOutput(KVI_OUT_INTERNAL,i18n("Far links (>=4 hops)         : %d (~%d %%)"),nFar,(nFar*100)/nHosts);
	else doFmtOutput(KVI_OUT_INTERNAL,i18n("Far links (>=4 hops)         : %d (0 %%)"),nFar);
	_debug_leavetrace("endOfLinks");
}
void KviLinkWnd::insertLink(const char *host,const char *parenthost,const char *description)
{
	doFmtOutput(KVI_OUT_INTERNAL,i18n("Received link : %s-%s : %s"),host,parenthost,description);
	KviLinkStruct *lpL = new KviLinkStruct;
	lpL->host=host;
	lpL->parenthost=parenthost;
	lpL->description=description;
	int idx=lpL->description.find(' ');
	if(idx != -1){
		lpL->hops=lpL->description.left(idx);
		lpL->description.remove(0,idx+1);
		bool bOk=false;
		lpL->nHops=lpL->hops.toInt(&bOk);
		if(!bOk)lpL->nHops=100;
	} else {
		lpL->hops=i18n("Unknown hops number");
		lpL->nHops=100;
	}
	//insert sorted by hops...the first letter in the description should be the hops number.
	KviLinkStruct *lpX=0;
	bool bFound=false;
	idx=0;
	for(lpX=m_lpLinkList->first();lpX && (!bFound);lpX=m_lpLinkList->next()){
		if(lpX->nHops > lpL->nHops){
			//we insert here
			bFound=true;
			m_lpLinkList->insert(idx,lpL);
		}
		idx++;
	}
	if(!bFound){
		m_lpLinkList->append(lpL);
	}
}
//============ insertLink ============//
void KviLinkWnd::insertLinkInternal(const char *host,const char *parenthost,const char *hops,const char *description)
{
	_debug_entertrace("insertLink");
	if(!m_lpView->firstChild()){
		//first link
		KviListViewItem *lpI=new KviListViewItem(m_lpView,host,i18n("Root server"),"0",description);
		lpI->setOpen(true);
		return;
	}
	KviListViewItem *lpP=getItemByHost(parenthost);
	if(!lpP){
		doFmtOutput(KVI_OUT_ERROR,i18n("WARNING : Misordered link from server : %s %s %s %s"),host,parenthost,hops,description);
		return;
	}
	QString szLnk=parenthost;
	szLnk+="-";
	szLnk+=host;
	KviListViewItem *lpI=new KviListViewItem(lpP,host,szLnk.data(),hops,description);
	lpI->setOpen(true);
	_debug_leavetrace("insertLink");
}
#include "m_kvi_link.moc"
