/*
 * cdcc.c xdcc in C.  This file tries to recreate the script xdcc in C,
 * this file contains all the functions needed to implement this.
 *
 * Written by Scott H Kilau
 *
 * Heavily modified by Flier
 *
 * Copyright(c) 1995
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT
 */

#ifdef BSD
#include <db.h>
#endif
#include <dirent.h>
#include <sys/file.h>
#include <sys/stat.h>
#include "irc.h"
#include "list.h"
#include "server.h"
#include "vars.h"
#include "ircaux.h"
#include "input.h"
#include "window.h"
#include "screen.h"
#include "output.h"
#include "edit.h"
#include "dcc.h"
#include "cdcc.h"
#include "myvars.h"

void Cdcc _((char *, char *, char *));
void CheckCdcc _((char *, char *, char *, char *));
void CheckAutoGet _((char *, char *, char *, char *));
void CheckDccIdleSends _((void));
void RemoveFromQueue _((void));
void helpmcommand _((char *));
int  matchmcommand _((char *, int));
static void sendmcommand _((char *));
static void send2mcommand _((char *, char *));
static void send3mcommand _((char *, char *));
static void resendmcommand _((char *));
static void resend2mcommand _((char *, char *));
static void resend3mcommand _((char *, char *));
static void getmcommand _((char *));
static void get2mcommand _((char *, char *));
static void doffermcommand _((char *));
static void doffer2mcommand _((char *, char *));
static void offermcommand _((char *));
static void offer2mcommand _((char *, char *));
static void offer3mcommand _((char *, char *));
static void offer4mcommand _((char *, char *));
#ifdef EXTRAS
static void renamepackmcommand _((char *));
#endif
static void listmcommand _((char *));
static void plistmcommand _((char *));
static void noticemcommand _((char *));
static void listcommand _((char *, char *));
static void helpcommand _((char *, char *));
static void sendcommand _((char *, char *));
static void versioncommand _((char *, char *));
static void psendmcommand _((char *));
static void psend2mcommand _((char *, char *));
static void psend3mcommand _((char *, char *));
static void autogetmcommand _((char *));
static void securemcommand _((char *));
static void closemcommand _((char *));
static void close2mcommand _((char *, char *));
#ifdef EXTRA_STUFF
static void emcommand _((char *));
static void mmcommand _((char *));
#endif
/****** Coded by Zakath ******/
static void requestmcommand _((char *));
/*****************************/
static void idlemcommand _((char *));
static void limitmcommand _((char *));
static void channelsmcommand _((char *));
static void ptimemcommand _((char *));
static void longstatusmcommand _((char *));
static void uldirmcommand _((char *));
static void dldirmcommand _((char *));
static void showdccsmcommand _((int));
static void loadmcommand _((char *));
static void savemcommand _((char *));
static void statusmcommand _((char *));
static void statsmcommand _((char *));
static void queuemcommand _((char *));
static void GetDir _((char *));
static void CleanList _((void));
static void AddFileToList _((char *, char *, int));
static int  AddFiles2List _((char *));
static void AddToOfferList _((char *, char *));
static void ShowPacks _((char *));
static void DeleteSend _((void));
static void AddToQueue _((Files *, char *, int));
static int  TotalSendDcc _((void));
static int  SeedFiles _((char *, int));
static int  GetSize _((char *, char *));
static int  compar _((struct dirent **, struct dirent **));
static int  selectent _((struct dirent *));

extern void AwaySave _((char *, int));
extern void PrintSetting _((char *, char *, char *, char *));
extern void NoWindowChannel _((void));
extern void PrintUsage _((char *));
extern void NumberCommand _((char *, char *, char *));
extern void OnOffCommand _((char *, char *, char *));
extern int  CheckChannel _((char *, char *));
extern char *OpenCreateFile _((char *, int));
extern struct friends *CheckUsers _((char *, char *));
extern void ColorUserHost _((char *, char *, char *));

extern void dcc_close _((char *));
extern void dcc_getfile _((char *));
extern void dcc_regetfile _((char *));
extern void dcc_filesend _((char *));
extern void dcc_resend _((char *));

static Packs *packs=(Packs *) 0;
static Files *files=(Files *) 0;
static FileQueue *queuelist=(FileQueue *) 0;
static struct dirent **CdccFileNames=NULL;
static struct stat CdccStatBuf;
static int CdccEntries;
static int c_entry_size;
static char *CdccString="\002[\002sz\002]\002";
static time_t LastIdleCheck=0;
/****** Coded by Zakath ******/
static int  CdccReqTog=0;
static char *CdccRequest=(char *) 0;
/*****************************/

static CdccCom CdccCommands[]={
    { "HELP",        helpmcommand },
    { "GET",         getmcommand },
    { "SEND",        sendmcommand },
    { "CLOSE",       closemcommand },
    { "RESEND",      resendmcommand },
    { "OFFER",       offermcommand },
    { "DOFFER",      doffermcommand },
    { "PLIST",       plistmcommand },
    { "LIST",        listmcommand },
    { "NOTICE",      noticemcommand },
#ifdef EXTRAS
    { "RENPACK",     renamepackmcommand },
#endif
/****** Coded by Zakath ******/
    { "REQUEST",     requestmcommand },
/*****************************/
    { "QUEUE",       queuemcommand },
    { "LOAD",        loadmcommand },
    { "SAVE",        savemcommand },
    { "LIMIT",       limitmcommand },
    { "CHANNELS",    channelsmcommand },
    { "PTIME",       ptimemcommand },
    { "LONGSTATUS",  longstatusmcommand },
    { "PSEND",       psendmcommand },
    { "IDLE",        idlemcommand },
    { "AUTOGET",     autogetmcommand },
    { "SECURE",      securemcommand },
    { "STATUS",      statusmcommand },
    { "STATS",       statsmcommand },
    { "ULDIR",       uldirmcommand },
    { "DLDIR",       dldirmcommand },
#ifdef EXTRA_STUFF
    { "E",           emcommand },
    { "M",           mmcommand },
#endif
    { NULL,          NULL }
};

extern char *dcc_types[];
extern char cdcc[];
extern char bold;
extern DCC_list *ClientList;

/************************************************************************
 * Cdcc: parse cdcc command line, and send off to correct function      *
 ************************************************************************/
void Cdcc(command, args, subargs)
char *command;
char *args;
char *subargs;
{
    char *word=(char *) 0;
    char *tmpstr;
    int  len=0;
    int  found=0;
    int  i;
    int  com=0;

    if (!(args && *args)) {
        showdccsmcommand(15);
        return;
    }
    word=next_arg(args,&args);
    len=strlen(word);
    for (i=0;CdccCommands[i].command && CdccCommands[i].function;i++)
        if (!my_strnicmp(CdccCommands[i].command,word,len)) {
            found++;
            if (!com) com=i;
        }
    if (found>1) {
        for (tmpstr=word;*tmpstr;tmpstr++) if (*tmpstr>='a' && *tmpstr<='z') *tmpstr-=' ';
        say("CDCC %s is ambiguous",word);
        return;
    }
    if (found) CdccCommands[com].function(args);
    else say("Try  /CDCC HELP");
}

/*********************************************************************
 * Gives user help                                                   *
 *********************************************************************/
void helpmcommand(line)
char *line;
{
    PrintUsage("/CDCC command where command is one of :");
    say("AUTOGET  CHAN     CLOSE   DLDIR   DOFFER  GET     IDLE   LIMIT  LIST");
    say("LOAD     LONGST   NOTICE  OFFER   PLIST   PTIME   PSEND  QUEUE  RENPACK");
    say("RESEND   REQUEST  SAVE    SECURE  SEND    STATUS  ULDIR");
    say("For more help on command do /SHELP CDCC command");
}

/***********************************************************************
 * listmcommand: List packs                                            *
 ***********************************************************************/
void listmcommand(line)
char *line;
{
    ShowPacks(line);
}

/***********************************************************************
 * Sets CDCC limit                                                     *
 ***********************************************************************/
void limitmcommand(line)
char *line;
{
    NumberCommand("LIMIT",line,NULL);
}

/***********************************************************************
 * Sets idle seconds before auto-close                                 *
 ***********************************************************************/
void idlemcommand(line)
char *line;
{
    NumberCommand("IDLE",line,NULL);
}

/***********************************************************************
 * Sets CDCC autoget on or off                                         *
 ***********************************************************************/
void autogetmcommand(line)
char *line;
{
    OnOffCommand("AUTOGET",line,NULL);
    update_all_status();
}

/***********************************************************************
 * Sets CDCC security on or off                                        *
 ***********************************************************************/
void securemcommand(line)
char *line;
{
    OnOffCommand("SECURE",line,NULL);
    update_all_status();
}

#ifdef EXTRA_STUFF
/**********************************************************************
* Sets CDCC encode string                                             *
***********************************************************************/
void emcommand(line)
char *line;
{
    if (*line) malloc_strcpy(&EString,line);
    PrintSetting("Cdcc E",EString,empty_string,empty_string);
}

/**********************************************************************
* Sets CDCC renaming on or off                                        *
***********************************************************************/
void mmcommand(line)
char *line;
{
    OnOffCommand("M",line,NULL);
}
#endif

/**********************************************************************
* Sets CDCC channels for PLIST                                        *
***********************************************************************/
void channelsmcommand(line)
char *line;
{
    if (line && *line) malloc_strcpy(&CdccChannels,line);
    if (CdccChannels) {
        if (!my_stricmp(CdccChannels,"current"))
            PrintSetting("Cdcc channels","current channel",empty_string,empty_string);
        else PrintSetting("Cdcc channels",CdccChannels,empty_string,empty_string);
    }
    else PrintSetting("Cdcc channels","none",empty_string,empty_string);
}

/***********************************************************************
 * Sets seconds to pass before repeating PLIST                         *
 ***********************************************************************/
void ptimemcommand(line)
char *line;
{
    NumberCommand("PTIME",line,NULL);
}

/***********************************************************************
 * Sets CDCC long status bar on or off                                 *
 ***********************************************************************/
void longstatusmcommand(line)
char *line;
{
    OnOffCommand("LONGSTATUS",line,NULL);
}

/***********************************************************************
 * Sets showing DCC status on status bar on or off                     *
 ***********************************************************************/
void statusmcommand(line)
char *line;
{
    OnOffCommand("STATUS",line,NULL);
    DCCDone=0;
    new_free(&CurrentDCC);
    update_all_status();
}

/***********************************************************************
 * Sets showing received/sent kB in PLIST on or off                    *
 ***********************************************************************/
void statsmcommand(line)
char *line;
{
    OnOffCommand("STATS",line,NULL);
}

/***********************************************************************
 * Sets CDCC upload dir                                                *
 ***********************************************************************/
void uldirmcommand(line)
char *line;
{
    char *fullname=(char *) 0;
    char tmpbuf[mybufsize/4];

    getcwd(tmpbuf,mybufsize);
    if (*line) {
        fullname=expand_twiddle(line);
        if (!chdir(fullname)) malloc_strcpy(&CdccUlDir,fullname);
        else {
#ifdef WANTANSI
            say("%sError%s, can't cd into %s%s%s",
                CmdsColors[COLWARNING].color1,Colors[COLOFF],
                CmdsColors[COLSETTING].color2,line,Colors[COLOFF]);
#else
            say("Error, can't cd into %s",line);
#endif
        }
        new_free(&fullname);
        chdir(tmpbuf);
    }
    if (CdccUlDir) PrintSetting("Cdcc upload dir",CdccUlDir,empty_string,empty_string);
    else PrintSetting("Cdcc upload dir",tmpbuf," - your current dir",empty_string);
}

/***********************************************************************
 * Sets CDCC download dir                                              *
 ***********************************************************************/
void dldirmcommand(line)
char *line;
{
    char *fullname=(char *) 0;
    char tmpbuf[mybufsize/4];

    getcwd(tmpbuf,mybufsize);
    if (*line) {
        fullname=expand_twiddle(line);
        if (!chdir(fullname)) malloc_strcpy(&CdccDlDir,fullname);
        else {
#ifdef WANTANSI
            say("%sError%s, can't cd into %s%s%s",
                CmdsColors[COLWARNING].color1,Colors[COLOFF],
                CmdsColors[COLSETTING].color2,line,Colors[COLOFF]);
#else
            say("Error, can't cd into %s",line);
#endif
        }
        new_free(&fullname);
        chdir(tmpbuf);
    }
    if (CdccDlDir) PrintSetting("Cdcc download dir",CdccDlDir,empty_string,empty_string);
    else PrintSetting("Cdcc download dir",tmpbuf," - your current dir",empty_string);
}

/***********************************************************************
 * showdccscommand: Lists all dccs                                     *
 ***********************************************************************/
void showdccsmcommand(type)
int type;
{
    int   fills;
    int   count=0,i,j;
    long  completed;
    char  *filename;
    char  *format;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/4];
    char  tmpbuf3[mybufsize/4];
    char  tmpbuf4[mybufsize/32];
    float rate;
    time_t timenow;
    time_t etatime;
    unsigned flags;
    DCC_list *Client;

    if (LongStatus) {
        format="%s %-7.7s %-9.9s %-2.2s %7s %s %s";
        say(format,"\002# ","Type","Nick","St","kb/s","   ETA ","Arguments\002");
    }
    else {
        format="%-2s %-7.7s %-9.9s %-2.2s %7s %s %s %s";
        say(format,"\002# ","Type","Nick","St","kb/s","Completed ","   ETA ","Arguments\002");
    }
    for (Client=ClientList;Client;Client=Client->next) {
        flags=Client->flags;
        if (type==15 || (flags&type)==type) {
            *tmpbuf1='\0';
            *tmpbuf2='\0';
            timenow=time((time_t *) 0);
            completed=0;
            flags&=DCC_TYPES;
            if (flags==DCC_FILEREAD) completed=Client->bytes_read;
            else if (flags==DCC_FILEREGET)
                completed=Client->bytes_read+Client->resendoffset;
            else if (flags==DCC_FILEOFFER) completed=Client->bytes_sent;
            else if (flags==DCC_RESENDOFFER)
                completed=Client->bytes_sent+Client->resendoffset;
            if (completed) {
                if (LongStatus) {
                    strcpy(tmpbuf1,"\026    ");
                    for (fills=1;fills<7;fills++) strcat(tmpbuf1,"          ");
                    fills=0;
                    if (Client->filesize>0) {
                        if (Client->filesize>=10000000) fills=completed/(Client->filesize/100);
                        else fills=completed*100/Client->filesize;
                    }
                    sprintf(tmpbuf2,"%3d%%  (%ld of %ld bytes)",fills,completed,Client->filesize);
                    fills=(fills+1)*63/100;
                    for (i=0,j=0;i<=fills;i++)
                        if (tmpbuf2[j] && i>14) {
                            tmpbuf1[i+1]=tmpbuf2[j];
                            j++;
                        }
                        else tmpbuf1[i+1]=' ';
                    if (!i) i++;
                    if (tmpbuf1[i]!=' ') tmpbuf1[i+1]=tmpbuf1[i];
                    tmpbuf1[i]='\026';
                    i++;
                    if (i<16) i=16;
                    while (tmpbuf2[j]) {
                        tmpbuf1[i+1]=tmpbuf2[j];
                        i++;
                        j++;
                    }
                }
                else {
                    strcpy(tmpbuf1,"\026___________");
                    fills=0;
                    if (Client->filesize>0) {
                        if (Client->filesize>=10000000) fills=completed/(Client->filesize/100);
                        else fills=completed*100/Client->filesize;
                    }
                    sprintf(tmpbuf2,"%d%%",fills);
                    fills=(fills+4)/10;
                    for (i=0,j=0;i<=fills;i++)
                        if (tmpbuf2[j] && i>3) {
                            tmpbuf1[i+1]=tmpbuf2[j];
                            j++;
                        }
                        else tmpbuf1[i+1]=' ';
                    if (!i) i++;
                    if (tmpbuf1[i]!='_' && tmpbuf1[i]!=' ') tmpbuf1[i+1]=tmpbuf1[i];
                    tmpbuf1[i]='\026';
                    i++;
                    if (i<5) i=5;
                    while (tmpbuf2[j]) {
                        tmpbuf1[i+1]=tmpbuf2[j];
                        i++;
                        j++;
                    }
                }
                strcpy(tmpbuf2,"  N/A");
                flags=Client->flags;
                if (flags&DCC_ACTIVE && timenow-Client->starttime>0) {
                    flags&=DCC_TYPES;
                    if (flags==DCC_FILEREGET || flags==DCC_RESENDOFFER)
                        completed-=Client->resendoffset;
                    rate=(float) (completed)/(float)(timenow-Client->starttime);
                    sprintf(tmpbuf2,"%6.2f",rate/1024.0);
                    if (rate>0.0 && completed<=Client->filesize) {
                        etatime=(float) (((float) (Client->filesize)-(float) completed)/(float) rate);
                        sprintf(tmpbuf3,"%3ld:%02ld ",etatime/60,etatime%60);
                    }
                }
                else strcpy(tmpbuf3,"   N/A ");
            }
            else {
                if (LongStatus) {
                    strcpy(tmpbuf2,"  N/A");
                    strcpy(tmpbuf3,"   N/A ");
                }
                else {
                    strcpy(tmpbuf1,"      N/A ");
                    strcpy(tmpbuf2,"  N/A");
                    strcpy(tmpbuf3,"   N/A ");
                }
            }
            count++;
            sprintf(tmpbuf4,"%-2d",count);
            filename=rindex(Client->description,'/');
            if (!filename) filename=Client->description;
            else filename++;
            if (!(*tmpbuf2)) strcpy(tmpbuf2,"      ");
            if (LongStatus) {
                flags=Client->flags;
                say(format,tmpbuf4,dcc_types[flags&DCC_TYPES],Client->user,
                    flags&DCC_OFFER?"O" :
                    flags&DCC_DELETE?"D" :
                    flags&DCC_ACTIVE?"A" :
                    flags&DCC_WAIT?"W" :
#ifdef DCC_CNCT_PEND
                    flags&DCC_CNCT_PEND?"C" :
#endif
                    "U",tmpbuf2,tmpbuf3,filename);
                flags&=DCC_TYPES;
                if (completed &&
                    (flags==DCC_FILEREAD  || flags==DCC_FILEREGET ||
                     flags==DCC_FILEOFFER || flags==DCC_RESENDOFFER))
                    say("[%s]",tmpbuf1);
            }
            else {
                flags=Client->flags;
                say(format,tmpbuf4,dcc_types[flags&DCC_TYPES],Client->user,
                    flags&DCC_OFFER?"O" :
                    flags&DCC_DELETE?"D" :
                    flags&DCC_ACTIVE?"A" :
                    flags&DCC_WAIT?"W" :
#ifdef DCC_CNCT_PEND
                    flags&DCC_CNCT_PEND?"C" :
#endif
                    "U",tmpbuf2,tmpbuf1,tmpbuf3,filename);
            }
        }
    }
}

/***********************************************************************
 * matchmcommand: Returns true if count matches line                   *
 ***********************************************************************/
int matchmcommand(origline,count)
char *origline;
int  count;
{
    int  startnum=0;
    int  endnum=0;
    register char *tmpstr;

    tmpstr=origline;
    if (!tmpstr) return(0);
    if (*tmpstr=='#') tmpstr++;
    if (*tmpstr=='*') return(1);
    while (*tmpstr) {
        startnum=0;
        endnum=0;
        if (*tmpstr=='-') {
            while (*tmpstr && !isdigit(*tmpstr)) tmpstr++;
            endnum=atoi(tmpstr);
            startnum=1;
            while (*tmpstr && isdigit(*tmpstr)) tmpstr++;
        }
        else {
            while (*tmpstr && !isdigit(*tmpstr)) tmpstr++;
            startnum=atoi(tmpstr);
            while (*tmpstr && isdigit(*tmpstr)) tmpstr++;
            if (*tmpstr=='-') {
                while (*tmpstr && !isdigit(*tmpstr)) tmpstr++;
                endnum=atoi(tmpstr);
                if (!endnum) endnum=1000;
                while (*tmpstr && isdigit(*tmpstr)) tmpstr++;
            }
        }
        if (count==startnum || (count>=startnum && count<=endnum)) return(1);
    }
    if (count==startnum || (count>=startnum && count<=endnum)) return(1);
    return(0);
}

/**********************************************************************
* closemcommand: Prompt User for type of dccs he wants to close       *
***********************************************************************/
void closemcommand(line)
char *line;
{
    if (!ClientList) {
        say("No dccs to close");
        return;
    }
    if (line && *line) close2mcommand(NULL,line);
    else {
        showdccsmcommand(15);
        add_wait_prompt("What to close (1-6,3 or *) ? ",close2mcommand,line,WAIT_PROMPT_LINE);
    }
}

/**********************************************************************
* close2mcommand  This closes all user specified dccs                 *
***********************************************************************/
void close2mcommand(blah,line)
char *blah;
char *line;
{
    int  count=0;
    int  packcount=0;
    char tmpbuf[mybufsize/4];
    DCC_list *Client;
    unsigned flags;

    if (line && *line) {
        for (Client=ClientList;Client;Client=Client->next) {
            packcount++;
            flags=Client->flags;
            if (matchmcommand(line,packcount)) {
                count++;
                sprintf(tmpbuf,"%s %s %s",dcc_types[flags&DCC_TYPES],Client->user,Client->description);
                dcc_close(tmpbuf);
            }
        }
        say("Total of %d dccs closed",count);
        RemoveFromQueue();
    }
    else say("You must specify what to close");
}

/**********************************************************************
* doffermcommand: Prompt User for removing pack                       *
***********************************************************************/
void doffermcommand(line)
char *line;
{
    if (packs) {
        if (line && *line) doffer2mcommand(NULL,line);
        else {
            ShowPacks(NULL);
            add_wait_prompt("Doffer what pack (1-6,3 or * for all) ? ",doffer2mcommand,line,WAIT_PROMPT_LINE);
        }
    }
    else say("No packs created");
}

/**********************************************************************
* doffer2mcommand  This parses offer file list                        *
*                  And puts in Files Linked List                      *
***********************************************************************/
void doffer2mcommand(blah,line)
char *blah;
char *line;
{
    int   packcount=0;
    Files *tmp2;
    Files *next;
    Packs *tmp;
    Packs *tmp1;
    Packs *tmp3;
    char  *tmpstr=(char *) 0;

    tmpstr=next_arg(line,&line);
    if (tmpstr) {
        for (tmp=packs;tmp;tmp=tmp3) {
            tmp3=tmp->next;
            packcount++;
            if (matchmcommand(tmpstr,packcount)) {
                if ((tmp1=(Packs *) list_lookup((List **) &packs,tmp->description,
                                                !USE_WILDCARDS,REMOVE_FROM_LIST))!=NULL) {
                    say("Removing pack %-2d : %s",packcount,tmp1->description);
                    for (tmp2=tmp1->files;tmp2;tmp2=next) {
                        next=tmp2->next;
                        new_free(&(tmp2->file));
                        new_free(&(tmp2->path));
                        new_free(&tmp2);
                    }
                    tmp1->files=NULL;
                    new_free(&(tmp1->description));
                    new_free(&tmp1);
/****** Coded by Zakath ******/
                    CdccPackNum--;
                    update_all_status();
/*****************************/
                }
                else say("DOH ERROR !");
            }
        }
    }
    else say("You must specify what packs to remove");
}

/**********************************************************************
* plistmcommand: Puts list to current channel                         *
***********************************************************************/
void plistmcommand(line)
char *line;
{
    int   count;
    int   number;
    int   current=0;
    int   oldserver;
    char  *channel;
    char  reverse=22;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/4];
    Packs *tmp;
    ChannelList *chan=server_list[curr_scr_win->server].chan_list;

    if (packs) {
        if (!CdccChannels) {
            say("You must set CDCC CHANNELS first");
            return;
        }
        for (tmp=packs,count=0;tmp;tmp=tmp->next) count++;
        if (!my_stricmp(CdccChannels,"current")) {
            current=1;
            channel=get_channel_by_refnum(0);
            if (!channel) {
                NoWindowChannel();
                return;
            }
            chan=lookup_channel(channel,from_server,0);
        }
        oldserver=from_server;
        for (;chan;chan=chan->next) {
            if (current || CheckChannel(chan->channel,CdccChannels)) {
                sprintf(tmpbuf1,"%s %c %d PACK%s OFFERED %c  /CTCP %s %cCDCC%c SEND N for pack N",
                        CdccString,reverse,count,count==1?empty_string:"S",reverse,
                        get_server_nickname(from_server),bold,bold);
                from_server=chan->server;
                send_text(chan->channel,tmpbuf1,"PRIVMSG");
                number=1;
                for (tmp=packs;tmp;tmp=tmp->next) {
                    sprintf(tmpbuf1,"%c#%c%-3d %-28s  %c[%c%.2f kB",bold,bold,number,
                            tmp->description,bold,bold,(float) (tmp->totalbytes)/1024.0);
                    if (tmp->minspeed>0.0)
                        sprintf(tmpbuf2,"%c/%c%d file%s%c/%cmin %.2f kB/s%c]%c",
                                bold,bold,tmp->totalfiles,tmp->totalfiles==1?empty_string:"s",
                                bold,bold,tmp->minspeed,bold,bold);
                    else
                        sprintf(tmpbuf2,"%c/%c%d file%s%c]%c",
                                bold,bold,tmp->totalfiles,tmp->totalfiles==1?empty_string:"s",bold,bold);
                    strcat(tmpbuf1,tmpbuf2);
                    send_text(chan->channel,tmpbuf1,"PRIVMSG");
                    number++;
                }
                if (CdccStats) {
                    sprintf(tmpbuf1,"%s Received %.2f kB  Sent %.2f kB",
                            CdccString,BytesReceived/1024.0,BytesSent/1024.0);
                    send_text(chan->channel,tmpbuf1,"PRIVMSG");
                }
                from_server=oldserver;
            }
            if (current) break;
        }
        LastPlist=time((time_t *) 0);
    }
    else say("No packs created");
}

/**********************************************************************
* noticemcommand: yells about offer to current channel                *
***********************************************************************/
void noticemcommand(line)
char *line;
{
    int   number;
    int   current=0;
    int   oldserver;
    char  *channel;
    char  reverse=22;
    char  tmpbuf[mybufsize/4];
    Packs *tmp;
    ChannelList *chan=server_list[curr_scr_win->server].chan_list;

    if (packs) {
        if (!CdccChannels) {
            say("You must set CDCC CHANNELS first");
            return;
        }
        for (tmp=packs,number=0;tmp;tmp=tmp->next) number++;
        if (!my_stricmp(CdccChannels,"current")) {
            current=1;
            channel=get_channel_by_refnum(0);
            if (!channel) {
                NoWindowChannel();
                return;
            }
            chan=lookup_channel(channel,from_server,0);
        }
        oldserver=from_server;
        for (;chan;chan=chan->next) {
            if (current || CheckChannel(chan->channel,CdccChannels)) {
                sprintf(tmpbuf,"%s %c %d PACK%s OFFERED %c  /CTCP %s %cCDCC%c LIST",
                        CdccString,reverse,number,number==1?empty_string:"S",reverse,
                        get_server_nickname(from_server),bold,bold);
                from_server=chan->server;
                send_text(chan->channel,tmpbuf,"PRIVMSG");
                from_server=oldserver;
            }
            if (current) break;
        }
    }
    else say("No packs created");
}

/***********************************************************************
 * Sets new pack description                                           *
 ***********************************************************************/
#ifdef EXTRAS
void renamepackmcommand(line)
char *line;
{
    int  number;
    int  i;
    char *pack;
    char *desc;
    Packs *tmp;

    if (*line) {
        if (packs) {
            pack=next_arg(line,&line);
            if (pack && *pack=='#') pack++;
            number=atoi(pack);
            desc=line;
            for (tmp=packs,i=1;tmp;tmp=tmp->next,i++) {
                if (i==number) {
                    say("Renamed pack #%d from %s to %s",number,tmp->description,desc);
                    malloc_strcpy(&tmp->description,desc);
                    return;
                }
            }
            say("Invalid pack number %d",number);
        }
        else say("No packs created");
    }
    else PrintUsage("/CDCC RENPACK #packno new description");
}
#endif

/***********************************************************************
 * Lists files in queue                                                *
 ***********************************************************************/
void queuemcommand(line)
char *line;
{
    int  count=0;
    int  countdel=0;
    char *tmpstr;
    char *files=(char *) 0;
    char *file=(char *) 0;
    char *nick=(char *) 0;
    FileQueue *tmp;
    FileQueue *prev=(FileQueue *) 0;
    FileQueue *tmpdel;

    if (!queuelist) {
        say("No files in queue");
        return;
    }
    tmpstr=next_arg(line,&line);
    if (tmpstr && *tmpstr && !my_stricmp(tmpstr,"LIST")) {
        tmpstr=next_arg(line,&line);
        if (tmpstr && *tmpstr) nick=tmpstr;
        if (nick) say("Listing all files in queue for %s",nick);
        else say("Listing all files in queue");
        for (tmp=queuelist;tmp;tmp=tmp->next)
            if (!nick || wild_match(nick,tmp->nick)) {
                count++;
                if (nick) {
                    if (!(file=rindex(tmp->file,'/'))) file=tmp->file;
                    else file++;
                    if (files) malloc_strcat(&files," ");
                    malloc_strcat(&files,file);
                    if (count && (count%70)==0) {
                        say("%s",files);
                        new_free(&files);
                    }
                }
                else say("#%-2d %s to %s",count,tmp->file,tmp->nick);
            }
        if (nick) {
            if (files) {
                say("%s",files);
                new_free(&files);
            }
            say("Total of %d files in queue for %s",count,nick);
        }
        else say("Total of %d files in queue",count);
    }
    else if (tmpstr && *tmpstr && !my_stricmp(tmpstr,"REMOVE")) {
        tmpstr=next_arg(line,&line);
        if (tmpstr && *tmpstr) {
            for (tmp=queuelist;tmp;) {
                count++;
                tmpdel=tmp;
                tmp=tmp->next;
                if (matchmcommand(tmpstr,count)) {
                    if (prev) prev->next=tmpdel->next;
                    else queuelist=tmpdel->next;
                    new_free(&(tmpdel->file));
                    new_free(&(tmpdel->nick));
                    new_free(&tmpdel);
                    countdel++;
                }
                else prev=tmpdel;
            }
            say("Total of %d files removed from queue",countdel);
        }
        else PrintUsage("/CDCC QUEUE REMOVE filter");
    }
    else {
        for (tmp=queuelist;tmp;tmp=tmp->next) count++;
        say("Total of %d files in queue",count);
    }
}

/**********************************************************************
* loadmcommand: loads packs from file                                 *
***********************************************************************/
void loadmcommand(line)
char *line;
{
    int   count=0;
    int   lineno=0;
    char  *file;
    char  *filepath;
    char  *tmpstr;
    char  *tmpstr1;
    char  tmpbuf[mybufsize/2];
    FILE  *fp;
    Packs *tmp=NULL;
    Packs *last;
    Files *tmpfile;
    Files *lastfile;

    if (line && *line) file=line;
    else file="ScrollZ.offer";
    filepath=OpenCreateFile(file,1);
    if ((fp=fopen(filepath,"r"))==NULL) {
#ifdef WANTANSI
        say("%sError%s: Can't open file %s !",
            CmdsColors[COLWARNING].color1,Colors[COLOFF],file);
#else
        say("Can't open file %s",file);
#endif
        return;
    }
    last=packs;
    while (last && last->next) last=last->next;
    while (fgets(tmpbuf,mybufsize/2,fp)) {
        lineno++;
        if (tmpbuf[0]=='#') continue;
        if (tmpbuf[strlen(tmpbuf)-1]=='\n') tmpbuf[strlen(tmpbuf)-1]='\0';
        tmpstr=tmpbuf;
        tmpstr1=next_arg(tmpstr,&tmpstr);
        if (tmpstr1 && *tmpstr1 && !my_strnicmp(tmpstr1,"PACK",4)) {
            tmp=(Packs *) new_malloc(sizeof(Packs));
            tmp->description=NULL;
            while (tmpstr && *tmpstr && isspace(*tmpstr)) tmpstr++;
            malloc_strcpy(&tmp->description,tmpstr);
            tmp->totalfiles=0;
            tmp->totalbytes=0;
            tmp->minspeed=0.0;
            tmp->files=NULL;
            tmp->next=NULL;
            if (last) last->next=tmp;
            else packs=tmp;
            last=tmp;
            count++;
        }
        if (tmpstr1 && *tmpstr1 && !my_strnicmp(tmpstr1,"SPEED",5)) {
            if (tmp) tmp->minspeed=atof(tmpstr);
#ifdef WANTANSI
            else {
                say("%sError%s in %s, %sline %d%s (SPEED should follow PACK)",
                     CmdsColors[COLWARNING].color1,Colors[COLOFF],file,
                     CmdsColors[COLWARNING].color3,lineno,Colors[COLOFF]);
                continue;
            }
#else
            else {
                say("%cError%c in %s, line %d (SPEED should follow PACK)",
                    bold,bold,file,lineno);
                continue;
            }
#endif
        }
        if (tmpstr1 && *tmpstr1 && !my_strnicmp(tmpstr1,"FILE",4)) {
            if (tmp) {
                for (tmpstr1=next_arg(tmpstr,&tmpstr);tmpstr1;tmpstr1=next_arg(tmpstr,&tmpstr))
                    if (SeedFiles(tmpstr1,0)) {
                        lastfile=tmp->files;
                        while (lastfile && lastfile->next) lastfile=lastfile->next;
                        if (lastfile) lastfile->next=files;
                        else tmp->files=files;
                        files=NULL;
                    }
#ifdef WANTANSI
                    else {
                        say("%sError%s in %s, %sline %d%s (can't stat %s)",
                            CmdsColors[COLWARNING].color1,Colors[COLOFF],file,
                            CmdsColors[COLWARNING].color3,lineno,Colors[COLOFF],tmpstr1);
                        continue;
                    }
#else
                    else {
                        say("%cError%c in %s, line %d (can't stat %s)",
                            bold,bold,file,lineno,tmpstr1);
                        continue;
                    }
#endif
            }
#ifdef WANTANSI
            else {
                say("%sError%s in %s, %sline %d%s (FILE should follow PACK)",
                    CmdsColors[COLWARNING].color1,Colors[COLOFF],file,
                    CmdsColors[COLWARNING].color3,lineno,Colors[COLOFF]);
                continue;
            }
#else
            else {
                say("%cError%c in %s, line %d (FILE should follow PACK)",
                    bold,bold,file,lineno);
                continue;
            }
#endif
        }
    }
    fclose(fp);
    for (tmp=packs;tmp;tmp=tmp->next) {
        tmp->totalbytes=0;
        tmp->totalfiles=0;
        for (tmpfile=tmp->files;tmpfile;tmpfile=tmpfile->next) {
            tmp->totalbytes+=tmpfile->size;
            tmp->totalfiles++;
        }
    }
    tmp=packs;
    last=packs;
    while (tmp) {
        if (!(tmp->totalfiles)) {
            count--;
            if (tmp==packs) {
                packs=tmp->next;
                last=packs;
            }
            else last->next=tmp->next;
            say("No files in pack %s, deleting...",tmp->description);
            new_free(&tmp->description);
            for (tmpfile=tmp->files;tmpfile;tmpfile=lastfile) {
                lastfile=tmpfile->next;
                new_free(&tmpfile->path);
                new_free(&tmpfile->file);
                new_free(&tmpfile);
            }
            new_free(&tmp);
            tmp=last;
        }
        else {
            last=tmp;
            tmp=tmp->next;
        }
    }
    say("Loaded %d pack%s from %s",count,count==1?empty_string:"s",file);
}

/**********************************************************************
* savemcommand: saves packs to file                                   *
***********************************************************************/
void savemcommand(line)
char *line;
{
    int   count=0;
    char  *file;
    char  *filepath;
    FILE  *fp;
    Packs *tmp;
    Files *tmpfile;

    if (packs) {
        if (line && *line) file=line;
        else file="ScrollZ.offer";
        filepath=OpenCreateFile(file,1);
        if ((fp=fopen(filepath,"w"))==NULL) {
#ifdef WANTANSI
            say("%sError%s: Can't open file %s !",
                CmdsColors[COLWARNING].color1,Colors[COLOFF],file);
#else
            say("Can't open file %s",file);
#endif
            return;
        }
        for (tmp=packs;tmp;tmp=tmp->next) {
            fprintf(fp,"PACK %s\n",tmp->description);
            if (tmp->minspeed>0.0) fprintf(fp,"SPEED %.2f\n",tmp->minspeed);
            for (tmpfile=tmp->files;tmpfile;tmpfile=tmpfile->next)
                fprintf(fp,"FILE %s/%s\n",tmpfile->path,tmpfile->file);
            count++;
        }
        fclose(fp);
        say("Saved %d pack%s to %s",count,count==1?empty_string:"s",file);
    }
    else say("No packs created");
}

/***********************************************************************
 * requestmcommand: Lets User tell leechers what he needs. By Zakath   *
 ***********************************************************************/
void requestmcommand(line)
char *line;
{
    if (line && *line) {
        CdccReqTog=1;    /* Assume request will be turned on, unless "OFF" */
        if (!my_stricmp(line,"OFF")) {
            CdccReqTog=0;
            new_free(&CdccRequest);   /* Not sure if new_free() is correct */
        }
        else malloc_strcpy(&CdccRequest,line);    /* It works */
    }
    if (!CdccReqTog) PrintSetting("Cdcc request","OFF",empty_string,empty_string);
    else if (CdccRequest && CdccReqTog) PrintSetting("Cdcc request",CdccRequest,empty_string,empty_string);
}

/**********************************************************************
* offermcommand: Prompt User for files                                *
***********************************************************************/
void offermcommand(line)
char *line;
{
    char *speed;
    char *desc;

    if (line && *line) {
        speed=index(line,',');
        if (!speed) {
            PrintUsage("/CDCC OFFER pattern1 pattern2 , speed , description");
            return;
        }
        *speed='\0';
        speed++;
        while (isspace(*speed)) speed++;
        desc=index(speed,',');
        if (!desc) {
            PrintUsage("/CDCC OFFER pattern1 pattern2 , speed , description");
            return;
        }
        *desc='\0';
        desc++;
        if (!(*desc)) {
            PrintUsage("/CDCC OFFER pattern1 pattern2 , speed , description");
            return;
        }
        if (AddFiles2List(line)) offer4mcommand(speed,desc);
    }
    else add_wait_prompt("Add what files to pack ? ",offer2mcommand,line,WAIT_PROMPT_LINE);
}

/**********************************************************************
* offer2mcommand  This parses offer file list                         *
*                 And puts in Files Linked List                       *
***********************************************************************/
void offer2mcommand(blah,line)
char *blah;
char *line;
{
    if (line && *line) {
        if (AddFiles2List(line))
            add_wait_prompt("Min speed for pack ? ",offer3mcommand,line,WAIT_PROMPT_LINE);
    }
    else say("You must specify file(s) to add to pack");
}

/**********************************************************************
* offer3mcommand  This asks for description                           *
***********************************************************************/
void offer3mcommand(blah,line)
char *blah;
char *line;
{
    add_wait_prompt("Pack description ? ",offer4mcommand,line,WAIT_PROMPT_LINE);
}

/**********************************************************************
* offer4mcommand  Final part of pack creation                         *
***********************************************************************/
void offer4mcommand(blah,line)
char *blah;
char *line;
{
    if (line && *line) AddToOfferList(blah,line);
    else say("You must specify pack description");
    DeleteSend();
}

/**********************************************************************
* AddToOfferList:  Adds Files to offer List                           *
***********************************************************************/
void AddToOfferList(speed,desc)
char *speed;
char *desc;
{
    int   totalfiles=0;
    int   totalbytes=0;
    char  tmpbuf1[mybufsize/4];
#ifdef WANTANSI
    char  tmpbuf2[mybufsize/8];
#endif
    Packs *new=(Packs *) 0;
    Packs *tmppack;
    Files *tmp=(Files *) 0;
    Files *tmp2=(Files *) 0;
    Files *tmp3=(Files *) 0;

    new=(Packs *) new_malloc(sizeof(Packs));
    new->description=(char *) 0;
    new->minspeed=(speed && *speed)?atof(speed):0.0;
    malloc_strcpy(&(new->description),desc);
    new->files=NULL;
    new->next=NULL;
    for (tmppack=packs;tmppack && tmppack->next;tmppack=tmppack->next);
    if (tmppack) tmppack->next=new;
    else packs=new;
    if (files) {
        tmp=files;
        for (;;) {
            tmp2=tmp->next;
            for (tmp3=new->files;tmp3 && tmp3->next;) tmp3=tmp3->next;
            if (tmp3) tmp3->next=tmp;
            else new->files=tmp;
            tmp->next=(Files *) 0;
            totalfiles++;
            totalbytes=totalbytes+tmp->size;
            if (!tmp2) break;
            tmp=tmp2;
        }
        files=(Files *) 0;
    }
    else files=(Files *) 0;
    new->totalbytes=totalbytes;
    new->totalfiles=totalfiles;
    sprintf(tmpbuf1,"%.2f kB/%d file%s",(float) (totalbytes)/1024.0,
            totalfiles,totalfiles==1?empty_string:"s");
#ifdef WANTANSI
    if (new->minspeed>0.0)
        sprintf(tmpbuf2,"%s/min %.2f kB/s%s",
                CmdsColors[COLCDCC].color5,new->minspeed,Colors[COLOFF]);
    else *tmpbuf2='\0';
    say("%sCdcc%s %screated new pack%s : [%s%s%s%s] ",
        CmdsColors[COLCDCC].color4,Colors[COLOFF],
        CmdsColors[COLCDCC].color3,Colors[COLOFF],
        CmdsColors[COLCDCC].color5,tmpbuf1,Colors[COLOFF],tmpbuf2);

#else
    if (new->minspeed>0.0) 
        say("Cdcc created new pack : [%s/min %.2f kB/s]",tmpbuf1,new->minspeed);
    else say("Cdcc created new pack : [%s]",tmpbuf1);
#endif
/****** Coded by Zakath ******/
    CdccPackNum++;
    update_all_status();
/*****************************/
}

/**********************************************************************
* sendmcommand: Prompt User for files                                 *
***********************************************************************/
void sendmcommand(line)
char *line;
{
    char *comma;

    if (line && *line) {
        comma=strchr(line,',');
        if (!comma) {
            PrintUsage("/CDCC SEND pattern1 pattern2 , nick1 nick2");
            return;
        }
        *comma='\0';
        comma++;
        if (AddFiles2List(line)) send3mcommand(NULL,comma);
    }
    else add_wait_prompt("Files to send ? ",send2mcommand,line,WAIT_PROMPT_LINE);
}

/**********************************************************************
* send2mcommand  This parses file send list                           *
*                Files Linked list                                    *
***********************************************************************/
void send2mcommand(blah,line)
char *blah;
char *line;
{
    if (line && *line) {
        if (AddFiles2List(line))
            add_wait_prompt("Send to whom ? (ie. nick1 nick2 nick3) ",send3mcommand,line,WAIT_PROMPT_LINE);
    }
    else say("You must specify file(s) to send");
}

/**********************************************************************
* send3mcommand  This parses nick send list, and sends all files in   *
*                Files Linked list                                    *
***********************************************************************/
void send3mcommand(blah,line)
char *blah;
char *line;
{
    int  count;
    int  total;
    int  queue=0;
    char *nick=(char *) 0;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/8];
    Files *tmp;
    unsigned int display;

    if (line && *line) {
        for (nick=next_arg(line,&line);nick;nick=next_arg(line,&line)) {
            total=0;
            count=0;
            display=window_display;
            window_display=0;
            for (tmp=files;tmp;tmp=tmp->next) {
                if (TotalSendDcc()<CdccLimit) {
                    sprintf(tmpbuf1,"%s %s/%s",nick,tmp->path,tmp->file);
                    dcc_filesend(tmpbuf1);
                }
                else {
                    AddToQueue(tmp,nick,1);
                    queue++;
                }
                count++;
                total+=tmp->size;
            }
            window_display=display;
            sprintf(tmpbuf2,"%.2f kB/%d file%s",(float) (total)/1024.0,
                    count,count==1?empty_string:"s");
            if (queue) sprintf(tmpbuf1,", %d files in queue",queue);
            else *tmpbuf1='\0';
            send_to_server("NOTICE %s :Sent : %c[%c%s%c]%c%s",
                                nick,bold,bold,tmpbuf2,bold,bold,tmpbuf1);
#ifdef WANTANSI
            sprintf(tmpbuf1,"%sCdcc%s %ssending%s %s%s%s : ",
                    CmdsColors[COLCDCC].color4,Colors[COLOFF],
                    CmdsColors[COLCDCC].color3,Colors[COLOFF],
                    CmdsColors[COLCDCC].color1,nick,Colors[COLOFF]);
            if (!queue) say("%s[%s%s%s]",tmpbuf1,
                            CmdsColors[COLCDCC].color5,tmpbuf2,Colors[COLOFF]);
            else say("%s[%s%s%s], %d files in queue",tmpbuf1,
                     CmdsColors[COLCDCC].color5,tmpbuf2,Colors[COLOFF],queue);
#else
            if (!queue) say("Cdcc sending %s : [%s]",nick,tmpbuf2);
            else say("Cdcc sending %s : [%s], %d files in queue",nick,tmpbuf2,queue);
#endif
        }
    }
    else say("You must specify nick(s) to send file(s) to");
    DeleteSend();
}

/**********************************************************************
* resendmcommand: Prompt User for files                               *
***********************************************************************/
void resendmcommand(line)
char *line;
{
    char *comma;

    if (line && *line) {
        comma=strchr(line,',');
        if (!comma) {
            PrintUsage("/CDCC RESEND pattern1 pattern2 , nick1 nick2");
            return;
        }
        *comma='\0';
        comma++;
        while (*comma==' ') comma++;
        if (AddFiles2List(line)) resend3mcommand(NULL,comma);
    }
    else add_wait_prompt("Files to resend ? ",resend2mcommand,line,WAIT_PROMPT_LINE);
}

/**********************************************************************
* resend2mcommand  This parses file resend list                       *
*                  Files Linked list                                  *
***********************************************************************/
void resend2mcommand(blah,line)
char *blah;
char *line;
{
    if (line && *line) {
        if (AddFiles2List(line))
            add_wait_prompt("Resend to whom ? (ie. nick1 nick2 nick3) ",resend3mcommand,line,WAIT_PROMPT_LINE);
    }
    else say("You must specify file(s) to resend");
}

/*************************************************************************
* resend3mcommand  This parses nick resend list, and resends all files   *
*                  in Files Linked list                                  *
**************************************************************************/
void resend3mcommand(blah,line)
char *blah;
char *line;
{
    int  count=0;
    int  total=0;
    char *nick=(char *) 0;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/8];
    Files *tmp;
    unsigned int display;

    if (line && *line) {
        for (nick=next_arg(line,&line);nick;nick=next_arg(line,&line)) {
            display=window_display;
            window_display=0;
            for (tmp=files;tmp;tmp=tmp->next) {
                sprintf(tmpbuf1,"%s %s/%s",nick,tmp->path,tmp->file);
                dcc_resend(tmpbuf1);
                count++;
                total=total+tmp->size;
            }
            window_display=display;
            sprintf(tmpbuf2,"%.2f kB/%d file%s",(float) (total)/1024.0,
                    count,count==1?empty_string:"s");
            send_to_server("NOTICE %s :Resent : %c[%c%s%c]%c",nick,bold,bold,tmpbuf2,
                           bold,bold);
#ifdef WANTANSI
            sprintf(tmpbuf1,"%sCdcc%s %sresending%s %s%s%s : ",
                    CmdsColors[COLCDCC].color4,Colors[COLOFF],
                    CmdsColors[COLCDCC].color3,Colors[COLOFF],
                    CmdsColors[COLCDCC].color1,nick,Colors[COLOFF]);
            say("%s[%s%s%s]",tmpbuf1,CmdsColors[COLCDCC].color5,tmpbuf2,Colors[COLOFF]);
#else
            say("Cdcc resending %s : [%s]",nick,tmpbuf2);
#endif
            total=0;
            count=0;
        }
    }
    else say("You must specify nick(s) to resend file(s) to");
    DeleteSend();
}

/**********************************************************************
* psendmcommand: Prompt User for pack                                 *
***********************************************************************/
void psendmcommand(line)
char *line;
{
    char *comma;

    if (packs) {
        if (line && *line) {
            comma=strchr(line,',');
            if (!comma) {
                PrintUsage("/CDCC PSEND pattern , nick1 nick2");
                return;
            }
            *comma='\0';
            comma++;
            while (*comma==' ') comma++;
            psend3mcommand(line,comma);
        }
        else {
            ShowPacks(NULL);
            add_wait_prompt("Packs to send (1-6,3 or * for all) ? ",psend2mcommand,line,WAIT_PROMPT_LINE);
        }
    }
    else say("No packs created");
}

/**********************************************************************
* psend2mcommand  This parses nick send list, and sends all files in  *
*                Pack Linked list                                     *
***********************************************************************/
void psend2mcommand(blah,line)
char *blah;
char *line;
{
    if (line && *line)
        add_wait_prompt("Send to whom ? (ie. nick1 nick2 nick3) ",psend3mcommand,line,WAIT_PROMPT_LINE);
    else say("You must specify pack(s) to send");
}

/**********************************************************************
* psend3mcommand  This parses pack send list                          *
***********************************************************************/
void psend3mcommand(blah,line)
char *blah;
char *line;
{
    int  packcount;
    char *nick;
    char *tmpstr;
    char tmpbuf[mybufsize/32];
    Packs *tmp;

    if (line && *line) {
        tmpstr=line;
        for (nick=next_arg(tmpstr,&tmpstr);nick;nick=next_arg(tmpstr,&tmpstr)) {
            packcount=1;
            for (tmp=packs;tmp;tmp=tmp->next) {
                if (matchmcommand(blah,packcount)) {
                    sprintf(tmpbuf,"%d",packcount);
                    sendcommand(nick,tmpbuf);
                }
                packcount++;
            }
        }
    }
    else say("You must specify nicks(s) to send pack(s) to");
    DeleteSend();
}

/**********************************************************************
* SeedFiles: Gets Users Line of Files with path, and rips em apart    *
* Puts path, file, and size into Files linked list.                   *
***********************************************************************/ 
int SeedFiles(line,error)
char *line;
int  error;
{
    int  i=0;
    int  size=0;
    int  count=0;
    char *string=(char *) 0;
    char *file=(char *) 0;
    char *fullname=(char *) 0;
    char *rest=(char *) 0;
    char  tmpbuf1[mybufsize/2];
    char  tmpbuf2[mybufsize/8];
    struct stat tmpstat;

    file=line;
    if (*file=='/') {
        if (strlen(file)>1 && rindex(file,'/')==file) sprintf(tmpbuf1,"/ %s",&file[1]);
        else strcpy(tmpbuf1,file);
    }
    else if (*file=='~') {
        if (0 == (fullname=expand_twiddle(file))) {
            if (error) yell("Unable to expand %s!",file);
            return(0);
        }
        strcpy(tmpbuf1,fullname);
        new_free(&fullname);
    }
    else {
        if (CdccUlDir) strcpy(tmpbuf1,CdccUlDir);
        else getcwd(tmpbuf1,mybufsize/2);
        strcat(tmpbuf1,"/");
        strcat(tmpbuf1,file);
    }
    tmpstat.st_mode=0;
    stat_file(tmpbuf1,&tmpstat);
    if (tmpstat.st_mode & S_IFDIR) {
        if (error) say("You tried to send a dir, please do a /* to send a whole dir");
        return(0);
    }
    rest=rindex(tmpbuf1,'/');
    if (rest==tmpbuf1) rest++;
    *rest='\0';
    rest++;
    if (access(tmpbuf1, R_OK)!=0) {
        if (error) say("Can't access %s",tmpbuf1);
        return(0);
    }
    GetDir(tmpbuf1);
    for (i=0;i<CdccEntries;i++) {
        strcpy(tmpbuf2,CdccFileNames[i]->d_name);
        string=tmpbuf2;
#ifdef lame_dgux
        string=string-2;
        if (string[0]=='.') continue;
#endif
        if (wild_match(rest,string))
            if ((size=GetSize(tmpbuf1,string))!=-1) {
                AddFileToList(tmpbuf1,string,size);
                count++;
            }
    }
    return(count);
}

/**********************************************************************
* Add File to  FILES linked list                                      *
***********************************************************************/
void AddFileToList(path,file,size)
char *path;
char *file;
int  size;
{
    Files *new;
    Files *tmp;

    new=(Files *) new_malloc(sizeof(Files));
    new->path=(char *) 0;
    new->file=(char *) 0;
    new->next=(Files *) 0;
    malloc_strcpy(&(new->path),path);
    malloc_strcpy(&(new->file),file);
    new->size=size;
    for (tmp=files;tmp && tmp->next;) tmp=tmp->next;
    if (tmp) tmp->next=new;
    else files=new;
}

/**********************************************************************
* Returns number of files added to list                               *
***********************************************************************/
int AddFiles2List(line)
char *line;
{
    int   count=0;
    char  *file=(char *) 0;
    char  tmpbuf[mybufsize];
    Files *tmpfile;

    for (file=next_arg(line,&line);file;file=next_arg(line,&line))
        count+=SeedFiles(file,1);
    if (count) {
#ifdef WANTANSI
        sprintf(tmpbuf,"%sCdcc%s %sadded%s %s%d%s file%s (",
                CmdsColors[COLCDCC].color4,Colors[COLOFF],
                CmdsColors[COLCDCC].color3,Colors[COLOFF],
                CmdsColors[COLCDCC].color5,count,Colors[COLOFF],count==1?empty_string:"s");
#else
        sprintf(tmpbuf,"Added %d file%s (",count,count==1?empty_string:"s");
#endif
        for (tmpfile=files;tmpfile;tmpfile=tmpfile->next) {
#ifdef WANTANSI
            strcat(tmpbuf,CmdsColors[COLCDCC].color5);
#endif
            strcat(tmpbuf,tmpfile->file);
            if (tmpfile->next) strcat(tmpbuf," ");
            else {
#ifdef WANTANSI
                strcat(tmpbuf,Colors[COLOFF]);
#endif
                strcat(tmpbuf,")");
            }
        }
        say("%s",tmpbuf);
    }
    else {
        say("No files found, aborting...");
        DeleteSend();
    }
    return(count);
}

/*********************************************************************
* Deletes the file link list                                         *
**********************************************************************/
void DeleteSend()
{
    Files *tmp;
    Files *next;

    if (files) {
        for (tmp=files;tmp;tmp=next) {
            next=tmp->next;
            new_free(&(tmp->path));
            new_free(&(tmp->file));
            new_free(&tmp);
        }
    }
    files=(Files *) 0;
}

/*********************************************************************
* Shows the offer link list                                          *
**********************************************************************/
void ShowPacks(args)
char *args;
{
    int   packcount=1;
    char  *word=(char *) 0;
    char  tmpbuf1[mybufsize/8];
    char  tmpbuf2[mybufsize/32];
    Packs *tmp;
    Files *tmp1;

    if (args && *args) word=next_arg(args,&args);
    if (word) {
        if (packs) {
            for (tmp=packs;tmp;tmp=tmp->next) {
                if (matchmcommand(word,packcount)) {
                    if (tmp->minspeed>0.0) 
                        say("Pack: %d  Description: %s  Min: %.2f kB/s",packcount,
                            tmp->description,tmp->minspeed);
                    else say("Pack: %d  Description: %s",packcount,tmp->description);
                    say("kBytes      File");
                    for (tmp1=tmp->files;tmp1;tmp1=tmp1->next) {
                        sprintf(tmpbuf1,"%-11.2f",(float) (tmp1->size)/1024.0);
                        say("%s %s",tmpbuf1,tmp1->file);
                    }
                    say("----------- ---------------");
                }
                packcount++;
            }
            say("Received %.2f kB  Sent %.2f kB",BytesReceived/1024.0,BytesSent/1024.0);
        }
        else say("No packs created");
    }
    else {
        if (packs) {
            for (tmp=packs;tmp;tmp=tmp->next) {
                if (tmp->minspeed>0.0)
                    sprintf(tmpbuf2,"%c/%cmin %.2f kB/s%c]%c",bold,bold,
                            tmp->minspeed,bold,bold);
                else sprintf(tmpbuf2,"%c]%c",bold,bold);
                sprintf(tmpbuf1,"%c[%c%.2f kB%c/%c%d file%s%s",bold,bold,
                        (float) (tmp->totalbytes)/1024.0,bold,bold,tmp->totalfiles,
                        tmp->totalfiles==1?empty_string:"s",tmpbuf2);
                say("%c#%c%-2d %-29s  %s",bold,bold,packcount,tmp->description,tmpbuf1);
                packcount++;
            }
            say("Received %.2f kB  Sent %.2f kB",BytesReceived/1024.0,BytesSent/1024.0);
        }
        else say("No packs created");
    }
}               

/*********************************************************************
* GetDir:  This gets the listing of all the files in *ptr dir.       *
**********************************************************************/
void GetDir(path)
char *path;
{
    c_entry_size = 0;

    CleanList();
    CdccEntries=scandir(path, &CdccFileNames,
                        (int (*) _((const struct dirent *))) selectent,
                        (const void *)(int (*) _((const struct dirent *, const struct dirent *))) compar);
}

/*********************************************************************
* CleanList: Cleans up global dirent                                 *
**********************************************************************/
void CleanList()
{
    int i;

    if (CdccFileNames) {
        for (i=0;i<CdccEntries;i++) new_free(&(CdccFileNames[i]));     
        new_free(&CdccFileNames);
        CdccEntries=0;
    }
}

/*********************************************************************
* compar: used by scandir to alphabetize files in dir                *
**********************************************************************/
int compar(e1,e2)
struct dirent **e1;
struct dirent **e2;
{
    return (my_stricmp((*e1)->d_name,(*e2)->d_name));
}

/***************************************************************************
* selectent: used by scandir to decide which entries to include in the dir *
*            listing.  Ignores ., includes all other files                 *
****************************************************************************/
int selectent(entry)
struct dirent *entry;
{
    if (*(entry->d_name)=='.') return (0);
    else {
        int len=strlen(entry->d_name);
        c_entry_size=(len>c_entry_size)?len:c_entry_size;
        return(1);
    }
}

/***********************************************************************
* GetSize: takes path, and filename, cats them together, and returns   *
*          filesize of file, if dir, returns -1                        *
************************************************************************/
int GetSize(path,file)
char *path;
char *file;
{
    char  tmpbuf[mybufsize/2];

    sprintf(tmpbuf,"%s/%s",path,file);
    stat_file(tmpbuf,&CdccStatBuf);
    if (CdccStatBuf.st_mode & S_IFDIR) return(-1);
    return(CdccStatBuf.st_size);
}

/**********************************************************************
* getmcommand: Prompt User for nick (* for all)                       *
***********************************************************************/
void getmcommand(line)
char *line;
{
    if (line && *line) get2mcommand(NULL,line);
    else {
        showdccsmcommand(66);
        add_wait_prompt("What to get (1-6,3 or * for all) ",get2mcommand,line,WAIT_PROMPT_LINE);
    }
}

/**********************************************************************
* get2mcommand:          This will parse your dcc list, and get files *
*                        Specified by your filter.                    *
***********************************************************************/
void get2mcommand(data,line)
char *data;
char *line;
{
    int  count=0;
    int  mode;
    char *tmp=(char *) 0;
    char  tmpbuf[mybufsize/4];
    DCC_list *Client;
    unsigned flags;

    tmp=next_arg(line,&line);
    if (!tmp) {
        say("You must specify what to get");
        return;
    }
    for (Client=ClientList;Client;Client=Client->next) {
        flags=Client->flags;
        if ((flags&DCC_OFFER)==DCC_OFFER) {
            mode=0;
            if ((flags&DCC_TYPES)==DCC_FILEREAD) mode=1;
            else if ((flags&DCC_TYPES)==DCC_FILEREGET) mode=2;
            count++;
            if (matchmcommand(tmp,count)) {
                sprintf(tmpbuf,"%s %s",Client->user,Client->description);
                if (mode==1) dcc_getfile(tmpbuf);
                else if (mode==2) dcc_regetfile(tmpbuf);
            }
        }
    }
}

/* Parse message to see what they want. */
void CheckCdcc(nick,args,to,rest)
char *nick;
char *args;
char *to;
char *rest;
{
    char *command=(char *) 0;
#ifdef WANTANSI
    char  tmpbuf1[mybufsize/2];
#endif
    char  tmpbuf2[mybufsize/8];
#ifdef WANTANSI
    char  tmpbuf3[mybufsize/8];
#endif
    char  tmpbuf4[mybufsize];

    if (Security && !FriendList) {
        send_to_server("NOTICE %s :This function has been disabled",nick);
        return;
    }
    command=next_arg(args,&args);
    if (command && (!my_stricmp(command,"CDCC") || !my_stricmp(command,"XDCC")))
        command=next_arg(args,&args);
    if (command) {
#ifdef WANTANSI
        ColorUserHost(FromUserHost,CmdsColors[COLCDCC].color2,tmpbuf3);
        *tmpbuf2='\0';
        if (to && is_channel(to))
            sprintf(tmpbuf2," to %s%s%s",
                   CmdsColors[COLCDCC].color6,to,Colors[COLOFF]);
        if (args && *args) {
            sprintf(tmpbuf1,"%sCdcc%s %s%s %s%s request received from %s%s",
                    CmdsColors[COLCDCC].color4,Colors[COLOFF],
                    CmdsColors[COLCDCC].color3,command,args,Colors[COLOFF],
                    CmdsColors[COLCDCC].color1,nick);
            sprintf(tmpbuf4,"%s%s %s%s",tmpbuf1,Colors[COLOFF],tmpbuf3,tmpbuf2);
        }
        else {
            sprintf(tmpbuf1,"%sCdcc%s %s%s%s request received from %s%s%s",
                    CmdsColors[COLCDCC].color4,Colors[COLOFF],
                    CmdsColors[COLCDCC].color3,command,Colors[COLOFF],
                    CmdsColors[COLCDCC].color1,nick,Colors[COLOFF]);
            sprintf(tmpbuf4,"%s %s%s",tmpbuf1,tmpbuf3,tmpbuf2);
        }
#else
        *tmpbuf2='\0';
        if (to && is_channel(to)) sprintf(tmpbuf2," to %s",to);
        if (args && *args)
            sprintf(tmpbuf4,"Cdcc %s %s request received from %s (%s)%s",
                    command,args,nick,FromUserHost,tmpbuf2);
        else sprintf(tmpbuf4,"Cdcc %s request received from %s (%s)%s",
                     command,nick,FromUserHost,tmpbuf2);
#endif
        say("%s",tmpbuf4);
        if (away_set || LogOn) AwaySave(tmpbuf4,SAVECDCC);
        if (!my_stricmp(command,"HELP")) helpcommand(nick,args);
        else if (!my_stricmp(command,"SEND")) sendcommand(nick,args);
        else if (!my_stricmp(command,"LIST")) listcommand(nick,args);
        else if (!my_stricmp(command,"VERSION")) versioncommand(nick,args);
        else if (!CTCPCloaking) send_to_server("NOTICE %s :Try /CTCP %s CDCC HELP to get Cdcc help",nick,get_server_nickname(from_server));
    }
    else if (!CTCPCloaking) send_to_server("NOTICE %s :Try /CTCP %s CDCC HELP to get Cdcc help",nick,get_server_nickname(from_server));
}

/* send msg'er help menu */
void helpcommand(from,args)
char *from;
char *args;
{
    if (CTCPCloaking) return;
    if (args && *args) {
        if (!my_stricmp(args,"VERSION")) send_to_server("NOTICE %s :Cdcc VERSION will report Cdcc version",from,get_server_nickname(from_server));
        else if (!my_stricmp(args,"HELP")) send_to_server("NOTICE %s :Cdcc HELP will give you Cdcc help",from,get_server_nickname(from_server));
        else if (!my_stricmp(args,"LIST")) {
            send_to_server("NOTICE %s :Cdcc LIST [filter] will list offered packs",from,get_server_nickname(from_server));
            send_to_server("NOTICE %s :          filter can be : -2,4,6-8,10- or * for all",from);
        }
        else if (!my_stricmp(args,"SEND")) {
            send_to_server("NOTICE %s :Cdcc SEND filter will send packs matching filter",from,get_server_nickname(from_server));
            send_to_server("NOTICE %s :          filter can be : -2,4,6-8,10- or * for all",from);
        }
    }
    else send_to_server("NOTICE %s :/CTCP %s CDCC HELP command where command is one of the following : HELP VERSION LIST or SEND",from,get_server_nickname(from_server));
}

/* send msg'er version reply */
void versioncommand(from, args)
char *from;
char *args;
{
    if (CTCPCloaking) return;
    send_to_server("NOTICE %s :Cdcc v1.5 written by Sheik & Flier",from);
    send_to_server("NOTICE %s :IRC's first XDCC clone in C !",from);
}

/* Check what listing type they want. */
void listcommand(from, args)
char *from;
char *args;
{
    int   packcount=1;
    int   count=0;
    /*int   sent=0;*/
    char  reverse=22;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/8];
    /*char  *word;*/
    Packs *tmp;
    /*Files *tmp1;

    word=next_arg(args,&args);
    if (word) {
        if (packs) {
            for (tmp=packs;tmp;tmp=tmp->next) {
                if (matchmcommand(word,packcount)) {
                    if (tmp->minspeed>0.0) 
                        send_to_server("NOTICE %s :Pack: %d  Description: %s  Min: %.2f kB/s",
                                       from,packcount,tmp->description,tmp->minspeed);
                    else send_to_server("NOTICE %s :Pack: %d  Description: %s",from,
                                        packcount,tmp->description);
                    send_to_server("NOTICE %s :kBytes      File",from);
                    sent++;
                    for (tmp1=tmp->files;tmp1;tmp1=tmp1->next) {
                        sprintf(tmpbuf1,"%-11.2f",(float) (tmp1->size)/1024.0);
                        send_to_server("NOTICE %s :%s %s",from,tmpbuf1,tmp1->file);
                    }
                    send_to_server("NOTICE %s :----------- ---------------",from);
                }
                packcount++;
            }
            if (sent) {
                send_to_server("NOTICE %s :%s Received %.2f kB  Sent %.2f kB",
                               from,CdccString,BytesReceived/1024.0,BytesSent/1024.0);
            }
            else send_to_server("NOTICE %s :No packs found matching %s",from,word);
        }
        else send_to_server("NOTICE %s :Sorry, there are no files offered",from);
    }
    else { */
    if (packs) {
        for (tmp=packs;tmp;tmp=tmp->next) count++;
        send_to_server("NOTICE %s :%s %c %d PACK%s OFFERED %c  /CTCP %s CDCC %cSEND%c N for pack N",
                       from,CdccString,reverse,count,count==1?empty_string:"s",
                       reverse,get_server_nickname(from_server),bold,bold);
        for (tmp=packs;tmp;tmp=tmp->next) {
            if (tmp->minspeed>0.0)
                sprintf(tmpbuf2,"%c/%cmin %.2f kB/s%c]%c",bold,bold,tmp->minspeed,
                        bold,bold);
            else sprintf(tmpbuf2,"%c]%c",bold,bold);
            sprintf(tmpbuf1,"%c[%c%.2f kB%c/%c%d file%s%s",bold,bold,
                    (float) (tmp->totalbytes)/1024.0,bold,bold,tmp->totalfiles,
                    tmp->totalfiles==1?empty_string:"s",tmpbuf2);
            send_to_server("NOTICE %s :%c#%c%-3d %-28s  %s",from,bold,bold,
                           packcount,tmp->description,tmpbuf1);
            packcount++;
        }
        if (CdccStats)
            send_to_server("NOTICE %s :%s Received %.2f kB  Sent %.2f kB",
                           from,CdccString,BytesReceived/1024.0,BytesSent/1024.0);
    }
    else send_to_server("NOTICE %s :Sorry, there are no files offered",from);
/*    }*/
}               

/* Send pack that they request */
void sendcommand(from, args)
char *from;
char *args;
{
    int   packcount=1;
    int   totalfiles=0;
    int   totalbytes=0;
    int   sent=0;
    int   queue=0;
    char  *word;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/4];
    Packs *tmp;
    Files *tmp1;
    unsigned int display;
    DCC_list *tmpdcc;

    word=next_arg(args,&args);
    if (word) {
        if (*word=='#') *word++='\0';
        if (packs) {
            for (tmp=packs;tmp;tmp=tmp->next) {
                if (matchmcommand(word,packcount)) {
                    if (tmp->minspeed>0.0)
                        sprintf(tmpbuf1,"%c[%cmin %.2f kB/s%c]%c",bold,bold,
                                tmp->minspeed,bold,bold);
                    else *tmpbuf1='\0';
                    send_to_server("NOTICE %s :Sending you pack %d : %s %s",from,
                                   packcount,tmp->description,tmpbuf1);
#ifdef WANTANSI
                    sprintf(tmpbuf1,"%sCdcc%s %ssending%s %s%s%s pack %s",
                            CmdsColors[COLCDCC].color4,Colors[COLOFF],
                            CmdsColors[COLCDCC].color3,Colors[COLOFF],
                            CmdsColors[COLCDCC].color1,from,Colors[COLOFF],
                            CmdsColors[COLCDCC].color5);
                    say("%s%d%s",tmpbuf1,packcount,Colors[COLOFF]);
#else
                    say("Cdcc sending %s pack %d",from,packcount);
#endif
                    display=window_display;
                    window_display=0;
                    sent++;
                    for (tmp1=tmp->files;tmp1;tmp1=tmp1->next) {
                        if (TotalSendDcc()<CdccLimit) {
                            sprintf(tmpbuf2,"%s/%s",tmp1->path,tmp1->file);
                            sprintf(tmpbuf1,"%s %s",from,tmpbuf2);
                            dcc_filesend(tmpbuf1);
                            tmpdcc=dcc_searchlist(NULL,from,DCC_FILEOFFER,0,tmpbuf2);
                            if (tmpdcc) tmpdcc->minspeed=tmp->minspeed;
                        }
                        else {
                            AddToQueue(tmp1,from,1);
                            queue++;
                        }
                        totalbytes+=tmp1->size;
                        totalfiles++;
                    }
                    window_display=display;
                }
                packcount++;
            }
            if (totalfiles) {
                sprintf(tmpbuf2,"%.2f kB/%d file%s",(float) (totalbytes)/1024.0,
                        totalfiles,totalfiles==1?empty_string:"s");
                if (!queue) send_to_server("NOTICE %s :Sent : %s",from,tmpbuf2);
                else send_to_server("NOTICE %s :Sent : %s, %d files in queue",from,
                                    tmpbuf2,queue);
#ifdef WANTANSI
                sprintf(tmpbuf1,"%sCdcc%s %ssent%s %s%s%s : ",
                        CmdsColors[COLCDCC].color4,Colors[COLOFF],
                        CmdsColors[COLCDCC].color3,Colors[COLOFF],
                        CmdsColors[COLCDCC].color1,from,Colors[COLOFF]);
                if (!queue) say("%s[%s%s%s]",tmpbuf1,
                                CmdsColors[COLCDCC].color5,tmpbuf2,Colors[COLOFF]);
                else say("%s[%s%s%s], %s%d%s files in queue",tmpbuf1,
                         CmdsColors[COLCDCC].color5,tmpbuf2,Colors[COLOFF],
                         CmdsColors[COLCDCC].color5,queue,Colors[COLOFF]);
#else
                if (!queue) say("Cdcc sent %s : %c[%c%s%c]%c",from,bold,bold,tmpbuf2,
                                bold,bold);
                else say("Cdcc sent %s : %c[%c%s%c]%c, %d files in queue",
                         from,bold,bold,tmpbuf2,bold,bold,queue);
#endif
/****** Coded by Zakath ******/
                if (CdccReqTog && CdccRequest)
                    send_to_server("NOTICE %s :I need : %c%s%c",from,bold,CdccRequest,bold);
/*****************************/
            }
            if (!sent) send_to_server("NOTICE %s :No packs found matching %s",from,word);
        }
        else send_to_server("NOTICE %s :Sorry, there are no files offered",from);
    }
    else send_to_server("NOTICE %s :Try /CTCP %s CDCC SEND N",from,get_server_nickname(from_server));
}

/**********************************************************************
* TotalSendDcc:          Return Total Dcc Sends Currently on Dcc      *
*                        Linked List.                                 *
***********************************************************************/
int TotalSendDcc()
{
    DCC_list *Client;
    unsigned flags;
    int counter=0;

    if (ClientList) {
        for (Client=ClientList;Client;Client=Client->next) {
            flags=Client->flags;
            if (((flags&DCC_TYPES)==DCC_FILEOFFER) ||
               ((flags&DCC_TYPES)==DCC_RESENDOFFER)) counter++;
        }
    }
    return(counter);
}

/* get offer files if auto-get is on */
void CheckAutoGet(nick,userhost,file,type)
char *nick;
char *userhost;
char *file;
char *type;
{
    char tmpbuf[mybufsize/4];
    struct friends *tmpfriend;

    if (Security) {
        sprintf(tmpbuf,"%s!%s",nick,userhost);
        tmpfriend=CheckUsers(tmpbuf,NULL);
        if (!tmpfriend) return;
        if (!((tmpfriend->privs)&64)) return;
    }
#ifdef WANTANSI
    sprintf(tmpbuf,"%sCdcc%s %sauto getting%s %s%s%s from ",
            CmdsColors[COLCDCC].color4,Colors[COLOFF],
            CmdsColors[COLCDCC].color3,Colors[COLOFF],
            CmdsColors[COLCDCC].color5,file,Colors[COLOFF]);
    say("%s%s%s%s",tmpbuf,CmdsColors[COLCDCC].color1,nick,Colors[COLOFF]);
#else
    say("Cdcc auto getting %s from %s",file,nick);
#endif
    if (away_set || LogOn) {
        sprintf(tmpbuf,"Cdcc auto getting %s from %s (%s)",file,nick,FromUserHost);
        AwaySave(tmpbuf,SAVECDCC);
    }
    sprintf(tmpbuf,"%s %s",nick,file);
    if (!my_stricmp(type,"SEND")) dcc_getfile(tmpbuf);
    if (!my_stricmp(type,"RESEND")) dcc_regetfile(tmpbuf);
}


/**********************************************************************
* CheckdccIdleSend:      Closes idle Dcc Sends Currently on Dcc       *
*                        Linked List.                                 *
***********************************************************************/
void CheckDccIdleSends()
{
    int  count;
    int  current=0;
    int  packcount;
    int  oldserver;
    char reverse=22;
    char *tmpstr;
    char *channel;
    char  tmpbuf1[mybufsize/4];
    char  tmpbuf2[mybufsize/4];
    time_t timenow;
    Packs  *tmp;
    unsigned flags;
    DCC_list *Client;
    ChannelList *chan=server_list[curr_scr_win->server].chan_list;

    timenow=time((time_t *) 0);
    if (timenow-LastIdleCheck>=60) {
        LastIdleCheck=timenow;
        for (Client=ClientList;CdccIdle && Client;Client=Client->next) {
            flags=Client->flags;
            if (!(flags&DCC_ACTIVE)) {
                tmpstr=rindex(Client->description,'/');
                if (tmpstr) tmpstr++;
                else tmpstr=Client->description;
                if ((flags&DCC_TYPES)==DCC_FILEOFFER || (flags&DCC_TYPES)==DCC_RESENDOFFER) {
                    if (timenow-Client->CdccTime>CdccIdle) {
#ifdef WANTANSI
                        sprintf(tmpbuf1,"%sCdcc%s %sclosing%s idle dcc %s%s%s",
                                CmdsColors[COLCDCC].color4,Colors[COLOFF],
                                CmdsColors[COLCDCC].color3,Colors[COLOFF],
                                CmdsColors[COLCDCC].color3,dcc_types[flags&DCC_TYPES],
                                Colors[COLOFF]);
                        sprintf(tmpbuf2," %s%s%s to %s%s%s",
                                CmdsColors[COLCDCC].color5,tmpstr,Colors[COLOFF],
                                CmdsColors[COLCDCC].color1,Client->user,Colors[COLOFF]);
                        say("%s%s",tmpbuf1,tmpbuf2);
#else
                        say("Cdcc closing idle dcc %s %s to %s",
                            dcc_types[flags&DCC_TYPES],tmpstr,Client->user);
#endif
                        send_to_server("NOTICE %s :Cdcc %s %s auto closed.",Client->user,
                                       dcc_types[flags&DCC_TYPES],tmpstr);
                        dcc_erase(Client);
                    }
                }
                else if (timenow-Client->CdccTime>3*CdccIdle) {
#ifdef WANTANSI
                    sprintf(tmpbuf1,"%sCdcc%s %sclosing%s idle dcc %s%s%s",
                            CmdsColors[COLCDCC].color4,Colors[COLOFF],
                            CmdsColors[COLCDCC].color3,Colors[COLOFF],
                            CmdsColors[COLCDCC].color3,dcc_types[flags&DCC_TYPES],
                            Colors[COLOFF]);
                    sprintf(tmpbuf2," %s%s%s from %s%s%s",
                            CmdsColors[COLCDCC].color5,tmpstr,Colors[COLOFF],
                            CmdsColors[COLCDCC].color1,Client->user,Colors[COLOFF]);
                    say("%s%s",tmpbuf1,tmpbuf2);
#else
                    say("Cdcc closing idle dcc %s %s from %s",dcc_types[flags&DCC_TYPES],
                        tmpstr,Client->user);
#endif
                    dcc_erase(Client);
                }
            }
        }
    }
    if (packs && CdccChannels && PlistTime && timenow-LastPlist>PlistTime) {
        for (tmp=packs,packcount=0;tmp;tmp=tmp->next) packcount++;
        if (!my_stricmp(CdccChannels,"current")) {
            current=1;
            channel=get_channel_by_refnum(0);
            if (!channel) {
                NoWindowChannel();
                return;
            }
            chan=lookup_channel(channel,from_server,0);
        }
        oldserver=from_server;
        for (;chan;chan=chan->next) {
            if (current || CheckChannel(chan->channel,CdccChannels)) {
                sprintf(tmpbuf1,"%s %c %d PACK%s OFFERED %c  /CTCP %s %cCDCC%c SEND N for pack N",
                        CdccString,reverse,packcount,packcount==1?empty_string:"S",
                        reverse,get_server_nickname(from_server),bold,bold);
                from_server=chan->server;
                send_text(chan->channel,tmpbuf1,"PRIVMSG");
                count=1;
                for (tmp=packs;tmp;tmp=tmp->next) {
                    sprintf(tmpbuf2,"%c#%c%-3d %-28s  ",bold,bold,count,tmp->description);
                    sprintf(tmpbuf1,"%c[%c%.2f kB%c/%c%d file%s",
                            bold,bold,(float) (tmp->totalbytes)/1024.0,bold,bold,
                            tmp->totalfiles,tmp->totalfiles==1?empty_string:"s");
                    strcat(tmpbuf2,tmpbuf1);
                    if (tmp->minspeed>0.0)
                        sprintf(tmpbuf1,"%c/%cmin %.2f kB/s%c]%c",bold,bold,
                                tmp->minspeed,bold,bold);
                    else sprintf(tmpbuf1,"%c]%c",bold,bold);
                    strcat(tmpbuf2,tmpbuf1);
                    send_text(chan->channel,tmpbuf2,"PRIVMSG");
                    count++;
                }
                if (CdccStats) {
                    sprintf(tmpbuf1,"%s Received %.2f kB  Sent %.2f kB",
                            CdccString,BytesReceived/1024.0,BytesSent/1024.0);
                    send_text(chan->channel,tmpbuf1,"PRIVMSG");
                }
                from_server=oldserver;
            }
            if (current) break;
        }
        LastPlist=timenow;
    }
}

/**********************************************************************
* CheckDCCSpeed:         Checks the pending DCC speed                 *
***********************************************************************/
void CheckDCCSpeed(Client,timenow)
DCC_list *Client;
time_t timenow;
{
    float  rate;
    char   *tmpstr;
#ifdef WANTANSI
    char   tmpbuf1[mybufsize/4];
    char   tmpbuf2[mybufsize/4];
#endif

    rate=((float) (Client->bytes_read)/(float)(timenow-Client->starttime))/1024.0;
    if (rate<Client->minspeed) {
        tmpstr=rindex(Client->description,'/');
        if (tmpstr) tmpstr++;
#ifdef WANTANSI
        sprintf(tmpbuf1,"%sCdcc%s %sclosing%s slow dcc %s%s%s ",
                CmdsColors[COLCDCC].color4,Colors[COLOFF],
                CmdsColors[COLCDCC].color3,Colors[COLOFF],
                CmdsColors[COLCDCC].color3,dcc_types[Client->flags&DCC_TYPES],
                Colors[COLOFF]);
        sprintf(tmpbuf2,"%s%s%s%s to %s%s%s ",
                tmpbuf1,CmdsColors[COLCDCC].color5,tmpstr,Colors[COLOFF],
                CmdsColors[COLCDCC].color1,Client->user,Colors[COLOFF]);
        say("%s(%srate %.2f kB/s%s/%smin %.2f kB/s%s)",tmpbuf2,
            CmdsColors[COLCDCC].color5,rate,Colors[COLOFF],
            CmdsColors[COLCDCC].color5,Client->minspeed,Colors[COLOFF]);
#else
        say("Cdcc closing slow dcc %s %s to %s (rate %.2f kB/s%c/%cmine %.2f kB/s)",
            dcc_types[Client->flags&DCC_TYPES],tmpstr,Client->user,rate,
            bold,bold,Client->minspeed);
#endif
        send_to_server("NOTICE %s :Cdcc %s %s auto closed, dcc was too slow (rate %.2f kB/s%c/%cmin %.2f kB/s)",
                       Client->user,dcc_types[Client->flags&DCC_TYPES],tmpstr,rate,bold,
                       bold,Client->minspeed);
        Client->flags|=DCC_DELETE;
    }
    else {
        if (rate>4*Client->minspeed) Client->CdccTime+=35L;
        else if (rate>2*Client->minspeed) Client->CdccTime+=25L;
        else Client->CdccTime+=15L;
    }
}

/**********************************************************************
* AddToQueue:            Adds file to queue                           *
***********************************************************************/
void AddToQueue(tmpfile,nick,flag)
Files *tmpfile;
char  *nick;
int   flag;
{
    char tmpbuf[mybufsize/4];
    FileQueue *tmp;
    FileQueue *newfile;

    sprintf(tmpbuf,"%s/%s",tmpfile->path,tmpfile->file);
    newfile=(FileQueue *) new_malloc(sizeof(FileQueue));
    newfile->file=(char *) 0;
    newfile->nick=(char *) 0;
    newfile->next=(FileQueue *) 0;
    malloc_strcpy(&(newfile->file),tmpbuf);
    malloc_strcpy(&(newfile->nick),nick);
    newfile->flag=flag;
    if (!queuelist) queuelist=newfile;
    else {
        for (tmp=queuelist;tmp && tmp->next;) tmp=tmp->next;
        tmp->next=newfile;
    }
}

/**********************************************************************
* RemoveFromQueue:       Removes file from queue                      *
***********************************************************************/
void RemoveFromQueue()
{
    int active=TotalSendDcc();
    char tmpbuf[mybufsize/4];
    unsigned int display;
    FileQueue *tmp=queuelist;

    while (tmp && active<CdccLimit) {
        queuelist=queuelist->next;
        sprintf(tmpbuf,"%s %s",tmp->nick,tmp->file);
        display=window_display;
        window_display=0;
        dcc_filesend(tmpbuf);
        window_display=display;
        new_free(&(tmp->nick));
        new_free(&(tmp->file));
        new_free(&tmp);
        active++;
        tmp=queuelist;
    }
}
