// DSTART 
// SmIRC - an X11R6/Motif 2.0 IRC client for Linux 
//  
// Current version is 0.70 
//  
// Copyright 1997-1999, Double Precision, Inc. 
//  
// This program is distributed under the terms of the GNU General Public 
// License. See COPYING for additional information. 
//  
// DEND 
#include	"widget/widget.h"
#include	"widget/widgetxms.h"
#include	"appshell.h"
#include	"smirc.h"
#include	"main.h"

static const char rcsid[]="$Id: appshell.C,v 1.7 1999/04/12 04:39:47 mrsam Exp $";


AppShell::AppShell(const char *name) : CShellWidget(name),
	m_main(NULL), m_server(NULL), m_channel("channel"), deiconify_flag(0)
{
}

// OK.  Here's the scoop how destruction happens.
//
//  SERVER shells:
//
//	1) Window manager close: AppShell::OnClose() is called.
//		A).  Connected to the server.  QUIT command is issued,
//                   and a flag is set.  After a socket is closed, the
//                   code checks whether this flag is set (it is cleared
//                   in all other situations.  If the flag is set,
//                   AppShell::Destroy() is called, which eventually
//                   calls AppShell:OnDestroy() (via a virtual call from
//                   CWidget::OnDestroy() ).  OnDestroy() calls "delete this",
//                   which causes a self-destruction, and AppShell::~AppShell
//                   to be called.
//              B).  Not connected to the server.  CWidget::OnClose() is
//                   called, which causes AppShell::Destroy() to be called
//                   via a virtual pointer, with the same results.
//
//      2) "Exit" menu option is selected.  This calls Channel::Exit(),
//         which duplicates AppShell:OnClose() logic.
//
//      DESTRUCTOR:
//
//      If this is the last server shell, exit(0) is called.
//
//	When Channel::~Channel() is called, it calls Channel::quit().
//	Channel::quit() is also called when a socket EOF is detected, and
//	the self-destruction flag is not set.  Channel::quit() goes through
//	Channel::m_chans and closes all channel windows opened for that
//	server.
//
//  CHANNEL shells:
//
//	1) Window manager close: AppShell::OnClose() is called.
//         AppShell::OnClose() calls Channel::CmdPART().  If we are connected
//         to a server (we better!), we issue a PART() command.
//         AppShell::OnClose() continues to call CWidget::OnClose(), which
//         calls AppShell::Destroy(), which eventually gets around to
//         AppShell::OnDestroy(), which deletes this, which gets around
//         to AppShell::~AppShell(), where this pointer is removed from
//         it's server's Channel::m_chans list.  This provides for
//         an instantaneous closure of the channel window, while still
//         issueing the PART command.
//      2) User manually issues a /PART command.  This eventually causes the
//         server to send us the /PART message back.  When this message is
//         received,  AppShell::Destroy() gets called, destroying the whole
//         window.  AppShell::Destroy() is also called when we get the message
//         of being kicked off the channel.

AppShell::~AppShell()
{
	if (m_main)
	{
		m_main->m_channels.RemoveAt(m_pos);
		if (m_main->m_channels.IsEmpty())
			CApplicationWidget::Quit();
			// When all server windows are closed,
			// this application is done.
	}
	if (m_server)
	{
		m_server->m_channel.m_chans.RemoveKey(IrcLower(m_channel_name));
	}

}

void AppShell::OnDestroy()
{
	CShellWidget::OnDestroy();
	delete this;
}

void AppShell::OnClose()
{
	try
	{
		// If this is a server window, check to see if we are connected
		// to the server, if so, be nice and send the QUIT command.

		if (m_main && !m_channel.IsChannel()
			&& m_channel.IsConnected())
		{
			m_channel.Exit();
			return;
		}

		if (m_channel.IsChannel() && !m_channel.IsPrivateChannel())
			m_channel.CmdPART("");
				// If user closed a channel window, issue
				// the PART command
	}
	catch (...)
	{
		CShellWidget::OnClose();
		throw;
	}
	CShellWidget::OnClose();
}

// This create function is called to initialize a shell for a server window

void AppShell::Create(CMain *parent)
{
	m_pos=parent->m_channels.AddTail(this);
	m_main=parent;

	CShellWidget::Create(DefaultTitle());
	m_channel.Create(this);
	m_channel.Manage();
}

// This create function is called to initialize a shell for a channel window

void AppShell::Create(AppShell *server, CString channel_name)
{
	m_channel_name=channel_name;
	m_channel.m_currentnick=server->m_channel.m_currentnick;
	m_channel.m_server=&server->m_channel;

	m_server=server;
	server->m_channel.m_chans[IrcLower(m_channel_name)]=this;
	CShellWidget::Create("");
	m_channel.Create(this);
	m_channel.Manage();
	Title(DefaultTitle());	// Can't really call DefaultTitle(),
				// until the channel is created.
}

// Renaming this channel: used for updating private message windows
// after a nick change.  We not only need to update m_channel_name,
// but also update the server's channel map

void AppShell::Rename(CString new_channel_name)
{
CString	loldname(IrcLower(m_channel_name));
CString lnewname(IrcLower(new_channel_name));

	if (loldname != lnewname)
	{
		m_server->m_channel.m_chans[lnewname]=this;
		m_server->m_channel.m_chans.RemoveKey(loldname);
	}
	m_channel_name=new_channel_name;
}



CString	AppShell::DefaultTitle()
{
	if (!m_server)
		return (
		"SmIRC " VERSION " Copyright 1997-1999, Double Precision, Inc." );

CStringArray	args;

	args.SetSize(4);
	args[0]=m_channel.m_currentnick;
	args[1]=m_channel_name;
	args[2]=m_channel.m_currenttopic;
	args[3]=m_channel.m_currentmode + "";
CXmString	s( m_server->m_main->
			GetString( (m_channel.IsPrivateChannel() ?
				"PRIVATECHANNELTITLE":"CHANNELTITLE"),
					args ));

	return ( (CString)s );
}

CString	AppShell::ConnectedTitle(CString system, CString nickname)
{
	if (!m_main)	return (DefaultTitle());

CStringArray	args;

	args.SetSize(3);
	args[0]=system;
	args[1]=nickname;
	args[2]=m_channel.m_currentmode + "";

CXmString	s( m_main->GetString("SERVERTITLE", args ));

	return ( "SmIRC " VERSION " - " + (CString)s );
}

void	AppShell::AutoClose()
{
	m_channel.Exit();
}

void	AppShell::Iconify()
{
	if (!CShellWidget::Iconic())
		m_channel.LogStrMessage("ICONIFIED");
	CShellWidget::Iconify();
}

void	AppShell::DeIconify()
{
	if (!deiconify_flag)
	{
		try
		{
			deiconify_flag=1;
			CShellWidget::DeIconify();
			if (CShellWidget::Iconic())
				m_channel.LogStrMessage("DEICONIFIED");
			deiconify_flag=0;
		}
		catch (...)
		{
			deiconify_flag=0;
			throw;
		}
	}
}
