/*
** Copyright (C) 1999 Erik de Castro Lopo <erikd@zip.com.au>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#include	<math.h>

#include	"sndfile.h"
#include	"config.h"
#include	"sfendian.h"
#include	"common.h"
#include	"wav.h"


typedef struct
{	unsigned int	channels, blocksize, samplesperblock, blocks, dataremaining ; 
	int				blockcount, samplecount ;
	int				previous [2] ;
	int				stepindex [2] ;
	unsigned char	*block ;
	short			*samples ;
	unsigned char	data	[4] ; /* Dummy size */
} MSADPCM_PRIVATE ;

/*============================================================================================
** MS ADPCM static data and functions.
*/

static int gaiP4 []    = 
{	230, 230, 230, 230, 307, 409, 512, 614,
	768, 614, 512, 409, 307, 230, 230, 230 
} ;

/* TODO : The first 7 coef's are are always hardcode and must
   appear in the actual WAVE file.  They should be read in
   in case a sound program added extras to the list. */

static int gaiCoef1[] = 
{	256, 512, 0, 192, 240, 460, 392 
} ;

static int gaiCoef2[] = 
{	0, -256, 0, 64, 0, -208, -232
} ;

static	int	__msadpcm_read_block (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms) ;
static	int __msadpcm_read (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms, short *ptr, int len) ;

/* 	When the encoder works we'll need these function prototypes.
**
**	static	int	__msadpcm_write_block (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms) ;
**	static	int __msadpcm_write (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms, short *ptr, int len) ;
*/

/*============================================================================================
** MS ADPCM Read Functions.
*/

int	__msadpcm_reader_init (SF_PRIVATE *psf, WAV_FMT *fmt)
{	MSADPCM_PRIVATE	*pms ;
	unsigned int	pmssize ;
	int				count ;
	
	pmssize = sizeof (MSADPCM_PRIVATE) + fmt->msadpcm.blockalign + 3 * fmt->msadpcm.channels * fmt->msadpcm.samplesperblock ;

	if (! (psf->fdata = malloc (pmssize)))
		return SFE_MALLOC_FAILED ;
	pms = (MSADPCM_PRIVATE*) psf->fdata ;
	memset (pms, 0, pmssize) ;
	
	pms->block   = (unsigned char*) pms->data ;
	pms->samples = (short*) (pms->data + fmt->msadpcm.blockalign) ;
	
	pms->channels        = fmt->msadpcm.channels ;
	pms->blocksize       = fmt->msadpcm.blockalign ;
	pms->samplesperblock = fmt->msadpcm.samplesperblock ;

	pms->dataremaining	 = psf->datalength ;

	if (psf->datalength % pms->blocksize)
		pms->blocks = psf->datalength / pms->blocksize  + 1 ;
	else
		pms->blocks = psf->datalength / pms->blocksize ;

	count = 2 * (pms->blocksize - 6 * pms->channels) / pms->channels ;
	if (pms->samplesperblock != count)
		__psf_sprintf (psf, "*** Warning : samplesperblock shoud be %d.\n", count) ;

	psf->sf.samples = (psf->datalength / pms->blocksize) * pms->samplesperblock ;
	psf->sf.samples += 2 * (psf->datalength - ((psf->datalength / pms->blocksize) * pms->blocksize)
						- 6 * pms->channels) / pms->channels ;

	__msadpcm_read_block (psf, pms) ;
	
	return 0 ;
} /* __msadpcm_reader_init */

static
int		__msadpcm_read_block (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms)
{	int		chan, k, blockindex, index [2], sampleindex ;
	short	bytecode, bpred [2] ;
	
    int predict ;
    int current ;
    int idelta ;

	pms->blockcount ++ ;
	pms->samplecount = 0 ;
	
	if (pms->blockcount > pms->blocks)
	{	memset (pms->samples, 0, pms->samplesperblock * pms->channels) ;
		return 1 ;
		} ;

	if ((k = fread (pms->block, 1, pms->blocksize, psf->file)) != pms->blocksize)
		__psf_sprintf (psf, "*** Warning : short read (%d != %d).\n", k, pms->blocksize) ;

	/* Read and check the block header. */
	
	blockindex = 0 ;
	bpred [0] = pms->block [blockindex++] ;
	bpred [1] = (pms->channels > 1) ? pms->block [blockindex++] : 0 ;
	
	if (bpred [0] >= 7 || bpred [1] >= 7)
		__psf_sprintf (psf, "MS ADPCM synchronisation error (%d %d).\n", bpred [0], bpred [1]) ;
	
	index [0] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
	blockindex += 2 ;
	
	if (pms->channels > 1)
	{	index [1] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
		blockindex += 2 ;
		} ;

    if (index [0] & 0x8000) 
		index [0] -= 0x10000 ;
    if (index [1] & 0x8000) 
		index [1] -= 0x10000 ;

	pms->samples [pms->channels] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
	blockindex += 2 ;
	if (pms->channels > 1)
	{	pms->samples [pms->channels+1] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
		blockindex += 2 ;
		} ;

	pms->samples [0] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
	blockindex += 2 ;
	if (pms->channels > 1)
	{	pms->samples [1] = pms->block [blockindex] | (pms->block [blockindex+1] << 8) ;
		blockindex += 2 ;
		} ;
		
	/* Pull apart the packed 4 bit samples and store them in their
	** correct sample positions.
	*/
	
	sampleindex = 2 * pms->channels ;
	while (blockindex <  pms->blocksize)
	{	bytecode = pms->block [blockindex++] ;
  		pms->samples [sampleindex++] = (bytecode >> 4) & 0x0F ;
		pms->samples [sampleindex++] = bytecode & 0x0F ;
		} ;
		
	/* Decode the encoded 4 bit samples. */
	
	for (k = 2 * pms->channels ; k < (pms->samplesperblock * pms->channels) ; k ++)
	{	chan = (pms->channels > 1) ? (k % 2) : 0 ;

		bytecode = pms->samples [k] & 0xF ;

	    /** Compute next Adaptive Scale Factor (ASF) **/
	    idelta = index [chan];
	    index [chan] = (gaiP4 [bytecode] * idelta) >> 8;
	    if (index [chan] < 16) 
			index [chan] = 16;
	    if (bytecode & 0x08) 
			bytecode = bytecode - 0x10;
		
    	predict = ((pms->samples [k - pms->channels] * gaiCoef1 [bpred [chan]]) 
					+ (pms->samples [k - 2 * pms->channels] * gaiCoef2 [bpred [chan]])) >> 8;
    	current = (bytecode * idelta) + predict;
    
	    if (current > 32767) 
			current = 32767;
	    else if (current < -32768) 
			current = -32768;
    
		pms->samples [k] = current ;
		} ;

	return 1 ;
} /* __msadpcm_read_block */

static
int __msadpcm_read (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms, short *ptr, int len)
{	int		count, total = 0, index = 0 ;
	
	while (index < len)
	{	if (pms->blockcount >= pms->blocks && pms->samplecount >= pms->samplesperblock)
		{	memset (&(ptr[index]), 0, (len - index) * sizeof (short)) ;
			return total ;
			} ;
		
		if (pms->samplecount >= pms->samplesperblock)
			__msadpcm_read_block (psf, pms) ;
		
		count = (pms->samplesperblock - pms->samplecount) * pms->channels ;
		count = (len - index > count) ? count : len - index ;
		
		memcpy (&(ptr[index]), &(pms->samples [pms->samplecount * pms->channels]), count * sizeof (short)) ;
		index += count ;
		pms->samplecount += count / pms->channels ;
		total = index ;
		} ;

	return total ;		
} /* __msadpcm_read */

int		__msadpcm_read_s (SF_PRIVATE *psf, short *ptr, int len)
{	MSADPCM_PRIVATE 	*pms ; 
	int				total ;

	if (! psf->fdata)
		return 0 ;
	pms = (MSADPCM_PRIVATE*) psf->fdata ;
	
	total = __msadpcm_read (psf, pms, ptr, len) ;

	return total ;
} /* __msadpcm_read_s */

int		__msadpcm_read_i  (SF_PRIVATE *psf, int *ptr, int len)
{	MSADPCM_PRIVATE *pms ; 
	short		*sptr ;
	int			k, bufferlen, readcount = 0, count ;
	int			index = 0, total = 0 ;

	if (! psf->fdata)
		return 0 ;
	pms = (MSADPCM_PRIVATE*) psf->fdata ;
	
	sptr = (short*) psf->buffer ;
	bufferlen = ((SF_BUFFER_LEN / psf->blockwidth) * psf->blockwidth) / sizeof (short) ;
	while (len > 0)
	{	readcount = (len >= bufferlen) ? bufferlen : len ;
		count = __msadpcm_read (psf, pms, sptr, readcount) ;
		for (k = 0 ; k < readcount ; k++)
			ptr [index+k] = (int) (sptr [k]) ;
		index += readcount ;
		total += count ;
		len -= readcount ;
		} ;
	return total ;
} /* __msadpcm_read_i */

int		__msadpcm_read_d  (SF_PRIVATE *psf, double *ptr, int len, int normalize)
{	MSADPCM_PRIVATE *pms ; 
	short		*sptr ;
	int			k, bufferlen, readcount = 0, count ;
	int			index = 0, total = 0 ;

	if (! psf->fdata)
		return 0 ;
	pms = (MSADPCM_PRIVATE*) psf->fdata ;
	
	sptr = (short*) psf->buffer ;
	bufferlen = ((SF_BUFFER_LEN / psf->blockwidth) * psf->blockwidth) / sizeof (short) ;
	while (len > 0)
	{	readcount = (len >= bufferlen) ? bufferlen : len ;
		count = __msadpcm_read (psf, pms, sptr, readcount) ;
		for (k = 0 ; k < readcount ; k++)
			ptr [index+k] = (double) (sptr [k]) ;
		index += readcount ;
		total += count ;
		len -= readcount ;
		} ;
	return total ;
} /* __msadpcm_read_d */


off_t    __msadpcm_seek   (SF_PRIVATE *psf, off_t offset, int whence)
{	MSADPCM_PRIVATE *pms ; 
	int			newblock, newsample ;
	
	if (! psf->fdata)
		return 0 ;
	pms = (MSADPCM_PRIVATE*) psf->fdata ;

	if (! (psf->blockwidth && psf->datalength && psf->dataoffset))
	{	psf->error = SFE_BAD_SEEK ;
		return	((off_t) -1) ;
		} ;
		
	switch (whence)
	{	case SEEK_SET :
				if (offset < 0 || offset > pms->blocks * pms->samplesperblock)
				{	psf->error = SFE_BAD_SEEK ;
					return	((off_t) -1) ;
					} ;
				newblock  = offset / pms->samplesperblock ;
				newsample = offset % pms->samplesperblock ;
				break ;
				
		case SEEK_CUR :
				if (psf->current + offset < 0 || psf->current + offset > pms->blocks * pms->samplesperblock)
				{	psf->error = SFE_BAD_SEEK ;
					return	((off_t) -1) ;
					} ;
				newblock  = (psf->current + offset) / pms->samplesperblock ;
				newsample = (psf->current + offset) % pms->samplesperblock ;
				break ;
				
		case SEEK_END :
				if (offset > 0 || pms->samplesperblock * pms->blocks + offset < 0)
				{	psf->error = SFE_BAD_SEEK ;
					return	((off_t) -1) ;
					} ;
				newblock  = (pms->samplesperblock * pms->blocks + offset) / pms->samplesperblock ;
				newsample = (pms->samplesperblock * pms->blocks + offset) % pms->samplesperblock ;
				break ;
				
		default : 
				psf->error = SFE_BAD_SEEK ;
				return	((off_t) -1) ;
		} ;
		
	if (psf->mode == SF_MODE_READ)
	{	fseek (psf->file, (int) (psf->dataoffset + newblock * pms->blocksize), SEEK_SET) ;
		pms->blockcount  = newblock ;
		__msadpcm_read_block (psf, pms) ;
		pms->samplecount = newsample ;
		}
	else
	{	/* What to do about write??? */ 
		psf->error = SFE_BAD_SEEK ;
		return	((off_t) -1) ;
		} ;

	psf->current = newblock * pms->samplesperblock + newsample ;
	return psf->current ;
} /* __msadpcm_seek */

/*==========================================================================================
** MS ADPCM Write Functions.
*/

/*========================================================================================
** Function skeletons for when I work out how to do an MS ADPCM encoder. 
*/

#ifdef	SKELETON_CODE_DO_NOT_COMPILE

int	__msadpcm_writer_init (SF_PRIVATE *psf, WAV_FMT *fmt)
{	MSADPCM_PRIVATE	*pms ;
	int 		pmssize ;
	
	if (fmt->format != 0x0002)
		__psf_sprintf (psf, "*** Warning : format tag != WAVE_FORMAT_MS_ADPCM.\n") ;
	
	return 0 ;
} /* __msadpcm_writer_init */

/*==========================================================================================
*/

static
int		__msadpcm_write_block (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms)
{				
	return 1 ;
} /* __msadpcm_write_block */

static
int __msadpcm_write (SF_PRIVATE *psf, MSADPCM_PRIVATE *pms, short *ptr, int len)
{	int		count, total = 0, index = 0 ;
	
	return total ;		
} /* __msadpcm_write */

int		__msadpcm_write_s (SF_PRIVATE *psf, short *ptr, int len)
{	MSADPCM_PRIVATE 	*pms ; 
	return total ;
} /* __msadpcm_write_s */

int		__msadpcm_write_i  (SF_PRIVATE *psf, int *ptr, int len)
{	MSADPCM_PRIVATE *pms ; 
	return total ;
} /* __msadpcm_write_i */

int		__msadpcm_write_d  (SF_PRIVATE *psf, double *ptr, int len, int normalize)
{	MSADPCM_PRIVATE *pms ; 
	return total ;
} /* __msadpcm_write_d */


#endif  

/*
** End of function skeletons. 
**========================================================================================
*/

int	__msadpcm_close	(SF_PRIVATE  *psf)
{	MSADPCM_PRIVATE *pms ; 
	unsigned int		dword ;

	if (! psf->fdata)
		return __wav_close (psf) ;

	pms = (MSADPCM_PRIVATE*) psf->fdata ;

	if (psf->mode == SF_MODE_WRITE)
	{	/*  Now we know for certain the length of the file we can
		**  re-write correct values for the RIFF and data chunks.
		*/
		 
		/*	When the encoder is working correctly, we probably 
		**	want to do this : 
		**	
		**	if (pms->samplecount && pms->samplecount < pms->samplesperblock)
		**		__msadpcm_write_block (psf, pms) ;	
		*/

		fseek (psf->file, 0, SEEK_END) ;
		psf->filelength = ftell (psf->file) ;

		/* Fix RIFF size. */
		dword = H2LE_INT (psf->filelength - 2 * sizeof (dword)) ;
		fseek (psf->file, sizeof (dword), SEEK_SET) ;
		fwrite (&dword, sizeof (dword), 1, psf->file) ;
		
		psf->datalength = psf->filelength - psf->dataoffset ;
		fseek (psf->file, (int) (psf->dataoffset - sizeof (dword)), SEEK_SET) ;
		dword = H2LE_INT (psf->datalength) ;
		fwrite (&dword, sizeof (dword), 1, psf->file) ;
		} ;

	if (psf->fdata)
		free (psf->fdata) ;
	psf->fdata = NULL ;

	return 0 ;
} /* __msadpcm_close */

