/*
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix)
*/
/* 
   gtk_tb.c: A GTK Toolbox
   Copyright (C) 1999 Tano Fotang

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free 
   Software Foundation; either version 2 of the License , or (at your option) 
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
   or FITNESS FOR A PARTICULAR PURPOSE. See the file LICENCE for details.

   You should have received a copy of the GNU General Public License along
   with this program; see the file COPYING.  If not, write to the Free
   Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 */

#include "gtk_tb.h"
#include <ctype.h>
#include <assert.h>
#include <string.h>

int spx_add_colour(const char *name);
int spx_max_colour;
char spx_format_char;           /* @ */
static void initialize_fonts(void);
static void initialize_colours(void);

/*

   0) int gtk_initialize(int *argc, char **argv,  const char *app,
   GTK_CMD_OPT cmd_opt_table, size_t table_size);
   argc, argv
   Command line parameters. Returns the values left
   after parsing.
   app_class_name
   The application class name. Should not contain '.',
   '?', or '*'. If inappropriate, modified copy will be made.
   cmd_opt_table
   A table of application command line arguments to be parsed.
   table_size
   The size of the table.

   gtk_initialize initializes the application resource database. See
   XrmParseCommand(3X11). Recognized options are removed from argv and
   new values for argc and argv returned.
   Database is built in the following order:
   0. /usr/X11/lib/X11/app-defaults/<app_class_name>
   1. $XAPPLRESDIR/<app_class_name>
   2. RESOURCE_MANAGER property or ~/.Xdefaults if RESOURCE_MANAGER is empty
   3. $XENVIRONMENT or ~/.Xdefaults-<hostname> if XENVIRONMENT is empty
   4. command line options

   gtk_initialize should be called after gtk_init since gtk_initialize
   depends uses gdk_display and g_get_prgname. You may prefer to just
   append gtk_initialize to the gtk_init instead of calling the former
   directly. Note, however, that gtk_init takes &argv while gtk_initialize
   takes just argv.

   1) void gtk_get_app_resources(GTK_resource *resource, size_t size);
   resource      A table of application resource
   size          Table size

   Retrieves the values of a list of application resources.
   May only be called after gtk_initialize.

   2) char *gtk_get_app_resource(const char *str_name, const char *str_class,
   GTK_RESOURCE_TYPE  type, const char *defval,
   void *buf, size_t bufsize);
   str_name
   The name of the resource to be retrieved, but
   excluding the leading application [class] name.
   E.g. "NC.debug" instead of myApp.NC.debug
   str_class
   The class name of the value to be retrieved.
   type
   The data type of the value 
   buf
   Buffer where retrieved value should be stored
   defval
   A default value (in string form) if resource not found
   bufsize
   Size of buffer for string data (type==GTK_STRING)

   Retrieves a single resource value.
   May only be called after gtk_initialize.

   3) TODO
   gtk_put/set_resource(const char *, [GTK_RESOURCE_TYPE,] const char *)
   ----    

   Corections to fotang@yahoo.com. 15Apr99.
 */

/* defined by gtk/gdk */
extern Display *gdk_display;
extern char *g_get_prgname();

/*
   static char *App_class_name2
   This is the application class name as used in system X resource files.
   It is usually the application name with the first character capitalized.
   We are given the name in the call to gtk_initialize.
   App_class_name2 is set to NULL if it is the same as program name.

 */
static char *App_class_name2;
static XrmDatabase app_res_db;

int gtk_initialize(int *argc,
                   char **argv,
                   const char *app_class_name,
                   GTK_CMD_OPT * cmd_opt,
                   size_t cmd_opt_count)
{
  XrmDatabase res_db = (XrmDatabase) 0;
  char bar[MAXPATHLEN];
  char *str;
  Atom atom;
  char found = 0;
  char *cl_name;

  XrmInitialize();
  app_res_db = (XrmDatabase) 0;

  cl_name = g_strdup(app_class_name);
  if (*cl_name)
  {
    char *p = cl_name;

    while (isspace(*p))         /* remove leading blanks */
      p++;
    if (p != cl_name)
    {                           /* overwrite leading blanks */
      memmove(cl_name, p, strlen(p) + 1);
      p = cl_name;
    }
    while (*p && !isspace(*p))  /* remove terminating blanks */
      p++;
    *p = 0;
  }
  if (*cl_name == 0)
  {
    g_free(cl_name);
    cl_name = g_strdup(g_get_prgname());
    if (*cl_name == 'x')        /* xdemo --> Xdemo */
      *cl_name = 'X';
    /* Xdemo --> XDemo
       demo  --> Demo
     */
    if (*cl_name == 'X')
    {
      if (*(cl_name + 1))
        *(cl_name + 1) = toupper(*(cl_name + 1));
    }
    else
      *cl_name = toupper(*cl_name);
  }
  if (!strcmp(g_get_prgname(), cl_name))
    App_class_name2 = NULL;
  else
    App_class_name2 = g_strdup(cl_name);
  sprintf(bar, "/usr/X11/lib/X11/app-defaults/%s", cl_name);
#if DEBUG_XRM
  fprintf(stderr, "Trying system default '%s'", bar);
#endif
  app_res_db = XrmGetFileDatabase(bar);
#if DEBUG_XRM
  fprintf(stderr, ". %s\n", app_res_db ? "ok" : "failed");
  fprintf(stderr, "Trying $XAPPLRESDIR");
#endif
  str = getenv("XAPPLRESDIR");
  if (str)
  {
    sprintf(bar, "%s/%s", str, cl_name);
    res_db = XrmGetFileDatabase(bar);
#if DEBUG_XRM
    fprintf(stderr, " '%s'. %s\n", bar, res_db ? "ok" : "failed");
#endif
    if (res_db)
      XrmMergeDatabases(res_db, &app_res_db);
  }
#if DEBUG_XRM
  else
    fprintf(stderr, "\n");
  fprintf(stderr, "Loading RESOURCE_MANAGER");
#endif
  atom = XInternAtom(gdk_display, "RESOURCE_MANAGER", True);
  if (atom != None)
  {                             /* what about BadAlloc and BadValue ? */
    unsigned char *buf = 0;
    Atom actual_type = None;
    int actual_fmt;
    unsigned long nitems_returned,
      nbytes_left;
    Status result;

    result = XGetWindowProperty(gdk_display,
                                RootWindow(gdk_display, 0),
                                atom,
                                0L, 1L,	/* 32-bit segments */
                                False, XA_STRING,
                                &actual_type,
                                &actual_fmt,
                                &nitems_returned,	/* 8-bit items */
                                &nbytes_left,	/* byte count */
                                &buf);
    if (result == Success && actual_type == XA_STRING && actual_fmt == 8)
    {
      if (buf)
        XFree(buf);
      result = XGetWindowProperty(gdk_display,
                                  RootWindow(gdk_display, 0),
                                  atom, 0L,
                               (long) ((nitems_returned + nbytes_left) / 4),
                                  False, XA_STRING,
                                  &actual_type, &actual_fmt,
                                  &nitems_returned,
                                  &nbytes_left, &buf);
      if (result == Success && actual_type == XA_STRING &&
          actual_fmt == 8 && buf)
      {
        res_db = XrmGetStringDatabase((char *) buf);
        XFree(buf);
        if (res_db)
        {
          XrmMergeDatabases(res_db, &app_res_db);
#if DEBUG_XRM
          fprintf(stderr, ". ok\n");
#endif
          found = 1;
        }
      }
      else if (nitems_returned > 0 && buf)
        XFree(buf);
    }
  }
  if (!found)                   /* try ~/.Xdefaults */
  {
    str = getenv("HOME");
    if (!str)
      str = ".";
    sprintf(bar, "%s/.Xdefaults", str);
#if DEBUG_XRM
    fprintf(stderr, " failed.\n Trying ~/.Xdefaults '%s'\n", bar);
#endif
    res_db = XrmGetFileDatabase(bar);
#if DEBUG_XRM
    fprintf(stderr, ". %s\n", res_db ? "ok" : "failed");
#endif
    if (res_db)
      XrmMergeDatabases(res_db, &app_res_db);
  }
  found = 0;
#if DEBUG_XRM
  fprintf(stderr, "Trying to merge $XENVIRONMENT\n");
#endif
  str = getenv("XENVIRONMENT");
  if (str)
  {
    res_db = XrmGetFileDatabase(str);
#if DEBUG_XRM
    fprintf(stderr, " %s: %s\n", str, res_db ? "ok" : "failed");
#endif
    if (res_db)
    {
      XrmMergeDatabases(res_db, &app_res_db);
      found = 1;
    }
  }
  if (!found)                   /* try ~/.Xdefaults-hostname */
  {
    char *host;

#if DEBUG_XRM
    fprintf(stderr, "Trying ~/.Xdefaults-<hostname>\n");
#endif
    host = getenv("HOSTNAME");
    if (host)
    {
      str = getenv("HOME");
      if (!str)
        str = ".";
      sprintf(bar, "%s/.Xdefaults-%s", str, host);
#if DEBUG_XRM
      fprintf(stderr, "Trying to merge '%s'", bar);
#endif
      res_db = XrmGetFileDatabase(bar);
#if DEBUG_XRM
      fprintf(stderr, ". %s\n", res_db ? "ok" : "failed");
#endif
      if (res_db)
        XrmMergeDatabases(res_db, &app_res_db);
    }
  }

  res_db = (XrmDatabase) 0;

  XrmParseCommand(&res_db,
                  cmd_opt, cmd_opt_count, cl_name, argc, argv);
  if (res_db)
    XrmMergeDatabases(res_db, &app_res_db);
  g_free(cl_name);
  if (app_res_db)
    XrmSetDatabase(gdk_display, app_res_db);
/* moved below for Sula Primerix:
  initialize_fonts();
  initialize_colours();
*/
  return 1;
}
void init_fonts(void)
{
  initialize_fonts();
  initialize_colours();
}
static void *get_resource(const char *str_name,
                          const char *str_class,
                          const char *defval)
{

  char *ret_type;
  XrmValue ret;
  char *name;
  Status result;

  name = (char *) g_malloc0(sizeof(char) * (strlen(str_name) + FILENAME_MAX + 2));	/* hmmmm... */

  if (App_class_name2)
    sprintf(name, "%s.%s", App_class_name2, str_name);
  else
    sprintf(name, "%s.%s", g_get_prgname(), str_name);
#if DEBUG_XRM
  fprintf(stderr, "Reading %s (%s)\n", name, str_class);
#endif
  result = XrmGetResource(app_res_db, name,
                          str_class, &ret_type,
                          &ret);
  if (result == False && App_class_name2)
  {
    sprintf(name, "%s.%s", g_get_prgname(), str_name);
#if DEBUG_XRM
    fprintf(stderr, "  Trying %s (%s)\n", name, str_class);
#endif
    result = XrmGetResource(app_res_db, name,
                            str_class, &ret_type,
                            &ret);
  }
  if (result == False)
  {
#if DEBUG_XRM
    fprintf(stderr, "\t%s (%s) not found\n", str_name, str_class);
#endif
    ret.addr = (XPointer) defval;
  }
  g_free(name);
  return (void *) ret.addr;
}

char *gtk_get_app_resource(const char *str_name,
                           const char *str_class,
                           GTK_RESOURCE_TYPE type,
                           const char *defval,
                           void *buf,
                           size_t bufsize)
{
  void *val;

  val = get_resource(str_name, str_class, defval);
  switch (type)
   {
     case GTK_SHORT:
       sscanf((char *) val, "%hd", (short *) buf);
       break;
     case GTK_BOOL:
       {
         char *str = g_strdup((char *) val);
         char *s = str;

         while ((*s = tolower(*s)))
           s++;
         *((int *) buf) = (int) (
                                  (*str == '1' && *(str + 1) == 0) ||
                    (*str == 'o' && *(str + 1) == 'n' && *(str + 2) == 0) ||
                                  (*str == 'y' && *(str + 1) == 'e' && *(str + 2) == 's' && *(str + 3) == 0) ||
                                  (*str == 't' && *(str + 1) == 'r' && *(str + 2) == 'u' && *(str + 3) == 'e' &&
                                   *(str + 4) == 0));
         g_free(str);
         break;
       }
     case GTK_INT:
       sscanf((char *) val, "%d", (int *) buf);
       break;
     case GTK_LONG:
       sscanf((char *) val, "%ld", (long *) buf);
       break;
     case GTK_STRING:
       memcpy(buf, val, bufsize);
       *((char *) buf + bufsize) = 0;
       break;
     case GTK_FLOAT:
       sscanf((char *) val, "%f", (float *) buf);
       break;
     case GTK_DOUBLE:
       sscanf((char *) val, "%lf", (double *) buf);
       break;
     default:
       break;
   }
  return (char *) val;
}

void gtk_get_app_resources(GTK_resource * res,
                           size_t count)
{
  GTK_resource *p = res,
   *end = (res + count);

  for (; p != end; p++)
    (void) gtk_get_app_resource(p->str_name, p->str_class, p->type,
                                p->defval, p->buf, p->size);
}

/*
   some functions for displaying text in text widgets
 */
int _spx_load_file(GtkWidget * text, const char *f, int expand)
{
  FILE *fp = fopen(f, "r");

  if (fp)
  {
#define RD_BUFSIZE 1024
    char buffer[RD_BUFSIZE + 1];

    gtk_text_freeze(GTK_TEXT(text));
    if (expand)
      while (fgets(buffer, RD_BUFSIZE, fp))
        spx_write(text, buffer);
    else
    {
      int nchars;

      while (1)
      {
        nchars = fread(buffer, 1, 1024, fp);
        gtk_text_insert(GTK_TEXT(text), NULL, NULL,
                        NULL, buffer, nchars);
        if (nchars < RD_BUFSIZE)
          break;
      }
    }
    fclose(fp);
    gtk_text_thaw(GTK_TEXT(text));
    return 0;
  }
  return -1;
}

static GdkFont *spx_font[STYLE_INVALID + 1];
__inline GdkFont *spx_idx2font(Font_idx n)
{
  return spx_font[n % STYLE_INVALID];
}

/* flags */
#define UNDERLINE	(1<<0)
#define INVERSE		(1<<1)
#define _HALTED    (1<<2)       /* @. in effect */
#define DEFAULT_FLAG ~(INVERSE|UNDERLINE|_HALTED)
/* style */
#define BOLD_FONT   (1<<0)
#define ITALIC_FONT	(1<<1)
#define FIXED_FONT	(1<<2)
#define NORMAL_FONT	(1<<3)
#define TIMES_FONT  (1<<4)
#define INVALID_FONT	(1<<5)

typedef struct
{
  Font_idx idx;
  char *res;
  char *def;                    /* default font */
  char *alt;                    /* alternative font */
}
Fontstyle;
Fontstyle fstyle[STYLE_INVALID + 1] =
{

/*small */
  {STYLE_SMALL, "spxFont.Small",
   "-*-times-medium-r-*-*-10-*-*-*-*-*-*-*", 0},
  {STYLE_BOLD_SMALL, "spxFont.Small.Bold",
   "-*-times-bold-r-*-*-10-*-*-*-*-*-*-*",
   NULL},
  {STYLE_ITALIC_SMALL, "spxFont.Small.Italic",
   "-*-times-medium-i-*-*-10-*-*-*-*-*-*-*",
   NULL},
  {STYLE_ITALICBOLD_SMALL, "spxFont.Small.Italic.Bold",
   "-*-times-bold-i-*-*-10-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXED_SMALL, "spxFont.Small.Fixed",
   "-*-lucidatypewriter-medium-r-*-*-10-*",
   "-*-courier-medium-r-*-*-10-*-*-*-*-*-*-*"},
  {STYLE_FIXEDBOLD_SMALL, "spxFont.Small.Fixed.Bold",
   "-*-lucidatypewriter-bold-r-*-*-10-*",
   "-*-courier-bold-r-*-*-10-*-*-*-*-*-*-*"},
  {STYLE_FIXEDITALIC_SMALL, "spxFont.Small.Fixed.Italic",
   "-*-courier-medium-i-*-*-10-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXEDBOLDITALIC_SMALL, "spxFont.Small.Fixed.Italic.Bold",
   "-*-courier-bold-i-*-*-10-*-*-*-*-*-*-*",
   NULL},
/*normal */
  {STYLE_NORMAL, "spxFont",
   "-*-times-medium-r-*-*-14-*-*-*-*-*-*-*",
   NULL},
  {STYLE_BOLD, "spxFont.Bold",
   "-*-times-bold-r-*-*-14-*-*-*-*-*-*-*",
   NULL},
  {STYLE_ITALIC, "spxFont.Italic",
   "-*-times-medium-i-*-*-14-*-*-*-*-*-*-*",
   NULL,
  },
  {STYLE_ITALICBOLD, "spxFont.Italic.Bold",
   "-*-times-bold-i-*-*-14-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXED, "spxFont.Fixed",
   "-*-lucidatypewriter-medium-r-normal-*-12-*",
   "-*-courier-medium-r-*-*-14-*-*-*-*-*-*-*"},
  {STYLE_FIXEDBOLD, "spxFont.Fixed.Bold",
   "-*-lucidatypewriter-bold-r-normal-*-12-*",
   "-*-courier-bold-r-*-*-14-*-*-*-*-*-*-*"},
  {STYLE_FIXEDITALIC, "spxFont.Fixed.Italic",
   "-*-courier-medium-i-*-*-14-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXEDBOLDITALIC, "spxFont.Fixed.Italic.Bold",
   "-*-courier-bold-i-*-*-14-*-*-*-*-*-*-*",
   NULL},
/*large */
  {STYLE_LARGE, "spxFont.Large",
   "-*-times-medium-r-*-*-28-*-*-*-*-*-*-*",
   NULL},
  {STYLE_BOLD_LARGE, "spxFont.Large.Bold",
   "-*-times-bold-r-*-*-28-*-*-*-*-*-*-*",
   NULL},
  {STYLE_ITALIC_LARGE, "spxFont.Large.Italic",
   "-*-times-medium-i-*-*-28-*-*-*-*-*-*-*",
   NULL},
  {STYLE_ITALICBOLD_LARGE, "spxFont.Large.Italic.Bold",
   "-*-times-bold-i-*-*-28-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXED_LARGE, "spxFont.Large.Fixed",
   "-*-lucidatypewriter-medium-*-*-*-28-*-*-*-*-*-*-*",
   "-*-courier-medium-r-*-*-28-*-*-*-*-*-*-*"},
  {STYLE_FIXEDBOLD_LARGE, "spxFont.Large.Fixed.Bold",
   "-*-lucidatypewriter-bold-r-*-*-28-*-*-*-*-*-*-*",
   "-*-courier-bold-r-*-*-28-*-*-*-*-*-*-*"},
  {STYLE_FIXEDITALIC_LARGE, "spxFont.Large.Fixed.Italic",
   "-*-courier-medium-i-*-*-28-*-*-*-*-*-*-*",
   NULL},
  {STYLE_FIXEDBOLDITALIC_LARGE, "spxFont.Large.Fixed.Italic.Bold",
   "-*-courier-bold-i-*-*-28-*-*-*-*-*-*-*",
   NULL},

  {STYLE_INVALID, NULL, 0},
};

typedef struct
{
  int style;
  Font_idx idx;
}
Stylename;

static Stylename normal[] =
{
  {BOLD_FONT | NORMAL_FONT, STYLE_BOLD},
  {BOLD_FONT | ITALIC_FONT | NORMAL_FONT, STYLE_ITALICBOLD},
  {FIXED_FONT, STYLE_FIXED},
  {BOLD_FONT | FIXED_FONT, STYLE_FIXEDBOLD},
  {BOLD_FONT | FIXED_FONT | ITALIC_FONT, STYLE_FIXEDBOLDITALIC},
  {FIXED_FONT | ITALIC_FONT, STYLE_FIXEDITALIC},
  {ITALIC_FONT | NORMAL_FONT, STYLE_ITALIC},
  {NORMAL_FONT, STYLE_NORMAL},
  {INVALID_FONT, STYLE_INVALID}
};
static Stylename small[] =
{
  {BOLD_FONT | NORMAL_FONT, STYLE_BOLD_SMALL},
  {BOLD_FONT | ITALIC_FONT | NORMAL_FONT, STYLE_ITALICBOLD_SMALL},
  {FIXED_FONT, STYLE_FIXED_SMALL},
  {BOLD_FONT | FIXED_FONT, STYLE_FIXEDBOLD_SMALL},
  {BOLD_FONT | FIXED_FONT | ITALIC_FONT, STYLE_FIXEDBOLDITALIC_SMALL},
  {FIXED_FONT | ITALIC_FONT, STYLE_FIXEDITALIC_SMALL},
  {ITALIC_FONT | NORMAL_FONT, STYLE_ITALIC_SMALL},
  {NORMAL_FONT, STYLE_SMALL},
  {INVALID_FONT, STYLE_INVALID}
};
static Stylename large[] =
{
  {BOLD_FONT | NORMAL_FONT, STYLE_BOLD_LARGE},
  {BOLD_FONT | ITALIC_FONT | NORMAL_FONT, STYLE_ITALICBOLD_LARGE},
  {FIXED_FONT, STYLE_FIXED_LARGE},
  {BOLD_FONT | FIXED_FONT, STYLE_FIXEDBOLD_LARGE},
  {BOLD_FONT | FIXED_FONT | ITALIC_FONT, STYLE_FIXEDBOLDITALIC_LARGE},
  {FIXED_FONT | ITALIC_FONT, STYLE_FIXEDITALIC_LARGE},
  {ITALIC_FONT | NORMAL_FONT, STYLE_ITALIC_LARGE},
  {NORMAL_FONT, STYLE_LARGE},

  {INVALID_FONT, STYLE_INVALID}
};
typedef enum
{

  SIZE_NORMAL, SIZE_SMALL,
  SIZE_LARGE,
  SIZE_INVALID
}
Font_size;

#define SIZE_MEDIUM SIZE_NORMAL
/*const Stylename *fstyles[SIZE_INVALID] =
   {normal, small, large}; */
const static Font_size font_sizes[SIZE_INVALID + 1] =
{SIZE_MEDIUM, SIZE_SMALL, SIZE_LARGE, SIZE_INVALID};

static Font_idx lookup_style(int style, Font_size size)
{

  if (size == SIZE_INVALID)
    return STYLE_INVALID;
  else
  {

    const Stylename *p;         /* = *(fstyles + size); */

    if (size == SIZE_NORMAL)
      p = normal;
    else if (size == SIZE_SMALL)
      p = small;
    else
      p = large;
    while (p->idx != STYLE_INVALID)
    {
      if (p->style == style)
        return p->idx;
      p++;
    }
    return STYLE_INVALID;
  }
}
#define GET_SIZE	0
#define GET_FONT	1
static int get_attrib_from_font_idx(Font_idx n, int what)
{
  int i = 0;
  const Stylename *p;

  while (font_sizes[i] != SIZE_INVALID)
  {
    //p = *(fstyles + i);
    if (font_sizes[i] == SIZE_NORMAL)
      p = normal;
    else if (font_sizes[i] == SIZE_SMALL)
      p = small;
    else
      p = large;
    while (p->idx != STYLE_INVALID)
      if (p->idx == n)
        return (what == GET_SIZE ? font_sizes[i] : p->style);
      else
        p++;
    i++;
  }
  return (what == GET_SIZE ? SIZE_INVALID : INVALID_FONT);
}

/* colours       */
//static int ncols;
static GdkColor *spx_colour;

__inline GdkColor spx_get_colour(unsigned int idx)
{
  return spx_colour[idx % spx_max_colour];
}

int spx_add_colour(const char *name)
{
  /*GdkColormap *cmap; */
  GdkColor c;

  /*cmap = gdk_colormap_get_system(); */
  if (!gdk_color_parse(name, &c))
    return -1;
  spx_colour = g_realloc(spx_colour, (spx_max_colour + 1) * sizeof(GdkColor));
  memcpy(spx_colour + spx_max_colour, &c, sizeof(GdkColor));
  spx_max_colour++;
  return (spx_max_colour - 1);
}

static void initialize_colours(void)
{
  GdkColormap *cmap;
  GdkColor text_colours[] =
  {
    {0, 0x0000, 0x0000, 0x0000},	//black
     {0, 0xFFFF, 0x0000, 0x0000},	//red
     {0, 0x0000, 0xFFFF, 0x0000},	//green
     {0, 0xFFFF, 0xFFFF, 0x0000},	//yellow
     {0, 0x0000, 0x0000, 0xFFFF},	//blue 
     {0, 0xFFFF, 0x0000, 0xFFFF},	//magenta
     {0, 0x0000, 0xFFFF, 0xFFFF},	//cyan
     {0, 0xFFFF, 0xFFFF, 0xFFFF},	//white

  };

#define DEFAULT_BG_COL  -1
#define DEFAULT_FG_COL  -1
  int n,
    i;

/*  fputs("Initializing colours...\n", stderr); */
  n = sizeof(text_colours) / sizeof(GdkColor);
  cmap = gdk_colormap_get_system();
  spx_colour = NULL;
  spx_max_colour = 0;
  for (i = 0; i < n; i++)
  {
    if (gdk_color_alloc(cmap, &text_colours[i]))
    {
      spx_colour = g_realloc(spx_colour, (spx_max_colour + 1) * sizeof(GdkColor));
      spx_colour[spx_max_colour] = text_colours[i];
      spx_max_colour++;
    }
  }
}

/* fonts */
static void initialize_fonts(void)
{
  const Fontstyle *p = fstyle;
  char *res;
  char bb[3];

  fputs("\nInitializing fonts. This will take some time.\n", stderr);
  while (p->idx != STYLE_INVALID)
  {
    res = gtk_get_app_resource(p->res, "FontClass", GTK_NONE, p->def, NULL, 0);
    spx_font[p->idx] = gdk_font_load(res);
    if (spx_font[p->idx] == NULL)
    {
      fprintf(stderr, "\n[%s %d] bad font for %s:\n  %s\n",
              __FILE__, __LINE__, p->res, res);
      if (strcmp(res, p->def))
      {
        fprintf(stderr, "  Retrying with %s\n", p->def);
        spx_font[p->idx] = gdk_font_load(p->def);
      }
      if (spx_font[p->idx] == NULL && p->alt)
      {
        fprintf(stderr, "    retrying with %s\n", p->alt);
        spx_font[p->idx] = gdk_font_load(p->alt);
      }
    }
    else
      putc('.', stderr);
    p++;
  }
  putc('\n', stderr);
  spx_font[p->idx] = NULL;
  (void) gtk_get_app_resource("spxFormat.Char",
                              "CharClass", GTK_STRING, "@", bb, 2);
  if (bb[0] == 0)
    spx_format_char = '@';
  else
    spx_format_char = bb[0];
}

static const char spx_format_chars[] = "CDbf.ismlv^!$t";
static int get_next2(const char *str,
                     char *p,
                     int *fg,
                     int *bg,
                     int *flag, //INVERSE, UNDERLINE
                      Font_idx * fx)
{
  //char done = 0;
  register char *s = (char *) str;
  int fontstyle = NORMAL_FONT;
  Font_size font_size = SIZE_INVALID;
  char got_size = 0,
    got_style = 0;

  if (*fx != STYLE_INVALID)
  {
    fontstyle = get_attrib_from_font_idx(*fx, GET_FONT);
    font_size = get_attrib_from_font_idx(*fx, GET_SIZE);
    got_size = 1;
  }
  *p = 0;
  while (*s /*&& !done */ )
    if (*s != spx_format_char)
      break;
    else
    {
      if (*(s + 1) == spx_format_char) /* @@ */
      {
        *p++ = *s;
        s += 2;
        if(*s==spx_format_char)
          break;  /* otherwise, next attribute, instead of the current, will act on this @ that we have*/
      }
      else
        switch (*(s + 1))
         {
           case 0:             /* @ at the end of line */
             *p++ = *s++;
             break;
           case 'C':
           case 'D':
             {
#define SSZ 8
               char buf[SSZ],
                *pbuf = &buf[0];
               char do_fg = *(s + 1) == 'C';

               *pbuf = 0;
               s += 2;
               while (*s && isdigit(*s) && (pbuf - &buf[0]) < SSZ)
                 *pbuf++ = *s++;
               *pbuf = 0;
               if (pbuf != &buf[0])
               {
                 if (do_fg)
                   *fg = atoi(buf);	/* *fg% % spx_max_colour; on return ; */
                 else
                   *bg = atoi(buf);
/**bg% % spx_max_colour; on return */
               }
               break;
             }
#if 0
           case '_':           // unused;how does one underline?

             s += 2;
             *flag |= UNDERLINE;
             break;
#endif
           case 'v':
             s += 2;
             *flag |= INVERSE;
             break;
           case 'b':
             s += 2;
             got_style = 1;
             fontstyle |= BOLD_FONT;
             break;
           case 'f':
             s += 2;
             got_style = 1;
             fontstyle = (fontstyle & (~(NORMAL_FONT | TIMES_FONT))) | FIXED_FONT;
             break;             /* no fixed bold itali */
           case 's':
             s += 2;
             font_size = SIZE_SMALL;
             got_size = 1;
             break;
           case 'm':
             s += 2;
             font_size = SIZE_MEDIUM;
             got_size = 1;
             break;
           case 'l':
             s += 2;
             font_size = SIZE_LARGE;
             got_size = 1;
             break;
#if 0
           case 't':
             s += 2;
             got_style = 1;
             fontstyle = (fontstyle & (~(FIXED_FONT | NORMAL_FONT))) | TIMES_FONT;
             break;
#endif
           case 'i':
             s += 2;
             got_style = 1;
             fontstyle |= ITALIC_FONT;
             break;
           case '^':
           case '.':
           case '$':
             goto halt;
           case '!':
             {
               s += 2;
               if (*s == 0)
                 break;
               /*if (*(s + 2) == '$')
                 goto halt;*/
               switch (*s)
                {
#if 0
                  case '_':
                    *flag &= (~UNDERLINE);
                    break;
#endif
                  case '$':
                    goto halt;
                  case 'v':
                    *flag &= (~INVERSE);
                    break;
                  case 'b':
                    fontstyle &= (~BOLD_FONT);
                    got_style = 1;
                    break;
                  case 'i':
                    fontstyle &= (~ITALIC_FONT);
                    got_style = 1;
                    break;
                  case 't':
                    fontstyle &= (~TIMES_FONT);
                    got_style = 1;
                    break;
                  case 'f':
                    got_style = 1;
                    fontstyle &= (~FIXED_FONT);
                    break;

                  case 'D':    /* reset colour */
                    *bg = DEFAULT_BG_COL;
                    break;
                  case 'C':
                    *fg = DEFAULT_FG_COL;
                    break;
                  default:
                    break;
                }
               s++;
               break;
             }                  /* case ! */
           default:
             s += 2;
             break;             /*unknown */
         }                      /*switch */
    }

check:
  while (*s != spx_format_char && (*p = *s))	// get text to draw

  {
    s++;
    p++;
  }
  if (*s == spx_format_char)
  {
    if (strchr(spx_format_chars, (*(s + 1))))
      *p = '\0';
    else if (*(s + 1) == spx_format_char)
    {
      *p++ = *s;
      s += 2;
      goto check;
    }
    else
      switch (*(s + 1))
       {
         case 0:
           *p++ = *s++;
           break;
         default:              /* unknown fmt char */
           s += 2;
           goto check;
       }
  }
halt:
  *p = 0;
  if (!got_size)
    if (got_style)
      font_size = SIZE_NORMAL;
  if (fontstyle != INVALID_FONT && !(fontstyle & (FIXED_FONT | TIMES_FONT | NORMAL_FONT)))
    fontstyle |= NORMAL_FONT;
  assert((fontstyle & (TIMES_FONT & NORMAL_FONT)) == 0);
  if ((font_size != SIZE_INVALID) && (fontstyle != INVALID_FONT))
  {
    *fx = lookup_style(fontstyle, font_size);
    assert(*fx != STYLE_INVALID);
  }
  return (s- str);
}
int spx_str2col_font(const char *fmt, int *bg, int *fg, GdkFont **font)
{
  Font_idx fidx = STYLE_INVALID;
  int pos=0;
  char *s=(char *) g_malloc(sizeof(char) * (strlen(fmt) + 1));
  int flag = DEFAULT_FLAG;
  
  *bg=*fg=-1;
  pos=get_next2(fmt, s, fg, bg, &flag, &fidx);
  if(font) *font = fidx == STYLE_INVALID ? NULL : spx_font[fidx];
  g_free(s);
  return pos;
}
#define swap_cols(__b,__f) do{if(__f==__b)\
  {(__b)=&text->style->black;(__f)=&text->style->white;}\
  else{GdkColor *_tmp_= (__f); (__f)=(__b); (__b)=_tmp_;}}while(0)
void spx_write(GtkWidget * text, const char *str)
{
  char *s;
  int b = -1,
    f = -1;
  GdkColor *bg = NULL,
   *fg = NULL;
  GdkFont *font = NULL;
  Font_idx fidx = STYLE_INVALID;
  int flag = DEFAULT_FLAG;
  size_t ll = strlen(str);
  int at = 0;

  s = (char *) g_malloc(sizeof(char) * (ll + 1));

  while (at < ll)
  {
    at += get_next2(str + at, s, &f, &b, &flag, &fidx);
    /* str=@xfoo@ybar
       read str up to @ybar */
#if 1
    bg = b < 0 ? NULL : spx_colour + (b % spx_max_colour);
    fg = f < 0 ? NULL : spx_colour + (f % spx_max_colour);
#else
    bg = b < 0 ? &text->style->base[GTK_STATE_NORMAL] : spx_colour + (b % spx_max_colour);
    fg = f < 0 ? &text->style->text[GTK_STATE_NORMAL] : spx_colour + (f % spx_max_colour);
#endif
#if 1
    font = fidx == STYLE_INVALID ? NULL : spx_font[fidx];
#else
    font = fidx == STYLE_INVALID ? text->style->font : spx_font[fidx];
#endif

    if (flag & INVERSE)
      swap_cols(bg, fg);
    if (*s)
      gtk_text_insert(GTK_TEXT(text), font, fg, bg, s, -1);

    if (*(str + at) == spx_format_char)
    {
      if (*(str + 1 + at) == '^')	/* str=@^foo */
      {
        flag = DEFAULT_FLAG;
        b = DEFAULT_BG_COL;
        f = DEFAULT_FG_COL;
        fidx = STYLE_INVALID;   /* currently the default */
        at += 2;
      }
      else if (*(str + 1 + at) == '.')	/* str=@.foo@dfdd@!.sds */
      {
        gtk_text_insert(GTK_TEXT(text), font, fg, bg, str + 2 + at, -1);
        break;
      }
      else if (*(str + 1 + at) == '$')
      {
        const char *old = str + 2 + at,
         *old0 = str + 2 + at;
        register char *ptr;
        char add_2 = 0;

        do
        {
          ptr = strchr(old, spx_format_char);
          if (ptr)
          {
            ptr++;
            if (*ptr != 0)
            {
              if (*(ptr + 1))
              {
                if (*(ptr) != '!')
                  ptr++;
                else if (*(ptr + 1) != '$')
                  ptr += 2;
                else
                {
                  ptr += 2;
                  add_2 = 2;
                  break;
                }
              }
            }
            old = ptr;
          }
        }
        while (*old && ptr);
        if (!ptr || *old == 0)
        {
          strcpy(s, old0);
          /*at=ll+1;*/
        }
        else
        {
          strncpy(s, old0, ptr - add_2 - old0);
          *(s + (ptr - add_2 - old0) - 1) = 0;
        }
        gtk_text_insert(GTK_TEXT(text), font, fg, bg, s, -1);
        if (ptr && *ptr)
          at = at + 2 + (ptr - old0);
        else
          break;
      }
      /*else
        at+=2;*/
    }
  }
  g_free(s);
}
