/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/


/*
 *   cmd_def.c  - more client commands. Also see cmd.c
 *
 *  Author: Tano Fotang, 1997-1999
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free 
 *  Software Foundation; either version 2 of the License , or (at your option) 
 *  any later version.
 *  See the file COPYING for details.
 *
 */

#include "spx.h"
#include <errno.h>
#include "config.h"
#include "nicks.h"
#include "server.h"
#include "parser.h"
#include "dcc.h"
#include "cmd_def.h"
#include "cmd.h"
#include "hooks.h"
#include "notify.h"
#if DEBUG
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#if !defined(DONT_HAVE_GETOPT)
#include <getopt.h>
#endif
#define iconnected(_w) (!win_invalid(_w) && winstruct[_w].server && winstruct[_w].server->fd>-1)

// strncasecmp(str1,str2,2) where str2[0]=='-',e,.g. str2:="-h"
#define streq(str,c0,c10,c11) \
           ((str)[0]==(c0) && ((str)[1]==(c10) || (str)[1]==(c11)))

static int save_nicknames(const char *path)
{
  char *s;
  FILE *fp;

  if (!path)
  {
    s = (char *) alloca(sizeof(char) *
                        (strlen(prog_home) + strlen(P_NICKNAMES) + 2));

    sprintf(s, "%s/%s", prog_home, P_NICKNAMES);
  }
  else
  {
    s = (char *) alloca(sizeof(char) * (strlen(path) + 1));

    sprintf(s, "%s", path);
  }
  if (access(s, F_OK) != -1)
  {
    if (access(s, W_OK) == -1)
      return -1;
    else
    {
      char *buf = malloc(sizeof(char) * (5 + strlen(s)));

      sprintf(buf, "%s.bak", s);
      rename(s, buf);
      free(buf);
    }
  }
  if ((fp = fopen(s, "w")) == NULL)
    return -1;
  else
  {
    time_t t = time(NULL);
    int n;

    fprintf(fp, "%c " sula_NAME " " sula_VERSION
            "\n%c Nick list created by %s, %s\n",
            COMMENT, COMMENT, getenv("USER"), ctime(&t));
    n = 0;
    for (n = 0; n < iNickListCount; n++)
      if (aNickList[n].label)
      {
        Nicks *p = aNickList[n].start->left;

        fprintf(fp, "<%s>\n", aNickList[n].label);
        while (p != aNickList[n].start)
        {
          fprintf(fp, "\t%s\n", p->nick);
          p = p->left;
        }
        fprintf(fp, "</%s>\n", aNickList[n].label);
      }
    fflush(fp);
  }
  fclose(fp);
  return 0;
}

void nick(int argc, char **argv, Winstruct * win)
{
/*
   nick [+|-] | [<nick>] | [s|l <filename>] 
 */
  char not_prev = 0;

  if (argc == 1)
  {
    if (win && win->server)
      say2(0, win - winstruct, 1, "Your nickname is %s", win->server->nick);
    return;
  }
  else if (argv[1][0] == '-')
  {
    if ((argv[1][1] == 's' || argv[1][1] == 'S'))
    {
      if (save_nicknames(argv[2]) == -1)
        say2(0, win ? win - winstruct : -1, -1,
             "(%s) Error saving nicknames: %s", argv[2] ? argv[2] : "",
             strerror(errno));
      return;
    }
    else if ((argv[1][1] == 'l' || argv[1][1] == 'L'))
    {
      char *path = argv[2];
      char *buf;

      if (!path)
      {
        buf =

           alloca(sizeof(char) *
                  (strlen(prog_home) + strlen(P_NICKNAMES) + 2));

        sprintf(buf, "%s/%s", prog_home, P_NICKNAMES);
      }
      else
        buf = path;
      if (load_nicks(buf))
        say2(0, win ? win - winstruct : -1, -1,
             "(%s) Error loading nicknames: %s", buf, strerror(errno));
      return;
    }
  }

  if ((argv[1][0] == '-' && argv[1][1] == 0) ||
      (not_prev = (argv[1][0] == '+' && argv[1][1] == 0)))
  {
    /*
       i.e. if( !strcmp(argv[1], "-") || !(prev=strcmp(argv[1], "+")) ){ 
     */
    NickList *nl;

    if (!win || !win->server)
    {
      if (gflags & BEEP_ERR)
        spx_bell(0);
      return;
    }
    assert(win->server->iNick >= 0);
    nl = &aNickList[win->server->iNick];
    if (nl->start != nl->start->left)
    {                           // list not empty

      Nicks *n;
      char *buf;

      if (nl->start->left == nl->start->right)	//just 1 nick entry

        return;
      n = not_prev ? nl->current->left : nl->current->right;
      if (n == nl->start)
        n = not_prev ? n->left : n->right;
      buf = alloca(sizeof(char) * (strlen(n->nick) + 8));

      sprintf(buf, "NICK %s\n", n->nick);
      sendto_server(win, buf);
    }
    return;
  }
  if (win && win->server)
  {
    char *buf = alloca(sizeof(char) * (strlen(argv[1]) + 8));

    sprintf(buf, "NICK %s\n", argv[1]);
    sendto_server(win, buf);
  }
}

static void save_browser(Winstruct * win, char *ptr, int save_all, int which)
{

  FILE *fp = fopen(ptr, "w");

  if (fp == NULL)
  {
    say2(0, win - winstruct, 1, PROMPT "%s %s", ptr, strerror(errno));
    if (gflags & BEEP_ERR)
      spx_bell(0);
  }
  else
  {
#if USE_XFORMS
    int total, i;
    FL_OBJECT *browser;

    browser = which == 2 ? win->chanwin->brow2 : win->chanwin->browser;
    total = fl_get_browser_maxline(browser);
    if (save_all)
      for (i = 1; i <= total; i++)
      {
        fputs(fl_get_browser_line(browser, i), fp);
        fputs("\n", fp);
      }
    else
      for (i = 1; i <= total; i++)
        if (fl_isselected_browser_line(browser, i))
        {
          fputs(fl_get_browser_line(browser, i), fp);
          fputs("\n", fp);
        }
    fclose(fp);
#else
    gchar *buf;

    buf =
       gtk_editable_get_chars(GTK_EDITABLE
                              (which ==
                               2 ? win->chanwin->brow2 : win->chanwin->
                               browser), 0, -1);
    fputs(buf, fp);
    fclose(fp);
    free(buf);
#endif
  }
}
static void load_a_file(Winstruct * win, char *ptr, int which)
{
  int fd = open(ptr, O_RDONLY);

  if (fd < 0)
  {
    say2(0, win - winstruct, -1, PROMPT "%s: %s", ptr, strerror(errno));
    if (gflags & BEEP_ERR)
      spx_bell(0);
  }
  else
#if USE_XFORMS
  {
#define BUFF	1024
    int n;
    char buf[BUFF + 1];

    FL_OBJECT *browser;

    browser = which == 2 ? win->chanwin->brow2 : win->chanwin->browser;
    fl_freeze_form(win->chanwin->chanwin);
    errno = 0;
    while ((n = read(fd, buf, BUFF)) > 0)
    {
      buf[n - 1] = '\0';
      fl_addto_browser_chars(browser, buf);
    }
    if (n == -1)
      perror("Error");
    close(fd);
    fl_unfreeze_form(win->chanwin->chanwin);
    if (n < 0)
    {
      sprintf(buf, PROMPT "%s: %s", ptr, strerror(errno));
      fl_addto_browser(win->chanwin->brow2, buf);
    }
  }
#else
    close(fd);
  spx_load_file(which == 2 ? win->chanwin->brow2 : win->chanwin->browser,
                ptr);
#endif
}

void window(int argc, char **argv, Winstruct * win)
{
/*
   window [--clear 1|2] window --save 1|2 --filename <file> [--marked] window 

   --load 1|2 --filename <f> window --radio 1|2|3|4|5
   (quote|query|channel|dcc|echo) window --help  
 */
  char *fname = NULL;
  int radio = -1;
  char selected = 0;
  int save = 0, load = 0;

  while (1)
  {
    register int i;

#if !defined(DONT_HAVE_GETOPT)
    int option_index = 0;
    static struct option long_options[] = {
      {"clear", 1, 0, 'c'},
      {"save", 1, 0, 's'},
      {"load", 1, 0, 'l'},
      {"filename", 1, 0, 'f'},
      {"radio", 1, 0, 'r'},
      {"marked", 0, 0, 'm'},
      {"help", 0, 0, 'h'},
      /*{"systemcomandlog", 0, 0, 'L'}, */
      {0, 0, 0, 0}
    };

    i = getopt_long(argc, argv, "c:s:l:f:mh", long_options, &option_index);
#else
    i = getopt(argc, argv, "c:s:l:f:mh");
#endif
    if (i == -1)
      break;

    switch (i)
    {
       case 'c':
         {
           int brow = atoi(optarg);

           if (brow == 1)
             TEXT_CLEAR(SPX_TEXT(win));
           else
             TEXT_CLEAR(SPX_TEXT2(win));
           break;
         }
       case 's':
         save = atoi(optarg);
         break;
       case 'l':
         load = atoi(optarg);
         break;
       case 'f':
         fname = optarg;
         break;
       case 'r':
         radio = atoi(optarg);
         break;
       case 'm':
         selected = 1;
         break;
/*       case 'L':
#if USE_XFORMS
         fl_show_command_log(FL_FULLBORDER);
#endif
         break;
*/
       case 'h':
         {
           uppercase2(*argv);
           show_std_help(*argv, NULL, win->chanwin);
           return;
         }
       default:
         break;
    }
  }
  optind = 0;
  if (radio > 0)
    switch (radio - 1)
    {
       case 0:
         TRIGGER_OBJECT(win->chanwin->r_quote);
         break;
       case 1:
         TRIGGER_OBJECT(win->chanwin->r_query);
         break;
       case 2:
         TRIGGER_OBJECT(win->chanwin->r_channel);
         break;
       case 3:
         TRIGGER_OBJECT(win->chanwin->r_dcc_chat);
         break;
       case 4:
         TRIGGER_OBJECT(win->chanwin->r_echo);
         break;
       default:
         break;
    }
  if (fname)
  {
    if (load)
      load_a_file(win, fname, load);
    if (save)
      save_browser(win, fname, !selected, save);
  }

}

#define do_error(_f) do{\
      say2(0, w, 1, PROMPT"%s: %s", _f, strerror(errno));\
        return;} while (0)
void history(int argc, char **argv, int w)
{
/*
 * history -clear history -load [filename] hist -reload histo
 * -save [filname] */

  static void clear_history(void);

  if (argc == 1)
    return;
  if (streq(argv[1], '-', 'c', 'C'))
  {
    clear_history();
    return;
  }
  else
  {
    char *fname = NULL;
    char buf[512];

    if (streq(argv[1], '-', 'l', 'L'))
    {
      fname = argv[2];
      if (!fname)
      {
        sprintf(buf, "%s/%s", prog_home, P_HISTORY);
        fname = &buf[0];
      }
      if (load_history(fname))
        do_error(fname);
    }
    else if (streq(argv[1], '-', 'r', 'R'))
    {
      sprintf(buf, "%s/%s", prog_home, P_HISTORY);
      fname = &buf[0];

      if (access(buf, R_OK) == 0)
        clear_history();
      if (load_history(buf))
        do_error(buf);
    }
    else if (streq(argv[1], '-', 's', 'S'))
    {
      if (save_history(argv[2]))
        say2(0, w, 1,
             PROMPT "Error saving (as %s if not null): %s", argv[2],
             strerror(errno));
    }
  }
}

static void clear_history(void)
{

  History *p = histStart->right;
  int i;

  while (p != histStart)
  {
    p->left->right = p->right;
    p->right->left = p->left;
    free(p->line);
    p = p->right;
  }
  histStart->left = histStart->right = histCur = histStart;

  for (i = 0; i < win_count; i++)
    if (winstruct[i].chanwin)
#if USE_XFORMS
      winstruct[i].chanwin->cmd_fwd->u_ldata = 0L;
#else
      gtk_object_set_data(GTK_OBJECT(winstruct[i].chanwin->cmd_fwd),
                          "H", 0);
#endif
}

extern int on_channel(const Server * s, const char *chan)
{
/*
   we're on this channel  
 */

  if (s->fd > -1)
  {
    gChannel *p = chanStart;

    while (p != chanEnd)
    {
      if (!strcasecmp(p->server, s->name) &&
          s->port == p->port && !strcasecmp(p->name, chan))
        return 1;
      p = p->next;
    }
  }
  return 0;
}
#define connected(win) ((win) && (win)->server && (win)->server->fd >-1)
void join_cmd(CmdStruct * cs)
{
/* /join channel [key] */
  Winstruct *win;

  if(win_invalid(cs->w)) return;
  win=winstruct+cs->w;
  if (!iconnected(cs->w))
  {
    if (gflags & BEEP_ERR)
      spx_bell(0);
    fit_object_label(win->chanwin->WinStatusLine,
                     "You are not connected to a server");
  }
  else
  {
    
    char *opt = nextword(cs->args, 1);

    if(*opt==0) return;
    if (opt[0] == '-' && ((opt[1] == 'l' || opt[1] == 'L') && opt[2] == 0))
    {

      if (win->lastPart->name)
      {
        char *buf =alloca(sizeof(char) *(strlen(win->lastPart->name) + 10 +
                   (win->lastPart->key ? strlen(win->lastPart->key) : 0)));

        addto_cmd_sent(win, "JOIN", win->lastPart->name);
        sprintf(buf, "JOIN %s %s\n", win->lastPart->name,
                win->lastPart->key ? win->lastPart->key : "");
        sendto_server(win, buf);
      }
    }
    else if (opt[0] == '-' && ((opt[1] == 'i' || opt[1] == 'I') && opt[2] == 0))
    {
      if (win->server->lastInvite)
      {
        char *buf =
           alloca(sizeof(char) * (strlen(win->server->lastInvite) + 10));

        addto_cmd_sent(win, "JOIN", win->server->lastInvite);
        sprintf(buf, "JOIN %s\n", win->server->lastInvite);
        sendto_server(win, buf);
      }
    }
    else
    {
      char *channame = NULL;

      if (*opt != '#' && *opt != '&')
      {
        channame = (char *) malloc(sizeof(char) * (strlen(opt) + 3));

        sprintf(channame, "#%s", opt);
      }
      else
        channame = strdup(opt);

      if (on_channel(win->server, channame))
        say2(1, cs->w, 1, PROMPT "already on channel @i@.%s", channame);
      else
      {
        char *key = nextword(cs->args, 2);
        char *buf =
           alloca(sizeof(char) *
                  (strlen(channame) + (*key ? strlen(key) : 0) + 10));

        sprintf(buf, "JOIN %s %s\n", channame, key);
        addto_cmd_sent(win, "JOIN", channame);
        sendto_server(win, buf);
        free(key);
      }
      free(channame);
    }
    free(opt);
  }
}

void part(int argc, char **argv, Winstruct * win)
{
  char buf[256];
  if (argc == 1)
  {
    if (win->chanCur != win->chanStart)
    {
      sprintf(buf, "PART %s\n", win->chanCur->name);
      sendto_server(win, buf);
    }
    return;
  }
  else if (argv[1][0] == '-' && (argv[1][1] == 'l' || argv[1][1] == 'L'))
  {
    if (win->lastJoin->name)
    {
      sprintf(buf, "PART %s\n", win->lastJoin->name);
      sendto_server(win, buf);
    }
    return;
  }
  else if (argv[1][0] == '-' && (argv[1][1] == 'a' || argv[1][1] == 'A'))
  {
    Channel *current;
    int errur = 0;

    current = win->chanStart->right;
    while (current != win->chanStart && !errur)
    {
      sprintf(buf, "PART %s\n", current->name);
      errur = sendto_server(win, buf);
      current = current->right;
      delay(500);
    }
    return;
  }
  else
  {
    char *channame = NULL;

    argv++;
    if (**argv == '*' && *(*argv + 1) == 0)
    {
      if (win->chanCur == win->chanStart)
        return;
      else
        channame = strdup(win->chanCur->name);
    }
    else if (**argv != '#' && **argv != '&')
    {
      channame = (char *) malloc(sizeof(char) * (strlen(*argv) + 3));

      sprintf(channame, "#%s", *argv);
    }
    else
      channame = strdup(*argv);
    sprintf(buf, "PART %s\n", channame);
    sendto_server(win, buf);
    free(channame);
  }
}

void invite(int argc, char **argv, int w)
{
/*
   invite nick1 nick 2 nick3 ... [to channel] 
 */
  int i, j = 0;
  char *channel = NULL;
  Winstruct * win;
  if (argc == 1)
  {
    if (gflags & BEEP_ERR)
      spx_bell(0);
    return;
  }
  win=winstruct + w;
  if (win->chanCur == win->chanStart)
    return;
#if 1
  i = 1;
  while (++i < argc)
    if (!strcasecmp(argv[i], "to") && (i + 2) == argc)
    {
      j = i;
      i++;
      if (!strcmp(argv[i], "*"))
        channel = win->chanCur->name;
      else
      {
        channel = alloca(sizeof(char) * (strlen(argv[i]) + 3));

        if (argv[i][0] != '#' && argv[i][0] != '&')
          sprintf(channel, "#%s", argv[i]);
        else
          strcpy(channel, argv[i]);
      }
      break;
    }
#endif
  if (j == 0)
    j = argc;
  if (!channel || !strcmp(channel, "*"))
    channel = win->chanCur->name;
  while (--j > 0)
  {
    char *buf =  alloca(sizeof(char) * (strlen(channel) + strlen(argv[j]) + 16));
    sprintf(buf, "INVITE %s %s\n", argv[j], channel);
    if (sendto_server(win, buf))
      break;
  }
}

// ///// server command /////////////////////
static void server_list(Server * p, SPX_OBJ(brow), int *i)
{
  if (p)
  {
    char *buf;

    server_list(p->left, brow, i);
    buf = (char *) malloc(sizeof(char) * (strlen(p->name) +
                                          strlen(p->nick) + 50));

    sprintf(buf, "@f%s@$%02d. %s@!$@C5 %hu @C6@.%s", p->fd > 0 ? "@C2" : "",
            ++(*i), p->name, p->port, p->nick);
    spx_write1(brow, buf);
    free(buf);
    server_list(p->right, brow, i);
  }
}
static void display_servers(SPX_OBJ(disp))
{
  int i;
  int n=0;
  if(iServerGroupCount>0)
    spx_write1(disp,
    "@f@C1@.__________________________________________");
  for(i=0;i<iServerGroupCount;i++)
    if(aServerGroup[i].name)
    {
      char *buf=malloc(sizeof(char)*(strlen(aServerGroup[i].name)+60));
      sprintf(buf,"@f@D15@C0@.%-42s",aServerGroup[i].name); 
      spx_write1(disp, buf);
      free(buf);
      server_list(aServerGroup[i].s, disp, &n);
    }
}

static void get_server_bynumber(Server * p, Server ** s,
                                int n, int *i, char *gotit)
{
/*    return server # n   */
  if (*gotit)
    return;
  if (p)
  {
    get_server_bynumber(p->left, s, n, i, gotit);
    if (n == ++(*i))
    {
      *gotit = 1;
      *s = p;
      return;
    }
    get_server_bynumber(p->right, s, n, i, gotit);
  }
}
static void idx2server(Server ** s, int idx, int *pos, char *gotit)
{
  int i;

  for(i=0; i<iServerGroupCount;i++)
    if(aServerGroup[i].name){
       get_server_bynumber(aServerGroup[i].s, s, idx, pos, gotit);
       if(*gotit) return;
   }
}
void server(char *args, Winstruct * win)
{
/*
   /server <number>|<servname> [port]  [-pas passwd] [-nick nickname] 
   /server   -ng <name>   ; remove notify group
   /server +ng <name>   ; add notify group 

 */
  int to;
  char *aargv[32], **argv;
  int argc;

  char *tmp = alloca((strlen(args) + 1) * sizeof(char));

  strcpy(tmp, args);
  new_str2args(args, &argc, aargv, 32);
  if (argc == 1)
  {
    display_servers(win ? win->chanwin->browser : main_window->browser);
    return;
  }
  to = win ? win - winstruct : -1;
  argv = aargv;
  if (argv[1][0] == '-')
  {
    if (!strcasecmp(argv[1], "-save"))
    {
      char *fname = argv[2];

      int status = save_servers(fname);

      if (status)
        say2(0, to, 1, "Error saving server list: %s", strerror(status));
      return;
    }
    else if (!strcasecmp(argv[1], "-load"))
    {
      if(argv[2])
        if(load_serverlist(argv[2]))
         say2(0, to, 1, "%s: %s", argv[2], strerror(errno));
      return;
    }
    else if (!strcasecmp(argv[1],"-reconnect"))
    {
      if (!win)
        return;
      if (!win->server)
      {
        fit_object_label(win->chanwin->WinStatusLine,
                         "Window has no server. Use /server to connect");
        if (gflags & BEEP_ERR)
          spx_bell(0);
      }
      else if (win->server->fd < 0)
        connect_to_server(win->server, win, NULL, 0);
      return;
    }
  }
  if ((argv[1][0] == '-' || argv[1][0] == '+') &&
      (argv[1][1] == 'n' || argv[1][1] == 'N')
      && (argv[1][2] == 'g' || argv[1][2] == 'G'))
  {                             //add/remove notify group

    if (!win || !win->server)
      return;
    else
    {
      char remove = (argv[1][0] == '-');
      char *group;
      Notify_t n;

      group = args + find_pos(tmp, 2);
      if (group[0] == 0)
      {
        say(PROMPT "Missing notify group name", to, -1);
        if (gflags & BEEP_ERR)
          spx_bell(0);
        return;
      }
      n = find_notify_list_bylabel(group);
      if (n == -1)
      {
        say2(0, to, -1, PROMPT "%s: No such notify group", group);
        return;
      }
      if (remove)
        detach_notify_group(win->server, n);
      else
      {
        if (attach_notify_group(win->server, n))
        {
          if (errno == 0)
            say2(0, to, -1, PROMPT "notify group \"%s\" is full", group);
          else
            error(-1, "system error");
        }
      }
    }
  }
  else
  {
    //there might be memory leaks in here
    char *server = 0, *nick = NULL, *password = NULL;
    u_short port = 0;
    Server *s = NULL;
    int found = 0;

    char *buf;
    int n;

    argv++;
    server = strdup(*argv);
    n = strtol(server, &buf, 0);
    if (*buf == 0)
    {
      int i = 0;
      char gotit = 0;

      s = malloc(sizeof(Server));
      if (n > 0)
        idx2server(&s, n, &i, &gotit);
      if (n <= 0 || !gotit)
      {
        say2(0, to, -1, PROMPT "invalid server number %d", n);
        if (gflags & BEEP_ERR)
          spx_bell(0);
        free(s);
        return;
      }
      found = 1;
    }
    while ((*++argv) != NULL)
    {
      if (!strncasecmp(*argv, "-pass", 3))
        if (!(*++argv))
        {
          say(PROMPT "-pass: password?", to, 1);
          return;
        }
        else
          password = strdup(*argv);
      else if (streq(*argv, '-', 'n', 'N'))
        if (!(*++argv))
        {

          say(PROMPT "-nick: missing nick", to, 1);
          return;
        }
        else
          nick = strdup(*argv);
      else if (port <= 0)
        port = atoi(*argv);
    }
    if (!found)
    {
      if (!port)
        port =
           win && win->server && win->server->fd < 0 ?
           win->server->port : DEFAULT_PORT;
      s = search_server_tree_byname(-1, server, port);
      if (!s)
      {
        found = 0;
        s = (Server *) my_malloc(sizeof(Server));
      }
      else
        found = 1;
    }                           //  if (!s) 

    if (found && s->fd > 0)
    {
      if (!check_msg_hook
          (SERVER_FAILED, to, "%s,%hu %s",
           s->name, s->port, nick ? nick : s->nick))
      {
        say2(0, to, 1,
             PROMPT "You are already connected to %s [%hu]",
             s->name, s->port);}
      return;
    }
    if (win && win->server && (win->server->connecting))
    {
      if (!check_msg_hook
          (SERVER_FAILED, to, "%s,%hu %s",
           win->server->name, win->server->port,
           nick ? nick : win->server->nick))
      {
        spx_bell(0);
        if (win->server->connecting > 0)
          say(PROMPT " connecting." " Wait or try different window", to, 1);
        else
          say(PROMPT "already resolving server name(?).", to, 1);
      }
      return;
    }
    if (nick)
      s->nick = nick;
    else if (!found)
      s->nick = Name;
    if (password)
      s->password = password;
    else if (!found)
      s->password = NULL;
    if (!found)
    {
      s->email = malloc(sizeof(char) * (strlen(Name) + strlen(host) + 3));

      sprintf(s->email, "%s@%s", Name, host);
      s->ircname = sula_NAME " " sula_VERSION;
      s->mode = NULL;
      s->flag = 0;
      s->read = 0;
      s->lag = NULL;
      s->lastInvite = NULL;
      s->port = port;
      s->name = server;
      s->alias = NULL;
      s->fd = -1;
      s->g=-1;
      s->state = -1;
      s->iNick = -1;
      s->notify = NULL;
    }
    // NOTE: xforms: does not return if there was no valid window
    if (connect_to_server(s, win, NULL, 0))
      check_msg_hook(SERVER_FAILED, to,
                     "%s,%hu %s", s->name, s->port, nick ? nick : s->nick);
    if (!found)
      free(s->email);
  }                             //                                                                                                     else 

}



void who(int argc, char **argv, Winstruct * win)
{
  char *buf = NULL;

  if (!win->server)
    return;
  if (argc == 1 || (argv[1][0] == '-' && argv[1][1] == 0))
  {
    if (win->chanCur != win->chanStart)
    {
      buf = malloc(sizeof(char) * (strlen(win->chanCur->name) + 8));

      sprintf(buf, "WHO %s\n", win->chanCur->name);
      addto_cmd_sent(win, "WHO", win->chanCur->name);
      if (sendto_server(win, buf))
        getwinbycmd(win->server, "WHO", win->chanCur->name, 1);
      free(buf);
    }
    return;
  }
  if (argv[1][0] == '-' && (argv[1][1] == 'u' || argv[1][1] == 'U'))
  {
    char *chan = NULL;

    if (argv[2])

    {
      Channel *ch;

      chan = argv[2];
      ch = get_chan_byname(win, chan);
      if (ch)
        ch->flag &= ~ch_UPDATE_USERS;
    }
    else if (win->chanCur != win->chanStart)
    {
      chan = win->chanCur->name;
      win->chanCur->flag &= ~ch_UPDATE_USERS;
    }
    else
      return;

    buf = malloc(sizeof(char) * (strlen(chan) + 8));

    sprintf(buf, "WHO %s\n", chan);
    addto_cmd_sent(win, "WHO_UPDATE", chan);
    if (sendto_server(win, buf))
      getwinbycmd(win->server, "WHO_UPDATE", chan, 1);
    free(buf);
  }
  else
  {
    buf = malloc(sizeof(char) * (strlen(argv[1]) + 8));

    sprintf(buf, "WHO %s\n", argv[1]);
    sendto_server(win, buf);
    free(buf);
    return;
  }
}
