/***************************************************
  This file is distributed as part of Sula PrimeriX
  (http://members.xoom.com/sprimerix).
****************************************************/


/*
 *  dcc.c  - handle DCC connections. Also see ctcp.c.
 *
 * (C) 1997-1999 Tano Fotang
 *
 * This software is distributed under the GNU General Public License,
 * either version 2, or any later version. The license is included
 * herein by reference.
 *
 */

#include "spx.h"
#include <errno.h>
#include <sys/param.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <glob.h>
#include <fnmatch.h>
#include "checksum.h"
#include "nc.h"
#include "parser.h"
#include "dcc.h"
#include "setting.h"
#include "ignore.h"
#include "cmd.h"
#include "hooks.h"
Key *keys = NULL;               /*encrypt */



u_short dcc_timeout;
char *chat_quit_text;
char *userinfo;                 /* ctcp */
char *finger_reply;             /* finger reply */
u_int max_sendbuf;              /* dcc send file; buffer size */
DCCChatWin *dccchatwinStart, *dccchatwinEnd;
DCC *dccStart, *dccEnd;
static int dcc_id = 0;
extern int dcc_getfile(int fd, DCC * f, Winstruct * win);
static int dcc_create_chatwin(int, char *, char *peer,
                              char *rem_host, Winstruct *, DCC *);
extern void remove_dcc_by_id(int id);
extern void terminate_file_transfer(DCC *, int w);


static char *tohost(unsigned long inetaddr)
{
  struct in_addr in;

/* confusion reigns here..did we accept? or connect?? remove this proc
   if yu figure that out... */
#if 0
//connect:ed
  in.s_addr = htonl(inetaddr);
#else
//accept:ed
  in.s_addr = inetaddr;
#endif
  return inet_ntoa(in);
}
void resume_request(const char *nick, const char *email,
                    u_short port, off_t offset, CheckSum cksum)
/*
   we've received a request to resume 

   first check to see whether what they think they have is actually what
   we're trying to send. If that's the case, send A DCC ACCEPT. otherwise
   send a DCC RENAME so they can rename the file instead of overwriting what
   they have. 
 */
{
  DCC *p = dccStart;

  if (debug)
    fprintf(stderr,
            "Resume req recvd from %s!%s for %hu (offs:%lu, chksum:%lu)\n",
            nick, email, port, offset, cksum);
  while (p != dccEnd)
  {
    if (p->lport == port && (p->type & ((DCC_FILE | DCC_SENT) & ~DCC_ACTIVE)))
      break;
    p = p->next;
  }
  if (p != dccEnd)
  {
    u_long cks;
    char buf[128];

    checksum(p->dcc.file->name, 0, offset, &cks);
    if (cks == cksum)
    {                           /*  they have a portion of the file */
      sprintf(buf, "PRIVMSG %s :%cDCC ACCEPT %lu %hu %lu%c\n", p->nick, 0x01, cksum,	/*    redundant       */
              port, offset, 0x01);
      p->dcc.file->offset = offset;
    }
    else                        /*   they have a different file */
      sprintf(buf, "PRIVMSG %s :%cDCC RENAME %lu %hu %lu%c\n", p->nick, 0x01, cks,	/*       redundant       */
              port, offset,     /*  offset is  redundant       */
              0x01);
    sendto_server(Win, buf);
  }
}
static void start_receiving(DCC *, Winstruct *);
extern void accept_request(const char *nick, const char *email,
                           u_short port, off_t offset)
/*
 * we get here when we get a DCC ACCEPT.  We have to assume that the sender
 * has made sure that we really  have a portion of what they're sending.
 * also, giving an offset is redundant: file will be opened with O_APPEND */
{
  DCC *q = dccStart;

  while (q != dccEnd)
  {
    if (q->rport == port && (q->type & ((DCC_FILE | DCC_RECV) & ~DCC_ACTIVE)))
      break;
    q = q->next;
  }
  if (q == dccEnd)
  /* either we didnt send A DCC RESUME, or it's been removed in the meantime*/
    return;
  if(debug)
  {
  if (q->dcc.file->resume < 0)
    fprintf(stderr, "Resuming %s from %s, offset %lu (size: %lu)\n",
            q->dcc.file->name, q->nick, offset, q->dcc.file->size);
  else
    fprintf(stderr,
            "Resume; rename %s offered by %s; offset was %lu (size: %lu)\n",
            q->dcc.file->name, q->nick, offset, q->dcc.file->size);
  }
  q->dcc.file->size -= offset;
  start_receiving(q, Win);
}
#define GET_CONTINUE	0
#define GET_IGNORE	1
#define	GET_RESUME	2
static int check_resume(DCC * f, Winstruct * win)
{
/*
   we have received a DCC SEND. check to see whether we have a file with a
   similar name.If that's the case, we compare file lengths. If ours is
   shorter, we send it's length together with its checksum using DCC RESUME.
   If ours is longer, we compare the checksums of size bytes. If they#re
   both the same, we ignore the DCC SEND request. If they differ, we just
   rename what we have and aceept the file. 
 */

  if (access(f->dcc.file->name, F_OK) != -1)
  {
    struct stat sbuf;
    char buf[512];

    if (stat(f->dcc.file->name, &sbuf) != -1)
    {
      int status;
      u_long cks;

      if (!S_ISREG(sbuf.st_mode))	/* what we have isnt a regular file */
        return GET_CONTINUE;
      if (sbuf.st_size >= f->dcc.file->size)
      {
        /*
           either ours is bigger or both are of same size. 
           * are the first f->size bytes the same? 
         */
        char ignore = 0;

        if (debug)
          fprintf(stderr, "%s exists. Comparing checksums...\n",
                  f->dcc.file->name);
        status = checksum(f->dcc.file->name, 0, f->dcc.file->size, &cks);
        if (status)
        {
          fprintf(stderr, "%s:"
                  " Unable to compute checksum: %s\n",
                  f->dcc.file->name, strerror(status));
          ignore = 1;

        }
        else if (cks == f->dcc.file->checksum)
          /* we have the file (the first f->size bytes are the same)*/
        {
          ignore = 1;
          fprintf(stderr, "%s exists and is the same:\n"
                  "\tchecksum to offs %lu:\E[4m%lu\E[m, size: \E[4m%lu\E[m\n",
                  f->dcc.file->name, f->dcc.file->size, cks, sbuf.st_size);
        }
        if (ignore)
        {
          fprintf(stderr, "DCC SEND from %s ignored for %s.\n",
                  f->nick, f->dcc.file->name);
          sprintf(buf, "PRIVMSG %s :%cDCC REJECT %hu %hu%c\n", f->nick, 0x01, f->rport,	// some clients expect somethin here 
                  f->rport, 0x01);
          sendto_server(win, buf);
          return GET_IGNORE;
        }
        else
        {
          fprintf(stderr, "%s exists but is %s:\n"
                  "\tchecksum of %lu bytes:\E[4m%lu\E[m, file size: \E[4m%lu\E[m.\n",
                  f->dcc.file->name,
                  sbuf.st_size == f->dcc.file->size ? "different" : "bigger",
                  f->dcc.file->size, cks, sbuf.st_size);
          f->dcc.file->resume = 0;
          /* ours is longer and the first bytes are different */
          return GET_CONTINUE;
        }
      }
      else
      {
        /* ours is shorter.  send checksum and request for resume  */
        status = checksum(f->dcc.file->name, 0, -1, &cks);
        f->dcc.file->resume = -1;	/*  our's is shorter */
        fprintf(stderr, "%s exists and is shorter:\n"
                "\tchecksum:\E[4m%lu\E[m, size: \E[4m%lu\E[m\n ",
                f->dcc.file->name, cks, sbuf.st_size);
      }
      f->starttime = time(NULL);
      fprintf(stderr, "Sending checksum and offset to %s for %s\n",
              f->nick, f->dcc.file->name);
      sprintf(buf, "PRIVMSG %s :%cDCC RESUME %lu %hu %lu%c\n",
              f->nick, 0x01, status ? 0 : cks, f->rport, sbuf.st_size, 0x01);
      sendto_server(win, buf);
      return GET_RESUME;
    }                           /*   if stat()                                                                                       */
  }                             /*   if(access(f->filename, F_OK..)  */
  return GET_CONTINUE;
}

extern void rename_request(const char *nick, const char *email, u_short port)
{
  DCC *q = dccStart;

  while (q != dccEnd)
    if (q->rport == port && (q->type & ((DCC_FILE | DCC_RECV) & ~DCC_ACTIVE)))
      break;
    else
      q = q->next;
  if (q == dccEnd || strcasecmp(q->peer, email))
  	/*  original file request didnt come from this address */
  {
#if DEBUG
    fprintf(stderr,
            "DCC rename: nick/host %s(%s) or bad offset or req gone\n", nick,
            email);
#endif
    return;
  }
#if DEBUG
  fprintf(stderr, "DCC rename from %s(%s): %s %hu\n",
          nick, email, q->dcc.file->name, port);
#endif
  q->dcc.file->resume = 1;
  start_receiving(q, Win);
}
extern void reject_request(const char *nick, const char *email, u_short port)
/*
   they've rejected our file offer.or we sent a dcc resume (?) 
 */
{
  DCC *q = dccStart;

  while (q != dccEnd)
    if (q->lport == port && (q->type & ((DCC_FILE | DCC_SENT) & ~DCC_ACTIVE)))
      break;
    else
      q = q->next;
  if (q == dccEnd   /*  either we didnt send A DCC RESUME or it's been removed  */
      || (q->peer && strcasecmp(q->peer, email))
      /* original file request didnt come from this address */
     )
  {
#if DEBUG
    fprintf(stderr,
            "DCC reject: bad nick/host %s(%s) or bad offset or req gone\n",
            nick, email);
#endif
    return;
  }
#if DEBUG
  fprintf(stderr, "DCC REJECTed for %s by %s!%s\n",
          q->dcc.file->name, nick, email);
#endif
  remove_dcc_by_id(q->id);
}

static void start_receiving(DCC * q, Winstruct * win)
{
  int fd;

  assert(q->type & ((DCC_FILE | DCC_RECV) & ~DCC_ACTIVE));
  fd = tcp_open(NULL, q->inetaddr, NULL, q->rport, connect_timeout);
  if (fd < 0)
  {
  failed:
    say2(1, win - winstruct, 1,
         "@i" PROMPT "DCC connection to %s (%lu, %d) failed",
         q->nick, q->inetaddr, q->rport);
    if (tcp_err_msg)
      say(tcp_err_msg, win - winstruct, 1);
    remove_dcc(q);
    return;
  }
  //q->inetaddr = tcp_rem_addr.sin_addr.s_addr;
  check_msg_hook(DCC_GET_START, win - winstruct, ", %s %s %s %lu %d",
                 q->nick, q->peer, q->dcc.file->name, q->dcc.file->size, fd);
  if (dcc_getfile(fd, q, win))
    goto failed;
}

extern void accept_file(DCC * q, Winstruct * win)
{
  if ((sflags & EXTENDED_DCC))
    switch (check_resume(q, win))
    {
       case GET_CONTINUE:
         start_receiving(q, win);
         break;
       case GET_IGNORE:
         remove_dcc_by_id(q->id);
         break;
       case GET_RESUME:
         break;
    }
  else
    start_receiving(q, win);
}

extern void receive_send_req(const char *nick, const char *email, char *fname,
                             u_long inaddr, u_short port, off_t fsize,
                             CheckSum cksum)
{
  DCC *p;
  char *tmp;

  if (!fname)
    return;
if(debug)
  fprintf(stderr, "From %s!%s(%lu %hu):%s; checksum:%lu size:%lu\n",
          nick, email, inaddr, port, fname, cksum, fsize);
  tmp = strrchr(fname, '/');
  if (tmp && !(*++tmp))
  {
    if ((gflags & VERBOSE_CLIENT))
    {
      say2(0, W, 1,
           PROMPT "%s (%s) is sending a directory %s..ignored.", nick, email,
           fname);
    }
    return;
  }
  if (!tmp)
    tmp = fname;
  if (*tmp == '~')
    tmp++;                      /*   ~yyydsds                                                                                        */
  if (!(*tmp))
    return;                     /*  sending a '~'  */
  p = dccStart;
  dccStart = my_malloc(sizeof(DCC));
  memset(dccStart,0, sizeof(DCC));
  dccStart->type = DCC_FILE | DCC_RECV;
  dccStart->id = ++dcc_id;
  dccStart->fd = -1;
  dccStart->w=W;
  dccStart->dcc.file = my_malloc(sizeof(DCCFile));
  memset(dccStart->dcc.file, 0, sizeof(DCCFile));
  dccStart->dcc.file->fp = -1;
  dccStart->dcc.file->name = strdup(tmp);
  dccStart->dcc.file->size = fsize;
  dccStart->dcc.file->checksum = cksum;
  dccStart->dcc.file->resume = 0;
  dccStart->nick = strdup(nick);
  dccStart->peer = strdup(email);
  dccStart->starttime = time(NULL);
  dccStart->rport = port;
  dccStart->inetaddr = inaddr;
  dccStart->next = p;
  if (gflags & BEEP_DCC)
    spx_bell(0);

  fmt_string = "%n%h%i%f%j%k%z%Z$";	//nick,user@host, port, inaddr. no fmt char to add id.
  tmp = format_msg(fmt_dcc_file_notify,
                   nick, dccStart->peer, dccStart->id,
                   dccStart->dcc.file->name,
                   dccStart->dcc.file->size,
                   dccStart->dcc.file->checksum, port, inaddr);
  if (tmp)
  {
    say(tmp, W, 1);
    free(tmp);
  }
  if (gflags & AUTO_GETFILE)
  {
    if (gflags & VERBOSE_CLIENT)
      say2(0, W, 1, "Autogetting %s (%d) from %s (%s)",
           dccStart->dcc.file->name, dccStart->dcc.file->size,
           dccStart->nick, dccStart->peer);

    tmp = malloc(sizeof(char) * 24);

    sprintf(tmp, "/dcc get %d", dccStart->id);
    process_cmd(tmp, Win);
  }

}

static int chat_req_exists(const char *email)
{
  DCC *p = dccStart;

  while (p != dccEnd)
  {
    if ((p->type & DCC_CHAT) && (p->type& DCC_RECV)
      //&& (p->type & ~DCC_ACTIVE)
        && !strcasecmp(email, p->peer))
      return 1;
    else
      p = p->next;
  }
  return 0;
}
extern void receive_chat_req(const char *nick, const char *email,
                             u_long inaddr, u_short port)
{

  DCC *p;
  char *buf;

  fprintf(stderr, "got CHAT req:%s %s %lu %d\n", nick, email, inaddr, port);
  if (chat_req_exists(email))
  {
    fprintf(stderr, "[ignored] new chat request from %s. DCC chat or request\n"
            " from %s already exists.", email, email);
    return;  /*
              ignore chat request if one already
              exists for that user@domain (in progress or waiting).
              this prevents flooding, but isnt good if two
              people are using same address */
  }
  p = dccStart;
  dccStart = my_malloc(sizeof(DCC));
  memset(dccStart,0, sizeof(DCC));
  dccStart->id = ++dcc_id;
  dccStart->fd = -1;
  dccStart->w=W;
  dccStart->type = DCC_CHAT | DCC_RECV;
  dccStart->nick = strdup(nick);
  dccStart->peer = strdup(email);
  dccStart->rport = port;
  dccStart->inetaddr = inaddr;
  dccStart->starttime = time(NULL);
  dccStart->next = p;

  fmt_string = "%n%h%i%z%Z$";   //nick,user@host, port, inaddr. no fmt char to add id.
  buf = format_msg(fmt_dcc_chat_notify,
                   nick, email, dccStart->id, port, inaddr);
  if (buf)
  {
    say(buf, W, 1);
    free(buf);
  }
  if (gflags & BEEP_DCC)
    spx_bell(0);
  if (gflags & AUTO_CHAT)
  {
    if (gflags & VERBOSE_CLIENT)
      say2(0, W, 1, "Accepting chat offer from %s (%s)",
           dccStart->nick, dccStart->peer);
    buf = malloc(sizeof(char) * 20);

    sprintf(buf, "/dcc chat %d", dccStart->id);
    process_cmd(buf, Win);
    free(buf);
  }

}

extern void accept_chat(DCC * q, Winstruct * win)
{
  int fd;
  int w = win ? win - winstruct : -1;
  struct sockaddr_in buf;
  int len;
  char *remote;

  if (debug)
    fprintf(stderr, "opening connection to DCC chat... #%d:%s %lu",
            q->id, q->nick, q->inetaddr);
  fd = tcp_open(NULL, q->inetaddr, NULL, q->rport, connect_timeout);
  if (fd < 0)
  {
    say2(0, w, 1, PROMPT "DCC chat connection to %s (%lu, %d) failed",
         q->nick, q->inetaddr, q->rport);
    if (debug)
      if (tcp_err_msg)
        fprintf(stderr, "failed: %s\n", tcp_err_msg);
    if (tcp_err_msg)
      say(tcp_err_msg, w, 1);
    remove_dcc(q);
    if (gflags & BEEP_ERR)
      spx_bell(0);
    return;
  }
  if (!getpeername(fd, (struct sockaddr *) &buf, &len))
  {
    q->inetaddr = buf.sin_addr.s_addr;
    remote = inet_ntoa(buf.sin_addr);
  }
  else
  {
    struct in_addr in;

    in.s_addr = q->type & DCC_SENT ? q->inetaddr : htonl(q->inetaddr);
    remote = inet_ntoa(in);
  }

  dcc_create_chatwin(fd, q->nick, q->peer, remote, win, q);
  remove_dcc(q);
}

/********************************************************
*   d c c   c h a t                                      
********************************************************/
extern DCCChatWin *search_dccwinsbyfd(int fd)
{
/*
   locate the window for an  fd when u read from fd 
 */
  DCCChatWin *p = dccchatwinStart;

  dccchatwinEnd->fd = &fd;
  while (*p->fd != fd)
    p = p->next;
  if (p == dccchatwinEnd)
    return NULL;
  else
    return p;
}

extern DCCChatWin *search_dccwinsbyid(int id)
{
/*
   find the window for an  fd when u read from fd 
 */
  DCCChatWin *p = dccchatwinStart;

  dccchatwinEnd->id = id;
  while (p->id != id)
    p = p->next;
  return p == dccchatwinEnd ? NULL : p;
}

extern DCCChat *get_windcc_byid(const Winstruct * win, int id)
{
/*
   search the dcc chats on this window for  id 
 */
  DCCChat *current;

  if (!win)
    return NULL;
  current = win->dccStart->right;
  while (current != win->dccStart)
  {
    if (current->id == id)
      return current;
    else
      current = current->right;
  }
  return NULL;
}

#if USE_XFORMS
static void dcc_chat_cb(int fd, void *data)
#else
static void dcc_chat_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
{
  static void parse_chat_output(DCCChatWin * dccwin, int read, char *msg);
  char line[MAXLINE + 1];
  int n;
  DCCChatWin *dccwin;
  char *buf;

  dccwin = search_dccwinsbyfd(fd);
  assert(dccwin != NULL);
  n = read(fd, line, MAXLINE);
  if (n <= 0)
  {
    DCCChat *p;

    if (n < 0 && errno == EAGAIN)
      return;
    spx_remove_io_callback(fd, SPX_READ, dcc_chat_cb);
    close(fd);
    *dccwin->fd = -1;
    buf = (char *) malloc(sizeof(char) * (64 + strlen(dccwin->nick)

                                          + strlen(PROMPT)));

    sprintf(buf, n < 0 ?
            PROMPT "Read error from %s" :
            PROMPT "DCC Connection closed from %s.", dccwin->nick);
    fit_object_label(winstruct[dccwin->w].chanwin->WinStatusLine, buf);
    say0(buf, dccwin->w, 1);
    free(buf);
    p = get_windcc_byid(winstruct + dccwin->w, dccwin->id);
    if (p)
      check_msg_hook(DCC_CHAT_LOST, dccwin->w,
                     ", %s %lu %s %s", p->nick, dccwin->read,
                     p->rem_host, p->email);

    n = remove_dccchat(dccwin->id, NULL);
    assert(n);
    UpdateStatus(winstruct + dccwin->w);
  }
  else
  {
    line[n] = '\0';
    *dccwin->read += n;
    if (line[n - 1] == '\n')
      line[--n] = '\0';
    if (n)
      parse_chat_output(dccwin, n, line);
  }
}
static void parse_chat_output(DCCChatWin * dccwin, int read, char *msg)
{

  if (!check_msg_hook(DCC_CHAT_MSG, dccwin->w, ", %s %s", dccwin->nick, msg))
  {
    char *bigbuf;
    int ll;

    ll = read;
    ctcp_unquote_it(msg, &ll);
    fmt_string = "%n%p%z%Z$";
    bigbuf = format_msg(fmt_other_dccchat, dccwin->nick, msg,
                        read, dccwin->read);
    if (bigbuf)
    {
      isay("DCC", bigbuf, dccwin->w, 0);
      free(bigbuf);
    }
  }
}

static int dcc_create_chatwin(int fd, char *nick,
                              char *email, char *rem_host,
                              Winstruct * win, DCC * dcc)
/*
   create a chat connection; create new window for chat if win==NULL.
 */
{
  DCCChatWin *p;
  DCCChat *d;
  struct linger l;
  int keepalive = 1;
  char new_win = 1;             /* creating a new window? */
  char *label = 0;              /* window title */

  d = my_malloc(sizeof(DCCChat));
  memset(d,0,sizeof(DCCChat));
  d->fd = fd;
  d->nick = strdup(nick);
  d->email = strdup2(email);
  d->rem_host = strdup(rem_host);
  d->port=dcc->rport;
  d->starttime = time(NULL);
  l.l_onoff = 0;
  l.l_linger = 0;
  if (setsockopt(d->fd, SOL_SOCKET, SO_LINGER, (void *) &l, sizeof(l)) < 0)
    error(-1, "(ignored) setsockopt SOL_SOCKET error");
  if (setsockopt(d->fd, SOL_SOCKET, SO_KEEPALIVE, (void *) &keepalive,
                 sizeof(keepalive)) < 0)
    error(-1, "(ignored) setsockopt SO_KEEPALIVE error");
  if (fcntl(d->fd, F_SETFL, O_NONBLOCK) == -1)
    error(-1, "[%s %d]fcntl error", __FILE__, __LINE__);
  p = dccchatwinStart;
  dccchatwinStart = (DCCChatWin *) my_malloc(sizeof(DCCChatWin));
  memset(dccchatwinStart,0, sizeof(DCCChatWin));
  d->id = ++dcc_id;
  dccchatwinStart->next = p;
  dccchatwinStart->id = d->id;
  /*
     *dccchatwinStart->fd=d->fd  below 
   */
  dccchatwinStart->nick = d->nick;
  dccchatwinStart->email = d->email;
  dccchatwinStart->rem_host=d->rem_host;
  dccchatwinStart->port=&d->port;
  dccchatwinStart->starttime = &d->starttime;
  if (win)
  {
    register int i;

    for (i = 0; i < win_count; i++)
      if (winstruct[i].chanwin && win == &winstruct[i])
      {
        dccchatwinStart->w = win - winstruct;
        new_win = 0;
        break;
      }
  }
  if (new_win)
  {
    win = new_chan_win();
    dccchatwinStart->w = win - winstruct;
  }
  d->left = win->dccCur;
  d->right = win->dccCur->right;
  win->dccCur->right->left = d;
  win->dccCur->right = d;
  win->dccCur = d;
  dccchatwinStart->read = &win->dccCur->read;
  dccchatwinStart->wrote = &win->dccCur->wrote;
  win->dccCur->wrote = win->dccCur->read = 0;
  dccchatwinStart->fd = &win->dccCur->fd;
  if (new_win)
  {
    label =malloc(sizeof(char) *(strlen(win->dccCur->nick) + 100 +
                             (win->dccCur->email ? strlen(win->dccCur->email) : 0)));

    sprintf(label, "[%d] " sula_NAME ": DCC chat with %s (%s)",
            (win - winstruct) + 1, win->dccCur->nick ,
            dccchatwinStart->email);
    show_chanwin(win, &label, 1, 0);
  }
  spx_add_io_callback(win->dccCur->fd, SPX_READ, dcc_chat_cb, 0);
  {
    char buf[255];

    if (win->server == NULL)
    {
      sprintf(buf, sula_NAME " " sula_VERSION ": DCC chat with %s",
              win->dccCur->nick);
      SET_WINDOW_TITLE(win->chanwin->chanwin, buf);
    }
    if (!check_msg_hook(DCC_CHAT_START,
                        win - winstruct,
                        ", %s %s %s", win->dccCur->nick,
                        win->dccCur->rem_host,
                        dccchatwinStart->
                        email ? dccchatwinStart->email : rem_host))
    {
      sprintf(buf, "DCC CHAT connection to %s (%s) established (%d)",
              win->dccCur->nick, email ? email : "-", win->dccCur->fd);
      fit_object_label(win->chanwin->WinStatusLine, buf);
      if (gflags & BEEP_DCC)
        spx_bell(0);
#if DEBUG
      fprintf(stderr, "You are talking to %s (%s) - %s.\n",
              win->dccCur->nick, dccchatwinStart->email,
              rem_host);
#endif
    }
  }
  SET_BUTTON(win->chanwin->r_dcc_chat, 1);
  UpdateStatus(win);
  if (new_win)
    show_chanwin(win, &label, 0, 1);
  return 0;
}

extern int sendto_dcc_chat(Winstruct * win, DCCChat * dcc_chat,
                           const char *buf)
{

  if (!win || dcc_chat == win->dccStart
      || win->dccStart->left == win->dccStart)
    return -1;
  if (dcc_chat->fd < 0)
  {
    spx_bell(0);
    fit_object_label(win->chanwin->WinStatusLine, "No DCC chat connection");
    return -1;
  }
  /*
     if (!check_msg_hook(SEND_CHATMSG, win - winstruct,
     ", %s %s %s", dcc_chat->nick,
     dcc_chat->email, buf))
   */
  {
    int n, w;

    n = strlen(buf);
    w = writen(dcc_chat->fd, buf, n);
    if (w != n)
    {
      if (errno == EAGAIN)
        return w;
      else
      {
        say2(0, win - winstruct, 1,
             "Dcc chat: write error to %s.", dcc_chat->nick);
        remove_dccchat(dcc_chat->id, NULL);
        return w;
      }
    }
    dcc_chat->wrote += w;
    return w;
  }
  return 0;
}

extern int remove_dccchat(int id, char *reason)
{
/*
   close a chat connecion 
 */
  DCCChat *p;
  DCCChatWin *q, *qq;
  Winstruct *win;
  ulong bytes_read, bytes_written;

  q = search_dccwinsbyid(id);
  if (!q)
    return 0;
  win = winstruct + q->w;
  p = get_windcc_byid(win, q->id);	/* find the chat connection on this window */
  if (!p)
    return 0;
  assert(p != win->dccStart);
  qq = q->next;
  bytes_read = *q->read;
  bytes_written = *q->wrote;
  if (qq == dccchatwinEnd)
    dccchatwinEnd = q;
  else
    *q = *qq;
  free(qq);
  if (p->fd > 0)
  {
    char *buf;

    spx_remove_io_callback(p->fd, SPX_READ, dcc_chat_cb);
    if (reason)
    {
      buf = (char *) alloca(sizeof(char) * (strlen(reason) + 2));

      if (buf)
      {
        sprintf(buf, "%s\n", reason);
        writen(p->fd, buf, strlen(buf));
      }
    }
    else if (chat_quit_text)
    {
      buf = (char *) alloca(sizeof(char) * (strlen(chat_quit_text) + 2));

      if (buf)
      {
        sprintf(buf, "%s\n", chat_quit_text);
        writen(p->fd, buf, strlen(buf));
      }
    }
    // go:
    close(p->fd);
    check_msg_hook(DCC_CHAT_DONE, win - winstruct,
                   ", %s %lu %lu %s %s %s", p->nick, bytes_read,
                   bytes_written, p->rem_host, p->email);
  }
  free(p->nick);
  free(p->email);
  free(p->rem_host);
  p->left->right = p->right;
  p->right->left = p->left;
  if (p == win->dccCur)
    win->dccCur = p->left;
  if (win->dccCur == win->dccStart)
    win->dccCur = win->dccStart->right;
  if (TO_DCC(win) &&
      (win->dccCur == win->dccStart || win->dccStart->left == win->dccStart))
    SET_BUTTON(win->chanwin->r_echo, 1);

  /*
     caller should update window status line 
   */
  return 1;
}
#if USE_XFORMS
static void dcc_sentchats_cb(int fd, void *data)
#else
static void dcc_sentchats_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
{
/*
   accept reply to a chat request that you sent out 
 */
  DCC *p = dccStart;
  int newfd, ll;
  struct sockaddr_in they;

  spx_remove_io_callback(p->fd, SPX_READ, dcc_sentchats_cb);
  while (p != dccEnd && p->fd != fd &&
         !(p->type & (DCC_CHAT | DCC_SENT) & ~DCC_ACTIVE))
    p = p->next;
  assert(p != dccEnd);
  if ((newfd = accept(p->fd, (struct sockaddr *) &they, &ll)) < 0)
  {
    error(-1, "accept error");
    remove_dcc(p);
    return;
  }
  dcc_create_chatwin(newfd, p->nick, p->peer,
                     (char *) inet_ntoa(they.sin_addr), p->dcc.chat->win, p);
  remove_dcc(p);
}

extern int dcc_sendchatreq(Winstruct * win, Winstruct * w,
                           const char *userhost)
{
/*
   send a chat request from within window win;
   w wil be used to chat when reply arrives (can be NULL).
 */
  DCC *p;
  int fd, ll;
  struct sockaddr_in here;

  char *buf;

  if ((sflags & DYNAMIC_SLIP) || me.sin_addr.s_addr == 0)
  {
    char *errstr="gethostname error";
    memset((void *) &me, 0, sizeof(me));
    if (gethostname(host, sizeof(host)) < 0)
    {
      fprintf(stderr, "%s: ", host);
      herror(errstr);
      me.sin_addr.s_addr = 0;
      return -1;
    }
    if ((me.sin_addr.s_addr = inet_addr(host)) == INADDR_NONE)
    {
      struct hostent *hp;

      hp = gethostbyname(host);
      if (hp == NULL)
      {
        herror(errstr);
        me.sin_addr.s_addr = 0;
        return -1;
      }
      memcpy((void *) &me.sin_addr, (void *) hp->h_addr, hp->h_length);
    }
  }
  if ((fd = tcp_create_server(1)) < 0)
    return -1;
  ll = sizeof(here);
  if (getsockname(fd, (struct sockaddr *) &here, &ll) < 0)
  {
    error(-1, "getsockname error");
    close(fd);
    return -1;
  }
  p = dccStart;
  dccStart = my_malloc(sizeof(DCC));
  memset(dccStart,0, sizeof(DCC));
  buf = strchr(userhost, '!');
  if (buf)
  {
    *buf = 0;
    dccStart->nick = strdup(userhost);
    dccStart->peer = strdup(buf + 1);
    *buf = '!';
  }
  else
  {
    dccStart->nick = strdup(userhost);
    dccStart->peer = strdup(userhost);
  }
  dccStart->id = ++dcc_id;
  dccStart->fd = fd;
  dccStart->next = p;
  dccStart->starttime = time(NULL);
  dccStart->dcc.chat = my_malloc(sizeof(DCCChatSent));
  dccStart->dcc.chat->server = win && win->server ? win->server->name : NULL;
  dccStart->dcc.chat->win = w;
  dccStart->type = DCC_CHAT | DCC_SENT;
  dccStart->lport = here.sin_port;
  spx_add_io_callback(dccStart->fd, SPX_READ, dcc_sentchats_cb, 0);
  buf = malloc(sizeof(char) * (strlen(dccStart->nick) + 60));

  sprintf(buf, "PRIVMSG %s :%cDCC CHAT chat %lu %u%c\n",
          dccStart->nick, 0x01,
          (unsigned long) ntohl(me.sin_addr.s_addr),
          ntohs(here.sin_port), 0x01);
  sendto_server(win, buf);
  free(buf);
  return 0;

}


static int check_hook_filestop(DCC * p)
{
/*
   remove  from the list. called by dcc_getfile_cb and offeredfiles_cb 
 */
  //char buf[256];
  int say_nothing;

  //sprintf(buf, "%s %s %lu %lu", p->email, p->filename, p->size, p->read);
  if (p->type & DCC_SENT)
  {
    say_nothing = check_msg_hook(DCC_SEND_DONE, -1,
                                 ", %s %s %lu %lu",
                                 p->peer,
                                 p->dcc.file->name,
                                 p->dcc.file->size, p->dcc.file->read);
  }
  else
  {
    say_nothing = check_msg_hook(DCC_GET_DONE, -1,
                                 ", %s %s %s %lu %lu",
                                 tohost(p->inetaddr),
                                 p->peer, p->dcc.file->name,
                                 p->dcc.file->size, p->dcc.file->read);
  }
  return say_nothing;
}
#if USE_XFORMS
static void dcc_getfile_cb(int fd, void *data)
#else
static void dcc_getfile_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
/*
   to be called only by the io_callback! This handles the actual file
   transfer 
 */
{
  char line[MAXLINE + 1];
  int n;
  DCC *p;

  p = dccStart;
  dccEnd->fd = fd;
  while (p->fd != fd)
    p = p->next;
  assert(p != dccEnd);
  if ((n = read(fd, line, MAXLINE)) < 0)
  {
    if (errno != EAGAIN)
    {
      spx_remove_io_callback(fd, SPX_READ | SPX_WRITE, dcc_getfile_cb);
      if (!check_hook_filestop(p))
      {

        say2(0, -1, 1, PROMPT
             "DCC get: Read error from %s for %s (read %lu of %lu bytes)",
             p->peer, p->dcc.file->name, p->dcc.file->read,
             p->dcc.file->size);
        if (gflags & BEEP_ERR)
          spx_bell(0);
      }
      remove_dcc(p);
    }
    return;
  }
  if (n == 0)
  {
    float diff;
    float read;
    struct timeval t;

    gettimeofday(&t, NULL);
    spx_remove_io_callback(fd, SPX_READ | SPX_WRITE, dcc_getfile_cb);

    read =
       p->dcc.file->size >
       0 ? (p->dcc.file->read * 100.0) / p->dcc.file->size : 100;
    diff =
       t.tv_sec - p->dcc.file->starttime.tv_sec +
       (float) ((t.tv_usec - p->dcc.file->starttime.tv_usec) / 1000000.0);
    if (!check_hook_filestop(p))
    {
      if (p->dcc.file->read >= p->dcc.file->size)
      {
        say2(0, -1, 1,
             PROMPT
             "File %s received from %s. Read %lu bytes (%.0f%%) in %f seconds"
             " (%0.3f kb/sec)", p->dcc.file->name, p->nick, p->dcc.file->read,
             read, diff,
             0.1 * p->dcc.file->read / ((0.1 * 1024) *
                                        (diff == 0 ? 1 : diff)));
      }
      else
      {
        say2(0, -1, 1,
             PROMPT
             "File transfer %s from %s terminated. Read %lu bytes (%.0f%%) in %f seconds"
             " (%0.3f kb/sec)", p->dcc.file->name, p->nick, p->dcc.file->read,
             read, diff,
             (0.1 * p->dcc.file->read) / ((0.1 * 1024) *
                                          (diff == 0 ? 1 : diff)));

      }
      remove_dcc(p);
      if (gflags & BEEP_DCC)
        spx_bell(0);
    }
    return;
  }
  gettimeofday(&p->dcc.file->lasttime, NULL);
  p->dcc.file->read += n;
re_write:
  if (write(p->dcc.file->fp, line, n) != n)
  {
    if (errno == EAGAIN)
      goto re_write;
    else
    {
      say2(0, -1, 1, PROMPT "DCC GET: Write error to local file ",
           p->dcc.file->name);
    out:
      spx_remove_io_callback(fd, SPX_READ | SPX_WRITE, dcc_getfile_cb);
      check_hook_filestop(p);
      remove_dcc(p);
    }
    return;
  }
  else
  {
    long read = htonl(p->dcc.file->read);

  again:
    if (writen(fd, (char *) &read, sizeof(read)) != sizeof(read))
    {
      if (errno == EAGAIN)
        goto again;
      error(-1, "Write error to %s after receiving %lu bytes",
            p->nick, p->dcc.file->read);
      goto out;
    }
  }
}

extern int dcc_getfile(int fd, DCC * f, Winstruct * win)
/*
   start receiving a file 
 */
{
  int keepalive = 1;

  f->fd = fd;

  f->dcc.file->offset = 0;
  f->dcc.file->read = 0;
  //f->dcc.file->w = -1;

  if (setsockopt(f->fd, SOL_SOCKET, SO_KEEPALIVE, (void *) &keepalive,
                 sizeof(keepalive)) < 0)
    error(-1, "(ignored) setsockopt SO_KEEPALIVE");
  if (fcntl(f->fd, F_SETFL, O_NONBLOCK) == -1)
    error(-1, "(ignored) O_NONBLOCK:fcntl error");

  if (f->dcc.file->resume == 1  /*
                                   a client which understands DCC RESUME but not
                                   DCC RENAME will send u an empty file now 
                                 */
      || (f->dcc.file->resume == 0 && access(f->dcc.file->name, F_OK) != -1))
  {
    char *buf=alloca(sizeof(char)*(strlen(f->dcc.file->name)+30));
    srandom(time(NULL));
    sprintf(buf, "%s.%d", f->dcc.file->name, (int) random() % 10000000);
    free(f->dcc.file->name);
    f->dcc.file->name = strdup(buf);
    say2(0, win - winstruct, 1,
         "File exists. Renamed to %s", f->dcc.file->name);
  }
  if ((f->dcc.file->fp = open(f->dcc.file->name, f->dcc.file->resume < 0 ?	// ours is shorter
                              O_WRONLY | O_APPEND | O_CREAT :
                              O_WRONLY | O_CREAT | O_TRUNC,
                              S_IRUSR | S_IWUSR | S_IRGRP)) < 0)
  {
    say2(0,win - winstruct, 1, "%s %s", strerror(errno), f->dcc.file->name);
    check_hook_filestop(f);
    remove_dcc(f);
    if (gflags & BEEP_ERR)
      spx_bell(0);
    return -1;
  }
  gettimeofday(&f->dcc.file->starttime, NULL);
  say2(0, win - winstruct, 1,
       PROMPT "%s transfer for %s (%lu bytes left) from %s (%s)",
       f->dcc.file->resume < 0 ? "Resuming" : "Starting", f->dcc.file->name,
       f->dcc.file->size, f->nick, f->peer);
  spx_add_io_callback(f->fd, SPX_READ, dcc_getfile_cb, 0);
  f->type |= DCC_ACTIVE;
  return 0;
}

/*
   FILE transfers

   offer 
 */
#if USE_XFORMS
static void dcc_sendfile_cb(int fd, void *data)
#else
static void dcc_sendfile_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
/*
   send file. to be called only by the io_callback! 
 */
{
  static void done_sending(DCC * p);
  char *line;
  int n;
  int w;                        /*  bytes written */
  DCC *p;

  p = dccStart;
  while (p != dccEnd && p->fd != fd)
    p = p->next;
  assert(p != dccEnd);
  line = my_malloc2(sizeof(char) * (max_sendbuf + 2));

  if (line == NULL)
  {
    done_sending(p);
    return;
  }
  if ((n = read(fd, line, sizeof(int))) < 0)
  {

    if (errno == EAGAIN)
    {
      free(line);
      return;
    }
    free(line);
    spx_remove_io_callback(fd, SPX_READ | SPX_WRITE, dcc_sendfile_cb);
    if (!check_hook_filestop(p))
      say2(0, -1, 1,
           PROMPT "DCC SEND: Read error from %s (sent %lu of %lu bytes)",
           p->peer, p->dcc.file->read, p->dcc.file->size);
    remove_dcc(p);
    return;
  }
  if (n == 0)
  {
    free(line);
    done_sending(p);
    return;
  }
  n = read(p->dcc.file->fp, line, max_sendbuf);
  if (n == 0)
  {
    free(line);
    if (gflags & VERBOSE_CLIENT)
      if (p->dcc.file->read < p->dcc.file->size - p->dcc.file->offset)
        say2(0, -1, 1, "unexpected EOF %s (local file)", p->dcc.file->name);
    done_sending(p);
    return;
  }
  if (n < 0)
  {
    n = errno;
    free(line);
    if (errno == EAGAIN)        /*wont happen with SA_RESTART */
    {
      errno = 0;
      return;
    }
    spx_remove_io_callback(fd, SPX_READ, dcc_sendfile_cb);
    if (!check_hook_filestop(p))
      say2(0, -1, 1, PROMPT "DCC SEND: Read error on local file %s: %s",
           p->dcc.file->name, strerror(n));
    remove_dcc(p);
    return;
  }
  w = writen(fd, line, n);
  if (w != n)
  {
    if ((errno == EAGAIN) && lseek(p->dcc.file->fp, w - n, SEEK_CUR) != -1)
    {                           /* wrote w bytes instead of n */
      goto weiter;
    }                           /*    let'S try again                                                                                */
    free(line);
    spx_remove_io_callback(fd, SPX_READ, dcc_getfile_cb);
    if (!check_hook_filestop(p))
      say2(0, -1, 1, PROMPT "DCC SEND: Write error to %s (%s) ",
           p->nick, p->peer);
    remove_dcc(p);
    return;
  }
weiter:
  free(line);
  gettimeofday(&p->dcc.file->lasttime, NULL);
  p->dcc.file->read += w;
  return;
}

static void done_sending(DCC * p)
{
  spx_remove_io_callback(p->fd, SPX_READ, dcc_sendfile_cb);
  if (!check_hook_filestop(p))
  {
    float diff;
    struct timeval t;
    float read;

    gettimeofday(&t, NULL);
    read =
       p->dcc.file->size >
       0 ? ((p->dcc.file->read * 100.0) /
            (p->dcc.file->size - p->dcc.file->offset)) : 100;
    diff =
       t.tv_sec - p->dcc.file->starttime.tv_sec +
       (float) ((t.tv_usec - p->dcc.file->starttime.tv_usec) / 1000000.0);

    if (p->dcc.file->read >= p->dcc.file->size)
    {
      say2(0, -1, 1,
           PROMPT
           "File sent to %s!%s (%s). Wrote %lu bytes (%.0f%%) in %f seconds"
           " (%0.3f kb/sec)", p->nick, p->peer, tohost(p->inetaddr),
           p->dcc.file->read, read, diff,
           (0.1 * p->dcc.file->read) / ((0.1 * 1024) *
                                        (diff == 0 ? 1 : diff)));
    }
  }
  remove_dcc(p);
  return;
}
void terminate_file_transfer(DCC * p, int w)
{
/*
   close a DCC-Get/send connection: stop reeivin a file,or stop sending one 
 */
  if (p->type & DCC_SENT)
    spx_remove_io_callback(p->fd, SPX_READ, dcc_sendfile_cb);
  else
    spx_remove_io_callback(p->fd, SPX_READ, dcc_getfile_cb);
  if (!check_hook_filestop(p))
  {
    if (p->type & DCC_ACTIVE)
    {
      float read;

      read = p->dcc.file->size > 0 ?
         ((p->dcc.file->read * 100.0) /
          (p->dcc.file->size - p->dcc.file->offset)) : 100;
      say2(0, w, 1,
           PROMPT
           "File transfer with %s (%s) disrupted. %s %lu bytes (%.0f%%).",
           p->nick, p->peer, p->type & DCC_SENT ? "Wrote" : "Read",
           p->dcc.file->read, read);
    }
    else
      say2(0, w, 1, PROMPT "File offer %s %s (%s) removed.",
           p->type & DCC_SENT ? "to" : "from", p->nick, p->peer);
  }
  remove_dcc(p);
  return;
}

#if USE_XFORMS
static void offered_files_cb(int fd, void *data)
#else
static void offered_files_cb(gpointer data, gint fd, GdkInputCondition junk)
#endif
{
/*
   accept reply to an offer that you sent 
 */
  DCC *p = dccStart;
  int newfd, ll;
  struct sockaddr_in they;
  int keepalive = 1;
  struct linger l;
  char *buf;
  int fp;
  int n;
  int w;

  spx_remove_io_callback(p->fd, SPX_READ, offered_files_cb);
  dccEnd->fd = fd;
  while (p->fd != fd)
    p = p->next;
  assert(p != dccEnd);
  assert(p->type & (DCC_FILE | DCC_SENT));
  if ((newfd = accept(p->fd, (struct sockaddr *) &they, &ll)) < 0)
  {
    error(-1, "accept error");
    remove_dcc(p);
    return;
  }
  if ((fp = open(p->dcc.file->name, O_RDONLY | O_NONBLOCK)) < 0)
  {
    say2(0, -1, 1, "%s: %s", p->dcc.file->name, strerror(errno));
    close(newfd);
    remove_dcc(p);
    return;
  }
  p->type = DCC_FILE | DCC_ACTIVE;
  p->fd = newfd;
  p->dcc.file->fp = fp;
  p->dcc.file->read = 0;
  //p->dcc.file->w = -1;
  lseek(p->dcc.file->fp, p->dcc.file->offset, SEEK_SET);
  l.l_onoff = 0;
  l.l_linger = 0;
  if (setsockopt(p->fd, SOL_SOCKET, SO_LINGER, (void *) &l, sizeof(l)) < 0)
    error(-1, "[%s %d] setsockopt SO_LINGER", __FILE__, __LINE__);
  if (setsockopt(p->fd, SOL_SOCKET, SO_KEEPALIVE, (void *) &keepalive,
                 sizeof(keepalive)) < 0)
    error(-1, "[%s %d] setsockopt SO_KEEPALIVE", __FILE__, __LINE__);
  if (fcntl(p->fd, F_SETFL, O_NONBLOCK) == -1)
    error(-11, "[%s %d] fcntl error", __FILE__, __LINE__);
  gettimeofday(&p->dcc.file->starttime, NULL);
  p->inetaddr = they.sin_addr.s_addr;
  if (!check_msg_hook(DCC_SEND_START, -1, ", %s %s %lu %s %s %lu",
                      p->nick,
                      p->dcc.file->name,
                      p->dcc.file->size,
                      tohost(p->inetaddr), p->peer, p->dcc.file->offset))
    if (gflags & VERBOSE_CLIENT)
    {
      char *buf = alloca(sizeof(char) * (strlen(p->dcc.file->name) + 255
                                         + strlen(p->nick) + 64));

      sprintf(buf, "%s transfer for %s (%ld bytes left) to %s (%s)[%s]",
              p->dcc.file->offset ? "Resuming" : "Starting",
              p->dcc.file->name,
              p->dcc.file->size - p->dcc.file->offset,
              p->nick, p->peer, tohost(p->inetaddr));
      say0(buf, -1, 1);
    }
  if (max_sendbuf < 512)
    max_sendbuf = 512;
  buf = (char *) alloca(sizeof(char) * (max_sendbuf + 2));

  if (!buf)
  {
    check_hook_filestop(p);
    error(ERR_MSG, "@C1Error: max_send_buf /Set too high: %s",
          strerror(errno));
    remove_dcc(p);
    return;
  }
  n = read(p->dcc.file->fp, buf, max_sendbuf);
  if (n <= 0)
  {
    if (!check_hook_filestop(p))
    {
      if (n == 0 && n == p->dcc.file->size)
        say2(0, -1, 1,
             PROMPT "File transfer to %s (%s)[%s] completed. Empty file.",
             p->nick, p->peer, tohost(p->inetaddr));
      else
        say2(0, -1, 1, "%s %s", p->dcc.file->name, strerror(errno));
    }
    remove_dcc(p);
    return;
  }
  w = writen(p->fd, buf, n);
  if (w != n)
  {
    if ((errno == EAGAIN) && lseek(p->dcc.file->fp, w - n, SEEK_CUR) != -1)
    {
      errno = 0;
      goto weiter;
    }
    if (!check_hook_filestop(p))
      say2(0, -1, 1, "Write error to %s (%s): %s",
           p->nick, tohost(p->inetaddr), strerror(errno));
    remove_dcc(p);
    return;
  }
weiter:
  gettimeofday(&p->dcc.file->lasttime, NULL);
  p->dcc.file->read = w;
  spx_add_io_callback(p->fd, SPX_READ, dcc_sendfile_cb, 0);
}

extern void
#if 0
remove_file_offer(int id)
#else
remove_dcc(DCC * p)
#endif
{
/*
   remove a DCC entry

   call spx_remove_io_callback(....) before this!!

 */
  DCC *q;

  close(p->fd);
  free(p->nick);
  free(p->peer);
  if (p->type & DCC_FILE)
  {
    free(p->dcc.file->name);
    if (p->dcc.file->fp > 0)
      close(p->dcc.file->fp);
    free(p->dcc.file);
  }
  else if ((p->type & DCC_SENT) & ~DCC_ACTIVE)
    free(p->dcc.chat);
  q = p->next;
  if (q == dccEnd)
    dccEnd = p;
  else
    *p = *q;
  free(q);
}
void remove_dcc_by_id(int id)
{
  DCC *p = dccStart;

  while (p != dccEnd && p->id != id)
    p = p->next;
  if (p == dccEnd)
    return;

  #if USE_GTK
  // uncomment after testing the others
  spx_remove_input(p->fd);
  #else
  if (p->type & ~DCC_ACTIVE)
  {
    if (p->type & (DCC_FILE | DCC_SENT))
      spx_remove_io_callback(p->fd, SPX_READ, offered_files_cb);
    else if (p->type & (DCC_CHAT | DCC_SENT))
      spx_remove_io_callback(p->fd, SPX_READ, dcc_sentchats_cb);
  }
  else
  {
    if (p->type & (DCC_FILE | DCC_SENT))
      spx_remove_io_callback(p->fd, SPX_READ, dcc_sendfile_cb);
    else if (p->type & (DCC_FILE | DCC_RECV))
      spx_remove_io_callback(p->fd, SPX_READ, dcc_getfile_cb);
    else if (p->type & (DCC_CHAT))
      spx_remove_io_callback(p->fd, SPX_READ, dcc_chat_cb);
  }
  #endif
  remove_dcc(p);
}

extern void offer(int argc, char **argv, int w)
{
/*
 *  offer files
 *   send nick!user@host file1 file2 ...
 *      or
 *    send file1 file2 to nick!user@host
 */
  if (argc < 3)
  {
    if (gflags & VERBOSE_CLIENT)
      say0(PROMPT "Send: missing args", w, 1);
    return;
  }
  else
  {
    /*
       note: there's now no need to use a queue here. will be changed 
     */
    typedef struct TFiles
    {
      char *name;
      struct TFiles *link;
    }
    Files;
    char *nick = NULL;
    Files *front = NULL, *rear = NULL;
    glob_t gl;
    char start = 1;

    argv++;
    while (*argv)
    {
      Files *p;

      if (!nick && (**argv == 't' || **argv == 'T')
          && ((*argv)[1] == 'o' || (*argv)[1] == 'O') && (*argv)[2] == 0)
      {
        if (*++argv)
          nick = strdup(*argv++);
        else
        {
          if (gflags & VERBOSE_CLIENT)
            say0(PROMPT "Send:missing dest.", w, -1);
          return;
        }
        continue;
      }
      p = (Files *) my_malloc2(sizeof(Files));
      if (!p)
        return;
      if (!rear)
        front = p;
      else
        rear->link = p;
      p->name = strdup(*argv);
      rear = p;
      argv++;
    }
    if (!nick && rear)
    {                           /*   first parameter shld be the nick  */
      Files *p = front;

      nick = strdup(front->name);
      if (front == rear)
        rear = NULL;
      else
        front = p->link;
      free(p->name);
      free(p);
    }
    if (!rear)
    {
      if (gflags & VERBOSE_CLIENT)
        say0(PROMPT "No files to send?", w, 1);
      return;
    }
    gl.gl_pathc = 0;
    while (rear)
    {
      Files *p = front;
      int result;
      char *file = front->name;

      if (*front->name == '~')
      {
        file = (expand_tilde(front->name));
        if (!file)
        {
          say2(0, w, 1, PROMPT "Cant expand %s: %s",
               front->name, strerror(errno));
          goto remove;
        }
      }
      else
        file = strdup(front->name);
      if (start)
      {
        result = glob(file, GLOB_NOSORT, NULL, &gl);
        start = 0;
      }
      else
        result = glob(file, GLOB_APPEND | GLOB_NOSORT, NULL, &gl);
      if (result && errno)
      {
        say2(0, w, 1, PROMPT "%s: %s", file, strerror(errno));
        if (result == GLOB_NOSPACE)
        {
          globfree(&gl);
          free(file);
          return;
        }
      }
      free(file);
    remove:
      if (front == rear)
        rear = NULL;
      else
        front = p->link;
      free(p->name);
      free(p);
    }
    if (gl.gl_pathc == 0)
    {
      say0(PROMPT "No matching files found", w, 1);
      return;
    }
    argv = gl.gl_pathv;

    while (*argv)
    {
      /*
       * shldnt have put the filenames in a queue in the first place..
       * now it is LIFO $&%-? i'm not rewriting this again.
       */
      struct stat buf;

      if (stat(*argv, &buf) < 0)
      {
        say2(0, w, 1, PROMPT "%s: stat error: %s",
             *argv, strerror(errno));
        argv++;
        continue;
      }
      if (!S_ISREG(buf.st_mode))
      {
        say2(0, w, 1, PROMPT "Send: %s: not a regular file",
             *argv);
        argv++;
        continue;
      }
      if (access(*argv, R_OK) < 0)
      {
        say2(0, w, 1, PROMPT "%s: no read access: %s",
             *argv, strerror(errno));
        argv++;
        continue;
      }
      if (addto_sentfileoffer(nick, *argv, buf.st_size, w))
        break;
      argv++;
    }
    globfree(&gl);
  }
}

extern int addto_sentfileoffer(const char *userhost,	/* nick!user@host */
                               char *file, off_t size, int w)
{
  int fd, ll;
  struct sockaddr_in here;

  DCC *p;
  char *buf;
  char *tmpfile;                /*  the name that receipient will see */

  tmpfile = strrchr(file, '/');
  if (tmpfile)
  {
    tmpfile++;
    if (*tmpfile == 0)
      return 0;
  }
  else
  {
    tmpfile = strrchr(file, '~');
    if (!tmpfile)
      tmpfile = file;
  }

  /*
   * we want to determine our internet socket address only once. 
   * on failure try later 
   */
  if ((sflags & DYNAMIC_SLIP) || me.sin_addr.s_addr == 0)
  {
    memset((void *) &me, 0, sizeof(me));
    if (gethostname(host, sizeof(host)) < 0)
    {
      fprintf(stderr, "%s: ", host);
      herror("gethostname error");
      me.sin_addr.s_addr = 0;
      return -1;
    }
    if ((me.sin_addr.s_addr = inet_addr(host)) == INADDR_NONE)
    {
      struct hostent *hp;

      hp = gethostbyname(host);
      if (hp == NULL)
      {
        herror("gethostbyname error");
        me.sin_addr.s_addr = 0;
        return -1;
      }
      memcpy((void *) &me.sin_addr, (void *) hp->h_addr, hp->h_length);
    }
  }
  if ((fd = tcp_create_server(1)) < 0)
    return errno;
  ll = sizeof(here);
  if (getsockname(fd, (struct sockaddr *) &here, &ll) < 0)
  {
    error(-1, "getsockname error");
    close(fd);
    return -1;
  }
  p = dccStart;
  dccStart = my_malloc(sizeof(DCC));
  memset(dccStart,0, sizeof(DCC));
  buf = strchr(userhost, '!');
  if (buf)
  {
    *buf = 0;
    dccStart->nick = strdup(userhost);
    dccStart->peer = strdup(buf + 1);
    *buf = '!';
  }
  else
  {
    dccStart->nick = strdup(userhost);
    dccStart->peer = NULL;
  }
  dccStart->type = DCC_FILE | DCC_SENT;
  dccStart->id = ++dcc_id;
  dccStart->dcc.file = my_malloc(sizeof(DCCFile));
  memset(dccStart->dcc.file, 0, sizeof(DCCFile));
  dccStart->dcc.file->fp = -1;
  dccStart->dcc.file->name = strdup(file);
  dccStart->dcc.file->size = size;
  dccStart->starttime = time(NULL);
  dccStart->next = p;
  dccStart->fd = fd;
  dccStart->w=w;
  dccStart->lport = ntohs(here.sin_port);
  spx_add_io_callback(dccStart->fd, SPX_READ, offered_files_cb, 0);
  buf = malloc(sizeof(char) * (strlen(dccStart->nick) +
                               strlen(tmpfile) + 100));

  if (!checksum
      (dccStart->dcc.file->name, 0, -1, &dccStart->dcc.file->checksum))
    sprintf(buf, "PRIVMSG %s :%cDCC SEND %s %lu %hu %lu %lu%c\n",
            dccStart->nick, 0x01, tmpfile,
            (unsigned long) ntohl(me.sin_addr.s_addr),
            ntohs(here.sin_port),
            dccStart->dcc.file->size, dccStart->dcc.file->checksum, 0x01);
  else
  {
    perror(dccStart->dcc.file->name);
    sprintf(buf, "PRIVMSG %s :%cDCC SEND %s %lu %hu %lu%c\n",
            dccStart->nick,
            0x01, tmpfile, (unsigned long) ntohl(me.sin_addr.s_addr),
            dccStart->lport, dccStart->dcc.file->size, 0x01);
  }
  sendto_server(winstruct+w, buf);
  free(buf);
  return 0;
}
